/* unit.m,
 *
 * The base unit class and other unit related stuff like handling
 * updating, shadow creation, etc.
 */

#include <allegro.h>
#include <assert.h>
#include <math.h>
#include "candy/explosion.h"
#include "candy/large-chunk.h"
#include "collision.h"
#include "common.h"
#include "debug.inc"
#include "difficulty.h"
#include "init.h"
#include "linklist.h"
#include "map.h"
#include "nuke.h"
#include "player.h"
#include "powerup.h"
#include "projectile.h"
#include "rotate.h"
#include "rottrans.h"
#include "seborrhea/seborrhea.h"
#include "sound.h"
#include "strlcpy.h"
#include "unit.h"
#include "unit-intern.h"
#include "units/all-units.h"


#define MAX4(alpha,beta,gamma,delta)		(MAX(alpha, MAX(beta, MAX(gamma, delta))))
#define MIN4(alpha,beta,gamma,delta)		(MIN(alpha, MIN(beta, MIN(gamma, delta))))

Unit<Boss> *boss;
int shadow_opacity = 0x40;

/*--------------------------------------------------------------*/

@implementation Unit
        /* Initialization. */
+ (BOOL) loadPrerequisites { return YES; }

+ (BOOL) loadData:(SebFile **)sebum :(const char *)dir
{
    *sebum = [SebFile new];
    return [(*sebum) loadSebumDirectory:dir];
}

- init
{
    [super init];

    critical_health = 0;

    angle = -M_PI_2;
    speed = 0.0;

    sprite = [base_sebum getSebumByName:"dummy48x48"];

    flags = FLAG_SPAWN_CHUNKS_ON_DEATH;
    chunk_colours = CHUNK_GRAY;

    return self;
}

- free
{
    /* Just in case. */
    projectiles_unlock_unit(self);
    return [super free];
}

- (id) setAngle:(double)theta { angle = theta; return self; }
- (id) setX:(double)x_ Y:(double)y_ { x = x_; y = y_; return self; }

- (void) delete
{
    projectiles_unlock_unit(self);
    flags |= FLAG_DEAD;
}

- (void) spawnDyingExplosions
{
    /* Bosses can replace this function for bigger, longer explosions. */
    if (health < -5)
        [self delete];

    if (health % 3 == 0)
	spawn_candy([BigExplosion class], x, y, HIGH_LAYER);

    health--;
    flash_tics += 2;
}

- (void) die
{
    if (flags & FLAG_SPAWN_CHUNKS_ON_DEATH)
	[self burstIntoChunks];

    health = 0;
    flags |= (FLAG_DYING|FLAG_INVINCIBLE);
    [self dropPowerups];

    /* Sounds. */
    play_explosion_sample(x);
}

	/* Drawing. */
- (void) drawShotFlicker:(BITMAP *)dest
{
    int x_ = x - offsetX;
    int y_ = y - offsetY;

    if (rotated_sprite)
	[sprite drawTo:dest X:x_ Y:y_ Tint:0xff:0xc0:0xa0:0xc0 Angle:angle];
    else {
	x_ -= [sprite width]/2;
	y_ -= [sprite height]/2;
	[sprite drawTo:dest X:x_ Y:y_ Tint:0xff:0xc0:0xa0:0xc0];
    }
}

- (void) drawHurtFlicker:(BITMAP *)dest
{
    int x_ = x - offsetX;
    int y_ = y - offsetY;

    if (rotated_sprite)
	[sprite drawTo:dest X:x_ Y:y_ Tint:0xff:0x80:0x60:0x90 Angle:angle];
    else {
	x_ -= [sprite width]/2;
	y_ -= [sprite height]/2;
	[sprite drawTo:dest X:x_ Y:y_ Tint:0xff:0x80:0x60:0x90];
    }
}

- (void) drawNormal:(BITMAP *)dest
{
    int x_ = x - offsetX;
    int y_ = y - offsetY;

    if (rotated_sprite)
	[sprite drawTo:dest X:x_ Y:y_ Angle:angle];
    else
	[super draw:dest];
}

- (void) draw:(BITMAP *)dest
{
    if (not sprite) {
	flags |= FLAG_DEAD;
	game_flags |= FLAG_LEVEL_COMPLETE;
	return;
    }

    if (flash_tics > 0)
	flash_tics--;
    else {
	if (health < critical_health) {
	    flash_tics--;
	    if (flash_tics < -6-4 - health/20)
		flash_tics = 0;
	}
    }

    if ((flash_tics > 0) && (flash_tics % 4 >= 2)) {
	[self drawShotFlicker:dest];
    }
    elif ((flash_tics < 0) && (flash_tics >= -6)) {
	[self drawHurtFlicker:dest];
    }
    else {
	[self drawNormal:dest];
    }
}

- (void) drawShadow:(BITMAP *)dest
{
    double x_, y_;
    int w_, h_;

    if (not shadow)
	return;

    x_ = (x * 3 / 4) + (screen_w / 8) - offsetX;
    y_ = y - offsetY;
    y_ = y_ + SHADOW_DISPLACEMENT * (1 - (2*y_/screen_h));

    if (shadow) {
	w_ = [shadow width]/2;
	h_ = [shadow height]/2;

	if (rotated_sprite)
	    [shadow drawTo:dest X:x_ Y:y_ Alpha:shadow_opacity Angle:angle];
	else
	    [shadow drawTo:dest X:x_-w_ Y:y_-h_ Alpha:shadow_opacity];
    }
}

	/* Update. */
- (enum INACTIVE_UNIT_UPDATE_STATE) readyToActivate
{
    /* Units which are still in the inactive list are queried here
       whether or not they are ready to be moved into the active list
       (ie. activate).

       If any one of the following conditions apply, the unit is
       considered to activate off the screen:
        1) x < 0
	2) x > screen_w
	3) y + activate_y >= top of screen

       These units will delay activation until it they top of the
       screen hits their activation line. This means they are not
       drawn, don't fire, etc. for the moment (intended behaviour). */
    int sprite_w_2 = [sprite width]/2;
    int sprite_h_2 = [sprite height]/2;

    if ((x + sprite_w_2 < 0) ||
	(x - sprite_w_2 > screen_w) ||
	(activate_y <= -screen_h)) {
	if (y-sprite_h_2 + activate_y - SHADOW_DISPLACEMENT >= offsetY)
	    goto activate;
    }

    /* Other units activate as soon as they reach the top of the
       screen or earlier (given by a positive activate_y), but not
       later.  Remember to allow some space for the shadows. */
    else {
        if (y+sprite_h_2 + MAX(0, activate_y) + SHADOW_DISPLACEMENT >= offsetY)
            goto activate;
    }

    return INACTIVE_UNIT_REST;

 activate:
    /* If this unit is marked 2 player only, but only one player is
       alive, kill it. */
    if (flags & FLAG_ONLY_SPAWN_IN_TWO_PLAYER) {
	if (not (player[0] && player[1]))
	    return INACTIVE_UNIT_DELETE;
    }

    return INACTIVE_UNIT_ACTIVATE;
}

- (void) activate
{
    /* This method is called just after the unit is inserted into the
       active unit list. */
}

- (BOOL) mayStillInfluenceGameplay
{
    /* Remember to leave some room for shadows! */
    int x_ = x - offsetX;
    int y_ = y - offsetY + SHADOW_DISPLACEMENT;
    int sprite_w_2 = [sprite width]/2;
    int sprite_h_2 = [sprite height]/2;
    BOOL above_screen = (y_+sprite_h_2 < 0);
    BOOL below_screen = (y_-sprite_h_2 > screen_h + 50);
    double yv = speed*sin(angle);

    if ((x_ + sprite_w_2 < -160) || (x_ - sprite_w_2 > screen_w + 160))
	return NO;

    /* We are facing up, are pass the top of the screen and moving
       faster than the map is scrolling. */
    else if (above_screen && (angle > 0) && (-yv < [current_map scrollRate]))
	return NO;

    else if (below_screen) {
	/* We are facing down. */
	if (angle < 0)
	    return NO;
	
	/* We are travelling up slower than the map scrolls. */
	else if (-yv > [current_map scrollRate])
	    return NO;
    }

    return YES;
}

- (enum THING_UPDATE_STATE) update
{
    /* It has now died.  Remove it from the lists. */
    if (flags & FLAG_DEAD)
	return THING_DEAD;

    /* Spawn some explosions while it still dying. */
    if (flags & FLAG_DYING) {
	[self spawnDyingExplosions];

	/* It has now died.  Remove it from the lists. */
	if (flags & FLAG_DEAD)
	    return THING_DEAD;
    }

    /* Only fire if we are not dying, and players alive. */
    else if ((flags & FLAG_FIRING_ENABLED) &&
	     (game_flags & FLAG_PLAYERS_ALIVE))
	[self fire];

    /* Move if we want. */
    if (flags & FLAG_MOVING_ENABLED)
        [self move];
    else
        [self enableMovement];

    /* Don't call [super update] because we don't want to remove
       things who are still spawning explosions, but have 0 health. */
    if ([self mayStillInfluenceGameplay])
	return THING_NORMAL;
    else
	return THING_DEAD;
}

- (void) enableMovement
{
    /* Units with -screen_w < activate_y < 0 want to begin their
       movement not immediately, but before scrolling off the screen.
       Make these move at the right time. */
    int sprite_h_2 = [sprite height] / 2;

    if (y + sprite_h_2 + activate_y + SHADOW_DISPLACEMENT >= offsetY)
	flags |= FLAG_MOVING_ENABLED;

    /* XXX: rotated units */
}

- (void) move
{
    x += speed * cos(angle);
    y -= speed * sin(angle);
}

- (void) fire {}

	/* Collisions. */
- (int) collisionRoutinePriority
{
    if (rotated_sprite)
	return COLLISION_PRIORITY_ROTATED_UNIT;
    else
	return COLLISION_PRIORITY_UNIT;
}

- (int) flags { return flags; }
- (int) collisionLists { return ALLY_LIST|COLLIDES_WITH_PROJECTILES_AND_NUKES; }

- (BOOL) checkCollisionWith:(Thing<DetectsCollision> *)object
{
    if (flags & FLAG_DYING)
	return NO;

    if (rotated_sprite) {
	double xx, yy;
	int ww, hh;

	[object getX:&xx Y:&yy W:&ww H:&hh];

	return rotated_bounding_box_collision(x, y, w, h, angle, 
					      xx-ww/2, yy-hh/2,
					      xx+ww/2, yy+hh/2);
    }
    else {
	return [super checkCollisionWith:object];
    }
}

- (void) getW:(int *)w_ H:(int *)h_
{
    /* Don't collide with things if you are dead. */
    if ((flags & FLAG_DYING) || (flags & FLAG_DEAD)) {
	*w_ = 0;
	*h_ = 0;
    }
    else if (rotated_sprite) {
	double x1, y1, x2, y2, x3, y3, x4, y4;

	rotate(x-w/2., y-h/2., angle, &x1, &y1);
	rotate(x+w/2., y-h/2., angle, &x2, &y2);
	rotate(x-w/2., y+h/2., angle, &x3, &y3);
	rotate(x+w/2., y+h/2., angle, &x4, &y4);

	if (w_) *w_ = MAX4(x1, x2, x3, x4) - MIN4(x1, x2, x3, x4);
	if (h_) *h_ = MAX4(y1, y2, y3, y4) - MIN4(y1, y2, y3, y4);
    }
    else {
	if (w_) *w_ = w;
	if (h_) *h_ = h;
    }
}

- (int) receiveDamage:(int)damage type:(enum DAMAGE_TYPE)type
{
    (void)type;

    /* Enemy ships can't be repaired. */
    if (damage <= 0 || flags & FLAG_INVINCIBLE || flags & FLAG_DYING)
	return 0;
    else {
	int ret = MIN(damage, health);
	health -= ret;
	flash_tics = 15;
        if (health <= 0)
            [self die];
	return ret;
    }
}

- (int) receiveDamage:(int)damage type:(enum DAMAGE_TYPE)type from:(int)unit_id
{
    int ret = [self receiveDamage:damage type:type];

    if (ret && unit_id >= 0 && player[unit_id]) {
	int score_multiplier;

	if (type == DAMAGE_TYPE_NUKE)
	    score_multiplier = 1;
	else
	    score_multiplier = 10;

	[player[unit_id] increaseScore:ret*score_multiplier];
	if (flags & FLAG_DYING)
	    [player[unit_id] incrementKillsFrom:x :y];
    }

    return ret;
}

	/* Load/Save.  Format: x;y;a[;flags]. */
- (void) importUnitProperties:(char *)str
{
    double x_, y_, theta;
    int f_;

    /* For flags, only keep flags below 0xff.  The others depend on
       whether they were saved or not.  Note: flags is optional.  */
    flags = flags & 0xff;
    if (sscanf(str, "%lf;%lf;%lf;%d", &x_, &y_, &theta, &f_) == 4)
	flags |= f_;

    /* If we are on the hardest difficulty, spawn the 2-player mode's
       ships which don't have powerups. */
    if ((difficulty >= DIFFICULTY_HARDEST) &&
	not (flags  & FLAG_DEATH_SPAWN_POWERUP))
	flags &=~FLAG_ONLY_SPAWN_IN_TWO_PLAYER;


    /* Pass it through setX:Y: since some objects may want to do fancy
       things depending on the initial location (mirroring paths). */
    [[self setX:x_ Y:y_] setAngle:deg2rad(theta)];
}

- (char *) exportUnitProperties:(char *)str
{
    int f;
    snprintf(str, 1024, "%g;%g;%g", x, y, rad2deg(angle));

    f = flags - (flags & 0xff);

    /* Any flags we want to save. */
    if (f) {
	char str2[1024];
	snprintf(str2, sizeof(str2), "%s;%d", str, f);
	strlcpy(str, str2, sizeof(str2));
    }

    return str;
}

	/* For use in map editor only! */
- (double) angle; { return angle; }

#ifdef DEBUG_BOUNDING_BOX
- (void) drawBoundingBox:(BITMAP *)dest
{
    if (rotated_sprite) {
	double x1, x2, x3, x4;
	double y1, y2, y3, y4;
	
	rotate(-(double)w/2.0, -(double)h/2.0, -angle, &x1, &y1);
	rotate( (double)w/2.0, -(double)h/2.0, -angle, &x2, &y2);
	rotate(-(double)w/2.0,  (double)h/2.0, -angle, &x3, &y3);
	rotate( (double)w/2.0,  (double)h/2.0, -angle, &x4, &y4);

        line(dest, x+x1-offsetX, y+y1-offsetY, x+x2-offsetX, y+y2-offsetY, BOUNDING_BOX_COLOUR);
        line(dest, x+x2-offsetX, y+y2-offsetY, x+x4-offsetX, y+y4-offsetY, BOUNDING_BOX_COLOUR);
        line(dest, x+x4-offsetX, y+y4-offsetY, x+x3-offsetX, y+y3-offsetY, BOUNDING_BOX_COLOUR);
        line(dest, x+x3-offsetX, y+y3-offsetY, x+x1-offsetX, y+y1-offsetY, BOUNDING_BOX_COLOUR);

        line(dest, x+x1-offsetX, y+y1-offsetY, x+x4-offsetX, y+y4-offsetY, BOUNDING_BOX_COLOUR);
        line(dest, x+x2-offsetX, y+y2-offsetY, x+x3-offsetX, y+y3-offsetY, BOUNDING_BOX_COLOUR);

    }
    else
	[super drawBoundingBox:dest];
}
#endif
@end


@implementation Unit (Debris)
- (void) burstIntoChunks
{
#define SPAWN_CHUNK(f,c)	if (chunk_colours & f) spawn_candy([c class], x, y, HIGH_LAYER)

    int n;

    for (n = 0; n < candy_amount+1; n++) {
	/* Determine which chunk colours to spawn */
	SPAWN_CHUNK(CHUNK_GRAY,		LargeChunk);
	SPAWN_CHUNK(CHUNK_BLUE,		LargeChunkBlue);
	SPAWN_CHUNK(CHUNK_COFFEE,	LargeChunkCoffee);
	SPAWN_CHUNK(CHUNK_GREEN,	LargeChunkGreen);
	SPAWN_CHUNK(CHUNK_RED,		LargeChunkRed);
	SPAWN_CHUNK(CHUNK_SARDAUKAR,	LargeChunkSardaukar);
    }

#undef SPAWN_CHUNK
}
@end


@implementation Unit (Powerups)
- (void) dropPowerups
{
#define SPAWN_POWERUP(f,c)	if (flags & f) spawn_powerup([c class], x, y)

    /* Some units give powerups when they die. */
    SPAWN_POWERUP(FLAG_DEATH_SPAWN_PRIMARY_POWERUP,	PrimaryPowerup);
    SPAWN_POWERUP(FLAG_DEATH_SPAWN_SECONDARY_POWERUP,	SecondaryPowerup);
    SPAWN_POWERUP(FLAG_DEATH_SPAWN_TERTIARY_POWERUP,	TertiaryPowerup);
    SPAWN_POWERUP(FLAG_DEATH_SPAWN_HEALTH_POWERUP,	HealthPowerup);
    SPAWN_POWERUP(FLAG_DEATH_SPAWN_NUKE_POWERUP,	NukePowerup);

#undef SPAWN_POWERUP
}
@end

/*--------------------------------------------------------------*/
/* Drawing.							*/
/*--------------------------------------------------------------*/

static inline void draw_units_in_list(BITMAP *dest, List *list)
{
    ListIterator *it;

    foreach (it, list) {
	[[it getItem] draw:dest];

#ifdef DEBUG_BOUNDING_BOX
	[[it getItem] drawBoundingBox:dest];
#endif
    }
}

void draw_units(BITMAP *dest, enum UNIT_LIST list)
{
    if (list == ACTIVE_AIR_LIST)
	draw_units_in_list(dest, active_air_list);
    elif (list == ACTIVE_GROUND_LIST)
	draw_units_in_list(dest, active_ground_list);
    elif (list == ALLY_LIST)
	draw_units_in_list(dest, ally_list);
}

void draw_unit_shadows(BITMAP *dest)
{
    ListIterator *it;

    foreach (it, active_air_list)
	[[it getItem] drawShadow:dest];

    foreach (it, ally_list)
	[[it getItem] drawShadow:dest];
}

/*--------------------------------------------------------------*/
/* Updating.							*/
/*--------------------------------------------------------------*/

static inline void update_inactive_list(List *inactive_list, List *active_list)
{
    ListIterator *it, *nx;

    foreach_nx (it, nx, inactive_list) {
	Unit *unit = [it getItem];
	nx = [it next];

	switch ([unit readyToActivate]) {
	  case INACTIVE_UNIT_REST:
	      break;
	  case INACTIVE_UNIT_DELETE:
	      [inactive_list removeItem:unit];
	      [unit delete];
	      unit = [unit free];
	      break;
	  case INACTIVE_UNIT_ACTIVATE:
	      [inactive_list removeItem:unit];
	      [active_list insertItem:unit];
	      [unit activate];
	      break;
	  default:
	      assert(NO);
	}
    }
}

static void update_active_list(List *list)
{
    ListIterator *it, *nx;
    int collision_lists;

    foreach_nx (it, nx, list) {
	Unit *unit = [it getItem];
	nx = [it next];

	collision_lists = [unit collisionLists];

	/* Check for collision with nukes. */
	if (collision_lists & COLLIDES_WITH_NUKES) {
	    Nuke *nuke = nuke_in_contact_with(unit);

	    if (nuke) {
		int d;

		/* Only receive the damage the unit received. */
		d = [unit receiveDamage:25 type:DAMAGE_TYPE_NUKE from:[nuke parentID]];
		[nuke receiveDamage:d];
	    }
	}

        /* Check for collision with allies in the air. */
	if (collision_lists & ALLY_LIST) {
            ListIterator *it2;

            foreach (it2, ally_list) {
		Unit *ally = [it2 getItem];

		/* This is NOT a bogus check.  Micro-nukes are
		   considered ally units, but do not hit air units. */
		if (not ([ally collisionLists] & ACTIVE_AIR_LIST))
		    continue;

                if ([unit collidesWith:ally]) {
		    /* Enemies receive double damage.  This is mainly
		       for satellites. */
                    [unit receiveDamage:20 type:DAMAGE_TYPE_UNIT];
                    [ally receiveDamage:1  type:DAMAGE_TYPE_UNIT];
                }
            }
	}

	if ([unit update] == THING_DEAD) {
	    if (unit == player[0]) {
		player[0] = nil;
		/* Force the status bar to redraw. */
		game_flags |= FLAG_REDRAW_STATUSBARS;
	    }
	    else if (unit == player[1]) {
		player[1] = nil;
		/* Force the status bar to redraw. */
		game_flags |= FLAG_REDRAW_STATUSBARS;
	    }
	    
	    [list removeItem:unit];
	    [unit free];
	}
    }
}

void update_units(void)
{
    /* Move inactive units on the screen to the active list. */
    update_inactive_list(ground_list, active_ground_list);
    update_inactive_list(air_list, active_air_list);

    /* Update the active enemies. */
    update_active_list(active_ground_list);
    update_active_list(active_air_list);

    /* Update the allies. */
    update_active_list(ally_list);
}

/*--------------------------------------------------------------*/
/* Creating/destroying units.					*/
/*--------------------------------------------------------------*/

Unit *spawn_unit(Class class, double x, double y,
		 enum UNIT_LIST list, BOOL at_start)
{
    Unit *unit;
    List *ll;
    assert(class != nil);

    ll = unit_list_number_to_list(list);
    assert(ll && "Attempted to add unit in non-proper list.");

    unit = [[class new] setX:x Y:y];
    if (not unit)
	return nil;

    if (at_start)
	[ll insertItem:unit];
    else
	[ll insertItemAtEnd:unit];

    return unit;
}

void destroy_unit(Unit *unit)
{
    if ([active_air_list removeItem:unit]) {
	[unit delete];
	unit = [unit free];
	return;
    }

    if ([active_ground_list removeItem:unit]) {
	[unit delete];
	unit = [unit free];
	return;
    }
}

void purge_units_below(double yy)
{
    /* Special for jump-starting from Redit. */
    ListIterator *it, *nx;
    Unit *unit;
    double x, y;

    foreach_nx (it, nx, air_list) {
	unit = [it getItem];
	nx = [it next];

	[unit getX:&x Y:&y];
	if (y > yy) {
	    [air_list removeItem:unit];
	    unit = [unit free];
	}
    }

    foreach_nx (it, nx, ground_list) {
	unit = [it getItem];
	nx = [it next];

	[unit getX:&x Y:&y];
	if (y > yy) {
	    [ground_list removeItem:unit];
	    unit = [unit free];
	}
    }
}

/*--------------------------------------------------------------*/
/* Init/shutdown.						*/
/*--------------------------------------------------------------*/

void unit_reset(void)
{
    unit_intern_shutdown();
    unit_intern_init();
    boss = nil;
}


void unit_init(void)
{
    unit_intern_init();
}


void unit_shutdown(void)
{
    boss = nil;
    unit_intern_shutdown();
    mark_all_unit_data_unnecessary();
}
