/* thing.m,
 *
 * Everything is a 'Thing'.  Units, projectiles, powerups, etc.
 */

#include <allegro.h>
#include <assert.h>
#include "collision.inc"
#include "collision.h"
#include "common.h"
#include "debug.inc"
#include "map.h"
#include "seborrhea/seborrhea.h"
#include "thing.h"


@implementation Thing
    /* Initialization. */
+ (BOOL) loadData { return YES; }
+ (void) shutdown {}

- init
{
    [super init];
    health = 1;
    return self;
}

- free
{
    if ([sprite conformsTo:@protocol(UniqueSebum)])
	sprite = [sprite free];
    return [super free];
}

- (id) setX:(double)x_ Y:(double)y_
{
    x = x_;
    y = y_;
    return self;
}

    /* Update. */
- (BOOL) mayStillInfluenceGameplay
{
    /* Determines whether this object will angle back onto the screen
       and influence gameplay.  If not, we can remove it.  Things
       which do not move at a fixed angle may want to change this. */
    int x_ = x - offsetX;
    int y_ = y - offsetY;
    int sprite_w_2 = [sprite width]/2;
    int sprite_h_2 = [sprite height]/2;

    if ((x_ + sprite_w_2 < 0) || (x_ - sprite_w_2 > screen_w) ||
	(y_ + sprite_h_2 < 0) || (y_ - sprite_h_2 > screen_h))
	return NO;
    return YES;
}

- (enum THING_UPDATE_STATE) update
{
    if (health <= 0)
	return THING_DEAD;

    if (not sprite)
	return THING_DEAD;

    if ([self mayStillInfluenceGameplay])
	return THING_NORMAL;
    else
	return THING_DEAD;
}

    /* Drawing. */
- (void) draw:(BITMAP *)dest
{
    if (sprite) {
	int sprite_w_2 = [sprite width]/2;
	int sprite_h_2 = [sprite height]/2;

	[sprite drawTo:dest X:x-sprite_w_2-offsetX Y:y-sprite_h_2-offsetY];
    }
}

- (void) drawBoundingBox:(BITMAP *)dest
{
#ifdef DEBUG_BOUNDING_BOX
    rect_wh(dest, x-w/2-offsetX, y-h/2-offsetY, w, h, BOUNDING_BOX_COLOUR);
#else
    (void)dest;
#endif
}
@end


/*--------------------------------------------------------------*/
/* CollidableThing.						*/
/*--------------------------------------------------------------*/

@implementation CollidableThing
- (int) collisionRoutinePriority
{
    return COLLISION_PRIORITY_LOWEST;
}

- (void) getX:(double* const)x_ Y:(double* const)y_
{
    if (x_) *x_ = x;
    if (y_) *y_ = y;
}

- (void) getW:(int* const)w_ H:(int* const)h_
{
    if (w_) *w_ = w;
    if (h_) *h_ = h;
}

- (void) getX:(double* const)x_ Y:(double* const )y_
	       W:(int* const)w_ H:(int* const)h_
{
    [self getX:x_ Y:y_];
    [self getW:w_ H:h_];
}

- (BOOL) checkCollisionWith:(Thing<DetectsCollision> *)object
{
    /* Replace this routine with we want something fancier than a
       simple bounding box collision. */
    double x1, y1, x2, y2;
    int w1, h1, w2, h2;

    [self   getX:&x1 Y:&y1 W:&w1 H:&h1];
    [object getX:&x2 Y:&y2 W:&w2 H:&h2];

    return bounding_box_collision(x1-w1/2.0, y1-h1/2.0, x1+w1/2.0, y1+h1/2.0,
				  x2-w2/2.0, y2-h2/2.0, x2+w2/2.0, y2+h2/2.0);
}

- (BOOL) collidesWith:(Thing<DetectsCollision> *)object
{
    /* Use the collision detection routine of the object with a higher
       routine priority. */
    if ([object collisionRoutinePriority] > [self collisionRoutinePriority])
	return [object checkCollisionWith:self];
    else
	return [self checkCollisionWith:object];
}
@end
