/* sound.m, 
 *
 * This file handles sound stuff.  Will be more useful when Allegro
 * dependence is removed.
 */

#include <assert.h>
#include "common.h"
#include "seborrhea/seborrhea-allegro.h" /* XXX */
#include "seborrhea/seborrhea.h"
#include "sound.h"


BOOL sound_initialized;
BOOL sound_reverse_stereo;
int sound_vol = 128;


static struct {
    Sebum<SebSample> *sample;
    int voice_n;
    int priority;
} reserved_channels[TOTAL_RESERVED_CHANNELS];

/*------------------------------------------------------------*/
/* Sound.                                                     */
/*------------------------------------------------------------*/

static int calculate_pan(int x)
{
    x = PANRANGE * (x - screen_w/2)/screen_w + (PANMAX/2);

    if (sound_reverse_stereo) {
	return MID(0, PANMAX-x, PANMAX);
    }
    else {
	return MID(0, x, PANMAX);
    }
}


void play_panned_sample(Sebum<SebSample> *spl, int x)
{
    if (spl)
	[spl playWithVolume:sound_vol/SOUND_VOLMAX Pan:calculate_pan(x)];
}


BOOL play_voice(Sebum<SebSample> *spl, int x, int priority,
		enum VOICE_CHANNEL ch)
{
    SAMPLE *s;
    assert(ch < TOTAL_RESERVED_CHANNELS);
    BOOL playing;
    int voice_n;
    int vol, pan;

    if ((not spl) || (sound_vol == 0))
	return YES;

    if ((reserved_channels[ch].sample == nil) ||
	(reserved_channels[ch].voice_n == -1))
	playing = NO;
    else
	playing = (voice_get_position(reserved_channels[ch].voice_n) != -1);

    if (playing) {
	if (reserved_channels[ch].priority >= priority)
	    return NO;
    }

    /* Start playing. */
    s = [(SebSampleAllegro *)spl sample];

    /* XXX?  Something wrong with reallocate voice?  Seems to loose
       some sound channels after a while. */
    if (reserved_channels[ch].voice_n != -1)
	deallocate_voice(reserved_channels[ch].voice_n);

    voice_n = allocate_voice(s);
    if (voice_n == -1)
	return YES;

    vol = [spl defaultVolume] * sound_vol/SOUND_VOLMAX;
    pan = calculate_pan(x);

    voice_set_volume(voice_n, vol);
    voice_set_pan(voice_n, pan);
    voice_start(voice_n);

    reserved_channels[ch].sample = spl;
    reserved_channels[ch].priority = priority;
    reserved_channels[ch].voice_n = voice_n;
    return YES;
}


void play_voice_on_channels(Sebum<SebSample> *spl, int x, int p,
			    enum VOICE_CHANNEL c)
{
    if (sound_vol == 0)
	return;

    if (c == PRIMARY_WEAPON_CHANNELS) {
	for (c = WEAPON_1; c <= WEAPON_4; c++) {
	    if (play_voice(spl, x, p, c))
		return;
	}

	/* Last resort: try to play on any other channel. */
/* 	play_panned_sample(spl, x); */
    }
}


void free_reserved_sounds(void)
{
    int i;

    for (i = 0; i < TOTAL_RESERVED_CHANNELS; i++) {
	reserved_channels[i].sample = nil;
	if (reserved_channels[i].voice_n >= 0) {
	    deallocate_voice(reserved_channels[i].voice_n);
	    reserved_channels[i].voice_n = -1;
	}
    }
}

/*------------------------------------------------------------*/

int sound_init(void)
{
    int i;

    reserve_voices(10, 0);

    set_volume_per_voice(1);
    if (install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL) < 0) {
        fprintf(stderr, "Error initializing sound: %s.\n", allegro_error);
	sound_vol = 0;
	return -1;
    }

    for (i = 0; i < TOTAL_RESERVED_CHANNELS; i++) {
	reserved_channels[i].sample = nil;
	reserved_channels[i].voice_n = -1;
    }

    sound_initialized = YES;
    return 0;
}

void sound_shutdown(void)
{
    free_reserved_sounds();
    sound_initialized = NO;
    remove_sound();
}
