/* ogg.m,
 *
 * Ogg file support using AllegroOGG.
 */

#ifndef NO_OGG

#include <assert.h>
#include "seborrhea/ogg.h"
#include "seborrhea/seborrhea-common.h"
#include "seborrhea/seborrhea-filetype.h"


#define BLOCK_SIZE	40960


static BOOL seborrhea_ogg_load(const char *filename, const char *name_sans_ext,
			       unsigned int _1, Sebum<SebContainer> *parent,
			       DirectoryList *_2);

/*--------------------------------------------------------------*/

@implementation SebOGG
+ (void) load
{
    seborrhea_register_filetype_handler("ogg", seborrhea_ogg_load);
}

- free
{
    [self unloadFile];

    if (music_file) {
	free(music_file);
	music_file = NULL;
    }

    return [super free];
}

/*--------------------------------------------------------------*/
/* Music filename.						*/
/*--------------------------------------------------------------*/

- (id<SebMusic>) setMusicFilename:(const char *)fn
{
    assert(fn);
    assert(not music_file);
    music_file = strdup(fn);
    return self;
}

- (const char *) getMusicFilename
{
    assert(music_file);
    return music_file;
}

/*--------------------------------------------------------------*/
/* Loading/Unloading music.					*/
/*--------------------------------------------------------------*/

- (BOOL) loadFile:(void *)mem :(const long)size
{
    FILE *fp;
    long music_size;
    assert(not music);
    assert(music_file);
    assert(mem);

    fp = fopen(music_file, "rb");
    if (not fp) {
	fprintf(stderr, "Warning: %s not found.\n", music_file);
	return NO;
    }

    music_size = fread(mem, 1, size, fp);
    fclose(fp);

    music = alogg_create_ogg_from_buffer(mem, music_size);
    if (music)
	return YES;
    else
	return NO;
}

- (void) unloadFile
{
    if (music) {
	alogg_destroy_ogg(music);
	music = NULL;
    }
}

/*--------------------------------------------------------------*/
/* Playing.							*/
/*--------------------------------------------------------------*/

- (BOOL) playWithVol:(const int)vol Loop:(const BOOL)loop
{
    return [self playWithVol:vol Pan:128 Speed:1000 Loop:loop];
}

- (BOOL) playWithVol:(const int)vol Pan:(const int)pan Speed:(const int)speed
		Loop:(const BOOL)loop
{
    assert(music);

    if (alogg_play_ex_ogg(music, 32768, vol, pan, speed, loop) == ALOGG_OK) {
	_vol = vol;
	_pan = pan;
	_speed = speed;
	_loop = loop;
	return YES;
    }
    else
	return NO;
}

- (void) adjustVol:(const int)vol
{
    if (music) {
	alogg_adjust_ogg(music, vol, _pan, _speed, _loop);
	_vol = vol;
    }
}

- (BOOL) poll
{
    if (!music)
	return NO;

    if (alogg_poll_ogg(music) == ALOGG_OK) {
	return YES;
    }
    else {
	[self unloadFile];
	return NO;
    }
}

- (void) stop
{
    if (music) {
	alogg_stop_ogg(music);
    }
}
@end


static BOOL seborrhea_ogg_load(const char *filename, const char *name_sans_ext,
			       unsigned int _1, Sebum<SebContainer> *parent,
			       DirectoryList *_2)
{
    SebOGG *seb;
    (void)_1, (void)_2;

    /* Don't load just the Ogg just yet.  Just keep a list of
       filenames for now. */
    seb = [[SebOGG alloc] initWithName:name_sans_ext];
    [seb setMusicFilename:filename];
    [parent addToList:seb];
    return YES;
}

#endif /* NO_OGG */
