/* mp3.m,
 *
 * MP3 file support using AllegroMP3.
 */

#ifndef NO_MP3

#include <assert.h>
#include "seborrhea/mp3.h"
#include "seborrhea/seborrhea-common.h"
#include "seborrhea/seborrhea-filetype.h"


static BOOL seborrhea_mp3_load(const char *filename, const char *name_sans_ext,
			       unsigned int _1, Sebum<SebContainer> *parent,
			       DirectoryList *_2);

/*--------------------------------------------------------------*/

@implementation SebMP3
+ (void) load
{
    seborrhea_register_filetype_handler("mp3", seborrhea_mp3_load);
}

- free
{
    [self unloadFile];

    if (music_file) {
	free(music_file);
	music_file = NULL;
    }

    return [super free];
}

/*--------------------------------------------------------------*/
/* Music filename.						*/
/*--------------------------------------------------------------*/

- (id<SebMusic>) setMusicFilename:(const char *)fn
{
    assert(fn);
    assert(not music_file);
    music_file = strdup(fn);
    return self;
}

- (const char *) getMusicFilename
{
    assert(music_file);
    return music_file;
}

/*--------------------------------------------------------------*/
/* Loading/Unloading music.					*/
/*--------------------------------------------------------------*/

- (BOOL) loadFile:(void *)mem :(const long)size
{
    FILE *fp;
    long music_size;
    assert(not music);
    assert(music_file);
    assert(mem);

    fp = fopen(music_file, "rb");
    if (not fp) {
	fprintf(stderr, "Warning: %s not found.\n", music_file);
	return NO;
    }

    music_size = fread(mem, 1, size, fp);
    fclose(fp);

    music = almp3_create_mp3(mem, music_size);
    if (music)
	return YES;
    else
	return NO;
}

- (void) unloadFile
{
    if (music) {
	almp3_destroy_mp3(music);
	music = NULL;
    }
}

/*--------------------------------------------------------------*/
/* Playing.							*/
/*--------------------------------------------------------------*/

- (BOOL) playWithVol:(const int)vol Loop:(const BOOL)loop
{
    return [self playWithVol:vol Pan:128 Speed:1000 Loop:loop];
}

- (BOOL) playWithVol:(const int)vol Pan:(const int)pan Speed:(const int)speed
		Loop:(const BOOL)loop
{
    assert(music);

    if (almp3_play_ex_mp3(music, 32768, vol, pan, speed, loop) == ALMP3_OK) {
	_vol = vol;
	_pan = pan;
	_speed = speed;
	_loop = loop;
	return YES;
    }
    else
	return NO;
}

- (void) adjustVol:(const int)vol
{
    if (music) {
	almp3_adjust_mp3(music, vol, _pan, _speed, _loop);
	_vol = vol;
    }
}

- (BOOL) poll
{
    if (!music)
	return NO;

    if (almp3_poll_mp3(music) == ALMP3_OK) {
	return YES;
    }
    else {
	[self unloadFile];
	return NO;
    }
}

- (void) stop
{
    if (music) {
	almp3_stop_mp3(music);
    }
}
@end


static BOOL seborrhea_mp3_load(const char *filename, const char *name_sans_ext,
			       unsigned int _1, Sebum<SebContainer> *parent,
			       DirectoryList *_2)
{
    SebMP3 *seb;
    (void)_1, (void)_2;

    /* Don't load just the MP3 just yet.  Just keep a list of
       filenames for now. */
    seb = [[SebMP3 alloc] initWithName:name_sans_ext];
    [seb setMusicFilename:filename];
    [parent addToList:seb];
    return YES;
}

#endif /* NO_MP3 */
