/* container.m,
 *
 * Abstract container class, from which we get Files, Animations, etc.
 */

#include <assert.h>
#include <stdlib.h>
#include "seborrhea/container.h"
#include "seborrhea/container-animation.h"
#include "seborrhea/container-lump.h"
#include "seborrhea/container-file.h"
#include "seborrhea/dirlist-recursive.h"
#include "seborrhea/dirlist-zzip.h"
#include "seborrhea/seborrhea-common.h"
#include "seborrhea/seborrhea-datadir.h"
#include "seborrhea/seborrhea-debug.h"
#include "seborrhea/seborrhea-filetype.h"
#include "seborrhea/strdup.h"


static int seb_container_filter(const char *filename)
{
    /* Ignore hidden files, =meta, =contents */
    return (filename[0] != '.' &&
	    strneq(filename, "=contents") &&
	    strneq(filename, "=ignore") &&
	    strneq(filename, "=meta"));
}


@implementation AbstractSebContainer
- init
{
    [super init];
    process_directories = NO;
    nelements = 0;
    element = NULL;
    return self;
}

- free
{
    while (nelements > 0) {
	nelements--;
	[element[nelements] free];
    }

    free(element);
    element = NULL;

    return [super free];
}

- (BOOL) addToList:(Sebum *)child
{
    void *new_elements;
    assert(child);

    new_elements = realloc(element, (nelements+1) * sizeof(Sebum *));
    if (not new_elements)
	return NO;

    element = new_elements;
    element[nelements] = child;
    nelements++;
    return YES;
}

- (BOOL) file:(char *)filename IsInCurrentDirectory:(const char *)current_dir
{
    char *slashie;
    unsigned int l;
    assert(filename && current_dir);

    l = strlen(current_dir);
    if (l >= strlen(filename))
	return NO;

    if (strncmp(filename, current_dir, l) != 0)
	return NO;

    filename = filename + l;
    slashie = strchr(filename, '/');
    if (slashie && (*(slashie+1) != '\0'))
	return NO;
    else
	return YES;
}

- (BOOL) loadContentsOfDirectory:(const char *)top_dir :(const char *)sub_dir WithDirList:(DirectoryList *)dirlist
{
    unsigned int ent;
    assert(top_dir && sub_dir && dirlist);

    for (ent = 0; ent < [dirlist numEntries]; ent++) {
	BOOL is_file, is_directory;
	dirent_t *entry = [dirlist getNthEntry:ent];

	/* This entry might have been loaded already (by metafiles). */
	if ([dirlist entryIsLoaded:ent] || not entry)
	    continue;

	/* If this ent is not in the current directory, skip it. */
	if (not [self file:entry->d_name IsInCurrentDirectory:sub_dir])
	    continue;

	/* Mark this entry as loaded. */
	if (seb_container_filter(entry->d_name + strlen(sub_dir)) == 0) {
	    [dirlist markEntryLoaded:ent];
	    continue;
	}

	is_file = [dirlist isFile:ent];
	is_directory = (process_directories) ? [dirlist isDirectory:ent] : NO;

	if (is_file) {
	    const char *ext;
	    char name_sans_ext[PATH_MAX];
	    char dir_plus_name[PATH_MAX];

	    {				/* Get filename extension. */ 
		unsigned int n;
		char *file_start = strrchr(entry->d_name, '/');
		const char *dot = strrchr(entry->d_name, '.');
		if (not dot)
		    goto try_directory;
		if (not file_start)
		    file_start = entry->d_name;
		else
		    file_start++;
		ext = dot+1;

		/* Get name without extension. */
		delimited_strdup(name_sans_ext, file_start, dot, sizeof name_sans_ext);

		/* Get directory + name. */ 
		n = snprintf(dir_plus_name, sizeof dir_plus_name, "%s%s", top_dir, entry->d_name);
		assert(n < sizeof dir_plus_name);
	    }

	    {				/* Determine file type. */
		int i;
		for (i = 0; i < MAX_FILETYPE_HANDLERS && filename_extension_handlers[i].proc; i++) {
		    if (strcaseeq(ext, filename_extension_handlers[i].ext)) {
			filename_extension_handlers[i].proc(dir_plus_name, name_sans_ext, ent, self, dirlist);
			is_directory = NO;
			break;
		    }
		}
		if (i >= MAX_FILETYPE_HANDLERS ||
		    not filename_extension_handlers[i].proc) {
		    fprintf(stderr, "Don't know what to do with %s\n", dir_plus_name);
		}
	    }
	}

    try_directory:
	if (is_directory) {
	    Sebum<SebContainer> *child;
	    char *type = strrchr(entry->d_name, '=');

	    if (type) {
		char new_name[1024];

		delimited_strdup(new_name, entry->d_name + strlen(sub_dir), type, sizeof new_name);
		type++;

		if (streq(type, "anim/"))
		    child = [SebAnimation alloc];
		else if (streq(type, "lump/"))
		    child = [SebLump alloc];
		else {
		    fprintf(stderr, "[%s] Unknown type (%s) for directory\n", [[self class] name], type);
		    return NO;
		}

		[child initWithName:new_name];
	    }
	    else {
		char new_name[1024];
		strncpy(new_name, entry->d_name + strlen(sub_dir), sizeof new_name);
		new_name[strlen(new_name) - 1] = '\0';
		child = [[SebFile alloc] initWithName:new_name];
	    }

	    if (not [child startLoadingSebumDirectory:top_dir :entry->d_name WithDirList:dirlist] ||
		not [self addToList:child]) {
		[child free];
		return NO;
	    }
	}
    }

    return YES;
}

- (BOOL) startLoadingSebumDirectory:(const char *)top_dir :(const char *)sub_dir WithDirList:(DirectoryList *)dirlist
{
    assert(top_dir && sub_dir);

    SEBORRHEA_MESSAGE(stdout, "[%s] Opening %s%s\n", [[self class] name], top_dir, sub_dir);

    return [self loadContentsOfDirectory:top_dir :sub_dir WithDirList:dirlist];
}

#if 0
- (void) listContents:(unsigned int)depth
{
    unsigned int i, j;

    for (i = 0; i < nelements; i++) {
	for (j = 0; j < depth; j++)
	    printf(" ");

	printf("%s\n", [element[i] name]);

	if ([element[i] isKindOf:[AbstractSebContainer class]])
	    [(AbstractSebContainer *)element[i] listContents:depth+1];
    }
}
#endif

- (BOOL) loadSebumDirectory:(const char *)dirname
{
    int i;
    assert(dirname);

    SEBORRHEA_MESSAGE(stdout, "[%s] Loading %s\n", [[self class] name], dirname);

    for (i = 0; seborrhea_data_root_directories[i]; i++) {
	DirectoryList *dirlist;
	char work[PATH_MAX];

	/* Try opening as a directory. */
	snprintf(work, sizeof work, "%s%s/", seborrhea_data_root_directories[i], dirname);
	dirlist = [RecursiveDirectoryList new];
	if (dirlist) {
	    if ([dirlist scanDirectory:work] &&
		[self startLoadingSebumDirectory:work :"" WithDirList:dirlist]) {
		[dirlist free];
		/*[self listContents:0];*/
                SEBORRHEA_MESSAGE(stdout, "[%s] Using %s.\n", [[self class] name], work);
		return YES;
	    }
	    dirlist = [dirlist free];
	}

#ifndef NO_ZZIP
	/* Try opening as a zip. */
        char work2[PATH_MAX];
	snprintf(work2, sizeof work2, "%s%s", seborrhea_data_root_directories[i], dirname);
	dirlist = [ZzipDirectoryList new];
	if (dirlist) {
	    if ([dirlist scanDirectory:work2] &&
		[self startLoadingSebumDirectory:work :"" WithDirList:dirlist]) {
		[dirlist free];
		/*[self listContents:0];*/
                SEBORRHEA_MESSAGE(stdout, "[%s] Using %s (zip).\n", [[self class] name], work);
		return YES;
	    }
	    dirlist = [dirlist free];
	}
#endif
    }

    SEBORRHEA_MESSAGE(stdout, "[%s] Failed to load %s\n", [[self class] name], dirname);
    [self free];
    return NO;
}
@end
