/* toothpaste.m,
 *
 * Toothpaste projectiles go straight if no enemy is locked, or
 * follows a spline to the locked enemy if one is locked.  A unit is
 * locked if another toothpaste hits something.
 */

#include <allegro.h>
#include <assert.h>
#include <math.h>
#include "candy/spark.h"
#include "common.h"
#include "linklist.h"
#include "map.h"
#include "player.h"
#include "projectiles/toothpaste.h"
#include "seborrhea/seborrhea.h"
#include "sound.h"
#include "unit.h"
#include "unit-seeking.h"


#define LEVEL0_DAMAGE		30
#define LEVEL2_DAMAGE		20	/* note: 3 fired. */
#define TOTAL_FLAVOURS		2
#define TOOTHPASTE_LENGTH	30
#define TOOTHPASTE_STEPS	25


static Sebum<SebSample> *fire_sample;
static Unit *target[MAX_PLAYERS];

/* Different set of blenders in case of Snow level, because it has a
   bright background, causing 'addition' blenders hard to see. */
int toothpaste_flavour;

/* toothpaste_colour[mode][flavour][player] */
static int toothpaste_colour[2][TOTAL_FLAVOURS][MAX_PLAYERS];
static int toothpaste_lock_shots[MAX_PLAYERS];

/*--------------------------------------------------------------*/

@interface Toothpaste (Private)
+ (void) initializeColours;
- (int) setNormalBlender;
- (int) setSeekingBlender;
- (BOOL) findClosestUnitWithinJumpRange;
- (void) calculateLockSpline:(int)steps;
@end

@interface ToothpasteHigh: Toothpaste
@end

/*--------------------------------------------------------------*/

@implementation Toothpaste
+ initialize
{
    fire_sample = [base_sebum getSebumByName:"weapon/paste-snd"];
    [self initializeColours];
    return self;
}

- init
{
    [super init];

    health = LEVEL0_DAMAGE;
    angle = M_PI_2;
    speed = 10.0;
    w = 3;
    h = 3;

    length = TOOTHPASTE_LENGTH;
    jump_range = 100;

    min = deg2rad(80.0);
    max = deg2rad(100.0);

    path = -1;		/* negative path means don't follow a path. */
    the_path = NULL;

    min_travel_dist = 5.0;

    return self;
}

- free
{
    FREE_SPLINE(the_path);
    return [super free];
}

- (id) setMinAngle:(double)theta_min maxAngle:(double)theta_max
{
    min = theta_min;
    max = theta_max;
    return self;
}

- (void) die
{
    spawn_candy([Spark class], x, y, MEDIUM_LAYER);
    health = 0;

    /* If no target locked, find closest target within jump_range. */
    if (not target[parent_id] &&
	[self findClosestUnitWithinJumpRange]) {
	path = -1;
	path_max = 10;
    }
}

static void thick_line(BITMAP *dest, int x1, int y1, int x2, int y2, int c)
{
    int i;

    for (i = -1; i <= 1; i++)
	line(dest, x1+i, y1, x2+i, y2, c);
}

- (void) draw:(BITMAP *)dest
{
    drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);

    if (path < 0) {
	int c = [self setNormalBlender];
        thick_line(dest, x, y - offsetY,
		   x + length*cos(angle),
		   y - length*sin(angle) - offsetY, c);
    }
    else {
	int i;
	int c = [self setSeekingBlender];

	for (i = MAX(0, path-2); (i <= path+3) && (i < path_max); i++) {
	    thick_line(dest,
		       the_path[i].x,    the_path[i].y-offsetY, 
		       the_path[path].x, the_path[path].y-offsetY, c);
	}
    }

    drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
}

- (BOOL) needNewTarget
{
    if (not target[parent_id])
	return YES;

    if ([target[parent_id] flags] & FLAG_DYING) {
	target[parent_id] = nil;
	return YES;
    }

    return NO;
}

- (enum THING_UPDATE_STATE) update
{
    if ((health < 0) ||
	(health == 0 && path >= 0))
	return THING_DEAD;
    else if (path < 0)
        return [super update];

    /* Reached the end of the spline, make a new path. */
    else if (path >= path_max-1) {
#if 0
	/* 0) shorten our path, or die. */
	path_max -= 4;
	if (path_max <= 1)
	    return THING_DEAD;
#else
	/* 0) Increase our minimum travel distance. */
	min_travel_dist += 5.0;
	if (min_travel_dist >= 5.0+5.0*4)
	    return THING_DEAD;
#endif

	/* 1) make sure our target is alive.  If not, find a new
              target. */
	if ([self needNewTarget]) {
	    if (not [self findClosestUnitWithinJumpRange])
		return THING_DEAD;
	}

	/* 2) make the new path. */
	[self calculateLockSpline:path_max];
    }

    else {
	double delta_x = 0.0, delta_y = 0.0;

	while ((SQ(delta_x) + SQ(delta_y) < SQ(min_travel_dist)) && (path < path_max-1)) {
	    path++;    
	    delta_x = the_path[path].x - x;
	    delta_y = the_path[path].y - y;
	}
	x = the_path[path].x;
	y = the_path[path].y;
    }

    return THING_NORMAL;
}

- (id) setParentID:(int)n
{
    parent_id = n;

    if (target[parent_id])
	[self calculateLockSpline :TOOTHPASTE_STEPS];

    return self;
}

+ (enum DAMAGE_TYPE) damageType
{
    return DAMAGE_TYPE_TOOTHPASTE;
 }

- (int) collisionRoutinePriority
{
    /* Higher priority than normal units, lower than nukes (although
       it shouldn't matter). */
    return 80;
}

/* XXX: Special collision routine. */

- (BOOL) collidesWith:(Thing<DetectsCollision> *)object
{
    /* Using 'collidesWith' instead of 'checkCollisionWith' is correct
       because we want the toothpaste locking effect (ir-)regardless
       of who did the collision detection. */

    if ([super collidesWith:object]) {
	/* If no target locked, the hitee is the new target. */
	if (path < 0) {//|| not target[parent_id]) {
	    target[parent_id] = (Unit *)object;
	    toothpaste_lock_shots[parent_id] = 5;
	}

	return YES;
    }
    else
	return NO;
}
@end


@implementation Toothpaste (Private)
+ (void) initializeColours
{
    /* toothpaste_colour[mode][flavour][player] */
    toothpaste_colour[0][0][0] = makecol(0xff, 0xa0, 0xff);
    toothpaste_colour[0][0][1] = makecol(0xff, 0xff, 0xa0);
    toothpaste_colour[0][1][0] = makecol(0xc0, 0x80, 0xc0);
    toothpaste_colour[0][1][1] = makecol(0xc0, 0xa0, 0x70);
    toothpaste_colour[1][0][0] = makecol(0x80, 0x40, 0x80); 
    toothpaste_colour[1][0][1] = makecol(0x80, 0x80, 0x40);
    toothpaste_colour[1][1][0] = makecol(0xc0, 0x70, 0xc0); 
    toothpaste_colour[1][1][1] = makecol(0xc0, 0xa0, 0x60); 
}

- (int) setNormalBlender
{
    if (toothpaste_flavour == 0) {
	set_add_blender(0x40, 0x40, 0x40, 0x40);
	return toothpaste_colour[0][0][parent_id];
    }
    else {
	set_trans_blender(0x80, 0x80, 0x80, 0x60);
	return toothpaste_colour[0][1][parent_id];
    }
}

- (int) setSeekingBlender
{
    if (toothpaste_flavour == 0) {
	set_add_blender(0x10, 0x10, 0x10, 0x20);
	return toothpaste_colour[1][0][parent_id];
    }
    else {
	set_trans_blender(0x80, 0x80, 0x80, 0x30);
	return toothpaste_colour[1][1][parent_id];
    }
}

- (BOOL) findClosestUnitWithinJumpRange
{
    Unit *new_target = find_closest_unit(x, y, ACTIVE_ENEMY_LISTS);
    double x_, y_;

    if (not new_target)
	goto no_target;

    [new_target getX:&x_ Y:&y_];
    if (SQ(x-x_) + SQ(y-y_) > SQ(jump_range))
	goto no_target;

    angle = angle_towards_unit(x, y, new_target);
    path = -1;
    target[parent_id] = new_target;
    return YES;

 no_target:
    target[parent_id] = nil;
    return NO;
}

- (void) calculateLockSpline:(int)steps
{
    double dest_x, dest_y, delta_x, delta_y, l, theta;
    double cp1x, cp1y, cp2x, cp2y;

    [target[parent_id] getX:&dest_x Y:&dest_y];

    /* Stupid splines! */
    delta_x = dest_x - x;
    delta_y = y - dest_y;
    l = sqrt(SQ(delta_x) + SQ(delta_y)) * 0.90;

    theta = MID(min, atan2(delta_y, delta_x), max);
    cp1x = +l*cos(theta);
    cp1y = -l*sin(theta);
    cp2x = (dest_x < x) ? -30 : 30;
    cp2y = -15;

    {
        List *list = [LIST_NEW];
        ControlPoint *cp1 = [ControlPoint new];
        ControlPoint *cp2 = [ControlPoint new];

        [[cp1 setX:x Y:y steps:steps] setControlPoint:1 X:cp1x Y:cp1y];
        [[cp2 setX:dest_x Y:dest_y]   setControlPoint:0 X:cp2x Y:cp2y];

        [list insertItem:cp2];
        [list insertItem:cp1];

	/* Don't need to allocate memory if the_path already exists,
	   so just use reticulate_spline.  Note that reticulate_spline
	   does not free any memory. */
	if (the_path)
	    the_path = reticulate_spline(the_path, list);
	else
	    the_path = articulate_spline(list);

        [list free];
    }

    path = 0;
    path_max = steps;
}
@end


@implementation ToothpasteHigh
- init
{
    [super init];
    health = LEVEL2_DAMAGE;
    length = TOOTHPASTE_LENGTH*2.0;
    jump_range = 125;
    return self;
}
@end

/*------------------------------------------------------------*/

void toothpaste_set_flavour(int flav)
{
    assert(0 <= flav && flav < TOTAL_FLAVOURS);
    toothpaste_flavour = flav;
}

void unlockToothpaste(int pid)
{
    assert(pid < MAX_PLAYERS);
    target[pid] = nil;
}

void unlockToothpaste2(const Unit *unit)
{
    int pid;

    for (pid = 0; pid < MAX_PLAYERS; pid++) {
	if (target[pid] == unit)
	    target[pid] = nil;
    }
}

/*------------------------------------------------------------*/

void fireToothpaste(double x, double y, primary_weapon_t *w, const int pid)
{
#define SPAWN_ALLY_PROJ(class,x,y)		(Toothpaste *)spawn_projectile(class,x,y,YES)

    /* Note: setParentID *must* be at the end due to laziness. */

    y -= 10;
    if (w->level == 0) {
        [SPAWN_ALLY_PROJ([Toothpaste class], x, y) setParentID:pid];
        w->tics = 7;
    }
    elif (w->level == 1) {
	Class proj = [ToothpasteHigh class];
        [[SPAWN_ALLY_PROJ(proj, x-16, y) setMinAngle:deg2rad(90.0) maxAngle:deg2rad(110.0)] setParentID:pid];
	[[SPAWN_ALLY_PROJ(proj, x+16, y) setMinAngle:deg2rad(70.0) maxAngle:deg2rad(90.0)]  setParentID:pid];
        w->tics = 8;
    }
    else {
	Class proj = [ToothpasteHigh class];
        [SPAWN_ALLY_PROJ(proj, x, y) setParentID:pid];
	[[SPAWN_ALLY_PROJ(proj, x-4, y) setMinAngle:deg2rad(-60.0)  maxAngle:deg2rad(-40.0)]  setParentID:pid];
        [[SPAWN_ALLY_PROJ(proj, x+4, y) setMinAngle:deg2rad(-120.0) maxAngle:deg2rad(-140.0)] setParentID:pid];
        w->tics = 10;
    }
    
    play_voice_on_channels(fire_sample, x, 192, PRIMARY_WEAPON_CHANNELS);

    if (toothpaste_lock_shots[pid] <= 0) {
	unlockToothpaste(pid);
    }
    else {
	toothpaste_lock_shots[pid]--;
    }

#undef SPAWN_ALLY_PROJ
}

