/* railgun.m,
 *
 * Railgun used by Miniboss0 (tank).  Near instantaneous.  You have 4
 * tics to dodge the shot after it shoots.  Good luck!
 */

#include <allegro.h>
#include <math.h>
#include "colour-conv.h"
#include "common.h"
#include "map.h"
#include "projectiles/railgun.h"
#include "seborrhea/seborrhea.h"
#include "seborrhea/seborrhea-allegro.h" /* XXX */


static SebFile *proj_data;
static SebAnimation *trail_anim;


static inline void swap(double *alpha, double *beta)
{
    double gamma = *alpha;
    *alpha = *beta, *beta = gamma;
}


@implementation Railgun
+ derive_registerProjectile();

+ (BOOL) loadData
{
    if (not [self loadData:&proj_data :"data/proj-railgun"])
	return NO;

    trail_anim = (SebAnimation *)[proj_data getSebumByName:"trail"];
    return YES;
}

+ (void) shutdown
{
    FREE_SEBFILE(proj_data);
    trail_anim = nil;
}

- init
{
    [super init];
    health = 5;			/* 4 tics before actually zapping. */
    drawn = NO;

    return self; 
}

- (enum THING_UPDATE_STATE) update
{
    if (health > 1) {
	health--;
	return THING_NORMAL;
    }
    if (health != 0 && not drawn)
	return THING_NORMAL;
    return THING_DEAD;
}

- (void) draw:(BITMAP *)dest
{
    (void)dest;

    /* Only draw once. */
    if (not drawn) {
	int i;
	double delta_x, delta_y, xx, yy;

	delta_x = 14*cos(angle);
	delta_y = 14*sin(angle);
	xx = x;
	yy = y;

	/* Spawn some railgun trails. */
	for (i = 0; i < 33; i++) {
	    RailgunTrail *t = [spawn_candy([RailgunTrail class], xx, yy, MEDIUM_LAYER)
					  setAngle:angle];
	    [t setYV:trail_velocity];

	    xx += delta_x;
	    yy -= delta_y;
	}

	drawn = YES;
    }
}

- (int) collisionRoutinePriority
{
    /* Higher priority than normal units, lower than nukes (although
       it shouldn't matter). */
    return 90;
}

- (BOOL) checkCollisionWith:(Thing<DetectsCollision> *)object
{
    double x2, y2, thetamin, thetamax;
    int w2, h2;

    /* A few tics delay before actually hurting the player. */
    if (health > 1)
	return NO;

    [object getX:&x2 Y:&y2 W:&w2 H:&h2];

    thetamin = atan2(y - y2, (x2 - w2/2 - 10) - x);
    thetamax = atan2(y - y2, (x2 + w2/2 + 10) - x);

    if (thetamin > thetamax)
	swap(&thetamin, &thetamax);

    if (angle >= thetamin && angle <= thetamax)
	return YES;
    else
	return NO;
}

- (void) setRailgunTrailVelocity:(double)v_
{
    trail_velocity = v_;
}
@end

/*------------------------------------------------------------*/

@implementation RailgunTrail
- init
{
    [super init];

    speed = 0;
    frame = 1;
    sprite = [[SebAnimator new] setAnimation:trail_anim];

    return self;
}

- (enum THING_UPDATE_STATE) update
{
    if ([(SebAnimator *)sprite animationEnded])
	return THING_DEAD;

    return [super update];
}

- (void) draw:(BITMAP *)dest
{
    [sprite drawTo:dest X:x-offsetX Y:y-offsetY Angle:angle];
}

- (void) setYV:(double)v_
{
    yv = v_;
    recalculate_velocity = NO;
}
@end
