/* homing-missile.m,
 *
 * Homing missiles are ally projectiles that lock onto the nearest
 * enemy that it thinks it can hit.
 */

#include <math.h>
#include "candy/ir-line.h"
#include "common.h"
#include "projectiles/homing-missile.h"
#include "rotate.h"
#include "seborrhea/seborrhea.h"
#include "unit-seeking.h"

#ifdef DEBUG_HOMING_MISSILE_DEAD_MEAT_TRACKER
#include <allegro.h>
#include "map.h"
#endif


#define LEVEL0_DAMAGE		35
#define LEVEL1_DAMAGE		30
#define LEVEL2_DAMAGE		25
#define HOMING_MISSILE_SPEED	4.5
#define MAX_ROTATION_ANGLE	deg2rad(3.0)
#define TRACKING_DELAY		20
#define TURNING_RADIUS		(HOMING_MISSILE_SPEED/MAX_ROTATION_ANGLE)

/*--------------------------------------------------------------*/

@interface HomingMissileMedium: HomingMissile
@end

@interface HomingMissileHigh: HomingMissile
@end

/*--------------------------------------------------------------*/

@implementation HomingMissile
- init
{
    [super init];

    health = LEVEL0_DAMAGE;
    speed = HOMING_MISSILE_SPEED;

    sprite = [base_sebum getSebumByName:"weapon/homing"];
    w = 20;	//[sprite width];
    h = 9;	//[sprite height];
    rotated_sprite = YES;

    drop_time = TRACKING_DELAY;
    target = nil;

    return self;
}

- (enum THING_UPDATE_STATE) update
{
    if (drop_time > 0)
        drop_time--;
    else
	[spawn_candy([IRLine class], x, y, MEDIUM_LAYER) setAngle:angle];

    if (not target) {
        target = find_closest_unit_with_some_genius(x, y, angle, TURNING_RADIUS, ACTIVE_ENEMY_LISTS);
    }
    else {
	double desired_angle = angle_towards_unit(x, y, target);
        limited_rotate(&angle, desired_angle, MAX_ROTATION_ANGLE);
	simplify_angle(&angle);
	recalculate_velocity = YES;
    }

    return [super update];
}

- (void) unlock:(const Unit *)unit
{
    if (target == unit)
	target = nil;
}

#ifdef DEBUG_HOMING_MISSILE_DEAD_MEAT_TRACKER
- (void) draw:(BITMAP *)dest
{
    [super draw:dest];

    if (target && unit_exists(target, ACTIVE_ENEMY_LISTS)) {
	double x_, y_;
	[target getX:&x_ Y:&y_];

	textprintf_centre(dest, font, x-offsetX, y-offsetY, makecol(0x00, 0x00, 0xff), "%02g", sqrt(SQ(x_-x) + SQ(y_-y)));
	line(dest, x-offsetX, y-offsetY, x_-offsetX, y_-offsetY, makecol(0xff, 0x00, 0x00));
    }

    /* Turning radiuses.  VERY large. */
    circle(dest, 
	   x-offsetX + TURNING_RADIUS*sin(angle), 
	   y-offsetY + TURNING_RADIUS*cos(angle),
	   TURNING_RADIUS, makecol(0x80, 0x00, 0x00));
    circle(dest, 
	   x-offsetX - TURNING_RADIUS*sin(angle), 
	   y-offsetY - TURNING_RADIUS*cos(angle),
	   TURNING_RADIUS, makecol(0x80, 0x00, 0x00));
}
#endif
@end


@implementation HomingMissileMedium
- init
{
    [super init];
    health = LEVEL1_DAMAGE;
    return self;
}
@end


@implementation HomingMissileHigh
- init
{
    [super init];
    health = LEVEL2_DAMAGE;
    return self;
}
@end

/*--------------------------------------------------------------*/

void fireHomingMissile(double x, double y, secondary_weapon_t *w, const int pid)
{
#define SPAWN_ALLY_PROJ(p,x,y,theta)	[[(AllyProjectile *)spawn_projectile([p class],x,y,YES) setAngle:theta] setParentID:pid]

    y -= 10;
    if (w->level == 0) {
        SPAWN_ALLY_PROJ(HomingMissile, x-10, y, deg2rad(100.0));
        SPAWN_ALLY_PROJ(HomingMissile, x+10, y, deg2rad(80.0));
    } 
    elif (w->level == 1) {
        SPAWN_ALLY_PROJ(HomingMissileMedium, x-10, y, deg2rad(110.0));
        SPAWN_ALLY_PROJ(HomingMissileMedium, x   , y, deg2rad(90.0));
        SPAWN_ALLY_PROJ(HomingMissileMedium, x+10, y, deg2rad(70.0));
    }
    else {
        SPAWN_ALLY_PROJ(HomingMissileHigh, x-10, y, deg2rad(120.0));
        SPAWN_ALLY_PROJ(HomingMissileHigh, x- 3, y, deg2rad(100.0));
        SPAWN_ALLY_PROJ(HomingMissileHigh, x+ 3, y, deg2rad(80.0));
        SPAWN_ALLY_PROJ(HomingMissileHigh, x+10, y, deg2rad(60.0));
    }

    w->tics = 50;

#undef SPAWN_ALLY_PROJ
}
