/* Projectile.m
 *
 * Projectiles collide with units and nukes, but not other
 * projectiles.
 */

#include <allegro.h>
#include <assert.h>
#include <math.h>
#include "candy/spark.h"
#include "collision.h"
#include "common.h"
#include "linklist.h"
#include "map.h"
#include "nuke.h"
#include "player.h"
#include "projectile.h"
#include "projectiles/toothpaste.h"
#include "unit-intern.h"
#include "unit.h"


static List *projectile_list;


@implementation Projectile
        /* Initialization. */
+ (BOOL) loadData:(SebFile **)sebum :(const char *)dir
{
    *sebum = [SebFile new];
    return [(*sebum) loadSebumDirectory:dir];
}

- init
{
    [super init];

    sprite = (Sebum<SebImage> *)[base_sebum getSebumByName:"dummy8x8"];
    w = [sprite width];
    h = [sprite height];
    rotated_sprite = NO;

    angle = -M_PI_2;
    speed = 2.0;
    recalculate_velocity = YES;

    alliance = NO;

    return self;
}

- (id) setAlliance:(BOOL)good { alliance = good; return self; }

- (id) setAngle:(double)theta
{
    angle = theta;
    recalculate_velocity = YES;
    return self;
}

- (void) die
{
    /* This function is called when projectiles hit an enemy.  It does
       not actually remove the projectile from the lists - that is
       done by update projectile. */

    while (health-- > 0)
	spawn_candy([Spark class], x, y, MEDIUM_LAYER);
}

	/* Drawing. */
- (void) draw:(BITMAP *)dest
{
    if (rotated_sprite)
	[sprite drawTo:dest X:x-offsetX Y:y-offsetY Angle:angle];
    else
	[super draw:dest];
}

	/* Update. */
- (enum THING_UPDATE_STATE) update
{
    /* The angle was changed, recalculate xv/yv. */
    if (recalculate_velocity) {
	recalculate_velocity = NO;
	xv = speed * cos(angle);
	yv = speed * sin(angle);
    }

    x += xv;
    y -= yv;

    return [super update];
}

	/* Collisions. */
+ (int) collisionLists { return ALLY_LIST|COLLIDES_WITH_NUKES; }
+ (enum DAMAGE_TYPE) damageType { return DAMAGE_TYPE_PROJECTILE; }

- (int) receiveDamage:(int)damage
{
    damage = health;
    [self die];
    return damage;
}

- (int) damage { return MAX(0, health); }
- (int) parentID { return -1; }

- (void) unlock:(const Unit *)_
{
    (void)_;
}
@end

/*--------------------------------------------------------------*/

@implementation AllyProjectile
+ (int) collisionLists { return ACTIVE_ENEMY_LISTS; }
- (id) setParentID:(int)n { parent_id = n; return self; }
- (int) parentID { return parent_id; }
@end

/*--------------------------------------------------------------*/

void draw_projectiles(BITMAP *dest)
{
    ListIterator *it;

    foreach (it, projectile_list)
	[[it getItem] draw:dest];
}

static BOOL collide_projectile_with_nuke(Projectile *proj)
{
    Nuke *nukular = nuke_in_contact_with(proj);

    if (nukular) {
	[nukular receiveDamage:[proj damage]];
	[proj die];
	return YES;
    }
    return NO;    
}

static BOOL collide_projectile_with_unit_in_list(Projectile *proj, List *list)
{
    ListIterator *it;

    foreach (it, list) {
	Unit *unit = [it getItem];
	int damage;

	if ([unit collisionLists] & COLLIDES_WITH_PROJECTILES) {
	    if ([proj collidesWith:unit]) {
		damage = [unit receiveDamage:[proj damage] type:[[proj class] damageType]
		      from:[proj parentID]];
		[proj receiveDamage:damage];
		return YES;
	    }
	}
    }
    return NO;
}

void update_projectiles(void)
{
    ListIterator *it, *nx;

    /* Ally projectiles collision with enemies. */
    foreach_nx (it, nx, projectile_list) {
	Projectile *proj = [it getItem];
	int collision_lists = [[proj class] collisionLists];
	nx = [it next];

	if (collision_lists & COLLIDES_WITH_NUKES &&
	    collide_projectile_with_nuke(proj));
	else if (collision_lists & ACTIVE_AIR_LIST &&
		 collide_projectile_with_unit_in_list(proj, active_air_list));
	else if (collision_lists & ACTIVE_GROUND_LIST &&
		 collide_projectile_with_unit_in_list(proj, active_ground_list));
	else if (collision_lists & ALLY_LIST)
	    collide_projectile_with_unit_in_list(proj, ally_list);

	if ([proj update] == THING_DEAD) {
	    [projectile_list removeItem:proj];
	    [proj free];
	}
    }
}

void projectiles_unlock_unit(const Unit *unit)
{
    ListIterator *it;

    foreach (it, projectile_list)
	[[it getItem] unlock:unit];

    unlockToothpaste2(unit);
}

Projectile *spawn_projectile(Class class, double x, double y, BOOL is_goodie)
{
    Projectile *proj = [class new];

    if (proj) {
        [[proj setX:x Y:y] setAlliance:is_goodie];
	[projectile_list insertItem:proj];
    }

    return proj;
}

/*--------------------------------------------------------------*/

void projectile_init(void)
{
    projectile_list = [LIST_NEW];
    assert(projectile_list);
}

void projectile_shutdown(void)
{
    if (projectile_list)
	projectile_list = [projectile_list free];

    mark_all_projectile_data_unnecessary();
}
