/* player.m,
 *
 * This is you, dummy!  After some time, I got lazy and started
 * relying on pid0 == player1, and pid1 == player2.
 */

#include <allegro.h>
#include <assert.h>
#include <math.h>
#include "candy/explosion.h"
#include "candy/large-chunk.h"
#include "candy/player-zoom.h"
#include "candy/spark.h"
#include "collision.h"
#include "common.h"
#include "demo.h"
#include "difficulty.h"
#include "hiscore.h"
#include "input.h"
#include "map.h"
#include "nuke.h"
#include "player-stats.h"
#include "player.h"
#include "pregen-circles.h"
#include "projectile.h"
#include "projectiles/homing-missile.h"
#include "projectiles/laser.h"
#include "projectiles/missile.h"
#include "projectiles/pink.h"
#include "projectiles/toothpaste.h"
#include "projectiles/vulcan.h"
#include "seborrhea/seborrhea.h"
#include "unit.h"
#include "units/micronuke.h"
#include "units/robot.h"
#include "units/satellite.h"


#define foreach_orientation(i)		for (i = 0; i < NUM_ORIENTATIONS; i++)
#define NUM_ORIENTATIONS	3

typedef enum ORIENTATION {
    ORIENTATION_CENTRE = 0,
    ORIENTATION_LEFT,
    ORIENTATION_RIGHT
} orientation_t;

Player *player[MAX_PLAYERS];
unsigned int num_players = 1;

static SebAnimation *body_anim[MAX_PLAYERS][NUM_ORIENTATIONS];
static SebAnimation *fire_anim_a[MAX_PLAYERS][NUM_ORIENTATIONS];
static SebAnimation *fire_anim_b[MAX_PLAYERS][NUM_ORIENTATIONS];
static Sebum<SebImage> *the_shadow[NUM_ORIENTATIONS];
static Sebum<SebImage> *status_bg;
static Sebum<SebFont> *status_font;
static Sebum<SebFont> *digital_font;

/*--------------------------------------------------------------*/
/* Playblast weapon selection. */

primary_weapon_t playblast_primary_weapon = {
    PRIMARY_WEAPON_VULCAN, 0, 0
};

secondary_weapon_t playblast_secondary_weapon = {
    SECONDARY_WEAPON_NONE, 0, 0, 0
};

enum TERTIARY_WEAPON playblast_tertiary_weapon = TERTIARY_WEAPON_NONE;

/*--------------------------------------------------------------*/

@interface Player (Private)
- (void) receiveRobots;
- (void) receiveSatellites;

- (void) setNormalAnimations;
- (void) setFiringAnimations;

- (void) drawStatusHealth:(BITMAP *)dest :(int)delta_x :(int)delta_y;
- (void) drawStatusPrimary:(BITMAP *)dest :(int)delta_x :(int)delta_y;
- (void) drawStatusSecondary:(BITMAP *)dest :(int)delta_x :(int)delta_y;
@end

/*--------------------------------------------------------------*/

@implementation Player
+ initialize
{
    /* This is supposed to be in initialize, not loadData. */
    body_anim[0][ORIENTATION_CENTRE]	= (SebAnimation *)[base_sebum getSebumByName:"player/p1-centre"];
    body_anim[0][ORIENTATION_LEFT]	= (SebAnimation *)[base_sebum getSebumByName:"player/p1-left"];
    body_anim[0][ORIENTATION_RIGHT]	= (SebAnimation *)[base_sebum getSebumByName:"player/p1-right"];
    fire_anim_a[0][ORIENTATION_CENTRE]	= (SebAnimation *)[base_sebum getSebumByName:"player/fa1-centre"];
    fire_anim_a[0][ORIENTATION_LEFT]	= (SebAnimation *)[base_sebum getSebumByName:"player/fa1-left"];
    fire_anim_a[0][ORIENTATION_RIGHT]	= (SebAnimation *)[base_sebum getSebumByName:"player/fa1-right"];
    fire_anim_b[0][ORIENTATION_CENTRE]	= (SebAnimation *)[base_sebum getSebumByName:"player/fb1-centre"];
    fire_anim_b[0][ORIENTATION_LEFT]	= (SebAnimation *)[base_sebum getSebumByName:"player/fb1-left"];
    fire_anim_b[0][ORIENTATION_RIGHT]	= (SebAnimation *)[base_sebum getSebumByName:"player/fb1-right"];

    the_shadow[ORIENTATION_CENTRE] = [base_sebum getSebumByName:"player/shadow-c"];
    the_shadow[ORIENTATION_LEFT]   = [base_sebum getSebumByName:"player/shadow-l"];
    the_shadow[ORIENTATION_RIGHT]  = [base_sebum getSebumByName:"player/shadow-r"];

    status_bg = [base_sebum getSebumByName:"glue/statusbar-bg"];
    status_font = [base_sebum getSebumByName:"fonts/nano10"];
    digital_font = [base_sebum getSebumByName:"fonts/digital2"];

    return self;
}

+ (void) shutdown
{
    orientation_t i;

    foreach_orientation (i) {
	body_anim[0][i]   = nil;
	fire_anim_a[0][i] = nil;
	fire_anim_b[0][i] = nil;
	the_shadow[i]	  = nil;
    }

    status_bg = nil;
    status_font = nil;
    digital_font = nil;
}

- init
{
    [super init];

    sprite = [SebAnimator new];
    [self setNormalAnimations];
    animation = anims[ORIENTATION_CENTRE];
    [(SebAnimator *)sprite setAnimation:animation];
    shadow = the_shadow[ORIENTATION_CENTRE];

    x = screen_w / 2;
    y = screen_h - 50;
    w = 8;
    h = 7;
    health = (num_players == 1) ? 10 : 7;

/*     primary_weapon.weapon	= PRIMARY_WEAPON_VULCAN; */
/*     secondary_weapon.weapon	= SECONDARY_WEAPON_NONE; */
/*     tertiary_weapon.weapon	= TERTIARY_WEAPON_NONE; */
    primary_weapon		= playblast_primary_weapon;
    primary_weapon.tics		= 0;
    secondary_weapon		= playblast_secondary_weapon;
    secondary_weapon.tics	= 0;
    tertiary_weapon.weapon	= playblast_tertiary_weapon;
    tertiary_weapon.children	= [LIST_NEW];

    switch (playblast_tertiary_weapon) {
      case TERTIARY_WEAPON_NONE: break;
      case TERTIARY_WEAPON_ROBOTS:
	  [self receiveRobots];
	  break;
      case TERTIARY_WEAPON_SATELLITE2:
	  tertiary_weapon.weapon = TERTIARY_WEAPON_SATELLITE;
	  [self receiveSatellites];
	  /* Fall through. */
      case TERTIARY_WEAPON_SATELLITE:
	  [self receiveSatellites];
	  break;
    }

    pid = 0;
    chunk_colours = CHUNK_BLUE;

    return self;
}

- free
{
    /* Check for a high score. */
    if (is_new_high_score(player_stats[pid].score)) {
	if (pid == 0) game_flags |= FLAG_PLAYER1_HIGH_SCORE;
	if (pid == 1) game_flags |= FLAG_PLAYER2_HIGH_SCORE;
    }

    if (tertiary_weapon.children) {
	/* We don't want to destroy units twice so unlink from this list. */
	[tertiary_weapon.children removeAllItems];
	tertiary_weapon.children = [tertiary_weapon.children free];
    }

    return [super free];
}

- (void) spawnDyingExplosions
{
    game_flags |= FLAG_SCREEN_SHAKE;

    health--;
    if (health == 0)			/* Just in case. */
	health = -1;

    /* Spit out come chunks, gradually get more. */
    if (rnd(health, 0) < -50) {
	if (chunk_colours & CHUNK_BLUE)
	    spawn_candy([LargeChunkBlue class], x, y, HIGH_LAYER);
	if (chunk_colours & CHUNK_COFFEE)
	    spawn_candy([LargeChunkCoffee class], x, y, HIGH_LAYER);
    }

    /* You have 3 seconds to grab a health powerup before you die! */
    if (health > - 3*SECONDS) {
	/* Pretend that the explosions are getting bigger by spawning
	   more small explosions earlier and bigger ones later. */
	if (rnd(health, 0) < -30)
	    spawn_candy([BigExplosion class], x, y, HIGH_LAYER);
	else
	    spawn_candy([Explosion class], x, y, HIGH_LAYER);
    }
    else if (not (game_flags & FLAG_LEVEL_COMPLETE)) {
	/* If you completed the level, you cheat death!
	   Congratulations.  Otherwise, die. */
	[self delete];
    }

    if (health > -150 + 15)
	flash_tics = 10;
    else if (flash_tics > 0) {
	/* Hack to make sure status bar is not displaced when we die. */
	flash_tics--;
    }
}

- (void) spawnNearDeathSparks
{
    if (rnd(0, 2) == 0) {
	spawn_candy([Spark class],
		    rnd(x-10, x+10), rnd(y-10, y+10), HIGH_LAYER);
    }
}

#ifdef WINFF
- (void) die
{
    [super die];

    if ((demo_state != DEMO_PLAYBACK) &&
	(demo_state != DEMO_PLAYBACK_AUTOMATIC))
	rumble_pad_for_player_dying(pid);
}
#endif

- (void) delete
{
    ListIterator *it, *nx;

    /* Free the robots and satellites. */
    foreach_nx (it, nx, tertiary_weapon.children) {
	Unit<PlayerChild> *unit = [it getItem];
	nx = [it next];
	[unit die];
    }

    [super delete];
}

- (void) draw:(BITMAP *)dest
{
    [super draw:dest];

    if (flags & FLAG_DYING) {
	int alpha = 0x60 * -health/150.0;
	int radius = 150;
	int colour = makecol(0xff, 0xb0, 0xb0);
	pregen_circlefill_add(dest, x-offsetX, y-offsetY, radius*2/3, colour, alpha);
	pregen_circlefill_add(dest, x-offsetX, y-offsetY, radius, colour, alpha);
    }
    else if (flash_tics > 0) {
	int alpha = MIN(0x10 * flash_tics, 0x40) / 3;
	int radius = 150 - 5 * ABS(10 - flash_tics);
	int colour = makecol(0xff, 0xb0, 0xb0);
	pregen_circlefill_add(dest, x-offsetX, y-offsetY, radius*2/3, colour, alpha);
	pregen_circlefill_add(dest, x-offsetX, y-offsetY, radius, colour, alpha);
    }
}

- (void) drawHealth:(BITMAP *)dest
{
    int delta_x, delta_y, c, shake_dist;

    shake_dist = MIN(flash_tics, 5);

    delta_x = rnd(-shake_dist, shake_dist);
    delta_y = rnd(-shake_dist, shake_dist);
    c  = makecol(0x40, 0x40, 0x40);

    /* Backdrop */
    [status_bg drawTo:dest X:0 Y:0 W:dest->w H:dest->h];

    [self drawStatusHealth:dest :delta_x :delta_y];
    [self drawStatusPrimary:dest :delta_x :delta_y];
    [self drawStatusSecondary:dest :delta_x :delta_y];

    [status_font putString:"SCORE" To:dest X:5+delta_x Y:150+delta_y Colour:0x40:0x40:0x40];
    [digital_font putStringTo:dest X:dest->w-15+delta_x Y:160+delta_y Colour:0x40:0x80:0x20 Alignment:ALIGN_RIGHT :"%d", player_stats[pid].score];
}

- (enum THING_UPDATE_STATE) update
{
    int w_ = [sprite width]/2 - 10;
    int h_ = 21;//[sprite height]/2;

    if (not (game_flags & FLAG_LEVEL_COMPLETE)) {
	/* Make sure the player doesn't scroll with the map, and keep
	   them on the screen. */
	y += [current_map scrollRate];
	x = MID(w_ + offsetX, x, screen_w - w_ + offsetX);
	y = MID(h_ + offsetY, y, screen_h - h_ + offsetY);
    }

    if (invulnerable_tics > 0) {
	if (--invulnerable_tics <= 0)
	    flags &=~FLAG_INVINCIBLE;
    }
    else if (flags & FLAG_DEAD) {	/* Death (rides a horse, Binky). */
	[spawn_nuke(x, y, -1) setNukeRadius:100.0];
	return THING_DEAD;
    }
    else if (health <= 0) {		/* Dying. */
	flags |= (FLAG_DYING|FLAG_INVINCIBLE);

	x += xv;
	y += yv;
	//xv *= 1.25;
	//yv *= 1.25;

	[self spawnDyingExplosions];
    }
    else if (health <= 3)		/* Low health warning. */
	[self spawnNearDeathSparks];

    /* Weapons and firing. */
    if (primary_weapon.tics > 0)
	primary_weapon.tics--;
    if (secondary_weapon.tics > 0)
	secondary_weapon.tics--;

    /* Slight penalty for holding down fire key. */
    if (fire_tics <= -15) {
        fire_was_down = NO;
    }
    else {
        fire_tics--;
        if (fire_tics > 0)
            [self fire];
    }

    if (fire_tics < 0)
	unlockToothpaste(pid);

    if ([(SebAnimator *)sprite animationEnded])
	[self setNormalAnimations];

    return THING_NORMAL;
}

- (void) increaseVelocity:(double)xv_ :(double)yv_
{
    orientation_t orientation;

    if (flags & FLAG_DYING) {
	xv = MID(-3.0, xv + xv_/3.0, 3.0);
	yv = MID(-3.0, yv + yv_/3.0, 3.0);
    }

    /* As XOP guy pointed out, rock hard movement gives better control. */
    else {
	x += xv_;
	y += yv_;
    }

    /* -1.0 < xv < 1.0, have centre animation. */
    if (xv_ < -1.0)
	orientation = ORIENTATION_LEFT;
    else if (xv_ > 1.0)
	orientation = ORIENTATION_RIGHT;
    else
        orientation = ORIENTATION_CENTRE;

    if (animation != anims[orientation]) {
	shadow = the_shadow[orientation];
	animation = anims[orientation];
	[(SebAnimator *)sprite setAnimation:animation];
    }
}

- (void) zoom
{
    /* Used by end-level sequence. */
    spawn_candy([PlayerZoom class], x, y, MEDIUM_LAYER);

    yv -= 0.2;
    y += yv;

    flags |= FLAG_INVINCIBLE;
}

- (void) openFire
{
    if (not fire_was_down) {
        fire_was_down = YES;
        fire_tics = 15;
    }
}

- (void) haltFire
{
    fire_was_down = NO;
}

- (void) fire
{    
    /* Primary weapon. */
    if (primary_weapon.tics <= 0) {
	if (primary_weapon.weapon == PRIMARY_WEAPON_LASER)
            fireLaser(x, y, &primary_weapon, pid);
	elif (primary_weapon.weapon == PRIMARY_WEAPON_TOOTHPASTE)
            fireToothpaste(x, y, &primary_weapon, pid);
	else
	    fireVulcan(x, y, &primary_weapon, pid);

	[self setFiringAnimations];
    }

    /* Secondary weapon. */
    if (secondary_weapon.tics <= 0) {
	if (secondary_weapon.weapon == SECONDARY_WEAPON_HOMING_MISSILE)
	    fireHomingMissile(x, y, &secondary_weapon, pid);
	elif (secondary_weapon.weapon == SECONDARY_WEAPON_MISSILE)
	    fireMissile(x, y, &secondary_weapon, pid);
	elif (secondary_weapon.weapon == SECONDARY_WEAPON_PINK_ORANGE_RED)
	    firePinkOrangeRed(x, y, &secondary_weapon, pid);
    }

    /* Do nothing for robots.  They will query us to determine whether
       or not to fire. */
}

- (BOOL) isFiring { return (fire_tics > 0) ? YES : NO; }

- (void) setPrimaryWeaponTo:(const enum PRIMARY_WEAPON)new_weapon
{
    if (new_weapon == primary_weapon.weapon) {
	/* Increase current weapon level, or spawn a mini-nuke. */
	if (primary_weapon.level == 2)
	    [spawn_nuke(x, y, pid) setNukeRadius:100];
	else
	    primary_weapon.level++;
    }
    else
	primary_weapon.weapon = new_weapon;

    record_demo_checkpoint = YES;
}

- (void) setSecondaryWeaponTo:(const enum SECONDARY_WEAPON)new_weapon
{
    if (new_weapon == secondary_weapon.weapon) {
	/* Increase current weapon level, or spawn a mini-nuke. */
	if (secondary_weapon.level == 2)
	    [spawn_nuke(x, y, pid) setNukeRadius:100];
	else
	    secondary_weapon.level++;
    }
    else
	secondary_weapon.weapon = new_weapon;

    record_demo_checkpoint = YES;
}

- (void) setTertiaryWeaponTo:(const enum TERTIARY_WEAPON)new_weapon
{
    tertiary_weapon.weapon = new_weapon;

    if (new_weapon == TERTIARY_WEAPON_ROBOTS)
	[self receiveRobots];
    else {
	[self receiveSatellites];
#ifdef CHEAT
	[self receiveSatellites];
	[self receiveSatellites];
#endif
    }

    record_demo_checkpoint = YES;
}

- (enum PRIMARY_WEAPON) weaponType
{
    return primary_weapon.weapon;
}

- (void) increaseScore:(int)n
{
    player_stats[pid].score += n * score_multiplier[difficulty];
}

- (void) incrementKillsFrom:(double)x_ :(double)y_
{
    stats_t *s = &player_stats[pid];
    int i;
    double dist_sq;
    
    s->kills++;
    dist_sq = SQ(x_-x) + SQ(y_-y);

    /* -10 is a small shift towards the short end. */
    i = NUM_KILL_RANGES_BUCKETS * (sqrt(dist_sq)-10) / (screen_h * 7/8);
    i = MID(0, i, NUM_KILL_RANGES_BUCKETS-1);
    s->kill_bucket[i]++;
}

- (void) disownChild:(Unit *)unit
{
    [tertiary_weapon.children removeItem:unit];
}

- (int) playerID { return pid; }

- (int) receiveDamage:(int)damage type:(enum DAMAGE_TYPE)type
{
    if (type == DAMAGE_TYPE_UNIT) {
	player_stats[pid].crashes++;
	player_stats[pid].hits++;
    }
    else if (damage > 0)
	player_stats[pid].hits++;

    if ((damage > 0 && flags & FLAG_INVINCIBLE) || 
	(damage > 0 && flags & FLAG_DYING) ||
	(flags & FLAG_DEAD))
	return 0;

    if (damage > 0) {			/* Was hurt. */
	health--;
	flash_tics = MIN(flash_tics+9, 25);
        if (health <= 0)
            [self die];
	else {
	    /* Make a MicroNuke to clear away nearby projectiles. */
	    spawn_unit([MicroNuke class], x, y, ALLY_LIST, YES);

	    flags |= FLAG_INVINCIBLE;
	    invulnerable_tics = 15;
	}

#ifdef WINFF
	if ((demo_state != DEMO_PLAYBACK) &&
	    (demo_state != DEMO_PLAYBACK_AUTOMATIC))
	    rumble_pad_for_player_hurt(pid);
#endif
    }
    else {				/* Health powerup. */
	if (flags & FLAG_DYING) {
	    health = -damage;
	    flash_tics = 0;
	    player_stats[pid].cheated_death_count++;
	    flags &=~FLAG_DYING;
	}
	else {
	    health -= damage;

	    /* Max out at 15 health.  Anything more causes a nuke. */
	    if (health > 15) {
		health = 15;
		[spawn_nuke(x, y, pid) setNukeRadius:100.0];
	    }
	}

	/* Picking up health also makes you invincible for a bit. */
	flags |= FLAG_INVINCIBLE;
	invulnerable_tics = 15;
    }

    return MIN(damage, 1);
}

- (int) collisionLists { return ACTIVE_AIR_LIST|COLLIDES_WITH_PROJECTILES; }

	/* Dummy to conform to OwnsChildren protocol. */
- (void) destroyChildren {}

- (BOOL) checkCollisionWith:(Thing<DetectsCollision> *)object
{
    /* Allow collisions even when dying. */
    double x1, y1, x2, y2;
    int w1, h1, w2, h2;

    [self   getX:&x1 Y:&y1 W:&w1 H:&h1];
    [object getX:&x2 Y:&y2 W:&w2 H:&h2];

    return bounding_box_collision(x1-w1/2.0, y1-h1/2.0, x1+w1/2.0, y1+h1/2.0,
				  x2-w2/2.0, y2-h2/2.0, x2+w2/2.0, y2+h2/2.0);
}
@end


@implementation Player (Demo)
- (void) recordDemoCheckPoint:(PACKFILE *)fp
{
    /* Health, location. */
    pack_putc(health, fp);
    pack_fwrite(&x, sizeof x, fp);
    pack_fwrite(&y, sizeof y, fp);

    /* Primary/secondary weapons. */
    pack_putc(primary_weapon.weapon,	fp);
    pack_putc(primary_weapon.level,	fp);
    pack_putc(secondary_weapon.weapon,	fp);
    pack_putc(secondary_weapon.level,	fp);
}

- (void) readDemoCheckPoint:(MEMORY_FILE *)fp
{
    /* Health, location. */
    health = (char)mmpk_getc(fp);
    mmpk_fread(&x, sizeof x, fp);
    mmpk_fread(&y, sizeof y, fp);

    /* Primary/secondary weapons. */
    primary_weapon.weapon	= (char)mmpk_getc(fp);
    primary_weapon.level	= mmpk_getc(fp);
    secondary_weapon.weapon	= (char)mmpk_getc(fp);
    secondary_weapon.level	= mmpk_getc(fp);
}
@end


@implementation Player (Private)
- (void) receiveRobots
{
#ifdef CHEAT
    int delta_x[8] = { 30, -30,  40, -40,  20, -20,  10, -10 };
    int delta_y[8] = {  5,   5,  15,  15,  -5,  -5, -15, -15 };
    int i = 7;
#else
    int delta_x[2] = { 30, -30 };
    int delta_y[2] = {  5,   5 };
    int i = 1;
#endif

    ListIterator *it, *nx;
    Unit<PlayerChild> *unit;

    /* Release all children, regardless of class. */
    foreach_nx (it, nx, tertiary_weapon.children) {
	unit = [it getItem];
	nx = [it next];
	[unit die];
    }

    for (; i >= 0; i--) {
	unit = (Robot *)spawn_unit([Robot class], x, y, ALLY_LIST, YES);
	/* Make the robots come from the bottom of the screen. */
	[[[unit setParent:self] setX:x Y:y + screen_h]
	    setDisplacementX:delta_x[i] Y:delta_y[i]];
	[tertiary_weapon.children insertItem:unit];
    }
}

- (void) receiveSatellites
{
    ListIterator *it, *nx;
    Unit<PlayerChild> *unit;
    int n = 2;

    /* foreach_nx because units may unlink from children list. */
    foreach_nx (it, nx, tertiary_weapon.children) {
	unit = [it getItem];
	nx = [it next];

	[unit exciteMeIgniteMe];
    }

    while (n > 0) {
	unit = (Shield *)spawn_unit([Shield class], x, y, ALLY_LIST, YES);
	[[unit setParent:self] setAngle:deg2rad(180.0) * n];
	[tertiary_weapon.children insertItem:unit];
	n--;
    }
}

- (void) setNormalAnimations
{
    orientation_t i;

    foreach_orientation (i) {
	if (animation == anims[i]) {
	    animation = body_anim[0][i];
	    sprite = [(SebAnimator *)sprite setAnimation:animation];
	}
	anims[i] = body_anim[0][i];
    }
}

- (void) setFiringAnimations
{
    BOOL use_animation_a = YES;
    orientation_t i;

    /* Special animation for level 2 laser and toothpaste. */
    if ((primary_weapon.weapon != PRIMARY_WEAPON_VULCAN) &&
	(primary_weapon.level == 1))
	use_animation_a = NO;

    foreach_orientation (i) {
	if (animation == anims[i]) {
	    animation = use_animation_a ? fire_anim_a[0][i] : fire_anim_b[0][i];
	    sprite = [(SebAnimator *)sprite setAnimation:animation];
	}

	anims[i] = use_animation_a ? fire_anim_a[0][i] : fire_anim_b[0][i];
    }
}

- (void) drawStatusHealth:(BITMAP *)dest :(int)delta_x :(int)delta_y
{
    Sebum<SebImage> *block = [base_sebum getSebumByName:"glue/health"];
    int i, xx, yy, hh;

    [status_font putString:"HEALTH" To:dest X:5+delta_x Y:10+delta_y Colour:0x40:0x40:0x40];
    hh = MIN(health, 15);

    for (yy = 20; hh > 0; yy += 10, hh -= 5) {
	for (xx = 10, i = (hh > 5 ? 5 : hh); i > 0; xx += 10, i--) {
	    [block drawTo:dest X:xx+2+delta_x Y:yy+2+delta_y];
	}
    }
}

- (void) drawStatusPrimary:(BITMAP *)dest :(int)delta_x :(int)delta_y
{
    const char *primary_weapon_level_sprite[NUM_PRIMARY_WEAPONS] = {
	"glue/level-l", "glue/level-t", "glue/level-v"
    };

    const char *blockname = primary_weapon_level_sprite[primary_weapon.weapon];
    Sebum<SebImage> *block = [base_sebum getSebumByName:blockname];
    int i, xx;

    [status_font putString:"PRIMARY" To:dest X:5+delta_x Y:60+delta_y Colour:0x40:0x40:0x40];

    for (i = 0, xx = 10; i < primary_weapon.level+1; i++, xx += 21) {
	[block drawTo:dest X:xx+2+delta_x Y:70+2+delta_y];
    }
}

- (void) drawStatusSecondary:(BITMAP *)dest :(int)delta_x :(int)delta_y
{
    const char *secondary_weapon_level_sprite[NUM_SECONDARY_WEAPONS] = {
	"glue/level-h", "glue/level-m", "glue/level-p"
    };

    const char *blockname;
    Sebum<SebImage> *block;
    int i, xx;

    [status_font putString:"SECONDARY" To:dest X:5+delta_x Y:90+delta_y Colour:0x40:0x40:0x40];

    if (secondary_weapon.weapon == SECONDARY_WEAPON_NONE)
	return;

    blockname = secondary_weapon_level_sprite[secondary_weapon.weapon];
    block = [base_sebum getSebumByName:blockname];

    for (i = 0, xx = 10; i < secondary_weapon.level+1; i++, xx += 21) {
	[block drawTo:dest X:xx+2+delta_x Y:100+2+delta_y];
    }
}
@end

/*--------------------------------------------------------------*/

@implementation Player2
+ initialize
{
    /* This is supposed to be in initialize, not loadData. */
    body_anim[1][ORIENTATION_CENTRE]	= (SebAnimation *)[base_sebum getSebumByName:"player/p2-centre"];
    body_anim[1][ORIENTATION_LEFT]	= (SebAnimation *)[base_sebum getSebumByName:"player/p2-left"];
    body_anim[1][ORIENTATION_RIGHT]	= (SebAnimation *)[base_sebum getSebumByName:"player/p2-right"];
    fire_anim_a[1][ORIENTATION_CENTRE]	= (SebAnimation *)[base_sebum getSebumByName:"player/fa2-centre"];
    fire_anim_a[1][ORIENTATION_LEFT]	= (SebAnimation *)[base_sebum getSebumByName:"player/fa2-left"];
    fire_anim_a[1][ORIENTATION_RIGHT]	= (SebAnimation *)[base_sebum getSebumByName:"player/fa2-right"];
    fire_anim_b[1][ORIENTATION_CENTRE]	= (SebAnimation *)[base_sebum getSebumByName:"player/fb2-centre"];
    fire_anim_b[1][ORIENTATION_LEFT]	= (SebAnimation *)[base_sebum getSebumByName:"player/fb2-left"];
    fire_anim_b[1][ORIENTATION_RIGHT]	= (SebAnimation *)[base_sebum getSebumByName:"player/fb2-right"];

    return self;
}

+ (void) shutdown
{
    orientation_t i;

    foreach_orientation (i) {
	body_anim[1][i]   = nil;
	fire_anim_a[1][i] = nil;
	fire_anim_b[1][i] = nil;
    }
}

- init
{
    [super init];
    chunk_colours = CHUNK_COFFEE;
    pid = 1;
    return self;
}
@end


@implementation Player2 (Private)
- (void) setNormalAnimations
{
    orientation_t i;

    foreach_orientation (i) {
	if (animation == anims[i]) {
	    animation = body_anim[1][i];
	    sprite = [(SebAnimator *)sprite setAnimation:animation];
	}
	anims[i] = body_anim[1][i];
    }
}

- (void) setFiringAnimations
{
    BOOL use_animation_a = YES;
    orientation_t i;

    /* Special animation for level 2 laser and toothpaste. */
    if ((primary_weapon.weapon != PRIMARY_WEAPON_VULCAN) &&
	(primary_weapon.level == 1))
	use_animation_a = NO;

    foreach_orientation (i) {
	if (animation == anims[i]) {
	    animation = use_animation_a ? fire_anim_a[1][i] : fire_anim_b[1][i];
	    sprite = [(SebAnimator *)sprite setAnimation:animation];
	}
	anims[i] = use_animation_a ? fire_anim_a[1][i] : fire_anim_b[1][i];
    }
}
@end
