/* newmenu.m,
 *
 * New main menu since someone keeps whinging about the old one.
 *
 * menu_1: Press START
 * menu_2: Play/Options/High Scores/Credits/Quit
 * PlayMenu: Single, Two, Custom (1), Custom (2), Demo
 * OptionsMenu: Controls/Volume/etc.
 */

#include <allegro.h>
#include <assert.h>
#include <math.h>
#include "candy.h"
#include "common.h"
#include "demo.h"
#include "demo-info.h"
#include "difficulty.h"
#include "game.h"
#include "init.h"
#include "input.h"
#include "music.h"
#include "newmenu-controls.h"
#include "newmenu-credits.h"
#include "newmenu-custom.h"
#include "newmenu-demo.h"
#include "newmenu-dirty.h"
#include "newmenu-hiscore.h"
#include "newmenu-log.h"
#include "newmenu.h"
#include "player.h"
#include "projectiles/all-projs.h"
#include "seborrhea/seborrhea.h"
#include "sound.h"
#include "units/all-units.h"
#include "video.h"


#define NUM_SHADOW_OPACITY_LEVELS	3
#define DIFFICULTY_SLIDER		0
#define DEMO_CHECKBOX			1
#define CANDY_SLIDER			0
#define SHADOW_SLIDER			1
#define FULLSCREEN_CHECKBOX		2
#define SOUND_SLIDER			0
#define MUSIC_SLIDER			1
#define REVERSE_BUTTON			2
#define NUM_CAMPAIGN_LEVELS		3


static const int shadow_opacity_level[NUM_SHADOW_OPACITY_LEVELS] = {
    0x00, 0x20, 0x40
};

static const char *campaign_levels[] = {
    "maps/level1.map", "maps/level2.map", "maps/level3.map", NULL, NULL
};

static const char *idle_demos[] = {
    "demos/demo1.rec", "demos/demo2.rec", "demos/demo3.rec", NULL
};

static int campaign_levels_unlocked = 3;
static BITMAP *dbuf;
static menu_t *current_menu;
static BOOL exit_raid = NO;
static Sebum<SebFont> *menu_font;
static Sebum<SebImage> *bg;

static enum {
    SHOW_HIGH_SCORES,
    SHOW_CREDITS,
    SHOW_DEMO
} idle_todo;

static int next_idle_demo = 0;

BOOL emergency_exit = NO;
SebFile *menu_sebum;

/*--------------------------------------------------------------*/

volatile int counter;
static volatile int idle_counter;
static void ticker(void) { counter++; idle_counter++; }

/*--------------------------------------------------------------*/

static void activate_menu_2(void);
static void activate_play_menu(void);
static void activate_options_menu(void);
static void do_high_scores(void);
static void do_credits(void);
static void do_quit(void);
static void do_play_single(void);
static void do_play_two(void);
static void do_play_custom(void);
static void do_play_demo(void);
static void do_begin_play(void);
static void toggle_difficulty(void);
static void toggle_demos(void);
static void activate_video_menu(void);
static void activate_audio_menu(void);
static void activate_customise_menu(void);
static void toggle_candy_level(void);
static void toggle_shadow_opacity(void);
static void toggle_screen_mode(void);
static void toggle_sound_level(void);
static void toggle_reverse_stereo(void);
static void do_customise_1(void);
static void do_customise_2(void);

static BOOL mainmenu_reload(void);
static void free_mainmenu_data(void);

/*--------------------------------------------------------------*/
/* The Menus.							*/
/*--------------------------------------------------------------*/

static menu_t menu_1 = {
    0, 1,
    { { "Press START", HOTKEY_MENU, 0, 0, 0, 0, 0, activate_menu_2 } },
};

static menu_t menu_2 = {
    0, 5,
    { { "Play",			0, 0, 0, 0, 0, 0, activate_play_menu },
      { "Options",		0, 0, 0, 0, 0, 0, activate_options_menu },
      { "High Scores",		0, 0, 0, 0, 0, 0, do_high_scores },
      { "Credits",		0, 0, 0, 0, 0, 0, do_credits },
      { "Quit",	      HOTKEY_MENU, 0, 0, 0, 0, 0, do_quit } }
};

static menu_t play_menu = {
    0, 5,
    { { "Single Player",	0, 0, 0, 0, 0, 0, do_play_single },
      { "Two Player",		0, 0, 0, 0, 0, 0, do_play_two },
      { "Custom",		0, 0, 0, 0, 0, 0, do_play_custom },
      { "Demo",			0, 0, 0, 0, 0, 0, do_play_demo },
      { "Back",	      HOTKEY_MENU, 0, 0, 0, 0, 0, activate_menu_2 } }
};

static menu_t pick_level_menu = {
    0, 5,
    { { "Part 1: The Desert",	0, 0, 0, 0, 0, 0, do_begin_play },
      { "Part 2: In Space",	0, 0, 0, 0, 0, 0, do_begin_play },
      { "Part 3: Arctic Adventure", 0, 0, 0, 0, 0, 0, do_begin_play },
      { "Part 4: Underground",	0, 0, 0, 0, 0, 0, do_begin_play },
      { "Back",	      HOTKEY_MENU, 0, 0, 0, 0, 0, activate_play_menu } }
};

static menu_t options_menu = {
    0, 6,
    { { "Difficulty",		0, 0, -(NUM_DIFFICULTY_LEVELS-1), 0, 0, 0, toggle_difficulty },
      { "Record Demos",		0, 0, 1, 0, 0, 0, toggle_demos },
      { "Video",		0, 0, 0, 0, 0, 0, activate_video_menu },
      { "Audio",		0, 0, 0, 0, 0, 0, activate_audio_menu },
      { "Customise Controls",	0, 0, 0, 0, 0, 0, activate_customise_menu },
      { "Back",	      HOTKEY_MENU, 0, 0, 0, 0, 0, activate_menu_2 } }
};

static menu_t video_menu = {
    0, 4,
    { { "Eye-candy",		0, 0, TOTAL_CANDY_LEVELS-1, 0, 0, 0, toggle_candy_level },
      { "Shadows",		0, 0, NUM_SHADOW_OPACITY_LEVELS-1, 0, 0, 0, toggle_shadow_opacity },
      { "Fullscreen",		0, 0, 1, 0, 0, 0, toggle_screen_mode },
      { "Back",	      HOTKEY_MENU, 0, 0, 0, 0, 0, activate_options_menu } }
};

static menu_t audio_menu = {
    0, 4,
    { { "Sound Volume",		0, 0, 20, 0, 0, 0, toggle_sound_level },
      { "Music Volume",		0, 0, 20, 0, 0, 0, toggle_sound_level },/* Same proc */
      { "Reverse Stereo",	0, 0, 1, 0, 0, 0, toggle_reverse_stereo },
      { "Back",	      HOTKEY_MENU, 0, 0, 0, 0, 0, activate_options_menu } }
};

static menu_t customise_menu = {
    0, 3,
    { { "Player 1",		0, 0, 0, 0, 0, 0, do_customise_1 },
      { "Player 2",		0, 0, 0, 0, 0, 0, do_customise_2 },
      { "Back",	      HOTKEY_MENU, 0, 0, 0, 0, 0, activate_options_menu } }
};

/*--------------------------------------------------------------*/
/* Useful tools, and for consistancy.				*/
/*--------------------------------------------------------------*/

void play_back_sound(void) { [(<SebSample>)[menu_sebum getSebumByName:"back"] playWithVolume:sound_vol/255.0]; }
void play_bing_sound(void) { [(<SebSample>)[menu_sebum getSebumByName:"bing"] playWithVolume:sound_vol/255.0]; }
void play_okay_sound(void) { [(<SebSample>)[menu_sebum getSebumByName:"okay"] playWithVolume:sound_vol/255.0]; }

static BOOL is_skippable(const menuitem_t* const m)
{
    return ((m->slider_max == 0) && (m->slider_val < 0));
}

BOOL menu_select_previous(void)
{
    if ([menu_keyboard_controls pressed:RAID_KEY_UP])
	return YES;

    if (joy_enabled) {
	int j;

	/* Check all joysticks (and hats). */
	for (j = 0; j < num_joysticks && joystick_controller[j]; j++) {
	    if ([joystick_controller[j] pressed:RAID_KEY_UP])
		return YES;
	}
    }

    return NO;
}

BOOL menu_select_next(void)
{
    if ([menu_keyboard_controls pressed:RAID_KEY_DOWN])
	return YES;

    if (joy_enabled) {
	int j;

	/* Check all joysticks (and hats). */
	for (j = 0; j < num_joysticks && joystick_controller[j]; j++) {
	    if ([joystick_controller[j] pressed:RAID_KEY_DOWN])
		return YES;
	}
    }

    return NO;
}

BOOL menu_select_left(void)
{
    if ([menu_keyboard_controls pressed:RAID_KEY_LEFT])
	return YES;

    if (joy_enabled) {
	int j;

	/* Check all joysticks (and hats). */
	for (j = 0; j < num_joysticks && joystick_controller[j]; j++) {
	    if ([joystick_controller[j] pressed:RAID_KEY_LEFT])
		return YES;
	}
    }

    return NO;
}

BOOL menu_select_right(void)
{
    if ([menu_keyboard_controls pressed:RAID_KEY_RIGHT])
	return YES;

    if (joy_enabled) {
	int j;

	/* Check all joysticks (and hats). */
	for (j = 0; j < num_joysticks && joystick_controller[j]; j++) {
	    if ([joystick_controller[j] pressed:RAID_KEY_RIGHT])
		return YES;
	}
    }

    return NO;
}

void selected_item_colour(int *r, int *g, int *b, int t)
{
    *r = 0xff;
    *g = 0xe0 - 0x1f*cos(deg2rad(t));
    *b = 0xc0 - 0x2f*cos(deg2rad(t));
}

static void switch_menu(menu_t *new)
{
    if (new == &menu_1) {
	/* Idle -> PRESS START. */
	menu_2.selected_item = 0;
	play_menu.selected_item = 0;
	options_menu.selected_item = 0;

	/* Options (sub-)menu -> PRESS START */
	//options_menu.selected_item = 0;
	bg = [menu_sebum getSebumByName:"backdrop"];
    }

    elif (new == &menu_2) {
	/* Play menu -> main menu. */
	if (current_menu == &play_menu)
	    play_menu.selected_item = 0;

	/* Options menu -> main menu. */
	elif (current_menu == &options_menu) {
	    options_menu.selected_item = 0;
	    bg = [menu_sebum getSebumByName:"backdrop"];
	}
    }

    elif (new == &pick_level_menu) {
	new->selected_item = 0;
    }

    elif (new == &options_menu) {
	/* Main menu -> options menu. */
	bg = [menu_sebum getSebumByName:"options"];
    }

    elif (new == &video_menu ||
	  new == &audio_menu ||
	  new == &customise_menu) {
	/* Options -> options' sub-menu. */
	new->selected_item = 0;
    }

    current_menu = new;
    mark_entire_screen_dirty();
}

/*--------------------------------------------------------------*/
/* Menu 1: Press START.						*/
/*--------------------------------------------------------------*/

static void activate_menu_2(void) { switch_menu(&menu_2); }

/*--------------------------------------------------------------*/
/* Menu 2: Play/Options/etc.					*/
/*--------------------------------------------------------------*/

static void activate_play_menu(void) { switch_menu(&play_menu); }
static void activate_options_menu(void) {switch_menu(&options_menu); }

static void do_high_scores(void)
{
    switch (roll_high_scores(dbuf)) {
      case EXIT_SCORES_INTERVENED:
	switch_menu(&menu_2);
	break;

      case EXIT_SCORES_ENDED:
	switch_menu(&menu_1);
	break;
    }
}

static void do_credits(void)
{
    switch (roll_credits(dbuf)) {
      case EXIT_CREDITS_INTERVENED:
	switch_menu(&menu_2);
	break;

      case EXIT_CREDITS_ENDED:
	switch_menu(&menu_1);
	break;
    }
}

static void do_quit(void) { exit_raid = YES; }

/*--------------------------------------------------------------*/
/* Play Menu.							*/
/*--------------------------------------------------------------*/

void play(const char *level_list[], unsigned int num_levels,
	  BOOL check_highscore, BOOL prefix_with_data_dir)
{
    free_mainmenu_data();
    free_music();
    remove_int(ticker);

    raid_start_levels(level_list, num_levels,
		      check_highscore, prefix_with_data_dir);

    mainmenu_reload();
}

static void activate_level_menu(void)
{
    /* If we haven't unlocked any levels yet, just play. */
    if (campaign_levels_unlocked <= 1)
	play(campaign_levels, NUM_CAMPAIGN_LEVELS, YES, YES);
    else {
	int i;

	for (i = 0; i < pick_level_menu.n_menuitems - 1; i++) {
	    pick_level_menu.menuitem[i].slider_val = 
		(i < campaign_levels_unlocked) ? 0 : -1;
	}



	switch_menu(&pick_level_menu);
    }
}

static void do_play_single(void) { num_players = 1; activate_level_menu(); }
static void do_play_two(void)    { num_players = 2; activate_level_menu(); }

static void do_begin_play(void)
{
    int start_lv = current_menu->selected_item;

    if (start_lv >= NUM_CAMPAIGN_LEVELS)
	start_lv = 0;

    demo_info_set_date(NULL);
    play(&(campaign_levels[start_lv]), NUM_CAMPAIGN_LEVELS-start_lv, YES, YES);
}

static void do_play_custom(void)
{
    char *level_list[1];
    level_list[0] = select_custom_level(dbuf);

    demo_info_set_date(NULL);
    if (level_list[0]) {
	play((const char **)level_list, 1, NO, NO);
	free(level_list[0]);
    }
}

static void do_play_demo(void)
{
    char *demo_name = select_demo(dbuf);

    if (!demo_name)
	return;

    if (open_demo_file_for_playback(demo_name)) {
	demo_state = DEMO_PLAYBACK;
	play(NULL, 99, NO, NO);	/* Hack. */
    }
}

/*--------------------------------------------------------------*/
/* Options Menu.						*/
/*--------------------------------------------------------------*/

static void toggle_difficulty(void)
{
    difficulty = options_menu.menuitem[DIFFICULTY_SLIDER].slider_val;
    options_menu.menuitem[DIFFICULTY_SLIDER].text = difficulty_text[difficulty];
}

static void toggle_demos(void) { record_demos = options_menu.menuitem[DEMO_CHECKBOX].slider_val; }
static void activate_video_menu(void) { switch_menu(&video_menu); }
static void activate_audio_menu(void) { switch_menu(&audio_menu); }
static void activate_customise_menu(void) { switch_menu(&customise_menu); }

static void toggle_candy_level(void)    { candy_amount   = video_menu.menuitem[CANDY_SLIDER].slider_val; }
static void toggle_shadow_opacity(void) { shadow_opacity = shadow_opacity_level[video_menu.menuitem[SHADOW_SLIDER].slider_val]; }

static void toggle_screen_mode(void)
{
    /* Unload all data in case we change colour depth. */
    mark_all_projectile_data_unnecessary();
    unload_unnecessary_projectile_data();
    mark_all_unit_data_unnecessary();
    unload_unnecessary_unit_data();
    video_bitmap_shutdown();

    if (fullscreen) {
	if (switch_gfx_mode(GFX_AUTODETECT_WINDOWED, SCREEN_W, SCREEN_H, 0) == 0)
	    fullscreen = NO;
	else
	    append_message("Error switching to windowed mode. %s.", allegro_error);
    }
    else {
	if (switch_gfx_mode(GFX_AUTODETECT_FULLSCREEN, SCREEN_W, SCREEN_H, 0) == 0)
	    fullscreen = YES;
	else
	    append_message("Error switching to fullscreen mode. %s.", allegro_error);
    }

    video_bitmap_init();
    mark_entire_screen_dirty(); 
}

static void toggle_sound_level(void)
{
    menuitem_t *sound_slider = &audio_menu.menuitem[SOUND_SLIDER];
    menuitem_t *music_slider = &audio_menu.menuitem[MUSIC_SLIDER];

    if ((sound_slider->slider_val == 0) &&
	(music_slider->slider_val == 0)) {
	sound_shutdown();
	music_shutdown();
    }
    else if (!sound_initialized) {
	if (sound_init() < 0) {
	    append_message("Error initializing sound. %s.", allegro_error);

	    sound_slider->slider_val = 0;
	    music_slider->slider_val = 0;
	    sound_vol = 0;
	    music_vol = 0;
	    return;
	}
	else
	    music_init();
    }

    sound_vol = 255 * sound_slider->slider_val/sound_slider->slider_max;
    music_vol = 255 * music_slider->slider_val/music_slider->slider_max;

    if (current_menu->selected_item == SOUND_SLIDER) {
	SebLump *lump = [base_sebum getSebumByName:"explosion-sounds"];
	[(<SebSample>)[lump getRandomSebum] playWithVolume:sound_vol/255.0];
    }
    else
	adjust_music_volume();
}

static void toggle_reverse_stereo(void)
{
    menuitem_t *rev_button = &audio_menu.menuitem[REVERSE_BUTTON];
    sound_reverse_stereo = (rev_button->slider_val == 0) ? NO : YES;
}

static void do_customise_1(void) { customise_controls_for_player(dbuf, 0); }
static void do_customise_2(void) { customise_controls_for_player(dbuf, 1); }

/*--------------------------------------------------------------*/
/* Init/Shutdown.						*/
/*--------------------------------------------------------------*/

static void switch_into_menu(void)
{
    mark_entire_screen_dirty(); 
}

static BOOL load_mainmenu_data(void)
{
    menu_sebum = [SebFile new];
    if (not [menu_sebum loadSebumDirectory:"data/menu"]) {
	menu_sebum = nil;
	fprintf(stderr, "Error loading menu datafiles.\n");
	return NO;
    }
    else {
	bg = [menu_sebum getSebumByName:"backdrop"];
	menu_font = [base_sebum getSebumByName:MENU_FONT_NAME];
/* 	[menu_font setFontSize:MENU_FONT_SIZE]; */
	return YES;
    }
}

static BOOL mainmenu_reload(void)
{
    install_int_ex(ticker, BPS_TO_TIMER(MENU_SPEED));

    if (not load_mainmenu_data())
	return NO;

    {					/* Fix the sliders. */
	menuitem_t *diff_slider   = &options_menu.menuitem[DIFFICULTY_SLIDER];
	menuitem_t *demo_checkbox = &options_menu.menuitem[DEMO_CHECKBOX];
	menuitem_t *candy_slider  = &video_menu.menuitem[CANDY_SLIDER];
	menuitem_t *shadow_slider = &video_menu.menuitem[SHADOW_SLIDER];
	menuitem_t *fs_checkbox	  = &video_menu.menuitem[FULLSCREEN_CHECKBOX];
	menuitem_t *sound_slider  = &audio_menu.menuitem[SOUND_SLIDER];
	menuitem_t *music_slider  = &audio_menu.menuitem[MUSIC_SLIDER];
	int i;

	diff_slider->slider_val = difficulty;
	toggle_difficulty();		/* Set the text. */
	demo_checkbox->slider_val = record_demos;

	/* Video. */
	candy_slider->slider_val = candy_amount;
	for (i = 0; i < NUM_SHADOW_OPACITY_LEVELS; i++) {
	    if (shadow_opacity == shadow_opacity_level[i]) {
		shadow_slider->slider_val = i;
		break;
	    }
	}
	fs_checkbox->slider_val = fullscreen;

	/* Audio. */
	sound_slider->slider_val = sound_slider->slider_max * sound_vol/255;
	music_slider->slider_val = music_slider->slider_max * music_vol/255;
    }

    set_display_switch_callback(SWITCH_IN, switch_into_menu);

    switch_menu(&menu_1);
    mark_entire_screen_dirty();
    play_music(MENU_MUSIC_DIRNAME);

    return YES;
}

static BOOL mainmenu_init(void)
{
    dbuf = create_bitmap(SCREEN_W, SCREEN_H);
    assert(dbuf);

    return mainmenu_reload();
}

static void free_mainmenu_data(void)
{
    FREE_SEBFILE(menu_sebum);
}

static void mainmenu_shutdown(void)
{
    free_mainmenu_data();
    FREE_BITMAP(dbuf);
    remove_int(ticker);
}

/*--------------------------------------------------------------*/

void menu_common_update(void)
{
    poll_raid_keys(0);
    poll_music();

    /* Alt-enter -> toggle full screen mode. */
    if ((key_shifts & KB_ALT_FLAG) &&
        [menu_keyboard_controls pressed:RAID_KEY_START]) {
	toggle_screen_mode();

	/* Note: if you want to be a tool and hold down alt-enter,
	   causing it to toggle screen mode, go ahead.  The problem
	   occurs (even tough I'm using 'pressed') because Allegro
	   clears key_shifts and key[].  These next lines are a hack
	   so enter released isn't detected by menu. */
	[menu_keyboard_controls poll];
	[menu_keyboard_controls poll];
    }
}

static BOOL menu_update(void)
{
    menuitem_t *sel = &current_menu->menuitem[current_menu->selected_item];
    int i;

    menu_common_update();

    if (menu_select_previous()) {
	if (current_menu->selected_item > 0) {
	    play_bing_sound();
	    do {
		current_menu->selected_item--;
	    } while (is_skippable(&current_menu->menuitem[current_menu->selected_item]));
	    return YES;
	}
    }

    if (menu_select_next()) {
	if (current_menu->selected_item < current_menu->n_menuitems-1) {
	    play_bing_sound();
	    do {
		current_menu->selected_item++;
	    } while (is_skippable(&current_menu->menuitem[current_menu->selected_item]));
	    return YES;
	}
    }

    if (sel->slider_max < 0 ||
	sel->slider_max > 1) {		/* Sliders */
	if (menu_select_left()) {
	    sel->slider_val = MAX(sel->slider_val-1, 0);
	    if (sel->action)
		sel->action();
	    return YES;
	}
	if (menu_select_right()) {
	    sel->slider_val = MIN(sel->slider_val+1, ABS(sel->slider_max));
	    if (sel->action)
		sel->action();
	    return YES;
	}
    }
    else {				/* Non-slider. */
	if (okay_released()) {
	    if (sel->slider_max == 1) {	/* Checkbox. */
		sel->slider_val = !sel->slider_val;
	    }

	    if (sel->action) {
		play_okay_sound();
		sel->action();
		mark_entire_screen_dirty();
		return YES;
	    }
	}
    }

    /* Hotkeys. */
    if (menu_released()) {
	for (i = 0; i < current_menu->n_menuitems; i++) {
	    if (current_menu->menuitem[i].hotkey == HOTKEY_MENU) {
		if (current_menu->selected_item != i)
		    current_menu->selected_item = i;
		else if (sel->action) {
		    play_back_sound();
		    sel->action();
		    mark_entire_screen_dirty();
		}
		return YES;
	    }
	}
    }

    if (idle_counter >= 8*MENU_SECONDS) {
	if (current_menu != &menu_1) {
	    /* Go back to start menu due to inactivity. */
	    switch_menu(&menu_1);
	}
	else {
	    /* Already at start menu, show credits, high scores or
	       demo instead. */
	    switch (idle_todo) {
	      case SHOW_HIGH_SCORES:
		idle_todo++;
		do_high_scores();
		break;

	      case SHOW_CREDITS:
		idle_todo++;
		do_credits();
		break;

	      case SHOW_DEMO:
		idle_todo = SHOW_HIGH_SCORES;
		if (open_demo_file_for_playback(idle_demos[next_idle_demo])) {
		    demo_state = DEMO_PLAYBACK_AUTOMATIC;
		    play(NULL, 99, NO, NO); /* Hack. */
		}

		next_idle_demo++;
		if (!idle_demos[next_idle_demo])
		    next_idle_demo = 0;
		break;
	    }

	    mark_entire_screen_dirty();
	}

	return YES;
    }

    return NO;
}

void do_newmenu(void)
{
    BOOL redraw = YES;
    int t = 0;

    if (not mainmenu_init())
	exit_raid = YES;
    else
	mark_entire_screen_dirty();

    while (not (exit_raid || emergency_exit)) {
	while (counter) {
	    counter--;
	    redraw = YES;

	    t += 12;
	    if (t >= 360)
		t = 0;

	    if (menu_update())
		idle_counter = 0;

	    update_log();
	    rest(0);
	}

	if (redraw) {
	    int i, y, h;

	    redraw = NO;
	    clear_to_color(dbuf, makecol(0x40,0x40,0x40)); /* XXX */
	    [bg drawTo:dbuf X:(dbuf->w - [bg width])/2 Y:(dbuf->h - [bg height])/2 W:dbuf->w H:dbuf->h];

	    y = 250;
	    h = [menu_font textHeight];
	    for (i = 0; i < current_menu->n_menuitems; i++, y += h) {
		menuitem_t *m = &current_menu->menuitem[i];
		int l;

		if (i == current_menu->selected_item) {
		    selected_item_colour(&m->r, &m->g, &m->b, t);
		}
		else {
		    int delta = ABS(i - current_menu->selected_item);
		    int r = MAX(0xf0 - delta*0x08, 0);
		    int g = MAX(0xa0 - delta*0x10, 0);
		    int b = MAX(0x30 - delta*0x08, 0);

		    m->r = (m->r + r) / 2;
		    m->g = (m->g + g) / 2;
		    m->b = (m->b + b) / 2;
		}


		if (is_skippable(m)) {
		    y -= h;
		    continue;
		}

		if (m->slider_max < 0)	/* Imaginary slider. */
		    l = [menu_font textLength:difficulty_text[DIFFICULTY_HARDEST]];
		else
		    l = [menu_font textLength:m->text];

		[menu_font putString:m->text To:dbuf X:dbuf->w/2 Y:y
			   Colour:m->r:m->g:m->b Alignment:ALIGN_CENTRE
			   Decoration:DECORATION_OUTLINE];

		/* Yes, yes.  I know the menu items don't always need
		   to be dirtied.  But I'm lazy. */
		mark_dirty_rectangle((dbuf->w-l)/2-1, y-1, l+2, h+2);

		if (m->slider_max == 1) { /* Check box. */
		    int x = dbuf->w * 88/100;

		    [menu_font putString:m->slider_val ? "Yes" : "No"
			       To:dbuf X:x Y:y Colour:m->r:m->g:m->b
			       Alignment:ALIGN_RIGHT
			       Decoration:DECORATION_OUTLINE];

		    mark_dirty_rectangle(x-100, y, 100, h);
		}

		if (m->slider_max > 1) { /* Slider. */
		    int x = dbuf->w * 72/100, w = 106;

		    rect(dbuf, x,   y+8, x+w-1, y+h-6, makecol(0x00, 0x00, 0x00));
		    rect(dbuf, x+1, y+9, x+w-2, y+h-7, makecol(m->r, m->g, m->b));

		    if (m->slider_val > 0) {
			rectfill(dbuf, x+3, y+11,
				 x+3 + (w-6)*m->slider_val/m->slider_max-1,
				 y+h-9, makecol(m->r, m->g, m->b));
		    }

		    mark_dirty_rectangle(x, y+8, w, h-8-6);
		}
	    }

	    /* XXX: don't use 'screen' when triple buffering. */
	    draw_log(dbuf);
	    blit_dirty_rectangles(dbuf, screen);
	}
    }

    mainmenu_shutdown();
}
