/* newmenu-fileselector.m,
 *
 * The file selector used by the custom game and play demo menus.
 */

#include <assert.h>
#include <allegro.h>
#include "common.h"
#include "input.h"
#include "newmenu.h"
#include "newmenu-dirty.h"
#include "newmenu-fileselector.h"

/*--------------------------------------------------------------*/

@interface MenuFileSelector (Private)
- (void) drawDecoration:(BITMAP *)dest;
- (void) drawCurrentDirectory:(BITMAP *)dest;
- (BOOL) moveSelection:(int)amount;
- (BOOL) updateMenuReleased;
- (void) updateFileListControls;
- (void) updateButtonControls;
- (const char *) selectItem;
@end

/*--------------------------------------------------------------*/

@implementation MenuFileSelector
static int directory_list_type_sort(const void *a, const void *b)
{
    const dirent_t *da = a;
    const dirent_t *db = b;

    if (da->d_type != db->d_type)
	return da->d_type - db->d_type;
    else
	return strcmp(da->d_name, db->d_name);
}

- init
{
    [super init];
    file_selection_side = YES;
    sorter = directory_list_type_sort;
    return self;
}

- (void) sort
{
    unsigned int i;

    [super sort];

    for (i = 0, selected_file = 0; i < nentries; i++) {
	/* Select a file when we enter a directory. */
	if ([self isFile:i]) {
	    selected_file = i;
	    break;
	}
    }
}

- (BOOL) updateOkayReleased:(const char **)selected_filename
{
    if (not file_selection_side &&
	selected_button == back_button) {
	return YES;
    }
    else {				/* Go! */
	(*selected_filename) = [self selectItem];
	
	if (*selected_filename)
	    return YES;
    }

    return NO;
}

- (BOOL) update:(const char **)selected_filename
{
    shown_offset_y = (shown_offset_y*4 + offset_y) / 5;

    if (menu_released()) {
	if ([self updateMenuReleased])
	    return YES;
    }

    if (okay_released()) {
	if ([self updateOkayReleased:selected_filename])
	    return YES;
    }

    if (file_selection_side)
	[self updateFileListControls];
    else
	[self updateButtonControls];

    if (menu_select_left() && not file_selection_side) {
	file_selection_side = YES;
	play_bing_sound();
    }
    if (menu_select_right() && file_selection_side) {
	file_selection_side = NO;
	play_bing_sound();
    }

    return NO;
}

- (void) draw:(BITMAP *)dest :(int)t
{
    unsigned int i, r, g, b;
    int y = shown_offset_y;
    int h = [font textHeight];

    [self drawDecoration:dest];

    set_clip(dest, 40, 0, dest->w-40, 60);
    [self drawCurrentDirectory:dest];

    set_clip(dest, 50, 65, 50+400, dest->h-35);
    for (i = 0, y += 65; i < nentries; i++, y += h) {
	BOOL is_directory, is_link;

	if (y+h < 65)
	    continue;
	if (y > dest->h-35)
	    break;

	if (i == selected_file)
	    selected_item_colour(&r, &g, &b, t);
	else if (file_selection_side) {
	    int delta = ABS((int)i - (int)selected_file);
	    r = MAX(0xff - delta*0x08, 0);
	    g = MAX(0xc0 - delta*0x10, 0);
	    b = MAX(0x60 - delta*0x10, 0);
	}
	else {
	    r = 0xff-5*0x08;
	    g = 0xc0-5*0x10;
	    b = 0x60-5*0x10;
	}

	is_directory = [self isDirectory:i];
	is_link = [self isLink:i];

	[font putStringTo:dest X:50 Y:y Colour:r:g:b
	      Decoration:DECORATION_OUTLINE
	      :"%s%s", entries[i].d_name, (is_link ? "@" :
					   is_directory ? "/" : "")];
    }
    set_clip(dest, 0, 0, dest->w, dest->h);
    mark_dirty_rectangle(50, 65, 400, dest->h-35-65+1);

    [self drawButtons:dest :t];
}

- (void) drawButtons:(BITMAP *)dest :(int)t
{
    (void)dest, (void)t;
}
@end


@implementation MenuFileSelector (Private)
- (void) drawDecoration:(BITMAP *)dest
{
    int black = makecol(0x00, 0x00, 0x00);
    int h = [font textHeight];

    drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);

    /* Current directory. */
    set_trans_blender(0xff, 0xff, 0xff, 0x60);
    rectfill(dest, 30, 60-h-8, dest->w-30, 60, black);

    /* File selection region. */
    if (file_selection_side)
	set_trans_blender(0xff, 0xff, 0xff, 0x40);
    rectfill(dest, 50, 65, 50+400, dest->h-35, black);

    drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
}

- (void) drawCurrentDirectory:(BITMAP *)dest
{
    int r = 0xff-5*0x08, g = 0xc0-5*0x10, b = 0x60-5*0x10;
    int l = [font textLength:working_directory];
    int h = [font textHeight];
    
    if (l > dest->w-40-40) {
	[font putString:working_directory To:dest X:dest->w-40 Y:60-h
	      Colour:r:g:b Alignment:ALIGN_RIGHT Decoration:DECORATION_OUTLINE];
    }
    else {
	[font putString:working_directory To:dest X:40 Y:60-h
	      Colour:r:g:b Decoration:DECORATION_OUTLINE];
    }
    mark_dirty_rectangle(40, 60-h, dest->w, h);
}

- (BOOL) moveSelection:(int)amount
{
    unsigned int h = [font textHeight];
    unsigned int old_selected_file = selected_file;

    if (amount < 0 && selected_file < ABS(amount))
	selected_file = 0;
    else {
	selected_file += amount;
	if (selected_file > nentries-1)
	    selected_file = nentries-1;
    }

    if (amount < 0) {
	if ((int)(selected_file*h) + offset_y < 0)
	    offset_y = -selected_file*h;
    }
    else {
	if (selected_file*h + offset_y > 13.0*h)
	    offset_y = 13.0*h - selected_file*h;
    }

    return (selected_file != old_selected_file);
}

- (BOOL) updateMenuReleased
{
    if (not file_selection_side &&
	selected_button == back_button) {
	return YES;
    }
    else {
	file_selection_side = NO;
	selected_button = back_button;
	return NO;
    }
}

- (void) updateFileListControls
{
    BOOL moved = NO;

    if (menu_select_previous() && [self moveSelection:-1])
	moved = YES;
    if (menu_select_next() && [self moveSelection:+1])
	moved = YES;
    if (key[KEY_PGUP] && [self moveSelection:-3])
	moved = YES;
    if (key[KEY_PGDN] && [self moveSelection:+3])
	moved = YES;

    if (moved)
	play_bing_sound();
}

- (void) updateButtonControls
{
    if (menu_select_previous() && selected_button > 0) {
	selected_button--;
	play_bing_sound();
    }

    if (menu_select_next() && selected_button < back_button) {
	selected_button++;
	play_bing_sound();
    }
}

- (const char *) selectItem
{
    BOOL is_directory;
    char selected_path[PATH_MAX];
    assert(selected_file < nentries);

    is_directory = [self isDirectory:selected_file];
    strncpy(selected_path, working_directory, sizeof selected_path);

    if (streq(entries[selected_file].d_name, "..")) {
	int l = strlen(selected_path);
	char *delim;

	/* Working_directory contains a / at the end.  Remove it. */
	if (l > 0) {
	    selected_path[l-1] = '\0';
	}

	delim = strrchr(selected_path, '/');
	if (not delim)
	    delim = selected_path;

	*delim = '\0';
    }
    else {
	strncat(selected_path, entries[selected_file].d_name, sizeof selected_path);
    }

    if (is_directory) {
	/* Add a '/'. */
	strncat(selected_path, "/", sizeof selected_path);

	[self freeEntries];
	[self scanDirectory:selected_path];
	selected_file = 0;
	offset_y = shown_offset_y = 0;
	return NULL;
    }
    else {
	return strdup(selected_path);
    }
}
@end
