/* music.m,
 *
 * This file handles music stuff.
 */

#include <allegro.h>
#include <assert.h>
#include <pthread.h>
#include <sys/stat.h>
#include "common.h"
#include "music-ad.h"
#include "music.h"
#include "osd.h"
#include "sound.h"
#include "seborrhea/seborrhea.h"


/* Music settings. */
#define PAN		128
#define SPEED		1000

static char *music_data;
static off_t music_data_size_allocated;
static SebShuffle *music_list;
static Sebum<SebMusic> *curr_music_file;
static Sebum<SebMusic> *last_music_file;
static enum {
    MUSIC_NOT_LOADING,
    MUSIC_LOADING_WAITING_FOR_THREAD,
    MUSIC_LOADING_THREAD_DONE
}  music_loading;

int music_vol = 255;

/*--------------------------------------------------------------*/
/* In-game volume adjusting.					*/
/*--------------------------------------------------------------*/

#define KEY_VOLUME_UP		KEY_F3
#define KEY_VOLUME_DOWN		KEY_F2

#ifdef ITOUCH
/* These used to work. */
# define KEY_VOLUME_UP2		95
# define KEY_VOLUME_DOWN2	96
#else
# define KEY_VOLUME_UP2		KEY_VOLUME_UP
# define KEY_VOLUME_DOWN2	KEY_VOLUME_DOWN
#endif

/*--------------------------------------------------------------*/

static pthread_t music_loader_thread;

static BOOL allocate_memory_for(const char *filename)
{
    struct stat stats;
    void *p;
    
    if (stat(filename, &stats) != 0) {
	fprintf(stderr, "[Music] Error statting %s.", filename);
	return NO;
    }

    /* Already allocated enough memory. */
    if (stats.st_size+1 <= music_data_size_allocated)
	return YES;

    p = realloc(music_data, stats.st_size+1);
    if (!p) {
	fprintf(stderr, "[Music] Not enough memory to load %s.", filename);
	return NO;
    }

    music_data = p;
    music_data_size_allocated = stats.st_size+1;
    return YES;
}


static void *music_loader(void *arg)
{
    Sebum<SebMusic> *new_music_file;
    assert(music_loading == MUSIC_LOADING_WAITING_FOR_THREAD);
    (void)arg;

#ifndef NO_ID3TAGS
    music_ad = [music_ad free];
#endif

    if ([music_list numElements] == 1)
	new_music_file = [music_list getRandomSebum];
    else {
	do {
	    /* Don't pick the same file twice in a row. */
	    new_music_file = [music_list getRandomSebum];
	} while (new_music_file == last_music_file);
    }

    if (!new_music_file)
	return NULL;

    /* Get the file size and reallocate if necessary. */
    if (not allocate_memory_for([new_music_file getMusicFilename]))
	goto end;

    if ([new_music_file loadFile:music_data :music_data_size_allocated]) {
	/* If only one file in directory, loop to avoid reloading. */
	BOOL loop = ([music_list numElements] == 1) ? YES : NO;

	curr_music_file = new_music_file;
	[curr_music_file playWithVol:music_vol Loop:loop];

#ifndef NO_ID3TAGS
	if (music_ad_enabled)
	    music_ad = [[MusicAd alloc]
			   initWithTagsFrom:[new_music_file getMusicFilename]];
#endif
    }

 end:
    music_loading = MUSIC_LOADING_THREAD_DONE;
    return NULL;
}

/*--------------------------------------------------------------*/

void play_music(const char *directory)
{
    if (music_list) {
	music_list = [music_list free];
	music_loading = MUSIC_NOT_LOADING;

	/* Don't check for playing the same file twice. */
	last_music_file = nil;
    }

    /* Load the directory.  No need to load the file yet.  Polling
       will pick a random file. */
    music_list = (SebShuffle *)[SebShuffle new];
    if (not [music_list loadSebumDirectory:directory]) {
	/* SebQueue will free itself when it fails.  */
	music_list = nil;
	return;
    }

    if ([music_list numElements] == 0)
	music_list = [music_list free];
}

void poll_music(void)
{
    int old_vol = music_vol;

    if (key[KEY_F4]) {
	if (curr_music_file) {
	    last_music_file = curr_music_file;
	    [curr_music_file unloadFile];
	    curr_music_file = nil;
	    music_loading = MUSIC_NOT_LOADING;
	}
	return;
    }

    /* In game volume adjusting. */
    if (key[KEY_VOLUME_UP] || key[KEY_VOLUME_UP2]) {
	music_vol += 5;
	[osd reset];
    }
    if (key[KEY_VOLUME_DOWN] || key[KEY_VOLUME_DOWN2]) {
	music_vol -= 5;
	[osd reset];
    }
    music_vol = MID(0, music_vol, 255);

    if (music_vol != old_vol)
	adjust_music_volume();

    if (music_vol == 0)
	return;

    if (curr_music_file) {
	if ([curr_music_file poll]) {
	    return;
	}
	else {
	    last_music_file = curr_music_file;
	    [curr_music_file unloadFile];
	    curr_music_file = nil;
	    return;
	}
    }

    /* Somehow out music has stopped playing.  Load a new file in
       another thread. */
    if (music_loading != MUSIC_LOADING_WAITING_FOR_THREAD && music_list) {
	music_loading = MUSIC_LOADING_WAITING_FOR_THREAD;
	pthread_create(&music_loader_thread, NULL, music_loader, NULL);
    }
    else if (music_loading == MUSIC_LOADING_THREAD_DONE) {
	pthread_join(music_loader_thread, NULL);
    }
}

void free_music(void)
{
    last_music_file = nil;
    curr_music_file = nil;
    music_list = [music_list free];

#ifndef NO_ID3TAGS
    music_ad = [music_ad free];
#endif
}

void adjust_music_volume(void)
{
    if (curr_music_file)
	[curr_music_file adjustVol:music_vol];
}

/*--------------------------------------------------------------*/
/* REdit needs these.						*/
/*--------------------------------------------------------------*/

#if 0
static ALMP3_MP3 *load_music_into_memory(const char *filename)
{
    PACKFILE *fp;
    long music_size;

    if (not allocate_memory_for(filename))
	return NULL;

    fp = pack_fopen(filename, F_READ);
    if (not fp) {
	fprintf(stderr, "Warning: %s not found.\n", filename);
	return NULL;
    }

    music_size = pack_fread(music_data, music_data_size_allocated, fp);
    pack_fclose(fp);
    return almp3_create_mp3(music_data, music_size);
}
#endif

void play_mp3(const char *filename)
{
    (void)filename;
/*     free_music(); */
/*     music = load_music_into_memory(filename); */
/*     if (music) */
/* 	almp3_play_ex_mp3(music, 32768, music_vol, PAN, SPEED, YES); */
}

void start_autopoll_music(void)
{
/*     if (music_vol != 0 && music) */
/* 	almp3_start_autopoll_mp3(music, 50); */
}

void stop_autopoll_music(void)
{
/*     if (music_vol != 0 && music) */
/* 	almp3_stop_autopoll_mp3(music); */
}

/*--------------------------------------------------------------*/

void music_init(void)
{
    if (not sound_initialized)
	music_vol = 0;
}

void music_shutdown(void)
{
    free_music();
    free(music_data);
    music_data = NULL;
}
