/* music-ad.m,
 *
 * Display the current songs ID3 tags when in game.  Uses TagLib.
 */

#ifndef NO_ID3TAGS

#include <allegro.h>
#include <assert.h>
#include <id3tag.h>
#ifndef NO_FBLEND
# include "fblend/include/fblend.h"
#endif
#include "common.h"
#include "music-ad.h"
#include "seborrhea/seborrhea.h"


MusicAd *music_ad;
BOOL music_ad_enabled = YES;


@implementation MusicAd
static char *get_field(struct id3_file *file, struct id3_tag *tag, 
		       const char *frame_str)
{
    struct id3_frame *frame = NULL;
    union id3_field *field = NULL;
    const id3_ucs4_t *data = NULL;
    char *ret = NULL;
    assert(file && tag && frame_str);

    frame = id3_tag_findframe(tag, frame_str, 0);
    if (not frame)
	goto exit;

    field = id3_frame_field(frame, 1);
    if (not field)
	goto exit;

    if (id3_field_getnstrings(field) < 1)
	goto exit;

    data = id3_field_getstrings(field, 0);
    ret = id3_ucs4_latin1duplicate(data);

 exit:
    /* field is automatically freed by id3_frame_delete. */
    if (frame)
	id3_frame_delete(frame);

    return ret;
}

- initWithTagsFrom:(const char *)filename
{
    //char album[128] = "", title[128] = "", artist[128] = "";
    char *album, *title, *artist;
    Sebum<SebFont> *font;

    [self init];

    /* Font. */
    font = [base_sebum getSebumByName:"fonts/elle16"];
    if (not font)
	goto error;

    {
        struct id3_file *file;
        struct id3_tag *tag;
	
        file = id3_file_open(filename, ID3_FILE_MODE_READONLY);
        if (not file)
	    goto error;
	
        tag = id3_file_tag(file);
        if (not tag) {
	    id3_file_close(file);
	    goto error;
	}
	
        /* Title/album/artist. */
        title  = get_field(file, tag, ID3_FRAME_TITLE);
        album  = get_field(file, tag, ID3_FRAME_ALBUM);
        artist = get_field(file, tag, ID3_FRAME_ARTIST);

	id3_tag_delete(tag);
	id3_file_close(file);
    }

    {
	unsigned int w, h;

	w = [font textLength:album];
	w = MAX(w, [font textLength:title]);
	w = MAX(w, [font textLength:artist]);
	if ((signed)w > SCREEN_W)
	    w = SCREEN_W;
	h = [font textHeight];

	alpha_text = create_bitmap(w, h*3);
	assert(alpha_text);
	clear_to_color(alpha_text, makecol(0x00, 0x00, 0x00));

	[font putString:title  To:alpha_text X:0 Y:0   Colour:0xff:0xff:0xff];
	[font putString:artist To:alpha_text X:0 Y:h   Colour:0xff:0xff:0xff];
	[font putString:album  To:alpha_text X:0 Y:h*2 Colour:0xff:0xff:0xff];
    }

    if (title)  free(title);
    if (album)  free(album);
    if (artist) free(artist);

    t = 2.5*SECONDS;
    return self;

 error:
    return [self free];
}

- free
{
    FREE_BITMAP(alpha_text);
    return [super free];
}

- update
{
    if (t > 0) {
	t--;
	return self;
    }
    else
	return [self free];
}

- (void) drawTo:(BITMAP *)dest
{
    unsigned int y, alpha;
    assert(dest);

    y = dest->h - alpha_text->h - 5;
    alpha = MIN(63, 64 * t / (1.5*SECONDS));

#ifndef NO_FBLEND
    fblend_add(alpha_text, dest, 10, y, alpha);
#else  /* Allegro routines. */
    drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
    set_trans_blender(0xff, 0xff, 0xff, alpha);
    rectfill(dest, 5, y-5, 5+alpha_text->w+10-1, dest->h-5,
	     makecol(0x40, 0x40, 0x40));
    rect(dest, 5, y-5, 5+alpha_text->w+10-1, dest->h-5,
	 makecol(0x40, 0x40, 0x40));
    drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);

    set_add_blender(0xff, 0xff, 0xff, alpha);
    draw_trans_sprite(dest, alpha_text, 10, y);
#endif /* NO_FBLEND. */
}
@end

#endif /* NO_ID3TAGS. */
