/* map-save.m,
 *
 * Map loading and saving routines.  Also does magic and version
 * stuff.
 */

#include <allegro.h>
#include <assert.h>
#include <objc/Object.h>
#include "common.h"
#include "map-save.h"
#include "map.h"
#include "projectiles/toothpaste.h"
#include "seborrhea/seborrhea.h"
#include "tile.h"
#include "unit-save.h"


#define foreach_tile(x,y)	for (y = 0; y < map_length; y++) foreach_x(x)
#define foreach_x(x)		for (x = 0; x < MAP_W_TILES; x++)

#define MAGIC_STR		"Raid Map "
#define MAP_VERSION		"0.3"
#define MAGIC_LENGTH		(9+1)


/*--------------------------------------------------------------*/

@interface Map (SavePrivate)
- (void) writeHeaderTo:(PACKFILE *)fp;
- (void) writeTilesTo:(PACKFILE *)fp;
- (void) writeUnitsTo:(PACKFILE *)fp :(BOOL)include_inactive_lists;

- (BOOL) readHeaderFrom:(MEMORY_FILE *)fp :(char *)version;
- (void) readTilesFrom:(MEMORY_FILE *)fp :(const char *)version;
- (BOOL) readUnitsFrom:(MEMORY_FILE *)fp :(const char *)version
		      :(BOOL)for_map_editor;
@end


@implementation Map (Save)
- (void) saveTo:(PACKFILE *)fp :(BOOL)include_inactive_lists
{
    assert(fp);

    [self writeHeaderTo:fp];
    [self writeTilesTo:fp];
    [self writeUnitsTo:fp :include_inactive_lists];
}

- (BOOL) loadFrom:(MEMORY_FILE *)fp :(BOOL)for_map_editor
{
    char version[16];
    assert(fp);

    if (not [self readHeaderFrom:fp :version])
	return NO;

    [self readTilesFrom:fp :version];

    if (not [self readUnitsFrom:fp :version :for_map_editor])
	return NO;

    return YES;
}
@end

/*--------------------------------------------------------------*/

@implementation Map (SavePrivate)
- (void) writeHeaderTo:(PACKFILE *)fp
{
    /* Magic and version number. */
    pack_fputs(MAGIC_STR, fp);
    pack_fputs(MAP_VERSION, fp);
    pack_putc('\n', fp);

    /* Map title and creator. */
    pack_fputs(title, fp);
    pack_putc('\n', fp);
    pack_fputs(creator, fp);
    pack_putc('\n', fp);

    /* The music file. */
    pack_fputs([current_map musicDirname], fp);
    pack_putc('\n', fp);
}

- (int) appendToIndex:(Sebum<SebImage> ***)tiles :(Sebum<SebImage> **)index :(int)len
{
    int i, x, y;
    const char *name;

    foreach_tile (x, y) {
	if (not tiles[y][x])
	    continue;

	name = [tiles[y][x] name];

	for (i = 0; i <= len; i++) {
	    if (i == len) {
		/* Tile not found in index. */
		assert(len < 1024);
		index[len] = tiles[y][x];
		len++;
		break;
	    }

	    /* This tile is already indexed. */
	    else if (streq(name, [index[i] name]))
		break;
	}
    }

    return len;
}

- (void) writeMap:(Sebum<SebImage> ***)tiles
		 :(Sebum<SebImage> **)index :(int)index_len :(PACKFILE *)fp
{
    int i, x, y;
    const char *name;
    assert(fp);

    foreach_tile (x, y) {
	if (not tiles[y][x]) {
	    pack_iputw(-1, fp);
	    continue;
	}

	name = [tiles[y][x] name];

	for (i = 0; i < index_len; i++) {
	    if (streq(name, [index[i] name])) {
		pack_iputw(i, fp);
		break;
	    }
	}
    }
}

- (void) writeTilesTo:(PACKFILE *)fp
{
    /* XXX: inefficient, but good enough? */
    Sebum<SebImage> *index[1024];
    int index_len = 0, i;

    assert(fp);

    /* Do we want stars? */
    pack_iputw(spawn_stars, fp);

    /* Toothpaste flavour. */
    pack_iputw(toothpaste_flavour, fp);

    /* Index each Sebum. */
    index_len = [self appendToIndex:base_tiles :index :0];
    if (spawn_stars)
	index_len = [self appendToIndex:parallax_tiles :index :index_len];

    /* How many tile types there are. */
    pack_iputw(index_len, fp);

    /* Write out the index of sebums. */
    for (i = 0; i < index_len; i++) {
	char name[1024];
	snprintf(name, sizeof name, "/tiles/%s", [index[i] name]);
	pack_fputs(name, fp);
        /* Don't lump the newline into the string. */
        pack_putc('\n', fp);
    }

    /* How long the map is. */
    pack_iputw(map_length, fp);

    /* The tiles and parallax tiles if necessary. */
    [self writeMap:base_tiles :index :index_len :fp];
    if (spawn_stars)
	[self writeMap:parallax_tiles :index :index_len :fp];
}

- (void) writeUnitsTo:(PACKFILE *)fp :(BOOL)include_inactive_lists
{
    save_unit_lists(fp, include_inactive_lists);
}


- (BOOL) readHeaderFrom:(MEMORY_FILE *)fp :(char *)version
{
    char magic[MAGIC_LENGTH];
    char music_fn[PATH_MAX];

    /* Magic and version number. */
    mmpk_fgets(magic, MAGIC_LENGTH, fp);
    mmpk_fgets(version, 16, fp);

    if (strneq(magic, MAGIC_STR)) {
	fprintf(stderr, "Magic mismatch!\nGot '%s', expecting '%s'.\n",
		magic, MAGIC_STR);
	return NO;
    }
    else if (strcmp(version, MAP_VERSION) > 0) {
	fprintf(stderr, "[Map] Warning: newer format (%s) detected.\n", version);
    }
    else if (strcmp(version, MAP_VERSION) < 0) {
	fprintf(stderr, "[Map] Warning: older format (%s) detected.\n", version);
    }

    /* Map title and creator. */
    mmpk_fgets(title, sizeof(title), fp);
    mmpk_fgets(creator, sizeof(creator), fp);

    /* The music file. */
    mmpk_fgets(music_fn, sizeof music_fn, fp);
    [self setMusicDirname:music_fn];
    return YES;
}

- (BOOL) readMap:(const char *)version :(Sebum<SebImage> ***)tiles
		:(Sebum<SebImage> **)index :(int)index_len :(MEMORY_FILE *)fp
{
    int i, x, y;
    BOOL error = NO;
    assert(fp);

    foreach_tile (x, y) {
	/* v0.1 maps were only 5 tiles wide -> skip one tile. */
	if (streq(version, "0.1") && x == 5) {
	    if (x == 5)
		continue;
	}

	/* -1 denotes tile = nil.  Since we are using words, be
	   careful not to mangle the -ve sign when we read it back. */
	i = (int)((short)mmpk_igetw(fp));
	if ((i >= index_len) || (i >= 0 && not index[i])) {
	    printf("Tile %d missing at (%d, %d).\n", i, x, y);
	    [self setTileAt:tiles :x :y to:nil];
	    error = YES;
	}
	else {
	    /* Go through setTileAt because we need to create
	       SebAnimator classes. */
	    if (i < 0)
		[self setTileAt:tiles :x :y to:nil];
	    else
		[self setTileAt:tiles :x :y to:index[i]];
	}
    }

    return error;
}

- (void) readTilesFrom:(MEMORY_FILE *)fp :(const char *)version
{
    Sebum<SebImage> *index[1024];
    int index_len, i;
    BOOL error = NO;
    assert(fp);

    /* Special: Stars. */
    if (strneq(version, "0.1"))
	spawn_stars = mmpk_igetw(fp);
    else
	spawn_stars = NO;

    /* Special: Toothpaste flavour. */
    if (strneq(version, "0.2"))
	toothpaste_set_flavour(mmpk_igetw(fp));
    else
	toothpaste_set_flavour(0);

    /* How many tile types there are. */
    index_len = mmpk_igetw(fp);
    assert(index_len < 1024);

    /* What Sebum number n refers to. */
    for (i = 0; i < index_len; i++) {
	char name[1024];	/* XXX? */
	mmpk_fgets(name, sizeof name, fp);
	index[i] = get_tile_by_name(name);
	if (not index[i])
	    fprintf(stderr, "[Map] Missing tile %s.\n", name);
    }

    /* How long the map is. */
    [self resize:mmpk_igetw(fp)];

    /* The map tiles. */
    error = [self readMap:version :base_tiles :index :index_len :fp];
    if (spawn_stars) {
	if ([self readMap:version :parallax_tiles :index :index_len :fp])
	    error = YES;
    }

    if (error) {
	fprintf(stderr,
		"Errors occured while loading the map.\n"
		"Are you missing some datafiles?\n");
    }

    current_scroll_rate = 0.0;
    desired_scroll_rate = DEFAULT_SCROLL_RATE;
    pixels_to_stop_scrolling = screen_h;
}

- (BOOL) readUnitsFrom:(MEMORY_FILE *)fp :(const char *)version
		      :(BOOL)for_map_editor
{
    (void)version;

    if (load_unit_lists_from_memory(fp, for_map_editor))
	return YES;
    else {
	fprintf(stderr, "Error loading units!\n");
	return NO;
    }
}
@end


/*--------------------------------------------------------------*/
/* Saving.							*/
/*--------------------------------------------------------------*/

void save_map(const char *filename)
{
    PACKFILE *fp;
    assert(filename && current_map);

    fp = pack_fopen(filename, "w");
    if (fp) {
	[current_map saveTo:fp :NO];
	pack_fclose(fp);
    }
}

/*--------------------------------------------------------------*/
/* Loading.							*/
/*--------------------------------------------------------------*/

MEMORY_FILE *load_map(const char *filename, BOOL for_map_editor)
{
    MEMORY_FILE *mfp;
    PACKFILE *fp = NULL;
    assert(filename);

    fp = pack_fopen(filename, "r");
    if (not fp)
	return NO;

    mfp = malloc(sizeof *mfp);
    assert(mfp);

    mfp->len  = file_size(filename); /* XXX? */
    mfp->data = malloc(mfp->len);
    mfp->pos  = mfp->data;
    assert(mfp->data);

    /* Load the packfile into memory. */
    pack_fread(mfp->data, mfp->len, fp);
    pack_fclose(fp);

    if (not [current_map loadFrom:mfp :for_map_editor]) {
	fprintf(stderr, "Errors while loading %s!  Aborting.\n", filename);
	mmpk_fclose(mfp);
	return NULL;
    }

    return mfp;
}
