/* redit.m,
 *
 * The Official (ie. dodgy) Raid'em map editor.  For historical
 * reasons, this one is also called REdit.
 */

#include <adime.h>
#include <allegro.h>
#include <assert.h>
#include <objc/Object.h>
#include "common.h"
#include "game.h"
#include "group.h"
#include "gui/gui-dirty.h"
#include "gui/gui-keyboard.h"
#include "gui/gui-mouse.h"
#include "gui/gui-raid.h"
#include "gui/gui-widgets.h"
#include "gui/gui.h"
#include "init.h"
#include "input.h"
#include "map-editor/help.h"
#include "map-editor/map-proc.h"
#include "map-editor/mouse-label-proc.h"
#include "map-editor/palette-air.h"
#include "map-editor/palette-formation.h"
#include "map-editor/palette-ground.h"
#include "map-editor/palette-parallax.h"
#include "map-editor/palette-switcher.h"
#include "map-editor/palette-tile.h"
#include "map-save.h"
#include "music.h"
#include "player.h"
#include "projectiles/all-projs.h"
#include "sound.h"
#include "unit.h"
#include "units/all-units.h"
#include "video.h"


#define EDITOR_NAME			"REdit"
#define EDITOR_VERSION			"0.2"
#define EDITOR_NAME_AND_VERSION		EDITOR_NAME " " EDITOR_VERSION

enum EDITING_MODE editing_mode;


static Frame *root_frame;
static BITMAP *dbuf;

static PaletteSwitcher *palette;
static MapProc *map_editing_area;

/*--------------------------------------------------------------*/
/* Editing mode button callbacks.				*/
/*--------------------------------------------------------------*/

static int set_editing_mode_parallax(Widget *_w, int _m, int _d)
{
    (void)_w, (void)_m, (void)_d;
    editing_mode = PARALLAX_MODE;
    [palette activateChild:editing_mode];
    return 0;
}

static int set_editing_mode_tiles(Widget *_w, int _m, int _d)
{
    (void)_w, (void)_m, (void)_d;
    editing_mode = TILE_MODE;
    [palette activateChild:editing_mode];
    return 0;
}

static int set_editing_mode_ground(Widget *_w, int _m, int _d)
{
    (void)_w, (void)_m, (void)_d;
    editing_mode = GROUND_MODE;
    [palette activateChild:editing_mode];
    return 0;
}

static int set_editing_mode_air(Widget *_w, int _m, int _d)
{
    (void)_w, (void)_m, (void)_d;
    editing_mode = AIR_MODE;
    [palette activateChild:editing_mode];
    return 0;
}

static int set_editing_mode_formation(Widget *_w, int _m, int _d)
{
    (void)_w, (void)_m, (void)_d;
    editing_mode = FORMATION_MODE;
    [palette activateChild:editing_mode];
    return 0;
}

/*--------------------------------------------------------------*/

static void pick_music(void)
{
    char filename[PATH_MAX] = "music/";

    start_autopoll_music();
    if (adime_file_select("Play which song?", filename, "mp3", sizeof filename, 600, 400)) {
	stop_autopoll_music();
	play_mp3(filename);
    }
    else
	stop_autopoll_music();
}

/*--------------------------------------------------------------*/

static BOOL quit;
static void close_window_hook(void) { quit = YES; }

static void switch_into_redit(void) { [root_frame receiveMessage:MSG_REDRAW :0]; }

/*--------------------------------------------------------------*/
/* REdit menu callbacks.					*/
/*--------------------------------------------------------------*/

static int callback_new(DIALOG *_)
{
    (void)_;
    [map_editing_area freeMap];
    [map_editing_area initMap];
    return -1;
}

static int callback_load(DIALOG *_)
{
    char filename[PATH_MAX] = "maps/";
    (void)_;

    if (adime_file_select("Load...", filename, RAID_MAP_EXTENSION,
			  sizeof filename, 600, 400)) {
	[map_editing_area loadMap:filename];
	return -1;
    }
    else {
	return 0;
    }
}

static int callback_save(DIALOG *_)
{
    char filename[PATH_MAX] = "maps/";
    (void)_;

    if (adime_file_select("Save as...", filename, RAID_MAP_EXTENSION,
			  sizeof filename, 600, 400)) {
	save_map(filename);
	return -1;
    }
    else {
	return 0;
    }
}

static int callback_test(DIALOG *_)
{
    int redit_w = SCREEN_W;
    int redit_h = SCREEN_H;
    Map *curr_map = current_map;
    const char *levels[] = {"./maps/__playblast__.map"};
    (void)_;

    save_map(levels[0]);

    show_mouse(NULL);
    if (redit_w != 640 || redit_h != 480) {
	switch_gfx_mode(fullscreen ? GFX_AUTODETECT_FULLSCREEN : GFX_AUTODETECT_WINDOWED, 640, 480, 0);
    }
    input_init();
    video_bitmap_init();
    num_players = 1;
    map_start_y = offsetY+screen_h*2;
    raid_start_levels(levels, 1, NO, NO);
    video_bitmap_shutdown();
    input_shutdown();

    mark_all_unit_data_necessary();

    if (redit_w != 640 || redit_h != 480) {
	switch_gfx_mode(fullscreen ? GFX_AUTODETECT_FULLSCREEN : GFX_AUTODETECT_WINDOWED, redit_w, redit_h, 0);
    }

    current_map = curr_map;
    [map_editing_area loadMap:levels[0]];
    [root_frame receiveMessage:MSG_REDRAW :0];
    show_mouse(screen);
    return -1;
}

static int callback_quit(DIALOG *_)
{
    (void)_;
    if (adime_dialogf("Quit REdit?",
		      ADIME_ALIGN_CENTRE, ADIME_ALIGN_CENTRE, 200,
		      "") != 2)
	quit = YES;
		  
    return -1;
}

static int callback_dummy(DIALOG *_)
{
    (void)_;
    return -1;
}

static void weapon_selection(void)
{
    int w1 = playblast_primary_weapon.weapon;
    int l1 = playblast_primary_weapon.level;
    int w2 = playblast_secondary_weapon.weapon+1;
    int l2 = playblast_secondary_weapon.level;
    int w3 = playblast_tertiary_weapon+1;

    adime_lowlevel_dialogf("Weapon Selection",
			   ADIME_ALIGN_CENTRE, ADIME_ALIGN_CENTRE, 200,
			   "Primary Weapon:%line[]"
			   "%list[Laser;Toothpaste;Vulcan]"
			   "%list[Level 1;Level 2;Level 3]"

			   "Secondary Weapon:%line[]"
			   "%list[None;Homing Missile;Missile;Pink Orange Red]"
			   "%list[Level 1;Level 2;Level 3]"

			   "Tertiary Weapon:%line[]"
			   "%list[None;Robots;Shields;Shields (2)]"
			   "%button[OK]",
			   &w1, &l1, &w2, &l2, &w3,
			   callback_dummy);

    playblast_primary_weapon.weapon	= w1;
    playblast_primary_weapon.level	= l1;
    playblast_secondary_weapon.weapon	= w2-1;
    playblast_secondary_weapon.level	= l2;
    playblast_tertiary_weapon		= w3-1;
}

/*--------------------------------------------------------------*/

void redit_loop(const char *fn)
{
    quit = NO;

    if (fn)
	[map_editing_area loadMap:fn];

    while (!quit) {
	[root_frame update];
	clear_to_color(dbuf, makecol(0xd0, 0xd0, 0xc0));
	[root_frame draw:dbuf];

	poll_music();

	if (key[KEY_F1]) {
	    do_help_dialog();
	    [root_frame receiveMessage:MSG_REDRAW :0];
	}

	if (key[KEY_F2]) {
	    pick_music();
	    [root_frame receiveMessage:MSG_REDRAW :0];
	}

	if (key[KEY_F5]) callback_save(NULL);

	/* 'Playblast'. */
	if (key[KEY_F7]) callback_test(NULL);
	if (key[KEY_F8]) weapon_selection();

	if (key[KEY_F9]) {
	    start_autopoll_music();
	    if (adime_lowlevel_dialogf("REdit",
				       ADIME_ALIGN_CENTRE, ADIME_ALIGN_CENTRE, 200,
				       "%button[&New Map]"
				       "%button[&Load Map]"
				       "F5:%button[&Save Map]"
				       "F7:%button[&Test Run]"
				       "\n\n"
				       "%button[Quit]"
				       "%button[Cancel]",
				       callback_new,
				       callback_load,
				       callback_save,
				       callback_test,
				       callback_quit,
				       callback_dummy) != 2) {
		[root_frame receiveMessage:MSG_REDRAW :0];
	    }
	    stop_autopoll_music();
	}
    }
}

/*--------------------------------------------------------------*/

static void make_gui(void);

int redit_init(void)
{
    if (adime_init() < 0)
	return -1;

    dbuf = create_bitmap(SCREEN_W, SCREEN_H);
    assert(dbuf);

    unit_init();
    unit_group_init();

    gui_init();
    make_gui();

    {					/* Load data. */
	unsigned int i;

	for (i = 0; i < air_units_table.num_classes; i++)
	    load_unit_data_for(air_units_table.class_data[i].class, YES);

	for (i = 0; i < ground_units_table.num_classes; i++)
	    load_unit_data_for(ground_units_table.class_data[i].class, NO);
    }

    set_window_title(EDITOR_NAME_AND_VERSION);
    set_window_close_hook(&close_window_hook);
    set_display_switch_callback(SWITCH_IN, switch_into_redit);

    return 0;
}

void redit_shutdown(void)
{
    root_frame = [root_frame free];

    adime_exit();
    gui_shutdown();

    if (dbuf) {
        destroy_bitmap(dbuf);
        dbuf = NULL;
    }

    mark_all_projectile_data_unnecessary();
    unit_group_shutdown();
    unit_shutdown();
}










/*------------------------------------------------------------*/
/* Yucky.  Stay away.                                         */
/*------------------------------------------------------------*/










static void make_gui(void)
{
    Pane *root;

    gui_raid_init();
    default_raid_font = [base_sebum getSebumByName:"fonts/am14"];

    root_frame = [Frame new];
    root = [Pane new];
    assert(default_raid_font && root && root_frame);

    {					/* Map area. */
	VPackingBox *vbox = [VPackingBox newWithName:"Map VBox"];

	/* The mouse position label. */
	[vbox insertWidget:[[MousePositionLabel newWithName:"Mouse Position"]
			       setMaxHeight:15]];

	/* The map. */
	map_editing_area = [[MapProc newWithName:"The Map"]
			       setMinWidth:320];
	[vbox insertWidget:map_editing_area];

	[root insertWidget:vbox];
    }

    {					/* Palette. */
	VPackingBox *vbox = [[VPackingBox newWithName:"Palette VBox"]
				setMaxWidth:(64+4)*4+30];
	Widget *w;

	{				/* Editing mode buttons. */
	    HPackingBox *hbox = [[HPackingBox newWithName:"Editing Mode Buttons"]
				    setMaxHeight:25];

	    [hbox insertWidget:[[[Button newWithName:"Parallax Button"]
				    setText:"Parallax"]
				   setPushedProc:set_editing_mode_parallax]];

	    [hbox insertWidget:[[[Button newWithName:"Tiles Button"]
				    setText:"Tiles"]
				   setPushedProc:set_editing_mode_tiles]];
	    
	    [hbox insertWidget:[[[Button newWithName:"Ground Button"]
				    setText:"Ground"]
				   setPushedProc:set_editing_mode_ground]];
	    
	    [hbox insertWidget:[[[Button newWithName:"Air Button"]
				    setText:"Air"]
				   setPushedProc:set_editing_mode_air]];

	    [hbox insertWidget:[[[Button newWithName:"Formation Button"]
				    setText:"Formations"]
				   setPushedProc:set_editing_mode_formation]];

	    [vbox insertWidget:hbox];
	}

	/* The palette(s). */
	palette = [PaletteSwitcher newWithName:"Palette Switcher"];

	w = [[PaletteParallax newWithName:"Parallax Palette"]
		setMapProcWidget:map_editing_area];
	[palette setChild:PARALLAX_MODE :w];

	w = [[PaletteTile newWithName:"Tiles Palette"]
		setMapProcWidget:map_editing_area];
	[palette setChild:TILE_MODE :w];

	w = [[PaletteGroundUnit newWithName:"Ground Units Palette"]
		setMapProcWidget:map_editing_area];
	[palette setChild:GROUND_MODE :w];

	w = [[PaletteAirUnit newWithName:"Air Units Palette"]
		setMapProcWidget:map_editing_area];
	[palette setChild:AIR_MODE :w];

	w = [[PaletteFormation newWithName:"Formations Palette"]
		setMapProcWidget:map_editing_area];
	[palette setChild:FORMATION_MODE :w];

	[vbox insertWidget:palette];
	[root insertWidget:vbox];
    }

    [[root_frame setChild:root] setX:0 Y:0 W:dbuf->w H:dbuf->h];

    /* Startup begins in tile editing mode. */
    set_editing_mode_tiles(NULL, 0, 0);
}
