/* palette-formation.m,
 *
 * Palette for pre-defined unit formations.
 */

#include <assert.h>
#include <allegro.h>
#include "common.h"
#include "gui/gui-clip.h"
#include "map-editor/palette-formation.h"
#include "map-editor/unit-editor.h"
#include "unit.h"
#include "units/all-units.h"


#define MAX_FORMATIONS			11
#define MAX_UNITS_PER_FORMATION		6


static struct {
    const char *name, *name2;
    struct {
	const BOOL air;
	const char *name;
	const int offx, offy;
    } units[MAX_UNITS_PER_FORMATION];
} formation[MAX_FORMATIONS] = {
    { "4 bugs", "short",
      { { YES, "Bug", -48,   0 },
	{ YES, "Bug", -16,   0 },
	{ YES, "Bug",  16,   0 },
	{ YES, "Bug",  48,   0 } }
    },

    { "4 bugs", "vanguard",
      { { YES, "Bug", -48,   0 },
	{ YES, "Bug", -16,  16 },
	{ YES, "Bug",  16,  16 },
	{ YES, "Bug",  48,   0 } }
    },

    { "6 cargos", "(normal)",
      { { YES, "Cargo",-160,   0 },
	{ YES, "Cargo",-128,   0 },
	{ YES, "Cargo", -96,   0 },
	{ YES, "Cargo", -64,   0 },
	{ YES, "Cargo", -32,   0 },
	{ YES, "Cargo",   0,   0 } }
    },

    { "6 cargos", "(health)",
      { { YES, "CargoHealth",-160,   0 },
	{ YES, "CargoHealth",-128,   0 },
	{ YES, "CargoHealth", -96,   0 },
	{ YES, "CargoHealth", -64,   0 },
	{ YES, "CargoHealth", -32,   0 },
	{ YES, "CargoHealth",   0,   0 } }
    },

    { "3 dragons", "diag\\",
      { { YES, "DragonFly", -32, -16 },
	{ YES, "DragonFly", -16,  -8 },
	{ YES, "DragonFly",   0,   0 } }
    },      

    { "3 dragons", "diag/",
      { { YES, "DragonFly",  32, -16 },
	{ YES, "DragonFly",  16,  -8 },
	{ YES, "DragonFly",   0,   0 } }
    },      

    { "5 lints", "line",
      { { YES, "Lint",   0, -64 },
	{ YES, "Lint",   0, -48 },
	{ YES, "Lint",   0, -32 },
	{ YES, "Lint",   0, -16 },
	{ YES, "Lint",   0,   0 } }
    },

    { "4 scourges", "vanguard",
      { { YES, "Scourge", -48,   0 },
	{ YES, "Scourge", -16,  16 },
	{ YES, "Scourge",  16,  16 },
	{ YES, "Scourge",  48,   0 } }
    },

    { "5 strumbles", "short",
      { { YES, "Strumble",-128,   0 },
	{ YES, "Strumble", -64,   0 },
	{ YES, "Strumble",  64,   0 },
	{ YES, "Strumble", 128,   0 },
	{ YES, "Strumble",   0,   0 },
      }
    },

    { "4 stumpies", "short",
      { { YES, "Stumpy", -72,   0 },
	{ YES, "Stumpy", -24,   0 },
	{ YES, "Stumpy",  24,   0 },
	{ YES, "Stumpy",  72,   0 } }
    },

    { "5 zippy", "short",
      { { YES, "Zippy", -64, 0 },
	{ YES, "Zippy", -32, 0 },
	{ YES, "Zippy",  32, 0 },
	{ YES, "Zippy",  64, 0 },
	{ YES, "Zippy",   0, 0 }}
    }
};


@implementation PaletteFormation
+ (unsigned int) countItems
{
    return MAX_FORMATIONS;
}

- (void) selectLMBItem:(int)n
{
    if ((unsigned)n >= [[self class] countItems])
	return;

    [label setText:formation[n].name];
    [map_proc setSelectedLMBItem:n];
}

- (void) drawNthItem:(BITMAP *)dest :(unsigned int)nth :(int)xx :(int)yy
{
    clip_t clip;

    save_clip(dest, &clip);
    set_clip_rect(dest, xx, yy, xx+64, yy+64);

    /* Draw a unit. */
    {
	Class class = find_class_by_name(formation[nth].units[0].name,
					 formation[nth].units[0].air);
	if (class) {
	    Unit *unit = [class new];
	    [unit drawTo:dest at:xx :yy];
	    [unit free];
	}
    }

    textout_ex(dest, font, formation[nth].name,  xx, yy,    makecol(0x00,0x80,0x00), -1);
    textout_ex(dest, font, formation[nth].name2, xx, yy+12, makecol(0x00,0x80,0x00), -1);

    restore_clip(dest, &clip);
    set_clip_rect(dest, clip.cl, clip.ct, clip.cr, clip.cb);
}
@end


/* Spawn the next unit in a formation.  Return nil if formation is
   complete.  i is the ith unit in the nth formation to spawn. */
Unit *spawn_formation_interate(unsigned int nth, unsigned int i, int x, int y)
{
    assert(nth < MAX_FORMATIONS);

    if ((i >= MAX_UNITS_PER_FORMATION) ||
	(!formation[nth].units[i].name)) {
	return nil;
    }
    else {
	const char *class_name = formation[nth].units[i].name;
	BOOL is_air = formation[nth].units[i].air;
	Class class = find_class_by_name(class_name, is_air);

	if (!class) {
	    fprintf(stderr, "[Redit] %s unit %s not found!\n",
		    is_air ? "Air" : "Ground", class_name);
	    return nil;
	}

	return spawn_unit(class,
			  x + formation[nth].units[i].offx, 
			  y + formation[nth].units[i].offy,
			  is_air ? ACTIVE_AIR_LIST : ACTIVE_GROUND_LIST,
			  YES);
    }
}
