/* map-proc.m,
 */

#include <allegro.h>
#include <assert.h>
#include <math.h>
#include "adime/include/adime.h"
#include "common.h"
#include "group.h"
#include "gui/gui-clip.h"
#include "gui/gui-dirty.h"
#include "gui/gui-raid.h"
#include "gui/gui-widgets.h"
#include "gui/gui.h"
#include "map-editor/map-proc.h"
#include "map-editor/palette-formation.h"
#include "map-editor/palette-proc.h"
#include "map-editor/tile-connection.h"
#include "map-editor/unit-editor.h"
#include "map-save.h"
#include "map.h"
#include "music.h"
#include "player.h"
#include "projectile.h"
#include "projectiles/toothpaste.h"
#include "seborrhea/seborrhea.h"
#include "sound.h"
#include "tile.h"
#include "unit-intern.h"
#include "unit.h"
#include "units/all-units.h"


#define SCROLL_RATE	(TILE_H/2)
#define GRID_SIZE	16
#define GRID_SIZE_2	(GRID_SIZE/2)

/*--------------------------------------------------------------*/
/* Map panning.							*/
/*--------------------------------------------------------------*/

static void shift_list(List *list, double deltax, double deltay)
{
    ListIterator *it;
    Unit *unit;
    double ux, uy;

    foreach (it, list) {
        unit = [it getItem];
	[unit getX:&ux Y:&uy];
	ux += deltax;
	uy += deltay;
	if (uy < 0)
	    uy = 0;
	[unit setX:ux Y:uy];
    }
}

/*--------------------------------------------------------------*/

@interface MapProc (Private)
- (void) setScrollbarPrecision;

/* Tile editing mode. */
- (void) plonkLMBTileAt:(const double)x_ :(const double)y_;
- (void) plonkRMBTileAt:(const double)x_ :(const double)y_;
- (BOOL) placeTileAt:(int)x0 :(int)y0 to:(Sebum<SebImage> *)tile;
- (void) autoPlaceAt:(const int)x0 :(const int)y0 :(const int)j;
- (void) floodfill;


/* Unit editing mode. */
- (void) processUnitModeLMBPressed:(const double)x_ :(const double)y_;


- (void) specialMapHacks;

- (void) customiseDisplay;
- (void) drawGrid:(BITMAP *)dest;
- (void) drawRulers:(BITMAP *)dest;

- (void) setActivationLine;
- (void) setTravelRange;
- (void) setGroup;
- (void) setScrollRate;
- (void) togglePowerups;
@end

@interface MapProc (Transform)
- (void) doPanMap;
- (void) doResizeMap;
@end

/*--------------------------------------------------------------*/

@implementation MapProc
- (id) setX:(int)x_ Y:(int)y_ W:(int)w_ H:(int)h_
{
    int sw, sh;
    [super setX:x_ Y:y_ W:w_ H:h_];

    sw = [vscrollbar maxWidth];
    sh = [hscrollbar maxHeight];

    [hscrollbar setX:x+sw Y:y+h-sh W:w-sw H:sh];
    [vscrollbar setX:x    Y:y      W:sw   H:h-sh];
    [self setScrollbarPrecision];

    return self;
}

- (int) minWidth  { return [hscrollbar minWidth]  + MAX(min_width,  [vscrollbar minWidth]); }
- (int) minHeight { return [vscrollbar minHeight] + MAX(min_height, [hscrollbar minHeight]); }

- (Widget *) getWidgetBounding:(int)mx :(int)my
{
    Widget *widget;

    widget = [hscrollbar getWidgetBounding:mx :my];
    if (widget) return widget;

    widget = [vscrollbar getWidgetBounding:mx :my];
    if (widget) return widget;

    return [super getWidgetBounding:mx :my];
}

- init
{
    /* Scrollbars. */
    hscrollbar = [[[RaidHScrollbar newWithName:"MapProc HScrollBar"]
		       setHPadding:0 VPadding:1]
		     setParentWidget:self];
    vscrollbar = [[[RaidVScrollbar newWithName:"MapProc VScrollBar"]
		      setHPadding:1 VPadding:0]
		     setParentWidget:self];

    h_padding = 0;
    v_padding = 0;

    [self initMap];

    auto_plonk = YES;
    selected_ground_class = ground_units_table.class_data[0].class;
    selected_air_class = air_units_table.class_data[0].class;

    visable.air_units = 1;
    visable.ground_units = 1;

    return [super init];
}

- free
{
    hscrollbar = [hscrollbar free];
    vscrollbar = [vscrollbar free];
    [self freeMap];
    return [super free];
}

- initMap
{
    the_map = [Map new];	assert(the_map);
    tagged_list = [LIST_NEW];	assert(tagged_list);

    [[[the_map resize:100] reset] setOffsetX:0 Y:0];
    if (not current_map)
        current_map = the_map;

    [self setScrollbarPrecision];

    return self;
}

- freeMap
{
    unit_reset();
    mark_all_unit_data_necessary();
    the_map = [the_map free];

    if (tagged_list) {
	[tagged_list removeAllItems];
	tagged_list = [tagged_list free];
    }

    return self;    
}

- (BOOL) draw:(BITMAP *)bmp :(BOOL)marked_dirty
{
    ListIterator *it;
    clip_t old_clip;

    marked_dirty = [super draw:bmp :marked_dirty];
    
    /* Draw the scrollbars. */
    [hscrollbar draw:bmp :marked_dirty];
    [vscrollbar draw:bmp :marked_dirty];

    if (!marked_dirty)
	return NO;

    save_clip(bmp, &old_clip);
    set_clip_rect(bmp, x+[vscrollbar width], y, x+w-1, y+h-[hscrollbar height]-1);

    /* Draw the map. */
    if ([the_map stars]) {
	[the_map drawParallaxLayer:bmp :NO];

	/* Don't draw the map tiles if we are in parallax mode. */
	if (editing_mode >= TILE_MODE)
	    [the_map drawTiles:bmp];
    }
    else
	[the_map drawTiles:bmp];

    /* Draw the units. */
    if (visable.ground_units || editing_mode == GROUND_MODE) {
	draw_units(bmp, ACTIVE_GROUND_LIST);
    }
    if (visable.air_units || editing_mode == AIR_MODE) {
	draw_unit_shadows(bmp);
	draw_units(bmp, ACTIVE_AIR_LIST);
    }

    /* Extra stats for the tagged unit. */
    foreach (it, tagged_list) {
        int rtm = text_mode(-1);
        [[it getItem] drawMapEditorExtras:bmp];
        text_mode(rtm);
    }

    if (auto_plonk)
	textout_ex(bmp, font, "auto-plonk", 20, 20, makecol(0x80,0x80,0xff), -1);

    [self drawGrid:bmp];
    [self drawRulers:bmp];
    
    restore_clip(bmp, &old_clip);

    return YES;
}

- (void) deleteUnit:(Unit *)unit
{
    if (not unit)
        return;

    [tagged_list removeItem:unit];

    destroy_unit(unit);
    flags |= FLAG_DIRTY;
}

- (void) processSpecialKeys:(int)data
{
    switch (data) {
      case KEY_A:
	  [self setActivationLine];
	  break;
      case KEY_D:
	  [self setTravelRange];
	  break;
      case KEY_G:
	  [self setGroup];
	  break;
      case KEY_S:
	  [self setScrollRate];
	  break;
      case KEY_1:
	  [self togglePowerups];
	  break;
      case KEY_2: {
	  ListIterator *it;
	  foreach (it, tagged_list)
	      [[it getItem] toggleFlag:FLAG_ONLY_SPAWN_IN_TWO_PLAYER];

	  break;
      }
      default:
	  return;
    }
    flags |= FLAG_DIRTY;
}

- (void) loadMap:(const char *)filename
{
    [tagged_list removeAllItems];
    load_map(filename, YES);
    [self setScrollbarPrecision];
    flags |= FLAG_DIRTY;
}

- (void) selectMapMusic 
{
    char filename[PATH_MAX] = "music/";

    start_autopoll_music();
    if (adime_file_select("Play which directory during gameplay?", filename, "mp3", sizeof filename, 600, 400)) {
	/* I assume the music is inside music/. */
	[the_map setMusicDirname:filename];
    }
    stop_autopoll_music();
}

- (int) HSliderToOffset:(int)data { return data - screen_w; }
- (void) offsetToHSlider:(int)off { [hscrollbar setHandlePosition:off + screen_w]; }
- (int) VSliderToOffset:(int)data { return data; }
- (void) offsetToVSlider:(int)off { [vscrollbar setHandlePosition:off]; }

- (void) receiveMessage:(int)msg :(int)data
{
    ListIterator *it;
    Unit *unit;
    int x_, offset_x, prev_offset_x;
    int y_, offset_y, prev_offset_y;

    offset_x = prev_offset_x = [the_map offsetX];
    offset_y = prev_offset_y = [the_map offsetY];
    x_ = mouse_x + offset_x;
    y_ = mouse_y + offset_y;

    if (snap_to_grid) {
        x_ += GRID_SIZE_2 * ((x_ > 0) ? 1 : -1);
        y_ += GRID_SIZE_2 * ((y_ > 0) ? 1 : -1);
	x_ -= x_ % GRID_SIZE;
	y_ -= y_ % GRID_SIZE;
    }

    switch (msg) {
	case MSG_KEY_PRESSED:
	    clear_keybuf();
	    if (data == KEY_ENTER) {
		auto_plonk = !auto_plonk;
		flags |= FLAG_DIRTY;
	    }
	    elif (data == KEY_SPACE) {
		snap_to_grid = !snap_to_grid;
                flags |= FLAG_DIRTY;
            }
            elif (data == KEY_DEL) {
		ListIterator *nx;
		foreach_nx (it, nx, tagged_list) {
		    unit = [it getItem];
		    nx = [it next];
		    [self deleteUnit:unit];
		}
	    }
            elif (data == KEY_ESC) {
		[tagged_list removeAllItems];
                flags |= FLAG_DIRTY;
            }
	    elif (data == KEY_R) [self doResizeMap];
	    elif (data == KEY_P) [self doPanMap];
	    elif (data == KEY_F3) [self selectMapMusic];
	    elif (data == KEY_F4) [self specialMapHacks];
	    elif (data == KEY_F12) [self customiseDisplay];
	    elif (data == KEY_F) [self floodfill];
            elif (not [tagged_list isEmpty])
                [self processSpecialKeys:data];
	    break;

	case MSG_KEY_HELD:
	    if (data == KEY_HOME)  offset_y = 0;
	    if (data == KEY_END)   offset_y = [the_map length] * 64 - h;
	    if (data == KEY_PGUP)  offset_y -= SCROLL_RATE * 5;
	    if (data == KEY_PGDN)  offset_y += SCROLL_RATE * 5;
	    if (data == KEY_LEFT)  offset_x -= SCROLL_RATE;
	    if (data == KEY_RIGHT) offset_x += SCROLL_RATE;
	    if (data == KEY_UP)    offset_y -= SCROLL_RATE;
	    if (data == KEY_DOWN)  offset_y += SCROLL_RATE;

	    if (![tagged_list isEmpty]) {
                double ux, uy, ua;	/* Unit x, y, angle. */
		double xx, yy, aa;	/* Displacements. */

		xx = yy = aa = 0.0;
		if (data == KEY_4_PAD) xx--;
		if (data == KEY_6_PAD) xx++;
		if (data == KEY_8_PAD) yy--;
		if (data == KEY_2_PAD) yy++;
		if (data == KEY_7_PAD) aa += deg2rad(1.0);
		if (data == KEY_9_PAD) aa -= deg2rad(1.0);

		if (!xx && !yy && !aa)
		    break;

		foreach (it, tagged_list) {
		    unit = [it getItem];

		    [unit getX:&ux Y:&uy];
		    ua = [unit angle];

		    ux += xx;
		    uy += yy;
		    ua += aa;

		    if (ua <= -M_PI) ua += 2 * M_PI;
		    if (ua >   M_PI) ua -= 2 * M_PI;

		    [[unit setX:ux Y:uy] setAngle:ua];
		}
                flags |= FLAG_DIRTY;
            }
	    break;

	case MSG_MOUSE_PRESSED:
	    mouse_click_x = mouse_x;
	    mouse_click_y = mouse_y;
	    last_mouse_x  = mouse_x+offset_x;
	    last_mouse_y  = mouse_y+offset_y;

	    if (data == 0) {	/* Left button. */
		if (editing_mode <= TILE_MODE)
		    [self plonkLMBTileAt:x_ :y_];
		else
		    [self processUnitModeLMBPressed:x_ :y_];
	    }
	    elif (data == 1) {	/* RMB. */
		if (editing_mode <= TILE_MODE)
		    [self plonkRMBTileAt:x_ :y_];
		else
		    [self deleteUnit:findUnitAt(x_, y_)];
	    }
	    break;

	case MSG_MOUSE_HELD:

	    if (data == 0) {	/* Left button. */
		if (editing_mode <= TILE_MODE) {
		    [self plonkLMBTileAt:x_ :y_];
		}
		elif (not [tagged_list isEmpty]) {
		    double ux, uy;
		    int dx, dy;
		    BOOL virgin = YES;

		    dx = x_-last_mouse_x;
		    dy = y_-last_mouse_y;

		    foreach (it, tagged_list) {
			unit = [it getItem];
			[unit getX:&ux Y:&uy];

			if (virgin && snap_to_grid) {
			    virgin = NO;

			    dx = (int)(ux+dx+GRID_SIZE_2 * ((x_ > 0) ? 1 : -1))/GRID_SIZE*GRID_SIZE - ux;
			    dy = (int)(uy+dy+GRID_SIZE_2 * ((y_ > 0) ? 1 : -1))/GRID_SIZE*GRID_SIZE - uy;
			}

			if ([unit reditSetX:ux+dx Y:uy+dy])
			    flags |= FLAG_DIRTY;
		    }
		}
	    }

	    elif (data == 1) {	/* Right button. */
		if (editing_mode <= TILE_MODE) {
		    [self plonkRMBTileAt:x_ :y_];
		}
		else {
		    double xx, yy, theta;
		    
		    if (not tagged_list)
			break;
		    
		    /* Rotate units to mouse. */
		    foreach (it, tagged_list) {
			unit = [it getItem];
			
			[unit getX:&xx Y:&yy];
			theta = atan2(yy - y_, x_ - xx);
			theta = deg2rad((int)rad2deg(theta)); /* Bluddy dogdy. */
			[unit setAngle:theta];
		    }
		    
		    flags |= FLAG_DIRTY;
		}
            }

	    elif (data == 2) {	/* Middle button. */
		offset_x += (mouse_x - mouse_click_x) / 10;
		offset_y += (mouse_y - mouse_click_y) / 10;
	    }

	    last_mouse_x = x_;
	    last_mouse_y = y_;
	    break;

	case MSG_MOUSE_WHEEL:
	    offset_y += SCROLL_RATE * data;
	    break;

	case MSG_MOUSE_ENTER:	/* XXX: what's this for, multiple maps? */
	    current_map = the_map;
	    /* Fall through. */

	default:
	    [super receiveMessage:msg :data];
	    break;
    }

    /* Clamp the offsets so we don't go too far off the map. */
    offset_x = MID(-screen_w, offset_x, 2*screen_w - w);
    offset_y = MID(-14, offset_y, [the_map length] * TILE_H - h/2);

    if ((offset_x != prev_offset_x) || (offset_y != prev_offset_y)) {
        [the_map setOffsetX:offset_x Y:offset_y];
	[self offsetToHSlider:offset_x];
	[self offsetToVSlider:offset_y];
	flags |= FLAG_DIRTY;
    }
}

- (void) receiveMessage:(int)msg :(int)data fromChild:(Widget *)child
{
    int offset_x = [the_map offsetX];
    int offset_y = [the_map offsetY];
    (void)child;

    if (msg == MSG_HSLIDER)
	offset_x = [self HSliderToOffset:data];
    elif (msg == MSG_VSLIDER)
	offset_y = [self VSliderToOffset:data];

    [the_map setOffsetX:offset_x Y:offset_y];
    flags |= FLAG_DIRTY;
}

- (void) setSelectedLMBItem:(int)data
{
    if (editing_mode <= TILE_MODE)
	lmb_tile = get_tile_by_index(data);
    elif (editing_mode == GROUND_MODE)
	selected_ground_class = ground_units_table.class_data[data].class;
    elif (editing_mode == AIR_MODE)
	selected_air_class = air_units_table.class_data[data].class;
    elif (editing_mode == FORMATION_MODE)
	selected_formation = data;
}

- (void) setSelectedRMBItem:(int)data
{
    assert(editing_mode == TILE_MODE);
    rmb_tile = get_tile_by_index(data);
}

- (Class) getSelectedUnitClass
{
    if (editing_mode == GROUND_MODE)
	return selected_ground_class;
    else
	return selected_air_class;
}
@end


@implementation MapProc (Transform)
- (void) doResizeMap
{
    int new_height = [the_map length];

    start_autopoll_music();
    if (adime_dialogf("Resize Map",
		      ADIME_ALIGN_CENTRE, ADIME_ALIGN_CENTRE, 50,
		      "New map length (>=50):%int[50,]", &new_height) != 2) {
	[the_map resize:new_height];
	[self setScrollbarPrecision];
	flags |= FLAG_DIRTY;
    }
    stop_autopoll_music();
}

- (void) doPanMap
{
    int delta_y = 0;

    start_autopoll_music();
    if (adime_dialogf("Pan Map",
		      ADIME_ALIGN_CENTRE, ADIME_ALIGN_CENTRE, 100,
		      "Pan by n tiles:%int[,]", &delta_y) != 2) {
	[the_map pan:delta_y];
	shift_list(active_air_list,    0, delta_y * TILE_H);
	shift_list(active_ground_list, 0, delta_y * TILE_H);
	flags |= FLAG_DIRTY;
    }
    stop_autopoll_music();
}
@end


@implementation MapProc (Private)
- (void) setScrollbarPrecision
{
    if (3*screen_w > w)
	[hscrollbar setSliderPrecision:3*screen_w - w];
    [vscrollbar setSliderPrecision:[the_map length] * TILE_H - h/2];
}

- (void) plonkLMBTileAt:(const double)x_ :(const double)y_
{
    assert((editing_mode == PARALLAX_MODE) ||
	   (editing_mode == TILE_MODE));

    if (editing_mode == TILE_MODE) {
	if ([self placeTileAt:x_/TILE_W :y_/TILE_H to:lmb_tile])
	    flags |= FLAG_DIRTY;
    }
    else {
	if ([the_map setParallaxTileAt:x_/TILE_W :y_/TILE_H to:lmb_tile])
	    flags |= FLAG_DIRTY;
    }
}

- (void) plonkRMBTileAt:(const double)x_ :(const double)y_
{
    assert((editing_mode == PARALLAX_MODE) ||
	   (editing_mode == TILE_MODE));

    if (editing_mode == TILE_MODE) {
	if ([the_map setTileAt:x_/TILE_W :y_/TILE_H to:rmb_tile])
	    flags |= FLAG_DIRTY; 
    }
    else {
	if ([the_map setParallaxTileAt:x_/TILE_W :y_/TILE_H to:nil])
	    flags |= FLAG_DIRTY;
    }
}

- (BOOL) placeTileAt:(int)x0 :(int)y0 to:(Sebum<SebImage> *)tile
{
    int j;

    if (not [the_map setTileAt:x0 :y0 to:tile])
	return NO;

    if (not auto_plonk)
	return YES;

    /* Auto-plonking. */
    if ((j = find_tile_connection_set_by_name([tile name])) >= 0) {
	int xx, yy;

	for (yy = -1; yy <= 1; yy++) {
	    for (xx = -1; xx <= 1; xx++)
		[self autoPlaceAt:x0+xx :y0+yy :j];
	}
    }
    return YES;
}

- (void) autoPlaceAt:(const int)x0 :(const int)y0 :(const int)j
{
#define SET			connection_set[j]
#define IDX(x_,y_)		(tt_idx[y_+1][x_+1])
#define CONNECTION(x_,y_,c)	((IDX(x_,y_) >= 0) && (SET.data[IDX(x_,y_)].connect[c]))

    const char *tile_name;
    int tl, tr, bl, br;
    int xx, yy;
    unsigned int i;
    int tt_idx[3][3];

    if (x0 < 0 || x0 >= MAP_W_TILES ||
	y0 < 0 || y0 >= [the_map length])
	return;

    for (yy = -1; yy <= 1; yy++) {
	for (xx = -1; xx <= 1; xx++) {
	    Sebum<SebImage> *tt = [the_map getTileAt:x0+xx :y0+yy];

	    if (tt)
		IDX(xx,yy) = find_in_set([tt name], j);
	    else
		IDX(xx,yy) = -1;
	}
    }

    tile_name = rename_tile([[the_map getTileAt:x0 :y0] name]);
    if (IDX(0,0) < 0 &&	strneq(tile_name, SET.base_name))
	return;

    tl = (CONNECTION(-1, -1, 3) || CONNECTION(0, -1, 2) ||
	  CONNECTION(-1,  0, 1) || CONNECTION(0,  0, 0));
    tr = (CONNECTION( 0, -1, 3) || CONNECTION(1, -1, 2) ||
	  CONNECTION( 0,  0, 1) || CONNECTION(1,  0, 0));
    bl = (CONNECTION(-1,  0, 3) || CONNECTION(0,  0, 2) ||
	  CONNECTION(-1,  1, 1) || CONNECTION(0,  1, 0));
    br = (CONNECTION( 0,  0, 3) || CONNECTION(1,  0, 2) ||
	  CONNECTION( 0,  1, 1) || CONNECTION(1,  1, 0));

    /* Update with new tile. */
    for (i = 0; i < connection_set[j].n_items; i++) {
	if ((SET.data[i].connect[0] == !!tl) &&
	    (SET.data[i].connect[1] == !!tr) &&
	    (SET.data[i].connect[2] == !!bl) &&
	    (SET.data[i].connect[3] == !!br)) {
	    Sebum<SebImage> *tt = get_tile_by_name(SET.data[i].tile_name);

	    [the_map setTileAt:x0 :y0 to:tt];
	    return;
	}
    }

#undef CONNECTION
#undef IDX
#undef SET
}

- (void) floodfill
{
    int mx = mouse_x + offsetX;
    int my = mouse_y + offsetY;

    [the_map floodfillAt:mx/TILE_W :my/TILE_H :lmb_tile];
    flags |= FLAG_DIRTY;
}

- (void) processUnitModeLMBPressed:(const double)x_ :(const double)y_
{
    Unit *unit = findUnitAt(x_, y_);

    if (unit) {
	/* Shift-click toggles this unit's tagged status. */
	if (key_shifts & KB_SHIFT_FLAG) {
	    if (not [tagged_list removeItem:unit])
		[tagged_list insertItem:unit];
	}
	/* CTRL-click selects the clicked unit class. */
	elif (key_shifts & KB_CTRL_FLAG) {
	    if (editing_mode == GROUND_MODE)
		selected_ground_class = [unit class];
	    elif (editing_mode == AIR_MODE)
		selected_air_class = [unit class];
	}
	/* Plain-click replaces the tagged units to current. */
	else {
	    [tagged_list removeAllItems];
	    [tagged_list insertItem:unit];
	}
    }

    else {
	/* Non-shift-click replaces tagged list. */
	if (not (key_shifts & KB_SHIFT_FLAG))
	    [tagged_list removeAllItems];

	if (editing_mode == GROUND_MODE) {
	    unit = spawn_unit(selected_ground_class, x_, y_, ACTIVE_GROUND_LIST, YES);
	    [tagged_list insertItem:unit];
	}
	elif (editing_mode == AIR_MODE) {
	    unit = spawn_unit(selected_air_class, x_, y_, ACTIVE_AIR_LIST, YES);
	    [tagged_list insertItem:unit];
	}
	else {	/* Formations */
	    unsigned int i = 0;

	    while ((unit = spawn_formation_interate(selected_formation, i, x_, y_))) {
		[tagged_list insertItem:unit];
		i++;
	    }
	}
    }

    flags |= FLAG_DIRTY;
}

/*--------------------------------------------------------------*/

- (void) specialMapHacks
{
    char title[MAX_MAP_TITLE_LENGTH] = "";
    char creator[MAX_MAP_CREATOR_LENGTH] = "";
    int stars = [the_map stars];
    int flavour = toothpaste_flavour;

    [the_map getTitle:title Creator:creator];

    if (adime_dialogf("Map Details",
		      ADIME_ALIGN_CENTRE, ADIME_ALIGN_CENTRE, 100,
		      "Title:%string[32]"
		      "Creator:%string[32]\n"
		      "Stars:%bool[]"
		      "Toothpaste:%bool[]",
		      &title, &creator, &stars, &flavour) != 2) {
	[the_map setTitle:title Creator:creator];
	[the_map setStars:stars];
	toothpaste_set_flavour((flavour == 0) ? 0 : 1);
    }
}

- (void) customiseDisplay
{
    int snap = snap_to_grid;
    struct map_proc_visuals vis = visable;

    if (adime_dialogf("Customise Display",
		      ADIME_ALIGN_CENTRE, ADIME_ALIGN_CENTRE, 100,
		      "Air units:%bool[]"
		      "Ground units:%bool[]"
		      "\n"
		      "Grid snap:%bool[]"
		      "Horizontal rulers:%bool[]"
		      "Vertical rulers:%bool[]",
		      &vis.air_units, &vis.ground_units,
		      &snap,
		      &vis.hruler, &vis.vruler) != 2) {
	snap_to_grid = snap;
	visable = vis;
    }

    flags |= FLAG_DIRTY;
}

- (void) drawGrid:(BITMAP *)dest
{
    int c1 = makecol(0xff, 0xc0, 0x00);
    int c2 = makecol(0x40, 0x40, 0x40);
    int xx, yy, x_, y_;

    if (not snap_to_grid)
	return;

    xx = (int)[the_map offsetX] % GRID_SIZE;
    yy = (int)[the_map offsetY] % GRID_SIZE;

    for (y_ = -yy; y_ < y + h; y_ += GRID_SIZE)
	for (x_ = -xx; x_ < x + w; x_ += GRID_SIZE) {
	    putpixel(dest, x_,   y_,   c1);
	    putpixel(dest, x_-1, y_,   c2);
	    putpixel(dest, x_+1, y_,   c2);
	    putpixel(dest, x_,   y_-1, c2);
	    putpixel(dest, x_,   y_+1, c2);
	}
}

- (void) drawRulers:(BITMAP *)dest
{
    int c = makecol(0xff, 0x80, 0x80);

    if (visable.vruler) {
	vline(dest, x-offsetX-160, y, y+h, c);
	vline(dest, x-offsetX, y, y+h, c);
	vline(dest, x-offsetX + screen_w/2, y, y+h, c);
	vline(dest, x-offsetX + screen_w, y, y+h, c);
	vline(dest, x-offsetX + screen_w+160, y, y+h, c);
    }

    if (visable.hruler && (h > screen_h)) {
        int yy = y + h/2;
        hline(dest, x, yy - screen_h/2, x+w-1, c);
        hline(dest, x, yy + screen_h/2, x+w-1, c);
    }
}

- (void) setActivationLine
{
    int i = 0, abs = 1;
    ListIterator *it;
    Unit *unit;

    start_autopoll_music();
    if (adime_dialogf("Set Activation Line",
		      ADIME_ALIGN_CENTRE, ADIME_ALIGN_CENTRE, 100,
		      "Activation Line (-480, 480):%int[-480,480]"
		      "Absolute:%bool[]",
		      &i, &abs) != 2) {
	foreach (it, tagged_list) {
	    unit = [it getItem];
	    if ([unit conformsTo:@protocol(VariableActivationLine)])
		[(<VariableActivationLine>)unit setActivationLine:i :abs];
	}
    }
    stop_autopoll_music();
}

- (void) setTravelRange
{
    int i = 0;
    ListIterator *it;
    Unit *unit;
    
    start_autopoll_music();
    if (adime_dialogf("Set Travel Range",
		      ADIME_ALIGN_CENTRE, ADIME_ALIGN_CENTRE, 100,
		      "Range:%int[0,1000]", &i) != 2) {
	foreach (it, tagged_list) {
	    unit = [it getItem];
	    if ([unit conformsTo:@protocol(VariableTravelRange)])
		[(<VariableTravelRange>)unit setTravelRange:i];
	}
    }
    stop_autopoll_music();
}

- (void) setGroup
{
    int i;
    ListIterator *it;
    Unit *unit;

    /* Find first unit which can be grouped, and take it's y value as
       the default group number. */
    foreach (it, tagged_list) {
	unit = [it getItem];
	if ([unit conformsTo:@protocol(Groupable)]) {
	    double x_, y_;
	    [unit getX:&x_ Y:&y_];
	    i = y_;
	    break;
	}
    }

    start_autopoll_music();
    if (adime_dialogf("Set Group",
		      ADIME_ALIGN_CENTRE, ADIME_ALIGN_CENTRE, 100,
		      "Group number:%int[,]", &i) != 2) {
	foreach (it, tagged_list) {
	    unit = [it getItem];
	    if ([unit conformsTo:@protocol(Groupable)])
		[(<Groupable>)unit setGroup:i];
	}
    }
    stop_autopoll_music();
}

- (void) setScrollRate
{
    float rate;
    ListIterator *it;
    Unit *unit = nil;

    /* Find first unit which can set scroll rate, which is the one to
       set; you should only be setting one at a time. */
    foreach (it, tagged_list) {
	unit = [it getItem];
	if ([unit conformsTo:@protocol(VariesScrollRate)])
	    goto found_unit;
    }
    return;

 found_unit:
    start_autopoll_music();
    if (adime_dialogf("Set Scroll Rate",
		      ADIME_ALIGN_CENTRE, ADIME_ALIGN_CENTRE, 100,
		      "Rate:%float[,]", &rate) != 2) {
	[(<VariesScrollRate>)unit setScrollRate:rate];
    }
    stop_autopoll_music();
}

- (void) togglePowerups
{
    ListIterator *it;
    int p1 = 0, p2 = 0, p3 = 0, p4 = 0, p5 = 0;
    int u = 0;

    /* Find the majority's powerup spawning flag for each powerup. */
    foreach (it, tagged_list) {
	int f = [[it getItem] flags];

	if (f & FLAG_DEATH_SPAWN_PRIMARY_POWERUP) p1++;
	if (f & FLAG_DEATH_SPAWN_SECONDARY_POWERUP) p2++;
	if (f & FLAG_DEATH_SPAWN_TERTIARY_POWERUP) p3++;
	if (f & FLAG_DEATH_SPAWN_NUKE_POWERUP) p4++;
	if (f & FLAG_DEATH_SPAWN_HEALTH_POWERUP) p5++;
	u++;
    }
    p1 = p1*2/u;
    p2 = p2*2/u;
    p3 = p3*2/u;
    p4 = p4*2/u;
    p5 = p5*2/u;

    if (adime_dialogf("Powerups",
		      ADIME_ALIGN_CENTRE, ADIME_ALIGN_CENTRE, 100,
		      "Primary:%bool[]"
		      "Secondary:%bool[]"
		      "Tertiary:%bool[]"
		      "Nuke:%bool[]"
		      "Health:%bool[]",
		      &p1, &p2, &p3, &p4, &p5) != 2) {
	int f = ((p1 ? FLAG_DEATH_SPAWN_PRIMARY_POWERUP : 0)	|
		 (p2 ? FLAG_DEATH_SPAWN_SECONDARY_POWERUP : 0)	|
		 (p3 ? FLAG_DEATH_SPAWN_TERTIARY_POWERUP : 0)	|
		 (p4 ? FLAG_DEATH_SPAWN_NUKE_POWERUP : 0)	|
		 (p5 ? FLAG_DEATH_SPAWN_HEALTH_POWERUP : 0));

	foreach (it, tagged_list) {
	    int ff = [[it getItem] flags];
	    [[it getItem] toggleFlag:(ff&FLAG_DEATH_SPAWN_POWERUP) ^ f];
	}
    }
}
@end

/*--------------------------------------------------------------*/

@interface Phony: Projectile
@end

@implementation Phony
- init
{
    alliance = YES;
    return self;
}

- (void) getX:(double *)x_ Y:(double *)y_ W:(int *)w_ H:(int *)h_;
{
    if (x_) *x_ = x;
    if (y_) *y_ = y;
    if (w_) *w_ = 1;
    if (h_) *h_ = 1;
}
@end

static Unit *unit_in_contact_with(Thing<DetectsCollision> *object, List *list)
{
    ListIterator *it;

    foreach (it, list) {
	Unit *unit = [it getItem];

	/* Use "obj collidesWith:unit" so that projectiles will
	   determine if a collision takes place.  This allows railguns
	   to do stuff. */
	if ([object collidesWith:unit]) {
	    [it free];
	    return unit;
	}
    }

    return nil;
}

Unit *findUnitAt(double x, double y)
{
    Phony *phony = [[Phony new] setX:x Y:y];
    Unit *unit;

    if (editing_mode == GROUND_MODE)
	unit = unit_in_contact_with(phony, active_ground_list);
    else
	unit = unit_in_contact_with(phony, active_air_list);

    [phony free];
    return unit;
}
