/* main.m,
 *
 * This just processes command line arguments.  See game.m for the
 * game loop, and map-editor/redit.m for editor loop.
 */

#include <allegro.h>
#include <assert.h>
#include <getopt.h>
#include <objc/Object.h>
#include <string.h>
#include <sys/stat.h>
#include "common.h"
#include "config.h"
#include "debug.inc"
#include "demo.h"
#include "difficulty.h"
#include "game.h"
#include "help.h"
#include "hiscore.h"
#include "init.h"
#include "input.h"
#include "map-editor/redit.h"
#include "music-ad.h"
#include "newmenu.h"
#include "player.h"
#include "projectiles/all-projs.h"
#include "seborrhea/seborrhea.h"
#include "seborrhea/seborrhea-datadir.h"
#include "units/all-units.h"


#ifdef DEBUG_MALLOC_TRACE

/*------------------------------------------------------------*/
/* MTrace.  Note: some memory leaks out of my control.        */
/*------------------------------------------------------------*/

#include <mcheck.h>

@interface mtraceInit
@end

@implementation mtraceInit
+ (void) load
{
    setenv("MALLOC_TRACE", "./memtrace.log", 0);
    mtrace();
}
@end

#endif

static void close_window_hook(void)
{
    game_flags |= FLAG_QUIT;
    emergency_exit = YES;		/* Menus. */
}

/*--------------------------------------------------------------*/

static void add_data_dirs(void)
{
    char *path;
    char tmp[PATH_MAX];

    /* Create ~/.raidem if it doesn't exist.  We will assume it does
       after this point. */
    {
	struct stat stats;

	snprintf(tmp, sizeof tmp, "%s/%s", homedir, RAIDEM_CONFIG_PATHNAME);

	if (stat(tmp, &stats) != 0) {
#ifdef __MINGW__
	    int mk = mkdir(tmp);
#else  /* Linux */
	    int mk = mkdir(tmp, S_IREAD|S_IWRITE|S_IEXEC);
#endif

	    if (mk != 0)
		fprintf(stderr, "Error creating '%s'\n", tmp);
	}
    }

    /* 'RAIDEM_DATA_DIR' environment variable. */
    path = getenv("RAIDEM_DATA_DIR");
    if (path) {
	/* Use the old name for compatability with Allegro 4.0.x */
	fix_filename_path(tmp, path, sizeof(tmp));
	seborrhea_add_root_directory(tmp);
    }

    /* The path in which Raid'em is currently placed. */
    get_executable_name(tmp, sizeof(tmp));
    path = strrchr(tmp, '/');
    if (path) {
	*path = '\0';
	seborrhea_add_root_directory(tmp);
	raid_binary_directory = strdup(tmp);
    }

#ifdef __unix__
    /* 'RAID_DATA_DIR' defined by Makefile. */
    seborrhea_add_root_directory(RAIDEM_DATA_DIR);
#endif

    /* Current directory. */
    seborrhea_add_root_directory("./");
    if (not raid_binary_directory)
	raid_binary_directory = strdup("./");
}


static int main_init(void)
{
    if (game_init() < 0)
	return -1;

    set_window_title(GAME_NAME_AND_VERSION);
    set_window_close_hook(&close_window_hook);
    return 0;
}


static void main_shutdown(void)
{
    game_shutdown();
}

/*--------------------------------------------------------------*/

int main(int argc, char *argv[])
{
    const char *load_filename = NULL;
    BOOL run_editor = NO, mute = NO;
    int gfx_mode = GFX_AUTODETECT, w = 640, h = 480, bpp = 0;
    int c;

    struct option long_opts[] = {	/* Long options. */
	{ "difficulty",	    required_argument,  0, 'D' },
	{ "jumpstart",	    required_argument,  0, 'J' },
	{ "nodemo",	    no_argument,	0, 'r' },
	{ "fs",		    no_argument,	0, 'f' },
	{ "fullscreen",	    no_argument,	0, 'f' },
	{ "win",	    no_argument,	0, 'w' },
	{ "windowed",	    no_argument,	0, 'w' },
	{ "depth",	    required_argument,	0, 'd' },

	{ "pageflipping",   no_argument,	0, 'p' },
	{ "nopageflipping", no_argument,	0, 'P' },
	{ "triplebuffer",   no_argument,	0, 't' },
	{ "notriplebuffer", no_argument,	0, 'T' },

	{ "nojoy",	    no_argument,	0, 'j' },
	{ "nojoystick",	    no_argument,	0, 'j' },
	{ "mute",	    no_argument,	0, 'm' },
	{ "noads",	    no_argument,	0, 'a' },
	{ "edit",	    optional_argument,	0, 'e' },
	{ "width",	    required_argument,	0, 'W' },
	{ "height",	    required_argument,	0, 'H' },
	{ "help",	    no_argument,	0, 'h' },
	{ "version",	    no_argument,	0, 'v' },
	{ 0, 0, 0, 0 }
    };

    /* Find 'HOME'. */
    homedir = getenv(HOME_PATH_ENV);
    read_config_file(&w, &h, &bpp);

    /* Command line options. */
    while ((c = getopt_long(argc, argv, "D:J:12rfwd:pPtTjmae::W:H:hv", long_opts, NULL)) != -1) {
	switch (c) {
	  case 'D': {
	      int diff = atoi(optarg);
	      if ((diff < DIFFICULTY_EASY) ||
		  (diff > DIFFICULTY_HARDEST)) {
		  fprintf(stderr, "Difficulty outside range!\n");
	      }
	      else {
		  difficulty = diff;
	      }

	      break;
	  }
	  case 'J': load_filename = optarg; break;
	  case '1': num_players = 1; break;
	  case '2': num_players = 2; break;
	  case 'r': record_demos = NO; break;

	      /* Video. */
	  case 'f': gfx_mode = GFX_AUTODETECT_FULLSCREEN; break;
	  case 'w': gfx_mode = GFX_AUTODETECT_WINDOWED; break;
	  case 'd':
	      bpp = atoi(optarg);
	      if ((bpp != 15) && (bpp != 16) && (bpp != 24) && (bpp != 32)) {
		  fprintf(stderr, "%d bpp not supported!\n", bpp);
		  bpp = 0;
	      }
	      break;

#ifdef NO_VIDEO_BITMAPS
	  case 'p':
	  case 'P':
	  case 't':
	  case 'T': break;
#else
	  case 'p': raid_use_page_flipping = YES; break;
	  case 'P': raid_use_page_flipping =  NO; break;
	  case 't': raid_use_triple_buffering = YES; break;
	  case 'T': raid_use_triple_buffering =  NO; break;
#endif

	      /* Input. */
	  case 'j': joy_enabled = NO; break;

	      /* Audio. */
	  case 'm': mute = YES; break;
#ifdef NO_ID3TAGS
	  case 'a': break;
#else
	  case 'a': music_ad_enabled = NO; break;
#endif

	  case 'e':
	      run_editor = YES;
	      record_demos = NO;
	      if (optarg)
		  load_filename = optarg;
	      break;
	  case 'W': w = atoi(optarg); break;
	  case 'H': h = atoi(optarg); break;

	  case '?':		/* Fall through. */
	  case 'h': display_help(); return 0;
	  case 'v': display_version(); return 0;
	}
    }

    if (allegro_init() != 0) {
	fprintf(stderr, "Error initialising Allegro.\n%s.\n", allegro_error);
	return -1;
    }

    add_data_dirs();

    if (run_editor) {
	/* Just in case we jump from editor to game. */
	rng_state = alrand_create(alrand_type_mersenne_twister, 0);
	assert(rng_state);

	if (raid_init(gfx_mode, w, h, bpp, mute) < 0)
	    goto exit;
	if (redit_init() < 0)
	    goto exit;

	redit_loop(load_filename);
	redit_shutdown();

	alrand_destroy(rng_state);
    }
    else {
	rng_state = alrand_create(alrand_type_mersenne_twister, 0);
	assert(rng_state);

	if (raid_init(gfx_mode, 640, 480, bpp, mute) < 0)
	    goto exit;
	if (main_init() < 0)
	    goto exit;

	read_high_scores();

	if (load_filename)
	    raid_start_levels(&load_filename, 1, NO, NO);
	else
	    do_newmenu();

	write_high_scores();
	main_shutdown();

	alrand_destroy(rng_state);
    }

 exit:
    raid_shutdown();

    write_config_file(w, h, bpp);
    unload_unnecessary_projectile_data();
    unload_unnecessary_unit_data();
    seborrhea_free_root_directories();

    free(raid_binary_directory);

    return 0;
}
END_OF_MAIN()
