/* input.m,
 *
 * This file handles players' controls.
 */

#include <allegro.h>
#include <assert.h>
#include <objc/Object.h>
#include "common.h"
#include "config.h"
#include "demo.h"
#include "input-joystick.h"
#include "input-keyboard.h"
#include "input.h"
#include "map.h"
#include "player.h"

/* Using the tokens to read from config files. */
#include "seborrhea/token.h"


#define MAX_SPEED		3.2
#define JOYSTICK_RANGE		96


KeyboardController *menu_keyboard_controls;
JoystickController *joystick_controller[MAX_JOYSTICKS];
player_controls_t player_controls[MAX_PLAYERS];

/*--------------------------------------------------------------*/

static void read_input_from_file(double *xv, double *yv, BOOL *fire, MEMORY_FILE *fp)
{
    mmpk_fread(xv, sizeof *xv, fp);
    mmpk_fread(yv, sizeof *yv, fp);
    *fire = mmpk_getc(fp);
}

static void read_input_from_controller(double *xv, double *yv, BOOL *fire, int pl)
{
    JoystickController *j = player_controls[pl].joystick;
    KeyboardController *k = player_controls[pl].keyboard;
    double xx = 0.0, yy = 0.0;

    /* Joystick, hopefully analog. */
    if (joy_enabled) {
	xx = [j xPosition];
	yy = [j yPosition];
	xx = (xx) * MAX_SPEED / JOYSTICK_RANGE;
	yy = (yy) * MAX_SPEED / JOYSTICK_RANGE;
	*fire = [j held:RAID_KEY_FIRE];
    }

    /* Keyboard controls, if you are not lucky enough to possess a
       Sony Dual-Shock 2 joystick and psx-to-usb adapter. */
    if ([k held:RAID_KEY_LEFT])  xx -= MAX_SPEED;
    if ([k held:RAID_KEY_RIGHT]) xx += MAX_SPEED;
    if ([k held:RAID_KEY_UP])    yy -= MAX_SPEED;
    if ([k held:RAID_KEY_DOWN])  yy += MAX_SPEED;
    if ([k held:RAID_KEY_FIRE])  *fire = YES;

    *xv = MID(-MAX_SPEED, xx, MAX_SPEED);
    *yv = MID(-MAX_SPEED, yy, MAX_SPEED);
}

static void do_input_for_player(int pl)
{
    double xv = 0.0, yv = 0.0;
    BOOL fire = NO;

    if ((demo_state == DEMO_PLAYBACK) ||
	(demo_state == DEMO_PLAYBACK_AUTOMATIC)) {
	read_input_from_file(&xv, &yv, &fire, &demo_file);
    }
    else {
	read_input_from_controller(&xv, &yv, &fire, pl);

	/* Demos. */
	if (demo_state == DEMO_RECORDING) {
	    pack_fwrite(&xv, sizeof xv, current_demo_file);
	    pack_fwrite(&yv, sizeof yv, current_demo_file);
	    pack_putc(fire, current_demo_file);
	}
    }

    [player[pl] increaseVelocity:xv :yv];

    if (fire)
        [player[pl] openFire];
    else
        [player[pl] haltFire];
}

/* Join pressed: used to switch from 1 player mode to 2 player.
   Trigger by pressing FIRE for player 2 or START joystick. */
static BOOL join_pressed(void)
{
    if (num_players >= 2)
	return NO;

    if ([player_controls[1].keyboard pressed:RAID_KEY_FIRE])
	return YES;

    if (joy_enabled && player_controls[1].joystick) {
	return ([player_controls[1].joystick pressed:RAID_KEY_START] ||
		[player_controls[1].joystick pressed:RAID_KEY_FIRE]);
    }

    return NO;
}

void do_input(void)
{
    int pl;

    /* No need to poll since handled in poll_raid_keys() */
    for (pl = 0; pl < MAX_PLAYERS; pl++) {
	if (player[pl])
	    do_input_for_player(pl);
    }

    /* Player 2 joins the action. */
    if (num_players < MAX_PLAYERS && game_flags & FLAG_PLAYERS_ALIVE) {
	BOOL join;

	if ((demo_state == DEMO_PLAYBACK) ||
	    (demo_state == DEMO_PLAYBACK_AUTOMATIC))
	    join = mmpk_getc(&demo_file);
	else
	    join = join_pressed();

	if (join) {
	    num_players = 2;
	    player[1] = (Player *)spawn_unit([Player2 class],
					     screen_w*2/3, offsetY+screen_h,
					     ALLY_LIST, YES);
	    record_demo_checkpoint = YES;
	}

	if (demo_state == DEMO_RECORDING)
	    pack_putc(join, current_demo_file);
    }
}

/*--------------------------------------------------------------*/

void poll_raid_keys(int n_players)
{
    [menu_keyboard_controls poll];
    [player_controls[0].keyboard poll];
    [player_controls[1].keyboard poll];

    if (joy_enabled) {
	poll_joystick();

	if (n_players == 0) {		/* Poll all joysticks. */
	    int j;

	    for (j = 0; j < num_joysticks && joystick_controller[j]; j++)
		[joystick_controller[j] poll];
	}
	else {
	    if (player_controls[0].joystick)
		[player_controls[0].joystick poll];
	    if (player_controls[1].joystick)
		[player_controls[1].joystick poll];
	}
    }
}

/* Start pressed: used to (un-)pause game. 
   Trigger by pressing START. */
BOOL start_pressed(void)
{
    /* All keyboard start buttons are the same, so just check one. */
    if ([player_controls[0].keyboard pressed:RAID_KEY_START])
	return YES;

    if (joy_enabled) {
	return ((player_controls[0].joystick && [player_controls[0].joystick pressed:RAID_KEY_START]) ||
		(player_controls[1].joystick && [player_controls[1].joystick pressed:RAID_KEY_START]));
    }

    return NO;
}

/* Start released: used to finish entering hiscore.
   Trigger by releasing ENTER or START. */
BOOL start_released(void)
{
    /* All keyboard start buttons are the same, so just check one. */
    if ([player_controls[0].keyboard released:RAID_KEY_START])
	return YES;

    if (joy_enabled) {
	return ((player_controls[0].joystick && [player_controls[0].joystick released:RAID_KEY_START]) ||
		(player_controls[1].joystick && [player_controls[1].joystick released:RAID_KEY_START]));
    }

    return NO;
}

/* Retry released: used for restarting.
   Trigger by pressing START or SPACEBAR. */
BOOL retry_released(void)
{
    return (key[KEY_SPACE] || start_pressed());
}

/* Okay released: used by menu.
   Trigger by releasing START (and/or FIRE on ANY joystick). */
BOOL okay_released(void)
{
    /* Enter, but not alt-enter. */
    if ([menu_keyboard_controls released:RAID_KEY_START] &&
	not (key_shifts & KB_ALT_FLAG))
	return YES;

    if (joy_enabled) {
	int j;

	for (j = 0; j < num_joysticks && joystick_controller[j]; j++) {
	    if ([joystick_controller[j] released:RAID_KEY_START] ||
		[joystick_controller[j] released:RAID_KEY_FIRE])
		return YES;
	}
    }

    return NO;
}

/* Menu released: used by menu, and to pause/leave game.
   Trigger by releasing ESC/MENU on ANY joystick. */
BOOL menu_released(void)
{
    /* All keyboard menu keys are the same, so just check one. */
    if ([menu_keyboard_controls released:RAID_KEY_MENU])
	return YES;

    if (joy_enabled) {
	int j;

	for (j = 0; j < num_joysticks && joystick_controller[j]; j++)
	    if ([joystick_controller[j] released:RAID_KEY_MENU])
		return YES;
    }

    return NO;
}

/*--------------------------------------------------------------*/

int input_init(void)
{
    install_keyboard();
    set_keyboard_rate(0, 1);

    /* For reading config files. */
    set_token_reader_stdio();

    /* main.m sets joy_enabled=NO to disable joystick support.  If
       joy_enabled=YES, then attempt to install joystick. */
    if (joy_enabled) {
	if (install_joystick(JOY_TYPE_AUTODETECT) == 0) {
	    int j;
	    FILE *fp;
	    
	    fprintf(stdout, "%d joystick(s) detected.\n", num_joysticks);
	    fp = fopen_cfg(JOYSTICK_CFG, "r");

	    for (j = 0; j < num_joysticks; j++) {
		joystick_controller[j] = [[JoystickController alloc]
					     initWithJoystickNumber:j
					     ConfigFile:fp];
	    }

	    if (fp)
		fclose(fp);
	}
	else
	    fprintf(stderr, "Warning: could not initialize joystick(s).\n");
    }

    {
	KeyboardController *k;
	FILE *fp = NULL;

	fp = fopen_cfg(KEYBOARD_CFG, "r");

	/* Player 1. */
	k = [[KeyboardController alloc] initWithConfigFile:fp :"keyboard0"];
	assert(k);
	player_controls[0].keyboard = k;
	player_controls[0].joystick = joystick_controller[0];


	/* Player 2. */
	k = [[KeyboardController alloc] initWithConfigFile:fp :"keyboard1"];
	assert(k);
	player_controls[1].keyboard = k;
	player_controls[1].joystick = joystick_controller[1];

	if (fp)
	    fclose(fp);
    }

    {					/* Menu controls. */
	menu_keyboard_controls = [KeyboardController new];
	assert(menu_keyboard_controls);
	[menu_keyboard_controls setKey:RAID_KEY_UP    To:KEY_UP];
	[menu_keyboard_controls setKey:RAID_KEY_DOWN  To:KEY_DOWN];
	[menu_keyboard_controls setKey:RAID_KEY_LEFT  To:KEY_LEFT];
	[menu_keyboard_controls setKey:RAID_KEY_RIGHT To:KEY_RIGHT];
    }

    return 0;
}


static void input_save_config(void)
{
    FILE *fp = NULL;
    int j;
    
    fp = fopen_cfg(KEYBOARD_CFG, "w");
    if (fp) {
	fprintf(fp, KEYBOARD_CFG_HEADER);
	[player_controls[0].keyboard writeConfigFile:fp :"keyboard0"];
	[player_controls[1].keyboard writeConfigFile:fp :"keyboard1"];
	fclose(fp);
    }

    fp = fopen_cfg(JOYSTICK_CFG, "w");
    if (fp) {
	char section[8];
	fprintf(fp, JOYSTICK_CFG_HEADER);

	for (j = 0; j < num_joysticks; j++) {
	    snprintf(section, sizeof section, "joy%d", j);
	    [joystick_controller[j] writeConfigFile:fp :section];
	}

	fclose(fp);
    }
}


void input_shutdown(void)
{
    int j;

    input_save_config();

    menu_keyboard_controls = [menu_keyboard_controls free];
    player_controls[0].keyboard = [player_controls[0].keyboard free];
    player_controls[1].keyboard = [player_controls[1].keyboard free];

    for (j = 0; j < num_joysticks; j++)
	joystick_controller[j] = [joystick_controller[j] free];
}
