/* input-keyboard.m,
 *
 * Syntax in keyboard config file is:
 *
 * (keyboard<num>
 *  (up-key <key>)
 *  (down-key <key>) ...)
 */

#include <allegro.h>
#include <assert.h>
#include "common.h"
#include "input-keyboard.h"


@interface KeyboardController (Private)
- (void) resetToDefaults:(const int)n;
@end

/*--------------------------------------------------------------*/

@implementation KeyboardController
- initWithConfigFile:(FILE *)fp :(const char *)section
{
    [self init];

    if (streq(section, "keyboard0"))
	[self resetToDefaults:0];
    else
	[self resetToDefaults:1];

    if ((not fp) ||
	(not [self seekInFile:KEYBOARD_CFG :fp For:section]))
	return self;

    /* Note: warnings are displayed by this function. */
    [self readConfigFile:fp :(const char *)section];
    return self;
}

- (void) pollUpdatePressed
{
    pressed[RAID_KEY_UP]    = key[up];
    pressed[RAID_KEY_DOWN]  = key[down];
    pressed[RAID_KEY_LEFT]  = key[left];
    pressed[RAID_KEY_RIGHT] = key[right];
    pressed[RAID_KEY_START] = key[KEY_ENTER] || key[KEY_ENTER_PAD];
    pressed[RAID_KEY_MENU]  = key[KEY_ESC];
    pressed[RAID_KEY_FIRE]  = key[fire[0]] || key[fire[1]];
}

- (void) setKey:(raid_key_t)k To:(int)k_
{
    if (k == RAID_KEY_UP) up = k_;
    elif (k == RAID_KEY_DOWN) down = k_;
    elif (k == RAID_KEY_LEFT) left = k_;
    elif (k == RAID_KEY_RIGHT) right = k_;
    elif (k == RAID_KEY_FIRE) fire[0] = k_;
    elif (k == RAID_KEY_FIRE2) fire[1] = k_;

    if (fire[0] == fire[1])
	fire[1] = 0;
}

- (int) getKey:(raid_key_t)k
{
    if (k == RAID_KEY_UP) return up;
    elif (k == RAID_KEY_DOWN) return down;
    elif (k == RAID_KEY_LEFT) return left;
    elif (k == RAID_KEY_RIGHT) return right;
    elif (k == RAID_KEY_FIRE) return fire[0];
    elif (k == RAID_KEY_FIRE2) return fire[1];
    else
	return -1;
}
@end


@implementation KeyboardController (Private)
- (void) resetToDefaults:(const int)n
{
    if (n == 0) {
	up      = KEY_UP;
	down    = KEY_DOWN;
	left    = KEY_LEFT;
	right   = KEY_RIGHT;
	fire[0] = KEY_LSHIFT;
	fire[1] = KEY_LCONTROL;
    }
    else {
	up      = KEY_8_PAD;
	down    = KEY_2_PAD;
	left    = KEY_4_PAD;
	right   = KEY_6_PAD;
	fire[0] = KEY_RSHIFT;
	fire[1] = KEY_RCONTROL;
    }
}
@end


@implementation KeyboardController (Config)
- (void) readConfigFile:(FILE *)fp :(const char *)section
{
#define SET_KEY(k)	[self setKey:k To:[ktok getFixnum]]

    assert(fp);

    while (expect_char(fp, '(')) {
	Token *ctok, *ktok;
	const char *cmd;

	ctok = read_symbol(fp);
	if (not ctok){
	    fprintf(stderr,
		    "[Input] Buggy " KEYBOARD_CFG " file (%s).  "
		    "Missing command token.\n", section);
	    return;
	}
	cmd = [ctok getSymbol];

	ktok = read_fixnum(fp);
	if (not ktok) {
	    fprintf(stderr,
		    "[Input] Buggy " KEYBOARD_CFG " file (%s:%s).  "
		    "Missing argument.\n", section, cmd);
	    ctok = [ctok free];
	    return;
	}

	if (not expect_char(fp, ')')) {
	    fprintf(stderr,
		    "[Input] Buggy " KEYBOARD_CFG " file (%s:%s).  "
		    "Missing ).\n", section, cmd);
	    ktok = [ktok free];
	    ctok = [ctok free];
	    return;
	}

	/* Process the commands. */
	if (streq(cmd, "up-key"))	SET_KEY(RAID_KEY_UP);
	elif (streq(cmd, "down-key"))	SET_KEY(RAID_KEY_DOWN);
	elif (streq(cmd, "left-key"))	SET_KEY(RAID_KEY_LEFT);
	elif (streq(cmd, "right-key"))	SET_KEY(RAID_KEY_RIGHT);
	elif (streq(cmd, "fire-key"))	SET_KEY(RAID_KEY_FIRE);
	elif (streq(cmd, "fire-key2"))	SET_KEY(RAID_KEY_FIRE2);
	else {
	    fprintf(stderr,
		    "[Input] Buggy " KEYBOARD_CFG " file (%s:%s).  "
		    "Unknown command.\n", section, cmd);
	}

	ktok = [ktok free];
	ctok = [ctok free];
    }

#undef SET_KEY
}

- (void) writeConfigFile:(FILE *)fp :(const char *)section
{
    assert(fp);

    fprintf(fp,
	    "(%s\n"
	    " (up-key\t%d)\n"
	    " (down-key\t%d)\n"
	    " (left-key\t%d)\n"
	    " (right-key\t%d)\n"
	    " (fire-key\t%d)\n"
	    " (fire-key2\t%d))\n",
	    section, up, down, left, right, fire[0], fire[1]);
}
@end
