/* input-joystick.m,
 *
 * Syntax in keyboard config file is:
 *
 * (joy<num>
 *  (name <string>)
 *  (start-button <button>)
 *  (menu-button <button>)
 *  (primary-stick <stick>)
 *  (ignore-stick <stick>))
 */

#include <assert.h>
#include "common.h"
#include "input-joystick.h"


BOOL joy_enabled = YES;

/*--------------------------------------------------------------*/

@interface JoystickController (Private)
- (int) guessStartButton;
- (int) guessMenuButton;
- (void) resetToDefaults;
@end

/*--------------------------------------------------------------*/

@implementation JoystickController
- initWithJoystickNumber:(int)n ConfigFile:(FILE *)fp
{
    char stick_string[8];
    assert(0 <= n && n < MAX_JOYSTICKS);

    [self init];

    joy_num = n;
    j = &joy[joy_num];
    snprintf(name, sizeof name, "Joy %d (%d buttons)",
	     joy_num+1, j->num_buttons);
    
    fprintf(stdout, "Joystick %d has %d sticks, %d buttons.\n",
	    joy_num+1, j->num_sticks, j->num_buttons);

    [self resetToDefaults];

    snprintf(stick_string, sizeof stick_string, "joy%d", joy_num);
    if ((not fp) ||
	(not [self seekInFile:JOYSTICK_CFG :fp For:stick_string]))
	return self;

    [self readConfigFile:fp :stick_string];
    return self;
}

- (const char *) name
{
    return name;
}

- (int) joystickNumber
{
    return joy_num;
}

- (void) pollUpdatePressed
{
    int b, s;

    if (not joy_enabled) {
	pressed[RAID_KEY_UP]    = NO;
	pressed[RAID_KEY_DOWN]  = NO;
	pressed[RAID_KEY_LEFT]  = NO;
	pressed[RAID_KEY_RIGHT] = NO;
	pressed[RAID_KEY_START] = NO;
	pressed[RAID_KEY_MENU]  = NO;
	pressed[RAID_KEY_FIRE]  = NO;
	return;
    }

    pressed[RAID_KEY_UP]    = NO;
    pressed[RAID_KEY_DOWN]  = NO;
    pressed[RAID_KEY_LEFT]  = NO;
    pressed[RAID_KEY_RIGHT] = NO;
    for (s = 0; s < j->num_sticks; s++) {
	/* Hack to fix our/my gamepads under Linux and Windows. */
	if (ignore_stick[s])
	    continue;

	if (j->stick[s].axis[y_axis].d1) pressed[RAID_KEY_UP]    = YES;
	if (j->stick[s].axis[y_axis].d2) pressed[RAID_KEY_DOWN]  = YES;
	if (j->stick[s].axis[x_axis].d1) pressed[RAID_KEY_LEFT]  = YES;
	if (j->stick[s].axis[x_axis].d2) pressed[RAID_KEY_RIGHT] = YES;
    }

    pressed[RAID_KEY_START] = j->button[start_b].b;
    pressed[RAID_KEY_MENU]  = j->button[menu_b].b;

    pressed[RAID_KEY_FIRE] = NO;
    for (b = 0; b < j->num_buttons; b++) {
	if (b == start_b || b == menu_b)
	    continue;

	if (j->button[b].b) {
	    pressed[RAID_KEY_FIRE] = YES;
	    break;
	}
    }
}

- (void) setKey:(raid_key_t)k To:(int)k_
{
    if (k == RAID_KEY_START) {
	if (menu_b == k_)
	    menu_b = start_b;

	start_b = k_;
    }
    elif (k == RAID_KEY_MENU) {
	if (start_b == k_)
	    start_b = menu_b;

	menu_b = k_;
    }
}

- (int) getKey:(raid_key_t)k
{
    if (k == RAID_KEY_START) return start_b;
    elif (k == RAID_KEY_MENU) return menu_b;
    else
	return -1;
}

- (int) xPosition
{
    return j->stick[x_stick].axis[x_axis].pos;
}

- (int) yPosition
{
    return j->stick[y_stick].axis[y_axis].pos;
}

#ifdef WINFF
- free
{
    if (ff_player_hurt) {
	ff_destroy_effect(ff_player_hurt);
	ff_player_hurt = NULL;
    }

    if (ff_player_dying) {
	ff_destroy_effect(ff_player_dying);
	ff_player_dying = NULL; 
    }

    return [super free];
}

- (void) rumbleHurt
{
    if (ff_player_hurt)
	ff_play_effect(ff_player_hurt);
}

- (void) rumbleDying
{
    if (ff_player_dying)
	ff_play_effect(ff_player_dying);
}
#endif
@end


@implementation JoystickController (Private)
- (int) guessStartButton
{
    if (not joy_enabled || joy_num >= num_joysticks)
	return 0;

    /* These values work for my gamepads :P */
    switch (j->num_buttons) {
      case  4: return 1;	/* Triangle (top) */
      case  8: return 5;
      case 12: return 8;
      case 24: printf("Oi!  24 joystick buttons detected.\n"); return 8;
      default: return 0;
    }
}

- (int) guessMenuButton
{
    if (not joy_enabled || joy_num >= num_joysticks)
	return 0;

    switch (j->num_buttons) {
      case  4: return 2;	/* Cross (bottom, bottom, bottom..hahaha) */
      case  8: return 4;
      case 12: return 9;
      case 24: printf("Oi!  24 joystick buttons detected.\n"); return 9;
      default: return 1;
    }
}

- (void) resetToDefaults
{
    x_stick = 0; x_axis = 0;
    y_stick = 0; y_axis = 1;

    start_b = [self guessStartButton];
    menu_b  = [self guessMenuButton];

#ifdef WINFF
    if (ff_joystick_has(joy_num)) {
	fprintf(stdout, "Joystick %d has force feedback.\n", joy_num+1);
	ff_player_hurt = ff_create_effect();
	ff_make_constant_force_effect(joy_num, ff_player_hurt, 5);
	ff_player_dying = ff_create_effect();
	ff_make_periodic_effect(joy_num, ff_player_dying);
    }
    else {
	ff_player_hurt = NULL;
	ff_player_dying = NULL;
    }
#endif
}
@end


@implementation JoystickController (Config)
- (void) readConfigFile:(FILE *)fp :(const char *)section
{
    assert(fp);

    while (expect_char(fp, '(')) {
	Token *ctok, *btok;
	const char *cmd;
	int b;

	ctok = read_symbol(fp);
	if (not ctok) {
	    fprintf(stderr,
		    "[Input] Buggy " JOYSTICK_CFG " file (%s).  "
		    "Missing command token.\n", section);
	    return;
	}
	cmd = [ctok getSymbol];

	btok = read_token(fp);
	if (not btok) {
	    fprintf(stderr, "[Input] Buggy " JOYSTICK_CFG " file (%s:%s).  "
		    "Missing argument.\n", section, cmd);
	    ctok = [ctok free];
	    return;
	}

	if (not expect_char(fp, ')')) {
	    fprintf(stderr,
		    "[Input] Buggy " JOYSTICK_CFG " file (%s:%s).  "
		    "Missing ).\n", section, cmd);
	    btok = [btok free];
	    ctok = [ctok free];
	    return;
	}

	/* Process the commands. */
	if (streq(cmd, "name")) {
	    assert([btok type] == TOKEN_STRING);
	    snprintf(name, sizeof name, [btok getString]);
	}
	else {
	    assert([btok type] == TOKEN_FIXNUM);
	    b = [btok getFixnum];
	    if (streq(cmd, "start-button")) {
		if (b < j->num_buttons)
		    start_b = b;
	    }
	    elif (streq(cmd, "menu-button")) {
		if (b < j->num_buttons)
		    menu_b = [btok getFixnum];
	    }
	    elif (streq(cmd, "primary-stick")) {
		if (b < j->num_sticks) {
		    x_stick = b;
		    y_stick = b;
		}
	    }
	    elif (streq(cmd, "ignore-stick")) {
		if (b < j->num_sticks)
		    ignore_stick[b] = YES;
	    }
	    else {
		fprintf(stderr,
			"[Input] Buggy " JOYSTICK_CFG " file (%s:%s).  "
			"Unknown command.\n", section, cmd);
	    }
	}

	btok = [btok free];
	ctok = [ctok free];
    }
}

- (void) writeConfigFile:(FILE *)fp :(const char *)section
{
    int s;
    assert(fp);

    fprintf(fp,
	    "(%s\n"
	    " (name\t\t\"%s\")\n"
	    " (start-button\t%d)\n"
	    " (menu-button\t%d)\n"
	    " (primary-stick\t%d)\n"
	    " ;(ignore-stick\t<num>)\n",
	    section, name, start_b, menu_b, x_stick);

    for (s = 0; s < j->num_sticks; s++) {
	if (ignore_stick[s])
	    fprintf(fp, " (ignore-stick\t%d)\n", s);
    }

    fprintf(fp, " )\n");
}
@end
