/* game.m,
 *
 * The game loop.  Filthy, but it works.  Note some variables may be
 * inside common.m because it's more useful there.
 */

#include <allegro.h>
#include <assert.h>
#include <math.h>
#include <objc/Object.h>
#include "candy.h"
#include "common.h"
#include "demo-info.h"
#include "demo.h"
#include "difficulty.h"
#include "end-level.h"
#include "enter-name.h"
#include "fps.h"
#include "game-timer.h"
#include "game.h"
#include "group.h"
#include "hiscore.h"
#include "input.h"
#include "map-save.h"
#include "map.h"
#include "maybe-fblend.h"
#include "music-ad.h"
#include "music.h"
#include "newmenu-log.h"
#include "newmenu.h"
#include "nuke.h"
#include "osd.h"
#include "player-stats.h"
#include "player.h"
#include "powerup.h"
#include "projectile.h"
#include "projectiles/toothpaste.h"
#include "scramble-letter.h"
#include "screenshot.h"
#include "seborrhea/seborrhea.h"
#include "seborrhea/seborrhea-datadir.h"
#include "sound.h"
#include "star-field.h"
#include "unit.h"
#include "units/all-units.h"
#include "video.h"


#define STATUS_BITMAP_W		80
#define STATUS_BITMAP_H		185 /*165*/
#define STATUS_BITMAP_X1	((((640-screen_w)/2) - STATUS_BITMAP_W)/2)
#define STATUS_BITMAP_X2	(screen_w+status_w + STATUS_BITMAP_X1)
#define STATUS_BITMAP_Y		200

static Sebum<SebFont> *main_font;
static BITMAP *real_dbuf;
static BITMAP *dbuf, *left_status, *right_status; /* Sub-bitmaps. */
static StarField *star_field;

static void blit_to_screen(int);
static void game_loop_init(void);
static void game_loop_shutdown(void);

/*--------------------------------------------------------------*/

static void grayout_bitmap(BITMAP *bmp)
{
    blender_begin_primitives();
    blender_set_trans(0x80);
    blender_rect_trans(bmp, 0, 0, bmp->w, bmp->h, makecol(0x40, 0x40, 0x40));
    blender_end_primitives();
}


static void blit_sidebars(BITMAP *blitme, int x)
{
#ifndef NO_VIDEO_BITMAPS
    if (raid_use_page_flipping || raid_use_triple_buffering) {
	video_blit_to_all_pages(blitme, x, 0, blitme->w, blitme->h);
	return;
    }
#endif

    blit(blitme, screen, 0, 0, x, 0, blitme->w, blitme->h);
}


static void redraw_sidebars(void)
{
    /* To avoid a BIG (REALLY BIG) speed hit, we only draw the left
       and right sidebars at the beginning of the game, and when we
       lose focus and come back (eg. alt-tabbing away). */

    Sebum<SebImage> *sidebar = [base_sebum getSebumByName:"glue/sidebar"];
    BITMAP *blitme = create_bitmap((SCREEN_W-screen_w)/2, SCREEN_H);

    if (not sidebar || not blitme)
	return;

    [sidebar drawTo:blitme X:0 Y:0 W:blitme->w H:blitme->h];

    /* Left sidebar. */
    blit(left_status, blitme, 0, 0, STATUS_BITMAP_X1, STATUS_BITMAP_Y, STATUS_BITMAP_W, STATUS_BITMAP_H);
    blit_sidebars(blitme, 0);

    /* Right sidebar.  Note X coordinate is correct. */
    blit(right_status, blitme, 0, 0, STATUS_BITMAP_X1, STATUS_BITMAP_Y, STATUS_BITMAP_W, STATUS_BITMAP_H);
    blit_sidebars(blitme, SCREEN_W-blitme->w);

    FREE_BITMAP(blitme);
}

#ifndef NO_ID3TAGS
static void redraw_sidebar_for_music_ad(void)
{
    Sebum<SebImage> *sidebar = [base_sebum getSebumByName:"glue/sidebar"];

    if (not sidebar)
	return;

    [sidebar drawFromX:0 Y:SCREEN_H-70
	     To:real_dbuf X:0 Y:SCREEN_H-70 W:[sidebar width] H:70];
}
#endif

/*--------------------------------------------------------------*/

static void draw_level_complete(BITMAP *dest)
{
    draw_letters(dest, level_complete_letters);
}


static void draw_game_over(BITMAP *dest)
{
    /* Game Over man, GAME OVER! */
    grayout_bitmap(dest);
    draw_letters(dest, game_over_letters);

    [main_font putString:"Press START to retry" To:dbuf X:dbuf->w/2 Y:dbuf->h/2+20 Colour:0xe0:0xe0:0xff Alignment:ALIGN_CENTRE Decoration:DECORATION_OUTLINE];
    [main_font putString:"or ESCAPE to quit"    To:dbuf X:dbuf->w/2 Y:dbuf->h/2+40 Colour:0xe0:0xe0:0xff Alignment:ALIGN_CENTRE Decoration:DECORATION_OUTLINE];
}


static void draw_statusbars(void)
{
    if (player[0])
	[player[0] drawHealth:left_status];
    else if (game_flags & FLAG_REDRAW_STATUSBARS)
	draw_death_statusbar(left_status, 0);

    if (player[1])
	[player[1] drawHealth:right_status];
    else if (num_players > 1 && game_flags & FLAG_REDRAW_STATUSBARS)
	draw_death_statusbar(right_status, 1);
}

/*--------------------------------------------------------------*/
/* Pause.							*/
/*--------------------------------------------------------------*/

static void do_game_paused_update(void) 
{
    if (start_pressed())
	game_flags ^= FLAG_GAME_PAUSED;
    if (menu_released())
	game_flags |= FLAG_QUIT;
    rest(0);				/* Yield! */
}


static void draw_game_paused(BITMAP *dest)
{
    Sebum<SebImage> *spr;
    grayout_bitmap(dest);

    spr = [base_sebum getSebumByName:"glue/paused"];
    [spr drawTo:dest X:(dest->w-[spr width])/2 Y:200];
    [main_font putString:"Press START to resume" To:dbuf X:dbuf->w/2 Y:dbuf->h/2+20 Colour:0xe0:0xe0:0xff Alignment:ALIGN_CENTRE Decoration:DECORATION_OUTLINE];
    [main_font putString:"or ESCAPE to quit"     To:dbuf X:dbuf->w/2 Y:dbuf->h/2+40 Colour:0xe0:0xe0:0xff Alignment:ALIGN_CENTRE Decoration:DECORATION_OUTLINE];
}

/*--------------------------------------------------------------*/
/* High score.							*/
/*--------------------------------------------------------------*/

char high_score_name_buf[MAX_PLAYERS][HIGH_SCORE_NAME_LEN+1] = {
#ifdef CHEAT
    "Cheater!", "Cheater!"
#else
    "Alice", "Bob"
#endif
};
static char current_high_score_char = 'a';
static BOOL has_entered_high_score[MAX_PLAYERS];


static void enter_high_score(const unsigned int pl)
{
    char *name;
    const int flag[MAX_PLAYERS] = {
	FLAG_PLAYER1_HIGH_SCORE, FLAG_PLAYER2_HIGH_SCORE
    };
    assert(pl < MAX_PLAYERS);

    name = high_score_name_buf[pl];

    if (enter_name(name, HIGH_SCORE_NAME_LEN+1, &current_high_score_char, pl)) {
	has_entered_high_score[pl] = YES;
	update_high_score(name, player_stats[pl].score, SOURCE_UNKNOWN);
	game_flags &=~flag[pl];
    }
}


static void draw_name(BITMAP *dest, const unsigned int pl)
{
    static int t;
    int l;
    char *name;
    char str[2];
    name = high_score_name_buf[pl];

    [main_font putString:"Enter your name" To:dest X:dest->w/2 Y:dest->h/2+10
	       Colour:0xb0:0xb0:0xb0 Alignment:ALIGN_CENTRE
	       Decoration:DECORATION_OUTLINE];

    [main_font putStringTo:dest X:dest->w/2 Y:dest->h/2+40
	       Colour:0xa0:0x60:0xe0 Alignment:ALIGN_CENTRE
	       Decoration:DECORATION_OUTLINE :"Player %d", pl+1];

    l = [main_font textLength:name];
    str[0] = current_high_score_char;
    str[1] = '\0';
    [main_font putString:name To:dest X:(dest->w-l)/2 Y:dest->h/2+70
	       Colour:0xb0:0xb0:0xb0 Decoration:DECORATION_OUTLINE];

    if (++t > 30)
	t = 0;

    if (t > 15) {		/* Dark */
	rectfill_wh(dest, (dest->w+l)/2+1, dest->h/2+70,
		    [main_font textLength:str]+4, 20, makecol(0xe0,0xa0,0x60));
	[main_font putString:str To:dest X:(dest->w+l)/2+3 Y:dest->h/2+70
		   Colour:0x40:0x40:0x40 Decoration:DECORATION_OUTLINE];
    }
    else {
	rectfill_wh(dest, (dest->w+l)/2+1, dest->h/2+70,
		    [main_font textLength:str]+4, 20, makecol(0x40,0x40,0x40));
	[main_font putString:str To:dest X:(dest->w+l)/2+3 Y:dest->h/2+70
		   Colour:0xe0:0xa0:0x60 Decoration:DECORATION_OUTLINE];
    }
}


static void draw_high_score(BITMAP *dest, const unsigned int pl)
{
    grayout_bitmap(dest);
    draw_letters(dest, high_score_letters);
    draw_name(dest, pl);

    /* Instructions. */
    [main_font putString:"MENU BUTTON"
	       To:dest X:180 Y:380 Colour:0xa0:0x80:0x80
	       Alignment:ALIGN_RIGHT Decoration:DECORATION_OUTLINE];
    [main_font putString:"to erase"
	       To:dest X:200 Y:380 Colour:0x80:0x80:0x80
	       Decoration:DECORATION_OUTLINE];

    [main_font putString:"START BUTTON"
	       To:dest X:180 Y:420 Colour:0xa0:0x80:0x80
	       Alignment:ALIGN_RIGHT Decoration:DECORATION_OUTLINE];
    [main_font putString:"to finish"
	       To:dest X:200 Y:420 Colour:0x80:0x80:0x80
	       Decoration:DECORATION_OUTLINE];
}


static void finish_campaign_high_score_loop(const unsigned int pl)
{
    BOOL redraw = YES;
    char *name;
    const int flag[MAX_PLAYERS] = {
	FLAG_PLAYER1_HIGH_SCORE, FLAG_PLAYER2_HIGH_SCORE
    };
    assert(pl < MAX_PLAYERS);

    name = high_score_name_buf[pl];

    scramble_letters(high_score_letters);

    while (game_flags & flag[pl]) {
	while (counter) {
	    counter--;
	    poll_raid_keys(num_players);
	    enter_high_score(pl);
	    unscramble_letters(high_score_letters);
	    redraw = YES;
	}

	if (redraw) {
	    redraw = NO;
	    clear_to_color(dbuf, makecol(0x40, 0x40, 0x40));
	    draw_high_score(dbuf, pl);

	    if (game_flags & FLAG_REDRAW_SIDEBARS) {
		game_flags &=~FLAG_REDRAW_SIDEBARS;
		redraw_sidebars();
	    }

	    blit_to_screen(0);
	}
    }
}


static void demo_name_loop(const unsigned int pl)
{
    BOOL redraw = YES;
    char *name;
    const int flag[MAX_PLAYERS] = {
	FLAG_PLAYER1_DEMO_NAME, FLAG_PLAYER2_DEMO_NAME
    };
    assert(pl < MAX_PLAYERS);

    name = high_score_name_buf[pl];

    while (game_flags & flag[pl]) {
	while (counter) {
	    counter--;
	    poll_raid_keys(num_players);

	    if (enter_name(name, HIGH_SCORE_NAME_LEN, &current_high_score_char, pl)) {
		game_flags &=~flag[pl];
	    }

	    redraw = YES;
	}

	if (redraw) {
	    redraw = NO;
	    clear_to_color(dbuf, makecol(0x40, 0x40, 0x40));
	    draw_high_score(dbuf, pl);

	    if (game_flags & FLAG_REDRAW_SIDEBARS) {
		game_flags &=~FLAG_REDRAW_SIDEBARS;
		redraw_sidebars();
	    }

	    blit_to_screen(0);
	}
    }
}

/*--------------------------------------------------------------*/

static void game_pre_reload_map(void)
{
    int pid;

    /* Reset our lists. */
    unit_reset();
    unit_group_shutdown();	unit_group_init();
    projectile_shutdown();	projectile_init();
    candy_shutdown();		candy_init();
    nuke_shutdown();		nuke_init();
    powerup_shutdown();		powerup_init();

    for (pid = 0; pid < MAX_PLAYERS; pid++)
	unlockToothpaste(pid);
}


static void game_post_reload_map(void)
{
    int map_length;
    long seed = time(0);
    alrand_seed(rng_state, seed);

    if (demo_state == DEMO_RECORDING)
	write_game_init(current_demo_file, seed);
    else if ((demo_state == DEMO_PLAYBACK) ||
	     (demo_state == DEMO_PLAYBACK_AUTOMATIC))
	read_game_init(&demo_file);

    if (map_start_y >= 0.0) {
	/* Hack for when we jump-start from REdit. */
	map_length = MIN(map_start_y, TILE_H * [current_map length]);
	purge_units_below(map_start_y-screen_h);
    }
    else {
	map_length = TILE_H * [current_map length];
    }

    [current_map setOffsetX:0 Y:map_length-screen_h];

    if (num_players == 1) {
	player[0] = (Player *)spawn_unit([Player class],  screen_w*1/2, map_length, ALLY_LIST, YES);
	player[1] = nil;
    }
    else {			/* Two-players. */
	player[0] = (Player *)spawn_unit([Player class],  screen_w*1/3, map_length, ALLY_LIST, YES);
	player[1] = (Player *)spawn_unit([Player2 class], screen_w*2/3, map_length, ALLY_LIST, YES);
    }

    /* The map wants stars, so make a star field. */
    if ([current_map stars])
	star_field = [StarField new];

    game_flags &=~FLAG_LEVEL_COMPLETE;
    game_flags |= FLAG_PLAYERS_ALIVE;
}


static MEMORY_FILE *raid_load_map(const char *map_fn, long *map_memory_offset)
{
    MEMORY_FILE *mfp;
    assert(map_memory_offset);

    game_pre_reload_map();

    /* The first time we start a map we need to load the map into
       memory.  After that we just load from memory. */
    if ((demo_state == DEMO_PLAYBACK) ||
	(demo_state == DEMO_PLAYBACK_AUTOMATIC)) {
	*map_memory_offset = mmpk_ftell(&demo_file) + DEMO_LEVEL_INIT_STRLEN;

	if (not read_level_init(&demo_file, YES))
	    return NULL;
	else
	    mfp = &demo_file;
    }
    else {
	mfp = load_map(map_fn, NO);
	*map_memory_offset = 0;		/* Semi-hack. */

	if (not mfp)
	    return NULL;

	/* We want to save the map into the demo file. */
	if (demo_state == DEMO_RECORDING)
	    write_level_init(current_demo_file);
    }

    game_post_reload_map();

    return mfp;
}


static void raid_reset(MEMORY_FILE *mfp, long map_memory_offset)
{
    long off;

    game_pre_reload_map();

    /* Reload the map. */
    off = mmpk_ftell(mfp);
    mmpk_fseek(mfp, map_memory_offset, SEEK_SET);
    read_level_init(mfp, NO);
    mmpk_fseek(mfp, off, SEEK_SET);

    game_post_reload_map();
}


static void do_game_update(BOOL check_high_score,
			   MEMORY_FILE *mfp, long map_memory_offset)
{
#ifdef DEBUG
    if (key[KEY_PGUP]) [current_map setScrollRate:-0.1 :NO];
    if (key[KEY_PGDN]) [current_map setScrollRate: 0.1 :NO];
    if (key[KEY_HOME]) [current_map setScrollRate:DEFAULT_SCROLL_RATE :YES];
    if (key[KEY_END])  [current_map setScrollRate:0.0 :YES];
#endif

#ifndef NO_ID3TAGS
    if (music_ad)
	music_ad = [music_ad update];
#endif

    if (demo_state == DEMO_RECORDING &&
	record_demo_checkpoint) {
	record_demo_checkpoint = NO;
	write_demo_checkpoint(current_demo_file);
    }
    else if ((demo_state == DEMO_PLAYBACK) ||
	     (demo_state == DEMO_PLAYBACK_AUTOMATIC)) {
	if (mmpk_feof(&demo_file))
	    game_flags |= FLAG_QUIT;

	if (memcmp(demo_file.pos, DEMO_CHECKPOINT_STRING,
		   DEMO_CHECKPOINT_STRLEN) == 0) {
	    /* +1 for newline char. */
	    mmpk_fseek(&demo_file, DEMO_CHECKPOINT_STRLEN+1, SEEK_CUR);
	    read_demo_checkpoint(&demo_file);
	}
    }

    if ((game_flags & FLAG_PLAYERS_ALIVE) &&
	not player[0] && not player[1]) {
	/* Just died recently. */
	game_flags &=~FLAG_PLAYERS_ALIVE;
	scramble_letters(game_over_letters);
	scramble_letters(high_score_letters);
	clear_keybuf();
    }

    if (game_flags & FLAG_LEVEL_COMPLETE && not boss) {
	if (player[0]) [player[0] zoom];
	if (player[1]) [player[1] zoom];

	if (not (game_flags & (FLAG_PLAYER1_HIGH_SCORE|FLAG_PLAYER2_HIGH_SCORE))) {
	    unscramble_letters(level_complete_letters);
	    end_level_tics--;
	}
    }
    else {
	do_input();
    }

    if (toggle_fps())			/* F12: toggle FPS. */
	game_flags ^= FLAG_DRAW_FPS;

    if (star_field)
	[star_field update];

    [current_map update];
    update_candy();
    update_powerups();
    update_nukes();
    update_projectiles();
    update_units();
    /* game_time++; */

    if (not (game_flags & FLAG_PLAYERS_ALIVE)) { /* Restart/Quit. */
	BOOL retry = NO;

	if (not check_high_score) {
	    game_flags &=~FLAG_PLAYER1_HIGH_SCORE;
	    game_flags &=~FLAG_PLAYER2_HIGH_SCORE;
	}

	if ((demo_state == DEMO_PLAYBACK) ||
	    (demo_state == DEMO_PLAYBACK_AUTOMATIC)) {
	    int c = mmpk_getc(&demo_file);

	    if (c == EOF)
		game_flags |= FLAG_QUIT;
	    else
		retry = c;
	}
	else {
	    if (demo_state == DEMO_RECORDING)
		pack_putc(0, current_demo_file);

	    if (game_flags & FLAG_PLAYER1_HIGH_SCORE) {
		unscramble_letters(high_score_letters);
		enter_high_score(0);
		return;
	    }

	    if (game_flags & FLAG_PLAYER2_HIGH_SCORE) {
		unscramble_letters(high_score_letters);
		enter_high_score(1);
		return;
	    }

	    /* Don't restart if we died but killed the boss. */
	    if (game_flags & FLAG_LEVEL_COMPLETE)
		return;

	    /* Press START to retry or ESCAPE to quit screen. */
	    if ((demo_state == DEMO_PLAYBACK) ||
		(demo_state == DEMO_PLAYBACK_AUTOMATIC)) {
	    }
	    else {
		retry = retry_released();

		if (demo_state == DEMO_RECORDING)
		    pack_putc(retry, current_demo_file);
	    }

	    if (menu_released())
		game_flags |= FLAG_QUIT;
	}

	if (game_flags & FLAG_QUIT);
	else if (retry) {
	    /* Don't put this inside raid_reset() because we want
	       score to carry to the next level. */
	    clear_stats_for_player(0);
	    clear_stats_for_player(1);

	    raid_reset(mfp, map_memory_offset);
	    free_reserved_sounds();
	    game_flags |= FLAG_PLAYERS_ALIVE;
	}
	else {
	    unscramble_letters(game_over_letters);
	}
    }
    else if (not (game_flags & FLAG_LEVEL_COMPLETE)) { /* Pause game. */
	if (start_pressed() || menu_released()) {
	    if (demo_state == DEMO_PLAYBACK_AUTOMATIC)
		game_flags |= FLAG_QUIT;
	    else
		game_flags ^= FLAG_GAME_PAUSED;
	}
    }
}

/*--------------------------------------------------------------*/

static void blit_to_screen(int ssx)
{
    /* If we are screen shaking, don't draw the left/right most column
       of dbuf because it will draw on the sidebars, which is bad.
       Here sx/dx if for source x, dest x. */

    int sx = MAX(-ssx, 0);
    int dx = MAX( ssx, 0);
    int ww = dbuf->w - ABS(ssx);
    BITMAP *active_page = video_active_page();

#ifndef NO_VIDEO_BITMAPS
    if (raid_use_page_flipping || raid_use_triple_buffering) {
	if (game_flags & FLAG_REDRAW_STATUSBARS) {
	    game_flags &=~FLAG_REDRAW_STATUSBARS;

	    video_blit_to_all_pages(left_status,  STATUS_BITMAP_X1, STATUS_BITMAP_Y, STATUS_BITMAP_W, STATUS_BITMAP_H);
	    video_blit_to_all_pages(right_status, STATUS_BITMAP_X2, STATUS_BITMAP_Y, STATUS_BITMAP_W, STATUS_BITMAP_H);
	    goto next;
	}
    }
#endif /* NO_VIDEO_BITMAPS */

    /* Try to draw as little as possible. */
    if (player[0] || game_flags & FLAG_REDRAW_STATUSBARS)
	blit(left_status,  active_page, 0, 0, STATUS_BITMAP_X1, STATUS_BITMAP_Y, STATUS_BITMAP_W, STATUS_BITMAP_H);
    if (player[1] || game_flags & FLAG_REDRAW_STATUSBARS)
	blit(right_status, active_page, 0, 0, STATUS_BITMAP_X2, STATUS_BITMAP_Y, STATUS_BITMAP_W, STATUS_BITMAP_H);
    game_flags &=~FLAG_REDRAW_STATUSBARS;
    goto next;			/* GCC: Shut up. */

 next:
#ifndef NO_ID3TAGS
    if (music_ad)
	blit(real_dbuf, active_page, 0, SCREEN_H-70, 0, SCREEN_H-70, status_w, 70);
#endif

    blit(dbuf, active_page, sx, 0, status_w+dx, 0, ww, dbuf->h);
    video_flip_to_screen();
}

/*--------------------------------------------------------------*/

/* raid_draw_play_field:
 *
 * Draw the main game area onto dbuf (global).
 */
static void raid_draw_play_field(void)
{
    /* Stars and parallax. */
    if ([current_map stars]) {
	clear_to_color(dbuf, makecol(0x00, 0x00, 0x00));
	[star_field draw:dbuf];
	[current_map drawParallaxLayer:dbuf :YES];
	draw_candy(dbuf, SUBTERRANIAN_LAYER);
    }

    /* Terrain and ground objects. */
    [current_map drawTiles:dbuf];
    draw_candy(dbuf, FLOOR_LAYER);
    draw_units(dbuf, ACTIVE_GROUND_LIST);
    draw_candy(dbuf, LOW_LAYER);

    /* Someone found shadows disorienting and asked for this.  Think
       it was Fenix Blade guy. */
    if (game_flags & FLAG_DRAW_SHADOW)
	draw_unit_shadows(dbuf);

    /* Air objects. */
    draw_candy(dbuf, MEDIUM_LAYER);
    draw_powerups(dbuf);
    draw_units(dbuf, ACTIVE_AIR_LIST);
    draw_units(dbuf, ALLY_LIST);
    draw_projectiles(dbuf);
    draw_candy(dbuf, HIGH_LAYER);
    draw_nukes(dbuf);

    /* Boss's health meter.  Note: boss == nil does not mean the boss
       is dead! */
    if (boss)
	[boss drawHealthMeter:dbuf];
}


/* raid_draw_demo_playback_info:
 *
 * Draw 'PRESS START' texts when automatically playing back a demo
 * (due to menu inactivity) to make it more arcade-y.
 */
static void raid_draw_demo_playback_info(void)
{
    [main_font putString:"Press Start"
	       To:dbuf X:dbuf->w/2 Y:30 Colour:0xff:0xff:0xff
	       Alignment:ALIGN_CENTRE Decoration:DECORATION_OUTLINE];
    [main_font putString:"Press Start"
	       To:dbuf X:dbuf->w/2 Y:dbuf->h-50 Colour:0xff:0xff:0xff
	       Alignment:ALIGN_CENTRE Decoration:DECORATION_OUTLINE];
}


/* raid_draw_game_progress_info:
 *
 * Write out game progress related info: paused, high score, dead.
 */
static void raid_draw_game_progress_info(void)
{
    draw_demo_info(dbuf);

    if (not (game_flags & FLAG_PLAYERS_ALIVE) ||
	(game_flags & FLAG_LEVEL_COMPLETE)) {
	if (game_flags & FLAG_PLAYER1_HIGH_SCORE)
	    draw_high_score(dbuf, 0);
	elif (game_flags & FLAG_PLAYER2_HIGH_SCORE)
	    draw_high_score(dbuf, 1);
	elif (game_flags & FLAG_LEVEL_COMPLETE)
	    draw_level_complete(dbuf);
	else
	    draw_game_over(dbuf);
    }
    else if (game_flags & FLAG_GAME_PAUSED)
	draw_game_paused(dbuf);
}

/*--------------------------------------------------------------*/

enum RAID_LOOP_EXIT_REASON {
    RAID_LOOP_LOAD_MAP_FAILED,
    RAID_LOOP_PLAYER_IS_A_QUITTER,
    RAID_LOOP_LEVEL_ENDED
};


static enum RAID_LOOP_EXIT_REASON
raid_loop(const char *level_name, BOOL check_high_score)
{
    BOOL last_key_n = NO;
    enum RAID_LOOP_EXIT_REASON exit_reason = RAID_LOOP_PLAYER_IS_A_QUITTER;
    MEMORY_FILE *mfp = NULL;
    long map_offset = 0;
    int screen_shake_x = 0, screen_shake_tics = 0;

    game_flags = (FLAG_REDRAW|FLAG_REDRAW_SIDEBARS|FLAG_REDRAW_STATUSBARS);

    if (shadow_opacity > 0)
	game_flags |= FLAG_DRAW_SHADOW;

    mfp = raid_load_map(level_name, &map_offset);
    if (not mfp) {
	game_flags |= FLAG_QUIT;
	exit_reason = RAID_LOOP_LOAD_MAP_FAILED;
    }
    else {
	char music_dir[PATH_MAX] = "music/";
	unload_unnecessary_unit_data();
	unload_unnecessary_projectile_data();
	strncat(music_dir, [current_map musicDirname], sizeof music_dir);
	play_music(music_dir);
    }

    if (demo_state != DEMO_PLAYBACK_AUTOMATIC) {
	demo_info_set_text([current_map getTitle],
			   difficulty_text2[difficulty]);
    }

    counter = 0;
    while (not (game_flags & FLAG_QUIT) &&
	   (not (game_flags & FLAG_LEVEL_COMPLETE) || end_level_tics > 0)) {
	while (counter) {
	    counter--;

	    if ((demo_state == DEMO_PLAYBACK) &&
		(key[KEY_N] && !last_key_n)) {
		switch (demo_skip_segment(&demo_file)) {
		  case DEMO_NO_SKIP:
		      break;
		  case DEMO_SKIP_TO_NEW_SEGMENT:
		      raid_reset(mfp, map_offset);
		      break;
		  case DEMO_SKIP_TO_NEW_MAP:
		      game_flags |= FLAG_LEVEL_COMPLETE;
		      goto out;
		}
	    }
	    last_key_n = key[KEY_N];

	    poll_raid_keys(num_players);

	    if (game_flags & FLAG_GAME_PAUSED) {
		do_game_paused_update();

		/* Special for screenshots. */
		if (key[KEY_F10])
		    game_flags |= FLAG_REDRAW;
	    }
	    else {
		do_game_update(check_high_score, mfp, map_offset);
		update_demo_info();
		game_flags |= FLAG_REDRAW;
	    }

	    /* Screen shakes. */
	    if (game_flags & FLAG_SCREEN_SHAKE) {
		game_flags &=~FLAG_SCREEN_SHAKE;
		screen_shake_tics = 20;
	    }

	    if (screen_shake_tics > 0) {
		screen_shake_tics--;
		screen_shake_x = rnd(-2, 2);
	    }
	    else
		screen_shake_x = 0;

	    /* Music. */
	    poll_music();
	}

	if (game_flags & FLAG_REDRAW) {
            game_flags &=~FLAG_REDRAW;

	    raid_draw_play_field();

	    /* Stats. */
	    if (game_flags & FLAG_REDRAW_SIDEBARS) {
		game_flags &=~FLAG_REDRAW_SIDEBARS;
		redraw_sidebars();
	    }
	    draw_statusbars();

	    /* Screenshot. */
	    if (key[KEY_F10])
		take_screenshot(dbuf);

	    if (demo_state == DEMO_PLAYBACK_AUTOMATIC)
		raid_draw_demo_playback_info();
	    else
		raid_draw_game_progress_info();

	    if (game_flags & FLAG_DRAW_FPS) {
		draw_fps(dbuf);
	    }

	    if (osd)
		[osd draw:dbuf :music_vol*100/255];

#ifndef NO_ID3TAGS
	    if (music_ad) {
		redraw_sidebar_for_music_ad();
		[music_ad drawTo:real_dbuf];
	    }
#endif

	    blit_to_screen(screen_shake_x);
	}
    }
 out:

    /* If a player has died, reset the score. */
    if (not player[0]) clear_stats_for_player(0);
    if (not player[1]) clear_stats_for_player(1);

    free_music();
    free_reserved_sounds();

    unit_shutdown();
    unit_group_shutdown();
    projectile_shutdown();
    candy_shutdown();
    nuke_shutdown();
    powerup_shutdown();
    if (star_field)
	star_field = [star_field free];

    if (game_flags & FLAG_LEVEL_COMPLETE)
	exit_reason = RAID_LOOP_LEVEL_ENDED;

    return exit_reason;
}


void raid_start_levels(const char *level_list[], int num_levels,
		       BOOL check_high_score, BOOL prefix_with_data_dir)
{
    enum RAID_LOOP_EXIT_REASON exit_reason = 0;
    BOOL exit = NO;
    int i, level_n = 0;

    game_loop_init();

    current_map = [Map new];
    assert(current_map);

    clear_stats_for_player(0);
    clear_stats_for_player(1);

    while (not exit && level_n < num_levels) {
	if (level_list == NULL) {
	    /* Demo playback. */
	    exit_reason = raid_loop(NULL, NO);
	}
	else if (prefix_with_data_dir) {
	    for (i = 0; seborrhea_data_root_directories[i]; i++) {
		char fn[PATH_MAX];
		snprintf(fn, sizeof fn, "%s/%s",
			 seborrhea_data_root_directories[i],
			 level_list[level_n]);

		exit_reason = raid_loop(fn, check_high_score);

		/* We loaded the map.  Don't play this level again. */
		if (exit_reason != RAID_LOOP_LOAD_MAP_FAILED)
		    break;
	    }
	}
	else {
	    exit_reason = raid_loop(level_list[level_n], check_high_score);
	}

	switch (exit_reason) {
	  case RAID_LOOP_LOAD_MAP_FAILED:
	    if (level_list) {
		append_message("Error opening %s!", level_list[level_n]);
		fprintf(stderr, "Error opening %s!\n", level_list[level_n]);
	    }
	    else {
		append_message("Demo ended prematurely: error loading map!");
		fprintf(stderr, "Demo ended prematurely: error loading map!\n");
	    }
	    exit = YES;
	    break;

	  case RAID_LOOP_PLAYER_IS_A_QUITTER:
	    exit = YES;
	    break;

	  case RAID_LOOP_LEVEL_ENDED:
	    /* If we are playing back demos, continue until it ends or
	       we choose to quit. */
	    if (level_list != NULL)
		level_n++;
	    break;
	}
    }

    current_map = [current_map free];

    if (not emergency_exit) {
	/* We finished the campaign, high score? */
	if (check_high_score) {
	    finish_campaign_high_score_loop(0);
	    finish_campaign_high_score_loop(1);
	}

	/* Ask for players' names if we didn't get around to it during
	   the game. */
	if (demo_state == DEMO_RECORDING) {
	    if (not has_entered_high_score[0]) {
		game_flags |= FLAG_PLAYER1_DEMO_NAME;
		demo_name_loop(0);
	    }

	    if (not has_entered_high_score[1] && (num_players > 1)) {
		game_flags |= FLAG_PLAYER2_DEMO_NAME;
		demo_name_loop(1);
	    }
	}
    }

    game_loop_shutdown();
}

/*--------------------------------------------------------------*/
/* Initialization and shutdown and some more.			*/
/*--------------------------------------------------------------*/

static void switch_into_game(void)
{
    /* Mark in FLAG_REDRAW in-case we were in paused mode. */
    game_flags |= FLAG_REDRAW|FLAG_REDRAW_SIDEBARS;
}


static void game_loop_init(void)
{
    real_dbuf = create_bitmap(SCREEN_W, SCREEN_H);
    assert(real_dbuf);

    dbuf = create_sub_bitmap(real_dbuf, status_w, 0, screen_w, screen_h);
    left_status  = create_sub_bitmap(real_dbuf, STATUS_BITMAP_X1, STATUS_BITMAP_Y,
				     STATUS_BITMAP_W, STATUS_BITMAP_H);
    right_status = create_sub_bitmap(real_dbuf, STATUS_BITMAP_X2, STATUS_BITMAP_Y,
				     STATUS_BITMAP_W, STATUS_BITMAP_H);
    assert(dbuf && left_status && right_status);

    video_flip_to_default();
    video_flip_to_screen();

    {					/* Status backdrops. */
	Sebum<SebImage> *bg = [base_sebum getSebumByName:"glue/statusbar-bg"];
	[bg drawTo:left_status  X:0 Y:0 W:STATUS_BITMAP_W H:STATUS_BITMAP_H];
	[bg drawTo:right_status X:0 Y:0 W:STATUS_BITMAP_W H:STATUS_BITMAP_H];
    }

    /* Resize some fonts now to avoid costly, constant resizing. */
    [(<SebFont>)[base_sebum getSebumByName:"fonts/digital2"] setFontSize:14];

    /* Timers. */
    install_int_ex(ticker, BPS_TO_TIMER(GAME_SPEED));
    fps_init();

    set_display_switch_callback(SWITCH_IN, switch_into_game);

    /* Demos. */
    if (record_demos &&
	demo_state != DEMO_PLAYBACK &&
	demo_state != DEMO_PLAYBACK_AUTOMATIC) {
	current_demo_file = open_demo_file_for_recording();
	record_demo_checkpoint = NO;
    }

    has_entered_high_score[0] = NO;
    has_entered_high_score[1] = NO;
}


static void game_loop_shutdown(void)
{
    /* Demos. */
    close_demo_file();

    /* Timers. */
    remove_int(ticker);
    fps_shutdown();

    FREE_BITMAP(left_status);
    FREE_BITMAP(right_status);
    FREE_BITMAP(dbuf);
    FREE_BITMAP(real_dbuf);

    video_flip_to_default();
}


int game_init(void)
{
    if (input_init() < 0)
        return -1;

    main_font = [base_sebum getSebumByName:"fonts/am20"];
    if (not main_font)
	return -1;

    video_bitmap_init();

    osd = [OSD new];
    return 0;
}


void game_shutdown(void)
{
    if (osd)
	osd = [osd free];

    video_bitmap_shutdown();
    input_shutdown();
}
