/* demo.m,
 *
 * Recording of demos.
 *
 * XXX: using packfiles because maps are in packfiles.
 */

#include <allegro.h>
#include <assert.h>
#include <sys/stat.h>
#include <time.h>
#include "unistd.h"
#include "candy.h"
#include "common.h"
#include "demo-info.h"
#include "demo.h"
#include "difficulty.h"
#include "game.h"
#include "map-save.h"
#include "newmenu-log.h"
#include "player.h"
#include "player-stats.h"


#define DEMO_MAGIC_LEN		(13+1)
#define DEMO_MAGIC_STR		"Raid'em Demo "
#define DEMO_VERSION		"0.3"

/* Demo format (v0.3):
 *
 * 0) HEADER
 *	magic, version
 *	date		(long)
 *	difficulty	(char)
 *	eye-candy	(char)
 *	player names	(strings)
 *
 * 1) LEVEL INIT
 *	[map]
 *	Map file
 *
 * 2) GAME INIT
 *	[game]
 *	seed		(long)
 *	initial num players	(char)
 *	score(s)	(word)
 *
 * 3) GAME LOOP
 *	Input
 *	[sync]...	-- only sometimes
 *
 * Checkpoint syncronisation occurs when:
 *	player collects powerup.
 *
 * Goto game init when we restart the level.
 * Goto level init when we pass a new level.
 */

BOOL record_demos = YES, record_demo_checkpoint;
enum DEMO_STATE demo_state;
PACKFILE *current_demo_file = NULL;
MEMORY_FILE demo_file;
char filename[PATH_MAX];

static struct {
    unsigned int num_players;
    enum DIFFICULTY_LEVEL difficulty;
    unsigned int candy_amount;
} backup_config;


/*--------------------------------------------------------------*/

static void reserve_space_for_string(PACKFILE *fp, unsigned int n)
{
    while (n > 0) {
	pack_putc('\0', fp);
	n--;
    }
}

/*--------------------------------------------------------------*/
/* Header.							*/
/*--------------------------------------------------------------*/

static void read_header03(MEMORY_FILE *fp);
static void write_header03(PACKFILE *fp);

void read_header(MEMORY_FILE *fp)
{
    read_header03(fp);
}


void write_header(PACKFILE *fp)
{
    write_header03(fp);
}


static void read_header03(MEMORY_FILE *fp)
{
    char name[HIGH_SCORE_NAME_LEN+1];
    char tmp[16];
    time_t demo_time;
    assert(fp);

    mmpk_fgets(tmp, DEMO_MAGIC_LEN, fp);	/* Magic	*/
    mmpk_fgets(tmp, sizeof(tmp), fp);		/* Version	*/

    demo_time    = mmpk_igetl(fp);
    difficulty   = mmpk_getc(fp);
    candy_amount = mmpk_getc(fp);
    demo_info_set_date(&demo_time);

    mmpk_fread(name, HIGH_SCORE_NAME_LEN, fp);
    demo_info_set_pilot(0, name);
    mmpk_fread(name, HIGH_SCORE_NAME_LEN, fp);
    demo_info_set_pilot(1, high_score_name_buf[1]);
}


static void write_header03(PACKFILE *fp)
{
    assert(fp);

    pack_fputs(DEMO_MAGIC_STR "0.3", fp);
    pack_putc('\n', fp);

    pack_iputl(time(0), fp);
    pack_putc(difficulty, fp);
    pack_putc(candy_amount, fp);

    reserve_space_for_string(fp, HIGH_SCORE_NAME_LEN);
    reserve_space_for_string(fp, HIGH_SCORE_NAME_LEN);
}


/*--------------------------------------------------------------*/
/* Level Init.							*/
/*--------------------------------------------------------------*/

static BOOL read_level_init03(MEMORY_FILE *fp, BOOL read_marker);
static void write_level_init03(PACKFILE *fp);


BOOL read_level_init(MEMORY_FILE *fp, BOOL read_marker)
{
    return read_level_init03(fp, read_marker);
}


void write_level_init(PACKFILE *fp)
{
    write_level_init03(fp);
}


static BOOL read_level_init03(MEMORY_FILE *fp, BOOL read_marker)
{
    char tmp[16];
    assert(current_map);
    assert(fp);

    if (read_marker) {
	mmpk_fread(tmp, DEMO_LEVEL_INIT_STRLEN, fp);

	if (memcmp(tmp, DEMO_LEVEL_INIT_STRING, DEMO_LEVEL_INIT_STRLEN) != 0) {
	    append_message("Demo broken");
	    game_flags &= FLAG_QUIT;
	    return NO;
	}
    }

    return [current_map loadFrom:fp :NO];
}


static void write_level_init03(PACKFILE *fp)
{
    assert(current_map);
    assert(fp);

    pack_fputs(DEMO_LEVEL_INIT_STRING, fp);
    [current_map saveTo:fp :YES];
}

/*--------------------------------------------------------------*/
/* Game Init.							*/
/*--------------------------------------------------------------*/

static void read_game_init03(MEMORY_FILE *fp);
static void write_game_init03(PACKFILE *fp, const long seed);


void read_game_init(MEMORY_FILE *fp)
{
    return read_game_init03(fp);
}


void write_game_init(PACKFILE *fp, const long seed)
{
    write_game_init03(fp, seed);
}


static void read_game_init03(MEMORY_FILE *fp)
{
    char tmp[16];
    alrand_uint32_t seed;
    assert(fp);

    mmpk_fread(tmp, DEMO_GAME_INIT_STRLEN, fp);
    if (memcmp(tmp, DEMO_GAME_INIT_STRING, DEMO_GAME_INIT_STRLEN) != 0) {
	append_message("Demo broken");
	game_flags &= FLAG_QUIT;
	return;
    }

    seed	= mmpk_igetl(fp);
    num_players	= mmpk_getc(fp);
    player_stats[0].score = mmpk_igetl(fp);
    player_stats[1].score = mmpk_igetl(fp);
    alrand_seed(rng_state, seed);
}


static void write_game_init03(PACKFILE *fp, const long seed)
{
    assert(fp);

    pack_fputs(DEMO_GAME_INIT_STRING, fp);

    pack_iputl(seed, fp);
    pack_putc(num_players, fp);
    pack_iputl(player_stats[0].score, fp);
    pack_iputl(player_stats[1].score, fp);
}

/*--------------------------------------------------------------*/
/* Checkpoints.							*/
/*--------------------------------------------------------------*/

void read_demo_checkpoint(MEMORY_FILE *fp)
{
    assert(fp);

    if (player[0]) [player[0] readDemoCheckPoint:fp];
    if (player[1]) [player[1] readDemoCheckPoint:fp];
}


void write_demo_checkpoint(PACKFILE *fp)
{
    assert(fp);

    pack_fputs(DEMO_CHECKPOINT_STRING, fp);
    pack_putc('\n', fp);

    if (player[0]) [player[0] recordDemoCheckPoint:fp];
    if (player[1]) [player[1] recordDemoCheckPoint:fp];
}

/*--------------------------------------------------------------*/
/* Skipping.							*/
/*--------------------------------------------------------------*/

enum DEMO_SKIP_STATE demo_skip_segment(MEMORY_FILE *fp)
{
#define cmp(str,len)	(memcmp(fp->pos, str, len) == 0)

    int c;
    long offset;
    assert(fp);
    offset = mmpk_ftell(fp);

    while ((c = mmpk_getc(fp)) != EOF) {
	if (c != '[')
	    continue;

	mmpk_fseek(fp, -1, SEEK_CUR);

	if (cmp(DEMO_GAME_INIT_STRING, DEMO_GAME_INIT_STRLEN)) {
	    return DEMO_SKIP_TO_NEW_SEGMENT;
	}
	else if (cmp(DEMO_LEVEL_INIT_STRING, DEMO_LEVEL_INIT_STRLEN)) {
	    return DEMO_SKIP_TO_NEW_MAP;
	}
	else {
	    mmpk_fseek(fp, 1, SEEK_CUR);
	}
    }

    mmpk_fseek(fp, offset, SEEK_SET);
    return DEMO_NO_SKIP;

#undef cmp
}


/*--------------------------------------------------------------*/
/* Writing.							*/
/*--------------------------------------------------------------*/

static PACKFILE *fopen_demo2(char *str, const size_t str_len,
			     const int c, const struct tm *tm)
{
    struct stat stats;

    if (stat(str, &stats) != 0)
	return NULL;
    strftime(str+c, str_len-c, "/%Y%m%d-%H%M.rec", tm);

    return pack_fopen(str, "wb");
}


static PACKFILE *fopen_demo(char *str, const size_t str_len)
{
    PACKFILE *fp;
    int c;
    time_t t = time(0);
    struct tm *tm = localtime(&t);
    struct stat stats;
    
    /* Try raidem/demos/, but don't create one.  Windows' stat doesn't
       like the trailing '/'. */
    c = snprintf(str, str_len, "%s/demos", raid_binary_directory);
    fp = fopen_demo2(str, str_len, c, tm);
    if (fp) {
	append_message("Demo recorded: raidem/demos%s", str+c);
	return fp;
    }

#ifndef __MINGW__
    c = snprintf(str, str_len, "%s/demos", RAIDEM_DATA_DIR);
    fp = fopen_demo2(str, str_len, c, tm);
    if (fp) {
	append_message("Demo recorded: raidem/demos%s", str+c);
	return fp;
    }
#endif


    /* Try ~/.raidem/demos/, creating if it doesn't exist. */
    c = snprintf(str, str_len, "%s/%s/demos",
		 homedir, RAIDEM_CONFIG_PATHNAME);

    if (stat(str, &stats) != 0) {
#ifdef __MINGW__
	int mk = mkdir(str);
#else  /* Linux */
	int mk = mkdir(str, S_IREAD|S_IWRITE|S_IEXEC);
#endif

	if (mk != 0)
	    return NULL;
    }

    strftime(str+c, str_len-c, "/%Y%m%d-%H%M.rec", tm);
    fp = pack_fopen(str, "wb");
    if (fp) {
	append_message("Demo recorded: home/%s/demos%s",
		       RAIDEM_CONFIG_PATHNAME, str+c);
	return fp;
    }

    return NULL;
}


PACKFILE *open_demo_file_for_recording(void)
{
    PACKFILE *fp;

    fp = fopen_demo(filename, sizeof(filename));
    if (not fp) {
	demo_state = DEMO_NOT_ACTIVE;
	return NULL;
    }

    demo_state = DEMO_RECORDING;
    write_header03(fp);

    return fp;
}

/*--------------------------------------------------------------*/
/* Reading.							*/
/*--------------------------------------------------------------*/

static long pilot1_offset(void)
{
    return (DEMO_MAGIC_LEN +	/* Magic	*/
	    3 +			/* Version	*/
	    4 +			/* Time		*/
	    1 +			/* Difficulty	*/
	    1			/* Eye Candy	*/
	    );
}

static long pilot2_offset(void)
{
    return pilot1_offset() + HIGH_SCORE_NAME_LEN;
}


MEMORY_FILE *open_demo_file_for_playback(const char *filename)
{
    PACKFILE *fp = pack_fopen(filename, "rb");

    if (demo_file.data) {
	free(demo_file.data);
	demo_file.len  = 0;
	demo_file.data = NULL;
	demo_file.pos  = NULL;
    }

    if (not fp) {
	demo_state = DEMO_NOT_ACTIVE;
	return NULL;
    }

    demo_file.len  = file_size(filename);
    demo_file.data = malloc(demo_file.len);
    pack_fread(demo_file.data, demo_file.len, fp);
    demo_file.pos  = demo_file.data;
    pack_fclose(fp);

    /* Keep the config to restore later. */
    backup_config.num_players  = num_players;
    backup_config.difficulty   = difficulty;
    backup_config.candy_amount = candy_amount;

    demo_state = DEMO_PLAYBACK;
    read_header(&demo_file);

    return &demo_file;
}

/*--------------------------------------------------------------*/

void close_demo_file(void)
{
    if (current_demo_file) {
	pack_fclose(current_demo_file);
	current_demo_file = NULL;
    }

    if (demo_state == DEMO_RECORDING) {
	/* (Over)-write the pilot names, which were 0 filled. */
	FILE *fp = fopen(filename, "r+b");
	fseek(fp, pilot1_offset(), SEEK_SET);
	fwrite(high_score_name_buf[0], sizeof(char), strlen(high_score_name_buf[0]), fp);
	fseek(fp, pilot2_offset(), SEEK_SET);
	fwrite(high_score_name_buf[1], sizeof(char), strlen(high_score_name_buf[1]), fp);
	fclose(fp);
    }

    {
	if (demo_file.data) {
	    free(demo_file.data);
	    demo_file.data = NULL;
	}

	demo_file.len = 0;
	demo_file.pos = NULL;
    }

    if (demo_state == DEMO_PLAYBACK) {
	/* Restore the config from before demo playback. */
	num_players  = backup_config.num_players;
	difficulty   = backup_config.difficulty;
	candy_amount = backup_config.candy_amount;
    }

    demo_state = DEMO_NOT_ACTIVE;
}
