/* config.m,
 */

#include <allegro.h>
#include <assert.h>
#include <stdio.h>
#include "candy.h"
#include "common.h"
#include "config.h"
#include "demo.h"
#include "difficulty.h"
#include "game.h"
#include "sound.h"
#include "strlcpy.h"
#include "music-ad.h"
#include "music.h"
#include "unit.h"
#include "seborrhea/token.h"


#define GAME_CFG	"game.cfg"


static BOOL is_true(const char *str)
{
    /* Accept true, yes, y, etc. */
    return ((str[0] == 't') || (str[0] == 'T') ||
	    (str[0] == 'y') || (str[0] == 'Y'));
}

/*--------------------------------------------------------------*/

static BOOL parse_config(const char *cmd, Token *atok,
			 int *w, int *h, int *bpp)
{
#define ABORT_UNLESS(x)	if (not [atok isType:x]) return NO;

    if (streq(cmd, "player1")) {
	ABORT_UNLESS(TOKEN_STRING);
	strlcpy(high_score_name_buf[0], [atok getString], sizeof(high_score_name_buf[0]));
    }
    elif (streq(cmd, "player2")) {
	ABORT_UNLESS(TOKEN_STRING);
	strlcpy(high_score_name_buf[1], [atok getString], sizeof(high_score_name_buf[1]));
    }

    /* Video */
    elif (streq(cmd, "redit-w")) {
	ABORT_UNLESS(TOKEN_FIXNUM);
	*w = [atok getFixnum];
    }
    elif (streq(cmd, "redit-h")) { 
	ABORT_UNLESS(TOKEN_FIXNUM);
	*h = [atok getFixnum];
    }
    elif (streq(cmd, "depth")) { 
	ABORT_UNLESS(TOKEN_FIXNUM);
	*bpp = [atok getFixnum];
    }
    elif (streq(cmd, "buffering")) {
#ifndef NO_VIDEO_BITMAPS
	const char *arg = [atok getSymbol];
	ABORT_UNLESS(TOKEN_SYMBOL);

	if (streq(arg, "triple")) {
	    raid_use_page_flipping = YES;
	    raid_use_triple_buffering = YES;
	}
	else if (streq(arg, "flip")) {
	    raid_use_page_flipping = YES;
	    raid_use_triple_buffering = NO;
	}
	else {
	    raid_use_page_flipping = NO;
	    raid_use_triple_buffering = NO;
	}
#endif
    }
    
    /* Audio */
    elif (streq(cmd, "tags")) {
	ABORT_UNLESS(TOKEN_SYMBOL);
#ifndef NO_ID3TAGS
	music_ad_enabled = is_true([atok getSymbol]);
#else
        [atok getSymbol];
#endif
    }
    elif (streq(cmd, "sound")) {
	ABORT_UNLESS(TOKEN_FIXNUM);
	sound_vol = [atok getFixnum];
    }
    elif (streq(cmd, "music")) {
	ABORT_UNLESS(TOKEN_FIXNUM);
	music_vol = [atok getFixnum];
    }
    elif (streq(cmd, "reversed")) {
	ABORT_UNLESS(TOKEN_SYMBOL);
	sound_reverse_stereo = is_true([atok getSymbol]);
    }

    /* Gameplay. */
    elif (streq(cmd, "difficulty")) {
	ABORT_UNLESS(TOKEN_FIXNUM);
	difficulty = MID(DIFFICULTY_EASY, [atok getFixnum], DIFFICULTY_HARDEST);
    }
    elif (streq(cmd, "candy")) {
	ABORT_UNLESS(TOKEN_FIXNUM);
	candy_amount = MID(0, [atok getFixnum], TOTAL_CANDY_LEVELS-1);
    }
    elif (streq(cmd, "shadows")) {
	ABORT_UNLESS(TOKEN_FIXNUM);
	shadow_opacity = MID(0, [atok getFixnum], 255);
    }
    elif (streq(cmd, "demos")) {
	ABORT_UNLESS(TOKEN_SYMBOL);
	record_demos = is_true([atok getSymbol]);
    }

    else {
	fprintf(stderr,
		"[Config] Buggy " GAME_CFG " file (%s).  "
		"Unknown command.\n", cmd);
    }

    return YES;

#undef ABORT_UNLESS
}


static void read_config_body(FILE *fp, int *w, int *h, int *bpp)
{
    while (expect_char(fp, '(')) {
	Token *ctok, *atok;
	const char *cmd;

	ctok = read_symbol(fp);
	if (not ctok) {
	    fprintf(stderr,
		    "[Config] Buggy " GAME_CFG " file.  "
		    "Missing command token.\n");
	    return;
	}
	cmd = [ctok getSymbol];

	atok = read_token(fp);
	if (not atok) {
	    fprintf(stderr,
		    "[Config] Buggy " GAME_CFG " file (%s).  "
		    "Missing argument.\n", cmd);
	    ctok = [ctok free];
	    return;
	}

	if (not expect_char(fp, ')')) {
	    fprintf(stderr,
		    "[Config] Buggy " GAME_CFG " file (%s).  "
		    "Missing ).\n", cmd);
	    atok = [atok free];
	    ctok = [ctok free];
	    return;
	}

	/* Process the commands. */
	if (not parse_config(cmd, atok, w, h, bpp)) {
	    fprintf(stderr,
		    "[Config] Buggy " GAME_CFG " file (%s).  "
		    "Wrong argument type.\n", cmd);
	}

 	atok = [atok free];
 	ctok = [ctok free];
    }
}

/*--------------------------------------------------------------*/

FILE *fopen_cfg(const char *cfg, char *mode)
{
    FILE *fp;
    char filename[PATH_MAX];

    /* Try ~/.raidem/ */
    snprintf(filename, sizeof filename, "%s/%s/%s",
	     homedir, RAIDEM_CONFIG_PATHNAME, cfg);
    fp = fopen(filename, mode);

    if (fp)
	return fp;

    /* Try global directory. */
    snprintf(filename, sizeof filename, "%s/%s",
	     raid_binary_directory, cfg);
    fp = fopen(filename, mode);

    /* Note: We are allowed to return NULL. */
    return fp;
}


void read_config_file(int *w, int *h, int *bpp)
{
    /* Too lazy to print out error messages. */

    FILE *fp;
    Token *tok;

    fp = fopen_cfg(GAME_CFG, "r");
    if (not fp)
	return;

    set_token_reader_stdio();

    if (not expect_char(fp, '('))
	goto buggy;

    tok = read_symbol(fp);
    if (not tok)
	goto buggy;

    if (strneq([tok getSymbol], "config")) {
	[tok free];
	goto buggy;
    }
    
    read_config_body(fp, w, h, bpp);
    expect_char(fp, ')');

 buggy:
    fclose(fp);
    return;
}

/*--------------------------------------------------------------*/

void write_config_file(const int w, const int h, const int bpp)
{
    FILE *fp;

    fp = fopen_cfg(GAME_CFG, "w");
    if (not fp)
	return;

    fprintf(fp,
	    "; This file should be name 'game.cfg' and placed in either the\n"
	    "; root Raid'em directory, or inside ~/.raidem/\n"
	    "\n"
	    "(config\n"
	    " (player1\t\"%s\")\n"
	    " (player2\t\"%s\")\n\n",
	    high_score_name_buf[0], high_score_name_buf[1]);

    fprintf(fp,
	    " ;; Video\n"
	    " (redit-w\t%d)\n"
	    " (redit-h\t%d)\n"
	    " (depth\t\t%d)\n"
	    "\n"
	    " ; double:\tdouble buffering\n"
	    " ; flip:\tpage-flipping\n"
	    " ; triple:\ttriple buffering\n"
	    " (buffering\t%s)\n\n",
	    w, h, bpp,
#ifndef NO_VIDEO_BITMAPS
	    (raid_use_triple_buffering ? "triple" :
	     raid_use_page_flipping ? "flip" :
	     "double")
#else
	    "double"
#endif
	    );

    fprintf(fp,
	    " ;; Audio\n"
	    " (sound\t\t%d)\n"
	    " (music\t\t%d)\n"
	    " (reversed\t%s)\n"
	    " (tags\t\t%s)\n\n",
	    sound_vol, music_vol,
	    sound_reverse_stereo ? "yes" : "no",
            music_ad_enabled ? "yes" : "no"
            );

    fprintf(fp,
	    " ;; Gameplay\n"
	    " (difficulty\t%d)\n"
	    " (candy\t\t%d)\n"
	    " (shadows\t%d)\n"
	    " (demos\t\t%s))\n",
	    difficulty, candy_amount, shadow_opacity,
	    record_demos ? "yes" : "no");

    fclose(fp);
}
