/* collision.m,
 */

#include "collision.h"
#include "common.h"
#include "rotate.h"


static BOOL is_between(const double a, const double b, const double c)
{
    if (a <= c)
        return (a <= b && b <= c);
    else
        return (c <= b && b <= a);
}


static inline void swap(double *a, double *b)
{
    double c = *a;
    *a = *b;
    *b = c;
}


static inline double max(const double a, const double b)
{
    return (a > b) ? a : b;
}


/*--------------------------------------------------------------*/
/* Bounding Box Collision.					*/
/*--------------------------------------------------------------*/

/* bounding_box_collision:
 *
 * Returns YES if the 2 non-rotated rectangles intersect.
 * Note: ax1 <= ax2, ay1 <= ay2, etc.
 */
BOOL bounding_box_collision(const double ax1, const double ay1,
			    const double ax2, const double ay2,
			    const double bx1, const double by1,
			    const double bx2, const double by2)
{
    return ((ax1 <= bx2) && (ax2 >= bx1) && 
	    (ay1 <= by2) && (ay2 >= by1));
}


/*--------------------------------------------------------------*/
/* Rotated Bounding Box Collision.				*/
/*--------------------------------------------------------------*/

/* point_bounded:
 *
 * Returns YES if a point (x,y) is bounded by a rotated rectangle.
 */
static BOOL point_bounded(const double ox, const double oy,
			  const double x1, const double y1,
			  const double x2, const double y2,
			  const double x3, const double y3, 
			  const double x4, const double y4)
{
    /*
		    /\ (x2,y2)
		   /  \ (x4,y4)
		  /   /
		 / . / <-- (ox,oy)
	(x1,y1)	/   /
		\  /
		 \/ (x3, y3)
    */

    /* Check 1: objy below the line made by (x1,y1) (x2,y2) */
    if (oy < (ox-x1) * (y2-y1)/(x2-x1) + y1)
	return NO;

    /* Check 2: objy above the line made by (x3,y3) (x4,y4) */
    if (oy > (ox-x3) * (y4-y3)/(x4-x3) + y3)
	return NO;

    /* Check 3: objx right of  line made by (x1,y1) (x3,y3) */
    if (ox < (oy-y1) * (x3-x1)/(y3-y1) + x1)
	return NO;

    /* Check 4: objx left  of  line made by (x2,y2) (x4,y4) */
    if (ox > (oy-y2) * (x4-x2)/(y4-y2) + x2)
	return NO;

    return YES;
    
}


/* hline_intersect:
 *
 * Returns YES if the line 'y = mx+c, y:[y1, y2]' intersects the
 * horizontal line, 'y = yi, x:[x1, x2]'.
 */
static BOOL hline_intersect(const double m, const double c,
			    const double y1, const double y2,
			    const double yi, const double x1, const double x2)
{
    return (is_between(x1, (yi-c)/m, x2) &&
	    is_between(y1, yi, y2));
}


/* vline_intersect:
 *
 * Returns YES if the line 'y = mx+c, x:[x1, x2]' intersects the
 * vertical line, 'x = xi, y:[y1, y2]'.
 */
static BOOL vline_intersect(const double m, const double c,
			    const double x1, const double x2,
			    const double xi, const double y1, const double y2)
{
    return (is_between(x1, xi, x2) &&
	    is_between(y1, m*xi+c, y2));
}


/* rotated_bounding_box_collision:
 *
 * Check whether a rotated rectangle, a, and a non-rotated rectangle,
 * B, intersect.  The width and height of the rotated rectangle should
 * be the width/height before rotation.  A bounding box is done
 * automatically.
 */
BOOL rotated_bounding_box_collision(const double ax, const double ay,
				    const double aw, const double ah,
				    const double theta,
				    const double bx1, const double by1,
				    const double bx2, const double by2)
{
    double m, c;

    /* Coordinates of the rotated rectangle. */
    double x1, x2, x3, x4, y1, y2, y3, y4;
    double r = max(aw, ah) / 2.0;

    if (not bounding_box_collision(ax-r, ay-r, ax+r, ay+r,
				   bx1, by1, bx2, by2))
	return NO;

    /* Check if the centre of the rotated rectangle is inside the non-rotated
       one. */
    if (is_between(bx1, ax, bx2) &&
        is_between(by1, ay, by2))
        return YES;

    rotate(-aw/2.0, -ah/2.0, -theta, &x1, &y1);
    rotate( aw/2.0, -ah/2.0, -theta, &x2, &y2);
    rotate(-aw/2.0,  ah/2.0, -theta, &x3, &y3);
    rotate( aw/2.0,  ah/2.0, -theta, &x4, &y4);

    if (y1 > y3) {
	swap(&x1, &x4); swap(&y1, &y4);
	swap(&x2, &x3); swap(&y2, &y3);
    }

    /* Check if either corner of the non-rotated rectangle is inside the
       rotated one. */
    if (point_bounded(bx1-ax, by1-ay, x1, y1, x2, y2, x3, y3, x4, y4) ||
        point_bounded(bx2-ax, by1-ay, y1, y1, x2, y2, x3, y3, x4, y4) ||
        point_bounded(bx1-ax, by2-ay, y1, y1, x2, y2, x3, y3, x4, y4) ||
        point_bounded(bx2-ax, by2-ay, y1, y1, x2, y2, x3, y3, x4, y4))
        return YES;

    /* Check line connecting (x1,y1) (x4,y4) intersection with rect. */
    m = (y4-y1) / (x4-x1);
    c = (ay+y1) - m*(ax+x1);
    if (vline_intersect(m, c, ax+x1, ax+x4, bx1, by1, by2) ||
        vline_intersect(m, c, ax+x1, ax+x4, bx2, by1, by2) ||
        hline_intersect(m, c, ay+y1, ay+y4, by1, bx1, bx2) ||
        hline_intersect(m, c, ay+y1, ay+y4, by2, bx1, bx2))
        return YES;

    /* Check line connecting bl->tr's intersection with rect. */
    m = (y2-y3) / (x2-x3);
    c = (ay+y3) - m*(ax+x3);
    if (vline_intersect(m, c, ax+x3, ax+x2, bx1, by1, by2) ||
        vline_intersect(m, c, ax+x3, ax+x2, bx2, by1, by2) ||
        hline_intersect(m, c, ay+y3, ay+y2, by1, bx1, bx2) ||
        hline_intersect(m, c, ay+y3, ay+y2, by2, bx1, bx2))
        return YES;

    return NO;
}
