//-----------------------------------------------------------------------------
// File: diutil.cpp
//
// Desc: DirectInput support
//
// Copyright (C) 1995-1999 Microsoft Corporation. All Rights Reserved.
//-----------------------------------------------------------------------------
#define DIRECTINPUT_VERSION 0x0700
#include <dinput.h>


typedef struct _FFEffectObject
{
	LPDIRECTINPUTEFFECT pDIEffect;

	DWORD               dwEffectType;
	DIEFFECT            diEffect;
	DIENVELOPE          diEnvelope;
	union 
	{
	    DICUSTOMFORCE   dicustomf;
		DICONSTANTFORCE dicf;
		DIPERIODIC      dipf;
	};
} FFEFFECT;


#include "diutil.h"

//-----------------------------------------------------------------------------
// Function prototypes
//-----------------------------------------------------------------------------
static BOOL CALLBACK EnumDeviceProc( DIDEVICEINSTANCE* pdidi, VOID* pv );
static HRESULT       CreateJoystick( HWND, LPDIRECTINPUTDEVICE2 pdidDevice );
static HRESULT			CreateMouse(HWND, LPDIRECTINPUTDEVICE2 pdidDevice);




//-----------------------------------------------------------------------------
// Global variables
//-----------------------------------------------------------------------------
#define MAX_INPUT_DEVICES 20

static DIDEVICEINSTANCE     g_didiDevices[MAX_INPUT_DEVICES+1];
static DWORD                g_dwNumDevices = 0;
static LPDIRECTINPUT        g_pDI;    // DirectInput object



//-----------------------------------------------------------------------------
// Name: 
// Desc: 
//-----------------------------------------------------------------------------
HRESULT DIUtil_GetDevices( DIDEVICEINSTANCE** ppDevice, DWORD* pdwCount )
{
	if( NULL==ppDevice || NULL==pdwCount )
		return E_INVALIDARG;

	(*ppDevice) = g_didiDevices;
	(*pdwCount) = g_dwNumDevices;
	
	return S_OK;
}




//-----------------------------------------------------------------------------
// Name: 
// Desc: 
//-----------------------------------------------------------------------------
BOOL CALLBACK EnumDeviceProc( DIDEVICEINSTANCE* pdidi, VOID* v)
{
	if( g_dwNumDevices >= MAX_INPUT_DEVICES )
		return DIENUM_STOP;

	memcpy( &g_didiDevices[g_dwNumDevices++], pdidi, sizeof(DIDEVICEINSTANCE) );

	return DIENUM_CONTINUE;
}




//-----------------------------------------------------------------------------
// Name: Initialize()
// Desc: Creates and initializes DirectInput objects
//-----------------------------------------------------------------------------
HRESULT DIUtil_Initialize( HWND hWnd )
{
    HRESULT hr;

    // Create the base DirectInput object
    hr = DirectInputCreate( (HINSTANCE)GetWindowLong( hWnd,GWL_HINSTANCE ),
		                    DIRECTINPUT_VERSION, &g_pDI, NULL );
    if( FAILED(hr) )
		return E_FAIL;

    // Enumerate all DirectInput devices
	hr = IDirectInput_EnumDevices( g_pDI, 0, (LPDIENUMDEVICESCALLBACK)EnumDeviceProc, NULL,
                             DIEDFL_ATTACHEDONLY );
    if( FAILED(hr) )
        return E_FAIL;

	return S_OK;
}


LPDIRECTINPUTDEVICE2 DIUtil_CreateDevice( HWND hWnd, DIDEVICEINSTANCE* pdidi )
{
    LPDIRECTINPUTDEVICE  pdidTempDevice = NULL;
    LPDIRECTINPUTDEVICE2 pdidDevice     = NULL;
    HRESULT              hr;

	if( NULL == pdidi )
		return NULL;

    // Create a temporary "Device 1" object
    hr = IDirectInput_CreateDevice(g_pDI, &pdidi->guidInstance, &pdidTempDevice, NULL );
    if( FAILED(hr) )
		return NULL;

    // Get the Device2 interface
    hr = IDirectInput_QueryInterface( pdidTempDevice, &IID_IDirectInputDevice, (VOID*)&pdidDevice );
	IDirectInputDevice_Release(pdidTempDevice);
    if( FAILED(hr) )
		return NULL;

	if( ( pdidi->dwDevType & 0xff ) == DIDEVTYPE_JOYSTICK )
		hr = CreateJoystick( hWnd, pdidDevice );

	if( FAILED(hr) )
		return NULL;

	return pdidDevice;
}


HRESULT CreateJoystick( HWND hWnd, LPDIRECTINPUTDEVICE2 pdidDevice )
{
    DIPROPRANGE dipr;
    DIPROPDWORD dipdw;
    HRESULT     hr;

	// Set the joystick's data format.
	hr = IDirectInputDevice_SetDataFormat(pdidDevice, &c_dfDIJoystick );
	if( FAILED(hr) )
		return E_FAIL;

	// ForceFeedback requires Exclusive access to the device.
	hr = IDirectInputDevice_SetCooperativeLevel(pdidDevice, hWnd, DISCL_EXCLUSIVE | DISCL_BACKGROUND );
	if( FAILED(hr) )
		return E_FAIL;

    // SetParameter() will fail if a device is currently acquired, we are
    // doing this here in case we get careless and forget to call this
    // function either before we call Acquire() or after we call Unacquire().
    IDirectInputDevice_Unacquire(pdidDevice);

    // Set the axis ranges for the device. Use the same range for X and Y axes.
	// Set them fairly low since we're only concerned with left, right, etc.
    dipr.diph.dwSize       = sizeof(DIPROPRANGE);
	dipr.diph.dwHeaderSize = sizeof(dipr.diph);
	dipr.diph.dwHow        = DIPH_BYOFFSET;
	dipr.lMin              = RANGE_MIN;  // negative to the left/top
	dipr.lMax              = RANGE_MAX;  // positive to the right/bottom
    dipr.diph.dwObj        = DIJOFS_X;
    
	// Set the x-axis range property
    hr = IDirectInputDevice_SetProperty(pdidDevice, DIPROP_RANGE, &dipr.diph );
    if( FAILED(hr) ) return E_FAIL;

	// Set the y-axis range property
    dipr.diph.dwObj = DIJOFS_Y;
    hr = IDirectInputDevice_SetProperty(pdidDevice, DIPROP_RANGE, &dipr.diph );
    if( FAILED(hr) ) return E_FAIL;

	// Set the z-axis range property
    dipr.diph.dwObj = DIJOFS_Z;
    hr = IDirectInputDevice_SetProperty(pdidDevice, DIPROP_RANGE, &dipr.diph );
   // if( FAILED(hr) ) return E_FAIL;
    
    
	// Set the rx-axis range property
    dipr.diph.dwObj = DIJOFS_RX;
    hr = IDirectInputDevice_SetProperty(pdidDevice, DIPROP_RANGE, &dipr.diph );
    //if( FAILED(hr) ) return E_FAIL;

	// Set the ry-axis range property
    dipr.diph.dwObj = DIJOFS_RY;
    hr = IDirectInputDevice_SetProperty(pdidDevice, DIPROP_RANGE, &dipr.diph );
    //if( FAILED(hr) ) return E_FAIL;

	// Set the rz-axis range property
    dipr.diph.dwObj = DIJOFS_RZ;
    hr = IDirectInputDevice_SetProperty(pdidDevice, DIPROP_RANGE, &dipr.diph );
    //if( FAILED(hr) ) return E_FAIL;
    
    dipr.diph.dwObj = DIJOFS_SLIDER(0);
    hr = IDirectInputDevice_SetProperty(pdidDevice, DIPROP_RANGE, &dipr.diph );
    dipr.diph.dwObj = DIJOFS_SLIDER(1);
    hr = IDirectInputDevice_SetProperty(pdidDevice, DIPROP_RANGE, &dipr.diph );
    

    // Acquire the device
    if( FAILED( IDirectInputDevice_Acquire(pdidDevice) ) )
		return E_FAIL;

    return S_OK;
}




BOOL DIUtil_IsForceFeedback( LPDIRECTINPUTDEVICE2 pdidDevice )
{
	// Check to see if the device is a force feedback device
    DIDEVCAPS didc;
	didc.dwSize = sizeof(DIDEVCAPS);
	
	if( SUCCEEDED( IDirectInputDevice_GetCapabilities(pdidDevice, &didc ) ) )
	{
		if( didc.dwFlags & DIDC_FORCEFEEDBACK )
			return TRUE;
	}

	return FALSE;
}




//-----------------------------------------------------------------------------
// Name: DIUtil_CleanupDirectInput()
// Desc: Cleans up DirectInput objects
//-----------------------------------------------------------------------------
VOID DIUtil_CleanupDirectInput()
{
    // Release() base object
    if( g_pDI )
        IDirectInput_Release(g_pDI);
    g_pDI = NULL;
}




BOOL CALLBACK EnumEffectCallback( LPCDIEFFECTINFO pei, VOID* pv )
{
    GUID* pEffectGUID = (GUID*)pv;;

    // Report back the guid of the effect we enumerated
    if( pEffectGUID )
        *pEffectGUID = pei->guid;

	// Could continue and build a list of effects, but we'll stop after one.
    return DIENUM_STOP;
}




//-----------------------------------------------------------------------------
// Name:
// Desc:
//-----------------------------------------------------------------------------
HRESULT DIUtil_CreateEffectH( LPDIRECTINPUTDEVICE2 pdidDevice,
							 FFEFFECT* pEffectObject )
{
	GUID    guidEffect;
	HRESULT hr;

	// Release the effect before recreating it
    if( pEffectObject->pDIEffect )
        IDirectInputEffect_Release(pEffectObject->pDIEffect);
    pEffectObject->pDIEffect = NULL;

    // Enumerate for a periodic effect
    hr = IDirectInputDevice2_EnumEffects( pdidDevice, (LPDIENUMEFFECTSCALLBACK)EnumEffectCallback,
                                  &guidEffect, pEffectObject->dwEffectType );
    if( FAILED(hr) )
		return E_FAIL;

    // Create the effect
    hr = IDirectInputDevice2_CreateEffect( pdidDevice, &guidEffect, &pEffectObject->diEffect, &pEffectObject->pDIEffect, NULL );
    if( FAILED(hr) )
		return E_FAIL;

    return S_OK;
}




//-----------------------------------------------------------------------------
// Name:
// Desc:
//-----------------------------------------------------------------------------
HRESULT DIUtil_SetupConstantForceEffect( FFEFFECT* pEffectObject,
										 LPDIRECTINPUTDEVICE2 pdidDevice,
										 BOOL bFudgeForBounce )
{
    HRESULT         hr;
    DWORD           rgdwAxes[2];
    LONG            rglDirections[2];
    DICONSTANTFORCE dicf;

    if( NULL == pEffectObject || NULL == pdidDevice )
        return E_FAIL;

    // These fields are the same for all effects we will be creating
    pEffectObject->diEffect.dwSamplePeriod          = 0; // use default sample period
    pEffectObject->diEffect.dwTriggerButton         = DIEB_NOTRIGGER;
    pEffectObject->diEffect.dwTriggerRepeatInterval = 0;
    pEffectObject->diEffect.rgdwAxes                = rgdwAxes;
    pEffectObject->diEffect.rglDirection            = rglDirections;

    // Prepare the DICONSTANTFORCE structure
    dicf.lMagnitude  = 10000;

	if( bFudgeForBounce ) 
	{
	    // Axes and directions to use
		rgdwAxes[0]      = DIJOFS_X;
		rgdwAxes[1]      = DIJOFS_Y;
		rglDirections[0] = 0;
		rglDirections[1] = 0;
	}
	else
	{
		// Axes and directions to use
		rgdwAxes[0]      = DIJOFS_Y;
		rglDirections[0] = 1;
	}

    // Prepare the DIEFFECT structure
	if( bFudgeForBounce )
	{
		pEffectObject->diEffect.dwFlags    = DIEFF_OBJECTOFFSETS | DIEFF_POLAR;
		pEffectObject->diEffect.dwDuration = 200000;
		pEffectObject->diEffect.cAxes      = 2;
	}
	else
	{
		pEffectObject->diEffect.dwFlags    = DIEFF_OBJECTOFFSETS | DIEFF_CARTESIAN;
        pEffectObject->diEffect.dwDuration = 20000;
        pEffectObject->diEffect.cAxes      = 1;
	}
    pEffectObject->diEffect.lpEnvelope            = NULL;
    pEffectObject->diEffect.cbTypeSpecificParams  = sizeof(DICONSTANTFORCE);
    pEffectObject->diEffect.lpvTypeSpecificParams = &dicf;

    pEffectObject->dwEffectType = DIEFT_CONSTANTFORCE;

	hr = DIUtil_CreateEffectH( pdidDevice, pEffectObject );
	if( FAILED(hr) )
		return E_FAIL;

    return S_OK;
}

        
//-----------------------------------------------------------------------------
// Name:
// Desc:
//-----------------------------------------------------------------------------
HRESULT DIUtil_SetupFireForceEffect( FFEFFECT* pEffectObject,
								    LPDIRECTINPUTDEVICE2 pdidDevice )
{
    HRESULT         hr;
    DWORD           rgdwAxes[2];
    LONG            rglDirections[2];
    DIPERIODIC      dipf;

    // Make sure that we have a non-NULL device object
    if( NULL == pEffectObject || NULL == pdidDevice )
        return E_FAIL;


	// prepare the DIENVELOPE structure
	//
	// We want to shape the explode effect so that it starts
	// at it's peak and then fades out
	pEffectObject->diEnvelope.dwSize                   = sizeof(DIENVELOPE);
	pEffectObject->diEnvelope.dwAttackLevel            = 0;
	pEffectObject->diEnvelope.dwAttackTime             = 0;
	pEffectObject->diEnvelope.dwFadeLevel              = 0;
	pEffectObject->diEnvelope.dwFadeTime               = 1000000;

	// prepare the DIPERIODIC structure
	//
	// this is the type-specific data for this force
	dipf.dwMagnitude                    = 10000;
	dipf.lOffset                        = 0;
	dipf.dwPhase                        = 0;
	dipf.dwPeriod                       = 100000;

	// what axes and directions to use?
	rgdwAxes[0]                         = DIJOFS_X;
	rglDirections[0]                    = 0;

	// prepare the DIEFFECT structure
	//
	// fill in the force-specific values
	pEffectObject->diEffect.dwFlags                    = DIEFF_OBJECTOFFSETS | DIEFF_CARTESIAN;
	pEffectObject->diEffect.dwDuration                 = 1000000;
	pEffectObject->diEffect.cAxes                      = 1;
	pEffectObject->diEffect.lpEnvelope                 = &pEffectObject->diEnvelope;
	pEffectObject->diEffect.cbTypeSpecificParams       = sizeof(DIPERIODIC);
	pEffectObject->diEffect.lpvTypeSpecificParams      = &dipf;


    pEffectObject->dwEffectType = DIEFT_FFATTACK;

	hr = DIUtil_CreateEffectH( pdidDevice, pEffectObject );
	if( FAILED(hr) )
		return E_FAIL;

	return S_OK;
}

        
//-----------------------------------------------------------------------------
// Name:
// Desc:
//-----------------------------------------------------------------------------
HRESULT DIUtil_SetupPeriodicEffect( FFEFFECT* pEffectObject,
								    LPDIRECTINPUTDEVICE2 pdidDevice )
{
    HRESULT         hr;
    DWORD           rgdwAxes[2];
    LONG            rglDirections[2];
    DIPERIODIC      dipf;

    // Make sure that we have a non-NULL device object
    if( NULL == pEffectObject || NULL == pdidDevice )
        return E_FAIL;

    // These fields are the same for all effects we will be creating
    pEffectObject->diEffect.dwSamplePeriod          = 0; // use default sample period
    pEffectObject->diEffect.dwTriggerButton         = DIEB_NOTRIGGER;
    pEffectObject->diEffect.dwTriggerRepeatInterval = 0;
    pEffectObject->diEffect.rgdwAxes                = rgdwAxes;
    pEffectObject->diEffect.rglDirection            = rglDirections;

    // Prepare the DIENVELOPE structure. Shape the explode effect so that
	// it starts at it's peak and then fades out
    pEffectObject->diEnvelope.dwAttackLevel = 0;
    pEffectObject->diEnvelope.dwAttackTime  = 0;
    pEffectObject->diEnvelope.dwFadeLevel   = 0;
    pEffectObject->diEnvelope.dwFadeTime    = 1000000;

    // Pepare the DIPERIODIC structure
    dipf.dwMagnitude = 10000;
    dipf.lOffset     = 0;
    dipf.dwPhase     = 0;
    dipf.dwPeriod    = 100000;

    // Axes and directions to use
    rgdwAxes[0]      = DIJOFS_X;
    rglDirections[0] = 0;

    // Prepare the DIEFFECT structure
    pEffectObject->diEffect.dwFlags               = DIEFF_OBJECTOFFSETS | DIEFF_CARTESIAN;
    pEffectObject->diEffect.dwDuration            = 1000000;
    pEffectObject->diEffect.cAxes                 = 1;
    pEffectObject->diEffect.lpEnvelope            = &pEffectObject->diEnvelope;
    pEffectObject->diEffect.cbTypeSpecificParams  = sizeof(DIPERIODIC);
    pEffectObject->diEffect.lpvTypeSpecificParams = &dipf;

    pEffectObject->dwEffectType = DIEFT_PERIODIC;

	hr = DIUtil_CreateEffectH( pdidDevice, pEffectObject );
	if( FAILED(hr) )
		return E_FAIL;

    return S_OK;
}




//-----------------------------------------------------------------------------
// Name:
// Desc:
//-----------------------------------------------------------------------------
HRESULT DIUtil_PlayDirectionalEffect( FFEFFECT* pEffectObject,
									  LONG lDirection )
{
	if( pEffectObject )
	{
		if( pEffectObject->pDIEffect )
		{
			DIEFFECT effect;
			// Set the direction. Since this is a polar coordinate effect, we will
			// pass the angle in as the direction relative to the x-axis, and will
			// leave 0 for the y-axis direction. Direction is passed in in degrees,
			// we convert to 100ths of a degree to make it easier for the caller.
			LONG rglDirections[2];
			rglDirections[0] = lDirection;
			rglDirections[1] = 0;
			
			// Initialize DIEFFECT structure
			ZeroMemory( &effect, sizeof(DIEFFECT) );
			effect.dwSize       = sizeof(DIEFFECT);
			effect.dwFlags      = DIEFF_OBJECTOFFSETS | DIEFF_POLAR;
			effect.cAxes        = 2;
			effect.rglDirection = rglDirections;

			IDirectInputEffect_SetParameters(pEffectObject->pDIEffect, &effect, DIEP_DIRECTION );

		    // Play the effect
			return IDirectInputEffect_Start(pEffectObject->pDIEffect, 1, 0 );
		}
	}

	return E_INVALIDARG;
}




//-----------------------------------------------------------------------------
// Name:
// Desc:
//-----------------------------------------------------------------------------
HRESULT DIUtil_PlayEffect( FFEFFECT* pEffectObject )
{
	if( pEffectObject )
	{
		if( pEffectObject->pDIEffect )
		{
		    // Play the effect
			return IDirectInputEffect_Start(pEffectObject->pDIEffect, 1, 0 );
		}
	}

	return E_INVALIDARG;
}




//-----------------------------------------------------------------------------
// Name:
// Desc:
//-----------------------------------------------------------------------------
FFEFFECT* DIUtil_CreateEffect()
{
	FFEFFECT* pEffectObject = (FFEFFECT *)malloc(sizeof(FFEFFECT));

    // Initialize DIEFFECT and DIENVELOPE structures
    ZeroMemory( &pEffectObject->diEffect, sizeof(DIEFFECT) );
    pEffectObject->diEffect.dwSize   = sizeof(DIEFFECT);
	pEffectObject->diEffect.dwGain   = 7500L;

    ZeroMemory( &pEffectObject->diEnvelope, sizeof(DIENVELOPE) );
    pEffectObject->diEnvelope.dwSize = sizeof(DIENVELOPE);

	pEffectObject->pDIEffect = NULL;
	
	return pEffectObject;
}




//-----------------------------------------------------------------------------
// Name:
// Desc:
//-----------------------------------------------------------------------------
VOID DIUtil_DeleteEffect( FFEFFECT* pEffectObject )
{
	if( pEffectObject )
	{
		if( pEffectObject->pDIEffect )
			IDirectInputEffect_Release(pEffectObject->pDIEffect);

		free(pEffectObject);
	}
}



/*
//-----------------------------------------------------------------------------
// Name: DIUtil_ReacquireInputDevices()
// Desc: Reacquires DirectInput devices as needed
//-----------------------------------------------------------------------------
HRESULT DIUtil_ReacquireInputDevices()
{
    g_bKeyboardAcquired = FALSE;

    if( NULL == g_pdidKeyboard )
        return E_FAIL;

    g_pdidKeyboard->Acquire();
    g_bKeyboardAcquired = TRUE;

  
    
    return S_OK;
}


*/