/***********************************
Copyright (c) 2006, Richard Cassan
All rights reserved.
***********************************/
#include "Proj.h"

extern CCamera camera;

ProjArray::ProjArray()
{
    init();
}

ProjArray::~ProjArray()
{

}
    
void ProjArray::init(void)
{
    iNumProjs=0;
    for(int n=0 ; n<MAX_PROJS ; n++)
    {
        projs[n].destroy();
        projs[n].init();
    }    
}

Proj* ProjArray::retProj(int n)
{
    if(n>=0 && n<MAX_PROJS)
        return &projs[n];
    else
        return 0;
}

Proj* ProjArray::retNextProj(void)
{
    /*iNumProjs++;
    if(iNumProjs>MAX_PROJS)
        iNumProjs=1;
        return &projs[iNumProjs-1];
     */   
    Proj *ret=0;
    int n;
    for(n=0 ; n<MAX_PROJS ; n++)
    {
        if(projs[n].retExist()==false)
        {
            ret = &projs[n];
            break;
        }
    }
    if(n>=iNumProjs)
        iNumProjs = n;
        
    if(ret==0)
        ret = &projs[0];
    return ret;
}



void Proj::create(TProjInfo _info, PObject *shooter, float offsetAng)
{
    CVector2 vel=shooter->retVel();
    CVector2 accel = shooter->retAccel();
    if(offsetAng!=0)
    {
        vel = vel.rotate(offsetAng);
        accel = accel.rotate(offsetAng);   
    }
    create(_info , shooter->retPos(), /*shooter->retVel()*/vel, /*shooter->retAccel()*/accel);
}

void Proj::playFireSound(void)
{
    SfxSingleton::instance()->playSound(info.fireSound);    
}    

void Proj::die(void)
{
    SfxSingleton::instance()->playSound(info.deathSound);
    BRect::die();
    if(onDeathPartGen)
    {
        onDeathPartGen->setPosition(makeVector3(pos.retX(),pos.retY(),0));
        onDeathPartGen->makeSparks();
        
        if(info.iDamageClass==EXPLOSIVE)
            camera.shake(3);
    }
}


CVector3 Proj::setPartGenPos(void)
{
    if(info.iType==INST)    //if it is an instant proj, we want the part get at its base
    {
        CVector3 p;
        p.setVector(p1.retX(),p1.retY(),0);
        return p; 
    }    
    else
        return PObject::setPartGenPos();    //otherwise, treat it normally  
}    

void Proj::create(TProjInfo _info, CVector2 _pos, CVector2 _vel, CVector2 _accel)
{
    //s->setBCirc(def.fCircRadius,pos,vel);
    //setBCirc(2,_pos,_vel);
    init();    
    exist=true;

    info = _info;
    
    if(info.iType!=INST)    //if it is not an instant proj, then it will have a bounding circle
    {
        //give the projectile the radius defined in the TProjInfo
        //give it the position and velocity passed to the function
        setBCirc(info.fRadius,_pos,_vel);
    }
    else if(info.iType==INST)   //an INST projectile moves instantly and is modeled with a line
    {
        CVector2 t = _accel, t2 = _accel;
        t.setMag(100);
        t2.setMag(4);
        setLine(_pos+t2, _pos + t, _vel);
    }    
        
    //give it the right mass
    setMass(info.fMass);
    accel.zero();
    
    //info.partGen->init();
    if(info.partGen)
        addPartsGen(info.partGen);
    if(info.explosionGen)
        setOnDeathPartGen(info.explosionGen);
        //addPartsGen(info.explosionGen);
        
    lifetime = info.startingLifetime;
    if(info.lifeRandomness!=0)
        lifetime += (int)(info.startingLifetime/2) - (int)(((float)(rand()%info.startingLifetime)) * info.lifeRandomness); //modify it by the randomness factor

    if(info.iType == ENERGY)
    {
        //energy shots need not accelerate, and are always traveling at their top speed
        //give the velocity the same direction as the accel (the direction the ship is pointing)
        vel = _accel;
        //but give it its max velocity for magnitude
        vel.setMag(info.fMaxVel);
        
        //energy shots don't accel
        accel=_accel;
        accel.setMag(0);
        bDoesAccel=false;
    }
    else if(info.iType == MISSILE)
    {
        vel = _accel;
        vel.setMag(info.fMusselVel);
        accel = _accel;
        accel.setMag(info.fAccel);
        bDoesAccel=true;
    }
    else if(info.iType == MINE)
    {
        //mines do not move
        /*vel.zero();
        accel = _accel;
        accel.setMag(0);
        bDoesAccel=true;*/
        vel = _accel;
        vel.setMag(info.fMusselVel);
        accel = _accel;
        accel.setMag(info.fAccel);
        bDoesAccel=true;
    }
    else if(info.iType == INST)
    {
        vel.zero();
        accel = _accel;
        accel.setMag(0);
        bDoesAccel=false;
    }    
    
    armor = _info.armor;
    
    setModel(_info.model);
    
    projType = info.iType;
    //iType=CIRC;    
    exist=true;
}

void Proj::init(void)
{
    exist=false;
    nPartGen=0;
    target=0;
    ang_vel=0;
    ang_accel=0;
    vel.zero();
    accel.zero();
    dieCounter=0;
    lifetime=-1;        //-1 for lifetime means it won't run out of life and die
    
    genType=TYPE_PROJ;
    onDeathPartGen=0;
}

CollisionInfo Proj::reactToCollision(void)
{
    CollisionInfo colInfo;
    
    if(info.iDamageClass==ENERGY)
    {
        colInfo.armorDamage=(float)info.iDamage*.75;
        colInfo.shieldDamage=(float)info.iDamage;
        colInfo.projDamage=colInfo.armorDamage;
    }
    else if(info.iDamageClass==EXPLOSIVE)
    {
        colInfo.armorDamage=(float)info.iDamage;
        colInfo.shieldDamage=(float)info.iDamage*.75;
        colInfo.projDamage=colInfo.armorDamage;
    }
    else if(info.iDamageClass==PROJ_KILLER)
    {
        colInfo.armorDamage=(float)info.iDamage*.5;
        colInfo.shieldDamage=(float)info.iDamage*.5;
        colInfo.projDamage=(float)info.iDamage*10;
    }    
    else if(info.iDamageClass==BALLISTIC)
    {
        colInfo.armorDamage=(float)info.iDamage;
        colInfo.shieldDamage=(float)info.iDamage;
        colInfo.projDamage=colInfo.armorDamage*3;        
    }    
    else if(info.iDamageClass==EMP)
    {
        colInfo.armorDamage=0;
        colInfo.shieldDamage=(float)info.iDamage;
        colInfo.projDamage=(float)info.iDamage;  
    }    

    colInfo.owner_id = owner_id;
    colInfo.owner_team = owner_team;
    colInfo.genType=genType;
    colInfo.ptrToOwner = ptrToOwner;
    
    return colInfo;
}

void Proj::reactToCollisionInfo(CollisionInfo &colInfo)
{
    if(colInfo.owner_team != owner_team || info.iType==ENERGY)
    {
        if(info.dieOnImpact==true)
        {
            if(info.iType==ENERGY)  //if it is an energy proj, it will die no matter what
            {
                die();
                exist=false;
                //outtext(LOGFILE,"e=f\n");
            }
            else if(colInfo.genType==TYPE_PROJ)  //if it is not an energy  proj, and it hit another projectile
            {
                //it will loose health
                armor-=(int)colInfo.projDamage;
                if(armor<=0)
                {
                    die();
                    exist=false;
                }
            }
            else    //otherwise, just kill it to be safe
            {
                die();
                exist=false;
            }
        }
        else if(colInfo.genType==TYPE_SHIP)     //if it hit a ship
        {
                //it will loose health
                armor-=(int)colInfo.projDamage;
                if(armor<=0)
                {
                    die();
                    exist=false;
                }            
        }    
    }  
}

void Proj::destroy(void)
{
    exist=false;
    model = (Model*)NULL;
}

void Proj::home(void)
{
    if(target)
    {
        if(target->retExist()==true)
        {
            CVector2 l,a;
            
            l = target->retPos() - pos; //line connecting proj to target
            a = accel;
            
            l.setVector(target->retPos().retX()-pos.retX(),
                        target->retPos().retY()-pos.retY(),false);
                        
            a.setVector(a.retX(),a.retY(),false);
            a=a.findNormal();
             
            if( (a|l) > 0)
            {
                ang_vel=info.fMaxAngVel;
                //vel*=info.fDampening;
            }
            else if((a|l) < 0)
            {
                ang_vel=-info.fMaxAngVel;
                //vel*=info.fDampening;
            }
            else
                ang_vel=0;
                
        }
        else
        {
            if(dieCounter==0)
                dieCounter = 10+rand()%20;
                
            if(dieCounter==1)
                die();
            
            if(dieCounter>0)
                dieCounter--;
        }
    }
    else
    {
        if(dieCounter==0)
            dieCounter = 10+rand()%20;
            
        if(dieCounter==1)
            die();
        
        if(dieCounter>0)
            dieCounter--;        
    }    
}

void Proj::update(BRect bounds)
{
    if(exist==true)
    {
        PObject::update();
        if(info.bHoming==true)
        {
            home();
            PObject::dampenVelocity(info.fDampening);
        }
        
        if(info.iType==MINE)
        {
            PObject::dampenVelocity(info.fDampening);
        }    
        
        if(info.bHoming==true || info.iType==MISSILE)
            PObject::accelerate(info.fAccel);
             
        //energy shots are always moving at the same speed (their max speed)
        if(info.iType==ENERGY)
        {
            vel.setMag(info.fMaxVel);
        }
        
        //if its not an inst projectile, it is a movie projectile, so it must be killed if it leaves the arena
        if(info.iType!=INST)    
        {
            if( (GameSettingsSingleton::instance())->getProjWrap() == false )
            {
                if(pos.retX()+radius < bounds.retP1().retX())
                    exist=false;
                else if(pos.retX()-radius > bounds.retP2().retX() )
                    exist=false;    
                    
                if(pos.retY()+radius < bounds.retP1().retY())
                    exist=false;
                else if(pos.retY()-radius > bounds.retP2().retY())
                    exist=false;
            }    
            else
            {    
                if(pos.retX()+radius < bounds.retP1().retX() /*bounds.retPos().retX()+bounds.retTl_offset().retX()*/)
                    pos.setX(bounds.retP2().retX()+radius);
                else if(pos.retX()-radius > bounds.retP2().retX() /*bounds.retPos().retX()+bounds.retBr_offset().retX()*/)
                    pos.setX(bounds.retP1().retX()-radius);    
                
                if(pos.retY()+radius < bounds.retP1().retY() /*bounds.retPos().retY()+bounds.retTl_offset().retY()*/)
                    pos.setY(bounds.retP2().retY()+radius);
                else if(pos.retY()-radius > bounds.retP2().retY() /*bounds.retPos().retY()+bounds.retBr_offset().retY()*/)
                    pos.setY(bounds.retP1().retY()-radius);     
                setPos(pos);
            }    
        }      
          
        if(lifetime>0)
            lifetime--;
        else if(lifetime==0)
        {
            die();
        }       
    }
}

void Proj::draw(bool wireframe)
{
    float nodeSpacing=.7;
    float lineLength;
    float numNodes;
    CVector2 p, tPos;
    CVector2 offset;
                
    float tx,ty;
    float rColour;

    
    if(info.iType==INST)    //if it is an instant moving (ie line) projectile
    {
        glDisable(GL_LIGHTING);
        glEnable(GL_BLEND);
        glBlendFunc(GL_SRC_ALPHA,GL_ONE);
        
        //so some special drawing code
        glBegin(GL_LINE_LOOP);
    
            //wall[n].draw();
            lineLength = slope.retMag();
            numNodes = lineLength / nodeSpacing;

            
            glColor4f(.8,.2,0,(float)lifetime/(float)info.startingLifetime);
            glVertex3f(p1.retX(),p1.retY(),0);
            
            rColour = (float)(rand()%100)/300;
            glColor4f(1,.6-rColour,1,.6*((float)lifetime/(float)info.startingLifetime) );
            
            p = p1;
            
            for(int n2=1 ; n2<numNodes-1 ; n2++)
            {

                /*tx+=slope.retX()*nodeSpacing;
                ty+=slope.retY()*nodeSpacing;*/
                p=makeVector2(p.retX()+slope.retX()*nodeSpacing,p.retY()+slope.retY()*nodeSpacing,true);
                

                offset= slope.findNormal();
                offset.isPosition(true);
                offset.setMag( (float)(rand()%100)/50 - 1 );

                tPos=makeVector2(p.retX()+offset.retX()*offset.retMag(),p.retY()+offset.retY()*offset.retMag(),true);

                glVertex3f(tPos.retX(),tPos.retY(),0);
            }

        glEnd();
        
        glDisable(GL_BLEND);
        glEnable(GL_LIGHTING);
    }
    else    //otherwise, just use the super's draw function
        PObject::draw(wireframe);    
}    
