/*   ______   ___    ___
    /\  _  \ /\_ \  /\_ \                             
    \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
     \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
      \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
       \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
        \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
                                       /\____/
         A game programming library    \_/__/   Version 4.2.0
         
         Space Battle using Dev-C++ 4.9.9.2   (c) 2006 by Gia T. Nguyen */

// ********************************************************************************
//
// This rudimentary 2-D game is based on the popular BattleShip board game 
// with an outer space theme.  It mimics the Milton Bradley Eletronic Battleship
// Advanced Mission toy in functionality.  This implementation is slightly
// different from the toy itself.
// 
//           Carrier    - 5 hits to sink
//           Battleship - 4 hits to sink
//           Destroyer  - 3 hits to sink
//           Corvette   - 2 hits to sink
//           Gunboat    - 2 hits to sink
//
//           Recon          - checks a sector (9 squares) for possible enemy ships
//           Salvos         - 1 shot per surviving ship
//           Advanced Salvo - custom salvo patterns per ship type
//
//           5 Levels of Difficulty:
//
//             Cadet      - Basic Battleship game, 1 shot per player, 
//                          hit once, sink entire ship (computer finishes off target immediately)
//             Ensign     - Like Cadet level, with Salvo, no Recon
//                          computer fires selectively using AI path finding,
//                          human-like limitation, clock-wise direction, with 4 shots per direction,
//                          new paths are pushed onto stack while tracking current target
//             Lieutenant - Like Ensign level, has 6 Recon missions,
//             Commander  - Like Lieutenant level, has 3 Recon missions, advanced Salvo
//             Captain    - Like Commander level, no Recon mission, advanced Salvo,
//                          computer finishes off current target with minimal shots in each direction
//                          while still stacking new paths (hardest level)
//
// This C++ program compiles under Windows XP on a Pentium IV CPU
// using the Dev-C++ IDE with MinGW32, and the Allegro game programming library.
// It utilizes 32-bit color, 800x600 screen resolution, and .WAV sound samples.
//
// Last Updated:  February 23, 2006
//
// Credits:
//           
// Ari Feldman, author of the SpriteLib GPL graphics library
// See more information at http://www.flyingyogi.com/fun/spritelib.html
// *******************************************************************************


#include "spacebattle.h"               /* Program Header */

/*** Begin of Program ***/

int main(int argc, char *argv[]) {                           /* Main entry */

    if (argc == 2) {
       if (strcmp(argv[1],"/d") >= 0) debug = true;
       if (strcmp(argv[1],"/D") >= 0) debug = true;
       if (strcmp(argv[1],"-d") >= 0) debug = true;
       if (strcmp(argv[1],"-D") >= 0) debug = true;
    }

    /* Main */
    
    init();    
    while (play());
    finish();
            
}
END_OF_MAIN()                          /* Allegro exit macro */

/*** End of Program ***/

void init() {                          /* Initialization */
	
    allegro_init();

    if (!exists(MASTER_CONFIG)) {
       allegro_message("Missing Allegro config file.");
       exit(1);
    }
    
    sound_on = get_config_int("spacebattle","sound",0);
    playlevel=get_config_int("spacebattle","level",0);
    volume = get_config_int("media","volume",0);
    int videox = get_config_int("media","videox",0);
    int videoy = get_config_int("media","videoy",0);
    int color = get_config_int("media","color",0);
  
    set_color_depth(color);	                               
	if (set_gfx_mode(GFX_AUTODETECT_WINDOWED, videox, videoy, 0, 0) != 0) {
       allegro_message("Video set-up error.  Program aborted.");
       exit(1);
    }
      
    install_timer(); 
	install_keyboard();
	install_mouse();
    enable_hardware_cursor();
        
    rndgen(0);	
	set_window_title("Space Battle");
   
    get_playlevel();
    switch (playlevel) {
           case CADET:
           case ENSIGN:
           case CAPTAIN:
                max_recon = 0;
                break;     
           case LIEUTENANT:
                max_recon = 6;
                break;
           case COMMANDER:
                max_recon = 3;
                break;
    }                	

    load_binaries();    
    load_bmp();
   	load_wav();
    draw_bkg();
	draw_grid(HUMAN);
	draw_grid(COMPUTER);
	debug_computer_data = load_datafile(COMPUTER,&skip);
	load_gameboard(COMPUTER);
	init_grid();
    draw_labels();
    show_tip();
    show_gridlegend();
    
    if (sound_on) {	
        select_mouse_cursor(MOUSE_CURSOR_BUSY);
        show_mouse(screen);	
        play_sample(intro_snd,volume,0,1000,0);
        rest(PAUSE);
    }    
    destroy_sample(intro_snd); 

    select_mouse_cursor(MOUSE_CURSOR_ARROW);
    show_mouse(screen);

    draw_scores(COMPUTER);

    if (playlevel > LIEUTENANT)
       show_salvolegend();
    if (playlevel > ENSIGN && playlevel < CAPTAIN) 
       show_reconlegend();
    
    set_config_int("spacebattle","level",playlevel);

}
   
bool play() {                                /* Accepts input until done or exit */

     bool click_ok;
     Coord firecoord;
     
     int res = get_inputs();
     switch (res) {
            case MOUSE_FIRE:
               if (load_data) {
                   firecoord = calc_mousepos(COMPUTER,&click_ok);
                   if (click_ok) {
                      if (sound_on) play_sample(blip_snd,volume,0,1000,0);
                      int shots_fired = 0;           
                      if (advanced_salvo > 0) {
                          bool fire_ok = false;
                          switch (advanced_salvo) {
                                  case CARRIER_SALVO1:
                                  case CARRIER_SALVO2:
                                       if (f5 < CARRIER) fire_ok = true;
                                       break;
                                  case BATTLESHIP_SALVO1:
                                  case BATTLESHIP_SALVO2:
                                       if (f4 < BATTLESHIP) fire_ok = true;
                                       break;
                                  case DESTROYER_SALVO1:
                                  case DESTROYER_SALVO2:
                                       if (f3 < DESTROYER) fire_ok = true;
                                       break;                                         
                          }
                          if (fire_ok) {
                              shots_fired = fire_advanced(firecoord,advanced_salvo);
                              rest(HALF_SEC);
                          } else
                              advanced_salvo = 0;   
                          if (shots_fired > escore) shots_fired = escore;
                          select_mouse_cursor(MOUSE_CURSOR_BUSY);
                          show_mouse(screen);
                          for (int i=0;i<shots_fired;i++) {
                                counter_shot();
                                rest(HALF_SEC);
                          }
                          select_mouse_cursor(MOUSE_CURSOR_ARROW);
                          show_mouse(screen);

                      } else {
                          select_mouse_cursor(MOUSE_CURSOR_BUSY);
                          show_mouse(screen);
                          if (fire_shot(firecoord)) {
                             rest(HALF_SEC);
                             counter_shot();
                          }
                          select_mouse_cursor(MOUSE_CURSOR_ARROW);
                          show_mouse(screen);
                      }
                   }      
               } else {
                   firecoord = calc_mousepos(HUMAN,&click_ok);
                   if (click_ok) {
                      rest(TENTH_SEC); 
                      manual_load(firecoord);
                   }                                     
               }    
               break;      
            case MOUSE_RECON:
                if (load_data && recons < max_recon) {
                   firecoord = calc_mousepos(COMPUTER,&click_ok);
                   if (click_ok) {
                      recon(firecoord);
                      erase_probe();
                      rest(HALF_SEC);
                   }
                } else {
                     if (!load_data) {  
                        rest(TENTH_SEC);
                        manual_load();
                     }   
                }
                break;     
            case AUTO_FIRE:
                 if (playlevel > CADET && playlevel < COMMANDER) {
                     select_mouse_cursor(MOUSE_CURSOR_BUSY);
                     show_mouse(screen);
                     if (sound_on) play_sample(blip_snd,volume,0,1000,0);
                     if (load_data) {
                         for (int i=0;i<fscore;i++) {
                             fire_shot(auto_fire(HUMAN));
                             rest(HALF_SEC);
                         }       
                         for (int i=0;i<escore;i++) {
                                counter_shot();
                                rest(HALF_SEC);
                         }
                     }
                     select_mouse_cursor(MOUSE_CURSOR_ARROW);
                     show_mouse(screen);
                 }
                 break;
            case LOAD_DATA:
                 if (!load_data && !manual_click) {
                    load_formation();
                    rest(HALF_SEC);
                    load_data = true;
                 }     
            case SOUND_TOGGLE:                   
                 break;
            case CARRIER_SALVO1:
                 if (f5 < CARRIER)
                    advanced_salvo = CARRIER_SALVO1;
                 break;
            case CARRIER_SALVO2:
                 if (f5 < CARRIER)
                    advanced_salvo = CARRIER_SALVO2;
                 break;
            case BATTLESHIP_SALVO1:
                 if (f4 < BATTLESHIP)
                    advanced_salvo = BATTLESHIP_SALVO1;
                 break;
            case BATTLESHIP_SALVO2:
                 if (f4 < BATTLESHIP) 
                    advanced_salvo = BATTLESHIP_SALVO2;
                 break;
            case DESTROYER_SALVO1:
                 if (f3 < DESTROYER) 
                    advanced_salvo = DESTROYER_SALVO1;
                 break;
            case DESTROYER_SALVO2:
                 if (f3 < DESTROYER) 
                    advanced_salvo = DESTROYER_SALVO2;
                 break;
            case SINGLE_SALVO:
                 advanced_salvo = 0;
                 break;     
     }
     
     if (escore > 0 && fscore > 0)    
        return true;                                  
     else 
        return false;                                 
}
                    
void finish() {                               /* Game Over */
    
    for (int x=0;x<MAX_X;x++)             
         for (int y=0;y<MAX_Y;y++)  
             if (enemy_grid[x][y] >= '1' && enemy_grid[x][y] <= '5')
                draw_bitmap(enemy,enemycoord.x,enemycoord.y,x,y);
    
    clear_keybuf();

	masked_blit(gameover,screen,0,0,362,354,300,60);

    if (fscore == 0)
        textout_centre_ex(screen,font,"YOU LOSE! PRESS ANY KEY ...",512,424,GREEN,-1);
    else
        textout_centre_ex(screen,font,"YOU WIN! PRESS ANY KEY ...",512,424,GREEN,-1);
	destroy_bitmap(gameover);
    
    bool wait = true;
    while (wait)
      if (keypressed()) deinit();
   
}

void deinit() {
     
     deinit(0);
     
}

void deinit(int err_code) {                          /* De-initialization prior to exit */

    clear_keybuf();
    
    destroy_sample(ping_snd);
    destroy_sample(explode_snd);
    destroy_sample(done_snd);
    destroy_sample(blip_snd);
    destroy_sample(announce_snd);
   
    destroy_bitmap(enemy);
    destroy_bitmap(friendly);
    destroy_bitmap(explosion);
    destroy_bitmap(missed);
    
    // unload_datafile(data);
    
    allegro_exit();
	exit(err_code);                             /* Exits Program */

}

int get_inputs() {         /* Main routine for keyboard and mouse ints */

   	 clear_keybuf();

     int res = 0;
     while (res == 0) {            
           if (mouse_b & LEFT_BUTTON) res = MOUSE_FIRE;
           if (mouse_b & RIGHT_BUTTON) res = MOUSE_RECON;      
           if (keypressed()) {
                if (key[KEY_F1]) res = LOAD_DATA;
                if (key[KEY_F2]) {         
                   sound_on = !sound_on;   
                   res = SOUND_TOGGLE;   
                }
                if (key[KEY_F4]) res = SINGLE_SALVO;
                if (key[KEY_F5]) res = CARRIER_SALVO1;
                if (key[KEY_F6]) res = CARRIER_SALVO2;
                if (key[KEY_F7]) res = BATTLESHIP_SALVO1;
                if (key[KEY_F8]) res = BATTLESHIP_SALVO2;
                if (key[KEY_F9]) res = DESTROYER_SALVO1;
                if (key[KEY_F10]) res = DESTROYER_SALVO2;                
                if (key[KEY_ESC]) deinit(); 
                if (key[KEY_Z]) res = AUTO_FIRE;    
           }
    }
   	
    clear_keybuf();

    return res;      
}

int fire_advanced(Coord firecoord,int salvo_type) {

     Coord newcoord;
     int x,y,cnt=0;
     switch (salvo_type) {
            case CARRIER_SALVO1:
            case BATTLESHIP_SALVO1:     
                 if (salvo_type == CARRIER_SALVO1) {
                    fire_shot(firecoord);
                    cnt++;
                 }   
                 for (int i=0;i<4;i++) {
                     switch (i) {
                            case 0:          
                                 x = firecoord.x - 1;
                                 y = firecoord.y - 1;
                                 break;
                            case 1:
                                 x = firecoord.x + 1;
                                 y = firecoord.y - 1;
                                 break;
                            case 2:
                                 x = firecoord.x - 1;
                                 y = firecoord.y + 1;
                                 break;
                            case 3:
                                 x = firecoord.x + 1;
                                 y = firecoord.y + 1;
                                 break;
                     }                                                                                
                     if (isvalid(x,y)) {
                        newcoord.x = x;
                        newcoord.y = y;
                        fire_shot(newcoord);
                        rest(TENTH_SEC);
                        cnt++;
                     }
                 }
                 break;    
            case CARRIER_SALVO2:
            case BATTLESHIP_SALVO2:
                 if (salvo_type == CARRIER_SALVO2) {
                    fire_shot(firecoord);
                    cnt++;
                 }
                 for (int i=0;i<4;i++) {
                     switch (i) {
                            case 0:          
                                 x = firecoord.x - 1;
                                 y = firecoord.y;
                                 break;
                            case 1:
                                 x = firecoord.x + 1;
                                 y = firecoord.y;
                                 break;
                            case 2:
                                 x = firecoord.x;
                                 y = firecoord.y - 1;
                                 break;
                            case 3:
                                 x = firecoord.x;
                                 y = firecoord.y + 1;
                                 break;
                     }                                                                                
                     if (isvalid(x,y)) {
                        newcoord.x = x;
                        newcoord.y = y;
                        fire_shot(newcoord);
                        rest(TENTH_SEC);
                        cnt++;
                     }
                 }    
                 break;
            case DESTROYER_SALVO1:
                 fire_shot(firecoord);
                 for (int i=0;i<2;i++) {
                     switch (i) {
                            case 0:          
                                 x = firecoord.x - 1;
                                 y = firecoord.y;
                                 break;
                            case 1:
                                 x = firecoord.x + 1;
                                 y = firecoord.y;
                                 break;
                     }                                                                                
                     if (isvalid(x,y)) {
                        newcoord.x = x;
                        newcoord.y = y;
                        fire_shot(newcoord);
                        rest(TENTH_SEC);
                        cnt++;
                     }
                 }    
                 break;
            case DESTROYER_SALVO2:
                 fire_shot(firecoord);
                 for (int i=0;i<2;i++) {
                     switch (i) {
                            case 0:          
                                 x = firecoord.x;
                                 y = firecoord.y  -1;
                                 break;
                            case 1:
                                 x = firecoord.x;
                                 y = firecoord.y + 1;
                                 break;
                     }                                                                                
                     if (isvalid(x,y)) {
                        newcoord.x = x;
                        newcoord.y = y;
                        fire_shot(newcoord);
                        rest(TENTH_SEC);
                        cnt++;
                     }
                 }    
                 break;
     }
     
     return cnt;
     
}

bool fire_shot(Coord firecoord) {                       /* Human player taking a shot */   

    bool res = false, saveprevhit = false;
    char a = enemy_grid[firecoord.x][firecoord.y];                  
    
    if (a == '-') {
        enemy_grid[firecoord.x][firecoord.y] = '+';                  
        draw_bitmap(missed,enemycoord.x,enemycoord.y,firecoord.x,firecoord.y);
        res = true;
    }   

    if (a >= '1' && a <= '5') {

        enemy_grid[firecoord.x][firecoord.y] = '@';
        draw_bitmap(enemy,enemycoord.x,enemycoord.y,firecoord.x,firecoord.y);
        draw_bitmap(explosion,enemycoord.x,enemycoord.y,firecoord.x,firecoord.y);
 
        if (sound_on) play_sample(explode_snd,volume,0,1000,0);
 
        draw_score();
        
        if (a == '1') e1++;
        if (a == '2') e2++;
        if (a == '3') e3++;
        if (a == '4') e4++;
        if (a == '5') e5++;
        

        if (playlevel == CADET) { // Modified Rule for CADET level, hit once, sink entire ship

            if (a == '1' && e1 == 1) saveprevhit = true;
            if (a == '2' && e2 == 1) saveprevhit = true;               
            if (a == '3' && e3 == 1) saveprevhit = true;               
            if (a == '4' && e4 == 1) saveprevhit = true;               
            if (a == '5' && e5 == 1) saveprevhit = true;
            
            if (saveprevhit) {                                         // first hit, push path onto stack
                Hit curr_hit = {firecoord.x,firecoord.y,a};             // for subsequent processing
                humanlist.push(curr_hit);
                rest(TENTH_SEC);
            }
                  
            Coord firecoord;           
            if (a == '1') {
                while (e1 < GUNBOAT) {
                    firecoord = compute_fire(HUMAN);
                    fire_shot(firecoord);
                }
            }   
            if (a == '2') {
                while (e2 < CORVETTE) {
                    firecoord = compute_fire(HUMAN);
                    fire_shot(firecoord);
                }
            }         
            if (a == '3') {
                while (e3 < DESTROYER) {
                    firecoord = compute_fire(HUMAN);
                    fire_shot(firecoord);
                }
             
            }         
            if (a == '4') {
                while (e4 < BATTLESHIP) {
                    firecoord = compute_fire(HUMAN);
                    fire_shot(firecoord);
                }
             
            }         
            if (a == '5') {
                while (e5 < CARRIER) {
                    firecoord = compute_fire(HUMAN);
                    fire_shot(firecoord);
                }
            }         
        }
               
        res = true;
    }
   
    return res;  
}

void counter_shot() {                 /* Computer taking a shot */

      Coord firecoord;
      
      bool saveprevhit = false;
      bool eraseprevhit = false;

      firecoord = compute_fire(COMPUTER);  

      char b = friendly_grid[firecoord.x][firecoord.y];

      if (b == '-') {
         friendly_grid[firecoord.x][firecoord.y] = '+';                  
         draw_bitmap(missed,friendlycoord.x,friendlycoord.y,firecoord.x,firecoord.y);
      }   

      if (b >= '1' && b <= '5') {

         friendly_grid[firecoord.x][firecoord.y] = '@';                  
         draw_bitmap(explosion,friendlycoord.x,friendlycoord.y,firecoord.x,firecoord.y);
       
         if (sound_on) play_sample(explode_snd,volume,0,1000,0);
         
         if (b == '1') f1++;
         if (b == '2') f2++;
         if (b == '3') f3++;
         if (b == '4') f4++;
         if (b == '5') f5++;
         
         draw_score();
            
         if (b == '1' && f1 == 1) saveprevhit = true;
         if (b == '2' && f2 == 1) saveprevhit = true;               
         if (b == '3' && f3 == 1) saveprevhit = true;               
         if (b == '4' && f4 == 1) saveprevhit = true;               
         if (b == '5' && f5 == 1) saveprevhit = true;
        
         if (saveprevhit) {                                         // first hit, push path onto stack
            Hit curr_hit = {firecoord.x,firecoord.y,b};             // for subsequent processing
            hitlist.push(curr_hit);
            rest(TENTH_SEC);
         }

        if (playlevel == CADET) {           //Modified rule for CADET, hit once, sink entire ship
            Coord firecoord;           
            if (b == '1')
                while (f1 < GUNBOAT)
                    counter_shot();
            if (b == '2')
                while (f2 < CORVETTE)
                    counter_shot();
            if (b == '3')
                while (f3 < DESTROYER)
                    counter_shot();
            if (b == '4')
                while (f4 < BATTLESHIP)
                    counter_shot();
            if (b == '5')
                while (f5 < CARRIER)
                    counter_shot();
        }
         
      }

}
     
Coord auto_fire(int who) {              /* Auto-selection of firing coordinates */

     int x,y;     
     bool found = false;

     while (!found) {
        x = rndgen(MAX_X);
        y = rndgen(MAX_Y);
        switch (who) {
            case HUMAN:   
                if (enemy_grid[x][y] != '@' && enemy_grid[x][y] != '+') 
                        found = true;
                break;
            case COMPUTER:           
                if (friendly_grid[x][y] != '@' && friendly_grid[x][y] != '+')
                        found = true;
                break;
       }
     }        

     Coord firecoord = {x,y};
     return firecoord;
     
}

Coord compute_fire(int who) {  /* computes future moves based on previous hit */

    Coord firecoord = {-1,-1};
    Hit prevhit;
    
    if (who == COMPUTER && hitlist.empty()) {  
       firecoord = auto_fire(COMPUTER);                 // if no existing path, randomize next move
    } else {
        bool done = false;
        bool done2 = false;
        int i=0;
        int firing_direction=0;
        int x = -1, y = -1;
        int tmpx=0,tmpy=0;
        if (who == COMPUTER)
           prevhit = hitlist.top();                  // get current path from stack
        else
           prevhit = humanlist.top();   
        while (!done) {              
            done2 = false;         
            firing_direction++;
            if (firing_direction < ROUND_TRIP) {  // path-finding, clock-wise direction
                i = 0;
                while (i < 5 && !done2) {     // attempt at least 4 shots in a given direction
                      i++;                    
                      switch (firing_direction) {
                             case UP:
                                  tmpx = prevhit.x;
                                  tmpy = prevhit.y - i;
                                  break;
                             case DOWN:
                                  tmpx = prevhit.x;
                                  tmpy = prevhit.y + i;
                                  break;
                             case LEFT:
                                  tmpx = prevhit.x - i;
                                  tmpy = prevhit.y;
                                  break;
                             case RIGHT:
                                  tmpx = prevhit.x + i;
                                  tmpy = prevhit.y;
                                  break;
                      }
                      if (isvalid(tmpx,tmpy)) {
                         if (playlevel == CADET) { // computer finishes off current target, hit once, sink entire ship
                            if (who == COMPUTER) {
                                   if (friendly_grid[tmpx][tmpy] == prevhit.type) {
                                       x = tmpx;
                                       y = tmpy;
                                       done2 = true;
                                   }    
                            } else {
                                   if (enemy_grid[tmpx][tmpy] == prevhit.type) {
                                       x = tmpx;
                                       y = tmpy;
                                       done2 = true;
                                   }
                            }        
                         }                                    
                         if (playlevel > CADET && playlevel < CAPTAIN) { // normal level, AI path finding 
                             if (friendly_grid[tmpx][tmpy] == '+') { 
                                   done2 = true; // exit out of current direction if dead end
                             } else {
                                if (friendly_grid[tmpx][tmpy] != '@') {      // take a shot if valid
                                   x = tmpx;
                                   y = tmpy;
                                   done2 = true;
                                }   
                             }
                         }
                         if (playlevel == CAPTAIN) {  // hardest level, computer seeks targets and queues new paths
                            if (friendly_grid[tmpx][tmpy] >= '1' && friendly_grid[tmpx][tmpy] <= '5') {
                               x = tmpx;
                               y = tmpy;
                               done2 = true;
                            }                             
                         }                                                              
                      }
                }
                if (isvalid(x,y)) {
                    firecoord.x = x;
                    firecoord.y = y;                         
                    done = true;        
                }
            } else {                                                      // last resort check in case AI path
               firecoord = auto_fire(COMPUTER);                           // leads to dead end, if stuck then
               done = true;                                               // randomize next move to prevent
            }                                                             // endless loop   
        }
    }
    
    return firecoord;
}

void recon(Coord firecoord) {        /* recon mission */

    int cnt=0,a,b;
    int color = BLUE;

    if (sound_on) play_sample(ping_snd,volume,0,1000,0);
    
    a = firecoord.x*SQUARE_SIZE+MOUSE_TOLERANCE;
    b = firecoord.y*SQUARE_SIZE+MOUSE_TOLERANCE;                                  
    rect(screen,a,b,a+SQUARE_SIZE,b+SQUARE_SIZE,YELLOW);
    
    for (int x = firecoord.x-1; x <= firecoord.x+1; x++)
        for (int y = firecoord.y-1; y <= firecoord.y+1; y++) 
            if (isvalid(x,y))
               if (enemy_grid[x][y] >= '1' && enemy_grid[x][y] <= '5') cnt++;
    
    if (cnt > 0) color = GREEN;
    
    rest(THIRD_SEC);
    
    for (int x = firecoord.x-1; x <= firecoord.x+1; x++)
       for (int y = firecoord.y-1; y <= firecoord.y+1; y++)
           if (isvalid(x,y)) {
              a = x*SQUARE_SIZE+MOUSE_TOLERANCE;
              b = y*SQUARE_SIZE+MOUSE_TOLERANCE;                                  
              rect(screen,a,b,a+SQUARE_SIZE,b+SQUARE_SIZE,color);
           }
    
    recons++;
   
}

Coord calc_mousepos(int who, bool *valid) {  /* Determines grid coordinates from mouse position */
     
     BoxCoord currgrid;
     Coord firecoord = {-1,-1}; 

     switch (who) {
            case HUMAN:
                 currgrid = friendlygrid;
                 break;
            case COMPUTER:
                 currgrid = enemygrid;
                 break;
     }
                 
     *valid = false;
     
     if ((mouse_x >= currgrid.top.x && mouse_x <= currgrid.bottom.x) &&
         (mouse_y >= currgrid.top.y && mouse_y <= currgrid.bottom.y)) {
              div_t resX = div(mouse_x,SQUARE_SIZE);      
              div_t resY = div(mouse_y,SQUARE_SIZE);      
              if (resX.rem < MOUSE_TOLERANCE)                 
                 firecoord.x = resX.quot - 1;    
              else
                 firecoord.x = resX.quot;   
              if (resY.rem < MOUSE_TOLERANCE)                 
                 firecoord.y = resY.quot - 1;
              else
                 firecoord.y = resY.quot;
              *valid = true;   
     }
    
     return firecoord;
     
}

void manual_load() {
     
     while (!manual_list.empty())
           manual_list.pop();
     draw_grid(HUMAN);
     loadship_click = 0;

}


void manual_load(Coord firecoord) {
     
     int a,b,x,y;
     char ship;

     x = firecoord.x;
     y = firecoord.y - 12;
     
     if (!isvalid(x,y)) return;
     if (friendly_grid[x][y] >= '1' && friendly_grid[x][y] <= '5') return;

     switch (loadship) {
            case CARRIER:
                 ship = '5';
                 break;
            case BATTLESHIP:
                 ship = '4';
                 break;
            case DESTROYER:
                 ship = '3';
                 break;
            case CORVETTE:
                 ship = '2';
                 break;
            case GUNBOAT1:
                 ship = '1';
                 break;
     }
                 
     Hit newcoord = {x,y,ship};
     Hit topcoord;
          
     a = firecoord.x*SQUARE_SIZE+10;
     b = firecoord.y*SQUARE_SIZE+5;
          
     bool ok = true;
     bool redraw = false;
     
     if (!manual_list.empty()) {
        topcoord = manual_list.top();
        if (!(topcoord.x == x || topcoord.y == y))
           ok = false;
        if ((topcoord.x == x) && (topcoord.y + 1 != y))
           ok = false;
        if ((topcoord.y == y) && (topcoord.x + 1 != x))
           ok = false;
     }
     
     if (ok) {
       rect(screen,a,b,a+SQUARE_SIZE,b+SQUARE_SIZE,YELLOW);
       manual_list.push(newcoord);
       loadship_click++;
       
       int checkship = loadship;
       if (checkship == GUNBOAT1) checkship++;
       
       if (loadship_click == checkship) {
         
         while (!manual_list.empty()) {
           topcoord = manual_list.top();
           x = topcoord.x;
           y = topcoord.y;
           ship = topcoord.type;
           if (friendly_grid[x][y] >= '1' && friendly_grid[x][y] <= '5') {
              ok = false;
              redraw = true;
           } else
               friendly_grid[x][y] = topcoord.type;
           manual_list.pop();    
         }
         
         if (ok) {
             manual_click = true;
             int z=0;
             int xscore = 600;
             int yscore = 650;
             switch (loadship) {
                    case CARRIER:
                         z = CARRIER;
                         show_shipdesc(BATTLESHIP);
                         break;
                    case BATTLESHIP:
                         z = BATTLESHIP;
                         yscore += 20;
                         show_shipdesc(DESTROYER);
                         break;
                    case DESTROYER:
                         z = DESTROYER;
                         yscore += 40;
                         show_shipdesc(CORVETTE);
                         break;
                    case CORVETTE:
                         z = CORVETTE;
                         yscore += 60;
                         show_shipdesc(GUNBOAT1);
                         break;
                    case GUNBOAT1:
                         z = GUNBOAT;
                         yscore += 80;
                         break;
             }
         
             for (int i=0;i<z;i++)
                 draw_bitmap(friendly,xscore+(i*20),yscore);
             
             if (sound_on) play_sample(done_snd,volume,0,1000,0);

             load_gameboard(HUMAN);     
             loadship_click = 0;
             loadship--;

             if (loadship == EMPTY) {
                redraw = true;
                load_data = true;
                show_ready();
                if (sound_on) play_sample(announce_snd,volume,0,1000,0);
             }
             
             if (redraw) draw_grid(HUMAN);
                                
           } 
        }
     }
     
}

void draw_scores(int who) {                 /* Displays scoreboard */

     int z;
     int x = 400;
     int x2 = 600;
     int y = 650;

     for (int j = 5;j > 0; j--) {
         z = j;
         if (j == 1) z = 2; 
         for (int i=0;i<z;i++) {
             switch (who) {
                    case COMPUTER:
                         draw_bitmap(enemy,x+(i*20),y);
                         break;
                    case HUMAN:
                         draw_bitmap(friendly,x2+(i*20),y);
                         break;
             }
         }
         y += 20;
     }    

    if (who == COMPUTER) 
        show_shipdesc(CARRIER);
    if (who == HUMAN) 
        for (int ship=BATTLESHIP;ship>EMPTY;ship--)
            show_shipdesc(ship);
        
}

void show_shipdesc(int ship) {
     
     switch (ship) {
            case CARRIER:
                 textout_ex(screen,font,"Carrier",515,660,YELLOW,-1);
                 break;
            case BATTLESHIP:
                 textout_ex(screen,font,"Battleship",515,680,YELLOW,-1);
                 break;
            case DESTROYER:     
                 textout_ex(screen,font,"Destroyer",515,700,YELLOW,-1);
                 break;
            case CORVETTE:
                 textout_ex(screen,font,"Corvette",515,720,YELLOW,-1);
                 break;
            case GUNBOAT1:
                 textout_ex(screen,font,"Gunboat",515,740,YELLOW,-1);
                 break;
     }

}
     
void draw_score() {                 /* Displays target hits on scoreboard */

     int erasehitlist = -1;
     
     int y = 650;
     int x = 400;
     int x2 = 600;
     int salvo = 0;
     
     int j=0;
     int who = -1;
     
     if (e5 == CARRIER || f5 == CARRIER) {
        if (e5 == CARRIER) {
           e5++;
           who = COMPUTER;
           escore--;
           if (playlevel == CADET) erasehitlist = COMPUTER;
        }
        if (f5 == CARRIER) {
           f5++;
           who = HUMAN;
           fscore--;
           salvo = CARRIER;
           erasehitlist = HUMAN;
        }
        j = CARRIER;
     }

     if (e4 == BATTLESHIP || f4 == BATTLESHIP) {
        if (e4 == BATTLESHIP) {
           e4++;
           escore--;
           who = COMPUTER;
           if (playlevel == CADET) erasehitlist = COMPUTER;           
        }   
        if (f4 == BATTLESHIP) {
           f4++;
           fscore--;
           salvo = BATTLESHIP;
           who = HUMAN;
           erasehitlist = HUMAN;
        }
        j = BATTLESHIP;
        y += 20;
     }

     if (e3 == DESTROYER || f3 == DESTROYER) {
        if (e3 == DESTROYER) {
           e3++;
           escore--;
           who = COMPUTER;
           if (playlevel == CADET) erasehitlist = COMPUTER;
        }
        if (f3 == DESTROYER) {
           f3++;
           fscore--;
           salvo = DESTROYER;
           who = HUMAN;
           erasehitlist = HUMAN;
        }
        j = DESTROYER;
        y += 40;
     }               

     if (e2 == CORVETTE || f2 == CORVETTE) {
        if (e2 == CORVETTE) {
           e2++;
           escore--;
           who = COMPUTER;
           if (playlevel == CADET) erasehitlist = COMPUTER;
        }
        if (f2 == CORVETTE) {
           f2++;
           fscore--;
           who = HUMAN;
           erasehitlist = HUMAN;
        }
        j = CORVETTE;
        y += 60;
     }               

     if (e1 == GUNBOAT || f1 == GUNBOAT) {
        if (e1 == GUNBOAT) {
           e1++;
           escore--;
           who = COMPUTER;
           if (playlevel == CADET) erasehitlist = COMPUTER;
        }
        if (f1 == GUNBOAT) {
           f1++;
           fscore--;
           who = HUMAN;
           erasehitlist = HUMAN;
        }
        j = GUNBOAT;
        y += 80;
     }
     
     for (int i=0;i<j;i++) {
        switch (who) {
           case COMPUTER: 
                draw_bitmap(explosion,x+(i*20),y);
                break;
           case HUMAN: 
                draw_bitmap(explosion,x2+(i*20),y);
                break;
        }
     }
     
     if (playlevel >= COMMANDER) {
         switch (salvo) {
                case CARRIER:
                     masked_blit(noprobe,screen,0,0,820,380,BMP_SIZE,BMP_SIZE);
                     masked_blit(noprobe,screen,0,0,890,380,BMP_SIZE,BMP_SIZE);
                     break;
                case BATTLESHIP:
                     masked_blit(noprobe,screen,0,0,820,480,BMP_SIZE,BMP_SIZE);    
                     masked_blit(noprobe,screen,0,0,890,480,BMP_SIZE,BMP_SIZE);    
                     break;
                case DESTROYER:
                     masked_blit(noprobe,screen,0,0,820,580,BMP_SIZE,BMP_SIZE);    
                     masked_blit(noprobe,screen,0,0,890,580,BMP_SIZE,BMP_SIZE);    
                     break;
         }
     }
         
     switch (erasehitlist) {
            case HUMAN:
                 hitlist.pop(); // done with current path, remove from stack
                 rest(TENTH_SEC);
                 break;
            case COMPUTER:
                 humanlist.pop();
                 rest(TENTH_SEC);
                 break;
     }    

}

void draw_labels() {                   /* Display screen labels */

    masked_blit(enemy_label,screen,0,0,400,630,100,25);
    masked_blit(friendly_label,screen,0,0,600,630,100,25);
    destroy_bitmap(enemy_label);
    destroy_bitmap(friendly_label);  
  
}

void load_gameboard(int who) {          /* Loads target ints from data file */

     switch (who) {
            case HUMAN:
                 for (int x=0;x<MAX_X;x++)          
                     for (int y=0;y<MAX_Y;y++) 
                         if (friendly_grid[x][y] >= '1' && friendly_grid[x][y] <= '5')
                            draw_bitmap(friendly,friendlycoord.x,friendlycoord.y,x,y);
                 break;
            case COMPUTER:
                 /* Debug only */
                 if (debug) {
                     for (int x=0;x<MAX_X;x++)          
                         for (int y=0;y<MAX_Y;y++) 
                             if (enemy_grid[x][y] >= '1' && enemy_grid[x][y] <= '5')
                                draw_bitmap(enemy,enemycoord.x,enemycoord.y,x,y);
                 }
                 /* */
                 break;
     }

}

void init_grid() {                         /* Initialize player game board */

     for (int x=0;x<MAX_X;x++)          
         for (int y=0;y<MAX_Y;y++) 
             friendly_grid[x][y] = '-';     
}

void load_bmp() {                          /* Loads bitmaps from files */

     int theme;
     switch (playlevel) {
            case CADET:
               theme = DATA_BKG1;
               break;
            case ENSIGN:
               theme = DATA_BKG2;
               break;
            case LIEUTENANT:
               theme = DATA_BKG3;
               break;
            case COMMANDER:
               theme = DATA_BKG4;
               break;
            case CAPTAIN:
               theme = DATA_BKG5;
               break;
            default:
               theme = DATA_BKG3;
               break;   
     }      
        
     bkg = (BITMAP *)data[theme].dat;
     enemy = (BITMAP *)data[DATA_ENEMY].dat;
     friendly = (BITMAP *)data[DATA_FRIEND].dat;
     explosion = (BITMAP *)data[DATA_EXPLOSION].dat;
     enemy_label = (BITMAP *)data[DATA_ENEMY_LABEL].dat;
     friendly_label = (BITMAP *)data[DATA_FRIENDLY_LABEL].dat;
     missed = (BITMAP *)data[DATA_MISSED].dat;
     probe = (BITMAP *)data[DATA_SALVO].dat;
     noprobe = (BITMAP *)data[DATA_X].dat;
     gameover = (BITMAP *)data[DATA_GAMEOVER].dat;
           
}

void load_wav() {                      /* Loads sound waves from files */

    if (install_sound(DIGI_AUTODETECT,MIDI_AUTODETECT,"") != 0) {
       allegro_message("Sound set-up error. Program aborted.");
       deinit(-1);
    }
    
    explode_snd = (SAMPLE *)data[DATA_EXPLODE].dat;
    ping_snd = (SAMPLE *)data[DATA_PING].dat;
    announce_snd = (SAMPLE *)data[DATA_GENQUTR].dat;
    done_snd = (SAMPLE *)data[DATA_DONE].dat;
    blip_snd = (SAMPLE *)data[DATA_BLIP].dat;
    intro_snd = (SAMPLE *)data[DATA_ALLHANDS].dat;
    
}

void get_playlevel() {

     load_intro();
             
     textout_ex(screen,font,"PLAY LEVEL: ",10,140,GREEN,-1);
     textout_ex(screen,font,"(1) CADET",20,150,GREEN,-1);
     textout_ex(screen,font,"(2) ENSIGN",20,160,GREEN,-1);
     textout_ex(screen,font,"(3) LIEUTENANT",20,170,GREEN,-1);
     textout_ex(screen,font,"(4) COMMANDER",20,180,GREEN,-1);
     textout_ex(screen,font,"(5) CAPTAIN",20,190,GREEN,-1);
     
     textprintf_ex(screen, font, 10, 210, GREEN,-1, "Previous selection:  %i",playlevel);
     textout_ex(screen,font,"Make new selection or press <Enter>: ",10,230,GREEN,-1);

     clear_keybuf();
     
     readkey();
     if (key[KEY_1]) playlevel = CADET;
     if (key[KEY_2]) playlevel = ENSIGN;         
     if (key[KEY_3]) playlevel = LIEUTENANT;
     if (key[KEY_4]) playlevel = COMMANDER;
     if (key[KEY_5]) playlevel = CAPTAIN;
     
     if (!(playlevel >= CADET && playlevel <= CAPTAIN))
        playlevel = LIEUTENANT;
        
     clear_keybuf();
     
}
         
void load_intro() {                    /* Splash screen at start of program */

     char linestr[255];
     
     if (!exists(CFG_FILE)) {
        allegro_message("Missing configuration files.");
        deinit(-1);
     }

     fstream headerin(CFG_FILE);
     
     int i = 0;
     while (!headerin.eof()) {
           headerin.getline(linestr,255);
           i += 10;
  	       textprintf_ex(screen, font, 10, i, GREEN,-1, "%s",linestr);
     }  

     headerin.close();

}

int load_datafile(int who,int *skip) {  /* Loads ship formations from file */

     int reverse = 0;
     char linestr[15], linestr2[MAX_X][MAX_Y];
     int y = -1;
     int ret;

      if (!exists(FORMATION_FILE)) {
        allegro_message("Missing configuration files.");
        deinit(-1);
     }
     
     int i = *skip;
     while (i == *skip) 
         i = rndgen(480);            // Randomize formation selection
     
     ret = i;
     
     if (i > 159 && i <= 319 ) {  // Tripling the # of formations
           i -= 160;                                                // by flipping formation horizontally or vertically
           reverse = 1;
           y = 10;
     }
     if (i > 319 && i <= 479) {
           i -= 320;
           reverse = 2;
     }
     
     int miny = i * MAX_Y; 
     int maxy = miny + 9;
     
     fstream in(FORMATION_FILE);
     
     int line = -1;
     int z=0;
          
     while(in) {
       line++;
       in.getline(linestr,15);
       if (line >= miny && line <= maxy) {
          if (reverse == 1)
              y--;      
          else
              y++;
          for (int x=0;x<MAX_X;x++) {
               if (reverse == 2)
                  z = (13-x);
               else
                  z = x;   
               linestr2[x][y] = linestr[z];
          }                                    
       }
     }
     
     for (int y=0;y<MAX_Y;y++)           
          for (int x=0;x<MAX_X;x++)
              switch (who) {
                 case COMPUTER:
                      enemy_grid[x][y] = linestr2[x][y];
                      break;
                 case HUMAN:
                      friendly_grid[x][y] = linestr2[x][y];
                      break;
              }
     
     in.close(); 
     *skip = ret;

     return ret;
     
}

void draw_grid(int who) {           /* Draws gameboard grids */

     switch (who) {
        case COMPUTER:
             for (int i=0;i<=MAX_X;i++)
                 line(screen,enemycoord.x+(i*SQUARE_SIZE),enemycoord.y,enemycoord.x+(i*SQUARE_SIZE),enemycoord_bottom.y,RED);
             for (int i=0;i<=MAX_Y;i++)
                     line(screen,friendlycoord.x,enemycoord.y+(i*SQUARE_SIZE),enemycoord_bottom.x,enemycoord.y+(i*SQUARE_SIZE),RED);
             break;
        case HUMAN:
             for (int i=0;i<=MAX_X;i++) 
                 line(screen,enemycoord.x+(i*SQUARE_SIZE),friendlycoord.y,enemycoord.x+(i*SQUARE_SIZE),friendlycoord_bottom.y,BLUE);
             for (int i=0;i<=MAX_Y;i++) 
                 line(screen,friendlycoord.x,friendlycoord.y+(i*SQUARE_SIZE),friendlycoord_bottom.x,friendlycoord.y+(i*SQUARE_SIZE),BLUE);
             break;                                       
     }

}

void draw_bkg() {                                     /* Draws main background graphic */

     blit(bkg,screen,0,0,0,0,SCREEN_W,SCREEN_H);
     destroy_bitmap(bkg);

}

void draw_bitmap(BITMAP *bitmap, int x, int y) {      /* draw bitmaps */

     masked_blit(bitmap,screen,0,0,x,y,BMP_SIZE,BMP_SIZE);

}

void draw_bitmap(BITMAP *bitmap,int X, int Y, int x, int y) {

     masked_blit(bitmap,screen,0,0,X+(x*25),Y+(y*25),BMP_SIZE,BMP_SIZE);

}

void erase_probe() {            /* Disables recon mission once used */
 
    int x;
    
    for (int i=1;i<=recons;i++)
        x = 800+(i*20);
    
    masked_blit(noprobe,screen,0,0,x,50,BMP_SIZE,BMP_SIZE);
    
    if (recons >= max_recon)
       destroy_bitmap(noprobe);
     
}

void load_binaries() {         /* Loads bitmaps and wave files from binary data file */

    if (!exists(DATA_FILE)) {
       allegro_message("Missing data file.");
       deinit(-1);
    }
    
    data = load_datafile(DATA_FILE); 
    
    if (!data) {
       allegro_message("Errors occurred while loading data.");
       deinit(-1);
    }        
    
}

void load_formation() { /* Player chooses preloaded formation */
     
    init_grid();
    draw_grid(HUMAN);
    debug_human_data = load_datafile(HUMAN,&skip);
    load_gameboard(HUMAN);
    draw_scores(HUMAN);
    show_ready();
    if (sound_on) play_sample(announce_snd,volume,0,1000,0);
                    
}

void show_reconlegend() {      /* Display recon map legend */

    textout_ex(screen,font,"Available Recons:",810,40,GREEN,-1);
    textout_ex(screen,font,"Recon Map Legend:",810,85,YELLOW,-1);
    for (int i=0;i<3;i++)
        for (int j=0;j<3;j++)
            rect(screen,810+i*15,100+j*15,825+i*15,115+j*15,BLUE);
    textout_ex(screen,font,"Negative Contact",875,120,YELLOW,-1);
    for (int i=0;i<3;i++)
        for (int j=0;j<3;j++)
            rect(screen,810+i*15,155+j*15,825+i*15,170+j*15,GREEN);
    textout_ex(screen,font,"Positive Contact",875,175,YELLOW,-1);     

    for (int i=1;i<=max_recon;i++)
        masked_blit(probe,screen,0,0,800+(i*20),50,BMP_SIZE,BMP_SIZE);
        
    destroy_bitmap(probe);
        
}

void show_salvolegend() {      /* Display salvo type legend */

    textout_ex(screen,font,"Salvo Types:",810,300,GREEN,-1);
    textout_ex(screen,font,"Carrier",840,320,YELLOW,-1);

    int cnt=0;
    for (int i=0;i<3;i++) {
        for (int j=0;j<3;j++) {
            cnt++;
            switch (cnt) {
                   case 1:
                   case 3:
                   case 5:
                   case 7:
                   case 9:
                     rectfill(screen,814+i*15,344+j*15,821+i*15,351+j*15,WHITE);
            }         
            rect(screen,810+i*15,340+j*15,825+i*15,355+j*15,RED);
        }
    }  
    
    cnt=0;        
    for (int i=0;i<3;i++) {
        for (int j=0;j<3;j++) {
            cnt++;
            switch (cnt) {
                   case 2:
                   case 4:
                   case 5:
                   case 6:
                   case 8:
                     rectfill(screen,884+i*15,344+j*15,891+i*15,351+j*15,WHITE);
            }         
            rect(screen,880+i*15,340+j*15,895+i*15,355+j*15,RED);
        } 
    }
    textout_ex(screen,font,"F5",825,390,GREEN,-1);
    textout_ex(screen,font,"F6",895,390,GREEN,-1);
    
    textout_ex(screen,font,"Battleship",836,420,YELLOW,-1);
    
    cnt=0; 
    for (int i=0;i<3;i++) {
        for (int j=0;j<3;j++) {
            cnt++;
            switch (cnt) {
                   case 1:
                   case 3:
                   case 7:
                   case 9:
                     rectfill(screen,814+i*15,444+j*15,821+i*15,451+j*15,WHITE);
            }         
            rect(screen,810+i*15,440+j*15,825+i*15,455+j*15,RED);
        }
    }
    cnt=0;  
    for (int i=0;i<3;i++) {
        for (int j=0;j<3;j++) {
            cnt++;
            switch (cnt) {
                   case 2:
                   case 4:
                   case 6:
                   case 8:
                     rectfill(screen,884+i*15,444+j*15,891+i*15,451+j*15,WHITE);
            }         
            rect(screen,880+i*15,440+j*15,895+i*15,455+j*15,RED);
        } 
    }

    textout_ex(screen,font,"F7",825,490,GREEN,-1);
    textout_ex(screen,font,"F8",895,490,GREEN,-1);
    
    textout_ex(screen,font,"Destroyer",838,520,YELLOW,-1);

    cnt=0;
    for (int i=0;i<3;i++) {
        for (int j=0;j<3;j++) {
            cnt++;
            switch (cnt) {
                   case 2:
                   case 5:
                   case 8:
                     rectfill(screen,814+i*15,544+j*15,821+i*15,551+j*15,WHITE);
            }         
            rect(screen,810+i*15,540+j*15,825+i*15,555+j*15,RED);
        }
    }  
    cnt=0;        
    for (int i=0;i<3;i++) {
        for (int j=0;j<3;j++) {
            cnt++;
            switch (cnt) {
                   case 4:
                   case 5:
                   case 6:
                     rectfill(screen,884+i*15,544+j*15,891+i*15,551+j*15,WHITE);
            }         
            rect(screen,880+i*15,540+j*15,895+i*15,555+j*15,RED);
        } 
    }
    textout_ex(screen,font,"F9",825,590,GREEN,-1);
    textout_ex(screen,font,"F10",895,590,GREEN,-1);
    
    textout_ex(screen,font,"F4 - back to single shot",805,620,GREEN,-1);
    
}

void show_gridlegend() {     /* Display grid labels and help information */

    textout_ex(screen,font,"Foe",38,265,GREEN,-1);       
    textout_ex(screen,font,"Friend",38,295,GREEN,-1);
    textout_ex(screen,font,"Left Mouse  - Fire",20,620,GREEN,-1);
    textout_ex(screen,font,"Esc - Quit",250,610,GREEN,-1);
    textout_ex(screen,font,"F2  - Sound",250,620,GREEN,-1);

    if (playlevel > CADET && playlevel < COMMANDER)
       textout_ex(screen,font," Z - Salvo",250,630,GREEN,-1);     

    if (playlevel > ENSIGN)
       textout_ex(screen,font,"Right Mouse - Recon",20,630,GREEN,-1);

    switch (playlevel) {
           case CADET:
                textout_ex(screen,font,"Play Level  - Cadet",20,610,GREEN,-1);
                break;
           case ENSIGN:
                textout_ex(screen,font,"Play Level  - Ensign",20,610,GREEN,-1);
                break;
           case LIEUTENANT:
                textout_ex(screen,font,"Play Level  - Lieutenant",20,610,GREEN,-1);
                break;
           case COMMANDER:
                textout_ex(screen,font,"Play Level  - Commander",20,610,GREEN,-1);
                break;
           case CAPTAIN:
                textout_ex(screen,font,"Play Level  - Captain",20,610,GREEN,-1);
                break;
    }
}

void show_tip() {

    textout_ex(screen, font, "Locate enemy ships on Red Grid", 10, 700, GREEN,-1);
    textout_ex(screen, font, "Mark your ship positions on Blue Grid:", 10, 710, GREEN,-1);
    textout_ex(screen, font, "  Left Mouse, mark left-right, top-bottom", 10, 720, GREEN,-1);
    textout_ex(screen, font, "  Right Mouse, clear markings", 10, 730, GREEN,-1);
    textout_ex(screen, font, "F1 - Load Ships",250,750,GREEN,-1);
    
}

void show_ready() {   /* Display ready prompt */
     
   	if (debug)
 	   textprintf_ex(screen, font, 225, 560, GREEN,-1, "Debug C=%i H=%i",debug_computer_data, debug_human_data);
    else
 	   textout_ex(screen, font, "Ready", 320, 560, GREEN,-1);
    
}

bool isvalid(int x, int y) {        /* Validates X,Y coordinates */                        
     
    if ((x >= 0 && x < MAX_X) && (y >= 0 && y < MAX_Y))
       return true;
    else
       return false;

}

int rndgen(int rndmax) {                   /* Random number generator */
    
    if (rndmax == 0) {
        srand(time(NULL));                 // randomizer seed
        return(0);
    } else
      return rand() % rndmax;

}
