#include <stdlib.h>
#include "render.h"
#include "math.h"
#include "world.h"
#include "video.h"
#include "clipping.h"

#define FRINGE_NORTH        1
#define FRINGE_EAST         2
#define FRINGE_SOUTH        4
#define FRINGE_WEST         8
#define FRINGE_NORTH_WEST  16
#define FRINGE_NORTH_EAST  32
#define FRINGE_SOUTH_EAST  64
#define FRINGE_SOUTH_WEST 128

int fringeConversionTable[3][3] = {
   {FRINGE_NORTH_WEST, FRINGE_NORTH, FRINGE_NORTH_EAST},
   {FRINGE_WEST,       0,            FRINGE_EAST},
   {FRINGE_SOUTH_WEST, FRINGE_SOUTH, FRINGE_SOUTH_EAST}
};

typedef struct RENDER_SQUARE {
   int squareX, squareY;
   float dist;
} RENDER_SQUARE;

int textureMode = POLYTYPE_PTEX_LIT;
int squareMode = SQUARE_TWO;

int polysDrawn = 0;
int squaresDrawn = 0;

int fogRange = 16;

bool sortOn = true;

bool dynamicLightingOn = true;

bool backfaceCullingOn = true;

bool edgeDraw = false;
bool vertexDraw = false;
bool textureDraw = true;
bool fringeDraw = true;

int zCompare(const void *e1, const void *e2)
{
   if (((RENDER_SQUARE *)e1)->dist > ((RENDER_SQUARE *)e2)->dist) {
      return -1;
   }
   else {
      return 1;
   }
}

void renderSquare(BITMAP *destination, World *world, Camera *camera, TextureSet *texture, int x, int y)
{
   V3D_f worldVertices[5];
   V3D_f tempVertices[10];
   V3D_f *tempPoly[6];
   const V3D_f *tri[4][3]; // This for GCC
//   V3D_f *tri[4][3]; // This for MSVC
   V3D_f *ntri[4][6];
   V3D_f ntriv[4][6];
   int vc[4];
   int out[10];
   int triCount = 0;
   int i, j;
   BITMAP *temp = NULL;
   char surr = 0;
   char sides, corners;
   float edgeChange = 0.0; // 0.5;

   // Set up the vetecies
   worldVertices[0].x = world->getPoint(x + 0, y + 0)->x;
   worldVertices[0].y = world->getPoint(x + 0, y + 0)->y;
   worldVertices[0].z = world->getPoint(x + 0, y + 0)->z;
   worldVertices[0].u = 0 - edgeChange;
   worldVertices[0].v = 0 - edgeChange;
   worldVertices[0].c = world->getPoint(x + 0, y + 0)->getLight();

   worldVertices[1].x = world->getPoint(x + 1, y + 0)->x;
   worldVertices[1].y = world->getPoint(x + 1, y + 0)->y;
   worldVertices[1].z = world->getPoint(x + 1, y + 0)->z;
   worldVertices[1].u = 31 + edgeChange;
   worldVertices[1].v = 0 - edgeChange;
   worldVertices[1].c = world->getPoint(x + 1, y + 0)->getLight();

   worldVertices[2].x = world->getPoint(x + 1, y + 1)->x;
   worldVertices[2].y = world->getPoint(x + 1, y + 1)->y;
   worldVertices[2].z = world->getPoint(x + 1, y + 1)->z;
   worldVertices[2].u = 31 + edgeChange;
   worldVertices[2].v = 31 + edgeChange;
   worldVertices[2].c = world->getPoint(x + 1, y + 1)->getLight();

   worldVertices[3].x = world->getPoint(x + 0, y + 1)->x;
   worldVertices[3].y = world->getPoint(x + 0, y + 1)->y;
   worldVertices[3].z = world->getPoint(x + 0, y + 1)->z;
   worldVertices[3].u = 0 - edgeChange;
   worldVertices[3].v = 31 + edgeChange;
   worldVertices[3].c = world->getPoint(x + 0, y + 1)->getLight();

   for (i = 0; i < 5; i++) {
      apply_matrix_f(&camera->viewMatrix, worldVertices[i].x, worldVertices[i].y, worldVertices[i].z, &worldVertices[i].x, &worldVertices[i].y, &worldVertices[i].z);
   }
   for (i = 0; i < 10; i++) {
      tempPoly[i] = &tempVertices[i];
   }

   switch (squareMode) {
   case SQUARE_TWO: // Draw square with two polys, faster
      triCount = 2;
      tri[0][0] = &worldVertices[0];
      tri[0][1] = &worldVertices[1];
      tri[0][2] = &worldVertices[2];
      tri[1][0] = &worldVertices[0];
      tri[1][1] = &worldVertices[2];
      tri[1][2] = &worldVertices[3];
      break;
   case SQUARE_FOUR: // Draw square with four polys, smoother
      worldVertices[4].x = (worldVertices[0].x + worldVertices[1].x + worldVertices[2].x + worldVertices[3].x) / 4;
      worldVertices[4].y = (worldVertices[0].y + worldVertices[1].y + worldVertices[2].y + worldVertices[3].y) / 4;
      worldVertices[4].z = (worldVertices[0].z + worldVertices[1].z + worldVertices[2].z + worldVertices[3].z) / 4;
      worldVertices[4].u = 15.5;
      worldVertices[4].v = 15.5;
      worldVertices[4].c = (worldVertices[0].c + worldVertices[1].c + worldVertices[2].c + worldVertices[3].c) / 4;

      triCount = 4;
      tri[0][0] = &worldVertices[0];
      tri[0][1] = &worldVertices[1];
      tri[0][2] = &worldVertices[4];
      tri[1][0] = &worldVertices[1];
      tri[1][1] = &worldVertices[2];
      tri[1][2] = &worldVertices[4];
      tri[2][0] = &worldVertices[2];
      tri[2][1] = &worldVertices[3];
      tri[2][2] = &worldVertices[4];
      tri[3][0] = &worldVertices[3];
      tri[3][1] = &worldVertices[0];
      tri[3][2] = &worldVertices[4];
      break;
   }

   if (textureMode == POLYTYPE_GRGB) {
      int colour = texture->getColour(world->getSquare(x, y)->texture);
      int red   = getr_depth(videoDepth, colour);
      int green = getg_depth(videoDepth, colour);
      int blue  = getb_depth(videoDepth, colour);
      worldVertices[0].c = makecol24(red * worldVertices[0].c / 255, green * worldVertices[0].c / 255, blue * worldVertices[0].c / 255);
      worldVertices[1].c = makecol24(red * worldVertices[1].c / 255, green * worldVertices[1].c / 255, blue * worldVertices[1].c / 255);
      worldVertices[2].c = makecol24(red * worldVertices[2].c / 255, green * worldVertices[2].c / 255, blue * worldVertices[2].c / 255);
      worldVertices[3].c = makecol24(red * worldVertices[3].c / 255, green * worldVertices[3].c / 255, blue * worldVertices[3].c / 255);
      worldVertices[4].c = makecol24(red * worldVertices[4].c / 255, green * worldVertices[4].c / 255, blue * worldVertices[4].c / 255);
   }
   else if (textureMode == POLYTYPE_FLAT) {
      int colour = texture->getColour(world->getSquare(x, y)->texture);
      worldVertices[0].c = colour;
      worldVertices[1].c = colour;
      worldVertices[2].c = colour;
      worldVertices[3].c = colour;
      worldVertices[4].c = colour;
   }

   for (j = 0; j < triCount; j++) {
      for (i = 0; i < 6; i++) {
         ntri[j][i] = &ntriv[j][i];
      }
   }
   for (j = 0; j < triCount; j++) {
      vc[j] = my_clip3d_f(textureMode, 0.0, fogRange * SQUARE_WIDTH * 1.0 / camera->zoom, 3, tri[j], ntri[j], tempPoly, out);
   }
   for (j = 0; j < triCount; j++) {
      for (i = 0; i < vc[j]; i++) {
         persp_project_f(ntriv[j][i].x, ntriv[j][i].y, ntriv[j][i].z, &ntriv[j][i].x, &ntriv[j][i].y);
      }
   }
   int t = ((x * y * world->getSquare(x, y)->texture) % 9 + (x + y) / 2) % 2;
   // Contruct texture
   if (textureDraw && world->getSquare(x, y)->fringe) {
      temp = create_bitmap(32, 32);
      if (t == 0) {
         blit(texture->getTexture(world->getSquare(x, y)->texture * 32 + 0), temp, 0, 0,  0,  0, 32, 32);
      }
      else {
         blit(texture->getTexture(world->getSquare(x, y)->texture * 32 + 16), temp, 0, 0,  0,  0, 32, 32);
      }
      if (fringeDraw) {
         for (int i = world->getSquare(x, y)->texture + 1; i < 8; i++) {
            surr = 0;
            for (int _x = 0; _x < 3; _x++) {
               for (int _y = 0; _y < 3; _y++) {
                  if (x + _x - 1 >= 0 && x + _x - 1 < world->getW() &&
                      y + _y - 1 >= 0 && y + _y - 1 < world->getH()) {
                     if (world->getSquare(x + _x - 1, y + _y - 1)->texture == i) {
                        surr += fringeConversionTable[_y][_x];
                     }
                  }
               }
            }
            sides   = (surr >> 0) & 15;
            corners = (surr >> 4) & 15;
            if (sides > 0) masked_blit(texture->getTexture(i * 32 +  0 + sides), temp, 0, 0, 0, 0, 32, 32);
            if (corners > 0) masked_blit(texture->getTexture(i * 32 + 16 + corners), temp, 0, 0, 0, 0, 32, 32);
         }
      }
   }
   for (j = 0; j < triCount; j++) {
      if (vc[j] > 0 && (polygon_z_normal_f(&ntriv[j][0], &ntriv[j][1], &ntriv[j][2]) >= 0 || !backfaceCullingOn)) {
         set_trans_blender(FADE_COLOR_RED, FADE_COLOR_GREEN, FADE_COLOR_BLUE, 127);
         if (textureDraw) {
            if (world->getSquare(x, y)->fringe) {
               polygon3d_f(destination, textureMode, temp, vc[j], ntri[j]);
            }
            else {
               if (t == 0) {
                  polygon3d_f(destination, textureMode, texture->getTexture(world->getSquare(x, y)->texture * 32 + 0), vc[j], ntri[j]);
               }
               else {
                  polygon3d_f(destination, textureMode, texture->getTexture(world->getSquare(x, y)->texture * 32 + 16), vc[j], ntri[j]);
               }
            }
         }
         set_add_blender(255, 255, 255, 255);
         drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
         if (edgeDraw) {
            for (i = 0; i < vc[j] - 1; i++) {
               line(destination, (int)ntri[j][i]->x, (int)ntri[j][i]->y, (int)ntri[j][i + 1]->x, (int)ntri[j][i + 1]->y, makecol_depth(videoDepth, 31, 31, 31));
            }
            line(destination, (int)ntri[j][vc[j] - 1]->x, (int)ntri[j][vc[j] - 1]->y, (int)ntri[j][0]->x, (int)ntri[j][0]->y, makecol_depth(videoDepth, 31, 31, 31));
         }
         if (vertexDraw) {
            for (i = 0; i < vc[j] - 1; i++) {
               putpixel(destination, (int)ntri[j][i]->x, (int)ntri[j][i]->y, makecol_depth(videoDepth, 31, 31, 31));
            }
         }
         polysDrawn++;
      }
   }
   // Deconstruct texture
   if (textureDraw && world->getSquare(x, y)->fringe) {
      destroy_bitmap(temp);
   }
}

void render(BITMAP *destination, World *world, Camera *camera, TextureSet *texture)
{
   int x, y;

   int cameraX, cameraY;
   int startX, startY, endX, endY;
   int range;

   cameraX = (int)(camera->position.x / SQUARE_WIDTH + 0.5);
   cameraY = (int)(camera->position.z / SQUARE_WIDTH + 0.5);
   range = fogRange + 2 + 1;
   startX = cameraX - range;
   startY = cameraY - range;
   endX   = cameraX + range;
   endY   = cameraY + range;
   if (startX < 0) startX = 0;
   if (startY < 0) startY = 0;
   if (endX > world->getW()) endX = world->getW();
   if (endY > world->getH()) endY = world->getH();

   set_trans_blender(FADE_COLOR_RED, FADE_COLOR_GREEN, FADE_COLOR_BLUE, 127);

   if (dynamicLightingOn) {
      for (x = startX; x <= endX; x++) {
         for (y = startY; y <= endY; y++) {
            world->getPoint(x, y)->dist = dist(camera->position.x, camera->position.y, camera->position.z, world->getPoint(x, y)->x, world->getPoint(x, y)->y, world->getPoint(x, y)->z);
            if (world->getPoint(x, y)->dist < fogRange * SQUARE_WIDTH) {
//               int l = (int)getLightLevel((fogRange / 2) * SQUARE_WIDTH, 255, fogRange * SQUARE_WIDTH, 0, world->getPoint(x, y)->dist);
               int l = (int)getLightLevel(0 * SQUARE_WIDTH, 255, fogRange * SQUARE_WIDTH, 0, world->getPoint(x, y)->dist);
               world->getPoint(x, y)->dynamicLight = l;
            }
            else {
               world->getPoint(x, y)->dynamicLight = 0.0;
            }
         }
      }
   }
   else {
      for (x = startX; x <= endX; x++) {
         for (y = startY; y <= endY; y++) {
            world->getPoint(x, y)->dist = dist(camera->position.x, camera->position.y, camera->position.z, world->getPoint(x, y)->x, world->getPoint(x, y)->y, world->getPoint(x, y)->z);
            world->getPoint(x, y)->dynamicLight = 255;
         }
      }
   }

   RENDER_SQUARE *zArray = new RENDER_SQUARE[((endX - 1) - startX) * ((endY - 1) - startY)];

   int fringeRange = fogRange / 4 * 3;

   int position = 0;
   for (x = startX; x <= endX - 1; x++) {
      for (y = startY; y <= endY - 1; y++) {
         float dist = (world->getPoint(x + 0, y + 0)->dist
                     + world->getPoint(x + 1, y + 0)->dist
                     + world->getPoint(x + 1, y + 1)->dist
                     + world->getPoint(x + 0, y + 1)->dist) / 4;
         if (dist < fogRange * SQUARE_WIDTH) {
            zArray[position].squareX = x;
            zArray[position].squareY = y;
            zArray[position].dist = dist;
            position++;
         }
         if (dist < fringeRange * SQUARE_WIDTH) {
            world->getSquare(x, y)->fringe = true;
         }
         else {
            world->getSquare(x, y)->fringe = false;
         }
      }
   }
   int amount = position;

   if (sortOn) {
      qsort(zArray, amount, sizeof(RENDER_SQUARE), zCompare);
   }

   for (int i = 0; i < amount; i++) {
      renderSquare(destination, world, camera, texture, zArray[i].squareX, zArray[i].squareY);
      squaresDrawn++;
   }

   delete [] zArray;
}
