// 3dland

#include <allegro.h>
#include <stdlib.h>
#include <math.h>
#include "world.h"
#include "math.h"
#include "texture.h"
#include "camera.h"
#include "render.h"
#include "object.h"
#include "video.h"
#include "font.h"

#define CAMERA_FREE 0
#define CAMERA_WALK 1
#define CAMERA_WAR  2

int standHeight = 32;

int cameraMode = CAMERA_WALK;

bool exitFlag = false;

int mapMode = 0;

bool allwaysRun = false;
bool allwaysStrafe = true;

// Time
volatile int milliseconds = 0;
volatile int seconds = 0;
volatile int minutes = 0;
volatile int hours = 0;

// Frames per second
volatile int frame = 0;
volatile long int frameTotal = 0;
volatile int fps = 0;

// Cycles per second
volatile int cycle = 0;
volatile long int cycleTotal = 0;
volatile int cps = 0;

// Speed control
volatile int speedCounter = 0;

void fpsUpdate(void)
{
   fps = frame;
   frame = 0;
}
END_OF_FUNCTION(fpsUpdate);

void cpsUpdate(void)
{
   cps = cycle;
   cycle = 0;
}
END_OF_FUNCTION(cpsUpdate);

void timerUpdate(void)
{
   milliseconds++;
   if (milliseconds >= 1000) {
      fpsUpdate();
      cpsUpdate();
      seconds++;
      milliseconds = 0;
   }
   if (seconds >= 60) {
      minutes++;
      seconds = 0;
   }
   if (minutes >= 60) {
      hours++;
      minutes = 0;
   }
}
END_OF_FUNCTION(timerUpdate);

void incrementSpeedCounter(void)
{
   speedCounter++;
}
END_OF_FUNCTION(incrementSpeedCounter);

void colouredRect(BITMAP *destination, int x1, int y1, int x2, int y2, int c1, int c2, int c3, int c4)
{
   int x, y;
   int w, h;
   int r1, g1, b1, a1,
       r2, g2, b2, a2,
       r3, g3, b3, a3,
       r4, g4, b4, a4;
   int rLeft, gLeft, bLeft, aLeft,
       rRight, gRight, bRight, aRight,
       rPos, gPos, bPos, aPos;
   int inDepth, outDepth;

   outDepth = bitmap_color_depth(destination);
   inDepth = 32;

   r1 = getr_depth(inDepth, c1);
   g1 = getg_depth(inDepth, c1);
   b1 = getb_depth(inDepth, c1);
   a1 = geta_depth(inDepth, c1);
   r2 = getr_depth(inDepth, c2);
   g2 = getg_depth(inDepth, c2);
   b2 = getb_depth(inDepth, c2);
   a2 = geta_depth(inDepth, c2);
   r3 = getr_depth(inDepth, c3);
   g3 = getg_depth(inDepth, c3);
   b3 = getb_depth(inDepth, c3);
   a3 = geta_depth(inDepth, c3);
   r4 = getr_depth(inDepth, c4);
   g4 = getg_depth(inDepth, c4);
   b4 = getb_depth(inDepth, c4);
   a4 = geta_depth(inDepth, c4);

   w = x2 - x1 + 1;
   h = y2 - y1 + 1;

   for (y = 0; y < h; y++) {
      rLeft = (r4 - r1) * y / h + r1;
      gLeft = (g4 - g1) * y / h + g1;
      bLeft = (b4 - b1) * y / h + b1;
      aLeft = (a4 - a1) * y / h + a1;
      rRight = (r3 - r2) * y / h + r2;
      gRight = (g3 - g2) * y / h + g2;
      bRight = (b3 - b2) * y / h + b2;
      aRight = (a3 - a2) * y / h + a2;
      for (x = 0; x < w; x++) {
         rPos = (rRight - rLeft) * x / w + rLeft;
         gPos = (gRight - gLeft) * x / w + gLeft;
         bPos = (bRight - bLeft) * x / w + bLeft;
         aPos = (aRight - aLeft) * x / w + aLeft;
         set_trans_blender(255, 255, 255, aPos);
         putpixel(destination, x1 + x, y1 + y, makecol_depth(outDepth, rPos, gPos, bPos));
      }
   }
}

void drawMap(BITMAP *destination, int _x, int _y, int w, int h, World *world, Camera *camera, TextureSet *texture)
{
   float squareW = (float)w / (float)(world->getW());
   float squareH = (float)h / (float)(world->getH());
   float x1, y1, x2, y2;
   int x, y;

   for (y = 0; y < world->getH(); y++) {
      for (x = 0; x < world->getW(); x++) {
         int colour = texture->getColour(world->getSquare(x, y)->texture);
         int red   = getr_depth(videoDepth, colour);
         int green = getg_depth(videoDepth, colour);
         int blue  = getb_depth(videoDepth, colour);
         int l1 = (int)world->getPoint(x + 0, y + 0)->staticLight;
         int l2 = (int)world->getPoint(x + 1, y + 0)->staticLight;
         int l3 = (int)world->getPoint(x + 1, y + 1)->staticLight;
         int l4 = (int)world->getPoint(x + 0, y + 1)->staticLight;
//         int c1 = makecol_depth(videoDepth, red * l1 / 255, green * l1 / 255, blue * l1 / 255);
//         int c2 = makecol_depth(videoDepth, red * l2 / 255, green * l2 / 255, blue * l2 / 255);
//         int c3 = makecol_depth(videoDepth, red * l3 / 255, green * l3 / 255, blue * l3 / 255);
//         int c4 = makecol_depth(videoDepth, red * l4 / 255, green * l4 / 255, blue * l4 / 255);
//         int c1 = makeacol_depth(32, red * l1 / 255, green * l1 / 255, blue * l1 / 255, l1);
//         int c2 = makeacol_depth(32, red * l2 / 255, green * l2 / 255, blue * l2 / 255, l2);
//         int c3 = makeacol_depth(32, red * l3 / 255, green * l3 / 255, blue * l3 / 255, l3);
//         int c4 = makeacol_depth(32, red * l4 / 255, green * l4 / 255, blue * l4 / 255, l4);
         int c1 = makeacol_depth(32, red, green, blue, l1);
         int c2 = makeacol_depth(32, red, green, blue, l2);
         int c3 = makeacol_depth(32, red, green, blue, l3);
         int c4 = makeacol_depth(32, red, green, blue, l4);
         x1 = (_x + (x + 0) * squareW + 0.5);
         y1 = (_y + (y + 0) * squareH + 0.5);
         x2 = (_x + (x + 1) * squareW - 0.5);
         y2 = (_y + (y + 1) * squareH - 0.5);
         if ((int)x1 < (int)(_x + (x + 0) * squareW)) {
            x1++;
         }
         if ((int)y1 < (int)(_y + (y + 0) * squareH)) {
            y1++;
         }
         if ((x2 - x1) > 0 && (y2 - y1) > 0) {
            colouredRect(destination, (int)x1, (int)y1, (int)x2, (int)y2, c1, c2, c3, c4);
         }
      }
   }
   int cameraX = (int)(squareW / SQUARE_WIDTH * camera->position.x + _x);
   int cameraY = (int)(squareH / SQUARE_DEPTH * camera->position.z + _y);
   int indicatorColour = makecol_depth(videoDepth, 255, 255, 255);
   circle(destination, cameraX, cameraY, (int)(squareW / 2) * 4, indicatorColour);
   int frontX = (int)(camera->front.x * (squareW / 2) * 8);
   int frontY = (int)(camera->front.z * (squareW / 2) * 8);
   line(destination, cameraX, cameraY, cameraX + frontX, cameraY + frontY, makecol_depth(videoDepth, 0, 0, 255));
   int upX = (int)(camera->up.x * (squareW / 2) * 8);
   int upY = (int)(camera->up.z * (squareW / 2) * 8);
   line(destination, cameraX, cameraY, cameraX + upX, cameraY + upY, makecol_depth(videoDepth, 0, 255, 0));
   int rightX = (int)(camera->right.x * (squareW / 2) * 8);
   int rightY = (int)(camera->right.z * (squareW / 2) * 8);
   line(destination, cameraX, cameraY, cameraX + rightX, cameraY + rightY, makecol_depth(videoDepth, 255, 0, 0));
}


int main(void)
{
   BITMAP *screenBuffer = NULL;
   BITMAP *hmap = NULL;
   BITMAP *tmap = NULL;
   TextureSet tex;
   int x, y;
   Object player;
   PALETTE palette;
   FONT *game_font = NULL;
   float headingChange;
   float pitchChange;
   float rollChange;

   // Initialise Allegro
   allegro_init();

   // Initialize timers
   install_timer();
   LOCK_VARIABLE(milliseconds);
   LOCK_VARIABLE(seconds);
   LOCK_VARIABLE(minutes);
   LOCK_VARIABLE(hours);
   LOCK_VARIABLE(frame);
   LOCK_VARIABLE(frameTotal);
   LOCK_VARIABLE(fps);
   LOCK_VARIABLE(cycle);
   LOCK_VARIABLE(cycleTotal);
   LOCK_VARIABLE(cps);
   LOCK_VARIABLE(speedCounter);
   LOCK_FUNCTION(timerUpdate);
   LOCK_FUNCTION(incrementSpeedCounter);
   install_int(timerUpdate, 1);
   install_int_ex(incrementSpeedCounter, BPS_TO_TIMER(30));

   // Initialise input
   install_keyboard();
   install_mouse();
   key_led_flag = FALSE;

   // Initialize video
   set_gfx_mode(GFX_SAFE, 320, 200, 0, 0);
   set_palette(default_palette);
   gui_fg_color = makecol_depth(8, 0, 0, 0);
   gui_bg_color = makecol_depth(8, 255, 255, 255);
   gfx_mode_select_ex(&videoCard, &videoWidth, &videoHeight, &videoDepth);
   set_color_depth(videoDepth);
   set_gfx_mode(videoCard, videoWidth, videoHeight, 0, 0);
   screenBuffer = create_bitmap(videoWidth, videoHeight);
   set_projection_viewport(0, 0, videoWidth, videoHeight);

//   set_color_conversion(COLORCONV_TOTAL);

   // Initialise font
   game_font = load_font("font.bmp");
   if (game_font == NULL) exit(69);
   font = game_font;

   // Load textures
   tex.open("textures.bmp");

   // Load maps
   set_color_depth(8);
   tmap = load_bitmap("tmap.pcx", NULL);
   hmap = load_bitmap("hmap.pcx", NULL);
   // Initialize world
   World world;
   world.setW(hmap->w - 1);
   world.setH(hmap->h - 1);
   world.createPoint();
   world.createSquare();
   // Read texture and height maps
   for (x = 0; x < world.getW(); x++) {
      for (y = 0; y < world.getH(); y++) {
         world.getSquare(x, y)->texture = getpixel(tmap, x, y);
      }
   }
   set_color_depth(videoDepth);
   for (x = 0; x < (world.getW() + 1); x++) {
      for (y = 0; y < (world.getH() + 1); y++) {
         world.getPoint(x, y)->x = x * SQUARE_WIDTH;
         world.getPoint(x, y)->y = getpixel(hmap, x, y) * SQUARE_HEIGHT;
         world.getPoint(x, y)->z = y * SQUARE_DEPTH;
      }
   }

   destroy_bitmap(tmap);
   destroy_bitmap(hmap);
   set_color_depth(videoDepth);

   // Initialize camera
   Camera camera;
   camera.position.x = world.getW() * SQUARE_WIDTH / 2;
   camera.position.y = 0.0;
   camera.position.z = world.getH() * SQUARE_DEPTH / 2;
   camera.velocity.x = 0.0;
   camera.velocity.y = 0.0;
   camera.velocity.z = 0.0;
   camera.front.x    = 0.0;
   camera.front.y    = 0.0;
   camera.front.z    = 1.0;
   camera.up.x       = 0.0;
   camera.up.y       = 1.0;
   camera.up.z       = 0.0;
   camera.right.x    = 1.0;
   camera.right.y    = 0.0;
   camera.right.z    = 0.0;
   camera.aspect     = 1.0;
   camera.fov        = 60.0;
   camera.zoom       = 1.0;

   // Setup light normal
   VECTOR lightNormal;
   lightNormal.x = -1.0;
   lightNormal.y = 1.0;
   lightNormal.z = -2.0;
   normalize_vector_f(&lightNormal.x, &lightNormal.y, &lightNormal.z);

   // Generate point normals
   for (x = 0; x < (world.getW() + 1); x++) {
      for (y = 0; y < (world.getH() + 1); y++) {
         VECTOR antiLight = makeVector(-lightNormal.x, -lightNormal.y, -lightNormal.z);
         VECTOR *up, *right, *down, *left;
         left = world.getPoint(x - 1, y);
         if (left == NULL) left = &antiLight;
         right = world.getPoint(x + 1, y);
         if (right == NULL) right = &antiLight;
         up = world.getPoint(x, y - 1);
         if (up == NULL) up = &antiLight;
         down = world.getPoint(x, y + 1);
         if (down == NULL) down = &antiLight;
         VECTOR n1, n2, n3, n4;
         n1 = planeNormal(world.getPoint(x, y), right, up);
         n2 = planeNormal(world.getPoint(x, y), down, right);
         n3 = planeNormal(world.getPoint(x, y), left, down);
         n4 = planeNormal(world.getPoint(x, y), up, left);
         world.getPoint(x, y)->normal.x = (n1.x + n2.x + n3.x + n4.x) / 4;
         world.getPoint(x, y)->normal.y = (n1.y + n2.y + n3.y + n4.y) / 4;
         world.getPoint(x, y)->normal.z = (n1.z + n2.z + n3.z + n4.z) / 4;
         normalize_vector_f(&world.getPoint(x, y)->normal.x, &world.getPoint(x, y)->normal.y, &world.getPoint(x, y)->normal.z);
      }
   }

   // Generate point static light
   for (x = 0; x < (world.getW() + 1); x++) {
      for (y = 0; y < (world.getH() + 1); y++) {
         if (x == 0 || x == world.getW() || y == 0 || y == world.getH()) {
            world.getPoint(x, y)->staticLight = 0;
         }
         else {
            float d = vectorDivergence(&world.getPoint(x, y)->normal, &lightNormal);
//            int l = (int)(255 * (d / 180));
            d = d / 180.0;
            if (d >= 0.0 && d < 0.5) {
               d = 0.125;
            }
            else if (d >= 0.5 && d < 0.75) {
               d = 0.125 + ((d - 0.5) / 0.25) * 0.875;
            }
            else if (d >= 0.75 && d <= 1.0) {
               d = 1.0;
            }
            int l = (int)(255 * d);
            world.getPoint(x, y)->staticLight = l;
         }
      }
   }

   // Main loop
   while (!exitFlag) {
      while (speedCounter > 0) {
         headingChange = 0.0;
         pitchChange = 0.0;
         rollChange = 0.0;
//         camera.velocity.x = 0.0;
//         camera.velocity.y = 0.0;
//         camera.velocity.z = 0.0;
         camera.velocity.x += -camera.velocity.x * (1.0 / 4.0);
         camera.velocity.y += -camera.velocity.y * (1.0 / 4.0);
         camera.velocity.z += -camera.velocity.z * (1.0 / 4.0);
         if (key[KEY_LEFT] || key[KEY_4_PAD]) {
            if ((!allwaysStrafe && (key_shifts & KB_ALT_FLAG)) || (allwaysStrafe && !(key_shifts & KB_ALT_FLAG))) {
               int amount = 4;
               if ((!allwaysRun && (key_shifts & KB_SHIFT_FLAG)) || (allwaysRun && !(key_shifts & KB_SHIFT_FLAG))) amount *= 2;
               camera.velocity.x += camera.right.x * amount;
               camera.velocity.y += camera.right.y * amount;
               camera.velocity.z += camera.right.z * amount;
            }
            else {
               headingChange -= 5.0;
            }
         }
         if (key[KEY_RIGHT] || key[KEY_6_PAD]) {
            if ((!allwaysStrafe && (key_shifts & KB_ALT_FLAG)) || (allwaysStrafe && !(key_shifts & KB_ALT_FLAG))) {
               int amount = 4;
               if ((!allwaysRun && (key_shifts & KB_SHIFT_FLAG)) || (allwaysRun && !(key_shifts & KB_SHIFT_FLAG))) amount *= 2;
               camera.velocity.x += -camera.right.x * amount;
               camera.velocity.y += -camera.right.y * amount;
               camera.velocity.z += -camera.right.z * amount;
            }
            else {
               headingChange += 5.0;
            }
         }
         if (key[KEY_PGUP] || key[KEY_9_PAD]) {
            pitchChange += 5.0;
         }
         if (key[KEY_PGDN] || key[KEY_3_PAD]) {
            pitchChange -= 5.0;
         }
         if (key[KEY_COMMA] || key[KEY_0_PAD]) {
            rollChange -= 5.0;
         }
         if (key[KEY_STOP] || key[KEY_DEL_PAD]) {
            rollChange += 5.0;
         }
         if (key[KEY_UP] || key[KEY_8_PAD]) {
            int amount = 4;
            if ((!allwaysRun && (key_shifts & KB_SHIFT_FLAG)) || (allwaysRun && !(key_shifts & KB_SHIFT_FLAG))) amount *= 2;
            camera.velocity.x += camera.front.x * amount;
            camera.velocity.y += camera.front.y * amount;
            camera.velocity.z += camera.front.z * amount;
         }
         if (key[KEY_DOWN] || key[KEY_2_PAD]) {
            int amount = 4;
            if ((!allwaysRun && (key_shifts & KB_SHIFT_FLAG)) || (allwaysRun && !(key_shifts & KB_SHIFT_FLAG))) amount *= 2;
            camera.velocity.x += -camera.front.x * amount;
            camera.velocity.y += -camera.front.y * amount;
            camera.velocity.z += -camera.front.z * amount;
         }
         if (key[KEY_A]) {
            if (key_shifts & KB_SHIFT_FLAG) {
               camera.aspect += -0.03125;
            }
            else {
               camera.aspect += 0.03125;
            }
         }
         if (key[KEY_F]) {
            if (key_shifts & KB_SHIFT_FLAG) {
               camera.fov += -1;
            }
            else {
               camera.fov += 1;
            }
         }
         if (key[KEY_R]) {
            if (key_shifts & KB_SHIFT_FLAG) {
               fogRange += -1;
            }
            else {
               fogRange += 1;
            }
         }
         if (key[KEY_H]) {
            if (key_shifts & KB_SHIFT_FLAG) {
               standHeight += -1;
            }
            else {
               standHeight += 1;
            }
         }
         if (key[KEY_Z]) {
            if (key_shifts & KB_SHIFT_FLAG) {
               camera.zoom += -0.03125;
            }
            else {
               camera.zoom += 0.03125;
            }
         }
         if (key[KEY_T]) {
            if (textureMode == POLYTYPE_ATEX) {
               textureMode = POLYTYPE_PTEX;
            }
            else if (textureMode == POLYTYPE_PTEX) {
               textureMode = POLYTYPE_ATEX_LIT;
            }
            else if (textureMode == POLYTYPE_ATEX_LIT) {
               textureMode = POLYTYPE_PTEX_LIT;
            }
            else if (textureMode == POLYTYPE_PTEX_LIT) {
               textureMode = POLYTYPE_GRGB;
            }
            else {
               textureMode = POLYTYPE_ATEX;
            }
         }
         if (key[KEY_C]) {
            if (cameraMode == CAMERA_FREE) {
               cameraMode = CAMERA_WALK;
            }
            else {
               cameraMode = CAMERA_FREE;
            }
         }
         if (key[KEY_S]) {
            if (squareMode == SQUARE_TWO) {
               squareMode = SQUARE_FOUR;
            }
            else {
               squareMode = SQUARE_TWO;
            }
         }
         if (key[KEY_TAB]) {
            if (mapMode == 0) {
               mapMode = 1;
            }
            else if (mapMode == 1) {
               mapMode = 2;
            }
            else {
               mapMode = 0;
            }
         }
         if (key[KEY_CAPSLOCK]) {
            allwaysRun = !allwaysRun;
         }
         if (key[KEY_NUMLOCK]) {
            allwaysStrafe = !allwaysStrafe;
         }
         if (key[KEY_E]) {
            edgeDraw = !edgeDraw;
         }
         if (key[KEY_V]) {
            vertexDraw = !vertexDraw;
         }
         if (key[KEY_U]) {
            textureDraw = !textureDraw;
         }
         if (key[KEY_O]) {
            sortOn = !sortOn;
         }
         if (key[KEY_M]) {
            cpu_mmx = !cpu_mmx;
         }
         if (key[KEY_D]) {
            dynamicLightingOn = !dynamicLightingOn;
         }
         if (key[KEY_B]) {
            backfaceCullingOn = !backfaceCullingOn;
         }
         if (key[KEY_I]) {
            fringeDraw = !fringeDraw;
         }
         if (key[KEY_ESC]) {
            exitFlag = true;
         }
         if (key[KEY_P]) {
            save_bitmap("screen.pcx", screenBuffer, palette);
         }
         if (mouse_b & 4) {
            exitFlag = true;
         }
         int mickeysX = 0, mickeysY = 0;
         get_mouse_mickeys(&mickeysX, &mickeysY);
         headingChange += mickeysX / 2;
         pitchChange += mickeysY / 2;

         // Update orientation vectors
         get_vector_rotation_matrix_f(&camera.headingMatrix, camera.up.x, camera.up.y, camera.up.z, B_DEG(headingChange));
         apply_matrix_f(&camera.headingMatrix, camera.right.x, camera.right.y, camera.right.z, &camera.right.x, &camera.right.y, &camera.right.z);
         apply_matrix_f(&camera.headingMatrix, camera.up.x, camera.up.y, camera.up.z, &camera.up.x, &camera.up.y, &camera.up.z);
         apply_matrix_f(&camera.headingMatrix, camera.front.x, camera.front.y, camera.front.z, &camera.front.x, &camera.front.y, &camera.front.z);
         get_vector_rotation_matrix_f(&camera.rollMatrix, camera.front.x, camera.front.y, camera.front.z, B_DEG(rollChange));
         apply_matrix_f(&camera.rollMatrix, camera.right.x, camera.right.y, camera.right.z, &camera.right.x, &camera.right.y, &camera.right.z);
         apply_matrix_f(&camera.rollMatrix, camera.up.x, camera.up.y, camera.up.z, &camera.up.x, &camera.up.y, &camera.up.z);
         apply_matrix_f(&camera.rollMatrix, camera.front.x, camera.front.y, camera.front.z, &camera.front.x, &camera.front.y, &camera.front.z);
         get_vector_rotation_matrix_f(&camera.pitchMatrix, camera.right.x, camera.right.y, camera.right.z, B_DEG(pitchChange));
         apply_matrix_f(&camera.pitchMatrix, camera.right.x, camera.right.y, camera.right.z, &camera.right.x, &camera.right.y, &camera.right.z);
         apply_matrix_f(&camera.pitchMatrix, camera.up.x, camera.up.y, camera.up.z, &camera.up.x, &camera.up.y, &camera.up.z);
         apply_matrix_f(&camera.pitchMatrix, camera.front.x, camera.front.y, camera.front.z, &camera.front.x, &camera.front.y, &camera.front.z);

         // Build view matrix
         get_camera_matrix_f(&camera.viewMatrix, camera.position.x, camera.position.y, camera.position.z, camera.front.x, camera.front.y, camera.front.z, camera.up.x, camera.up.y, camera.up.z, B_DEG(camera.fov), camera.aspect);
         get_scaling_matrix_f(&camera.zoomMatrix, 1.0, 1.0, 1.0 / camera.zoom);
         matrix_mul_f(&camera.viewMatrix, &camera.zoomMatrix, &camera.viewMatrix);

         // limit tilt
         switch (cameraMode) {
         case CAMERA_WALK:
            float tiltAngle;
            float maxTiltAngle;
            float retiltAngle;
            MATRIX_f temp;
            VECTOR tiltAxis;
            maxTiltAngle = 45.0;
            tiltAngle = DEG(acos(dot_product_f(0.0, 1.0, 0.0, camera.up.x, camera.up.y, camera.up.z)));
            if (tiltAngle == 0) { // Don't divide by zero
            }
            else if (fabs(tiltAngle) > fabs(maxTiltAngle)) {
               cross_product_f(0.0, 1.0, 0.0, camera.up.x, camera.up.y, camera.up.z, &tiltAxis.x, &tiltAxis.y, &tiltAxis.z);
               retiltAngle = tiltAngle - maxTiltAngle;
               while (retiltAngle < 0) retiltAngle += 360.0;
               while (retiltAngle >= 360.0) retiltAngle -= 360.0;
               get_vector_rotation_matrix_f(&temp, tiltAxis.x, tiltAxis.y, tiltAxis.z, B_DEG(retiltAngle));
               apply_matrix_f(&temp, camera.front.x, camera.front.y, camera.front.z, &camera.front.x, &camera.front.y, &camera.front.z);
               apply_matrix_f(&temp, camera.up.x, camera.up.y, camera.up.z, &camera.up.x, &camera.up.y, &camera.up.z);
               apply_matrix_f(&temp, camera.right.x, camera.right.y, camera.right.z, &camera.right.x, &camera.right.y, &camera.right.z);
            }
            else {
/*               cross_product_f(0.0, 1.0, 0.0, camera.up.x, camera.up.y, camera.up.z, &tiltAxis.x, &tiltAxis.y, &tiltAxis.z);
               retiltAngle = tiltAngle;
               retiltAngle += -(tiltAngle / 8.0);
               while (retiltAngle < 0) retiltAngle += 360.0;
               while (retiltAngle >= 360.0) retiltAngle -= 360.0;
               get_vector_rotation_matrix_f(&temp, tiltAxis.x, tiltAxis.y, tiltAxis.z, B_DEG(retiltAngle));
               apply_matrix_f(&temp, camera.front.x, camera.front.y, camera.front.z, &camera.front.x, &camera.front.y, &camera.front.z);
               apply_matrix_f(&temp, camera.up.x, camera.up.y, camera.up.z, &camera.up.x, &camera.up.y, &camera.up.z);
               apply_matrix_f(&temp, camera.right.x, camera.right.y, camera.right.z, &camera.right.x, &camera.right.y, &camera.right.z);
*/            }
            break;
         }

         camera.position.x += camera.velocity.x;
         camera.position.y += camera.velocity.y;
         camera.position.z += camera.velocity.z;

         if (camera.position.x < 0) camera.position.x = 0;
         if (camera.position.z < 0) camera.position.z = 0;
         if (camera.position.x > ((world.getW()) * SQUARE_WIDTH)) camera.position.x = ((world.getW()) * SQUARE_WIDTH);
         if (camera.position.z > ((world.getH()) * SQUARE_DEPTH)) camera.position.z = ((world.getH()) * SQUARE_DEPTH);

         switch (cameraMode) {
         case CAMERA_FREE:
            // Leave it be
            break;
         case CAMERA_WALK:
            // Restrict to a fixed height above terrain
            int mapX = (int)(camera.position.x / SQUARE_WIDTH);
            int mapZ = (int)(camera.position.z / SQUARE_DEPTH);
            if (mapX < 0) mapX = 0;
            if (mapZ < 0) mapZ = 0;
            if (mapX > world.getW() - 1) mapX = world.getW() - 1;
            if (mapZ > world.getH() - 1) mapZ = world.getH() - 1;
            float subX = (camera.position.x / SQUARE_WIDTH) - mapX;
            float subZ = (camera.position.z / SQUARE_DEPTH) - mapZ;
            float y0 = world.getPoint(mapX + 0, mapZ + 0)->y;
            float y1 = world.getPoint(mapX + 1, mapZ + 0)->y;
            float y2 = world.getPoint(mapX + 1, mapZ + 1)->y;
            float y3 = world.getPoint(mapX + 0, mapZ + 1)->y;
            float topY = y0 + (subX * (y1 - y0));
            float bottomY = y3 + (subX * (y2 - y3));
            camera.position.y = topY + (subZ * (bottomY - topY)) + standHeight;
            break;
         }
         cycle++;
         cycleTotal++;
         speedCounter--;
      }
      // Draw
      clear_to_color(screenBuffer, makecol_depth(videoDepth, FADE_COLOR_RED, FADE_COLOR_GREEN, FADE_COLOR_BLUE));
      render(screenBuffer, &world, &camera, &tex);
      set_add_blender(255, 255, 255, 127);
      drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
      if (mapMode > 0) {
         drawMap(screenBuffer, videoWidth - (videoWidth - videoHeight) * 2, 0, (videoWidth - videoHeight) * 2 - 1, (videoWidth - videoHeight) * 2 - 1, &world, &camera, &tex);
      }
      text_mode(-1);
      int textColour = makecol_depth(videoDepth, 255, 255, 255);
      textprintf(screenBuffer, font,   0,   0, textColour, "position: <%6.2f,%6.2f,%6.2f>", camera.position.x, camera.position.y, camera.position.z);
      textprintf(screenBuffer, font,   0,   8, textColour, "front: <%5.2f,%5.2f,%5.2f>, up: <%5.2f,%5.2f,%5.2f>, right: <%5.2f,%5.2f,%5.2f>", camera.front.x, camera.front.y, camera.front.z, camera.up.x, camera.up.y, camera.up.z, camera.right.x, camera.right.y, camera.right.z);
      textprintf(screenBuffer, font,   0,  16, textColour, "fov: %6.2f, aspect: %6.2f, zoom: %6.2f", camera.fov, camera.aspect, camera.zoom);
      textprintf(screenBuffer, font,   0,  24, textColour, "squares: %d, polys: %d", squaresDrawn, polysDrawn);
      textprintf(screenBuffer, font,   0,  32, textColour, "fog range: %d, stand height: %d", fogRange, standHeight);
      textprintf(screenBuffer, font,   0,  40, textColour, "time: %02d:%02d:%02d.%03d, fps: %d, cps: %d", hours, minutes, seconds, milliseconds, fps, cps);
      vsync();
      blit(screenBuffer, screen, 0, 0, 0, 0, videoWidth, videoHeight);
      frame++;
      frameTotal++;
      squaresDrawn = 0;
      polysDrawn = 0;
   }

   destroy_bitmap(screenBuffer);
   tex.close();

   destroy_font(game_font);

   remove_int(timerUpdate);
   remove_int(incrementSpeedCounter);

   return 0;
}
END_OF_MAIN();
