#include "font.h"

#include <allegro/aintern.h>
#include <string.h>
#include <stdio.h>

/* splits bitmaps into sub-sprites, using regions bounded by col #255 */
static void find_character(AL_CONST BITMAP *bmp, int *x, int *y, int *w, int *h)
{
   int c1;
   int c2;

   if (bitmap_color_depth(bmp) == 8) {
      c1 = 255;
      c2 = 255;
   }
   else {
      c1 = makecol_depth(bitmap_color_depth(bmp), 255, 255, 0);
      c2 = makecol_depth(bitmap_color_depth(bmp), 0, 255, 255);
   }

   /* look for top left corner of character */
   while ((getpixel(bmp, *x,   *y)   != c1) ||
          (getpixel(bmp, *x+1, *y)   != c2) ||
          (getpixel(bmp, *x,   *y+1) != c2) ||
          (getpixel(bmp, *x+1, *y+1) == c1) ||
          (getpixel(bmp, *x+1, *y+1) == c2)) {
      (*x)++;
      if (*x >= bmp->w) {
         *x = 0;
         (*y)++;
         if (*y >= bmp->h) {
            *w = 0;
            *h = 0;
            return;
         }
      }
   }

   /* look for right edge of character */
   *w = 0;
   while ((getpixel(bmp, *x+*w+1, *y)   == c2) &&
          (getpixel(bmp, *x+*w+1, *y+1) != c2) &&
          (*x+*w+1 <= bmp->w))
      (*w)++;

   /* look for bottom edge of character */
   *h = 0;
   while ((getpixel(bmp, *x,   *y+*h+1) == c2) &&
          (getpixel(bmp, *x+1, *y+*h+1) != c2) &&
          (*y+*h+1 <= bmp->h))
      (*h)++;
}



/* exports a font into an external file */
int save_font(AL_CONST char *filename, FONT *fnt)
{
   BITMAP *b;
   FONT_GLYPH *g;
   PALETTE pal;
   int w, h, c;
   int chars;
   int col;

   w = 0;
   h = 0;

   chars = 0;
   if (fnt->mono == TRUE) {
      for (c = fnt->start; c <= fnt->end; c++) {
         g = (FONT_GLYPH *)fnt->glyphs[c - fnt->start];
         if (g->w > w)
            w = g->w;
         if (g->h > h)
            h = g->h;
         chars++;
      }
   }
   else {
      for (c = fnt->start; c <= fnt->end; c++) {
         b = (BITMAP *)fnt->glyphs[c - fnt->start];
         if (b->w > w)
            w = b->w;
         if (b->h > h)
            h = b->h;
         chars++;
      }
   }

   w = (w+16) & 0xFFF0;
   h = (h+16) & 0xFFF0;

   b = create_bitmap_ex(8, 1 + w * 16, 1 + h * ((chars + 15) / 16));
   rectfill(b, 0, 0, b->w, b->h, 255);
   text_mode(0);

   chars = 0;
   col = (fnt->mono) ? 1 : -1;
   for (c = fnt->start; c <= fnt->end; c++) {
      textprintf(b, fnt, 1 + w * (chars & 15), 1 + h * (chars / 16), col, "%c", c);
      chars++;
   }

   pal[0].r = 63;
   pal[0].g = 0;
   pal[0].b = 63;

   for (c = 1; c < 255; c++) {
      col = (c - 1) * 63 / 253;
      pal[c].r = col;
      pal[c].g = col;
      pal[c].b = col;
   }

   pal[255].r = 63;
   pal[255].g = 63;
   pal[255].b = 0;

   save_bitmap(filename, b, pal);
   destroy_bitmap(b);

   return (errno == 0);
}



/* import routine for the Allegro .pcx font format */
FONT *load_font(AL_CONST char *filename)
{
   BITMAP **bmp;
   PALETTE junk;
   FONT *fnt;
   FONT_GLYPH *g;
   int x, y, w, h;
   int c, d;
   int max_h;
   int stride;
   int col;
   BITMAP *import_bmp = NULL;
   int import_x = 0;
   int import_y = 0;
   int minchar = ' ';
   int maxchar = -1;

   if (filename) {
      set_color_conversion(COLORCONV_NONE);
      import_bmp = load_bitmap(filename, junk);

      import_x = 0;
      import_y = 0;
   }

   if (!import_bmp)
      return NULL;

   if (bitmap_color_depth(import_bmp) != 8) {
      destroy_bitmap(import_bmp);
      import_bmp = NULL;
      return NULL;
   }

   bmp = (BITMAP **)malloc(65536*sizeof(BITMAP *));
   max_h = 0;
   col = 0;
   c = 0;

   for (;;) {
      find_character(import_bmp, &import_x, &import_y, &w, &h);

      if ((w <= 0) || (h <= 0))
         break;

      bmp[c] = create_bitmap_ex(8, w, h);
      blit(import_bmp, bmp[c], import_x+1, import_y+1, 0, 0, w, h);
      max_h = MAX(max_h, h);
      import_x += w;

      if (col >= 0) {
         for (y=0; y<h; y++) {
            for (x=0; x<w; x++) {
               if (bmp[c]->line[y][x]) {
                  if ((col) && (col != bmp[c]->line[y][x]))
                     col = -1;
                  else
                     col = bmp[c]->line[y][x];
               }
            }
         }
      }

      c++;

      if ((maxchar > 0) && (minchar+c > maxchar))
         break;
   }

   if (c > 0) {
      fnt = (FONT *)_al_malloc(sizeof(FONT));
      fnt->mono = (col >= 0) ? TRUE : FALSE;
      fnt->start = minchar;
      fnt->end = minchar+c-1;
      fnt->next = NULL;
      fnt->renderhook = NULL;
      fnt->widthhook = NULL;
      fnt->heighthook = NULL;

      fnt->glyphs = (void **)_al_malloc(c*sizeof(void *));

      for (d=0; d<c; d++) {
         w = bmp[d]->w;
         h = max_h;

         if (col >= 0) {
            stride = (w+7)/8;

            g = (FONT_GLYPH *)_al_malloc(sizeof(FONT_GLYPH) + stride*h);
            g->w = w;
            g->h = h;

            memset(g->dat, 0, stride*h);

            for (y=0; y<bmp[d]->h; y++) {
               for (x=0; x<bmp[d]->w; x++) {
                  if (bmp[d]->line[y][x])
                     g->dat[y*stride + x/8] |= (0x80 >> (x&7));
               }
            }

            fnt->glyphs[d] = g;
         }
         else {
            fnt->glyphs[d] = create_bitmap_ex(8, w, h);
            clear((BITMAP *)fnt->glyphs[d]);
            blit(bmp[d], (BITMAP *)fnt->glyphs[d], 0, 0, 0, 0, w, h);
         }
      }
   }
   else
      fnt = NULL;

   for (d=0; d<c; d++)
      destroy_bitmap(bmp[d]);

   free(bmp);

   destroy_bitmap(import_bmp);
   import_bmp = NULL;

   return fnt;
}

