#include "prototypes.h"

volatile int speed_counter = 0;								// Timer handler stuff!
void increment_speed_counter (void) { speed_counter++; }
END_OF_FUNCTION(increment_speed_counter)


int main (void)
{	
	// Allegro-specific objects
	BITMAP *buffer;						// Video memory buffer
	BITMAP *status_bar_bmp;				// Holds the image of the status bar
	BITMAP *sprites[MAX_SPRITES];		// Holds the sprite data
	RLE_SPRITE *background_image;		// Background image to go behind the blocks of the level
	RLE_SPRITE *rle_sprites[MAX_SPRITES];	// Holds the RLE sprite data (faster drawing format)
//	DATAFILE *dat;						// Holds info from the datafile
//	FONT *game_font;					// Font object, holds the unique game font

	// Objects of types defined here
	LEVEL_TYPE level;					// Holds all level data
	PLAYER_TYPE player;					// Holds all player data 
	ENEMY_TYPE enemies[MAX_ACTIVE_ENEMIES];		// Holds variable enemy data
	ENEMY_DATA_TYPE enemy_data[NUM_ENEMY_DATA];	// Holds constant enemy data
	WINDOW_TYPE window;					// Holds all window data
	KEYS_TYPE keysdown;					// Holds keypress data
	CELL_DATA_TYPE cell_data;			// Holds individual cell data
	COLLISION_INPUT_TYPE collision_input_data;		// Holds data to pass to detect_collisions()
	COLLISION_DETECTION_TYPE collision_data[2][4];	// Holds results of collision detection
	ANIMATION_TYPE animations[MAX_ANIMATIONS];		// Holds data about animations on screen
	SAMPLE *background_music = NULL;	// Digital sample of the background music

	char is_rle_sprite[MAX_SPRITES];	// Indicates whether a sprite index is an RLE sprite
	char include_o2_bar;				// Indicates whether or not to refresh the o2 bar also
										// when updating the status bar
	char draw_items;					// Indicates whether or not to re-draw the item icon
	char refresh_items;					// Indicates whether or not to refresh the items bar
										// when updating the status bar
	char refresh_status_bar_flag[2];	// Indicates whether or not to refresh the status bar
	int music_volume = 128;				// Volume of the background music. Range: 0-255
	char music_filename[5000] = "";		// Holds the filename of the music file to play
		
	LOCK_VARIABLE(speed_counter);
	LOCK_FUNCTION(increment_speed_counter);
	
	// Set 640x480x65536 graphics and initialize everything
	init_beginning(640, 480, 16, &buffer, &status_bar_bmp, &level, &player, &window, &keysdown);
	
	// Load the sprites from their respective bitmap files into the sprites array
	cell_data.num_sprites = load_sprites(sprites, rle_sprites, &player, enemy_data, &cell_data,
		is_rle_sprite);

	// Assign the cell types to the cells
	assign_cell_types (&cell_data);

	// Load the timer handler stuff
	install_int_ex(increment_speed_counter, BPS_TO_TIMER(GAME_MSEC_DELAY));
		
	while (player.lives > 0) {	// Loop until the player runs out of lives
	
		player.throw_angle = 32;
		player.has_item = ITEM_NONE;
		
		// Load the level
		load_level(&background_image, &status_bar_bmp, sprites, rle_sprites, &level, &player,
			enemies, enemy_data, &window, &cell_data, animations, player.cur_level, 
			music_filename);
		
		// Load the music 
		//load_music (background_music, music_filename, &music_volume);

		// Play the music
		//play_music (background_music, &music_volume);

		while (player.health > 0)	// Loop until the player dies
		{
			while (speed_counter > 0 && player.health > 0)	// Loop until it's time to draw eerything
			{
				// By default, do not refresh the status bar
				refresh_status_bar_flag[0] = FALSE;
				refresh_status_bar_flag[1] = FALSE;
			
				// Respond to keyboard input
				refresh_status_bar_flag[0] =
					check_input (&buffer, &status_bar_bmp, sprites, rle_sprites, is_rle_sprite,
					&level, &player, &window, &keysdown, &cell_data, animations, &music_volume);
		
				// Prepare the collision input data object to pass to detect_collisions()
				get_collision_input (&collision_input_data, DETECT_PLAYER, -1, &keysdown,
						  &player, enemies, enemy_data, animations);
				
				// Perform collision detection on the player
				detect_collisions (&collision_input_data, &level, &player, enemies, enemy_data,
					&keysdown, &cell_data, collision_data);

				// React in response to the data obtained by collision detection
				refresh_status_bar_flag[1] =
					react_to_collision_detection (&status_bar_bmp, sprites, rle_sprites, 
					is_rle_sprite, &level, &window, &player, collision_data, &keysdown,
					&cell_data, animations);

				if (player.health <= 0) {		// If the player has zero health
					kill_player (&player);		// Play the death routine
					break;						// Exit and re-load the level
				}

				// Move the player
				move_player (&level, &player, enemies, &window, &keysdown, collision_data, 
					&cell_data);

				// Move the enemies
				if (move_enemies (&level, &player, &keysdown, enemies, enemy_data, &window, 
					&collision_input_data, collision_data, &cell_data, animations, sprites,
					rle_sprites, is_rle_sprite) == TRUE)
				{
					refresh_status_bar_flag[0] = TRUE;
				}

				// Move the animations
				if (move_animations (&collision_input_data, collision_data, &level, &player, 
					&window, &cell_data, animations, enemies, enemy_data, &keysdown, sprites,
					rle_sprites, is_rle_sprite) == TRUE)
				{
					refresh_status_bar_flag[0] = TRUE;
				}

				// Update the status bar if necessary
				if (refresh_status_bar_flag[0] <= TRUE || refresh_status_bar_flag[1] <= TRUE) {
					
					// Determine what to include
					draw_items = FALSE;
					refresh_items = FALSE;
					if (player.is_swimming) {
						include_o2_bar = TRUE;
					}
					else {
						include_o2_bar = FALSE;
						if (refresh_status_bar_flag[0] == TRUE - 1 ||
							refresh_status_bar_flag[1] == TRUE - 1)
						{
							draw_items = TRUE;
							refresh_items = TRUE;
						}
					}
					
					// Refresh the status buar
					refresh_status_bar
						(FALSE, TRUE, 
						FALSE, TRUE, 
						FALSE, TRUE, 
						FALSE, include_o2_bar,
						draw_items, refresh_items,
						&status_bar_bmp, rle_sprites, &player, &window, 
						&cell_data);
				}
			
				// Update the animation of the cells
				update_animation (sprites, rle_sprites, is_rle_sprite, &level, &player,
					&cell_data, animations);

				// We have completed the game logic - one step closer to a refresh
				speed_counter--;

			}

			// Draw the graphics to the buffer
			draw_screen(&buffer, &status_bar_bmp, &background_image, sprites, rle_sprites, 
				&level,	&player, enemies, enemy_data, &window, &cell_data, animations,
				&keysdown, is_rle_sprite);

			// Copy the graphics to the screen
			copy_page(buffer, screen);

			rest(5);
		}

		// If the program has reached this point, the player has died and his health must be refilled
		player.health = player.max_health;
	}

	return 0;
}

END_OF_MAIN ()				// Needed for allegro library


/*
Initializes the keyboard, mouse, timer, graphics mode, default directory,
and text mode.

 * Sets 640x480x65536 graphics mode
 * Initializes the keyboard and mouse
 * Initializes the keyboard keys
 * Initializes the level variables
 * Initializes the video memory buffer

 */
void init_beginning (int screen_x_size, int screen_y_size, int color_depth, BITMAP **buffer,
					 BITMAP **status_bar_bmp, LEVEL_TYPE *level, PLAYER_TYPE *player,
					 WINDOW_TYPE *window, KEYS_TYPE *keysdown)
{	
//	int x, y, offset, loops;
	int gfx_mode;
	
	// Init the level data
	level->width = 0;		// Tells the load_level function not to free the memory
	level->height = 0;		// allocated to the level variables

	// Init the player data
	player->top_buffer = 3;
	player->bottom_buffer = 2;
	player->left_buffer = 6;
	player->right_buffer = 8;
	player->left_slip_buffer = 8;
	player->right_slip_buffer = 8;
	player->max_health = 150;
	player->lives = 5;
	player->max_lives = 99;
	player->score = 0;
	player->max_score = 99999;
	player->health = player->max_health - 50;
	player->cur_level = 1;

	// Init the window
	window->x_cell = 0;
	window->y_cell = 0;
	window->x_offset = 0;
	window->y_offset = 0;
	window->top_buffer = screen_y_size / 3;
	window->bottom_buffer = screen_y_size / 3;
	window->left_buffer = screen_x_size / 3;
	window->right_buffer = screen_x_size / 3;
	// Center the lifebar horizontally on the screen	
//	window->lifebar_x = (screen_x_size / 2) - (player->max_health / 2 + 25 + 11 + 11);
	window->lifebar_x = 4;
	window->lifebar_y = 3;

	// Init the keyboard keys
	keysdown->esc = FALSE;
	keysdown->lalt = FALSE;
	keysdown->lctrl = FALSE;
	
	allegro_init();
	install_keyboard();
	install_mouse();
	install_timer();
		
	// Init graphics mode
	set_color_depth (color_depth);			// Value of 16 sets 256 color mode	
	gfx_mode = GFX_AUTODETECT_FULLSCREEN;	// Fullscreen view
	if (set_gfx_mode(gfx_mode, screen_x_size, screen_y_size, 0, 0) != 0)	{
		allegro_message("An error initializing the graphics (%d by %d) occured.\n%s\n", 
			screen_x_size, screen_y_size, allegro_error);
		exit (1);
    }

	// Init audio
	reserve_voices(NUM_SOUND_VOICES, 1);
	if (install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, 0) != 0)
	{
		allegro_message("Error setting audio mode\n%s\n", allegro_error);
		exit (1);		
	}

	*buffer = create_bitmap(SCREEN_W, SCREEN_H);					// Create the video buffer
	*status_bar_bmp = create_bitmap(SCREEN_W, STATUS_BAR_HEIGHT);	// Create status bar bitmap

	if (*buffer == NULL) {
		allegro_message("Error allocating memory for the offscreen buffer!\n%s\n",
			allegro_error);
		exit(1);
	}

	clear_bitmap (*buffer);				// clear the buffer to color 0
	clear_bitmap (screen);				// clear the screen to color 0

} // End init_beginning



// Loads the background music from a file and plays it 
void load_music (SAMPLE *background_music, char *music_filename, int *music_volume)
{
	long filesize_k;
	
	char file_size_str[5000];		// String that holds the size of the file in KB
	char load_msg[5000];			// Loading message to display to user
	char full_filename[5000];		// Filename of the music file to load
	char wav_filename[5000];		// Filename of the music file to load relative to Music/

	// Start the filename of the music file rooted at the music folder
	strcpy(full_filename, "Music\\");

	// Decide the wave filename
	strcpy(wav_filename, music_filename);	
	//strcpy(wav_filename, "coldplay - brothers and sisters.wav");	

	// Add the wave filename
	strcat(full_filename, wav_filename);

 	fix_filename_slashes(full_filename);

	// Get the size of the wave file
	filesize_k = file_size(full_filename) / 1024;
	
	// Convert the file size to a string
	get_str_from_long (file_size_str, filesize_k);

	// Display a loading message
	strcpy(load_msg, "Loading music (");
	strcat(load_msg, file_size_str);
	strcat(load_msg, " KB)...");
	TEXTPRINTF(screen, font, 0, text_height(font) + 3, TEXTPRINTF_COL, load_msg);

	background_music = load_sample (full_filename);
								 //("Music\\air - moon safari - la femme d'argent.wav");

	/*
	SAMPLE *load_sample(const char *filename);

	Loads a sample from a file, returning a pointer to it, or NULL on error. At present
	this function supports both mono and stereo WAV and mono VOC files, in 8 or 16 bit formats.
	*/

	if (background_music == NULL)
	{
		shout ("A problem occurred loading the background music!");
		exit(0);
	}

	if (allocate_voice (background_music) == -1)			// Reserve the sound voice
	{
		shout ("No sound voices were avaiable!");
		exit(0);
	}
	
	play_sample (background_music, *music_volume, 128, 1000, TRUE);		// Loop the song

}



// Starts the background music playing
void play_music (SAMPLE *background_music, int *music_volume)
{
	/*
	int allocate_voice(const SAMPLE *spl);
	
	Allocates a soundcard voice and prepares it for playing the specified sample, setting 
	up sensible default parameters (maximum volume, centre pan, no change of pitch, no 
	looping). When you are finished with the voice you must free it by calling 
	deallocate_voice() or release_voice(). Returns the voice number, or -1 if no voices 
	are available.
	*/

	//play_sample (background_music, *music_volume, 128, 1000, TRUE);		// Loop the song
//	play_sample (background_music, 100, 128, 1000, TRUE);		// Loop the song

	/*
	int play_sample(const SAMPLE *spl, int vol, int pan, int freq, int loop);

	Triggers a sample at the specified volume, pan position, and frequency. The volume and
	pan range from 0 (min/left) to 255 (max/right). Frequency is relative rather than 
	absolute: 1000 represents the frequency that the sample was recorded at, 2000 is twice 
	this, etc. If the loop flag is set, the sample will repeat until you call stop_sample(),
	and can be manipulated while it is playing by calling adjust_sample(). Returns the voice 
	number that was allocated for the sample (a non-negative number if successful). 
	*/
}



// Loads the sprites from filenames specified in tilelist.txt into the sprite array
// RETURNS the number of sprites
int load_sprites (BITMAP *sprites[], RLE_SPRITE *rle_sprites[], PLAYER_TYPE *player, 
				  ENEMY_DATA_TYPE enemy_data[], CELL_DATA_TYPE *cell_data,
				  char is_rle_sprite[])
{
	FILE *fp_in;
	char str[2000];
	char filename[2000];
	int which_file, num_files;
	int which_enemy = 0;
	int which_enemy_data_type = 0;

	// Display a loading message
	TEXTPRINTF(screen, font, 0, 0, TEXTPRINTF_COL, "Loading graphics: ");
	// Draw the progress bar
	rect(screen, text_length(font, "Loading graphics: "), 0, SCREEN_W - 1, text_height(font), 
		WHITE);

	// Determine the total number of files to load
	strcpy(str, "Images\\sprlist.txt");
	fix_filename_slashes(str);
	fp_in = fopen(str, "r");		// Open the sprite file for input	
	for (num_files = 0; fgets(str, 2000, fp_in) != NULL; num_files++) {}	// Read each line
	fclose(fp_in);															//	Close the file

	strcpy(str, "Images\\sprlist.txt");
	fix_filename_slashes(str);
	fp_in = fopen(str, "r");		// Open the sprite file for input	
	
	// Loop until no more files are found
	for (which_file = 0; fgets(str, 2000, fp_in) != NULL; which_file++) {
		
		if (strlen(str) == 0) continue;			// Empty string: skip one iteration
		if (FILESYSTEM_WINDOWS == TRUE)
			str[strlen(str) - 1] = '\0';			// Remove the trailing newline
		if (FILESYSTEM_LINUX == TRUE)
			str[strlen(str) - 2] = '\0';			// Remove the trailing newline
		strcpy(filename, "Images\\");
		strcat(filename, str);

		is_rle_sprite[which_file] = FALSE;		// Mark as non-RLE by default
		
		// Load the sprite from the bitmap file
		fix_filename_slashes(filename);
		//fix_filename_escape_chars(filename);
		
		if ((sprites[which_file] = load_bitmap(filename, NULL)) == NULL) {
			// If the load failed, display an error message
			allegro_message("Error loading sprite #%d from file \"%s\"", which_file, filename);
			exit (1);
		}
		
		if (strcmp(str, "player swim 1.bmp") == 0 ||						// Do not convert
			strcmp(str, "player swim 2.bmp") == 0 ||						// to RLE sprite!
			strcmp(str, "player swim 3.bmp") == 0);
		else if (strcmp(str, "cannon active.bmp") == 0);
		else if (strcmp(str, "smack cannon.bmp") == 0);
		// If the string starts with "enemy"
		else if (str[0] == 'e' && str[1] == 'n' && str[2] == 'e' && str[3] == 'm' &&
			str[4] == 'y') {

			// If the string ends with "1.bmp"
			if (str[strlen(str) - 5] == '1') {
//				TEXTPRINTF(screen, font, 0, (which_enemy_data_type * 10) + 50, TEXTPRINTF_COL, 
//					"cell_data->enemy_index[%d] = %d", which_enemy_data_type, which_file);
				
				// Set some characteristics of the enemy object
				cell_data->enemy_index[which_enemy_data_type] = which_file;
				enemy_data[which_enemy_data_type].width = sprites[which_file]->w;
				enemy_data[which_enemy_data_type].height = sprites[which_file]->h;
				enemy_data[which_enemy_data_type].cell_index = which_file;
		
				which_enemy_data_type++;
			}
		} else if (strcmp(str, "player right 1.bmp") == 0) {
			// Set the player's width and height
			player->width = sprites[which_file]->w - (player->left_buffer + player->right_buffer);
			player->height = sprites[which_file]->h - (player->top_buffer + player->bottom_buffer);
		} else if (strcmp(str, "player in cannon.bmp") == 0) {
		} else if (strcmp(str, "crosshair angles.bmp") == 0) {
		} else {
			rle_sprites[which_file] = get_rle_sprite(sprites[which_file]);	// Convert to RLE
			destroy_bitmap(sprites[which_file]);							// Destroy bitmap
			is_rle_sprite[which_file] = TRUE;								// Mark as RLE
		}

		// Sprite indexes
		if (strcmp(str, "player swim 1.bmp") == 0)
			cell_data->swim_index = which_file;
		else if (strcmp(str, "player front.bmp") == 0)
			cell_data->player_index = which_file;
		else if (strcmp(str, "empty space.bmp") == 0)
			cell_data->empty_cell_index = which_file;
		else if (strcmp(str, "lifebar icon.bmp") == 0)
			cell_data->lifebar_icon_index = which_file;		
		else if (strcmp(str, "o2 icon.bmp") == 0)
			cell_data->o2_icon_index = which_file;			
		else if (strcmp(str, "lifebar left.bmp") == 0)
			cell_data->lifebar_left_index = which_file;		
		else if (strcmp(str, "lifebar middle.bmp") == 0)
			cell_data->lifebar_middle_index = which_file;	
		else if (strcmp(str, "lifebar right.bmp") == 0)
			cell_data->lifebar_right_index = which_file;	
		else if (strcmp(str, "bananas 1.bmp") == 0)
			cell_data->bananas_index = which_file;			
		else if (strcmp(str, "apple little 1.bmp") == 0)
			cell_data->apples_index = which_file;			
		else if (strcmp(str, "grapes 1.bmp") == 0)
			cell_data->grapes_index = which_file;			
		else if (strcmp(str, "watermelon 1.bmp") == 0)
			cell_data->watermelon_index = which_file;		
		else if (strcmp(str, "heart 1.bmp") == 0)
			cell_data->heart_index = which_file;			
		else if (strcmp(str, "coffee 1.bmp") == 0)
			cell_data->coffee_index = which_file;			
		else if (strcmp(str, "number small 0.bmp") == 0)
			cell_data->number_small_zero_index = which_file;	
		else if (strcmp(str, "number big 0.bmp") == 0)
			cell_data->number_big_zero_index = which_file;	
		else if (strcmp(str, "water 1.bmp") == 0)
			cell_data->water_index = which_file;			
		else if (strcmp(str, "waterfall 1.bmp") == 0)
			cell_data->waterfall_index = which_file;		
		else if (strcmp(str, "conveyor right left end 1.bmp") == 0)	
			cell_data->conveyor_index[0] = which_file;				
		else if (strcmp(str, "conveyor right middle 1.bmp") == 0)	
			cell_data->conveyor_index[1] = which_file;				
		else if (strcmp(str, "conveyor right right end 1.bmp") == 0)	
			cell_data->conveyor_index[2] = which_file;				
		else if (strcmp(str, "conveyor left left end 4.bmp") == 0)	
			cell_data->conveyor_index[3] = which_file;				
		else if (strcmp(str, "conveyor left middle 4.bmp") == 0)	
			cell_data->conveyor_index[4] = which_file;				
		else if (strcmp(str, "conveyor left right end 4.bmp") == 0)	
			cell_data->conveyor_index[5] = which_file;				
		else if (strcmp(str, "spring up 1.bmp") == 0)				
			cell_data->spring_index[0] = which_file;				
		else if (strcmp(str, "spring down 1.bmp") == 0)				
			cell_data->spring_index[1] = which_file;				
		else if (strcmp(str, "spring left 1.bmp") == 0)				
			cell_data->spring_index[2] = which_file;				
		else if (strcmp(str, "spring right 1.bmp") == 0)			
			cell_data->spring_index[3] = which_file;				
		else if (strcmp(str, "rock diamond square.bmp") == 0)		
			cell_data->rocksquare_index = which_file;				
		else if (strcmp(str, "spikes up.bmp") == 0)					
			cell_data->spikes_index[0] = which_file;				
		else if (strcmp(str, "spikes down.bmp") == 0)				
			cell_data->spikes_index[1] = which_file;				
		else if (strcmp(str, "spikes left.bmp") == 0)				
			cell_data->spikes_index[2] = which_file;				
		else if (strcmp(str, "spikes right.bmp") == 0)				
			cell_data->spikes_index[3] = which_file;				
		else if (strcmp(str, "lava 1.bmp") == 0)					
			cell_data->lava_index[0] = which_file;						
		else if (strcmp(str, "lava 2.bmp") == 0)					
			cell_data->lava_index[1] = which_file;						
		else if (strcmp(str, "lava 3.bmp") == 0)					
			cell_data->lava_index[2] = which_file;						
		else if (strcmp(str, "electricity 1 left.bmp") == 0)		
			cell_data->e_index[0] = which_file;						
		else if (strcmp(str, "electricity 1 right.bmp") == 0)		
			cell_data->e_index[1] = which_file;						
		else if (strcmp(str, "switch block blue.bmp") == 0)			
			cell_data->switch_block_index = which_file;				
		else if (strcmp(str, "dotted outline blue.bmp") == 0)		
			cell_data->dotted_outline_index = which_file;			
		else if (strcmp(str, "filled block blue.bmp") == 0)			
			cell_data->filled_block_index = which_file;				
		else if (strcmp(str, "x icon.bmp") == 0)			
			cell_data->x_icon_index = which_file;				
		else if (strcmp(str, "bubble 1.bmp") == 0)			
			cell_data->bubble_index = which_file;				
		else if (strcmp(str, "cannon base.bmp") == 0)
			cell_data->cannon_index = which_file;				
		else if (strcmp(str, "player in cannon.bmp") == 0)
			cell_data->player_in_cannon_index = which_file;				
		else if (strcmp(str, "smack.bmp") == 0)
			cell_data->smack_index = which_file;
		else if (strcmp(str, "crosshair angles.bmp") == 0)
			cell_data->crosshair_angles_index = which_file;
		else if (strcmp(str, "bomb.bmp") == 0)
			cell_data->bomb_index = which_file;
		else if (strcmp(str, "bomb explosion 1.bmp") == 0)
			cell_data->explosion_index = which_file;
		else if (strcmp(str, "bomb powerup.bmp") == 0)
			cell_data->bomb_powerup_index = which_file;
		
		// Fill the appropriate part of the progress bar
		rectfill (screen, text_length(font, "Loading graphics: ") + 1, 1, 
			text_length(font, "Loading graphics: ") + 1 + 
			(((SCREEN_W - text_length(font, "Loading graphics: ") - 1) 
				* which_file) / num_files),
			text_height(font) - 1, GRAY);
	}

	fclose(fp_in);								// Close tilelist.txt for input
	
	return which_file;
}



// Fills up the cell_types array
void assign_cell_types (CELL_DATA_TYPE *cell_data)
{
	FILE *fp_in;		// For text input from tilelist.txt
	FILE *fp_in2;		// For text input from sprlist.txt
	char str[2000];
	char str2[2000];
	char filename[2000];
	int which_tile, which_sprite;
	char which_cell_type;
	int which_spring;
	char num_cell_frames;
	int num_files;


	// Determine the total number of files to load
	strcpy(filename, "Images\\tilelist.txt");
	fix_filename_slashes(filename);
	fp_in = fopen(filename, "r");				// Open the sprite file for input	
	for (num_files = 0; fgets(str, 2000, fp_in) != NULL; num_files++) {}	// Read each line
	fclose(fp_in);															//	Close the file

	strcpy(filename, "Images\\tilelist.txt");
	fix_filename_slashes(filename);
	fp_in = fopen(filename, "r");		// Open tilelist.txt for text input
	for (which_tile = 0; fgets(str, 2000, fp_in) != NULL; which_tile++) {

		// Check to see if this filename also appears in sprlist.txt
		strcpy(filename, "Images\\sprlist.txt");
		fix_filename_slashes(filename);
		fp_in2 = fopen(filename, "r");	// Open sprlist.txt for text input
		for (which_sprite = 0; fgets(str2, 2000, fp_in2) != NULL; which_sprite++) {
			if (strcmp(&str[6], str2) == 0) {		// If we found the filename in tilelist.txt
				str[2] = '\0';						// End the string at the third character
				str[5] = '\0';						// End the string again at the sixth char
				which_cell_type = (char) atoi(&str[0]);		// Get the cell type from the string
				num_cell_frames = (char) atoi(&str[3]);	
				cell_data->cell_types[which_sprite] = which_cell_type;		// Assign the type
				cell_data->num_frames[which_sprite] = num_cell_frames;
				break;
			}
		}
		fclose(fp_in2);								// Close sprlist.txt input
	}
	fclose(fp_in);									// Close tilelist.txt input

	// Manually assign the spring cell type data
	for (which_spring = 0; which_spring < 4; which_spring++) {
		cell_data->num_frames[cell_data->spring_index[which_spring] + 1] = 1;
		cell_data->num_frames[cell_data->spring_index[which_spring] + 2] = 1;

		cell_data->cell_types[cell_data->spring_index[which_spring] + 1] = TYPE_ACTION;
		cell_data->cell_types[cell_data->spring_index[which_spring] + 2] = TYPE_ACTION;
	}
}



// Loads the level from a file
void load_level (RLE_SPRITE **background_image, BITMAP **status_bar_bmp, BITMAP *sprites[],
				 RLE_SPRITE *rle_sprites[], LEVEL_TYPE *level, PLAYER_TYPE *player,
				 ENEMY_TYPE enemies[], ENEMY_DATA_TYPE enemy_data[], WINDOW_TYPE *window,
				 CELL_DATA_TYPE *cell_data, ANIMATION_TYPE animations[], int which_level,
				 char music_filename[])
{
	FILE *fp_in;							// Used to read from the level
	BITMAP *bmp;							// Used to load the level's background image
	int row;
	char filename_dat[2000];
	char filename_inf[2000];
	char filename_bmp[2000];
	int which_ani;
	int which_enemy = 0;
	char include_items;						// Whether or not to include items when drawing the
											// status bar
	char str[5000];
	
	if (level->width != 0 && level->height != 0) {	// If the width and height are set to 0...
		free (level->title);						// Free the memory allocated to the  title
		for (row = 0; row < level->height; row++) {	// Free the memory allocated to the tiles
			free (level->tiles[row]);					
		}
		free(level->tiles);
		destroy_rle_sprite(*background_image);		// Free the memory of the background image
	}
	
	strcpy(filename_dat, "Levels\\level");		// Generate the filename of the level data file
	get_str_from_int(str, which_level);
	strcat(filename_dat, str);
	strcpy(filename_inf, filename_dat);
	strcat(filename_dat, ".dat");
	
	fix_filename_slashes(filename_dat);
	fix_filename_slashes(filename_inf);
	fp_in = fopen(filename_dat, "rb");					// Open the level file for binary input
	fread(str, MAX_LEVEL_TITLE_LENGTH, 1, fp_in);		// Read in the level title
		
	// Allocate memory for the level title
	level->title = (char *) calloc (sizeof(char) * (strlen(str) + 1), sizeof(char));
	
	strcpy(level->title, str);							// Copy the level title to our local variable
	fread(&level->width,  sizeof(int), 1, fp_in);		// Read the level width
	fread(&level->height, sizeof(int), 1, fp_in);		// Read the level height
	fread(&level->type,   sizeof(int), 1, fp_in);		// Read the level type
	
	// Allocate memory for the level->tiles array
	level->tiles = (int **) calloc (level->height, sizeof(int *));
	for (row = 0; row < level->height; row++) {
		level->tiles[row] = (int *) calloc (level->width, sizeof(int));
		if (level->tiles[row] == NULL) {		// If something went wrong allocating the memory
			allegro_message("Error allocating memory for loading level #%d, row %d out of %d", 
				which_level, row, level->height);
			exit (1);							// Exit with error message
		}
	}

	// Read the actual data
	for (row = 0; row < level->height; row++) 
		fread(level->tiles[row], sizeof(int), level->width, fp_in);
	fclose(fp_in);												// Close the level file				
	//shout("Here!"); exit(0);

	strcpy(filename_bmp, "Levels\\level");		// Generate the filename of the level image file
	get_str_from_int(str, which_level);
	strcat(filename_bmp, str);
	strcpy(filename_inf, filename_bmp);
	strcat(filename_bmp, ".bmp");
	strcat(filename_inf, ".inf");

	fix_filename_slashes(filename_bmp);
	fix_filename_slashes(filename_inf);
	bmp = load_bitmap(filename_bmp, NULL);		// Load the level backgrond image from the file
	*background_image = get_rle_sprite(bmp);	// Create the RLE sprite from the loaded bitmap
	destroy_bitmap(bmp);						// Free the memory taken by the bitmap image

	// Create the status bar bitmap
	rectfill(*status_bar_bmp, 0, 0, (*status_bar_bmp)->w - 1, (*status_bar_bmp)->h - 1, 
		TRANSPARENT_COL);
	
	strcpy(filename_inf, "Levels\\level");		// Generate the filename of the level image file
	get_str_from_int(str, which_level);
	strcat(filename_inf, str);
	strcat(filename_inf, ".inf");

	initialize_player (level, player, window, cell_data);

	fix_filename_slashes(filename_inf);
	fp_in = fopen(filename_inf, "r");						// Open the level .inf file
	fgets(str, 5000, fp_in);								// Read in and ignore a comment line
	fgets(music_filename, 5000, fp_in);						// Get the background music filename
	music_filename[strlen(music_filename) - 1] = '\0';		// Cut off string trailing character
	fgets(str, 5000, fp_in);								// Read in and ignore a comment line
	
	// Load all the enemy data
	for (which_enemy = 0; which_enemy < NUM_ENEMY_DATA; which_enemy++)
		load_enemy_data (which_enemy, fp_in, enemy_data);

	fclose(fp_in);								// Close the level .inf file
	enemy_data[0].num_enemies = which_enemy;	// Set the number of enemies

	// Initialize all enemies and animations to not active
	for (which_enemy = 0; which_enemy < MAX_ACTIVE_ENEMIES; which_enemy++)
		enemies[which_enemy].active = FALSE;
	for (which_ani = 0; which_ani < MAX_ANIMATIONS; which_ani++)
		animations[which_ani].active = FALSE;
	
	// Scan the entire viewing window for enemies
	scan_area_for_enemies(window->x_cell, window->y_cell, window->x_cell + SCREEN_W / CELL_SIZE,
		window->y_cell + SCREEN_H / CELL_SIZE, level, enemies, enemy_data, cell_data);

	// Draw the status bar info
	if (player->has_item == ITEM_NONE) include_items = FALSE; else include_items = TRUE;
	refresh_status_bar
		(TRUE, TRUE,
		TRUE, TRUE,
		TRUE, TRUE,
		FALSE, FALSE,
		include_items, include_items,
		status_bar_bmp, 
		rle_sprites, player, window, cell_data);
/*
	// (Display some debugging information)
	TEXTPRINTF(screen, font, 0, 0,  TEXTPRINTF_COL, "Filename: \"%s\"", filename);
	TEXTPRINTF(screen, font, 0, 10, TEXTPRINTF_COL, "Title: \"%s\"", level->title);
	TEXTPRINTF(screen, font, 0, 20, TEXTPRINTF_COL, "Size: \"%d, %d\"", level->width, level->height);
	TEXTPRINTF(screen, font, 0, 30, TEXTPRINTF_COL, "Type: \"%d\"", level->type);
	TEXTPRINTF(screen, font, 0, 40, TEXTPRINTF_COL, "Tiles [0][4], [1][4]: \"%d, %d\"", 
		level->tiles[0][4], level->tiles[1][4]);

	for (y = 0; y < level->height; y++) {
		for (x = 0; x < level->width; x++) {
			putpixel(screen, x, y + 60, level->tiles[y][x] * 10000);
		}
	}
	
	PAUSE;
*/
}



// Find the player and enemies and initialize the data
void initialize_player (LEVEL_TYPE *level, PLAYER_TYPE *player, WINDOW_TYPE *window,
						CELL_DATA_TYPE *cell_data)
{
	int row, col, which_spring;

	for (row = 0; row < level->height; row++) {
		for (col = 0; col < level->width; col++) {
			if (level->tiles[row][col] == cell_data->player_index) {
				
				// Initialize the player's data
				player->x_cell = col;
				player->y_cell = row;
				player->x_offset = 0;
				player->y_offset = CELL_SIZE * 2 - player->height;
				player->x_velocity = 0;
				player->y_velocity = 0;
				player->max_x_velocity = 2;
				player->max_y_velocity = 2;
				player->max_swim_x_velocity = 1;
				player->max_swim_y_velocity = 1;
				player->facing = FACING_FRONT;
				player->sprite_index = cell_data->player_index;
				player->is_blinking = FALSE;
				player->animation_counter = 0;
				player->animation_frame = 0;
				player->is_jumping = FALSE;
				player->is_falling = FALSE;
				player->is_on_ice = FALSE;
				player->is_in_cannon = FALSE;
				player->cannon_angle = 0;
				player->cannon_x_cell = -1;
				player->cannon_y_cell = -1;
				player->conveyed_left = FALSE;
				player->conveyed_right = FALSE;
				player->is_swimming = FALSE;
				player->in_waterfall = FALSE;
				player->max_o2 = player->max_health;
				player->throw_angle = 32;	
				
				// Init the springboards
				for (which_spring = 0; which_spring < MAX_SPRINGBOARDS; which_spring++) {
					cell_data->spring_active[which_spring] = FALSE;
				}

				// Initialize the window data
				window->x_cell = player->x_cell - ((SCREEN_W / CELL_SIZE) / 2);
				if (window->x_cell < 0) window->x_cell = 0;
				if (window->x_cell > level->width - (SCREEN_W / CELL_SIZE))
					window->x_cell = level->width - (SCREEN_W / CELL_SIZE);
				window->y_cell = player->y_cell - ((SCREEN_H / CELL_SIZE) / 2);
				if (window->y_cell < 0) window->y_cell = 0;
				if (window->y_cell > level->height - (SCREEN_H / CELL_SIZE))
					window->y_cell = level->height - (SCREEN_H / CELL_SIZE);
				window->x_offset = 0;
				window->y_offset = 0;

				// Change the player's cells to empty spaces
				level->tiles[row][col] = cell_data->empty_cell_index;
				level->tiles[row + 1][col] = cell_data->empty_cell_index;

				// Init the cell data
				cell_data->animation_counter = 0;			
			}
		}
	}
}



// Load the enemy data for an enemy data type from the level .inf file
void load_enemy_data (int which_enemy, FILE *fp_in, ENEMY_DATA_TYPE enemy_data[])
{
	char str[2000];
	int val[4];
	
	// Read one line of text
	fgets(str, 2000, fp_in);	
	
	// Parse the line of text into our values
	sscanf(str, "%d, %d, %d, %d", &val[0], &val[1], &val[2], &val[3]);

	// Copy the values into the enemy object
	enemy_data[which_enemy].max_health = val[0];
	enemy_data[which_enemy].max_x_velocity = val[1];
	enemy_data[which_enemy].max_y_velocity = val[2];
	enemy_data[which_enemy].player_damage = val[3];
}



// Initializes an enemy at cell (x_cell, y_cell) in the enemies array
void start_enemy (int x_cell, int y_cell, int cell_index, ENEMY_TYPE enemies[],
				  ENEMY_DATA_TYPE enemy_data[], CELL_DATA_TYPE *cell_data)
{
	int counter, which_enemy_data, which_enemy_active;
	
	// Find which enemy in the level data to copy data from, to the new active enemy
	which_enemy_data = -1;
	for (counter = 0; counter < NUM_ENEMY_DATA; counter++) {
		if (cell_data->enemy_index[counter] == cell_index) {
			which_enemy_data = counter;
			break;
		}
	}
	
	// If an enemy does not exist at the given cell
	if (which_enemy_data == -1) {
		TEXTPRINTF(screen, font, 0, 0, TEXTPRINTF_COL, "No enemy of sprite index %d exists at cell (%d, %d).",
			cell_index, x_cell, y_cell);
		PAUSE;
		exit(0);
	}

	// Find an empty entry in the enemies array
	for (which_enemy_active = 0; which_enemy_active < MAX_ACTIVE_ENEMIES; which_enemy_active++) {
		if (enemies[which_enemy_active].active == FALSE) break;
	}

	// If an inactive entry in the enemies array could not be found
	if (which_enemy_active == MAX_ACTIVE_ENEMIES - 1) {
		TEXTPRINTF(screen, font, 0, 0, TEXTPRINTF_COL,
			"No inactive enemy index could be found (out of %d).", MAX_ACTIVE_ENEMIES);
		exit(0);
	}

	enemies[which_enemy_active].active = TRUE;
	enemies[which_enemy_active].prev_x_cell = x_cell;		// Save the enemy's starting cell
	enemies[which_enemy_active].prev_y_cell = y_cell;
	enemies[which_enemy_active].x_cell = x_cell;
	enemies[which_enemy_active].y_cell = y_cell;
	enemies[which_enemy_active].x_offset = 0;
	enemies[which_enemy_active].x_velocity = 0;
	enemies[which_enemy_active].y_velocity = 0;
	enemies[which_enemy_active].animation_counter = 0;
	enemies[which_enemy_active].animation_frame = 0;
	enemies[which_enemy_active].blinking = FALSE;
	enemies[which_enemy_active].has_item = 0;
	enemies[which_enemy_active].is_falling = FALSE;
	enemies[which_enemy_active].is_jumping = FALSE;
	enemies[which_enemy_active].enemy_data_type = which_enemy_data;
	enemies[which_enemy_active].facing = FACING_RIGHT;

	enemies[which_enemy_active].health = enemy_data[which_enemy_data].max_health;

	// Determine the enemy's y offset
	if (enemy_data[which_enemy_data].height <= CELL_SIZE)
		enemies[which_enemy_active].y_offset =
			CELL_SIZE * 1 - enemy_data[which_enemy_data].height;
	else if (enemy_data[which_enemy_data].height <= CELL_SIZE * 2) {
		enemies[which_enemy_active].y_offset =
			CELL_SIZE * 2 - enemy_data[which_enemy_data].height;
	}
	else if (enemy_data[which_enemy_data].height <= CELL_SIZE * 3) {
		enemies[which_enemy_active].y_offset =
			CELL_SIZE * 3 - enemy_data[which_enemy_data].height;
	}

	// Determine the enemy's initial velocity
	switch (which_enemy_data) {
	case ENEMY_DATA_TYPE_BIRD:
	case ENEMY_DATA_TYPE_ICE_BEAR:
	case ENEMY_DATA_TYPE_RED:
		enemies[which_enemy_active].x_velocity =
			enemy_data[enemies[which_enemy_active].enemy_data_type].max_x_velocity;
		// if (enemies[which_enemy_active].facing == FACING_LEFT)
		//	enemies[which_enemy_active].x_velocity *= -1;
		enemies[which_enemy_active].y_velocity = FORCE_GRAVITY;
		break;
	default:;
	}
}



// Searches a rectangle of cells for enemies. If any are found, their cells of origin are
// cleared and the appropriate entry in the enemies array is activated.
void scan_area_for_enemies (int x1_cell, int y1_cell, int x2_cell, int y2_cell, 
							LEVEL_TYPE *level, ENEMY_TYPE *enemies, ENEMY_DATA_TYPE enemy_data[],
							CELL_DATA_TYPE *cell_data)
{
	int x_cell, y_cell;
	int cell_index;

	for (y_cell = y1_cell; y_cell <= y2_cell; y_cell++) {
		for (x_cell = x1_cell; x_cell <= x2_cell; x_cell++) {
			cell_index = get_cell(level, cell_data, x_cell, y_cell);//level->tiles[y_cell][x_cell];
			if (cell_data->cell_types[cell_index] >= TYPE_ENEMY) {				
				start_enemy(x_cell, y_cell, cell_index, enemies, enemy_data, cell_data);
				set_cell(level, cell_data, x_cell, y_cell, cell_data->empty_cell_index);	// Clear
			}
		}
	}
}



// Update the animation of the individual cells
void update_animation (BITMAP *sprites[], RLE_SPRITE *rle_sprites[], char is_rle_sprite[],
					   LEVEL_TYPE *level, PLAYER_TYPE *player, CELL_DATA_TYPE *cell_data,
					   ANIMATION_TYPE animations[])
{
	int x_cell, y_cell;
	double x_offset, y_offset;
	int which_spring;
	int which_ani;

	// Increment the animation counter, which is used to determine which frame to draw
	// the cells in draw_screen
	cell_data->animation_counter++;
	if (cell_data->animation_counter >= 2 * 3 * 31 * 10)
		cell_data->animation_counter = 0;

	// Move the springs along
	for (which_spring = 0; which_spring < MAX_SPRINGBOARDS; which_spring++) {
		if (cell_data->spring_active[which_spring]) {
			cell_data->spring_counter[which_spring]++;
			if (cell_data->spring_counter[which_spring] == SPRING_SWITCH_FRAMES) {
				
				// Switch to the next tile if necessary
				level->tiles[cell_data->spring_y_cell[which_spring]]
							[cell_data->spring_x_cell[which_spring]]
						= cell_data->spring_index[cell_data->spring_direction[which_spring]] + 2;
			}
			else if (cell_data->spring_counter[which_spring] == SPRING_SWITCH_FRAMES * 2) {
				
				// Switch to the next tile if necessary
				level->tiles[cell_data->spring_y_cell[which_spring]]
							[cell_data->spring_x_cell[which_spring]]
						= cell_data->spring_index[cell_data->spring_direction[which_spring]] + 1;
			}

			// If it's time to reset the spring
			if (cell_data->spring_counter[which_spring] == SPRING_RESET_FRAMES) {
				
				// Reset the tile
				level->tiles[cell_data->spring_y_cell[which_spring]]
							[cell_data->spring_x_cell[which_spring]] =
					cell_data->spring_index[cell_data->spring_direction[which_spring]];
				
				// Mark the spring as inactive
				cell_data->spring_active[which_spring] = FALSE;
			}	
		}
	}

	// Generate bubbles if necessary (DISABLE FOR NOW--BUGGY!)
	if (player->is_swimming == TRUE && rand() % BUBBLE_FREQUENCY < 2 && FALSE) {
		
		// Position the bubble
		if (player->swim_angle >= 0 && player->swim_angle <= 64) {
			x_cell = player->x_cell;
			y_cell = player->y_cell + 1;
			x_offset = player->x_offset - player->height / 2;
			y_offset = player->y_offset;
		}
		else if (player->swim_angle > 64 && player->swim_angle <= 128) {
			x_cell = player->x_cell;
			y_cell = player->y_cell;
			x_offset = player->x_offset - player->height / 2;
			y_offset = player->y_offset - player->height / 2;
		}
		if (player->swim_angle > 128 && player->swim_angle <= 192) {
			x_cell = player->x_cell + 1;
			y_cell = player->y_cell;
			x_offset = player->x_offset;
			y_offset = player->y_offset;
		}
		if (player->swim_angle > 192 && player->swim_angle <= 255) {
			x_cell = player->x_cell + 1;
			y_cell = player->y_cell + 1;
			x_offset = player->x_offset;
			y_offset = player->y_offset;
		}
		
		x_offset -= rle_sprites[cell_data->bubble_index]->w / 2;

		add_animation (x_cell, y_cell, x_offset, y_offset, 
			cell_data->bubble_index + (rand() % 2), 10, -1, 1, 0, 0, 0, animations,
			sprites, rle_sprites, is_rle_sprite);
	}
	
	for (which_ani = 0; which_ani < MAX_ANIMATIONS; which_ani++) {		// Go thru each ani
		
		if (animations[which_ani].active == TRUE)						// If it is active
			animations[which_ani].iterations++;							// Increment iterations
		
		if (animations[which_ani].num_iterations > 0 && animations[which_ani].iterations >=
			animations[which_ani].num_iterations) {		// If the animation's time is up
			animations[which_ani].active = FALSE;		// End the animation
			
			// If the animation is of a switch block
			if (animations[which_ani].cell_index == cell_data->switch_block_index ||
				animations[which_ani].cell_index == cell_data->switch_block_index + 1 ||
				animations[which_ani].cell_index == cell_data->switch_block_index + 2) {
				
				// Replace the switch block
				level->tiles[animations[which_ani].y_cell][animations[which_ani].x_cell] =
					animations[which_ani].cell_index;
			}
		}
	}
}



// Returns the animation frame of the cell type given
int get_animation_frame (int which_cell, CELL_DATA_TYPE *cell_data)
{
	int slow_factor = 10;
	
	int num_frames = cell_data->num_frames[which_cell];		// Get the number of frames

	if (which_cell == cell_data->waterfall_index)
		slow_factor = 1;
	else if (which_cell == cell_data->water_index)
		slow_factor = 3;
	else if (which_cell == cell_data->conveyor_index[0] ||
			 which_cell == cell_data->conveyor_index[1] ||
			 which_cell == cell_data->conveyor_index[2] ||
			 which_cell == cell_data->conveyor_index[3] ||
			 which_cell == cell_data->conveyor_index[4] ||
			 which_cell == cell_data->conveyor_index[5])
		slow_factor = 3;

	// If the cell is any of the springboard sprites
	if (which_cell == cell_data->spring_index[0] || 
		which_cell == cell_data->spring_index[1] || 
		which_cell == cell_data->spring_index[2] || 
		which_cell == cell_data->spring_index[3]) {

		return 0;		// Use the first frame
	}
	
	return (cell_data->animation_counter / slow_factor % num_frames);
}



// Draws one screen of the level
void draw_screen (BITMAP **buffer, BITMAP **status_bar_bmp, RLE_SPRITE **background_image, 
				  BITMAP *sprites[], RLE_SPRITE *rle_sprites[], LEVEL_TYPE *level,
				  PLAYER_TYPE *player, ENEMY_TYPE enemies[], ENEMY_DATA_TYPE enemy_data[],
				  WINDOW_TYPE *window, CELL_DATA_TYPE *cell_data, ANIMATION_TYPE animations[],
				  KEYS_TYPE *keysdown, char is_rle_sprite[])
{
	int x, y;
	int row, col;
	int pass;
	int which_cell;
	int which_frame;
	int which_angle;		// Used for drawing the cannons, if necessary
	char is_foreground;		// Flag that indicates whether a sprite goes in the foreground or not

	// Paste the background image
	x = -(60 * (window->x_cell * CELL_SIZE + window->x_offset) / 
		(1 * (level->width - (SCREEN_W / CELL_SIZE)) * CELL_SIZE));
	y = -(60 * (window->y_cell * CELL_SIZE + window->y_offset) / 
		(1 * (level->height - (SCREEN_H / CELL_SIZE)) * CELL_SIZE));
	draw_rle_sprite (*buffer, *background_image, x, y);

	for (pass = 0; pass < 3; pass++) {						// Make three passes
		if (pass == 1)										// On the second pass
		{									
			// If the player is not on the invisible phase of blinking, or in a cannon
			if ((player->is_blinking != TRUE || player->blink_counter % 2 == 0) &&
				player->is_in_cannon == FALSE)
			{
				draw_player (buffer, sprites, rle_sprites, player,			// Draw the player
					window, cell_data, keysdown, is_rle_sprite);
			}

			draw_enemies(buffer, sprites, window, enemies, enemy_data);		// Draw the enemies
		} 
		else	// On the first or third passes...
		{
			// Examine each row and column to draw the cells
			for (row = 0; row < SCREEN_H / CELL_SIZE + 1; row++) {
				for (col = 0; col < SCREEN_W / CELL_SIZE + 1; col++)
				{	
					// Get the cell
					which_cell = get_cell(level, cell_data, col + window->x_cell, 
						row + window->y_cell);
					
					// Start an enemy if one exists at the cell
					if (cell_data->cell_types[which_cell] >= TYPE_ENEMY) {
						start_enemy(col + window->x_cell, row + window->y_cell, which_cell,
							enemies, enemy_data, cell_data);
						level->tiles[row + window->y_cell][col + window->x_cell] =
							cell_data->empty_cell_index;
						continue;
					}

					// Determine whether the sprite gets drawn in the foreground or not
					if (cell_data->cell_types[which_cell] == TYPE_FOREGROUND || 
						which_cell == cell_data->lava_index[0] || 
						which_cell == cell_data->lava_index[0] + 3 ||
						which_cell == cell_data->lava_index[0] + 4 ||
						which_cell == cell_data->cannon_index) {
						is_foreground = TRUE;
					} else is_foreground = FALSE;
							
					// If we're drawing cells and not the player this pass around
					if ((pass == 0 && is_foreground == FALSE) || 
						(pass == 2 && is_foreground == TRUE)) {
					
						// Position each tile
						x = col * CELL_SIZE - window->x_offset;
						y = row * CELL_SIZE - window->y_offset;
					
						// If the tile is within the bounds of the level
						if (col + window->x_cell >= 0 && row + window->y_cell >= 0 && 
							col + window->x_cell < level->width && 
							row + window->y_cell < level->height) {
							
							// Get the animation frame for this type of cell
							which_frame = get_animation_frame (which_cell, cell_data);
							
							// Draw the tile
							draw_rle_sprite(*buffer, rle_sprites[which_cell + which_frame], 
								x, y);
		
							// Draw the cannon barrel if necessary
							if (which_cell == cell_data->cannon_index) {
								// If the player is in this cannon
								if (player->is_in_cannon == TRUE && player->cannon_x_cell == 
									col + window->x_cell && player->cannon_y_cell == row +
									window->y_cell) {
									
									// Draw the player in it
									rotate_sprite(*buffer, sprites[cell_data->
										player_in_cannon_index], x + 0, y - 36,
										itofix(player->cannon_angle + 1));
								}
								if (player->cannon_x_cell == col + window->x_cell &&
									player->cannon_y_cell == row + window->y_cell)
									which_angle = player->cannon_angle;
								else
									which_angle = 0;
								rotate_sprite(*buffer, sprites[which_cell + 1], x - 4, y - 10,
									itofix(which_angle));
							}
						}
					}
				}
			}
		}
	}

	// Draw the animations
	draw_animations(buffer, sprites, rle_sprites, player, window, cell_data, animations, 
		is_rle_sprite);

	// Draw the status bar
	draw_sprite(*buffer, *status_bar_bmp, 0, 0);
	
	// Draw some stats
/*
	TEXTPRINTF(*buffer, font, 0, 0, TEXTPRINTF_COL, "Bombs left: %d", player->num_projectiles);
	TEXTPRINTF(*buffer, font, 0, 0, TEXTPRINTF_COL, "x_cell: %3d", player->x_cell);
	TEXTPRINTF(*buffer, font, 0, 10, TEXTPRINTF_COL, "y_cell: %3d", player->y_cell);
	TEXTPRINTF(*buffer, font, 0, 20, TEXTPRINTF_COL, "x_offset: %3d", (int)player->x_offset);
	TEXTPRINTF(*buffer, font, 0, 30, TEXTPRINTF_COL, "y_offset: %3d", (int)player->y_offset);
	TEXTPRINTF(*buffer, font, 0, 40, TEXTPRINTF_COL, "x_velocity: %3.2f", player->x_velocity);
	TEXTPRINTF(*buffer, font, 0, 50, TEXTPRINTF_COL, "y_velocity: %3.2f", player->y_velocity);
	TEXTPRINTF(*buffer, font, 0, 60, TEXTPRINTF_COL, "width: %3d", (int)player->width);
	TEXTPRINTF(*buffer, font, 0, 70, TEXTPRINTF_COL, "height: %3d", (int)player->height);
	TEXTPRINTF(*buffer, font, 0, 80, TEXTPRINTF_COL, "facing: %3d", (int)player->facing);
*/
}



// Draw the player onto the onscreen buffer
void draw_player(BITMAP **buffer, BITMAP *sprites[], RLE_SPRITE *rle_sprites[], PLAYER_TYPE *player,
				 WINDOW_TYPE *window, CELL_DATA_TYPE *cell_data, KEYS_TYPE *keysdown, 
				 char is_rle_sprite[])
{
	int x, y;
	int x_add_crosshair;
	int cell_index;

	// Draw the player
	x = get_x_pixel(player->x_cell, player->x_offset, window) - player->left_buffer;
	y = get_y_pixel(player->y_cell, player->y_offset, window) - player->top_buffer;
				
	if (player->is_swimming)
	{
		if (player->swim_angle >= 128)
		{
			rotate_sprite(*buffer, sprites[cell_data->swim_index +
			player->animation_frame], x, y, itofix(player->swim_angle + 128));
		}
		else
		{
			rotate_sprite_v_flip(*buffer, sprites[cell_data->swim_index +
			player->animation_frame], x, y, itofix(player->swim_angle));				
		}
	}
	else	// (Player is not swimming)
	{
		if (is_rle_sprite[cell_data->player_index +	player->animation_frame] == TRUE)
		{
			draw_rle_sprite(*buffer, rle_sprites[cell_data->player_index +
				player->animation_frame], x, y);
		}
		else {
			draw_sprite(*buffer, sprites[cell_data->player_index + 
				player->animation_frame], x, y);
		}
	}

	// If the player can throw bombs
	if (player->has_item == ITEM_THROW_BOMBS && player->throw_prepared == TRUE)
	{	
		// Position the crosshair
		x_add_crosshair = player->width;
		cell_index = cell_data->crosshair_angles_index + 1;

		if (player->facing == FACING_RIGHT)
		{
			// Draw the crosshair pointer
			pivot_sprite(*buffer, sprites[cell_data->crosshair_angles_index],
				x + x_add_crosshair, y + 10, 7, 40, itofix(player->throw_angle));
		}
		else if (player->facing == FACING_LEFT)
		{
			// Draw the crosshair pointer
			pivot_sprite(*buffer, sprites[cell_data->crosshair_angles_index],
				x + x_add_crosshair, y + 10, 7, 40, itofix(256 - player->throw_angle));
		}

	}
}



// Draw the enemies onto the onscreen buffer
void draw_enemies(BITMAP **buffer, BITMAP *sprites[], WINDOW_TYPE *window, ENEMY_TYPE enemies[],
				  ENEMY_DATA_TYPE enemy_data[])
{
	int which_enemy;
	int x, y;

	for (which_enemy = 0; which_enemy < MAX_ACTIVE_ENEMIES; which_enemy++) {
		if (enemies[which_enemy].active == TRUE) {
			
			// Position the enemy
			x = get_x_pixel(enemies[which_enemy].x_cell,
				enemies[which_enemy].x_offset, window);
			y = get_y_pixel(enemies[which_enemy].y_cell,
				enemies[which_enemy].y_offset, window);

			// Move the enemy lower if necessary
			if (enemies[which_enemy].enemy_data_type == ENEMY_DATA_TYPE_RED) y++;
			if (enemies[which_enemy].enemy_data_type == ENEMY_DATA_TYPE_ICE_BEAR) y += 2;
		
			// Draw the enemy
			if (enemies[which_enemy].facing == FACING_RIGHT) {
				draw_sprite_h_flip (*buffer, sprites[enemy_data[
					enemies[which_enemy].enemy_data_type].cell_index + 
					enemies[which_enemy].animation_frame], x, y);
			} else {
				draw_sprite (*buffer, sprites[enemy_data[
					enemies[which_enemy].enemy_data_type].cell_index +
					enemies[which_enemy].animation_frame], x, y);
			}
		}
	}
}



// Draw the animations onto the onscreen buffer
void draw_animations (BITMAP **buffer, BITMAP *sprites[], RLE_SPRITE *rle_sprites[], 
					  PLAYER_TYPE *player, WINDOW_TYPE *window, CELL_DATA_TYPE *cell_data, 
					  ANIMATION_TYPE animations[], char is_rle_sprite[])
{
	int which_ani;
	int which_cell;
	int x, y;

	for (which_ani = 0; which_ani < MAX_ANIMATIONS; which_ani++) {
		if (animations[which_ani].active == TRUE) {
			
			// Determine where to draw the animation
			x = get_x_pixel(animations[which_ani].x_cell, animations[which_ani].x_offset,
				window);
			y = get_y_pixel(animations[which_ani].y_cell, animations[which_ani].y_offset,
				window);
			
			which_cell = animations[which_ani].cell_index;	// Determine the sprite to draw

			// If we are to draw a number
			if (which_cell == cell_data->number_small_zero_index) {
				draw_small_numbers(x, y, animations[which_ani].tag,		// Draw small numbers
					buffer, rle_sprites, cell_data);
			} else if (which_cell == cell_data->smack_index + 1) {		// Drawing a cannon blast
				rotate_sprite(*buffer, sprites[cell_data->
					smack_index + 1], x, y,
					itofix(player->cannon_angle + 1));
			} else {
				// Draw an RLE sprite if appropriate
				if (is_rle_sprite[which_cell] == TRUE) {
					draw_rle_sprite (*buffer, rle_sprites[which_cell + 
						animations[which_ani].cur_frame], x, y);
				}
				else {
					// Draw a regular sprite (upside down if indicated) as appropriate
					if (animations[which_ani].tag == TAG_UPSIDE_DOWN) {
						draw_sprite_v_flip (*buffer, sprites[which_cell + 
							animations[which_ani].cur_frame], x, y);
					} else {
						draw_sprite (*buffer, sprites[which_cell], x, y);
					}
				}
			}
		}
	}
}



// Returns the x pixel coordinate on screen to draw an object, given its x cell and offset
int get_x_pixel (int x_cell, double x_offset, WINDOW_TYPE *window)
{
	return (x_cell * CELL_SIZE + (int) x_offset) - 
		   (window->x_cell * CELL_SIZE + (int) window->x_offset);
}



// Returns the y pixel coordinate on screen to draw an object, given its y cell and offset
int get_y_pixel (int y_cell, double y_offset, WINDOW_TYPE *window)
{
	return (y_cell * CELL_SIZE + (int) y_offset) - 
		   (window->y_cell * CELL_SIZE + (int) window->y_offset);
}



// Scans for certain keypresses. If the keys are pressed, execute the appropriate actions.
int check_input (BITMAP **buffer, BITMAP **status_bar_bmp, BITMAP *sprites[], 
				 RLE_SPRITE *rle_sprites[], char is_rle_sprite[],
				  LEVEL_TYPE *level, PLAYER_TYPE *player, WINDOW_TYPE *window,
				  KEYS_TYPE *keysdown, CELL_DATA_TYPE *cell_data, ANIMATION_TYPE animations[],
				  int *music_volume)
{
	char update_flag = FALSE;
	
	poll_keyboard();
	
	// "Hold down" keys - continuous action as long as keypress is there
	if (key[KEY_W])														// If W is pressed
	{
		if (*music_volume < 255)
		{
			*music_volume += 1;
			set_volume (*music_volume, 0);
		}
	}
	//	scroll_window(window, level, 0, -(int)player->max_y_velocity);	// Scroll window up
	if (key[KEY_S])														// If S is pressed
	{
		if (*music_volume > 0)
		{
			*music_volume -= 1;
			set_volume (*music_volume, 0);
		}
	}
	//	scroll_window(window, level, 0, (int)player->max_y_velocity);	// Scroll window down
	if (key[KEY_A])	{}													// If A is pressed
	//	scroll_window(window, level, -(int)player->max_x_velocity, 0);	// Scroll window left
	if (key[KEY_D])	{}													// If D is pressed
	//	scroll_window(window, level, (int)player->max_x_velocity, 0);	// Scroll window right

	if (key[KEY_UP])	// Up arrow key
	{					
		if (player->is_in_cannon == TRUE)
		{
			if (player->cannon_angle > 0 && player->cannon_angle <= 128) player->cannon_angle--;
			else if (player->cannon_angle != 0) player->cannon_angle++;
		}
		else if (player->is_swimming)
		{
			player->y_velocity -= FORCE_SWIM;
			player->facing = FACING_UP;

			if (player->y_velocity < -player->max_swim_y_velocity)	// Keep velocity in bounds
				player->y_velocity = -player->max_swim_y_velocity;		
			
			if (player->swim_angle < 128) player->swim_angle++;
			else if (player->swim_angle > 128) player->swim_angle--;
		}
		else if (player->has_item == ITEM_THROW_BOMBS)
		{
			player->throw_angle--;
			if (player->throw_angle < 0) player->throw_angle = 0;
		}

		keysdown->up_arrow = TRUE;
	}
	else if (key[KEY_DOWN])	// Down arrow key
	{
		if (player->is_in_cannon == TRUE)
		{
			if (cell_data->cell_types[level->tiles[player->y_cell + 2][player->x_cell]] != TYPE_EMPTY)
			{
				if (player->cannon_angle >= 0 && player->cannon_angle < 64) player->cannon_angle++;
				else if (player->cannon_angle > 192) player->cannon_angle--;
			}
			else
			{
				if (player->cannon_angle >= 0 && player->cannon_angle < 128) player->cannon_angle++;
				else if (player->cannon_angle != 128) player->cannon_angle--;
			}
		} 
		else if (player->is_swimming)
		{
			player->y_velocity += FORCE_SWIM;
			player->facing = FACING_DOWN;
			
			if (player->y_velocity > player->max_swim_y_velocity)	// Keep velocity in bounds
				player->y_velocity = player->max_swim_y_velocity;
			
			if (player->swim_angle >= 128 && player->swim_angle != 0) player->swim_angle++;
			else if (player->swim_angle < 128 && player->swim_angle != 0) player->swim_angle--;
		}
		else if (player->has_item == ITEM_THROW_BOMBS)
		{
			player->throw_angle++;
			if (player->throw_angle > 80) player->throw_angle = 80;
		}

		keysdown->down_arrow = TRUE;
	}
	else
	{
		keysdown->up_arrow = FALSE;
		keysdown->down_arrow = FALSE;
	}
	
	if (key[KEY_RIGHT]) {						// Right arrow key
		
		if (player->is_in_cannon == TRUE) {
			if (player->cannon_angle < 64 || player->cannon_angle >= 192) player->cannon_angle++;
			else if (player->cannon_angle != 64) player->cannon_angle--;
		} else if (player->is_on_ice) {
			if (player->x_velocity < player->max_x_velocity)
				player->x_velocity += FORCE_WALK_ICE;
		} else if (player->is_swimming) {
			if (player->x_velocity < player->max_swim_x_velocity)
				player->x_velocity += FORCE_SWIM;
			if (player->swim_angle < 192 && player->swim_angle >= 64) player->swim_angle++;
			else if (player->swim_angle < 64 || player->swim_angle > 192) player->swim_angle--;
		} else {
			if (player->x_velocity < player->max_x_velocity)
				player->x_velocity += FORCE_WALK;
			else
				if (player->x_velocity < player->max_x_velocity + FORCE_WALK * 2)
					player->x_velocity = player->max_x_velocity;
		}
		
		player->facing = FACING_RIGHT;
		player->settle_counter = 0;
		keysdown->right_arrow = TRUE;
	
	} else if (key[KEY_LEFT]) {					// Left arrow key
	
		if (player->is_in_cannon == TRUE) {
			if (player->cannon_angle > 64 && player->cannon_angle < 192) player->cannon_angle++;
			else if (player->cannon_angle != 192) player->cannon_angle--;
		} else if (player->is_on_ice) {
			if (player->x_velocity > -player->max_x_velocity)
				player->x_velocity -= FORCE_WALK_ICE;
		} else if (player->is_swimming) {
			if (player->x_velocity > -player->max_swim_x_velocity)
				player->x_velocity -= FORCE_SWIM;
			if (player->swim_angle < 64 || player->swim_angle >= 192) player->swim_angle++;
			else if (player->swim_angle > 64 && player->swim_angle < 192) player->swim_angle--;
		} else {
			if (player->x_velocity > -player->max_x_velocity)
				player->x_velocity -= FORCE_WALK;
			else
				if (player->x_velocity > -player->max_x_velocity - FORCE_WALK * 2)
					player->x_velocity = -player->max_x_velocity;
		}
		
		player->facing = FACING_LEFT;
		player->settle_counter = 0;
		keysdown->left_arrow = TRUE;

	// If neither the left nor right arrow keys are pressed and the player is not moving fast
	} else {
		if (player->settle_counter < 1000)		// Increase the settle counter;
			player->settle_counter++;			// Make him want to settle

		keysdown->left_arrow = FALSE;			// Tell other functions that neither the right
		keysdown->right_arrow = FALSE;			//   nor the left arrow keys are down
	}

	if (player->swim_angle > 255) player->swim_angle -= 256;		// Keep the angles in bounds
	if (player->swim_angle < 0) player->swim_angle += 256;
	if (player->cannon_angle > 255) player->cannon_angle -= 256;
	if (player->cannon_angle < 0) player->cannon_angle += 256;

	if (player->is_swimming) {
		if (keysdown->up_arrow == TRUE && keysdown->left_arrow == TRUE) {
			if (player->swim_angle < 96) player->swim_angle++;
			else if (player->swim_angle > 96) player->swim_angle--;
		}
		else if (keysdown->up_arrow == TRUE && keysdown->right_arrow == TRUE) {
			if (player->swim_angle < 160) player->swim_angle++;
			else if (player->swim_angle > 160) player->swim_angle--;
		}
		else if (keysdown->down_arrow == TRUE && keysdown->left_arrow == TRUE) {
			if (player->swim_angle < 32) player->swim_angle++;
			else if (player->swim_angle > 32) player->swim_angle--;
		}
		else if (keysdown->down_arrow == TRUE && keysdown->right_arrow == TRUE) {
			if (player->swim_angle < 224) player->swim_angle++;
			else if (player->swim_angle > 224) player->swim_angle--;
		}
	}
	
	// "Repeat keypress" keys - action occurs once per keypress
	if (key[KEY_LCONTROL]) {				// If left ctrl key is pressed
		if (keysdown->lctrl == FALSE) {		// and it was previously up
			keysdown->lctrl = TRUE;			// Tell the function that it is now down
				
			if (player->is_in_cannon == TRUE) {
				player->is_in_cannon = FALSE;

				// Set the player's velocity based on the cannon angle
				player->x_velocity = FORCE_CANNON * cos(((player->cannon_angle - 64) % 256) * 
					PI / 128);
				player->y_velocity = FORCE_CANNON * sin(((player->cannon_angle - 64) % 256) *
					PI / 128);

				if (cell_data->cell_types[level->tiles[player->y_cell + 2][player->x_cell]] !=
					TYPE_EMPTY) {
					
					player->y_offset = CELL_SIZE * 2 - player->height;
				}

				// Add the explosion animation
				add_animation(player->cannon_x_cell, player->cannon_y_cell, 0, -36,
					cell_data->smack_index + 1, -1, 10, -1, player->cannon_angle, 0, 0, 
					animations, sprites, rle_sprites, is_rle_sprite);

				// player->x_offset += player->x_velocity;	// Move the player away from the cannon
				// player->y_offset += player->y_velocity;

				player->cannon_delay = 100;		// Wait 100 game loops before returning the cannon
				player->is_jumping = FALSE;		// Do not enable Ctrl key jump lengthening
			} 
			// If the player is not jumping or falling or swimming
			else if (player->is_jumping == FALSE && player->is_falling == FALSE && 
				player->is_swimming == FALSE && player->y_velocity >= 0) {
				
				player->y_velocity = -FORCE_JUMP;	// Send him upwards
				player->is_jumping = TRUE;			// We are now jumping
			}
		}
	}
	else keysdown->lctrl = FALSE;

	if (key[KEY_ALT]) {						// If left alt key is pressed
		if (keysdown->lalt == FALSE) {		// and it was previously up
			keysdown->lalt = TRUE;			// Tell the function that it is now down
		
			// If the player has bombs and is not swimming, prepare a throw
			if (player->has_item == ITEM_THROW_BOMBS && player->is_swimming != TRUE)
				player->throw_prepared = TRUE;
		}
	}
	else
	{
		keysdown->lalt = FALSE;
		if (player->throw_prepared == TRUE)
		{
			player->throw_prepared = FALSE;
			throw_projectile (status_bar_bmp, sprites, rle_sprites, is_rle_sprite, player, 
				cell_data, animations, window);
			
			if (player->has_item != ITEM_NONE)
				update_flag = TRUE - 1;
			else
				update_flag = TRUE;
		}
	}

	if (key[KEY_ESC]) {					// If esc key is pressed
//		if (keysdown->esc == FALSE) {		// and it was previously up
//			keysdown->esc = TRUE;			// Tell the function that it is now down
			exit(0);					// Esc key action
//		}
	}
	else keysdown->esc = FALSE;

	if (key[KEY_F2]) {
		save_bmp ("screenshot.bmp", *buffer, NULL);
	}

	return update_flag;
}



// Reacts to the data obtained by collision detection
int react_to_collision_detection (BITMAP **status_bar_bmp, BITMAP *sprites[], 
								  RLE_SPRITE *rle_sprites[], char is_rle_sprite[],
								  LEVEL_TYPE *level, WINDOW_TYPE *window, PLAYER_TYPE *player,
								  COLLISION_DETECTION_TYPE collision_data[2][4], 
								  KEYS_TYPE *keysdown, CELL_DATA_TYPE *cell_data,
								  ANIMATION_TYPE animations[])
{
	int x, y;
	int x_cell, y_cell;
	int which_direction, which_cell, which_type, which_spring;
	int cell_index;
	int cell_type_left, cell_type_right;	// Used in determining to skirt a brick
	int spring_direction;
	int direction;
	int update_flag = FALSE;				// Flags
	int swim_flag;							// See if the player is swimming before checking
	char spring_found;
	char active_spring_found;
	char hit_switch_block;

	// Set default attributes of the player
	if (player->y_velocity == 0) player->is_standing = TRUE; else player->is_standing = FALSE;
	if (player->y_velocity < 0); else player->is_jumping = FALSE;
	if (player->y_velocity > 0) player->is_falling = TRUE; else player->is_falling = FALSE;
	swim_flag = player->is_swimming;
	player->is_on_ice = FALSE;
	player->conveyed_left = FALSE;
	player->conveyed_right = FALSE;
	player->is_swimming = FALSE;
	player->in_waterfall = FALSE;
	
	// React to PRESENT STATE collision detection
	for (which_cell = 0; which_cell < collision_data[0][0].num_scanned_cells; which_cell++) {
		
		cell_index = collision_data[0][0].which_cell[which_cell];
		which_type = cell_data->cell_types[collision_data[0][0].which_cell[which_cell]];
		x_cell = collision_data[0][0].x_cell[which_cell];
		y_cell = collision_data[0][0].y_cell[which_cell];

		switch (which_type) {
		
		case TYPE_SQUARE:						// Solid cell
		case TYPE_ICE:							// Ice (solid) cell
		case TYPE_MOVING_RIGHT:					// Conveyor belts
		case TYPE_MOVING_LEFT:
			
			if (player->is_in_cannon) {}
			else if (x_cell == player->x_cell + 1) {
				player->x_offset = CELL_SIZE - player->width - 1;
				if (player->x_velocity > 0) player->x_velocity = 0;
			}	
			else if (x_cell == player->x_cell) {
				player->x_cell++;
				player->x_offset = 1;
				if (player->x_velocity < 0) player->x_velocity = 0;
			}
			break;
		
		case TYPE_ITEM:
			collect_item(x_cell, y_cell, which_type, level, player, cell_data, status_bar_bmp,
				rle_sprites, window);
			update_flag = TRUE;		// Update the status bar
			break;
		
		case TYPE_WATERFALL:
			player->in_waterfall = TRUE;
			break;

		case TYPE_WATER:
			if (y_cell == player->y_cell) {
				if (swim_flag == FALSE) {		// Player was not swimming before
					player->swim_angle = 128;	// Set the swim angle to vertical
												// Create a splash animation
					player->o2 = player->max_o2;	// Fill the player's o2 bar
					
					// Draw the o2 bar
					refresh_status_bar
						(FALSE, FALSE,
						FALSE, FALSE,
						FALSE, FALSE,
						TRUE, TRUE,
						FALSE, FALSE,
						status_bar_bmp, rle_sprites, player, window, cell_data);
				}
				player->is_swimming = TRUE;
			}
			break;
		
		case TYPE_HARM:
			if (cell_index == cell_data->e_index[0] || cell_index == cell_data->e_index[1])
				player->health -= DAMAGE_ELECTRICITY;
			else if (cell_index == cell_data->lava_index[0] || 
					 cell_index == cell_data->lava_index[1] ||
					 cell_index == cell_data->lava_index[2])
				player->health -= DAMAGE_LAVA;
			
			update_flag = TRUE;		// Update the status bar
			break;

		case TYPE_ACTION:
			
			// Determine if a spring was found and if so, which direction
			spring_found = FALSE;
			for (direction = 0; direction < 4; direction++) {
				if (cell_index == cell_data->spring_index[direction] ||
					cell_index == cell_data->spring_index[direction] + 1 ||
					cell_index == cell_data->spring_index[direction] + 2) {
						
					spring_direction = direction;
					spring_found = TRUE;
				}
			}
				
			// (1) If the player is in contact with a springboard of some kind
			// (2) And if the player is in the right range to activate the springboard
			if (spring_found &&
				(spring_direction == 0 &&
				(player->y_offset >= CELL_SIZE * 2 - player->height + 
					CELL_SIZE - SPRING_HEIGHT_1 || player->y_cell == y_cell - 1)) ||
				(spring_direction == 1 &&
				(player->y_offset <= SPRING_HEIGHT_1 &&	player->y_cell <= y_cell)) ||
				(spring_direction == 2 &&
				(player->x_offset >= CELL_SIZE - player->width + 
					CELL_SIZE - SPRING_HEIGHT_1 || player->x_cell == x_cell)) ||
				(spring_direction == 3 &&
				(player->x_offset <= SPRING_HEIGHT_1 && player->x_cell <= x_cell))) {
			
				// Find a springboard 
				active_spring_found = FALSE;
				for (which_spring = 0; which_spring < MAX_SPRINGBOARDS; which_spring++) {	
										
					// If we found an inactive spring
					if (cell_data->spring_active[which_spring] == FALSE ||
						(cell_data->spring_x_cell[which_spring] ==
						collision_data[0][0].x_cell[which_cell] &&
						cell_data->spring_y_cell[which_spring] ==
						collision_data[0][0].y_cell[which_cell])) {
						
						active_spring_found = TRUE;			// Use it
						break;
					}
				}

				if (active_spring_found == FALSE) exit(0);
					
				// Set the data for the springboard
				cell_data->spring_active[which_spring] = TRUE;
				cell_data->spring_direction[which_spring] = spring_direction;
				cell_data->spring_counter[which_spring] = 0;
				cell_data->spring_x_cell[which_spring] = x_cell;
				cell_data->spring_y_cell[which_spring] = y_cell;

				// Increment the spring tile
				level->tiles[y_cell][x_cell] = cell_data->spring_index[spring_direction] + 1;
					
				// If the player made contact with an upwards springboard
				if (cell_index == cell_data->spring_index[0] || 
					cell_index == cell_data->spring_index[0] + 1 || 
					cell_index == cell_data->spring_index[0] + 2) {
						
					// Send the player upwards
					player->y_velocity = -FORCE_SPRINGBOARD;
					player->is_falling = FALSE;
					player->is_jumping = FALSE;		// Do not allow Ctrl key jump raising
				}
				// If the player made contact with a downwards springboard
				else if (cell_index == cell_data->spring_index[1] || 
					cell_index == cell_data->spring_index[1] + 1 || 
					cell_index == cell_data->spring_index[1] + 2) {

					// Send the player downwards
					player->y_velocity = FORCE_SPRINGBOARD;
					player->is_falling = TRUE;
					player->is_jumping = FALSE;
				}
				// If the player made contact with a leftwards springboard
				else if (cell_index == cell_data->spring_index[2] || 
					cell_index == cell_data->spring_index[2] + 1 || 
					cell_index == cell_data->spring_index[2] + 2) {

					// Send the player leftwards
					player->x_velocity = -FORCE_SPRINGBOARD;
				}
				// If the player made contact with a rightwards springboard
				else if (cell_index == cell_data->spring_index[3] || 
					cell_index == cell_data->spring_index[3] + 1 || 
					cell_index == cell_data->spring_index[3] + 2) {
						
					// Send the player rightwards
					player->x_velocity = FORCE_SPRINGBOARD;	
				}
			}
		
		default:;
		}
	}
	
	// React to FUTURE STATE collision detection
	for (which_direction = 0; which_direction < 4; which_direction++) {
		for (which_cell = 0; which_cell < collision_data[1][which_direction].num_scanned_cells;
		which_cell++) {

			cell_index = collision_data[1][which_direction].which_cell[which_cell];
			which_type = cell_data->cell_types[cell_index];
			x_cell = collision_data[1][which_direction].x_cell[which_cell];
			y_cell = collision_data[1][which_direction].y_cell[which_cell];
			
			// Decide what to do based on the cell type
			switch (which_type) {
			case TYPE_EMPTY:						// Empty cell
			case TYPE_FOREGROUND:					// Cell that appears in the foreground
			case TYPE_CLIMBABLE:	break;			// Climbable background cell
			
			case TYPE_HARM:
				
				// Spikes
				if (collision_data[1][which_direction].which_cell[which_cell] == 
					cell_data->spikes_index[0]) {					// Upward pointing spikes
					
					if (which_direction == 1) {						// Player moving down
						player->health = 0;							// Kill the player
						return TRUE;
					} else if (which_direction == 0 || which_direction == 2 || 
						which_direction == 3) {						// Up, Left, right
						
						collision_data[0][0].allow_x_movement = FALSE;
						if ((player->x_velocity < 0 && which_direction == 2) ||
							(player->x_velocity > 0 && which_direction == 3)) 
						player->x_velocity = 0;			// Stop the player's horizontal movemnt
						player->animation_counter = 0;	// Reset the player to standing frame
					}
				}
				if (collision_data[1][which_direction].which_cell[which_cell] == 
					cell_data->spikes_index[1]) {					// Downward pointing spikes
					
					if (which_direction == 0) {						// Player moving up
						player->health = 0;							// Kill the player
						return TRUE;
					} else if (which_direction == 1 || which_direction == 2 || 
						which_direction == 3) {						// Down, Left, right
						
						collision_data[0][0].allow_x_movement = FALSE;
						if ((player->x_velocity < 0 && which_direction == 2) ||
							(player->x_velocity > 0 && which_direction == 3)) 
						player->x_velocity = 0;			// Stop the player's horizontal movemnt
						player->animation_counter = 0;	// Reset the player to standing frame
					}
				}
				else if (collision_data[1][which_direction].which_cell[which_cell] == 
					cell_data->spikes_index[2]) {					// Left pointing spikes
					
					if (which_direction == 3) {						// Player moving right
						player->health = 0;							// Kill the player
						return TRUE;
					} else if (which_direction == 0 || which_direction == 1 || 
						which_direction == 2) {						// Up, down, left
						
						collision_data[0][0].allow_y_movement = FALSE;
						if ((player->y_velocity < 0 && which_direction == 0) ||
							(player->y_velocity > 0 && which_direction == 1)) 
						player->y_velocity = 0;			// Stop the player's horizontal movemnt
						player->animation_counter = 0;	// Reset the player to standing frame
					}
				}
				else if (collision_data[1][which_direction].which_cell[which_cell] == 
					cell_data->spikes_index[3]) {					// Right pointing spikes
					
					if (which_direction == 2) {						// Player moving left
						player->health = 0;							// Kill the player
						return TRUE;
					} else if (which_direction == 0 || which_direction == 1 || 
						which_direction == 3) {						// Up, down, right
						
						collision_data[0][0].allow_y_movement = FALSE;
						if ((player->y_velocity < 0 && which_direction == 0) ||
							(player->y_velocity > 0 && which_direction == 1)) 
						player->y_velocity = 0;			// Stop the player's horizontal movemnt
						player->animation_counter = 0;	// Reset the player to standing frame
					}
				}

				break;
			
			case TYPE_ACTION:
				if (cell_index == cell_data->cannon_index) {		// If player hit a cannon
					if (player->is_in_cannon == FALSE)	{			// And wasn't already in one
						player->is_in_cannon = TRUE;				// Put him in one
						player->cannon_angle = 0;
						player->cannon_x_cell = x_cell;
						player->cannon_y_cell = y_cell;

						player->x_cell = x_cell;					// Reposition the player
						player->y_cell = y_cell - 1;
						player->x_offset = 9;
						player->y_offset = 10;
					}
				}

				break;
			
			case TYPE_SQUARE:						// Solid cell
			case TYPE_ICE:							// Ice (solid) cell
			case TYPE_MOVING_RIGHT:					// Conveyor belts
			case TYPE_MOVING_LEFT:
			case TYPE_PLATFORM:						// Platform-type cells
				
				if (which_direction == 0 && which_type != TYPE_PLATFORM) {		// UP
					
					if (collision_data[1][0].num_scanned_cells == 2) {
						cell_type_left = 
							cell_data->cell_types[collision_data[1][0].which_cell[0]];
						cell_type_right = 
							cell_data->cell_types[collision_data[1][0].which_cell[1]];
					}
					
					// If the player's head hit a switch block
					if (cell_index == cell_data->switch_block_index ||
						cell_index == cell_data->switch_block_index + 1 ||
						cell_index == cell_data->switch_block_index + 2)
					{
						// Activate the switch block
						activate_switch_block(level, cell_data, x_cell, y_cell, cell_index,
							animations, sprites, rle_sprites, is_rle_sprite);

						hit_switch_block = TRUE;
					} 
					else
					{
						hit_switch_block = FALSE;
					}
					
					// If the player's head hit one cell slightly on the left
					if (hit_switch_block == FALSE &&
						collision_data[1][0].num_scanned_cells == 2 && 
						!keysdown->right_arrow && (
						cell_type_left == TYPE_EMPTY || 
						cell_type_left == TYPE_ITEM || 
						cell_type_left == TYPE_FOREGROUND || 
						cell_type_left == TYPE_CLIMBABLE ) && (
						cell_type_right != TYPE_EMPTY && 
						cell_type_right != TYPE_ITEM && 
						cell_type_right != TYPE_FOREGROUND &&
						cell_type_right != TYPE_CLIMBABLE )) {
						
						player->x_offset--;				// Move the player one pixel to the left
						if (player->x_velocity > 0)		// If the player is moving to the right
							player->x_velocity = 0;		// Stop that movement
						
						collision_data[0][0].allow_y_movement = FALSE;		// Ban vertical mvt
						player->y_velocity -= FORCE_GRAVITY;				// Maintain velocity
					} 
					// If the player's head hit one cell slightly on the right
					else if (hit_switch_block == FALSE &&
						collision_data[1][0].num_scanned_cells == 2 &&
						!keysdown->left_arrow && (
			
						cell_type_right == TYPE_EMPTY || 
						cell_type_right == TYPE_ITEM ||
						cell_type_right == TYPE_FOREGROUND || 
						cell_type_right == TYPE_CLIMBABLE ) && (
						cell_type_left != TYPE_EMPTY && 
						cell_type_left != TYPE_ITEM && 
						cell_type_left != TYPE_FOREGROUND &&
						cell_type_left != TYPE_CLIMBABLE )) {
						
						player->x_offset++;				// Move the player one pixel to the right
						if (player->x_velocity < 0)		// If the player is moving to the left
							player->x_velocity = 0;		// Stop that movement

						collision_data[0][0].allow_y_movement = FALSE;		// Ban vertical movement
						player->y_velocity -= FORCE_GRAVITY;				// Maintain velocity
					}
					// If the player's head hit one solid cell
					else {
						player->is_standing = FALSE;		// The player is now not standing
						player->is_jumping = FALSE;			// The player is now not jumping
						player->is_falling = TRUE;			// The player is now not falling
						player->y_velocity = 0;				// Stop the player's movement
						collision_data[0][0].allow_y_movement = FALSE;
						player->animation_counter = 0;		// Reset the player to standing frame
					}
				} else if (which_direction == 1) {			// DOWN
					
					if (player->y_velocity >= 0) {			// If the player is not moving up
						player->is_standing = TRUE;			// The player is standing
						player->is_jumping = FALSE;			// The player is not jumping
						player->is_falling = FALSE;			// The player is not falling
						player->y_velocity = 0;				// Stop the player's vertical movement
						player->y_offset = CELL_SIZE * 2 - player->height;	// Place on ground
						collision_data[0][0].allow_y_movement = FALSE;	// Stop from falling
						
						// React to different types of cells
						if (which_type == TYPE_ICE)					player->is_on_ice = TRUE;
						else if (which_type == TYPE_MOVING_RIGHT)	player->conveyed_right = TRUE;
						else if (which_type == TYPE_MOVING_LEFT)	player->conveyed_left = TRUE;
					}

				} else if ((which_direction == 2 || which_direction == 3) &&   // LEFT or RIGHT
					which_type != TYPE_PLATFORM) {
										
				//	textprintf_right(screen, font, SCREEN_W - 1, 0, TEXTPRINTF_COL, "%d,%d",
				//		collision_data[2].cell_type[0], collision_data[2].cell_type[1]);
					
					collision_data[0][0].allow_x_movement = FALSE;

//					if ((player->x_velocity < 0 && which_direction == 2) ||
//						(player->x_velocity > 0 && which_direction == 3)) 
					
					player->x_velocity = 0;				// Stop the player's horizontal movement
					player->animation_counter = 0;		// Reset the player to standing animation frame
				}

				break;
			default: break;
			}
		}
	}
	
	// If the player was swimming
	if (swim_flag == TRUE)
	{	
		if (player->is_swimming == FALSE)						// If the player is not swimming
		{	
			if (keysdown->up_arrow == TRUE)						// If up arrow key is pressed
			{
				player->y_velocity = -FORCE_JUMP_FROM_WATER;	// Start a jump!
				player->is_jumping = FALSE;						// Don't allow jump raising
				player->facing = FACING_RIGHT;
			}
			
			// Erase the o2 bar
			x = window->lifebar_x;
			y = window->lifebar_y + rle_sprites[cell_data->lifebar_icon_index]->h + 4;
			rectfill(*status_bar_bmp, x, y, x + rle_sprites[cell_data->o2_icon_index]->w + 5 + 
				player->max_o2 + 1, y + rle_sprites[cell_data->o2_icon_index]->h,
				TRANSPARENT_COL);

			if (player->has_item != ITEM_NONE)
				update_flag = TRUE - 1;					// Indicates to re-draw the item bar icon
			else
				update_flag = TRUE;						// Indicates not to re-draw the item bar icon
			srand(cell_data->animation_counter);	// Seed the random number generator
		}
		// Reduce the o2 bar (or health, if the player is out of o2)
		else if (cell_data->animation_counter % O2_LOSS_SPEED == 0)
		{
			if (player->o2 > 0)
				player->o2--;	
			else
				player->health--;
			update_flag = TRUE;		// Update the status bar!
		}
	}

	if (player->y_cell >= level->height) {
		player->health -= 3;
		update_flag = TRUE;
	}
		
	return update_flag;		// Return, telling main whether or not to update the status bar

}



// Adds an animation to the array.
// Returns the index of the new active animation, or -1 on failure.
int add_animation (int x_cell, int y_cell, double x_offset, double y_offset, int cell_index, 
				   int frames_delay, int num_iterations, int num_frames, int tag,
				   double x_velocity, double y_velocity, ANIMATION_TYPE animations[],
				   BITMAP *sprites[], RLE_SPRITE *rle_sprites[], char is_rle_sprite[])
{
	int which_ani = 0;

	while (which_ani < MAX_ANIMATIONS) {
		if (animations[which_ani].active == FALSE) {
			animations[which_ani].active = TRUE;
			animations[which_ani].x_cell = x_cell;
			animations[which_ani].y_cell = y_cell;
			animations[which_ani].prev_x_cell = x_cell;
			animations[which_ani].prev_y_cell = y_cell;
			animations[which_ani].x_offset = x_offset;
			animations[which_ani].y_offset = y_offset;
			animations[which_ani].prev_x_offset = x_offset;
			animations[which_ani].prev_y_offset = y_offset;
			animations[which_ani].frames_delay = frames_delay;
			animations[which_ani].num_iterations = num_iterations;
			animations[which_ani].num_frames = num_frames;
			animations[which_ani].cell_index = cell_index;
			animations[which_ani].tag = tag;
			animations[which_ani].cur_cyle = 0;
			animations[which_ani].cur_frame = 0;
			animations[which_ani].iterations = 0;
			animations[which_ani].x_velocity = x_velocity;
			animations[which_ani].y_velocity = y_velocity;
			animations[which_ani].ani_counter = 0;
			
			if (is_rle_sprite[cell_index])
			{
				animations[which_ani].width = rle_sprites[cell_index]->w;
				animations[which_ani].height = rle_sprites[cell_index]->h;
			}
			else
			{
				animations[which_ani].width = sprites[cell_index]->w;
				animations[which_ani].height = sprites[cell_index]->h;
			}

			return which_ani;
		}
		which_ani++;
	}

	return -1;		// Could not find an empty slot
}



// Changes all of the dotted outline blocks to solid and vice versa, for a particular color.
void switch_solid_and_dotted_blocks (LEVEL_TYPE *level, CELL_DATA_TYPE *cell_data, 
									 int which_switch)
{
	int x, y;

	for (y = 0; y < level->height; y++) {
		for (x = 0; x < level->width; x++) {
			if (level->tiles[y][x] == cell_data->dotted_outline_index + which_switch)
				level->tiles[y][x] = cell_data->filled_block_index + which_switch;
			else if (level->tiles[y][x] == cell_data->filled_block_index + which_switch)
				level->tiles[y][x] = cell_data->dotted_outline_index + which_switch;
		}
	}

}



// Perform the animation for the player's death, and restart the level
void kill_player (PLAYER_TYPE *player)
{
	player->lives--;
	player->has_item = 0;
}



// Move the player.
// This function performs the appropriate actions if collision detection reports any.
// Returns whether or not to update the status bar
void move_player (LEVEL_TYPE *level, PLAYER_TYPE *player, ENEMY_TYPE enemies[], 
					  WINDOW_TYPE *window, KEYS_TYPE *keysdown,
					  COLLISION_DETECTION_TYPE collision_data[2][4], CELL_DATA_TYPE *cell_data)
{
//	int item_found, harm_found, water_found;
//	int pixels;
	float stop_factor;
	int animation_factor;
	int player_x_pixels, player_y_pixels, window_x_pixels, window_y_pixels;
	int x_add, y_add;		// For scrolling the window

	// If the player is in the cannon, scroll the window to center the cannon
	if (player->is_in_cannon == TRUE) {		
		player_x_pixels = player->x_cell * CELL_SIZE + player->x_offset;
		player_y_pixels = player->y_cell * CELL_SIZE + player->y_offset;
		window_x_pixels = window->x_cell * CELL_SIZE + window->x_offset;
		window_y_pixels = window->y_cell * CELL_SIZE + window->y_offset;
		
		if (player_x_pixels > window_x_pixels + SCREEN_W / 2 + player->width / 2) x_add = 1;
		else if (player_x_pixels < window_x_pixels + SCREEN_W / 2 + player->width / 2) x_add = -1;
		else x_add = 0;

		if (player_y_pixels > window_y_pixels + SCREEN_H / 2 + player->height / 2) y_add = 1;
		else if (player_y_pixels < window_y_pixels + SCREEN_H / 2 + player->height / 2) y_add = -1;
		else y_add = 0;

		if (x_add != 0 || y_add != 0) scroll_window(window, level, x_add, y_add);
		
		return;		// Do not move the player
	}
	
	if (collision_data[0][0].allow_x_movement == TRUE) {	// If horizontal movement is allowed
		player->x_offset += player->x_velocity;				// Move the player horizontally

		if (player->conveyed_right)				// If the player stands on a rightwards conveyor
			player->x_offset += FORCE_CONVEYOR;	// Move him to the right
		else if (player->conveyed_left)			// If the player stands on a leftwards conveyor
			player->x_offset -= FORCE_CONVEYOR;	// Move him to the left

		// If the player has moved far enough to the right
		if (player->x_cell * CELL_SIZE + (int)floor(player->x_offset) + player->width > 
			window->x_cell * CELL_SIZE + (int)floor(window->x_offset) +
			SCREEN_W - window->right_buffer) {
			
			// Scroll the viewing window to the right			
			scroll_window(window, level, 
				(player->x_cell * CELL_SIZE + (int)floor(player->x_offset) + player->width) -
				(window->x_cell * CELL_SIZE + (int)floor(window->x_offset) +
				(SCREEN_W - window->right_buffer)), 0);
		}

		// If the player has moved far enough to the left
		if (player->x_cell * CELL_SIZE + (int)floor(player->x_offset) < 
			window->x_cell * CELL_SIZE + (int)floor(window->x_offset) +
			window->left_buffer) {
			
			// Scroll the viewing window to the left
			scroll_window(window, level, 
				(player->x_cell * CELL_SIZE + (int)floor(player->x_offset)) -
				(window->x_cell * CELL_SIZE + (int)floor(window->x_offset) +
				window->left_buffer), 0);
		}
	}

	if (collision_data[0][0].allow_y_movement) {		// If vertical movement is allowed
		player->y_offset += player->y_velocity;			// Move the player vertically
		
		if (player->in_waterfall == TRUE)
			player->y_velocity += FORCE_WATERFALL;
		
		// If the player has moved far enough downwards
		if (player->y_cell * CELL_SIZE + (int)floor(player->y_offset) + player->height > 
			window->y_cell * CELL_SIZE + (int)floor(window->y_offset) +
			SCREEN_H - window->bottom_buffer) {
			
			// Scroll the viewing window downwards
			scroll_window(window, level, 0,
				(player->y_cell * CELL_SIZE + (int)floor(player->y_offset) + player->height) -
				(window->y_cell * CELL_SIZE + (int)floor(window->y_offset) +
				(SCREEN_H - window->bottom_buffer)));
		}

		// If the player has moved far enough upwards
		if (player->y_cell * CELL_SIZE + (int)floor(player->y_offset) < 
			window->y_cell * CELL_SIZE + (int)floor(window->y_offset) +
			window->top_buffer) {
			
			// Scroll the viewing window upwards
			scroll_window(window, level, 0, 
				(player->y_cell * CELL_SIZE + (int)floor(player->y_offset)) -
				(window->y_cell * CELL_SIZE + (int)floor(window->y_offset) +
				window->top_buffer));
		}		
	}

	
	// Keep the cell offset values in range, adjusting the cell values accordingly
	while (player->x_offset < 0) {
		player->x_offset += CELL_SIZE;
		player->x_cell--;
	}
	while (player->x_offset >= CELL_SIZE) {
		player->x_offset -= CELL_SIZE;
		player->x_cell++;
	}
	while (player->y_offset < 0) {
		player->y_offset += CELL_SIZE;
		player->y_cell--;
	}
	while (player->y_offset >= CELL_SIZE) {
		player->y_offset -= CELL_SIZE;
		player->y_cell++;
	}

	// Adjust the player's animation frame
	if (player->facing != FACING_FRONT && player->facing != FACING_BACK) {
		
		// If the player is facing to the right
		if (player->is_on_ice)
			player->animation_counter += 2;
		else if (player->is_swimming)
			player->animation_counter += 1;
		else
			player->animation_counter += abs((int) player->x_velocity);	// Increase the counter

		if (player->is_swimming) animation_factor = 30; else animation_factor = 15;

		switch ((int) player->animation_counter / animation_factor) {// Select the frame to use
		case 0: player->animation_frame = 1; break;
		case 1: player->animation_frame = 2; break;
		case 2: player->animation_frame = 3; break;
		case 3: player->animation_frame = 2; break;
		case 4: player->animation_frame = 1; break;
		case 5: player->animation_frame = 4; break;
		case 6: player->animation_frame = 5; break;
		case 7:
		default:
			player->animation_frame = 4; break;
		}

		while (player->animation_counter >= 8 * 15)		// Keep the counter between 0 and 120
			player->animation_counter -= 8 * 15;

		if (player->is_swimming) {
			
			// Adjust for only having three swimming animation frames
			if (player->animation_frame == 1) player->animation_frame = 0;
			if (player->animation_frame == 2) player->animation_frame = 1;
			if (player->animation_frame == 3) player->animation_frame = 2;
			if (player->animation_frame == 4) player->animation_frame = 1;
			if (player->animation_frame == 5) player->animation_frame = 2;

		} else {	
		
			if (player->settle_counter > 1)				// If the player has settled long enough
				player->animation_frame = 1;			// Give him the standing animation frame
			if (player->is_falling)						// If the player is falling,
				player->animation_frame = 3;			//   set the appropriate frame
			if (player->y_velocity < 0)					// If the player is jumping
				player->animation_frame = 6;			//   set the appropriate frame

			if (player->facing == FACING_RIGHT)			// Use a different set of frames if the
				player->animation_frame += 6;			//  player faces a different direction

			if (player->settle_counter >= 1000) {		// If the player has settled a long time
				if (player->x_velocity == 0 && player->y_velocity >= 0 && 
					player->y_velocity < 1) {
					
					player->animation_frame = 0;		// Give him the forward-facing frame
					player->facing = FACING_FRONT;
				} else {
					player->settle_counter = 0;
				}
			}
		}
	}

	// Adjust the player's velocity	
	if (player->is_swimming) {

		stop_factor = STOP_FACTOR_SWIMMING;

		if (keysdown->left_arrow == FALSE && keysdown->right_arrow == FALSE) {
			if (player->x_velocity >= stop_factor)			// And he is moving to the right
				player->x_velocity -= stop_factor;			// Slow him down
			else if (player->x_velocity <= -stop_factor)	// If he is moving to the left
				player->x_velocity += stop_factor;			// Slow him down
			else									// If he is not moving or moving slowly
				player->x_velocity = 0;				// Stop his horizontal movement
		}
		
		if (keysdown->up_arrow == FALSE && keysdown->down_arrow == FALSE) {
			if (player->y_velocity >= stop_factor)			// And he is moving down
				player->y_velocity -= stop_factor;			// Slow him down
			else if (player->y_velocity <= -stop_factor)	// If he is moving up
				player->y_velocity += stop_factor;			// Slow him down
			else									// If he is not moving or moving slowly
				player->y_velocity = 0;				// Stop his vertical movement
		}

		if (player->y_velocity > player->max_swim_y_velocity)
			player->y_velocity -= stop_factor * 4;
	
	} else if (keysdown->left_arrow == FALSE && keysdown->right_arrow == FALSE && 
		player->is_on_ice == FALSE) {
	
		// Determine how much to slow the player's velocity
		if (player->is_standing) stop_factor = STOP_FACTOR_WALKING; else stop_factor = 0;
		if (player->x_velocity >= stop_factor)			// And he is moving to the right
			player->x_velocity -= stop_factor;			// Slow him down
		else if (player->x_velocity <= -stop_factor)	// If he is moving to the left
			player->x_velocity += stop_factor;			// Slow him down
		else									// If he is not moving or moving slowly
			player->x_velocity = 0;				// Stop his horizontal movement
	}
	
	if (player->is_swimming == FALSE)				// If the player is not swimming
		player->y_velocity += FORCE_GRAVITY;		// Exert the force of gravity on him

	// If the player is jumping
	if (player->is_jumping && player->y_velocity >= -FORCE_JUMP && player->is_swimming == FALSE)
		if (!key[KEY_LCONTROL])							// And the left Ctrl key is not pressed
			player->y_velocity += FORCE_CONTINUE_JUMP;	// Make the player jump further

	if (player->is_blinking == TRUE) {			// If the player is blinking
		player->blink_counter--;				// Decrement the blink counter
		if (player->blink_counter <= 0) {		// If the blink counter has run out
			player->is_blinking = FALSE;		// Stop the blinking
		}
	}

	// If the player is not in the cannon but it hasn't been returned to an upright angle,
	// either decrement the counter or move the cannon upwards
	if (player->is_in_cannon == FALSE && player->cannon_angle != 0) {
		if (player->cannon_delay > 0) player->cannon_delay--;
		else if (player->cannon_angle <= 128) player->cannon_angle--; else player->cannon_angle++;
	}
}



// Move the enemies
int move_enemies (LEVEL_TYPE *level, PLAYER_TYPE *player, KEYS_TYPE *keysdown,
				   ENEMY_TYPE enemies[], ENEMY_DATA_TYPE enemy_data[], WINDOW_TYPE *window,
				   COLLISION_INPUT_TYPE *collision_input_data, 
				   COLLISION_DETECTION_TYPE collision_data[2][4], CELL_DATA_TYPE *cell_data,
				   ANIMATION_TYPE animations[], BITMAP *sprites[], RLE_SPRITE *rle_sprites[],
				   char is_rle_sprite[])
{
	int which_enemy;
	int which_direction;	// React to collision detection
	int which_cell;
	int cell_index;
	int which_type;
	int x_cell, y_cell;
	int enemy_x_pixels, enemy_y_pixels;
	int player_x_pixels, player_y_pixels;
	int num_rows_occupied, num_cols_occupied;
	double x_add, y_add;	// Determine the enemy's old and new positions for collision det.
	long x_pixels[2];		// Used to determine if the enemy is to the left or right of player
	long y_pixels[2];
	int facing_direction;
	int update_flag = FALSE;	// Flag to update the status bar
	char reverse_direction;		// Flag to reverse direction if necessary


	for (which_enemy = 0; which_enemy < MAX_ACTIVE_ENEMIES; which_enemy++) {
		if (enemies[which_enemy].active == TRUE) {

			// Perform collision detection on the cells that the enemy will soon occupy
			get_collision_input (collision_input_data, DETECT_ENEMY, which_enemy, keysdown,
				player, enemies, enemy_data, animations);
			
			detect_collisions (collision_input_data, level, player, enemies, enemy_data, 
				keysdown, cell_data, collision_data);

			reverse_direction = FALSE;		// Do not reverse horizontal direction by default
			
			// React to collision detection for the enemies
			// ALSO CHECK start_enemy() here!
			switch (enemies[which_enemy].enemy_data_type) {
			case ENEMY_DATA_TYPE_BIRD:		// Bird
			case ENEMY_DATA_TYPE_ICE_BEAR:	// Ice bear
			case ENEMY_DATA_TYPE_RED:		// Red walking guy
	
				// Calculate basic enemy data
				num_cols_occupied = enemy_data[enemies[which_enemy].enemy_data_type].width /
					CELL_SIZE;
				num_rows_occupied = enemy_data[enemies[which_enemy].enemy_data_type].height /
					CELL_SIZE;
				enemy_x_pixels = enemies[which_enemy].x_cell * CELL_SIZE +
					enemies[which_enemy].x_offset;
				enemy_y_pixels = enemies[which_enemy].y_cell * CELL_SIZE +
					enemies[which_enemy].y_offset;
				player_x_pixels = player->x_cell * CELL_SIZE + player->x_offset;
				player_y_pixels = player->y_cell * CELL_SIZE + player->y_offset;
				
				// React to FUTURE STATE collision detection of the enemy
				for (which_direction = 0; which_direction < 4; which_direction++) {
					for (which_cell = 0; which_cell < 
						collision_data[1][which_direction].num_scanned_cells; which_cell++) {
						
						// Calculate basic enemy data to be used later
						cell_index = collision_data[1][which_direction].which_cell[which_cell];
						which_type = cell_data->cell_types[cell_index];
						x_cell = collision_data[1][which_direction].x_cell[which_cell];
						y_cell = collision_data[1][which_direction].y_cell[which_cell];

						// Decide what to do based on the cell type
						switch (which_type) {
						case TYPE_EMPTY:						// Empty cell
						case TYPE_WATER:
						case TYPE_WATERFALL:
						case TYPE_ITEM:
						case TYPE_HARM:
							
							// If the enemy in question is not a bird, and no block is detected 
							// beneath it, reverse its horizontal direction
							if (enemies[which_enemy].enemy_data_type != ENEMY_DATA_TYPE_BIRD) {
								if (which_direction == 1) {			// Down direction collision

									// If the enemy is about to walk off the edge of a lower left
									// block
									if (enemies[which_enemy].facing == FACING_RIGHT && 
										which_cell == 1 && enemies[which_enemy].x_offset >= 8) {
										
										// If the lower left block is itself empty
										switch (cell_data->cell_types[collision_data[1][1].
											which_cell[0]]) {
										case TYPE_EMPTY:						// Empty cell
										case TYPE_WATER:
										case TYPE_WATERFALL:
										case TYPE_ITEM:
											break;
										default:
											// If the lower left block is not empty,
											// reverse the enemy's direction
											reverse_direction = TRUE;
										}
									}
							
									// If the enemy is about to walk off the edge of a lower
									// right block
									else if (enemies[which_enemy].facing == FACING_LEFT && 
										which_cell == 0 && enemies[which_enemy].x_offset <=
										enemy_data[enemies[which_enemy].enemy_data_type].width -
											8)
									{
										
										// If the lower right block is itself empty
										switch (cell_data->cell_types[collision_data[1][1].
											which_cell[1]]) {
										case TYPE_EMPTY:						// Empty cell
										case TYPE_WATER:
										case TYPE_WATERFALL:
										case TYPE_ITEM:
											break;
										default:
											// If the lower right block is not empty, reverse the
											// enemy's direction
											reverse_direction = TRUE;
										}
									}
							
								}
							}
							
							break;
						
						case TYPE_SQUARE:						// Solid cell
						case TYPE_ICE:							// Ice (solid) cell
						case TYPE_MOVING_RIGHT:					// Conveyor belts
						case TYPE_MOVING_LEFT:
						case TYPE_PLATFORM:						// Platform-type cells
							
							if ((which_direction == 0 && which_type != TYPE_PLATFORM) || 
								which_direction == 1)
							{								
								// Stop the enemy's movement
								collision_data[0][0].allow_y_movement = FALSE;
								enemies[which_enemy].y_velocity = 0;

								// Align the enemy with the nearest row of cells
								if (which_direction == 0) enemies[which_enemy].y_offset = 0;
								
								if (which_direction == 1)
								{
									enemies[which_enemy].y_cell = y_cell - num_rows_occupied;
									enemies[which_enemy].y_offset = 
										num_rows_occupied  * CELL_SIZE -
										enemy_data[enemies[which_enemy].enemy_data_type].height;
								}
							}
							if (which_direction == 2 || which_direction == 3)
								if (which_type != TYPE_PLATFORM) reverse_direction = TRUE;
							
							break;
						default:;
						}
					}
				}

				// Reverse horizontal direction if necessary
				if (reverse_direction == TRUE)
				{
					if (enemies[which_enemy].facing == FACING_LEFT ||
						enemies[which_enemy].x_velocity < 0)
					{	
						enemies[which_enemy].facing = FACING_RIGHT;
						enemies[which_enemy].x_velocity =
							enemy_data[enemies[which_enemy].enemy_data_type].max_x_velocity;
					}
					else if (enemies[which_enemy].facing == FACING_RIGHT ||
						enemies[which_enemy].x_velocity > 0)
					{
						enemies[which_enemy].facing = FACING_LEFT;
						enemies[which_enemy].x_velocity = -enemy_data[
							enemies[which_enemy].enemy_data_type].max_x_velocity;
					}
				}

				enemies[which_enemy].animation_counter++;			// Determine the enemy's
				enemies[which_enemy].animation_frame =				// animation
					enemies[which_enemy].animation_counter / 6 % 2;
				while (enemies[which_enemy].animation_counter >= 6 * 2 * 2)
					enemies[which_enemy].animation_counter -= 6 * 2 * 2;
			
				// If the enemy is a bird
				if (enemies[which_enemy].enemy_data_type == ENEMY_DATA_TYPE_BIRD)
				{	
					// Reverse the bird's direction if it is too far horizontally from the player
					if (enemies[which_enemy].facing == FACING_RIGHT && 
						enemy_x_pixels > player_x_pixels + CELL_SIZE * 4 &&
						rand() % 3 == 0)
					{
						enemies[which_enemy].x_velocity = 
							-abs((int)enemies[which_enemy].x_velocity);
						enemies[which_enemy].facing = FACING_LEFT;
					}						
					else if (enemies[which_enemy].facing == FACING_LEFT &&
						enemy_x_pixels + CELL_SIZE * 4 < player_x_pixels &&
						rand() % 3 == 0)
					{
						enemies[which_enemy].x_velocity =
							abs((int)enemies[which_enemy].x_velocity);
						enemies[which_enemy].facing = FACING_RIGHT;
					}						
					
					// If vertical movement is allowed (the enemy has not hit a solid surface)
					if (collision_data[0][0].allow_y_movement == TRUE)
					{
						if (enemy_y_pixels + CELL_SIZE * 4 < player_y_pixels && rand() % 1 == 0)
						{	
							// Accelerate the enemy downwards
							enemies[which_enemy].y_velocity += FORCE_BIRD;				

							// Limit its downwards speed
							if (enemies[which_enemy].y_velocity > MAX_BIRD_Y_VELOCITY)	
								enemies[which_enemy].y_velocity = MAX_BIRD_Y_VELOCITY;	
						}
						else if (enemy_y_pixels + CELL_SIZE * 4 > player_y_pixels && rand() % 1 == 0)
						{													
							// Accelerate the enemy upwards
							enemies[which_enemy].y_velocity -= FORCE_BIRD / 2;
							
							// Limit its upwards speed
							if (enemies[which_enemy].y_velocity < -MAX_BIRD_Y_VELOCITY)
								enemies[which_enemy].y_velocity = -MAX_BIRD_Y_VELOCITY;
						}
					}
				}
				else
				{
					// Accelerate the enemy downwards
					enemies[which_enemy].y_velocity += FORCE_GRAVITY;
				}

				break;
			default:;
			}
			
			// Determine how much/far to move the enemy
			if (collision_data[0][0].allow_x_movement == TRUE)
				x_add = enemies[which_enemy].x_velocity;
			else
				x_add = 0;

			if (collision_data[0][0].allow_y_movement == TRUE)
				y_add = enemies[which_enemy].y_velocity;
			else
				y_add = 0;

			// Determine if the enemy's new position will collide with that of the player
	/*		if ((rectangles_collide(player->x_cell, player->y_cell, player->x_offset,
				player->y_offset, player->width, player->height, enemies[which_enemy].x_cell,
				enemies[which_enemy].y_cell, enemies[which_enemy].x_offset, 
				enemies[which_enemy].y_offset, 
				enemy_data[enemies[which_enemy].enemy_data_type].width,
				enemy_data[enemies[which_enemy].enemy_data_type].height) == FALSE) &&
	*/			
			
			// Determine what happens if a collision occurs between the player and the enemy
			if	((rectangles_collide(player->x_cell, player->y_cell, player->x_offset,
				player->y_offset, player->width, player->height, enemies[which_enemy].x_cell,
				enemies[which_enemy].y_cell, enemies[which_enemy].x_offset + x_add, 
				enemies[which_enemy].y_offset + y_add, 
				enemy_data[enemies[which_enemy].enemy_data_type].width,
				enemy_data[enemies[which_enemy].enemy_data_type].height) == TRUE))
			{
				// Determine the raw pixel locations of the player and the enemy
				x_pixels[0] = player->x_cell * CELL_SIZE + player->x_offset;
				x_pixels[1] = enemies[which_enemy].x_cell * CELL_SIZE + 
					enemies[which_enemy].x_offset;
				y_pixels[0] = player->y_cell * CELL_SIZE + player->y_offset;
				y_pixels[1] = enemies[which_enemy].y_cell * CELL_SIZE + 
					enemies[which_enemy].y_offset;
				
				// If the player landed on top of the enemy's head
				if (player->y_velocity > 0 &&
					abs((int)(y_pixels[1] - y_pixels[0] - player->height)) >= 0 &&
					abs((int)(y_pixels[1] - y_pixels[0] - player->height)) <=
						player->max_y_velocity + 5)
				{
					enemies[which_enemy].health--;				// Hurt the enemy
					player->score += SCORE_DEFEAT_ENEMY;		// Add the score
						
					// Add the score animation
					add_animation(player->x_cell, player->y_cell, 
						player->x_offset + player->width / 2,
						player->y_offset + player->height - 12, 
						cell_data->number_small_zero_index, 10, 50, 1,
						SCORE_DEFEAT_ENEMY, 0, 0, animations, sprites, rle_sprites, 
						is_rle_sprite);

					// Add the smack animation
					add_animation(enemies[which_enemy].x_cell, enemies[which_enemy].y_cell,
						enemies[which_enemy].x_offset, enemies[which_enemy].y_offset - 18,
						cell_data->smack_index, 100, 7, 1, -1, 0,0, animations, sprites, 
						rle_sprites, is_rle_sprite);
						
					if (enemies[which_enemy].health <= 0) 	// If its health is gone, kill it
					{
						kill_enemy (which_enemy, enemies, enemy_data, animations, sprites, 
							rle_sprites, is_rle_sprite);
					}

					player->y_velocity = -FORCE_JUMP;	// Make the player jump
					player->is_jumping = TRUE;			// Allow Ctrl key jump raising
				}
				else	// If the player did not land on the enemy's head
				{
					if (player->is_blinking != TRUE)
					{
						if (x_pixels[0] < x_pixels[1]) facing_direction = FACING_LEFT;
						else facing_direction = FACING_RIGHT;
				
						hurt_player (facing_direction, enemy_data[enemies[which_enemy].
							enemy_data_type].player_damage, TRUE, player);
					}
				}
				update_flag = TRUE;		
			}

			// Move the enemy
			enemies[which_enemy].x_offset += x_add;
			enemies[which_enemy].y_offset += y_add;

			while (enemies[which_enemy].x_offset < 0) {			// Keep enemy coordinates in
				enemies[which_enemy].x_offset += CELL_SIZE;		// bounds
				enemies[which_enemy].x_cell--;
			}
			while (enemies[which_enemy].y_offset < 0) {
				enemies[which_enemy].y_offset += CELL_SIZE;
				enemies[which_enemy].y_cell--;
			}
			while (enemies[which_enemy].x_offset >= CELL_SIZE) {
				enemies[which_enemy].x_offset -= CELL_SIZE;
				enemies[which_enemy].x_cell++;
			}
			while (enemies[which_enemy].y_offset >= CELL_SIZE) {
				enemies[which_enemy].y_offset -= CELL_SIZE;
				enemies[which_enemy].y_cell++;
			}
		}
	}

	return update_flag;
}



// Make the enemy inactive and create whatever animation to show that it is daed
void kill_enemy (int which_enemy, ENEMY_TYPE enemies[], ENEMY_DATA_TYPE enemy_data[], 
				 ANIMATION_TYPE animations[], BITMAP *sprites[], RLE_SPRITE *rle_sprites[],
				 char is_rle_sprite[])
{
	// Add the defeated enemy animation
	add_animation(enemies[which_enemy].x_cell, enemies[which_enemy].y_cell,
		enemies[which_enemy].x_offset, enemies[which_enemy].y_offset + 1,
		enemy_data[enemies[which_enemy].enemy_data_type].cell_index,
		10, -1, 1, TAG_UPSIDE_DOWN, 0, 0, animations, sprites, rle_sprites, 
		is_rle_sprite);
						
	enemies[which_enemy].active = FALSE;
}



// Returns whether or not a rectangle collides with another rectangle
int rectangles_collide (int x_cell_1, int y_cell_1, double x_offset_1, double y_offset_1,
						int width_1, int height_1, int x_cell_2, int y_cell_2, 
						double x_offset_2, double y_offset_2, int width_2, int height_2)
{
	int x_pixels_1, y_pixels_1, x_pixels_2, y_pixels_2;

	x_pixels_1 = x_cell_1 * CELL_SIZE + x_offset_1;		// Convert the cell, offset arguments
	y_pixels_1 = y_cell_1 * CELL_SIZE + y_offset_1;		// to raw pixels form
	x_pixels_2 = x_cell_2 * CELL_SIZE + x_offset_2;
	y_pixels_2 = y_cell_2 * CELL_SIZE + y_offset_2;

	// Determine if the rectangles collide
	if (((x_pixels_2 - x_pixels_1 >= 0 && x_pixels_2 - x_pixels_1 < width_1) ||
		(x_pixels_1 - x_pixels_2 >= 0 && x_pixels_1 - x_pixels_2 < width_2)) &&
		((y_pixels_2 - y_pixels_1 >= 0 && y_pixels_2 - y_pixels_1 < height_1) ||
		(y_pixels_1 - y_pixels_2 >= 0 && y_pixels_1 - y_pixels_2 < height_2))) {
		return TRUE;
	} else return FALSE;
}



// Hurt the player due to: enemy damage (that's it thus far)
void hurt_player (int move_direction, int damage, char start_blinking, PLAYER_TYPE *player)
{
	if (move_direction == FACING_LEFT) {				// Make the player move
		player->x_velocity = -FORCE_ENEMY_COLLISION_X;
		player->y_velocity = -FORCE_ENEMY_COLLISION_Y;
	}
	else if (move_direction == FACING_RIGHT) {
		player->x_velocity = FORCE_ENEMY_COLLISION_X;
		player->y_velocity = -FORCE_ENEMY_COLLISION_Y;
	}
	
	player->health -= damage;							// Reduce the player's health
	
	if (start_blinking == TRUE) {						// Make the player blink
		player->is_blinking = TRUE;
		player->blink_counter = 125;
	}
}



// Move all the animations
int move_animations (COLLISION_INPUT_TYPE *collision_input_data, 
					  COLLISION_DETECTION_TYPE collision_data[2][4],
					  LEVEL_TYPE *level, PLAYER_TYPE *player, WINDOW_TYPE *window,
					  CELL_DATA_TYPE *cell_data, ANIMATION_TYPE animations[],
					  ENEMY_TYPE enemies[], ENEMY_DATA_TYPE enemy_data[], KEYS_TYPE *keysdown,
					  BITMAP *sprites[], RLE_SPRITE *rle_sprites[], char is_rle_sprite[])
{
	int which_ani;
	int cell_index;
	int update_flag = FALSE;

	for (which_ani = 0; which_ani < MAX_ANIMATIONS; which_ani++) {	// Go thru each animation
		if (animations[which_ani].active == TRUE) {					// If it is active,
			cell_index = animations[which_ani].cell_index;			// Get its cell index
			
			// Increment tha animation's sprite frame to use
			animations[which_ani].ani_counter++;
			if (animations[which_ani].ani_counter >= animations[which_ani].frames_delay)
			{
				animations[which_ani].ani_counter = 0;
				animations[which_ani].cur_frame++;
				if (animations[which_ani].cur_frame >= animations[which_ani].num_frames)
				{
					//if (animations[which_ani].num_frames > 1) exit(0);
					animations[which_ani].cur_frame = 0;
				}
			}
			
			if (cell_index == cell_data->bubble_index ||			// If it is a bubble
				cell_index == cell_data->bubble_index + 1) {
				animations[which_ani].y_offset--;					// Move it up one pixel
				if (animations[which_ani].y_offset < 0) {
					animations[which_ani].y_offset += CELL_SIZE;
					animations[which_ani].y_cell--;
				}
				if (cell_data->cell_types[level->tiles					// If the bubble hit
					[animations[which_ani].y_cell]						// something,
					[animations[which_ani].x_cell]] != TYPE_WATER) {
					animations[which_ani].active = FALSE;				// Remove it
				}
			}
			else if (cell_index == cell_data->switch_block_index ||	// If it is a switch block
				cell_index == cell_data->switch_block_index + 1 ||
				cell_index == cell_data->switch_block_index + 2)
			{	
				// Position the block animation
				animations[which_ani].y_offset = -sin(animations[which_ani].iterations * PI /
					animations[which_ani].num_iterations) * 15;

				// If the block animation has lived long enough, end it
				if (animations[which_ani].iterations >= animations[which_ani].num_iterations)
					animations[which_ani].active = FALSE;
			}
			else if (animations[which_ani].cell_index == 
				cell_data->number_small_zero_index)
			{
				animations[which_ani].y_offset = animations[which_ani].prev_y_offset - 
					sin(animations[which_ani].iterations * PI /
					animations[which_ani].num_iterations / 2) * 20;
			}
			else if (animations[which_ani].cell_index ==			// If it is an enemy type
				enemy_data[ENEMY_DATA_TYPE_RED].cell_index || 
					animations[which_ani].cell_index == 
				enemy_data[ENEMY_DATA_TYPE_ICE_BEAR].cell_index ||
					animations[which_ani].cell_index == 
				enemy_data[ENEMY_DATA_TYPE_BIRD].cell_index)
			{	
				// Move the animation downwards
				animations[which_ani].y_offset += animations[which_ani].y_velocity;
				
				animations[which_ani].y_velocity += FORCE_GRAVITY / 2;	// Speed up its descent
				while (animations[which_ani].y_offset >= CELL_SIZE) {	// Keep values in bounds
					animations[which_ani].y_offset -= CELL_SIZE;
					animations[which_ani].y_cell++;
				}
				if (animations[which_ani].y_cell >= level->height)		// If the ani is out of
					animations[which_ani].active = FALSE;				// bounds, end it
			}
			else if (animations[which_ani].cell_index == cell_data->bomb_index)
			{
				if (move_projectiles (which_ani, collision_input_data, collision_data, level, 
					player, window, cell_data, animations, enemies, enemy_data, keysdown,
					sprites, rle_sprites, is_rle_sprite) == TRUE)
				{
					update_flag = TRUE;
				}
			}
		}
	}

	return update_flag;
}



// Move all the projectiles (bombs, etc.). Called from the move_animations(...) function.
int move_projectiles (int which_ani, COLLISION_INPUT_TYPE *collision_input_data, 
					  COLLISION_DETECTION_TYPE collision_data[2][4],
					  LEVEL_TYPE *level, PLAYER_TYPE *player, WINDOW_TYPE *window,
					  CELL_DATA_TYPE *cell_data, ANIMATION_TYPE animations[],
					  ENEMY_TYPE enemies[], ENEMY_DATA_TYPE enemy_data[], KEYS_TYPE *keysdown,
					  BITMAP *sprites[], RLE_SPRITE *rle_sprites[], char is_rle_sprite[])
{
	int which_direction, which_cell, cell_index, which_type, x_cell, y_cell;
	int which_enemy;
	int update_flag = FALSE;

	// Move the projectile
	animations[which_ani].x_offset += animations[which_ani].x_velocity;
	animations[which_ani].y_offset += animations[which_ani].y_velocity;
	
	while (animations[which_ani].x_offset < 0) {
		animations[which_ani].x_offset += CELL_SIZE;
		animations[which_ani].x_cell--;
	}
	while (animations[which_ani].x_offset >= CELL_SIZE) {
		animations[which_ani].x_offset -= CELL_SIZE;
		animations[which_ani].x_cell++;
	}
	while (animations[which_ani].y_offset < 0) {
		animations[which_ani].y_offset += CELL_SIZE;
		animations[which_ani].y_cell--;
	}
	while (animations[which_ani].y_offset >= CELL_SIZE) {
		animations[which_ani].y_offset -= CELL_SIZE;
		animations[which_ani].y_cell++;
	}

	// Increase the downward speed of the projectile
	animations[which_ani].y_velocity += FORCE_GRAVITY / 2;

	// If the animation has fallen below the bottom of the level
	if (animations[which_ani].y_cell >= level->height)
		animations[which_ani].active = FALSE;
	
	// Get collision detection data for the moving projectile
	get_collision_input (collision_input_data, DETECT_PROJECTILE, which_ani,
		keysdown, player, enemies, enemy_data, animations);
	
	detect_collisions (collision_input_data, level, player, enemies, enemy_data,
		keysdown, cell_data, collision_data);
	
	// React to FUTURE STATE collision detection
	for (which_direction = 0; which_direction < 4; which_direction++)
	{
		for (which_cell = 0; which_cell < 
			collision_data[1][which_direction].num_scanned_cells; which_cell++)
		{
			cell_index = collision_data[1][which_direction].which_cell[which_cell];
			which_type = cell_data->cell_types[cell_index];
			x_cell = collision_data[1][which_direction].x_cell[which_cell];
			y_cell = collision_data[1][which_direction].y_cell[which_cell];

			switch (which_type) {
			case TYPE_EMPTY:
			case TYPE_DIAGONAL_UPPER_LEFT:
			case TYPE_DIAGONAL_UPPER_RIGHT:
			case TYPE_DIAGONAL_LOWER_LEFT:
			case TYPE_DIAGONAL_LOWER_RIGHT:
			case TYPE_ITEM:
			case TYPE_ACTION:
			case TYPE_FOREGROUND:
			case TYPE_WATER:
			case TYPE_CLIMBABLE:
			case TYPE_WATERFALL:
				break;
	
			case TYPE_PLATFORM:	if (which_direction != 1) break;
			case TYPE_SQUARE:
			case TYPE_HARM:
			case TYPE_ICE:
			case TYPE_MOVING_RIGHT:
			case TYPE_MOVING_LEFT:

				// Explosion!
				animations[which_ani].active = FALSE;

				// Create the explosion animation
				add_animation (animations[which_ani].x_cell, animations[which_ani].y_cell,
					animations[which_ani].x_offset - 20, animations[which_ani].y_offset - 20,
					cell_data->explosion_index + 0, 8, 32, 3, 0, 0, 0, animations, sprites,
					rle_sprites, is_rle_sprite);

				// If the projectile hit the switch
				if (cell_index == cell_data->switch_block_index || 
					cell_index == cell_data->switch_block_index + 1 || 
					cell_index == cell_data->switch_block_index + 2)
				{
					// Activate the switch block
					activate_switch_block(level, cell_data, x_cell, y_cell, cell_index,
						animations, sprites, rle_sprites, is_rle_sprite);
				}
			default:;
			}
		}
	}

	// Analyze each enemy
	for (which_enemy = 0; which_enemy < MAX_ACTIVE_ENEMIES; which_enemy++)
	{
		if (enemies[which_enemy].active == TRUE)
		{
			// If there is a collison between the bomb and the enemy
			if	((rectangles_collide(animations[which_ani].x_cell, 
				animations[which_ani].y_cell, animations[which_ani].x_offset,
				animations[which_ani].y_offset, animations[which_ani].width, 
				animations[which_ani].height, 
							
				enemies[which_enemy].x_cell, enemies[which_enemy].y_cell,
				enemies[which_enemy].x_offset, enemies[which_enemy].y_offset, 
				enemy_data[enemies[which_enemy].enemy_data_type].width,
				enemy_data[enemies[which_enemy].enemy_data_type].height) == TRUE)) {
				
				animations[which_ani].active = FALSE;	// End the animation

				// Create the explosion animation
				add_animation (animations[which_ani].x_cell, animations[which_ani].y_cell,
					animations[which_ani].x_offset - 20, animations[which_ani].y_offset - 20,
					cell_data->explosion_index + 0, 8, 32, 3, 0, 0, 0, animations, sprites,
					rle_sprites, is_rle_sprite);

				// Kill the enemy
				kill_enemy (which_enemy, enemies, enemy_data, animations, sprites, 
					rle_sprites, is_rle_sprite);

				update_flag = TRUE;		// Update the status bar
			}
		}
	}

	// Check to see if the projectile will collide with the player
	if	((rectangles_collide(animations[which_ani].x_cell, 
		animations[which_ani].y_cell, animations[which_ani].x_offset,
		animations[which_ani].y_offset, animations[which_ani].width, 
		animations[which_ani].height, 
							
		player->x_cell, player->y_cell,	player->x_offset, player->y_offset, 
		player->width, player->height) == TRUE))
	{
		if (animations[which_ani].tag == 1)			// If the rectangle has left the player
		{
			if (player->is_blinking == FALSE)		// If the player is not blinking
			{
				animations[which_ani].active = FALSE;	// End the animation

				// Create the explosion animation
				add_animation (animations[which_ani].x_cell, animations[which_ani].y_cell,
					animations[which_ani].x_offset - 20, animations[which_ani].y_offset - 20,
					cell_data->explosion_index + 0, 8, 32, 3, 0, 0, 0, animations, sprites,
					rle_sprites, is_rle_sprite);

				// Hurt the player
				hurt_player(FACING_LEFT + rand() % 2, DAMAGE_BOMB, TRUE, player);

				update_flag = TRUE;		// Update the status bar
			}
		}
	}
	else
	{
		animations[which_ani].tag = 1;				// The rectangle has now left the player
	}

	return update_flag;
}
	


// Collects an item on cell (x_cell, y_cell) of type type
// Returns TRUE if the status bar now needs to be updated
int collect_item (int x_cell, int y_cell, int type, LEVEL_TYPE *level, PLAYER_TYPE *player,
				   CELL_DATA_TYPE *cell_data, BITMAP **status_bar_bmp, RLE_SPRITE *rle_sprites[],
				   WINDOW_TYPE *window)
{
	int which_type;	
	int draw_items, refresh_items, include_o2_bar;
	int update_flag = FALSE;
	
	// Decide what to do
	which_type = level->tiles[y_cell][x_cell];

	if (which_type == cell_data->apples_index) {					// If the item is apples
		level->tiles[y_cell][x_cell] = cell_data->empty_cell_index;	// Empty the cell
		player->score += SCORE_APPLES;								// Add the score increment
		player->health += RECOVER_APPLES;							// Add the health increment
		update_flag = TRUE;											// Refresh the status bar
	}
	else if (which_type == cell_data->bananas_index) {				// If the item is bananas
		level->tiles[y_cell][x_cell] = cell_data->empty_cell_index;	// Empty the cell
		player->score += SCORE_BANANAS;								// Add the score increment
		player->health += RECOVER_BANANAS;							// Add the health increment
		update_flag = TRUE;											// Refresh the status bar
	}
	else if (which_type == cell_data->grapes_index) {				// If the item is grapes
		level->tiles[y_cell][x_cell] = cell_data->empty_cell_index;	// Empty the cell
		player->score += SCORE_GRAPES;								// Add the score increment
		player->health += RECOVER_GRAPES;							// Add the health increment
		update_flag = TRUE;											// Refresh the status bar
	}
	else if (which_type == cell_data->watermelon_index) {			// If the item is a melon
		level->tiles[y_cell][x_cell] = cell_data->empty_cell_index;	// Empty the cell
		player->score += SCORE_WATERMELON;							// Add the score increment
		player->health += RECOVER_WATERMELON;						// Add the health increment
		update_flag = TRUE;											// Refresh the status bar
	}
	else if (which_type == cell_data->heart_index) {				// If the item is a heart
		level->tiles[y_cell][x_cell] = cell_data->empty_cell_index;	// Empty the cell
		player->score += SCORE_HEART;								// Add the score increment
		player->lives++;											// Add the lives increment
		update_flag = TRUE;											// Refresh the status bar
	}
	else if (which_type == cell_data->bomb_powerup_index) {		// If the item is a bomb powerup
		level->tiles[y_cell][x_cell] = cell_data->empty_cell_index;	// Empty the cell
		if (player->has_item != ITEM_THROW_BOMBS)
			player->num_projectiles = 0;
		player->has_item = ITEM_THROW_BOMBS;						// Give the player bombs
		player->max_projectiles = MAX_BOMBS;						// Set the maximum number
		player->num_projectiles += BOMB_POWERUP_INCREMENT;			// Increase the # of bombs
		if (player->num_projectiles > player->max_projectiles)		// Limit the max # of bombs
			player->num_projectiles = player->max_projectiles;
		
					// Determine what to include
					draw_items = TRUE;
					refresh_items = TRUE;
					include_o2_bar = FALSE;
					
					// Refresh the status buar
					refresh_status_bar
						(FALSE, TRUE, 
						FALSE, TRUE, 
						FALSE, TRUE, 
						FALSE, (char) include_o2_bar,
						(char) draw_items, (char) refresh_items,
						status_bar_bmp, rle_sprites, player, window, 
						cell_data);

		
		update_flag = TRUE;
	}

	if (player->score > player->max_score)		// Keep everything in bounds
		player->score = player->max_score;
	if (player->health > player->max_health) 
		player->health = player->max_health;
	if (player->lives > player->max_lives) 
		player->lives = player->max_lives;


	return update_flag;
}



// Updates different parts of the status bar
void refresh_status_bar (char draw_health, char refresh_health, 
						 char draw_lives, char refresh_lives, 
						 char draw_score, char refresh_score, 
						 char draw_o2_bar, char refresh_o2_bar, 
						 char draw_item, char refresh_item, 
						 BITMAP **status_bar_bmp, RLE_SPRITE *rle_sprites[],
						 PLAYER_TYPE *player, WINDOW_TYPE *window, CELL_DATA_TYPE *cell_data)
{
	int x, y;
	int x_start, x_end;		// For determining the location of the bar drawings
	int y_start, y_end;
	int row, pixel;
	int r, g, b;
	int add_x, add_y;
	int cell_index;
	int counter;				// Counts two iterations for drawing the health/o2 bars
	
	if (draw_health)
	{	
		// Draw the lifebar icon
		draw_rle_sprite(*status_bar_bmp, rle_sprites[cell_data->lifebar_icon_index], 
			window->lifebar_x, window->lifebar_y);
	}

	if (draw_o2_bar)
	{
		// Draw the o2 bar icon
		draw_rle_sprite(*status_bar_bmp, rle_sprites[cell_data->o2_icon_index], 
			window->lifebar_x, window->lifebar_y + rle_sprites[cell_data->lifebar_icon_index]->h
			+ 4);
	}

	if (draw_item)
	{
		// Draw the item icon
		switch (player->has_item) {
		case ITEM_NONE:			cell_index = cell_data->empty_cell_index;	break;
		case ITEM_THROW_BOMBS:	cell_index = cell_data->bomb_index;			break;
		default:;
		}

		draw_rle_sprite(*status_bar_bmp, rle_sprites[cell_index], 
			window->lifebar_x + rle_sprites[cell_data->lifebar_icon_index]->w / 2 -
			rle_sprites[cell_index]->w / 2, 
			window->lifebar_y + 
			rle_sprites[cell_data->lifebar_icon_index]->h + 4 +
			rle_sprites[cell_data->lifebar_icon_index]->h / 2 -
			rle_sprites[cell_index]->h / 2);
	}

	// If the function was told to refresh certain status bars
	if (refresh_health || draw_health || refresh_o2_bar || draw_o2_bar || refresh_item ||
		draw_item)
	{	
		// Loop twice: once for the health bar, once for the o2/item bar
		for (counter = 0; counter < 2; counter++)
		{	
			// Skip this iteration if we are not told to draw the specified target
			if (counter == 0 && (refresh_health == FALSE && draw_health == FALSE)) continue;
			if (counter == 1 && (refresh_o2_bar == FALSE && draw_o2_bar == FALSE &&
								 refresh_item == FALSE && draw_item == FALSE)) continue;

			// Determine the row of pixels to draw the bar
			if (counter == 0)
				y = window->lifebar_y;
			else
				y = window->lifebar_y + rle_sprites[cell_data->lifebar_icon_index]->h + 4;

			// Draw the left end of the bar
			draw_rle_sprite(*status_bar_bmp, rle_sprites[cell_data->lifebar_left_index], 
				window->lifebar_x + rle_sprites[cell_data->lifebar_icon_index]->w + 3, y + 1);
			
			// Determine where to draw the vertical strips of the bar
			x_start = window->lifebar_x + rle_sprites[cell_data->lifebar_icon_index]->w + 3 +
				rle_sprites[cell_data->lifebar_left_index]->w;
			
			if (counter == 0)
			{
				x_end = x_start + player->max_health -
					(rle_sprites[cell_data->lifebar_left_index]->w +
					rle_sprites[cell_data->lifebar_right_index]->w) + 4;
			}
			else if (counter == 1)
			{
				if (refresh_item)
				{
					x_end = x_start + player->max_projectiles -
						(rle_sprites[cell_data->lifebar_left_index]->w +
						rle_sprites[cell_data->lifebar_right_index]->w) + 8;
				}
				else if (refresh_o2_bar)
				{
					x_end = x_start + player->max_o2 -
						(rle_sprites[cell_data->lifebar_left_index]->w +
						rle_sprites[cell_data->lifebar_right_index]->w) + 4;
				}
			}

			// Draw vertical strips of the bar
			for (x = x_start; x < x_end; x++) {
				draw_rle_sprite(*status_bar_bmp, rle_sprites[cell_data->lifebar_middle_index], 
					x, y + 1);
			}
	
			// Determine where to draw the right end of the bar
			if (counter == 0)
			{
				x_start = window->lifebar_x + rle_sprites[cell_data->lifebar_icon_index]->w + 3
					+ rle_sprites[cell_data->lifebar_left_index]->w + player->max_health -
					(rle_sprites[cell_data->lifebar_left_index]->w + 
					rle_sprites[cell_data->lifebar_right_index]->w) + 4;
			}
			else if (counter == 1)
			{
				if (refresh_item)
				{
					x_start = window->lifebar_x + rle_sprites[cell_data->lifebar_icon_index]->w
						+ 3	+ rle_sprites[cell_data->lifebar_left_index]->w +
						player->max_projectiles -(rle_sprites[cell_data->lifebar_left_index]->w + 
						rle_sprites[cell_data->lifebar_right_index]->w) + 8;
				}
				else if (refresh_o2_bar)
				{
					x_start = window->lifebar_x + rle_sprites[cell_data->lifebar_icon_index]->w
						+ 3	+ rle_sprites[cell_data->lifebar_left_index]->w + player->max_o2 -
						(rle_sprites[cell_data->lifebar_left_index]->w + 
						rle_sprites[cell_data->lifebar_right_index]->w) + 4;
				}
			}

			// Draw the right end of the life bar
			draw_rle_sprite(*status_bar_bmp, rle_sprites[cell_data->lifebar_right_index],
				x_start, y + 1);

			// Determine how far to the right to fill in the bar
			x_start = window->lifebar_x + rle_sprites[cell_data->lifebar_icon_index]->w + 3 + 4;
			if (counter == 0)
			{
				x_end = window->lifebar_x + rle_sprites[cell_data->lifebar_icon_index]->w + 3 + 
					player->health;
			} 
			else if (counter == 1)
			{
				if (refresh_item)
				{
					x_end = window->lifebar_x + rle_sprites[cell_data->lifebar_icon_index]->w + 3
						+ player->num_projectiles + 4;
				}
				else if (refresh_o2_bar)
				{
					x_end = window->lifebar_x + rle_sprites[cell_data->lifebar_icon_index]->w + 3
						+ player->o2;
				}
			}

			// Fill in the bar
			for (x = x_start; x < x_end; x++) {
			
				// Draw one vertical strip of the bar
				y_start = y + 5;
				y_end = y + rle_sprites[cell_data->lifebar_middle_index]->h - 2;
				
				for (row = y_start; row < y_end; row++) {
					pixel = getpixel(*status_bar_bmp, x, row);		// Get the pixel
					r = getr(pixel);								// Get its RGB components
					g = getg(pixel);
					b = getb(pixel);
					if ((r != 0 || g != 0 || b != 0) &&			// If the pixel is not black
						(r < 100 || g < 100 || b < 100)) {		// and is not too bright
						
						add_x = (x - (window->lifebar_x + rle_sprites[
							cell_data->lifebar_icon_index]->w + 3 + 4));
						add_y = (row - (y + 4));
						
						if (counter == 0)		// Determine how to change the color of the bar
						{
							r += 25;			// Tint it green for health bar
							g += 185;
							b += 25;
						} 
						else if (counter == 1)
						{
							if (refresh_item)
							{
								r += 185;		// Tint it red for item bar
								g += 25;
								b += 25;
							}
							else if (refresh_o2_bar)
							{
								r += 25;		// Tint it blue for oxygen bar
								g += 25;
								b += 185;
							}
						}

						if (r > 255) r = 255;							// Keep the RGB values
						if (g > 255) g = 255;							//   in bounds
						if (b > 255) b = 255;
						
						pixel = makecol(r, g, b);						// Make the new pixel
						putpixel(*status_bar_bmp, x, row, pixel);		// Draw the new pixel
					}
				}
			}
		}
	}

	if (refresh_score || draw_score)
	{
		draw_big_numbers(SCREEN_W - 4, window->lifebar_y, player->score, status_bar_bmp,
		rle_sprites, cell_data);
	}

	if (draw_lives || refresh_lives)
	{	
		// Clear the area that the lives section goes on
		rectfill(*status_bar_bmp, SCREEN_W - rle_sprites[cell_data->number_big_zero_index]->w * 5,
			window->lifebar_y + rle_sprites[cell_data->number_big_zero_index]->h + 1,
			SCREEN_W - 1, window->lifebar_y + rle_sprites[cell_data->number_big_zero_index]->h +
			1 +	rle_sprites[cell_data->number_big_zero_index]->h + 5, TRANSPARENT_COL);
		
		// Draw the number sprites
		x_start = draw_big_numbers(SCREEN_W - 4, window->lifebar_y + 
			rle_sprites[cell_data->number_big_zero_index]->h + 5, (long) player->lives,
			status_bar_bmp, rle_sprites, cell_data);
		
		x_start -= rle_sprites[cell_data->x_icon_index]->w + 2;
		
		// Draw the "X" sprite
		draw_rle_sprite (*status_bar_bmp, rle_sprites[cell_data->x_icon_index], 
			x_start, window->lifebar_y + rle_sprites[cell_data->number_big_zero_index]->h + 7);

		x_start -= rle_sprites[cell_data->heart_index + 1]->w + 0;

		// Draw the heart
		draw_rle_sprite (*status_bar_bmp, rle_sprites[cell_data->heart_index + 1],
			x_start, window->lifebar_y + rle_sprites[cell_data->number_big_zero_index]->h + 1);
	}
	
}



// Draws an integer value, right-justified, using the big number sprites
// Returns the column of pixels that marks the left edge of the images drawn
int draw_big_numbers (int pos_x, int pos_y, long value, BITMAP **status_bar_bmp,
					   RLE_SPRITE *rle_sprites[], CELL_DATA_TYPE *cell_data)
{
	int x, number;
	int digit, width;

	// Draw the sprites for the numbers
	for (x = pos_x, digit = 0; value >= pow(10, digit) || digit == 0; digit++) {
		
		// Determine which digit to draw
		number = ((value / (int)(pow(10, digit))) % 10);
		
		// Bounds check the number we obtained
		if (number < 0 || number > 9) {
			shout("An error occured with draw_big_numbers (...) !");
			exit(1);
		}
		
		// Determine the width of the sprite		
		width = rle_sprites[cell_data->number_big_zero_index + number]->w;
		
		x -= width;		// Position where to draw the sprite
		
		// Erase the area behind the sprite
		rectfill(*status_bar_bmp, x - 6, pos_y, x + width, pos_y + 
			rle_sprites[cell_data->number_big_zero_index]->h, TRANSPARENT_COL);
	
		// Draw the sprite itself
		draw_rle_sprite(*status_bar_bmp, 
			rle_sprites[cell_data->number_big_zero_index + number], x, pos_y);
			x -= 3;
	}

	return x;
}



// Draws an integer value, center-justified, using the small number sprites
void draw_small_numbers (int pos_x, int pos_y, long value, BITMAP **buffer,
					   RLE_SPRITE *rle_sprites[], CELL_DATA_TYPE *cell_data)
{
	int x, number;
	int digit, width;
	int total_width = 0;

	// Determine the width of everything
	// Draw the sprites for the numbers
	for (x = pos_x, digit = 0; value >= pow(10, digit) || digit == 0; digit++) {
		number = ((value / (int)(pow(10, digit))) % 10);
		total_width += rle_sprites[cell_data->number_small_zero_index + number]->w;
	}

	// Draw the sprites for the numbers
	for (x = pos_x + total_width / 2, digit = 0; value >= pow(10, digit) || digit == 0;
		digit++) {
		
		// Determine which digit to draw
		number = ((value / (int)(pow(10, digit))) % 10);
		
		// Bounds check the number we obtained
		if (number < 0 || number > 9) {
			shout("An error occured with draw_big_numbers (...) !");
			exit(1);
		}
		
		// Determine the width of the sprite		
		width = rle_sprites[cell_data->number_small_zero_index + number]->w;
				
		// Erase the area behind the sprite
	//	rectfill(*buffer, x - 6, pos_y, x + width, pos_y + 
	//		rle_sprites[cell_data->number_small_zero_index]->h, TRANSPARENT_COL);
	
		// Draw the sprite itself
		draw_rle_sprite(*buffer, 
			rle_sprites[cell_data->number_small_zero_index + number], x, pos_y);
		
		x -= width + 1;		// Position where to draw the sprite
	}
}



// Sets the data of the passed collison input type structure, which will then be ready to be 
// passed to detect_collisions (...).
// type = DETECT_PLAYER, DETECT_ENEMY, DETECT_PROJECTILE
// which = the corresponding array element, if using an enemy, projectile, etc.
void get_collision_input (COLLISION_INPUT_TYPE *collision_input_data, int type, int which,
						  KEYS_TYPE *keysdown, PLAYER_TYPE *player, ENEMY_TYPE enemies[],
						  ENEMY_DATA_TYPE enemy_data[], ANIMATION_TYPE animations[])
{
	int width, height, x_cell, y_cell;
	double x_offset, y_offset, x_velocity, y_velocity, x_offset_shift, y_offset_shift;
	
	if (type == DETECT_PLAYER)					// Use the player's data
	{				
		which = 0;
		width = player->width;
		height = player->height;
		x_cell = player->x_cell;
		y_cell = player->y_cell;
		x_offset = player->x_offset;
		y_offset = player->y_offset;
		x_velocity = player->x_velocity;
		y_velocity = player->y_velocity;
		
		if (player->conveyed_left == TRUE)
		{
			x_offset_shift = -FORCE_CONVEYOR;
			if (keysdown->left_arrow) x_offset_shift -= FORCE_WALK;
		}
		else if (player->conveyed_right == TRUE)
		{
			x_offset_shift = FORCE_CONVEYOR;
			if (keysdown->right_arrow) x_offset_shift += FORCE_WALK;
		}
		else x_offset_shift = 0;
		
		if (player->in_waterfall == TRUE) y_offset_shift = FORCE_WATERFALL;
		else y_offset_shift = 0;
	} 
	else if (type == DETECT_ENEMY)				// Use an enemy's data
	{
		// (The active enemy type to detect was passed in as which)
		width = enemy_data[enemies[which].enemy_data_type].width;
		height = enemy_data[enemies[which].enemy_data_type].height;
		x_cell = enemies[which].x_cell;
		y_cell = enemies[which].y_cell;
		x_offset = enemies[which].x_offset;
		y_offset = enemies[which].y_offset;
		x_velocity = enemies[which].x_velocity;
		y_velocity = enemies[which].y_velocity;
		x_offset_shift = 0;
		y_offset_shift = 0;
	}
	else if (type == DETECT_PROJECTILE)			// Use a projectile's data
	{
		// (The active projectile type to detect was passed in as which)
		width = animations[which].width;
		height = animations[which].height;
		x_cell = animations[which].x_cell;
		y_cell = animations[which].y_cell;
		x_offset = animations[which].x_offset;
		y_offset = animations[which].y_offset;
		x_velocity = animations[which].x_velocity;
		y_velocity = animations[which].y_velocity;
		x_offset_shift = 0;
		y_offset_shift = 0;
	}

	// Set the data of the collision input data object
	collision_input_data->which = which;
	collision_input_data->width = width;
	collision_input_data->height = height;
	collision_input_data->x_cell = x_cell;
	collision_input_data->y_cell = y_cell;
	collision_input_data->x_offset = x_offset;
	collision_input_data->y_offset = y_offset;
	collision_input_data->x_velocity = x_velocity;
	collision_input_data->y_velocity = y_velocity;
	collision_input_data->x_offset_shift = x_offset_shift;
	collision_input_data->y_offset_shift = y_offset_shift;
}



// Performs collision detection and returns the data
// collision_data[0][0]: Present state - UP
// collision_data[0][1]: Present state - DOWN
// collision_data[0][2]: Present state - LEFT
// collision_data[0][3]: Present state - RIGHT
// collision_data[1][0]: Future state - UP
// collision_data[1][1]: Future state - DOWN
// collision_data[1][2]: Future state - LEFT
// collision_data[1][3]: Future state - RIGHT
void detect_collisions (COLLISION_INPUT_TYPE *collision_input_data, LEVEL_TYPE *level,
						PLAYER_TYPE *player, ENEMY_TYPE enemies[], ENEMY_DATA_TYPE enemy_data[],
						KEYS_TYPE *keysdown, CELL_DATA_TYPE *cell_data,
						COLLISION_DETECTION_TYPE collision_data[2][4])
{
	int height;				// Height and width of the rectangle
	int width;
	int x_cell;				// Coordinates of the cell in which the upper left pixel of the 
	int y_cell;				// rectangle resides
	double x_offset;		// Coordinates of the upper left pixel of the rectangle, relative
	double y_offset;		// to the cell in which it resides
	int num_rows_occupied;	// How many rows or columns the rectangle occupies at present
	int num_cols_occupied;
	double x_velocity;			// For future state collision detection
	double y_velocity;

	double future_x_offset;		// Coordinates of the FUTURE LOCATION of the upper left pixel of
	double future_y_offset;		// the rectangle, relative to the cell in which it WILL reside

	double x_offset_shift, y_offset_shift;		// How much to shift the future_x_offset,
												// future_y_offset values after calculation
												// due to conveyor belts, waterfalls, etc.

	char will_enter_new_row_up = FALSE;		// Inidicate whether the object will enter a new
	char will_enter_new_row_down = FALSE;	// row or column of cells (for FUTURE STATE
	char will_enter_new_col_left = FALSE;	// collision detection)
	char will_enter_new_col_right = FALSE;
		
	int scan_x_cell[2][6];			// Array of six cells to scan
	int scan_y_cell[2][6];
		
	int which_cell;				// Which cell is being scanned
	int which_direction;		// Which direction collision detection is being performed in
	int num_scan_cells = 0;		// Number of cells to scan (ranges from 3-6)
	int which_row, which_col;	// Used for scanning cells

	double value;
	int x, y;	// Dummy values

	// Get the data from the collision input type object we passed in
	width = collision_input_data->width;
	height = collision_input_data->height;
	x_cell = collision_input_data->x_cell;
	y_cell = collision_input_data->y_cell;
	x_offset = collision_input_data->x_offset;
	y_offset = collision_input_data->y_offset;
	x_velocity = collision_input_data->x_velocity;
	y_velocity = collision_input_data->y_velocity;
	x_offset_shift = collision_input_data->x_offset_shift;
	y_offset_shift = collision_input_data->y_offset_shift;

	num_cols_occupied = 1;
	value = x_offset + width;
	while (value > CELL_SIZE) {
		value -= CELL_SIZE;
		num_cols_occupied++;
	}

	num_rows_occupied = 1;
	value = y_offset + height;
	while (value > CELL_SIZE) {
		value -= CELL_SIZE;
		num_rows_occupied++;
	}

	// Calculate the variables - Future state
	future_x_offset = x_offset + x_velocity + x_offset_shift;
	future_y_offset = y_offset + y_velocity + y_offset_shift;

	// *** Calculate whether the rectangle will enter a new row or column ***
	if (x_offset >= 0 && future_x_offset < 0 && x_velocity < 0)
	{
		will_enter_new_col_left = TRUE;
		will_enter_new_col_right = FALSE;	
	}
	// If the right edge of the rectangle is in a columm of cells and the future right edge of the rectangle
	// will be in a more rightwards row of cells
	else if (x_offset + width <= CELL_SIZE * num_cols_occupied && 
		future_x_offset + width > CELL_SIZE * num_cols_occupied)
	{
		will_enter_new_col_left = FALSE;	// Mark that the rectangle will enter a new rightwards col of cells
		will_enter_new_col_right = TRUE;
	} 
	else
	{
		will_enter_new_col_left = FALSE;
		will_enter_new_col_right = FALSE;
	}


	if (y_offset >= 0 && future_y_offset < 0 && y_velocity < 0)
	{
		will_enter_new_row_up = TRUE;
		will_enter_new_row_down = FALSE;
	}
	// If the bottom edge of the rectangle is in a row of cells and the future bottom edge of the rectangle
	// will be in a lower row of cells
	else if (y_offset + height <= CELL_SIZE * num_rows_occupied && 
		future_y_offset + height > CELL_SIZE * num_rows_occupied)
	{
		will_enter_new_row_up = FALSE;		// Mark that the rectangle will enter a new lower row of cells
		will_enter_new_row_down = TRUE;	
	}
	else
	{
		will_enter_new_row_up = FALSE;
		will_enter_new_row_down = FALSE;		
	}

	
	// By default, allow horizontal and vertical movement
	collision_data[0][0].allow_x_movement = TRUE;
	collision_data[0][0].allow_y_movement = TRUE;

	// PRESENT STATE collision detection
	// Scan each of the cells that the player currently occupies.
	// Data is assigned to collision_data[0][0] because present state collision detection
	// is not associated with any scanning direction!
	which_cell = 0;
	for (y = y_cell; y < y_cell + num_rows_occupied; y++) {
		for (x = x_cell; x < x_cell + num_cols_occupied; x++) {

			// Set the cell coordinates
			collision_data[0][0].x_cell[which_cell] = x;
			collision_data[0][0].y_cell[which_cell] = y;

			// Set the cell index
			collision_data[0][0].which_cell[which_cell] =				
				get_cell(level, cell_data, x, y);

			which_cell++;	// On to the next cell!
		}
	}

	collision_data[0][0].num_scanned_cells = which_cell;	// Set the number of scanned cells
	
	// FUTURE STATE collision detection	
	// Scan any new cells that the player will occupy.
	// Data is assigned to collision_data[1][which_direction]
	for (which_direction = 0; which_direction < 4; which_direction++) {	
	
		num_scan_cells = 0;

		if (which_direction == 0) {					// UP direction (collision_data[1][0])
			if (will_enter_new_row_up == TRUE) {	// If the rectangle will move to a new row
				
				for (which_col = 0; which_col < num_cols_occupied; which_col++) {
					scan_x_cell[1][num_scan_cells] = x_cell + which_col;
					scan_y_cell[1][num_scan_cells] = y_cell - 1;
					num_scan_cells++;
				}
			}	
		} else if (which_direction == 1) {			// DOWN direction (collision_data[1][1])
			if (will_enter_new_row_down == TRUE) {	// If the rectangle will move to a new row
				for (which_col = 0; which_col < num_cols_occupied; which_col++) {
					scan_x_cell[1][num_scan_cells] = x_cell + which_col;
					scan_y_cell[1][num_scan_cells] = y_cell + num_rows_occupied;
					num_scan_cells++;
				}
			}
		} else if (which_direction == 2) {			// LEFT direction (collision_data[1][2])
			if (will_enter_new_col_left == TRUE) {	// If the rectangle will move to a new col
								
				for (which_row = 0; which_row < num_rows_occupied; which_row++) {
					scan_x_cell[1][num_scan_cells] = x_cell - 1;
					scan_y_cell[1][num_scan_cells] = y_cell + which_row;
					num_scan_cells++;
				}
			}	
		} else if (which_direction == 3) {			// RIGHT direction (collision_data[1][3])
			if (will_enter_new_col_right == TRUE) {	// If the rectangle will move to a new col
				for (which_row = 0; which_row < num_rows_occupied; which_row++) {
					scan_x_cell[1][num_scan_cells] = x_cell + num_cols_occupied;
					scan_y_cell[1][num_scan_cells] = y_cell + which_row;
					num_scan_cells++;
				}
			}
		}

		// Set the num of scanned cells
		collision_data[1][which_direction].num_scanned_cells = num_scan_cells;

		// Collect the data for each cell
		for (which_cell = 0; which_cell < num_scan_cells; which_cell++) {
			
			// Set the cell coordinates
			collision_data[1][which_direction].x_cell[which_cell] = scan_x_cell[1][which_cell];
			collision_data[1][which_direction].y_cell[which_cell] = scan_y_cell[1][which_cell];

			// Set the cell index
			collision_data[1][which_direction].which_cell[which_cell] =				
				get_cell(level, cell_data, scan_x_cell[1][which_cell], 
						 scan_y_cell[1][which_cell]);
		}

	} // END FUTURE STATE collision detection
}



// Starts a projectile
int throw_projectile (BITMAP **status_bar_bmp, BITMAP *sprites[], RLE_SPRITE *rle_sprites[], 
					   char is_rle_sprite[], PLAYER_TYPE *player, CELL_DATA_TYPE *cell_data,
					   ANIMATION_TYPE animations[], WINDOW_TYPE *window)
{
	int x, y;
	int update_flag = FALSE;
	double x_velocity, y_velocity;
	
	switch (player->has_item) {
	case ITEM_THROW_BOMBS:
		
		// Determine the speed and direction of the bomb's motion
		x_velocity = FORCE_THROW_BOMB * cos(((player->throw_angle - 64) % 256) * PI / 128);
		y_velocity = FORCE_THROW_BOMB * sin(((player->throw_angle - 64) % 256) * PI / 128);
		if (player->facing == FACING_LEFT) x_velocity *= -1;
		if (player->is_on_ice == TRUE)
			x_velocity += player->x_velocity;
		else
			x_velocity += player->x_velocity / 3;
		y_velocity += player->y_velocity / 3;

		// Add the bomb animation
//		add_animation(player->x_cell, player->y_cell, player->x_offset, player->y_offset,
//			cell_data->bomb_index, 10, -1, -1, player->throw_angle, 
//			x_velocity, y_velocity, animations, sprites, rle_sprites, is_rle_sprite)
		// Add the bomb animation
		add_animation(player->x_cell, player->y_cell, player->x_offset, player->y_offset,
			cell_data->bomb_index, 10, -1, -1, 0, 
			x_velocity, y_velocity, animations, sprites, rle_sprites, is_rle_sprite);

		player->num_projectiles--;
		if (player->num_projectiles <= 0)	// If the player has run out of bombs
		{
			player->has_item = ITEM_NONE;

			// Erase the o2 bar
			x = window->lifebar_x;
			y = window->lifebar_y + rle_sprites[cell_data->lifebar_icon_index]->h + 4;
			rectfill(*status_bar_bmp, x, y, x + rle_sprites[cell_data->o2_icon_index]->w + 5 + 
				player->max_o2 + 1, y + rle_sprites[cell_data->o2_icon_index]->h,
				TRANSPARENT_COL);

			update_flag = TRUE;		// Refresh the status bar
		}

		break;
	default:;
	}

	return update_flag;
}



// Returns the cell located at (x, y) in the level. If it is above or below the level's bounds,
// returns an empty cell.
int get_cell(LEVEL_TYPE *level, CELL_DATA_TYPE *cell_data, int x, int y)
{
	if (x < 0 || x >= level->width)
		return cell_data->rocksquare_index;
	if (y < 0 || y >= level->height)
		return cell_data->empty_cell_index;
	else
		return level->tiles[y][x];
}



// Changes the cell located at (x, y) in the level to cell_index. If it is above or below the 
// level's bounds, this function does nothing.
void set_cell(LEVEL_TYPE *level, CELL_DATA_TYPE *cell_data, int x, int y, int cell_index)
{
	if (x < 0 || x >= level->width);
	else if (y < 0 || y >= level->height);
	else
		level->tiles[y][x] = cell_index;
}


// Moves the video window in any direction
void scroll_window (WINDOW_TYPE *window, LEVEL_TYPE *level, int x_add, int y_add)
{
	// Keep the window in bounds so we don't look outside of the level
	if (x_add < 0) 
		if (window->x_cell < 0 || (window->x_cell == 0 && window->x_offset + x_add < 0)) return;
	if (y_add < 0) 
		if (window->y_cell < 0 || (window->y_cell == 0 && window->y_offset + y_add < 0)) return;
	if (x_add > 0) 
		if (window->x_cell > level->width - (SCREEN_W / CELL_SIZE) - 1 || 
			(window->x_cell == level->width - (SCREEN_W / CELL_SIZE) - 1 && 
			window->x_offset + x_add >= CELL_SIZE)) return;
	if (y_add > 0) 
		if (window->y_cell > level->height - (SCREEN_H / CELL_SIZE) - 1 || 
			(window->y_cell == level->height - (SCREEN_H / CELL_SIZE) - 1 && 
			window->y_offset + y_add >= CELL_SIZE)) return;
	
	// Move the window's pixel offset by the given values
	window->x_offset += x_add;
	window->y_offset += y_add;
	
	// Keep the window's pixel offset in range, adjusting its starting cell
	while (window->x_offset < 0) {
		window->x_offset += CELL_SIZE;
		window->x_cell--;
	}
	while (window->y_offset < 0) {
		window->y_offset += CELL_SIZE;
		window->y_cell--;
	}
	while (window->x_offset > CELL_SIZE) {
		window->x_offset -= CELL_SIZE;
		window->x_cell++;
	}
	while (window->y_offset > CELL_SIZE) {
		window->y_offset -= CELL_SIZE;
		window->y_cell++;
	}
}

// Copies one video page to another
void copy_page (BITMAP *source, BITMAP *dest)  {
	blit (source, dest, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
}



// Inserts a string representation of an integer (0-99) into the string
void get_str_from_int (char *str, int num)
{
	if (num >= 0 && num <= 9) {
		str[0] = '0' + num;
		str[1] = '\0';
	} else if (num >= 10 && num <= 99) {
		str[0] = '0' + num / 10;
		str[1] = '0' + num % 10;
		str[2] = '\0';
	}
}



// Inserts a string representation of an long integer (0-99999) into the string
void get_str_from_long (char *str, long num)
{
	if (num >= 0 && num <= 9) {
		str[0] = '0' + num;
		str[1] = '\0';
	} else if (num >= 10 && num <= 99) {
		str[0] = '0' + num / 10;
		str[1] = '0' + num % 10;
		str[2] = '\0';
	} else if (num >= 100 && num <= 999) {
		str[0] = '0' + num / 100;
		str[1] = '0' + num / 10;
		str[2] = '0' + num % 10;
		str[3] = '\0';
	} else if (num >= 1000 && num <= 9999) {
		str[0] = '0' + num / 1000;
		str[1] = '0' + num / 100;
		str[2] = '0' + num / 10;
		str[3] = '0' + num % 10;
		str[4] = '\0';
	} else if (num >= 10000 && num <= 99999) {
		str[0] = '0' + (num / 10000) % 10;
		str[1] = '0' + (num / 1000) % 10;
		str[2] = ',';
		str[3] = '0' + (num / 100) % 10;
		str[4] = '0' + (num / 10) % 10;
		str[5] = '0' + num % 10;
		str[6] = '\0';
	} else if (num >= 100000 && num <= 999999) {
		str[0] = '0' + num / 100000;
		str[1] = '0' + num / 10000;
		str[2] = '0' + num / 1000;
		str[3] = ',';
		str[4] = '0' + num / 100;
		str[5] = '0' + num / 10;
		str[6] = '0' + num % 10;
		str[7] = '\0';
	}
}



// Prints a string on screen, and pauses
void shout (char *str)
{
	TEXTPRINTF(screen, font, 0, 0, TEXTPRINTF_COL, str);
	PAUSE;
}



// Prints a number on screen, and pauses
void snum (int num)
{
	TEXTPRINTF(screen, font, 0, 0, TEXTPRINTF_COL, "%d", num);
	PAUSE;
}



// Starts the animation of a switch block type cell located at x_cell, y_cell,
// with cell sprite index cell_indez
void activate_switch_block (LEVEL_TYPE *level, CELL_DATA_TYPE *cell_data, 
							int x_cell, int y_cell,	int cell_index,
							ANIMATION_TYPE animations[], BITMAP *sprites[],
							RLE_SPRITE *rle_sprites[], char is_rle_sprite[])
{
	// Switch the solid and dotted blocks of that color in the level
	switch_solid_and_dotted_blocks (level, cell_data, cell_index - 
		cell_data->switch_block_index);

	// Remove the switch block
	level->tiles[y_cell][x_cell] = cell_data->empty_cell_index;

	// Add an animation of the switch block
	add_animation(x_cell, y_cell, 0, 0, cell_index, -1, 40, 1, 0, 0, 0, 
		animations, sprites, rle_sprites, is_rle_sprite);
}



// Inserts a backward slash in front of each space in a filename, if one does
// not exist there already.
// ASSUMES the first character of filename is not a space
void fix_filename_escape_chars(char *filename)
{
	char new_str[2000] = "";
	int which_char_old;
	int which_char_new;
	int len = strlen(filename);

	for (which_char_old = 0; which_char_old < len; which_char_old++)
	{
		if (filename[which_char_old] == ' ')
		{
			strcat(new_str, "\\");
			which_char_new++;
		}
		new_str[which_char_new] = filename[which_char_old];
		which_char_new++;
	}

	strcpy(filename, new_str);
}
