/*
Simple Space Invaders-style game.
By: Danny Tenedorio
*/

// included headers
#include "allegro.h"
#include "stdio.h"
#include "stdlib.h"
#include "string.h"
#include "math.h"

// defined constants
#define FLUSH				while (getchar() != '\n')
#define DEBUG				TRUE
#define PLAYER_INDEX		1
#define STATUS_BAR_END		24
#define RIGHT				1
#define LEFT				2
#define ENEMY_X_STEP		10
#define ENEMY_Y_STEP		7
#define FONT1				0
#define FONT2				1
#define X_TO_PAN			(255 / (float)screen_x_size)
#define TITLE_FORE_COLOR	31
#define TITLE_BACK_COLOR    197
#define TITLE_SHADE_COLOR	195
#define MENU_BEGINS			83
#define MAX_LEVELS			5
#define MAX_LIVES			7

// global structures
typedef struct  {
	int x;
	int y;
	int width;
	int height;
	int sprite_index;
	int frame;
	int shots_left;
	int alive;
} SCREEN_OBJECT;

typedef struct  {
	int x;
	int y;
	int length;
	int col;
	int col_up;
	int alive;
} SHOT;

typedef struct  {
	int x;
	int y;
	int col;
	int speed;
} POINT;

// function prototype definitions
void init_beginning (void);
void init_level (void);
void load_palette (void);
void load_sprites (void);
void load_sounds (void);
void put_sprite (int x, int y, int which);
void title_screen (void);
void load_game (void);
void save_game (void);
void respond_to_input (void);
void clear_screen (void);
void update_screen (void);
void update_status_bar (void);
void update_variables (void);
void move_player_shots (void);
void erase_player_shots (void);
void draw_player_shots (void);
int total_enemy_shots (void);
int all_enemies_dead (void);
int in_box (int x_given, int y_given, SCREEN_OBJECT monster);
int actual_x_aliens (void);
int actual_y_aliens (void);
int actual_left_boundary (void);
int total_enemies (void);
void play_sound (int which, int the_pan);
void do_game_over (void);
void do_high_scores (void);
void show_high_scores (void);
void win_level (void);
void kill_player (void);
void draw_barriers (void);
void copy_buffer_to_screen (void);
void shrink_screen (void);
void show_rainbow (void);
void erase_save_files (void);
void get_text_line (FILE *stream, char *dest_str);
char make_window(int win_x1, int win_y1, int win_x2, int win_y2, int col1, int col2,
				 int col3, int col4, int text_color, char *message);

// global variables
int x, y, x1, x2, y2, index, index2, flag;
int screen_x_size, screen_y_size;
int draw_solid = TRUE;
unsigned long game_counter;
int total_player_shots;
int lives;
long score;
int level;
int shot_probability;
int enemy_wait_frames;
int enemy_which_direction;
int move_down;
int switch_dir;
int won_level;
int game_ended;
int space_down, esc_down;
int delay_msecs = 10;
int x_aliens, y_aliens;
int shot_switch;
int game_loaded;

// global objects
PALETTE pal;				// stores the current palette in memory
BITMAP *buffer;				// memory buffer to which we send the data
BITMAP *sprite[50];			// array that holds the sprite data
DATAFILE *dat;				// datafile object that holds the datafile info
FONT *game_font;			// font object to hold the nifty game font
FONT *title_font;			// font object to hold the nifty game font
SCREEN_OBJECT player;
SCREEN_OBJECT enemy[100];
SHOT player_shot[10];
SHOT enemy_shot[100];
SAMPLE *game_sound[20];
GFX_MODE_LIST *the_list;
int pan = 128;
int pitch = 1000;
int sound_voice[20];


int main (void)
{
	init_beginning();							// initialize everything
	
	while (TRUE)
	{
		lives = 3;
		score = 0;
		level = 1;
		game_ended = FALSE;
		game_loaded = FALSE;

		title_screen();

		while (game_ended == FALSE)
		{
			if (game_loaded == FALSE) 
				init_level ();	// set all the variables and fade the level in
			else  {
				copy_buffer_to_screen ();			// copy it all onscreen
				fade_in (pal, 5);					// fade it in
			}
			game_loaded = FALSE;

			while (won_level == FALSE)
			{
				respond_to_input ();			// respond to the actions of the player
				if (game_ended == TRUE) break;	// if the game has ended, exit the loop
				clear_screen ();				// clear the main playing area
				move_player_shots ();			
				update_variables ();			// move the enemies and the shots
				if (game_ended == TRUE) break;	// if the game has ended, exit the loop
				update_screen ();				// re-draw everything
				copy_buffer_to_screen ();		// copy it all onscreen
				rest(delay_msecs);				// pause 
			}
		}
	}
	
}

END_OF_MAIN ();								// who knows wtf this macro does, but i need it :/



// does all the initializing
void init_beginning (void)
{
	// initializations
	allegro_init();
	install_keyboard();
	install_timer();
	// install a digital sound driver
	reserve_voices (8, 0);
	if (install_sound(DIGI_AUTODETECT, MIDI_NONE, "") != 0) {
		allegro_message("Error initialising sound system\n%s\n", allegro_error);
		exit (1);
	}
	
	// init graphics mode
	screen_x_size = 320;
	screen_y_size = 240;
	set_color_depth (8);
	if (set_gfx_mode(GFX_AUTODETECT_FULLSCREEN, screen_x_size, screen_y_size, 0, 0) != 0)	{
		allegro_message("Error setting graphics mode\n%s\n", allegro_error);
		exit (1);
    }
	
	if (DEBUG == TRUE) chdir ("C:\\Danny\\Programs\\spaceInv\\Files\\");

	dat = load_datafile ("font.dat");	// load the font
	game_font = (FONT *)dat[FONT1].dat;
	title_font = (FONT *)dat[FONT2].dat;

	buffer = create_bitmap(screen_x_size, screen_y_size);	// make an off-screen buffer in system RAM
	clear_bitmap (buffer);				// clear the buffer to color 0
	clear_bitmap (screen);				// clear the screen to color 0
	load_palette ();					// load the game palette from file
	set_palette (pal);					// set the palette
	load_sprites ();					// load the sprites
	load_sounds();						// load the sounds
	
	srand (time (NULL));				// seed the random number generator
	
	erase_save_files ();				// erase the save files

	return;
} // init_beginning



// sets everything up to begin the level
void init_level (void)
{
	char level_word[10];
	char level_phrase[100];

	game_counter = 1;
	total_player_shots = 0;
	won_level = FALSE;
	draw_solid = FALSE;
		
	for (index = 0; index <= 5; index++)  {		// init the shots (player)
		player_shot[index].col = 0;
		player_shot[index].col_up = TRUE;
		player_shot[index].alive = TRUE;
	}
	
	for (index = 0; index <= 99; index++)  {	// init the shots (enemy)
		enemy_shot[index].col_up = TRUE;
		enemy_shot[index].alive = FALSE;
	}

	player.width = 25;							// init the player
	player.height = 25;
	player.x = (screen_x_size * 1 / 8) - 15 + 2;	// start the player underneath the leftmost barrier
	player.y = screen_y_size - player.height;
	player.frame = 0;
	player.alive = TRUE;
	
	if		(level == 1) { x_aliens = 4; y_aliens = 3; enemy_wait_frames = 400; shot_probability = 26; }
	else if (level == 2) { x_aliens = 5; y_aliens = 4; enemy_wait_frames = 300; shot_probability = 24; }
	else if (level == 3) { x_aliens = 6; y_aliens = 2; enemy_wait_frames = 200; shot_probability = 10; }
	else if (level == 4) { x_aliens = 8; y_aliens = 5; enemy_wait_frames = 370; shot_probability = 20; }
	else if (level == 5) { x_aliens = 9; y_aliens = 3; enemy_wait_frames = 270; shot_probability = 12; }
	else if (level == 6) { x_aliens = 9; y_aliens = 3; enemy_wait_frames = 200; shot_probability = 20; }

	for (y = 1; y <= y_aliens; y++)  {
		for (x = 1; x <= x_aliens; x++)  {
			index = x + ((y - 1) * x_aliens);
			enemy[index].alive = TRUE;
			enemy[index].sprite_index = 16 + (((x - 1) % 7) * 2);
			enemy[index].frame = 0;
			enemy[index].shots_left = 1;
			enemy[index].width = sprite[enemy[index].sprite_index]->w;
			enemy[index].height = sprite[enemy[index].sprite_index]->h;
			enemy[index].x = 5 + ((x - 1) * 31);
			enemy[index].y = STATUS_BAR_END + 2 + ((y - 1) * 27);
		}
	}

	enemy_which_direction = RIGHT;
	
	set_palette (black_palette);		// make the screen black until we fade everything in
	clear_bitmap (buffer);				// clear the buffer to color 0
	clear_bitmap (screen);				// clear the screen to color 0
		
	update_status_bar ();				// draw the status bar
	draw_barriers ();					// draw the barriers that protect the player
	
	if (level == 1)  strcpy(level_word, "One");		// determine which word represents which number
	if (level == 2)  strcpy(level_word, "Two");
	if (level == 3)  strcpy(level_word, "Three");
	if (level == 4)  strcpy(level_word, "Four");
	if (level == 5)  strcpy(level_word, "Five");
	if (level == 6)  strcpy(level_word, "Six");
	if (level == 7)  strcpy(level_word, "Seven");
	if (level == 8)  strcpy(level_word, "Eight");
	if (level == 9)  strcpy(level_word, "Nine");
	if (level == 10) strcpy(level_word, "Ten");

	strcpy(level_phrase, "Level ");
	strcat(level_phrase, level_word);
	strcat(level_phrase, "");
	textprintf_centre (buffer, game_font, screen_x_size / 2,	// display message: Level One 
		((screen_y_size - 55) / 2) - 4, 31, level_phrase);		//  (or Level Two or whatever)
	textprintf_centre (buffer, game_font, screen_x_size / 2,	// display message: Press any key
		((screen_y_size - 55) / 2) - 4 + 13, 31, "Press any key to begin");		//  to begin
	copy_buffer_to_screen ();			// copy it all onscreen
	fade_in (pal, 5);					// fade it in
	clear_keybuf();						// clear the keyboard buffer
	readkey ();							// wait until the user presses any key
	rectfill (buffer, 0, ((screen_y_size - 55) / 2) - 4, screen_x_size, ((screen_y_size - 55) / 2) - 4 + 25, 0);


} // init_level



// copies the contents of the off-screen buffer to the visible screen
void copy_buffer_to_screen (void)
{
	blit (buffer, screen, 0, 0, 0, 0, screen_x_size, screen_y_size);		// copy memory buffer to the screen
}



// loads the game's palette into the pal object from spaceInv.pal
void load_palette (void)
{
	FILE *the_file;								// file object for file i/o

	the_file = fopen("spaceInv.pal", "rb");		// open the file
	fread (&pal, sizeof(pal), 1, the_file);		// read the info from the file
	fclose (the_file);							// close the file
}



// draws a sprite which at location x, y
void put_sprite (int x, int y, int which)
{
	if (draw_solid == TRUE)	
		blit(sprite[which], buffer, 0, 0, x, y, sprite[which]->w, sprite[which]->h);
	else
		draw_sprite (buffer, sprite[which], x, y);
}




// responds to input
void respond_to_input (void)
{
	poll_keyboard();
	
	if (key[KEY_RIGHT])	 {							// if the right arrow key is pressed
		if ((player.x + player.width) < screen_x_size)  {		// if the player isn't too far to the right
			(player.x)++;							// move it one pixel to the right
		}
	}
	else if (key[KEY_LEFT])  { 						// if the left arrow key is pressed
		if (player.x > 0) 	{						// if the player isn't too far to the left
			(player.x)--;							// move it one pixel to the left
		}
	}
	if (key[KEY_SPACE])  {							// if the space key is pressed
		if (space_down == FALSE)  {					// and it was released last loop
			if (total_player_shots == 0)  {			// and no player shots are on screen
				total_player_shots++;				// init one...
				player_shot[total_player_shots].length = 8;
				player_shot[total_player_shots].x = player.x + 12;
				player_shot[total_player_shots].y = player.y - player_shot[total_player_shots].length;
				player_shot[total_player_shots].col = 39;
				player_shot[total_player_shots].col_up = TRUE;
				play_sound (2, player.x * X_TO_PAN);
			}
		}
		space_down = TRUE;
	}
	else
		space_down = FALSE;
	
	if (key[KEY_P])  {
		make_window ((screen_x_size / 2) - 30, (screen_y_size / 2) - 12 - (screen_y_size / 24),
					 (screen_x_size / 2) + 30, (screen_y_size / 2) + 12 - (screen_y_size / 24),
					 36, 0, 36, 18, 15, "Pause");
	}
	
	if (key[KEY_ESC])  {
		if (esc_down == FALSE)  {
		
				save_game();
				fade_out (5);
				won_level = TRUE;
				game_ended = TRUE;
			
		}
		esc_down = TRUE;
	}
	else  {
		esc_down = FALSE;
	}

	if (key[KEY_G] && DEBUG)  {
		//do_high_scores();
		do_game_over();
	}
	
}



// update the variables
void update_variables (void)
{
	player.frame = (player.x / 5) % 2;								// determine the player's sprite's frame
	
	game_counter++;													// advance the game counter
	if (game_counter > (ULONG_MAX)) game_counter = 1;

	// erase the aliens
	for (index = 1; index <= (x_aliens * y_aliens); index++)  {
		if (enemy[index].alive == TRUE)  {
			rectfill (buffer, enemy[index].x, enemy[index].y, enemy[index].x + enemy[index].width - 1,
				enemy[index].y + enemy[index].height - 1, 0);
		}
	}
	

	for (index = 1; index <= 80; index++)  {						// change/move the enemy shot(s)
	
		if (enemy_shot[index].alive == TRUE)  {

		if ((game_counter % 4) == 0)  {								// color cycle the shot
			if (enemy_shot[index].col_up == TRUE)  {
				(enemy_shot[index].col)++;
				if (enemy_shot[index].col == 63)
					enemy_shot[index].col_up = FALSE;
			}
			else  {
				(enemy_shot[index].col)--;
				if (enemy_shot[index].col == 55)
					enemy_shot[index].col_up = TRUE;
			}
		}

		(enemy_shot[index].y)++;									// move the shot

		if (enemy_shot[index].y + enemy_shot[index].length + 1 > screen_y_size)  {					// if the shot went too low
			enemy_shot[index].alive = FALSE;
		}
		else if (getpixel (buffer, enemy_shot[index].x, enemy_shot[index].y + enemy_shot[index].length)  != 0)  {	// if the shot hit a barrier
			if ((enemy_shot[index].x >= player.x) & (enemy_shot[index].x <= player.x + player.width) &
				(enemy_shot[index].y + enemy_shot[index].length >= player.y) &
				(enemy_shot[index].y + enemy_shot[index].length <= player.y + player.height))  {
				enemy_shot[index].alive = FALSE;
				kill_player();
				if (lives == 0)  {
					do_game_over();
				}
				break;
			}
			flag = FALSE;
			if (enemy_shot[index].y + enemy_shot[index].length < screen_y_size - 60) flag = TRUE;
			if (flag == FALSE)  {
				vline (buffer, enemy_shot[index].x, enemy_shot[index].y + enemy_shot[index].length, 		// make a hole in the barrier
						enemy_shot[index].y + enemy_shot[index].length + ((rand() % 2) + 2), 0);
				play_sound (5, enemy_shot[index].x * X_TO_PAN);
				enemy_shot[index].alive = FALSE;
			}
		}
		else  {

		}

		}

	} // move shots

	// if the enemies advanced too far..
	if (enemy[actual_y_aliens() * x_aliens].y + enemy[actual_y_aliens() * x_aliens].height > player.y + 2)  {
		kill_player();
		fade_out (5);
		won_level = TRUE;
	}
	
	if (game_ended == TRUE) return;

	rectfill (buffer, player.x, player.y, player.x + player.width, player.y + player.height, 0);	// erase the player

	if (all_enemies_dead())												// if no more enemies are left
		win_level ();													// win the level
	else if (((rand() % shot_probability) == 0) && (total_enemy_shots () < 80) &&
		(enemy[index].shots_left == 0) )  {	// generate a new enemy shot
		do {
			index = (rand() % (x_aliens * y_aliens)) + 1;
		} while (enemy[index].alive == FALSE);
		flag = FALSE;
		for (index2 = 1; (index2 <= 80) || (flag == TRUE); index2++)  {
			if (enemy_shot[index2].alive == FALSE)  {
				break;
			}
		}
		enemy[index].shots_left =	9;
		enemy_shot[index2].alive =	TRUE;
		enemy_shot[index2].x =		enemy[index].x + (enemy[index].width / 2) + (rand() % 3) - 1;
		enemy_shot[index2].y =		enemy[index].y + (enemy[index].height / 2);
		enemy_shot[index2].col =	55;
		enemy_shot[index2].col_up =	TRUE;
		enemy_shot[index2].length =	8;
		if (shot_switch == TRUE)  {
			play_sound (4, enemy_shot[index].x * X_TO_PAN);
			shot_switch = FALSE;
		}
		else  {
			play_sound (7, enemy_shot[index].x * X_TO_PAN);
			shot_switch = TRUE;
		}
		
		//textprintf(screen, font, 1, 1, 15, "total shots: %d  index: %d", total_enemy_shots (), index);
		//readkey();
	}

	
	for (index = 1; index <= (x_aliens * y_aliens); index++)  {		// erase the aliens
		if (enemy[index].alive == TRUE)  {
			draw_character(buffer, sprite[enemy[index].sprite_index + enemy[index].frame],
							enemy[index].x, enemy[index].y, 0);
		}
	}

	
	// change/move the aliens
	if ((game_counter % (enemy_wait_frames / 10)) == 0)  {	// if the time has come
		switch_dir = FALSE;
		if ((enemy_which_direction == RIGHT) & 
			((enemy[actual_x_aliens()].x + enemy[actual_x_aliens()].width + ENEMY_X_STEP) > screen_x_size) ||
			((enemy_which_direction == LEFT) & ((enemy[actual_left_boundary()].x - ENEMY_X_STEP) < 0)))  {
			switch_dir = TRUE;
		}
		for (y = 1; y <= y_aliens; y++)  {							// move the aliens
			for (x = 1; x <= x_aliens; x++)  {
				index = x + ((y - 1) * x_aliens);
				enemy[index].shots_left--;
				if (enemy[index].shots_left < 0) enemy[index].shots_left = 0;
				if (switch_dir == TRUE)  {
					enemy[index].y += ENEMY_Y_STEP;
				}
				else  {
					if (enemy_which_direction == RIGHT)
						enemy[index].x += ENEMY_X_STEP;
					else
						enemy[index].x -= ENEMY_X_STEP;
				}
				if (enemy[index].frame == 0)
					enemy[index].frame = 1;
				else
					enemy[index].frame = 0;
			}
		}
		if (switch_dir == TRUE)  {
			if (enemy_which_direction == RIGHT)
				enemy_which_direction = LEFT;
			else
				enemy_which_direction = RIGHT;
		}
	}



} // update_variables



// clears the sprites, shots, etc. from the screen
void clear_screen (void)
{
	
	erase_player_shots ();

	for (index = 1; index <= 80; index++)  {						// erase the enemys shot(s)
		if (enemy_shot[index].alive == TRUE)  {
			vline (buffer, enemy_shot[index].x, enemy_shot[index].y, 
						   enemy_shot[index].y + enemy_shot[index].length, 0);
		}
	}

} // clear_screen



// redraws everything on the screen
void update_screen (void)
{
	put_sprite (player.x, player.y, PLAYER_INDEX + player.frame);	// draw the player

	draw_player_shots ();

	for (index = 1; index <= 80; index++)  {		// draw the enemy's shot(s)
		if (enemy_shot[index].alive == TRUE)  {
			vline (buffer, enemy_shot[index].x, enemy_shot[index].y, 
			   enemy_shot[index].y + enemy_shot[index].length, enemy_shot[index].col);
		}
	}

	for (index = 1; index <= (x_aliens * y_aliens); index++)  {		// draw the aliens
		if (enemy[index].alive == TRUE)
			put_sprite (enemy[index].x, enemy[index].y, enemy[index].sprite_index + enemy[index].frame);
	}

//	textprintf (buffer, game_font, 0, 0, 15, "tps: %d", total_player_shots);


} // update_screen



// updates the status bar at the top of the screen
void update_status_bar (void)
{
	long digit;
	long value;
	int denominator;
	int num_digits = 7;

	// display the score
	for (index = 1; index <= num_digits; index++)  {		
		if (index == 1) denominator = 1; else denominator = pow(10, index - 1);
		value = score / denominator;
		digit = value % 10;												// get the digit
		if ((score < pow(10, index - 1)) & (index > 1)) digit = -1;		// change leading zeros to empty space
		put_sprite ((num_digits * 19) - (index * 19) + 10, 0, 4 + digit);	// draw the sprite
	}
	
	// display the lives
	rectfill (buffer, 160, 0, screen_x_size, 30, 0);					// clear the lives area
	for (index = 1; index <= lives; index++)  {		
		put_sprite (screen_x_size - (index * 22), 1, 14);				// draw the sprite
	}


} // update_status_bar



// draw the barriers
void draw_barriers (void)
{
	if (level == 1 || level == 1)  {
		put_sprite ((screen_x_size * 1 / 8) - 14, screen_y_size - 50, 15);
		put_sprite ((screen_x_size * 3 / 8) - 14, screen_y_size - 50, 15);
		put_sprite ((screen_x_size * 5 / 8) - 14, screen_y_size - 50, 15);
		put_sprite ((screen_x_size * 7 / 8) - 14, screen_y_size - 50, 15);
	}
	else if (level == 2 || level == 3 || level == 5)  {
		put_sprite ((screen_x_size * 1 / 6) - 14, screen_y_size - 50, 15);
		put_sprite ((screen_x_size * 3 / 6) - 14, screen_y_size - 50, 15);
		put_sprite ((screen_x_size * 5 / 6) - 14, screen_y_size - 50, 15);
	}
	else if (level == 4)  {
		put_sprite ((screen_x_size * 1 / 8) - 14, screen_y_size - 50, 15);
		put_sprite ((screen_x_size * 7 / 8) - 14, screen_y_size - 50, 15);
	}

}



// returns TRUE if the given point is within the boundaries of the monster.
int in_box (int x_given, int y_given, SCREEN_OBJECT monster)
{
	if ((x_given >= monster.x) & (x_given <= (monster.x + monster.width)) &
		(y_given >= monster.y) & (y_given <= (monster.y + monster.height)))
		return TRUE;
	else
		return FALSE;
}



// returns how many enemy shots are on screen at any time
int total_enemy_shots (void)
{
	int	number = 0;

	for (index = 0; index <= 80; index++)  {
		if (enemy_shot[index].alive == TRUE) number++;
	}
		
	return number;
}



// returns TRUE if all enemies are dead; FALSE if not
int all_enemies_dead (void)
{
	for (index = 0; index <= (x_aliens * y_aliens); index++)
		if (enemy[index].alive == TRUE) return FALSE;

	return TRUE;

} // all_enemies_dead



// run this function after all the enemies are dead to advance to the next level
void win_level (void)
{
	rectfill (buffer, 0, STATUS_BAR_END + 1, screen_x_size, screen_y_size - 56, 0); // clear the main area
	update_screen ();			// update the screen
	copy_buffer_to_screen();	// copy the buffer to the screen
	fade_out (5);				// fade to black
	level++;					// add 1 to the level
	won_level = TRUE;			// tell the main loop that we won the level
	
}



// returns the amount of aliens in the longest row. used to determine when the enemies
// should switch direction
int actual_x_aliens (void)
{
	int biggest_row = 0;
	int current_row = 0;
	for (y = 1; y <= y_aliens; y++)  {
		for (x = 1; x <= x_aliens; x++)  {
			index = x + ((y - 1) * x_aliens);
			if (enemy[index].alive == TRUE)  {
				current_row = x;
				if (current_row > biggest_row) biggest_row = current_row;
			}
		}
	}

	return biggest_row;
} // actual_x_aliens




// returns the amount of aliens in the longest column. used to determine when the enemies
// have taken over
int actual_y_aliens (void)
{
	int biggest_column = 0;
	int current_column = 0;
	for (x = 1; x <= x_aliens; x++)  {
		for (y = 1; y <= y_aliens; y++)  {
			index = x + ((y - 1) * x_aliens);
			if (enemy[index].alive == TRUE)  {
				current_column = y;
				if (current_column > biggest_column) biggest_column = current_column;
			}
		}
	}

	return biggest_column;
} // actual_x_aliens



// returns the leftmost alien. used to determine when the enemies
// should switch direction
int actual_left_boundary (void)
{
	int smallest_row = x_aliens;
	int current_row = 0;
	for (y = 1; y <= y_aliens; y++)  {
		for (x = x_aliens; x >= 1; x--)  {
			index = x + ((y - 1) * x_aliens);
			if (enemy[index].alive == TRUE)  {
				current_row = x;
				if (current_row < smallest_row) smallest_row = current_row;
			}
		}
	}

	return smallest_row;
} // actual_left_boundary



// 
void kill_player (void)
{
	int the_color;
	int which_shot;
	
	lives--;												// decrease lives by 1
	update_status_bar();									// update the status bar
	play_sound (6, (player.x + (player.width / 2)) * X_TO_PAN);	// play death sound
	
	update_screen();
	for (the_color = 63; the_color >= 48; the_color--)  {		// fade the player from white to green to black
		draw_character (buffer, sprite[PLAYER_INDEX], player.x, player.y, the_color);
		copy_buffer_to_screen();
		vsync(); vsync();
	}

	draw_character (buffer, sprite[PLAYER_INDEX], player.x, player.y, 0);	// erase the player
	copy_buffer_to_screen();								// copy the buffer to the screen
	rest (1000);											// wait for 2 seconds
	
	clear_screen();											// erase the shots from the screen

	for (which_shot = 1; which_shot <= 80; which_shot++)	// kill all enemy shots
		enemy_shot[which_shot].alive = FALSE;

	player_shot[0].alive = FALSE;							// kill the player's shot
	player_shot[1].alive = FALSE;							// kill the player's shot
	total_player_shots = 0;
}	



// display all the colors in the current palette
void show_rainbow (void)
{
	for (x = 0; x < 255; x++)
		rect (screen, x, 0, x, screen_y_size, x);	
}



// returns the amount of living enemies on screen
int total_enemies (void)
{
	int num_enemies = 0;

	for (y = 1; y <= y_aliens; y++)  {
		for (x = 1; x <= x_aliens; x++)  {
			index = x + ((y - 1) * x_aliens);	
			if (enemy[index].alive == TRUE) num_enemies++;
		}
	}

	return num_enemies;
}



// game over sequence
void do_game_over (void)
{
	draw_player_shots();				// re-draw everything
	copy_buffer_to_screen ();		// copy it all onscreen
	shrink_screen();
	clear_bitmap (buffer);
	set_palette (black_palette);

	textprintf_centre (buffer, game_font, screen_x_size / 2, (screen_y_size / 2) - 36, 31,
		"Game Over!");
	textprintf_centre (buffer, game_font, screen_x_size / 2, (screen_y_size / 2) - 12, 199,
		"The alien invasion was successful.");
	textprintf_centre (buffer, game_font, screen_x_size / 2, (screen_y_size / 2) + 0,  198,
		"Once the enemy forces found your head,");
	textprintf_centre (buffer, game_font, screen_x_size / 2, (screen_y_size / 2) + 12, 197,
		"they put it on display as a warning to");
	textprintf_centre (buffer, game_font, screen_x_size / 2, (screen_y_size / 2) + 24, 196,
		"any other planets that might get in ");
	textprintf_centre (buffer, game_font, screen_x_size / 2, (screen_y_size / 2) + 36, 195,
		"their way.");

	put_sprite ((screen_x_size / 2) - 12, (screen_y_size / 2) - 68, 20);
	
	copy_buffer_to_screen();
	
	fade_in (pal, 5);
	play_sound (3, 255 / 2);
	clear_keybuf();
	readkey();
	fade_out (5);
	clear_bitmap (screen);
	set_palette (black_palette);

	game_ended = TRUE;

	do_high_scores();
}



// shrink the screen to nothing
void shrink_screen (void)
{
	int w, h, counter;

	for (counter = 0; counter < (screen_x_size / 2); counter++)  {
		x = counter;
		y = counter * screen_y_size / screen_x_size;
		w = screen_x_size - (counter * 2);
		h = w * screen_y_size / screen_x_size;
		stretch_blit(buffer, screen, 0, 0, screen_x_size, screen_y_size, x, y, w, h);
		rect (screen, x, y, x + w, y + h, 0);
		rect (screen, 0, y + h + 1, screen_x_size - 1, screen_y_size - 1, 0);
		rest (2);
	}
}



// loads the sprites from the specified files
void load_sprites (void)
{
	int total_files;
	char *filename[30];
	PALETTE junkPal;
				
	filename[1] =  "player1.bmp";
	filename[2] =  "player2.bmp";
	filename[3] =  "num.bmp";
	filename[4] =  "num0.bmp";
	filename[5] =  "num1.bmp";
	filename[6] =  "num2.bmp";
	filename[7] =  "num3.bmp";
	filename[8] =  "num4.bmp";
	filename[9] =  "num5.bmp";
	filename[10] = "num6.bmp";
	filename[11] = "num7.bmp";
	filename[12] = "num8.bmp";
	filename[13] = "num9.bmp";
	filename[14] = "plyricon.bmp";
	filename[15] = "barrier.bmp";
	filename[16] = "enemy11.bmp";
	filename[17] = "enemy12.bmp";
	filename[18] = "enemy21.bmp";
	filename[19] = "enemy22.bmp";
	filename[20] = "enemy31.bmp";
	filename[21] = "enemy32.bmp";
	filename[22] = "enemy41.bmp";
	filename[23] = "enemy42.bmp";
	filename[24] = "enemy51.bmp";
	filename[25] = "enemy52.bmp";
	filename[26] = "enemy71.bmp";
	filename[27] = "enemy72.bmp";
	filename[28] = "enemy61.bmp";
	filename[29] = "enemy62.bmp";
	total_files = 29;

	for (index = 1; index <= total_files; index++)		// go through all the bitmap files
		sprite[index] = load_bitmap(filename[index], junkPal);	// load sprite in bitmap format

} // load_sprites



void load_sounds (void)
{
	game_sound[1] = load_sample("pop.wav");
	game_sound[2] = load_sample("whistle.wav");
	game_sound[3] = load_sample("gameover.wav");
	game_sound[4] = load_sample("badshot.wav");
	game_sound[5] = load_sample("plunk.wav");
	game_sound[6] = load_sample("death.wav");
	game_sound[7] = load_sample("badshot.wav");

	for (index = 1; index <= 7; index++)  {
		sound_voice[index] = allocate_voice (game_sound[index]);
	}


		
}



void play_sound (int which, int the_pan)
{
	// play the sample
	voice_stop (sound_voice[which]);				// stop the sound if it's already playing
	voice_set_position(sound_voice[which], 0);		// start at the beginning of the sound
	voice_set_pan (sound_voice[which], the_pan);	// set which part of the sound comes from which speaker
	voice_start (sound_voice[which]);				// play the sound
}



char make_window(int win_x1, int win_y1, int win_x2, int win_y2, int col1, int col2, int col3, int col4, int text_color, char *message)
{
	int counter, temp_var, the_y1;
	int win_width, win_height;
	char a_char;
	BITMAP *saved_area;
	
	if (win_x1 > win_x2)  {		
		temp_var = win_x2;			// swaps win_x1 and win_x2
		win_x2 = win_x1;
		win_x1 = temp_var;
	}
	if (win_y1 > win_y2)  {
		temp_var = win_y2;			// swaps win_y1 and win_y2
		win_y2 = win_y1;
		win_y1 = temp_var;
	}
	
	win_width = win_x2 - win_x1;	// determine window width
	win_height = win_y2 - win_y1;	// determine window height

	saved_area = create_bitmap (win_width, win_height + 1);
	blit (buffer, saved_area, win_x1, win_y1, 0, 0, win_width, win_height + 1);

	// open the window	
	for (counter = win_width / 2 - (2 * win_width / win_height); counter >= 1; counter -= 2)  {
		x1 = win_x1 + counter;
		the_y1 = win_y1 + (counter * win_height) / win_width;
		x2 = win_x2 - (counter);
		y2 = win_y2 - (counter * win_height) / win_width;
		
		// colors: 124, 122, 120, 159
		rect	 (buffer, x1 + 0, the_y1 + 0, x2 - 0, y2 - 0, col1);
		rect	 (buffer, x1 + 1, the_y1 + 1, x2 - 1, y2 - 1, col2);
		rect	 (buffer, x1 + 2, the_y1 + 2, x2 - 2, y2 - 2, col3);
		rectfill (buffer, x1 + 3, the_y1 + 3, x2 - 3, y2 - 3, col4);
		
		copy_buffer_to_screen();
		rest (1);
	}

	text_mode (-1);		// set opaque text mode
	textout (buffer, game_font, message, win_x1 + 13, win_y1 + 7, text_color);

	copy_buffer_to_screen();

	clear_keybuf();
	a_char = readkey();
	while (keypressed()) { }

	// close the window	
	for (counter = 1; counter <= win_width / 2 - (2 * win_width / win_height); counter += 2)  {
		x1 = win_x1 + counter;
		the_y1 = win_y1 + (counter * win_height) / win_width;
		x2 = win_x2 - (counter);
		y2 = win_y2 - (counter * win_height) / win_width;
		
		// colors: 124, 122, 120, 159
		
		blit(saved_area, buffer, 0, 0, win_x1, win_y1, saved_area->w, saved_area->h);
		rect	 (buffer, x1 + 0, the_y1 + 0, x2 - 0, y2 - 0, col1);
		rect	 (buffer, x1 + 1, the_y1 + 1, x2 - 1, y2 - 1, col2);
		rect	 (buffer, x1 + 2, the_y1 + 2, x2 - 2, y2 - 2, col3);
		rectfill (buffer, x1 + 3, the_y1 + 3, x2 - 3, y2 - 3, col4);
		
		copy_buffer_to_screen();
		rest (1);
	}

	// paste the window area back
	blit(saved_area, buffer, 0, 0, win_x1, win_y1, saved_area->w, saved_area->h);
	
	copy_buffer_to_screen();


	destroy_bitmap (saved_area);
	clear_keybuf();
	return a_char;

}



void move_player_shots (void)
{
	int counter;	
	
	for (index = 1; index <= total_player_shots; index++)  {		// change/move the player's shot(s)
		
		if ((game_counter % 4) == 0)  {								// color cycle the shot
			if (player_shot[index].col_up == TRUE)  {
				(player_shot[index].col)++;
				if (player_shot[index].col >= 47)
					player_shot[index].col_up = FALSE;
			}
			else  {
				(player_shot[index].col)--;
				if (player_shot[index].col <= 39)
					player_shot[index].col_up = TRUE;
			}
		}

		for (counter = 1; counter <= 2; counter++)  {
		
			(player_shot[index].y) --;									// move the shot

		// scan to see if two shots have collided head-on
		for (index2 = 1; index2 <= 80; index2++)  {
			if (enemy_shot[index2].alive == TRUE)  {
				if ((player_shot[index].x == enemy_shot[index2].x) &
					((player_shot[index].y >= enemy_shot[index2].y) &
					(player_shot[index].y <= enemy_shot[index2].y + enemy_shot[index2].length)))  {
					// if we got here, two shots collided head-on
					player_shot[index].alive = FALSE;				// kill the player's shot
					enemy_shot[index2].alive = FALSE;				// kill the enemy's shot
					score += 10;									// add 10 to the score
					update_status_bar ();							// update the status bar
				}
			}
		}
		
		if (player_shot[index].y < STATUS_BAR_END)  				// if the shot went too high
			total_player_shots--;									// kill the shot
		else if (getpixel (buffer, player_shot[index].x, player_shot[index].y)  != 0)  {	// if the shot hit a barrier
			flag = FALSE;
			for (index2 = 1; (index2 <= (x_aliens * y_aliens)) & (flag == FALSE); index2++)  {
				if ((enemy[index2].alive == TRUE) & (in_box (player_shot[index].x,
					// index refers to player_shot[]; ndex2 refers to enemy[]
					player_shot[index].y, enemy[index2])))  {		// the shot hit an enemy
					play_sound (1, player_shot[index].x * X_TO_PAN);
					total_player_shots--;							// kill the shot
					enemy[index2].alive = FALSE;					// kill the enemy
					draw_character(buffer, sprite[enemy[index2].sprite_index + enemy[index2].frame],
						enemy[index2].x, enemy[index2].y, 0);
					score += 100;									// add 100 to the scorezorz
					update_status_bar ();							// update the status bar
					flag = TRUE;

					if (total_enemies() == 5) enemy_wait_frames = 110;
					if (total_enemies() == 3) enemy_wait_frames = 80;
					if (total_enemies() == 1) enemy_wait_frames = 50;
				}
			}
			if (flag == FALSE)  {
				srand (time (NULL));											// seed the random number generator
				vline (buffer, player_shot[index].x, player_shot[index].y -		// make a hole in the barrier
						((rand() % 2) + 1),	player_shot[index].y, 0);		
				player_shot[index].alive = FALSE;
				play_sound (5, player_shot[index].x * X_TO_PAN);
				total_player_shots--;									// kill the shot
			}
		}
		else  {

		}

		if (total_player_shots <= 0) break;
		
		}

	} // move player shots

}



void erase_player_shots (void)
{
	for (index = 1; index <= total_player_shots; index++)  {		// erase the player's shot(s)
		vline (buffer, player_shot[index].x, player_shot[index].y, 
		   player_shot[index].y + player_shot[index].length, 0);
	}
}



void draw_player_shots (void)
{
	for (index = 1; index <= total_player_shots; index++)  {		// draw the player's shot(s)
		vline (buffer, player_shot[index].x, player_shot[index].y, 
			player_shot[index].y + player_shot[index].length, player_shot[index].col);
	}
}



// get, show high scores
void do_high_scores (void)
{
	// local variables
	int curKey, curLetter;
	int counter;
	int temp_int;
	char str[100];
	char name_entered[40];
	char player_name[16][40];
	long player_score[16];
	FILE *fp_highscores;

	// clear the screen
	clear_bitmap (buffer);
	
	// get the high scores
	fp_highscores = fopen ("hiscores.lst", "a");			// create the file if it does not exist 
	fclose (fp_highscores);									// close the file
	fp_highscores = fopen ("hiscores.lst", "r");			// open the file for input
	if (fscanf(fp_highscores, "%s", str) == EOF)  {			// if the file is empty:
		fclose (fp_highscores);								// close the file
		fp_highscores = fopen ("hiscores.lst", "w");		// open the file for output
		// write default high scores
		fprintf (fp_highscores, "Johnny the Wild Axe Man\n"			"10000\n");
		fprintf (fp_highscores, "Robert and His Camels\n"			"9000\n");
		fprintf (fp_highscores, "William the Racecar Driver\n"		"8000\n");
		fprintf (fp_highscores, "Jeremy the Mushroom Picker\n"		"7000\n");
		fprintf (fp_highscores, "Kevin and the Cool Kids\n"			"6000\n");
		fprintf (fp_highscores, "Andrew the Magnificent\n"			"5000\n");
		fprintf (fp_highscores, "Horatio's Band of Merry Men\n"		"4000\n");
		fprintf (fp_highscores, "Valerie's Dwarf Tossing Brigade\n"	"3000\n");
		fprintf (fp_highscores, "Josh the Video Store Owner\n"		"2000\n");
		fprintf (fp_highscores, "Aaron and the Shoeshiners\n"		"1000\n");
		fprintf (fp_highscores, "Skip and Company\n"				"900\n");
		fprintf (fp_highscores, "Albert the Fish Gulper\n"			"800\n");
		fprintf (fp_highscores, "Jethro the Mighty\n"				"700\n");
		fprintf (fp_highscores, "Zeke and Ryan\n"					"600\n");
		fprintf (fp_highscores, "Moe the Glass Blower\n"			"500\n");
	}
	fclose (fp_highscores);									// close the file
	fp_highscores = fopen ("hiscores.lst", "r");			// open the file for input
	for (index = 1; index <= 15; index++)  {				// read the scores from file
		get_text_line (fp_highscores, player_name[index]);	// read the player's name
		get_text_line (fp_highscores, str);					// read the player's score
		player_score[index] = strtol(str, NULL, 0);			// convert the score string to long integer
	}
	fclose (fp_highscores);									// close the file

	if (score > player_score[15])  {
	
		// display messages
		textprintf_centre (buffer, game_font, (screen_x_size / 2), (screen_y_size / 2) - 36, 15, "Congratulations!");
		textprintf_centre (buffer, game_font, (screen_x_size / 2), (screen_y_size / 2) - 24, 15, "You made the high score list.");
		textprintf_centre (buffer, game_font, (screen_x_size / 2), (screen_y_size / 2) - 2, 15, "Please enter your name:");

		copy_buffer_to_screen();
		fade_in (pal, 5);									// fade in
	
		clear_keybuf();										// clear the keyboard buffer

		// get the player's name here
		x = 203; y = 71;
		curLetter = 0;
		strcpy (str, "");

		while ((curKey >> 8) != KEY_ENTER)  {

			rectfill (buffer, 0, (screen_y_size / 2) + 12, screen_x_size, (screen_y_size / 2) + 24, 0);			// clear the area
			textout_centre (buffer, game_font, str, (screen_x_size / 2), (screen_y_size / 2) + 12, 15);

			blit (buffer, screen, 0, 0, 0, 0, 320, 200);	// copy memory buffer to the screen

			curKey = readkey();
			if (											// if the key pressed is
				(((curKey >> 8)   >= KEY_A) & ((curKey >> 8)   <= KEY_Z)) |		// between A-Z, or
				(((curKey & 0xff) >= '0')   & ((curKey & 0xff) <= '9'))   |		// between 0-9, or
				(((curKey & 0xff) == '-')) |									// a hyphen, or
				(((curKey & 0xff) == ' ')) |									// a hyphen, or
				(((curKey >> 8)	  == KEY_STOP) & (strchr(str, (int) '.') == 0)))  {	// a period
				if (curLetter < 36)  {						// if the string < 36 characters...
					// curKey = tolower (curKey);
					str[curLetter] = (char) curKey;			// add the letter to the string
					str[curLetter + 1] = '\0';				// end the string after the letter
					curLetter++;
				}
			}
			else if (((curKey >> 8) == KEY_BACKSPACE) & (curLetter > 0)) {
				curLetter--;
				str[curLetter] = '\0';						// end the string after the letter
			}
		} 
		
		strcpy (name_entered, str);							// copy the string entered into the filename variable
	}

	
	// add player's name and score to high score table, then sort the table from highest to lowest score
	strcpy (player_name[16], name_entered);					
	player_score[16] = score;
	for (counter = 1; counter <= 16; counter++)  {
		for (index = 1; index <= 15; index++)  {
			if (player_score[index] < player_score[index + 1])  {		// if the 2 scores are out of order:
				strcpy (str, player_name[index]);						// swap player_name[index] with
				strcpy (player_name[index], player_name[index + 1]);	// player_name[index + 1]
				strcpy (player_name[index + 1], str);
				temp_int = player_score[index];							// swap player_score[index] with
				player_score[index] = player_score[index + 1];			// player_score[index + 1]
				player_score[index + 1] = temp_int;
			}
		}
	}

	// write the high scores to file
	fp_highscores = fopen ("hiscores.lst", "w");				// open the file for output
	for (index = 1; index <= 15; index++)  {					
		fprintf (fp_highscores, player_name[index]);			// write the person's name
		fprintf (fp_highscores, "\n");							// go to the next line
		fprintf (fp_highscores, "%ld", player_score[index]);	// write the person's score
		fprintf (fp_highscores, "\n");							// go to the next line
	}
	fclose (fp_highscores);										// close the file

	show_high_scores();	

	return;
}



// reads one line of text from a file and puts it into dest_str
void get_text_line (FILE *stream, char *dest_str)
{
	char the_char;
	char the_str[60] = "";
	int which_char;

	for (which_char = 0; which_char < 59; which_char++)  {
		the_char = getc (stream);							// read one character from the input stream
		if (the_char == '\n')  {							// if we've reached the end of the line:
			the_str[which_char] = '\0';							// end the string
			break;											// exit the loop
		}
		else  {												// if we're still inputting characters:
			the_str[which_char] = the_char;						// add each character to the string
		}
	}
	
	strcpy (dest_str, the_str);								// copy the string we're reading to dest_str
}



void title_screen (void)
{
	char menu_line[15][50];
	int the_key;
	int menu_choice = 1;
	int chosen_choice = 0;
	int col;
	int total_menu_choices;
	int save_file_exists = TRUE;
	BITMAP *version_bmp;
	
	// figure out if the save file is empty or not
	fp_savefile = fopen("game.sav", "ab");					// open save file for append
	fclose (fp_savefile);
	fp_savefile = fopen("game.sav", "rb");					// open save file for input
	if (feof(fp_savefile)) {
		save_file_exists = FALSE;
		return;
	}
	fclose (fp_savefile);

	clear_bitmap (buffer);

	set_palette (black_palette);
	
	// display the title, before formatting and style effects
	textprintf_centre (buffer, title_font, (screen_x_size / 2), 20, TITLE_FORE_COLOR, "Galaxy Invaders");
	
	// add lower darker edge to font
	for (x = 0; x <= screen_x_size; x++)  {
		for (y = 20; y <= 100; y++)  {
			if (((getpixel (buffer, x, y - 5) == TITLE_FORE_COLOR) || (getpixel (buffer, x, y - 4) == TITLE_FORE_COLOR) ||
				(getpixel (buffer, x, y - 3) == TITLE_FORE_COLOR) || (getpixel (buffer, x, y - 2) == TITLE_FORE_COLOR) ||
				(getpixel (buffer, x, y - 1) == TITLE_FORE_COLOR)) && getpixel (buffer, x, y) != TITLE_FORE_COLOR)  {
				putpixel (buffer, x, y, TITLE_BACK_COLOR);
			}
		}
	}

	// add shading around font's edges
	for (y = 20; y <= 100; y++)  {
		for (x = 0; x <= screen_x_size; x++)  {
			if (getpixel (buffer, x, y + 1) == TITLE_FORE_COLOR && getpixel (buffer, x, y) != TITLE_FORE_COLOR && getpixel (buffer, x, y) != TITLE_SHADE_COLOR)  {
				putpixel (buffer, x, y + 1, TITLE_SHADE_COLOR);
			}
			if (getpixel (buffer, x, y - 1) == TITLE_FORE_COLOR && getpixel (buffer, x, y) != TITLE_FORE_COLOR)  {
				putpixel (buffer, x, y - 1, TITLE_SHADE_COLOR);
			}
			if (getpixel (buffer, x + 1, y) == TITLE_FORE_COLOR && getpixel (buffer, x, y) != TITLE_FORE_COLOR && getpixel (buffer, x, y) != TITLE_SHADE_COLOR)  {
				putpixel (buffer, x + 1, y, TITLE_SHADE_COLOR);
			}
			if (getpixel (buffer, x - 1, y) == TITLE_FORE_COLOR && getpixel (buffer, x, y) != TITLE_FORE_COLOR)  {
				putpixel (buffer, x - 1, y, TITLE_SHADE_COLOR);
			}
		}
	}
	
	version_bmp = load_bitmap ("version.bmp", NULL);
	draw_sprite (buffer, version_bmp, (screen_x_size / 2) - 64, 70);

	// set menu choices
	strcpy (menu_line[1], "Start New Game");
	strcpy (menu_line[2], "Continue Game");
	strcpy (menu_line[3], "See High Scores");
	strcpy (menu_line[4], "Adjust Speed: %d msec delay");
	strcpy (menu_line[5], "Start On Level: %d");
	strcpy (menu_line[6], "Start With Lives: %d");
	strcpy (menu_line[7], "Exit to OS");
	total_menu_choices = 7;
	
	for (index = 1; index <= total_menu_choices; index++)  {
		if (menu_choice == index) col = 192; else col = 15;
		if (menu_choice == 2 && save_file_exists == FALSE) col = 20;
		if (index == 4)  {
			textprintf_centre (buffer, game_font, (screen_x_size / 2), MENU_BEGINS + (index * 13), col, menu_line[index], delay_msecs);
		}
		else if (index == 5)  {
			textprintf_centre (buffer, game_font, (screen_x_size / 2), MENU_BEGINS + (index * 13), col, menu_line[index], level);
		}  
		else if (index == 6)  {
			textprintf_centre (buffer, game_font, (screen_x_size / 2), MENU_BEGINS + (index * 13), col, menu_line[index], lives);
		}  
		else  {
			textprintf_centre (buffer, game_font, (screen_x_size / 2), MENU_BEGINS + (index * 13), col, menu_line[index]);
		}
	}
	copy_buffer_to_screen();
	fade_in (pal, 5);	// fade in

	while (chosen_choice == 0)  {
		for (index = 1; index <= total_menu_choices; index++)  {
			if (menu_choice == index) col = 192; else col = 15;
			if (menu_choice == 2 && save_file_exists == FALSE) col = 20;
			if (index == 4)  {
				rectfill (buffer, 0, MENU_BEGINS + (index * 13), screen_x_size, MENU_BEGINS + (index * 13) + 12, 0);
				textprintf_centre (buffer, game_font, (screen_x_size / 2), MENU_BEGINS + (index * 13), col, menu_line[index], delay_msecs);
			}  
			else if (index == 5)  {
				rectfill (buffer, 0, MENU_BEGINS + (index * 13), screen_x_size, MENU_BEGINS + (index * 13) + 12, 0);
				textprintf_centre (buffer, game_font, (screen_x_size / 2), MENU_BEGINS + (index * 13), col, menu_line[index], level);
			}  
			else if (index == 6)  {
				rectfill (buffer, 0, MENU_BEGINS + (index * 13), screen_x_size, MENU_BEGINS + (index * 13) + 12, 0);
				textprintf_centre (buffer, game_font, (screen_x_size / 2), MENU_BEGINS + (index * 13), col, menu_line[index], lives);
			}  
			else  {
				textprintf_centre (buffer, game_font, (screen_x_size / 2), MENU_BEGINS + (index * 13), col, menu_line[index]);
			}
		}
		copy_buffer_to_screen();

		clear_keybuf();
		the_key = readkey();

		if ((the_key >> 8) == KEY_UP)  {
			menu_choice--;
			if (menu_choice == 2 && save_file_exists == FALSE) menu_choice--;
			if (menu_choice < 1) menu_choice = 1;
		}
		if ((the_key >> 8) == KEY_DOWN)  {
			menu_choice++;
			if (menu_choice == 2 && save_file_exists == FALSE) menu_choice++;
			if (menu_choice > total_menu_choices) menu_choice = total_menu_choices;
		}
		if ((the_key >> 8) == KEY_RIGHT && (menu_choice == 4 || menu_choice == 5 || menu_choice == 6))  {
			if (menu_choice == 4)  {
				delay_msecs++;
				if (delay_msecs > 20) delay_msecs = 20;
			}
			else if (menu_choice == 5)  {
				level++;
				if (level > MAX_LEVELS) level = MAX_LEVELS;
			}
			else if (menu_choice == 6)  {
				lives++;
				if (lives > MAX_LIVES) lives = MAX_LIVES;
			}
		}
		if ((the_key >> 8) == KEY_LEFT && (menu_choice == 4 || menu_choice == 5 || menu_choice == 6))  {
			if (menu_choice == 4)  {
				delay_msecs--;
				if (delay_msecs < 5) delay_msecs = 5;
			}
			else if (menu_choice == 5)  {
				level--;
				if (level < 1) level = 1;
			}
			else if (menu_choice == 6)  {
				lives--;
				if (lives < 1) lives = 1;
			}
		}
		if ((the_key >> 8) == KEY_ENTER && menu_choice != 4 && menu_choice != 5 && menu_choice != 6)  {
			chosen_choice = menu_choice;
		}
	}

	fade_out (5);
	switch (menu_choice)  {
	case 1: break;						// Start New Game
	case 2: load_game(); break;			// Continue Game
	case 3: 	
		show_high_scores();
		game_ended = TRUE;
		break;
	case 7: exit(0);		// Exit to OS
	}

}



// saves a snapshot of the action to files game.sav and dump.scr
void save_game (void)
{
	FILE *fp_savefile;
	BITMAP *the_bmp;
	
	the_bmp = create_sub_bitmap(buffer, 0, 0, screen_x_size, screen_y_size);
	save_bmp("dump.scr", the_bmp, NULL);
	destroy_bitmap(the_bmp);

	fp_savefile = fopen("game.sav", "wb");					// open save file for output
	fwrite (&level, sizeof(int), 1, fp_savefile);			// write level to file
	fwrite (&lives, sizeof(int), 1, fp_savefile);			// write lives to file
	fwrite (&score, sizeof(int), 1, fp_savefile);			// write score to file
	fwrite (&player.x, sizeof(int), 1, fp_savefile);		// write player x coordinate to file
	fwrite (&player.y, sizeof(int), 1, fp_savefile);		// write player y coordinate to file
	fwrite (&player.frame, sizeof(int), 1, fp_savefile);	// write player frame to file
	fwrite (&total_player_shots, sizeof(int), 1, fp_savefile);	// write the total amount of player shots to file
	fwrite (&enemy_which_direction, sizeof(int), 1, fp_savefile);	// write the direction the enemies are moving to file
	fwrite (&shot_probability, sizeof(int), 1, fp_savefile);	// write the probability of enemy shooting to file
	fwrite (&enemy_wait_frames, sizeof(int), 1, fp_savefile);	// write the time it takes for the enemies to move to file
	
	for (index = 1; index <= 100; index++)  {				// write each enemy to file
		fwrite (&enemy[index], sizeof(enemy[index]), 1, fp_savefile);
	}
	for (index = 1; index <= 10; index++)  {				// write each player shot to file
		fwrite (&player_shot[index], sizeof(player_shot[index]), 1, fp_savefile);
	}
	for (index = 1; index <= 100; index++)  {				// write the enemies' shots to file
		fwrite (&enemy_shot[index], sizeof(enemy_shot[index]), 1, fp_savefile);
	}
	fclose (fp_savefile);									// close the save file


}



// loads a snapshot of the game from files game.sav and dump.scr
void load_game (void)
{
	int test_int = 0;
	FILE *fp_savefile;
	PALETTE junk_pal;

	fp_savefile = fopen("game.sav", "ab");					// open save file for append
	fclose (fp_savefile);
	fp_savefile = fopen("game.sav", "rb");					// open save file for input
	if (feof(fp_savefile)) {
		clear_bitmap(buffer);
		fclose (fp_savefile);								// close the file
		return;
	}
	buffer = load_bmp("dump.scr", junk_pal);

	fp_savefile = fopen("game.sav", "rb");					// open save file for input
	fread (&level, sizeof(int), 1, fp_savefile);			// read level to file
	fread (&lives, sizeof(int), 1, fp_savefile);			// read lives to file
	fread (&score, sizeof(int), 1, fp_savefile);			// read score to file
	fread (&player.x, sizeof(int), 1, fp_savefile);			// read player x coordinate to file
	fread (&player.y, sizeof(int), 1, fp_savefile);			// read player y coordinate to file
	fread (&player.frame, sizeof(int), 1, fp_savefile);		// read player frame to file
	fread (&total_player_shots, sizeof(int), 1, fp_savefile);		// read the total amount of player shots from file
	fread (&enemy_which_direction, sizeof(int), 1, fp_savefile);	// read the direction the enemies are moving from file
	fread (&shot_probability, sizeof(int), 1, fp_savefile);		// read the probability of enemy shooting from file
	fread (&enemy_wait_frames, sizeof(int), 1, fp_savefile);	// read the time it takes for the enemies to move from file

	for (index = 1; index <= 100; index++)  {				// read the enemies from file
		fread (&enemy[index], sizeof(enemy[index]), 1, fp_savefile);
	}
	for (index = 1; index <= 10; index++)  {				// read the player's shots from file
		fread (&player_shot[index], sizeof(player_shot[index]), 1, fp_savefile);
	}
	for (index = 1; index <= 100; index++)  {				// read the enemies' shots from file
		fread (&enemy_shot[index], sizeof(enemy_shot[index]), 1, fp_savefile);
	}
	fclose (fp_savefile);									// close the save file
	
	if		(level == 1) { x_aliens = 4; y_aliens = 3; }
	else if (level == 2) { x_aliens = 5; y_aliens = 4; }
	else if (level == 3) { x_aliens = 6; y_aliens = 2; }
	else if (level == 4) { x_aliens = 8; y_aliens = 5; }
	else if (level == 5) { x_aliens = 9; y_aliens = 3; }
	else if (level == 6) { x_aliens = 9; y_aliens = 3; }

	won_level = FALSE;
	draw_solid = FALSE;

	game_loaded = TRUE;
}



// erases the save files game.sav and dump.pcx
void erase_save_files (void)
{
	FILE *fp_savefile;

	fp_savefile = fopen("game.sav", "w");					// open and erase save file
	fclose (fp_savefile);									// close save file
	fp_savefile = fopen("dump.scr", "w");					// open and erase screen dump file
	fclose (fp_savefile);									// close screen dump file

}



// displays the high score table
void show_high_scores (void)
{
	int c;
	char str[100];
	char player_name[16][40];
	long player_score[16];
	RGB temp;
	FILE *fp_highscores;
	PALETTE palette;

	fade_out (5);										// fade to black

	clear_bitmap (buffer);								// clear the screen

		// get the high scores
	fp_highscores = fopen ("hiscores.lst", "a");			// create the file if it does not exist 
	fclose (fp_highscores);									// close the file
	fp_highscores = fopen ("hiscores.lst", "r");			// open the file for input
	if (fscanf(fp_highscores, "%s", str) == EOF)  {			// if the file is empty:
		fclose (fp_highscores);								// close the file
		fp_highscores = fopen ("hiscores.lst", "w");		// open the file for output
		// write default high scores
		fprintf (fp_highscores, "Johnny the Wild Axe Man\n"			"10000\n");
		fprintf (fp_highscores, "Robert and His Camels\n"			"9000\n");
		fprintf (fp_highscores, "William the Racecar Driver\n"		"8000\n");
		fprintf (fp_highscores, "Jeremy the Mushroom Picker\n"		"7000\n");
		fprintf (fp_highscores, "Kevin and the Cool Kids\n"			"6000\n");
		fprintf (fp_highscores, "Andrew the Magnificent\n"			"5000\n");
		fprintf (fp_highscores, "Horatio's Band of Merry Men\n"		"4000\n");
		fprintf (fp_highscores, "Valerie's Dwarf Tossing Brigade\n"	"3000\n");
		fprintf (fp_highscores, "Josh the Video Store Owner\n"		"2000\n");
		fprintf (fp_highscores, "Aaron and the Shoeshiners\n"		"1000\n");
		fprintf (fp_highscores, "Skip and Company\n"				"900\n");
		fprintf (fp_highscores, "Albert the Fish Gulper\n"			"800\n");
		fprintf (fp_highscores, "Jethro the Mighty\n"				"700\n");
		fprintf (fp_highscores, "Zeke and Ryan\n"					"600\n");
		fprintf (fp_highscores, "Moe the Glass Blower\n"			"500\n");
	}
	fclose (fp_highscores);									// close the file
	fp_highscores = fopen ("hiscores.lst", "r");			// open the file for input
	for (index = 1; index <= 15; index++)  {				// read the scores from file
		get_text_line (fp_highscores, player_name[index]);	// read the player's name
		get_text_line (fp_highscores, str);					// read the player's score
		player_score[index] = strtol(str, NULL, 0);			// convert the score string to long integer
	}
	fclose (fp_highscores);									// close the file


	
	// draw some circles onto the screen
	for (c = 255; c > 0; c--)
		circlefill (buffer, screen_x_size / 2, screen_y_size / 2, c, c % 127);
   
	// fill our palette with a gradually altering sequence of colors (black to red)
	for (c = 0; c <= 63; c++)  {
		palette[c].r = 0;
		palette[c].b = c;
		palette[c].g = 0;
	}
	for (c = 64; c <= 127; c++) {
		palette[c].r = 0;
		palette[c].b = 127 - c;
		palette[c].g = 0 ;
		
	}
	for (c = 128; c <= 255; c++)  {
		palette[c].r = 0;
		palette[c].g = 0;
		palette[c].b = 0;
	}
	palette[255].r = 63;
	palette[255].g = 63;
	palette[255].b = 63;

	text_mode (-1);		// draw transparent text
	textprintf_centre (buffer, game_font, (screen_x_size / 2), 10, 255, "Galaxy Invaders High Scores");

	// display the scores onscreen
	for (index = 1; index <= 15; index++)  {
		textprintf (buffer, game_font, 10, 27 + (index * 13), 255, player_name[index]);	// display the name
		textprintf_right (buffer, game_font, (screen_x_size - 10), 27 + (index * 13), 255, 
			"%ld", player_score[index]);					// display the score
	}
	
	text_mode (0);				// set opaque text mode
		
	copy_buffer_to_screen ();	// copy buffer to screen
	
	fade_in (palette, 5);		// fade in
	
	// animate the image by rotating the palette until a key is pressed
	clear_keybuf();
	while (!keypressed()) {
		temp = palette[127];
		for (c = 127; c >= 1; c--)
			palette[c] = palette[c - 1];
		palette[0] = temp;
		set_palette(palette);
	}
	
	fade_out (5);		// fade to black	
}