/* -------------------------------------------------------------------------- */
/*                                                                            */
/*                     Another Tetris Clone v0.1alpha                         */
/*                      Copyright (c) 2005 by Buster                          */
/*                  sebastian dot windisch at gmx dot net                     */
/*                                                                            */
/* -------------------------------------------------------------------------- */

#include <allegro.h>

#include <stdlib.h>
#include <time.h>
#include <stdio.h>

#ifndef TRUE
#   define TRUE -1
#   define FALSE 0
#endif

#define HOFFSET 320 / 2 - 10 * 10 / 2 + 50
#define VOFFSET 240 / 2 - 20 * 10 / 2

struct COORD
{
    int x, y;
};

struct COORD __o1[] = {{1,1},{2,1},{1,2},{2,2}};
struct COORD __i1[] = {{0,1},{1,1},{2,1},{3,1}};
struct COORD __i2[] = {{2,0},{2,1},{2,2},{2,3}};
struct COORD __s1[] = {{1,2},{2,1},{2,2},{3,1}};
struct COORD __s2[] = {{2,0},{2,1},{3,1},{3,2}};
struct COORD __z1[] = {{1,1},{2,1},{2,2},{3,2}};
struct COORD __z2[] = {{3,0},{2,1},{3,1},{2,2}};
struct COORD __l1[] = {{1,1},{1,2},{2,1},{3,1}};
struct COORD __l2[] = {{2,0},{2,1},{2,2},{3,2}};
struct COORD __l3[] = {{1,1},{2,1},{3,0},{3,1}};
struct COORD __l4[] = {{1,0},{2,0},{2,1},{2,2}};
struct COORD __j1[] = {{1,1},{2,1},{3,1},{3,2}};
struct COORD __j2[] = {{2,0},{2,1},{2,2},{3,0}};
struct COORD __j3[] = {{1,0},{1,1},{2,1},{3,1}};
struct COORD __j4[] = {{1,2},{2,0},{2,1},{2,2}};
struct COORD __t1[] = {{1,1},{2,1},{2,2},{3,1}};
struct COORD __t2[] = {{2,0},{2,1},{2,2},{3,1}};
struct COORD __t3[] = {{1,1},{2,0},{2,1},{3,1}};
struct COORD __t4[] = {{1,1},{2,0},{2,1},{2,2}};

struct COORD *__o[] = {__o1, NULL};
struct COORD *__i[] = {__i1, __i2, NULL};
struct COORD *__s[] = {__s1, __s2, NULL};
struct COORD *__z[] = {__z1, __z2, NULL};
struct COORD *__l[] = {__l1, __l2, __l3, __l4, NULL};
struct COORD *__j[] = {__j1, __j2, __j3, __j4, NULL};
struct COORD *__t[] = {__t1, __t2, __t3, __t4, NULL};

struct COORD **stone[] = {__o, __i, __s, __z, __l, __j, __t};

static volatile int timer_count = 0;
static volatile int game_active = TRUE;

static BITMAP *framebuffer = NULL;
static struct COORD pos = {3,-2};
static int map[10][20] = {{0}};
static int curr_stone;
static int curr_rotation = 0;
static int next_stone;
static int frames = 0;
static int speed = 30;
static int game_paused = FALSE;
static int total_lines = 0;
static int level = 0;
static int lost = FALSE;
static int win = FALSE;

static int init_game(void);
static void clean_game(void);
static void timer_callback(void);
static void exit_callback(void);
static void game_loop(void);
static void game_logic(void);
static void game_draw(void);
static int is_line_full(int line);
static void kill_lines(void);

enum
{
    MOVE_DOWN,
    MOVE_RIGHT,
    MOVE_LEFT,
    ROTATE
};

static int check_collision(int type);
static void check_control(void);

int main(int argc, char *argv[])
{
    if (init_game())
        game_loop();
    
    return 0;
}
END_OF_MAIN();

static int init_game(void)
{
    if (allegro_init() == 0)
    {
        set_color_depth(8);

        if (set_gfx_mode(GFX_AUTODETECT_WINDOWED, 320, 240, 0, 0) == 0)
        {
            set_window_title("Another Tetris Clone v0.1alpha");
            
            LOCK_FUNCTION(exit_callback);
            LOCK_FUNCTION(timer_callback);
            LOCK_VARIABLE(timer_count);
            
            if (set_close_button_callback(&exit_callback) == 0
            && install_int_ex(&timer_callback, BPS_TO_TIMER(30)) == 0
            && install_keyboard() == 0)
            {
                framebuffer = create_bitmap(screen->w, screen->h);
                
                if (framebuffer)
                {
                    atexit(&clean_game);
                    srand(time(NULL));
                    curr_stone = rand()%7;
                    next_stone = rand()%7;
                    return TRUE;
                }
            }
        }
    }
    
    return FALSE;
}

static void clean_game(void)
{
    if (framebuffer)
        destroy_bitmap(framebuffer);
}

static void timer_callback(void)
{
    ++timer_count;
}
END_OF_STATIC_FUNCTION(timer_callback);


static void exit_callback(void)
{
    game_active = FALSE;
}
END_OF_STATIC_FUNCTION(exit_callback);

static void game_loop(void)
{
    while (game_active)
    {
        while (timer_count > 0)
        {
            --timer_count;
            if (frames++ == ~((int)0))
                frames = 0;
            game_logic();
        }
        
        game_draw();
        
        while (timer_count == 0)
            rest(1);
    }
}

static void game_logic(void)
{
    if (frames % 5 == 0)
        check_control();

    if (!game_paused)
    {
        if (frames % speed == 0)
        {
            if (check_collision(MOVE_DOWN))
                ++pos.y;
            else
            {
                int i = 0;
                struct COORD *s = stone[curr_stone][curr_rotation];

                for (; i < 4; ++i)
                    if (pos.y+s[i].y >= 0)
                    {
                        map[pos.x+s[i].x][pos.y+s[i].y] = 1;
                    }
                    else
                    {
                        game_paused = TRUE;
                        lost = TRUE;
                    }

                curr_stone = next_stone;
                next_stone = rand()%7;
                pos.x = 3;
                pos.y = -2;
                curr_rotation = 0;

                kill_lines();
            }
        }
    }
}

static int is_line_full(int line)
{
    int i = 0;
    
    for (; i < 10; ++i)
    {
        if (!map[i][line])
            return FALSE;
    }
    
    return TRUE;
}

static void kill_lines(void)
{
    int i = 0;
    
    for (; i < 20; ++i)
    {
        if (is_line_full(i))
        {
            int j = 0, k;
            
            ++total_lines;
            
            if (total_lines % 10 == 0)
            {
                speed-=3;
                ++level;
                
                if (level == 10)
                    win = TRUE;
            }
            
            for (; j < 10; ++j)
                map[j][i] = 0;
                
            for (j = 0; j < 10; ++j)
                for (k = i; k > 0; --k)
                    map[j][k] = map[j][k-1];
        }
    }
}

static void game_draw(void)
{
    int i = 0, j;
    char buffer[32];
    struct COORD *s = stone[curr_stone][curr_rotation];

    snprintf(buffer, 32, "Lines: %d", total_lines);
    textout_ex(framebuffer, font, buffer, 10, 10, makecol8(255,255,255), -1);
    
    snprintf(buffer, 32, "Level: %d", level);
    textout_ex(framebuffer, font, buffer, 10, 20, makecol8(255,255,255), -1);

    if (game_paused || lost)
        textout_ex(framebuffer, font, "P A U S E D !", 10, 30, makecol8(0,255,0), -1);

    if (lost)
        textout_ex(framebuffer, font, "L O S E R !", 10, 40, makecol8(255,0,0), -1);
        
    if (win)
        textout_ex(framebuffer, font, "W I N N E R !", 10, 40, makecol8(0,255,0), -1);

    for (; i < 4; ++i)
        if (pos.y+stone[curr_stone][curr_rotation][i].y >= 0)
            rectfill(framebuffer,HOFFSET+pos.x*10+s[i].x*10,VOFFSET+pos.y*10+s[i].y*10,
            HOFFSET+pos.x*10+s[i].x*10+10,VOFFSET+pos.y*10+s[i].y*10+10,makecol(255,0,0));

    for (i = 0; i < 10; ++i)
        for (j = 0; j < 20; ++j)
            if (map[i][j])
                rectfill(framebuffer,HOFFSET+i*10,VOFFSET+j*10,HOFFSET+i*10+10,VOFFSET+j*10+10,makecol8(255,0,0));

    for (i = 0; i <= 10; ++i)
        line(framebuffer, HOFFSET+i*10,VOFFSET,HOFFSET+i*10,VOFFSET+200,makecol8(255,255,255));
    for (i = 0; i <= 20; ++i)
        line(framebuffer,HOFFSET,VOFFSET+i*10,HOFFSET+100,VOFFSET+i*10,makecol8(255,255,255));
        
    textout_ex(framebuffer, font, "Next:", 10, 50, makecol8(255,255,255), -1);
        
    s = stone[next_stone][0];
        
    for (i = 0; i < 4; ++i)
            rectfill(framebuffer,10+s[i].x*10,60+s[i].y*10,s[i].x*10+20,s[i].y*10+70,makecol(255,0,0));
        
    for (i = 0; i <= 4; ++i)
        line(framebuffer, 10+i*10,60,10+i*10,100,makecol8(255,255,255));
    for (i = 0; i <= 4; ++i)
        line(framebuffer,10,60+i*10,50,60+i*10,makecol8(255,255,255));
    
    acquire_screen();
    blit(framebuffer, screen, 0, 0, 0, 0, framebuffer->w, framebuffer->h);
    release_screen();
    
    clear_bitmap(framebuffer);
}

static void check_control(void)
{
    if (keyboard_needs_poll())
        poll_keyboard();
        
    if (key[KEY_ESC])
    {
        game_active = FALSE;
        return;
    }
    
    if (key[KEY_P] && !lost)
    {
        game_paused = !game_paused;
        return;
    }

    if (!game_paused)
    {
        if (key[KEY_SPACE])
        {
            if (check_collision(ROTATE))
            {
                if (stone[curr_stone][curr_rotation+1])
                    ++curr_rotation;
                else
                    curr_rotation = 0;
            }
        }
        else if (key[KEY_RIGHT])
        {
            if (check_collision(MOVE_RIGHT))
                ++pos.x;
        }
        else if (key[KEY_LEFT])
        {
            if (check_collision(MOVE_LEFT))
                --pos.x;
        }
        else if (key[KEY_DOWN])
        {
            if (check_collision(MOVE_DOWN))
                ++pos.y;
        }
    }
}

static int check_collision(int type)
{
    switch (type)
    {
        case ROTATE:
        {
            int i = 0, j, r;
            struct COORD *s;
            
            if (stone[curr_stone][curr_rotation+1])
                r = curr_rotation + 1;
            else
                r = 0;
            
            s = stone[curr_stone][r];

            for (; i < 10; ++i)
            {
                for (j = 0; j < 20; ++j)
                {
                    if (map[i][j])
                    {
                        int n = 0;
                        
                        for (; n < 4; ++n)
                        {
                            if (pos.x+s[n].x == i
                            && pos.y+s[n].y == j)
                                return FALSE;
                        }
                    }
                }
            }

            for (i = 0; i < 4; ++i)
            {
                if (pos.x+s[i].x < 0
                || pos.x+s[i].x > 9
                || pos.y+s[i].y > 19)
                    return FALSE;
            }
        }
        break;
            
        case MOVE_LEFT:
        {
            int i = 0, j;
            struct COORD *s = stone[curr_stone][curr_rotation];

            for (; i < 10; ++i)
            {
                for (j = 0; j < 20; ++j)
                {
                    if (map[i][j])
                    {
                        int n = 0;

                        for (; n < 4; ++n)
                        {
                            if (pos.x+s[n].x-1 == i
                            && pos.y+s[n].y == j)
                                return FALSE;
                        }
                    }
                }
            }

            for (i = 0; i < 4; ++i)
            {
                if (pos.x+s[i].x-1 < 0)
                    return FALSE;
            }
        }
        break;
        
        case MOVE_RIGHT:
        {
            int i = 0, j;
            struct COORD *s = stone[curr_stone][curr_rotation];

            for (; i < 10; ++i)
            {
                for (j = 0; j < 20; ++j)
                {
                    if (map[i][j])
                    {
                        int n = 0;

                        for (; n < 4; ++n)
                        {
                            if (pos.x+s[n].x+1 == i
                            && pos.y+s[n].y == j)
                                return FALSE;
                        }
                    }
                }
            }

            for (i = 0; i < 4; ++i)
            {
                if (pos.x+s[i].x+1 > 9)
                    return FALSE;
            }
        }
        break;
        
        case MOVE_DOWN:
        {
            int i = 0, j;
            struct COORD *s = stone[curr_stone][curr_rotation];

            for (; i < 10; ++i)
            {
                for (j = 0; j < 20; ++j)
                {
                    if (map[i][j])
                    {
                        int n = 0;

                        for (; n < 4; ++n)
                        {
                            if (pos.x+s[n].x == i
                            && pos.y+s[n].y+1 == j)
                                return FALSE;
                        }
                    }
                }
            }

            for (i = 0; i < 4; ++i)
            {
                if (pos.y+s[i].y+1 > 19)
                    return FALSE;
            }
        }
        break;
    }
    
    return TRUE;
}
