#include <stdlib.h>
#include <math.h>
#include "Map.h"
#include "Buildings.h"
#include "MatrixStack.h"
#include "GlobalTimer.h"

#define SPACING	0.7f
#define SCALER 3.0f

float GetX1(int id, float lowx, float highx) { return (id&1) ? (lowx+highx)*0.5f : lowx;}
float GetX2(int id, float lowx, float highx) { return (id&1) ? highx : (lowx+highx)*0.5f;}
float GetY1(int id, float lowx, float highx) { return (id&2) ? (lowx+highx)*0.5f : lowx;}
float GetY2(int id, float lowx, float highx) { return (id&2) ? highx : (lowx+highx)*0.5f;}

CMap City;

CMap::CMap()
{
	Tree = NULL;
}

CMap::~CMap()
{
	delete Tree;
}

void CMap::Draw()
{
	if(Tree)
	{
		Tree->Draw(-128, -128, 128, 128);
		switch((GlobalTimer >> 2)&1)
		{
			default:
			case 0: SetColour(GREEN); break;
			case 1: SetColour(RED); break;
		}
			
		alPushMatrix();
			alTranslatef(TargetX, 0, TargetY);
			alScalef(TargetScale, TargetScale, TargetScale);
			alTranslatef(0, 1, 0);
			DrawObject3d(Target->Shape);
		alPopMatrix();
	}
}

void CMap::Setup(int round)
{
	/* generate city */
	delete Tree;
	Tree = new Node;
	Tree->Setup(1, round, -128, -128, 128, 128);

	/* pick a target */
	while(!Tree->SetTarget(this, 1, round, -128, -128, 128, 128));

	/* pick a start position */
	switch(round / 5)
	{
		case 0:	/* rounds 1 - 5 */
			PlayerAngle = 90 + (randr(90));
		break;
		case 1:
		case 2: /* rounds 5 - 15 */
			PlayerAngle = randr(180);
		break;
		case 3:
		case 4: /* rounds 15 - 25 */
			PlayerAngle = randr(360);
		break;
		case 5:
		case 6: /* rounds 25-35 */
			PlayerAngle = (randr(180))+180;
		break;
		default: /* rounds 35+ */
			PlayerAngle = 270 + (randr(90));
		break;
	}
	PlayerDistance = (round+12)*25.0f + round*25.0f*randf();
}

void CMap::GetStartPos(float &Angle, float &Distance)
{
	Angle = PlayerAngle;
	Distance = PlayerDistance;
}

void CMap::CentreOnTarget()
{
	alTranslatef(-TargetX, 0, -TargetY);
}

float CMap::GetDistanceFromTarget(float *PlayerPos)
{
	return sqrt((PlayerPos[0]-TargetX)*(PlayerPos[0]-TargetX) + (PlayerPos[2]-TargetY)*(PlayerPos[2]-TargetY));
}

CMap::Node::Node()
{
	Shape = NULL;
	Children = NULL;
	Target = false;
}

int CMap::GetHit(float *StartPos, float *EndPos)
{
	if(Tree)
	{
		return Tree->GetHit(StartPos, EndPos, -128, -128, 128, 128);
	}

	return NOHIT;
}
int CMap::Node::GetHit(float *StartPos, float *EndPos, float lowx, float lowy, float highx, float highy)
{
	/* trivial separating planes */
	if(StartPos[0] < lowx*2.0f*SPACING && EndPos[0] < lowx*2.0f*SPACING) return NOHIT;
	if(StartPos[0] > highx*2.0f*SPACING && EndPos[0] > highx*2.0f*SPACING) return NOHIT;
	if(StartPos[2] < lowy*2.0f*SPACING && EndPos[2] < lowy*2.0f*SPACING) return NOHIT;
	if(StartPos[2] > highy*2.0f*SPACING && EndPos[2] > highy*2.0f*SPACING) return NOHIT;
	if(StartPos[1] > ((highy-lowy)*2.0f)/SCALER && EndPos[1] > ((highy-lowy)*2.0f)/SCALER) return NOHIT;
//	if(StartPos[1] < 0) return NOHIT;

	if(Children)
	{
		int res = 0;
		int c = 4;
		while(c--)
			res |= Children[c].GetHit(StartPos, EndPos, GetX1(c, lowx, highx), GetY1(c, lowy, highy), GetX2(c, lowx, highx), GetY2(c, lowy, highy));
		return res;
	}
	else
	{
		if(Shape && Shape != GetBuilding(CRATER))
		{
			float FictionalStartVec[3], FictionalEndVec[3];
			float scaler = (highx - lowx) / SCALER;

			FictionalStartVec[0] = (StartPos[0] - (lowx + highx)*SPACING)/scaler;
			FictionalStartVec[1] = StartPos[1]/scaler - 1;
			FictionalStartVec[2] = (StartPos[2] - (lowy + highy)*SPACING)/scaler;

			FictionalEndVec[0] = (EndPos[0] - (lowx + highx)*SPACING)/scaler;
			FictionalEndVec[1] = EndPos[1]/scaler - 1;
			FictionalEndVec[2] = (EndPos[2] - (lowy + highy)*SPACING)/scaler;

//			if(fabs(FictionalVec[0]) < 1.0f && fabs(FictionalVec[1]) < 1.0f && fabs(FictionalVec[2]) < 1.0f)
			{
				if(QueryPenetration(Shape, FictionalStartVec, FictionalEndVec))
				{
					Shape = GetBuilding(CRATER);
					return Target ? GOODHIT : BADHIT;
				}
			}
		}
	}
	return NOHIT;
}

bool CMap::Node::SetTarget(CMap *Parent, int level, int round, float lowx, float lowy, float highx, float highy)
{
	if(Children)
	{
		/* normal order */
		int Order[4] = {0, 1, 2, 3};

		/* two transitions to make a random order */
		if(level > 1)
		{
			int pos1, pos2, sw;

			pos1 = randr(4); pos2 = randr(4);
			sw = Order[pos1]; Order[pos1] = Order[pos2]; Order[pos2] = sw;
			pos1 = randr(4); pos2 = randr(4);
			sw = Order[pos1]; Order[pos1] = Order[pos2]; Order[pos2] = sw;
			pos1 = randr(4); pos2 = randr(4);
			sw = Order[pos1]; Order[pos1] = Order[pos2]; Order[pos2] = sw;
		}

		/* try them all */
		int c = 3;
		while(c--)
			if(Children[Order[c]].SetTarget(Parent, level+1, round, GetX1(Order[c], lowx, highx), GetY1(Order[c], lowy, highy), GetX2(Order[c], lowx, highx), GetY2(Order[c], lowy, highy))) return true;
		return false;
	}
	else
		if(Shape)
		{
			if(level <= ((round >> 2)+4))
			{
				Target = true;
				Parent->TargetX = (lowx+highx)*SPACING;
				Parent->TargetY = (lowy+highy)*SPACING;
				Parent->TargetScale = (highx - lowx) / SCALER;
				Parent->Target = this;
				return true;
			}
		}
	return false;
}

#define MAX_LEVELS	5
void CMap::Node::Setup(int level, int round, float lowx, float lowy, float highx, float highy)
{
	/* always subdivide for first three levels */
	int topdivider = (round >> 2)+4;
	if(topdivider > MAX_LEVELS) topdivider = MAX_LEVELS;
	
	if((level < 4) || (((randr(round)) > 2) && level < topdivider))
	{
		Children = new Node[4];
		int c = 4;
		while(c--)
			Children[c].Setup(level+1, round, GetX1(c, lowx, highx), GetY1(c, lowy, highy), GetX2(c, lowx, highx), GetY2(c, lowy, highy));
	}
	else
	{
		/* with 1 in 5 probability do nothing, otherwise pick a random shape */
		if(randr(5))
		{
			Shape = GetBuilding(randr(NUM_BUILDINGS));
		}
	}
}

CMap::Node::~Node()
{
	delete[] Children;
}

void CMap::Node::Draw(float lowx, float lowy, float highx, float highy)
{
	if(Shape)
	{
		if(!Target)
		{
			SetColour(WHITE);
			alPushMatrix();
				alTranslatef((lowx+highx)*SPACING, 0, (lowy+highy)*SPACING);
				float scaler = (highx - lowx) / SCALER;
				alScalef(scaler, scaler, scaler);
				alTranslatef(0, 1, 0);
				DrawObject3d(Shape);
			alPopMatrix();
		}
	}
	else
		if(Children)
		{
			int c = 4;
			while(c--)
				Children[c].Draw(GetX1(c, lowx, highx), GetY1(c, lowy, highy), GetX2(c, lowx, highx), GetY2(c, lowy, highy));
		}
}
