/**********************************************/
/* Script virtual machine                     */
/* Evert Glebbeek 2002                        */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include <math.h>
#include "vm.h"
#include "script.h"
#include "skeyword.h"
#include "linked.h"

typedef struct {
   void *obj;
   SCRIPT *script;
} ACTIVE_SCRIPT;

typedef struct {
   CPOPCODE opcode_implementation[256];
   int num_scripts;
   LINK *running_scripts;
   LINK *current_script;
} VIRTUAL_MACHINE;

VIRTUAL_MACHINE vm = {{}, 0, NULL, NULL};

/* Default opcode handler */
static int unknown_opcode(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   TRACE("Unknown byte code encountered at 0x%04x: 0x%02x - program corrupt!\n",
         script->ip, script->script[script->ip]);

   script->ip++;
   return 0;
}

/* Initialize virtual machine */
void vm_init(void)
{
   int c;

   for (c = 0; c < 256; c++) {
      vm.opcode_implementation[c] = unknown_opcode;
   }
   
   vm.current_script = NULL;
   vm.running_scripts = NULL;
   vm.num_scripts = 0;
}

/* Shutdown virtual machine */
void vm_shutdown(void)
{
   void *data;
   
   while (vm.running_scripts) {
      data = vm.running_scripts->data;
      vm.running_scripts = remove_from_list(data, vm.running_scripts);
      free(data);
   }
}

/* Register a handler function for the VM */
void register_opcode(CPOPCODE func, const int opcode)
{
   vm.opcode_implementation[opcode] = func;
}

/* Run next instruction for script on model */
static int update_script(void *obj, SCRIPT *script)
{
   int script_loop;
   int res = 0;
   
   ASSERT(obj);
   ASSERT(script);

   /* check wait cycle */

   script_loop = 0;
   do {
      if (script->wait) {
         script->wait--;
         return 2;
      }

      if (script == NULL) {
         return 3;
      }
      
      if (script->script == NULL) {
         return 4;
      }

      //printf("\toc: %02x\n", script->script[script->ip]);
      res = vm.opcode_implementation[script->script[script->ip]](obj, script, &script_loop);

      script_loop++;
   } while (script_loop < 1);
   return res;
}

/* Run a script for some object */
/* Returns a pointer to the running script */
SCRIPT* run_script(void *obj, SCRIPT *script)
{
   ACTIVE_SCRIPT *as = malloc(sizeof *as);
   
   ASSERT(as);
   
   as->obj = obj;
   as->script = malloc(sizeof *as->script);
   //memcpy(as->script, script, sizeof *script);
   //printf("run_script: %08x\n", script->script);
   as->script->script = script->script;
   as->script->ip = 0;
   as->script->sp = 8;
   as->script->wait = 0;
   vm.running_scripts = add_to_list(as, vm.running_scripts);
   vm.num_scripts++;
   
   return as->script;
}

/* Returns TRUE if a script is currently influencing obj */
int running_script(void *obj)
{
   ACTIVE_SCRIPT *as;
   LINK *l;
   
   l = vm.running_scripts; 
   while(l) {
      as = l->data;
      if (as->obj == obj)
         return TRUE;
      l=l->next;
   }
   
   return FALSE;
}

/* Halt a script */
void halt_script(void *obj, SCRIPT *script)
{
   ACTIVE_SCRIPT *as;
   LINK *l;
   LINK *ln;
   
   l = vm.running_scripts; 
   while(l) {
      as = l->data;
      ln = l->next;
      if ((as->script == script) && (as->obj == obj)) {
         if (vm.current_script==l)
            vm.current_script=vm.current_script->next;
         if (vm.running_scripts==l)
            vm.running_scripts=vm.running_scripts->next;
         free(l->data);
         free_link(l);
         vm.num_scripts--;
      }
      l=ln;
   }
}

/* Update all active scripts */
void update_active_scripts(void)
{
   ACTIVE_SCRIPT *as;
   
   vm.current_script = vm.running_scripts; 
   //printf ("%d %d\n", get_link_count(), vm.num_scripts);
   while(vm.current_script) {
      as = vm.current_script->data;
      //printf ("ip: %d ret: ", as->script->ip);
      //printf("%d\n", update_script(as->obj, as->script));
      
      //printf ("%p: %p\n", as->obj, as->script);
      update_script(as->obj, as->script);
      if (vm.current_script)
         vm.current_script=vm.current_script->next;
   }
}
