/**********************************************/
/* Rise of the Tribes                         */
/* C version, Take 2                          */
/* Evert Glebbeek 1998, 2003                  */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include <stdio.h>
#include "global.h"
#include "map.h"
#include "unit.h"
#include "unitmap.h"

/* A* node structure */
typedef struct ASTAR_NODE {
   struct ASTAR_NODE *next;
   struct ASTAR_NODE *parent;
   struct ASTAR_NODE *child[8];
   int f, g, h;
   int x, y;
   int node_index;
} ASTAR_NODE;

/* Structure to check for existence of a path */
typedef struct PATH_EXIST_STRUC {
   int startx;
   int starty;
   int layer;
   int *data;
} PATH_EXIST_STRUC;

static ASTAR_NODE *open_list;
static ASTAR_NODE *closed_list;
static PATH_EXIST_STRUC node_eval = { -1, -1, -1, NULL };

/* A* heuristic distance function */
#define heuristic(x1,y1,x2,y2) (((x2)-(x1))*((x2)-(x1))+((y2)-(y1))*((y2)-(y1)))

/* Static variables that are nice for easy access */
static int dir_cost[8] = { 10, 14, 10, 14, 10, 14, 10, 14 };

/* Generate tile index number from an (x, y) pair */
static inline int tile_index(int x, int y)
{
   return x + y * get_map_width();
}

/* Clear all nodes on the stack. */
static void clear_nodes()
{
   ASTAR_NODE *p;
   ASTAR_NODE *save = NULL;

   p = open_list;
   while (p != NULL) {
      save = p->next;
      free(p);
      p = save;
   }

   p = closed_list;
   while (p != NULL) {
      save = p->next;
      free(p);
      p = save;
   }

   open_list = closed_list = NULL;
}

static ASTAR_NODE *alloc_node(void)
{
   return calloc(1, sizeof(ASTAR_NODE));
}

/* Place a node on the open_list list, using merge sort */
static void insert_open_list_node(ASTAR_NODE *node)
{
   ASTAR_NODE *n1;
   ASTAR_NODE *n2;
   int f;

   if (!open_list) {
      open_list = node;
   } else {
      f = node->f;
      n2 = n1 = open_list;
      while ((n1) && (n1->f < f)) {
         n2 = n1;
         n1 = n1->next;
      }
      node->next = n1;
      if (n1 == open_list) {
         open_list = node;
      } else {
         n2->next = node;
      }
   }
}

/* Pop the best node from the open_list list, ie, the first one */
static ASTAR_NODE *pop_best_node(void)
{
   ASTAR_NODE *node;

   node = open_list;

   if (node) {
      open_list = open_list->next;

      /* Put the node on the closed_list list */
      node->next = closed_list;
      closed_list = node;
   }

   return node;
}

/* Find a node labeled index on the open_list list */
static ASTAR_NODE *check_open_list(int index)
{
   ASTAR_NODE *node = open_list;

   while (node) {
      if (node->node_index == index)
         return node;
      node = node->next;
   }

   return NULL;
}

/* Find a node labeled index on the closed_list list */
static ASTAR_NODE *check_closed_list(int index)
{
   ASTAR_NODE *node;

   for (node = closed_list; node; node = node->next)
      if (node->node_index == index)
         return node;

   return NULL;
}

static void propagate_down(ASTAR_NODE *node)
{
   int c;
   int g;
   ASTAR_NODE *child;

   g = node->g;

   for (c = 0; c < 8; c++) {
      if (!node->child[c])
         break;
      child = node->child[c];
      g = node->g +
         dir_cost[dx_dy_to_dir(node->x - child->x, node->y - child->y)];
      if (g < child->g) {
         child->g = g;
         child->f = child->g + child->h;
         child->parent = node;

         /* Now propagate through the children */
         /* It would seem *very* unlikely that any stack we create ourselves */
         /*  will be more efficient than a proper recursive formulation */
         propagate_down(child);
      }
   }

}

/* helper function for generate_scuccessors: perform a single step */
static void generate_successor(ASTAR_NODE *node, int x, int y, int x2,
                               int y2, int cost)
{
   int g;
   int index;
   int c;
   ASTAR_NODE *old;
   ASTAR_NODE *new;

   g = node->g + cost;
   index = tile_index(x, y);
   /* Find the node in the open_list */
   if ((old = check_open_list(index))) {
      /* Set children */
      for (c = 0; c < 8; c++)
         if (!node->child[c])
            break;
      node->child[c] = old;

      /* Set the new cost value */
      if (g < old->g) {
         old->parent = node;
         old->g = g;
         old->f = old->g + old->h;
      }
   } else if ((old = check_closed_list(index))) {  /* Now look on the closed_list */
      /* Set children */
      for (c = 0; c < 8; c++)
         if (!node->child[c])
            break;
      node->child[c] = old;

      /* If we've found a better path, then propagate this downward */
      if (g < old->g) {
         old->parent = node;
         old->g = g;
         old->f = old->g + old->h;

         propagate_down(old);
      }
   } else {                     /* Node neither on the open_list, nor on the closed_list */
      new = alloc_node();
      new->parent = node;
      new->g = g;
      new->h = heuristic(x, y, x2, y2);
      new->f = new->g + new->h;
      new->x = x;
      new->y = y;
      new->node_index = index;

      insert_open_list_node(new);

      for (c = 0; c < 8; c++)
         if (!node->child[c])
            break;
      node->child[c] = new;
   }
}

/* Generate next step from node to (x2, y2) */
static void generate_successors(ASTAR_NODE *node, int x2, int y2, int layer)
{
   int dir;
   int x, y;
   
   ASSERT(node);

   for (dir = 0; dir < 8; dir++) {
      x = node->x + dir_to_dx(dir);
      y = node->y + dir_to_dy(dir);
      if (!tile_blocked(x, y, layer, NULL) &&
          in_rect(x, y, 0, 0, get_map_width(), get_map_height())) {
         /* Generate successor for node from (x, y) to (x2, y2) */
         generate_successor(node, x, y, x2, y2, dir_cost[dir]);
      }
   }
}

/* The actual path generator */
MAP_TILE_LIST *create_path(const int x1, const int y1, const int x2,
                           const int y2, const int layer)
{
   MAP_TILE_LIST *mtlst = NULL;
   MAP_TILE_LIST *mtl = NULL;
   ASTAR_NODE *node;
   int dest_index = tile_index(x2, y2);

   if (tile_blocked(x2, y2, layer, NULL))
      return NULL;

   clear_nodes();

   node = alloc_node();
   node->g = 0;
   node->h = heuristic(x1, y1, x2, y2);
   node->f = node->g + node->h;
   node->node_index = tile_index(x1, y1);
   node->x = x1;
   node->y = y1;

   /* Place node on the open_list list */
   insert_open_list_node(node);

   while (1) {
      /* Get the best node from the Open list */
      node = pop_best_node();

      if (!node)
         break;

      /* Have we reached our goal? */
      if (node->node_index == dest_index)
         break;

      /* Sadly, no. Generate the next iteration step */
      generate_successors(node, x2, y2, layer);
   }

   /* Ok, so now we have the path stored as a series of A* nodes. */
   /* Convert this to a MAP_TILE_LIST */

   if (node)
      while (node->parent) {
         mtl = alloc_maptile();

         mtl->tx = node->x;
         mtl->ty = node->y;
         node = node->parent;

         /* Now add the new tile to the maptile list */
         mtl->next = mtlst;
         mtlst = mtl;
      }

   mtl = alloc_maptile();

   mtl->tx = x1;
   mtl->ty = y1;

   /* Now add the new tile to the maptile list */
   mtl->next = mtlst;
   mtlst = mtl;


   /* Cleanup */
   clear_nodes();

   return mtlst;
}

/* Clears the path variable of the specified unit */
void clear_path(UNITDATA *u)
{
   MAP_TILE_LIST *tl;

   tl = u->path;
   while (tl) {
      u->path = u->path->next;

      free_maptile(tl);

      tl = u->path;
   }
}

void set_path(UNITDATA *u, MAP_TILE_LIST *path)
{
   u->path = path;
}

/* Remove the head of the path */
void clear_first_step(UNITDATA *u)
{
   MAP_TILE_LIST *tl = u->path;

   if (tl) {
      u->path = u->path->next;
      free_maptile(tl);
   }
}

/* Reset the path finding engine */
void reset_pathfinder(void)
{
   clear_nodes();

   if (node_eval.data) {
      free(node_eval.data);
      node_eval.data = NULL;
      node_eval.startx = -1;
      node_eval.starty = -1;
      node_eval.layer = -1;
   }
}

/* Generate children from a node and set a flag wether or not they can be */
/*  reached */
static void paint_node_reach(int x, int y, int layer)
{
   int c;
   int i;
   int tx;
   int ty;

   for (c = 0; c < 8; c++) {
      tx = x + dir_to_dx(c);
      ty = y + dir_to_dy(c);
      i = tile_index(tx, ty);

      if (!node_eval.data[i] && !tile_blocked(tx, ty, layer, NULL)) {
         node_eval.data[i] = TRUE;
         paint_node_reach(tx, ty, layer);
      }
   }
}

/* Check if a path exists between node 1 and node 2 */
int path_exists(const int x1, const int y1, const int x2, const int y2,
                int layer)
{
   int c;
   int n;
   int tx, ty;

   ASSERT(x1 > -1);
   ASSERT(y1 > -1);
   ASSERT(x1 < get_map_width());
   ASSERT(y1 < get_map_height());
   ASSERT(x2 > -1);
   ASSERT(y2 > -1);
   ASSERT(x2 < get_map_width());
   ASSERT(y2 < get_map_height());

   if (node_eval.startx != x1 || node_eval.starty != y1 ||
       node_eval.layer != layer) {
      node_eval.startx = x1;
      node_eval.starty = y1;
      n = get_map_width() * get_map_height();
      /* Clear the pathing stack */
      if (!node_eval.data)
         node_eval.data = malloc(n * sizeof(int));
      ASSERT(node_eval.data);

      for (c = 0; c < n; c++)
         node_eval.data[c] = FALSE;
      for (c = 0; c < 8; c++) {
         tx = x1 + dir_to_dx(c);
         ty = y1 + dir_to_dy(c);

         if (!tile_blocked(tx, ty, layer, NULL)) {
            node_eval.data[tile_index(tx, ty)] = TRUE;
            paint_node_reach(tx, ty, layer);
         }
      }
   }

   return node_eval.data[tile_index(x2, y2)];
}

#define FCT_TARGET(tx, ty, dx, dy, sx, sy, cx, cy, cost) = \
            if (in_rect\
                (tx + dx, ty + dy, 0, 0, get_map_width() - 1,\
                 get_map_height() - 1)) {\
               if (node_eval.data[tile_index(tx + dx, ty + dy)]) {\
                  cost =\
                     heuristic(tx, ty, tx + dx, ty + dy) * heuristic(sx, sy,\
                                                                     tx + dx,\
                                                                     ty + dy);\
                  if (cost < max_cost) {\
                     *cx = tx + dx;\
                     *cy = ty + dy;\
                     max_cost = cost;\
                  }\
               }\
            }

/* Return the tile closest to (tx, ty) that is unblocked */
/* Uses the output from the last path_exists() call */
void find_closest_target(const int sx, const int sy, const int tx,
                         const int ty, int *cx, int *cy)
{
   int dx;
   int dy;
   int r;
   int max_cost = heuristic(sx, sy, tx, ty);
   int cost = heuristic(sx, sy, tx, ty);

   *cx = sx;
   *cy = sy;

   max_cost *= cost;

   for (r = 0; r < 4; r++) {
      for (dx = -r; dx < r + 1; dx++) {
         for (dy = -r; dy < r + 1; dy++) {
            if (in_rect
                (tx + dx, ty + dy, 0, 0, get_map_width() - 1,
                 get_map_height() - 1)) {
               if (node_eval.data[tile_index(tx + dx, ty + dy)]) {
                  cost =
                     heuristic(tx, ty, tx + dx, ty + dy) * heuristic(sx, sy,
                                                                     tx + dx,
                                                                     ty + dy);
                  if (cost < max_cost) {
                     *cx = tx + dx;
                     *cy = ty + dy;
                     max_cost = cost;
                  }
               }
            }
         }
      }
      
      if (*cx != sx && *cy != sy)
         return;
   }
}
