/**********************************************/
/* Evert Glebbeek 2003, 2005                  */
/* eglebbk@dds.nl                             */
/**********************************************/
#include <allegro.h>
#include <stdlib.h>
#include <string.h>
#include "items.h"
#include "bbox.h"

#define MAX_ITEMS    (128*1024)
#define MAX_PROJECTS (128*1024)

typedef struct ITEM_DATA {
   int id;              /* ID number */
   char *name;          /* Descriptive name */
   int pickup_sound;    /* Sound to be played when picked up */
   int blocks;          /* Blocks movement yes or no */
   int lifetime;        /* Item lifetime, indicates how long it will last */
                        /* 0 for items that will last indefinately */
   BSHAPE *bbox;        /* collision bounding box */
   char *desc;          /* Descriptive string */
} ITEM_DATA;

static ITEM *item_chunk = NULL;
static ITEM **item_stack = NULL;
static int item_stack_counter = 0;

static PROJECTILE *projectile_chunk = NULL;
static PROJECTILE **projectile_stack = NULL;
static int projectile_stack_counter = 0;

static ITEM_DATA item_data[] = {
   { ITEM_BOW,          "Bow",                  11, FALSE,  0, &tile_bb,   "E: %dd%d   I: %dd%d   C: %dd%d   P: %dd%d" },
   { ITEM_CROSSBOW,     "Crossbow",             11, FALSE,  0, &tile_bb,   "E: %dd%d   I: %dd%d   C: %dd%d   P: %dd%d" },
   { ITEM_CLUB,         "Wooden club",          11, FALSE,  0, &tile_bb,   "E: %dd%d   I: %dd%d   C: %dd%d   P: %dd%d" },
   { ITEM_MACE,         "Mace",                 11, FALSE,  0, &tile_bb,   "E: %dd%d   I: %dd%d   C: %dd%d   P: %dd%d" },
   { ITEM_DAGGER,       "Dagger",               11, FALSE,  0, &tile_bb,   "E: %dd%d   I: %dd%d   C: %dd%d   P: %dd%d" },
   { ITEM_DAGGER2,      "Dagger of Striking",   11, FALSE,  0, &tile_bb,   "E: %dd%d   I: %dd%d   C: %dd%d   P: %dd%d" },
   { ITEM_SWORD,        "Sword",                11, FALSE,  0, &tile_bb,   "E: %dd%d   I: %dd%d   C: %dd%d   P: %dd%d" },
   { ITEM_SWORD2,       "Excalibur",            11, FALSE,  0, &tile_bb,   "E: %dd%d   I: %dd%d   C: %dd%d   P: %dd%d" },
   { ITEM_SHIELD,       "Shield",               11, FALSE,  0, &tile_bb,   "Reduced damage from impact (I) attacks" },
   { ITEM_RINGREGEN,    "Ring of Regeneration", 11, FALSE,  0, &tile_bb,   "Revive when slain" },
   { ITEM_TELLRING,     "Tell's Ring",          11, FALSE,  0, &tile_bb,   "Set arrow target" },
   { ITEM_OWLFIG,       "Owl Figurine",         11, FALSE,  0, &tile_bb,   "?" },
   { ITEM_CATFIG,       "Cat figurine",         11, FALSE,  0, &tile_bb,   "?" },
   { ITEM_SNAKEFIG,     "Snake figurine",       11, FALSE,  0, &tile_bb,   "?" },
   { ITEM_CHAINMAIL,    "Chainmail",            11, FALSE,  0, &tile_bb,   "Reduced damage from cutting (C) attacks" },
   { ITEM_HELMET,       "Helmet",               11, FALSE,  0, &tile_bb,   "Reduced damage" },

   { ITEM_ARROW,        "Arrows",               14, FALSE,320, &item_bb,   NULL },
   { ITEM_SARROW,       "Steel-tipped arrows",  14, FALSE,320, &item_bb,   NULL },
   { ITEM_FARROW,       "Fiery arrows",         14, FALSE,320, &item_bb,   NULL },
   { ITEM_BOMB,         "Bombs",                14, FALSE,180, &item_bb,   NULL },
   { ITEM_QHEART,       "QHeart",               11, FALSE,  0, &tile_bb,   NULL },
   { ITEM_HEARTC,       "Heart Container",      11, FALSE,  0, &tile_bb,   NULL },
   { ITEM_KEY,          "Key",                  14, FALSE,  0, &item_bb,   NULL },
   { ITEM_POTION,       "Potion",               13, FALSE,  0, &item_bb,   NULL },
   { ITEM_HEART,        "Heart",                13, FALSE,140, &item_bb,   NULL },
   { ITEM_EXIT,         "Exit",                 -1, FALSE,  0, &tile_bb,   NULL },
   { ITEM_CANDLEHOLD,   "Candleholder",         16, FALSE,180, &item_bb,   NULL },
   { ITEM_CHALICE,      "Chalice",              16, FALSE,180, &item_bb,   NULL },
   { ITEM_GEM,          "Gem",                  16, FALSE,180, &item_bb,   NULL },
   { ITEM_POISON,       "Poison",                6, FALSE,160, &item_bb,   NULL },
   { ITEM_TRAP,         "Trap",                  6, FALSE,  0, &tile_bb,   NULL },
   
   { ITEM_SWITCHUP,     "Switch(up)",            7, FALSE,  0, &tile_bb,   NULL },
   { ITEM_SWITCHDN,     "Switch(down)",         -1, FALSE,  0, &tile_bb,   NULL },
   { ITEM_SWITCHWGHTU,  "Switch(presup)",       -1, FALSE,  0, &tile_bb,   NULL },
   { ITEM_SWITCHWGHTD,  "Switch(presdown)",     -1, FALSE,  0, &tile_bb,   NULL },
      
   { ITEM_BOSSGATE,     "Bossgate",             -1, FALSE,  0, &tile_bb,   NULL },
   { ITEM_BOSSEXIT,     "Gameexit",             -1, FALSE,  0, &tile_bb,   NULL },
   
   { ITEM_CHESTOPEN,    "Open chest",           -1, TRUE,   0, &tile_bb,   NULL },
   { ITEM_CHESTCLOSED,  "Closed chest",          5, TRUE,   0, &tile_bb,   NULL },
   { ITEM_BLOCK,        "Block",                -1, TRUE,   0, &tile_bb,   NULL },
   { ITEM_BLOCKCRACK,   "Cracked block",        -1, TRUE,   0, &tile_bb,   NULL },
   { ITEM_BLOCKLOCKED,  "Locked block",          7, TRUE,   0, &tile_bb,   NULL },
   { ITEM_STATUE,       "Statue",               -1, TRUE,   0, &tile_bb,   NULL },

   { 0,                 NULL,                   -1, FALSE,  0, NULL,       NULL },
};

ITEM *alloc_item(void)
{
   ASSERT(item_stack_counter<MAX_ITEMS);
   ASSERT(item_stack);
   
   /* return the address of the last free item in the list */
   return item_stack[item_stack_counter++];
}

/* Frees the memory in use by an item, including its children */
void free_item(ITEM *l)
{
   ASSERT(item_stack_counter);
   ASSERT(item_stack);
   
   if (l->child) free_item(l->child);   
   item_stack[--item_stack_counter]=l;
}

int get_item_count(void)
{
   return item_stack_counter;
}


PROJECTILE *alloc_projectile(void)
{
   ASSERT(projectile_stack_counter<MAX_PROJECTS);
   ASSERT(projectile_stack);
   
   /* return the address of the last free projectile in the list */
   return projectile_stack[projectile_stack_counter++];
}

void free_projectile(PROJECTILE *l)
{
   ASSERT(projectile_stack_counter);
   ASSERT(projectile_stack);
   
   projectile_stack[--projectile_stack_counter]=l;
}

int get_projectile_count(void)
{
   return projectile_stack_counter;
}

int init_objects(void)
{
   int c;

   item_chunk = realloc(item_chunk, sizeof(ITEM)*MAX_ITEMS);
   item_stack = realloc(item_stack, sizeof(ITEM *)*MAX_ITEMS);
   
   item_stack_counter = 0;
   for (c=0; c<MAX_ITEMS; c++)
      item_stack[c] = &(item_chunk[c]);

   projectile_chunk = realloc(projectile_chunk, sizeof(PROJECTILE)*MAX_PROJECTS);
   projectile_stack = realloc(projectile_stack, sizeof(PROJECTILE *)*MAX_PROJECTS);
   
   projectile_stack_counter = 0;
   for (c=0; c<MAX_PROJECTS; c++)
      projectile_stack[c] = &(projectile_chunk[c]);

   return (sizeof(ITEM)+sizeof(ITEM *))*MAX_ITEMS;
}

void free_objects(void)
{
   free(item_chunk);
   free(item_stack);
   
   item_chunk = NULL;
   item_stack = NULL;
   item_stack_counter = 0;

   free(projectile_chunk);
   free(projectile_stack);
   
   projectile_chunk = NULL;
   projectile_stack = NULL;
   projectile_stack_counter = 0;
}

/* Get the name (string) of the item associated with the id */
char *get_item_name(int id)
{
   int c;
   for(c=0; item_data[c].name; c++) {
      if (item_data[c].id == id)
         return item_data[c].name;
   }
   return NULL;
}

/* Get the short description of the item associated with the id */
char *get_item_desc(int id)
{
   int c;
   for(c=0; item_data[c].name; c++) {
      if (item_data[c].id & id)
         return item_data[c].desc;
   }
   return NULL;
}

/* Get the id of the item associated with the string name */
int get_item_id(char *name)
{
   int c;
   for(c=0; item_data[c].name; c++) {
      if (ustricmp(item_data[c].name, name) == 0)
         return item_data[c].id;
   }
   return ITEM_NONE;
}

/* Get the sound played on item pickup  */
int get_item_sound(int id)
{
   int c;
   for(c=0; item_data[c].name; c++) {
      if (item_data[c].id == id)
         return item_data[c].pickup_sound;
   }
   return -1;
}

/* Returns TRUE if the item blocks the passage */
int item_blocks(int id)
{
   int c;
   for(c=0; item_data[c].name; c++) {
      if (item_data[c].id == id)
         return item_data[c].blocks;
   }
   return FALSE;
}

/* Get the lifetime of the item associated with the id */
int get_item_lifetime(int id)
{
   int c;
   for(c=0; item_data[c].name; c++) {
      if (item_data[c].id == id)
         return item_data[c].lifetime;
   }
   return 0;
}

BSHAPE *get_item_bbox(int id)
{
   int c;
   for(c=0; item_data[c].name; c++) {
      if (item_data[c].id == id)
         return item_data[c].bbox;
   }
   return &tile_bb;
}
