/**********************************************/
/* Script compiler and parser                 */
/* Evert Glebbeek 2002, 2004                  */
/* eglebbk@dds.nl                             */
/**********************************************/
#include <allegro.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "script.h"
#include "skeyword.h"
#include "global.h"
#include "str.h"

/* Temporary structure for storing labels */
typedef struct {
   char *name;
   int location;
} PARSE_LABEL;

/* Temporary structure for storing a script while compiling */
typedef struct {
   unsigned char *code;
   int size;
   int max_size;
} PARSE_SCRIPT;

static int pass = 1;
static PARSE_LABEL *pl = NULL;
static size_t num_labels = 0;
static SCRIPT *script = NULL;
static int num_scripts = 0;

/* Output compiler messages */
//static void output_msg(char *msg, ...) __attribute__ ((format(printf, 1, 2)));
static void output_msg(char *msg, ...)
{
   char *s = malloc(4096);

   va_list ap;

   va_start(ap, msg);
   uvsprintf(s, msg, ap);
   va_end(ap);

   allegro_message(s);

   free(s);
}

/* Returns TRUE if the string s is an integer number */
static int is_numeric_integer(char *s)
{
   char *parse;

   if (s[0]=='-' || s[0]=='+')
      s++;

   for (parse = s; isdigit(parse[0]); parse++);

   if (parse[0])
      return FALSE;
   else
      return TRUE;
}

/* Returns TRUE if the string s is a floating point number */
static int is_numeric_float(char *s)
{
   char *parse;
   int dotc = 0;

   if (s[0]=='-' || s[0]=='+')
      s++;

   for (parse = s; isdigit(parse[0])||parse[0]=='.'; parse++)
      if (parse[0]=='.')
         dotc++;

   if (parse[0] || dotc!=1)
      return FALSE;
   else
      return TRUE;
}

static int is_numeric(char *s)
{
   int c;

   if (is_numeric_integer(s)||is_numeric_float(s)) {
      return TRUE;
   } else {
      for(c=0; identifier[c].parse_name; c++) {
         if (streq(s, identifier[c].parse_name)) {
            return identifier[c].is_a_number;
         }
      }
   }

   return FALSE;
}

/* Helper function: make sure there is enough space to store an additional */
/*  n bytes of script space */
static inline void reserve_script_size(PARSE_SCRIPT *ps, int n)
{
   while (ps->size+n>=ps->max_size) {
      ps->max_size+=8;
      ps->code = realloc(ps->code, ps->max_size * sizeof *(ps->code));
   }
}

/* Helper function: write opcode */
static inline void write_script_opcode(PARSE_SCRIPT *ps, int op)
{
   reserve_script_size(ps, 1);
   ps->code[ps->size] = op;
   ps->size++;
}

static inline void write_script_int(PARSE_SCRIPT *ps, int n)
{
   reserve_script_size(ps, 4);
   ps->code[ps->size+0] = n & 0xFF;
   ps->code[ps->size+1] = (n>>8) & 0xFF;
   ps->code[ps->size+2] = (n>>16) & 0xFF;
   ps->code[ps->size+3] = (n>>24) & 0xFF;
   ps->size+=4;
}

static inline void write_script_float(PARSE_SCRIPT *ps, float f)
{
   write_script_int(ps, ftofix(f));
}

/* Convert the string s to its numerical translation */
static inline void write_string_translation(PARSE_SCRIPT *ps, char *s)
{
   float f;
   int n;

   if (is_numeric_integer(s)) {
      sscanf(s, "%d", &n);
      write_script_int(ps, n);
   } else if (is_numeric_float(s)) {
      sscanf(s, "%g", &f);
      write_script_float(ps, f);
   } else {
      for (n=0; identifier[n].parse_name; n++) {
         if (streq(s, identifier[n].parse_name)) {
            write_script_int(ps, identifier[n].num_value);
            return;
         }
      }
      /* Identifier not found --> there is an error in the script! */
      if (pass==2)
         output_msg("Unknown identifier (treated as 0): %s\n", s);
      write_script_int(ps, 0);
   }
}

/* Generate the correct push command for the argument string s */
static inline void write_script_push(PARSE_SCRIPT *ps, char *arg)
{
   if (is_numeric(arg)) {
      write_script_opcode(ps, CMD_PUSHN);
   } else {
      write_script_opcode(ps, CMD_LOAD);
   }
   write_string_translation(ps, arg);
}

/* Generate the correct pop command for the argument string s */
static inline void write_script_pop(PARSE_SCRIPT *ps, char *arg)
{
   if (is_numeric(arg)) {
      write_script_opcode(ps, CMD_POPN);
   } else {
      write_script_opcode(ps, CMD_STORE);
      write_string_translation(ps, arg);
   }
}

/* Compiler helper function: store label and it's location */
/* Has no effect during pass 2 */
static void push_label(char *name, int ptr)
{
   if (pass==1) {
      pl = realloc(pl, (num_labels + 1) * sizeof(PARSE_LABEL));
      pl[num_labels].name = strdup(name);
      pl[num_labels].location = ptr;
      num_labels++;
   }
}

/* Compiler helper function: convert label name to it's offset */
/* Has no effect during pass 1 */
static int find_label(char *name)
{
   unsigned int c;

   TRACE("#%d:label %s", pass, name);
   if (pass==2) {
      for (c = 0; c < num_labels; c++) {
         if (streq(name, pl[c].name)) {
            TRACE("@%d\n", pl[c].location);
            return pl[c].location;
         }
      }

   }
   TRACE(" not found\n");
   return 0;
}

/* Clears up all temporary data from the label parser */
static void clear_label_stack(void)
{
   unsigned int c;

   for (c = 0; c < num_labels; c++) {
      free(pl[c].name);
   }
   if (pl)
      free(pl);
   pl = NULL;
   num_labels = 0;
}

/* Convert a comma seperated list of arguments to an argc and argv structure */
static void string_to_argc_argv(char *l, int *argc, char ***argv)
{
   char *s;
   char *eol;
   int c;

   c = 0;
   /* Find all the comma's in s */
   for (s = l; s; s = strstr(s, ",")) {
      c++;
      s++;
   }

   (*argc) = c;

   if (c) {
      *argv = malloc(c * sizeof *argv);
      s = l;
      c = 0;
      while (s) {
         eol = strstr(s, ",");
         if (eol) {
            eol[0]='\0';
            eol++;
         }
         (*argv)[c] = strdup(s);

         strflushl((*argv)[c]);
         strflushr((*argv)[c]);

         c++;

         s = eol;
      }
   }
}

/* Returns 0 if the script was read succesfully, or -1 for end-of-file */
static int parse_script(FILE *f, char *filename, int *line, SCRIPT *script)
{
   PARSE_SCRIPT ps = {NULL, 0, 0};
   char s[256];
   char *opt;
   char *eol;
   char **argv;
   int argc;
   int opcode_only;
   int c;
   int n;
   int opcode = CMD_NOP;
   int section;

   ASSERT(script);
   script->script_name = NULL;
   script->script = NULL;

   while(fgets(s, 256, f)) {
      /* Notify caller that we've read a new line from the file */
      (*line)++;

      /* Skip comments */
      eol = strstr(s, ";");
      if (eol)
         eol[0] = '\0';

      /* Clear end-of-line marker if it is present */
      eol = strstr(s, "\n");
      if (eol)
         eol[0] = '\0';

      /* Replace tab characters by spaces */
      for (eol = strstr(s, "\t"); eol; eol = strstr(eol, "\t"))
         eol[0]=' ';

      /* clear spaces from the beginning and end of the string */
      strflushl(s);
      strflushr(s);

      /* Now see if we have anything left to parse */
      if (s[0]) {
         /* Split command and arguments */

         opt = s;

         eol = strstr(opt, "= ");
         if (eol) {
            eol[1] = ',';
         }

         eol = strstr(opt, " ");
         if (eol) {
            eol[0] = '\0';
            eol++;

            /* Remove spaces from the start of the string */
            strflushl(eol);

            string_to_argc_argv(eol, &argc, &argv);
         } else {
            argc = 0;
         }

         /* commands can be specified in two ways:    */
         /*                                           */
         /*  opc arg1                                 */
         /*                                           */
         /*    or                                     */
         /*                                           */
         /*  push arg                                 */
         /*  $opc                                     */
         /*                                           */
         /* Detect which of these we're dealing with  */

         if (opt[0]=='$') {
            opt++;
            opcode_only=TRUE;
         } else {
            opcode_only=FALSE;
         }

         section = -1;

         /* Parse commands */

         /* General statements */
         if (streq(opt, "proc")) {
            ASSERT(argc);

            /* Command is a command to the compiler ,*/
            section = 0;

            script->script_name = strdup(argv[0]);
         } else if (argc==1 && streq(argv[0], "endp")) {
            /* Make sure commands are executed only within a script procedure */
            ASSERT(script->script_name);

            /* Command is a command to the compiler ,*/
            section = 0;

            script->script = ps.code;
            for(c=0; c<argc; c++)
               free(argv[c]);
            free(argv);

            /*
            printf("%s:\n", script->script_name);
            for(c=0; c<ps.size; c++)
               printf("%02x\t", script->script[c]);
            printf("\n");
            */

            return 0;
         } else if (streq(opt, "context")) {
            section = 0;
         } else if (streq(opt, "#include")) {
            section = 0;
            load_scripts(argv[0]);
         } else if (argc==1 && streq(opt, "label")) {
            char *l;

            /* Command is a command to the compiler ,*/
            section = 0;

            /* Make sure commands are executed only within a script procedure */
            ASSERT(script->script_name);

            l = malloc(strlen(script->script_name) + strlen(argv[0]) + 4);

            sprintf(l, "%s.%s", script->script_name, argv[0]);
            push_label(l, ps.size);

            free(l);
         } else if (streq(opt, "push")) {
            /* Push can be either pushn, or load, depending on it's argument  */
            /* The compiler will have to treat it as a special command and    */
            /* translate it in the proper context                             */
            section = 0;

            /* Make sure commands are executed only within a script procedure */
            ASSERT(script->script_name);
            ASSERT(argc==1);
            write_script_push(&ps, argv[0]);
         } else if (streq(opt, "pop")) {
            /* Push can be either pushn, or load, depending on it's argument  */
            /* The compiler will have to treat it as a special command and    */
            /* translate it in the proper context                             */
            section = 0;

            /* Make sure commands are executed only within a script procedure */
            ASSERT(script->script_name);
            if (argc==1) {
               write_script_pop(&ps, argv[0]);
            } else {
               write_script_opcode(&ps, CMD_POPN);
            }
         }

         /* Special `higher level' statements; these are compiled to several */
         /*  low level statements */
         if (argc==2 && streq(opt, "LOOP")) {
            char *l;

            /* High-level command: loop counter, label */
            /* Jumps to label if counter is not zero and decreases the counter */
            /* Compiled to:                                          */
            /* dec counter clrzf, cmp counter,0, je label            */
            ASSERT(script->script_name);
            ASSERT(!is_numeric(argv[0]));

            l = malloc(strlen(script->script_name) + strlen(argv[1]) + 4);
            sprintf(l, "%s.%s", script->script_name, argv[1]);

            write_script_push(&ps, argv[0]);
            write_script_opcode(&ps, CMD_DEC);
            write_script_opcode(&ps, CMD_CLRZF);
            write_script_opcode(&ps, CMD_PUSHN);
            write_script_int(&ps, 0);
            write_script_opcode(&ps, CMD_CMP);
            write_script_opcode(&ps, CMD_JE);
            write_script_int(&ps, find_label(l));

            free(l);

            section = 0;
         } else if (argc == 2 && streq(argv[0], "+=")) {
            /* High-level command: a+=b              */
            /* push a, push b, add, pop a            */
            ASSERT(script->script_name);
            ASSERT(!is_numeric(opt));
            write_script_push(&ps, opt);
            write_script_push(&ps, argv[1]);
            write_script_opcode(&ps, CMD_ADD);
            write_script_pop(&ps, opt);

            section = 0;
         } else if (argc == 2 && streq(argv[0], "-=")) {
            /* High-level command: a-=b              */
            /* push a, push b, sub, pop a            */
            ASSERT(script->script_name);
            ASSERT(!is_numeric(opt));
            write_script_push(&ps, opt);
            write_script_push(&ps, argv[1]);
            write_script_opcode(&ps, CMD_SUB);
            write_script_pop(&ps, opt);

            section = 0;
         } else if (argc == 2 && streq(argv[0], "*=")) {
            /* High-level command: a*=b              */
            /* push a, push b, mul, pop a            */
            ASSERT(script->script_name);
            ASSERT(!is_numeric(opt));
            write_script_push(&ps, opt);
            write_script_push(&ps, argv[1]);
            write_script_opcode(&ps, CMD_MUL);
            write_script_pop(&ps, opt);

            section = 0;
         } else if (argc == 2 && streq(argv[0], "/=")) {
            /* High-level command: a/=b              */
            /* push a, push b, div, pop a            */
            ASSERT(script->script_name);
            ASSERT(!is_numeric(opt));
            write_script_push(&ps, opt);
            write_script_push(&ps, argv[1]);
            write_script_opcode(&ps, CMD_DIV);
            write_script_pop(&ps, opt);

            section = 0;
         } else if (argc == 2 && streq(argv[0], ":=")) {
            /* High-level command: a:=b              */
            /* push b, pop a                         */
            ASSERT(script->script_name);
            ASSERT(!is_numeric(opt));
            write_script_push(&ps, argv[1]);
            write_script_pop(&ps, opt);

            section = 0;
         } else if (argc == 2 && streq(argv[0], "==")) {
            /* High-level command: a==b              */
            /* push a, push b, cmp                    */
            ASSERT(script->script_name);
            ASSERT(!is_numeric(opt));
            write_script_push(&ps, opt);
            write_script_push(&ps, argv[0]);
            write_script_opcode(&ps, CMD_CMP);

            section = 0;
         }


         /* Check keywords, but only if no section has yet been set */
         ASSERT(opt);

         if (section) {
            for (c=0; keyword[c].mnemonic; c++)
               if (streq(opt, keyword[c].mnemonic)) {
                  break;
               }

            /* Normal keyword? */
            if (keyword[c].mnemonic) {
               section = 1;
               opcode = keyword[c].bytecode;
            } else {
               /* No - check flow control */
               for (c=0; fkeyword[c].mnemonic; c++)
                  if (streq(opt, fkeyword[c].mnemonic))
                     break;

               if (fkeyword[c].mnemonic) {
                  section = 2;
                  opcode = fkeyword[c].bytecode;
               } else {
                  /* Again no - check stack based operators */
                  for (c=0; skeyword[c].mnemonic; c++)
                     if (streq(opt, skeyword[c].mnemonic))
                        break;

                  if (skeyword[c].mnemonic) {
                     section = 3;
                     opcode = skeyword[c].bytecode;
                  } else {
                     /* Check the special syntax `push' and `pop' */
                     /*  for the explicid forms pushn/popn/load/store */
                  }
               }
            }

            switch (section) {
               case 0:        /* System/compiler command - already processed */
                  break;
               case 1:        /* Keyword/normal operator */
                  /* Make sure commands are executed only within a script procedure */
                  ASSERT(script->script_name);

                  /* Write arguments if they were specified */
                  if (argc) {
                     for (n=0; n<argc; n++) {
                        write_script_push(&ps, argv[n]);
                     }
                  }
                  write_script_opcode(&ps, opcode);
                  break;
               case 2:        /* Flow control            */
                  {
                     char *l;
                     /* Make sure commands are executed only within a script procedure */
                     ASSERT(script->script_name);
                     ASSERT(argc==1);

                     l = malloc(strlen(script->script_name) + strlen(argv[0]) + 4);
                     sprintf(l, "%s.%s", script->script_name, argv[0]);

                     write_script_opcode(&ps, opcode);
                     write_script_int(&ps, find_label(l));

                     free(l);
                     break;
                  }
               case 3:        /* Stack based operation - explicid   */
                  /* Make sure commands are executed only within a script procedure */
                  ASSERT(script->script_name);
                  ASSERT(argc<2);

                  write_script_opcode(&ps, opcode);
                  if (argc)
                     write_string_translation(&ps, argv[0]);
                  break;
               default:
                  /* Not a recognized command */
                  n = *line;
                  output_msg("Error: %s, line %d: unknown command %s\n", filename, n, opt);
            }
         }

         if (argc) {
            for(c=0; c<argc; c++)
               free(argv[c]);
            free(argv);
         }
      }
   }

   return EOF;
}

/* Load scripts from a script source file */
int load_scripts(char *filename)
{
   SCRIPT test;
   FILE *f;
   int ln;
   int r;

   ln = 0;

   f = fopen(filename, "r");
   if (!f) {
      output_msg("Cannot open file: %s (read error)\n", filename);
      return FALSE;
   }

   switch (pass) {
      case 1:
         do {
            r = parse_script(f, filename, &ln, &test);
            /*
            if (r>-1) {
               int c;
               for (c=0; test.script[c]!=CMD_RETURN; c++) {
                  printf("%02X ", test.script[c]);
               }
               printf("\n");
            }
            */
         } while(r!=-1);


         /* Now do a second pass to get label offsets right */
         pass = 2;
         load_scripts(filename);
         break;
      case 2:
         do {
            r = parse_script(f, filename, &ln, &test);
            if (r==0) {    /* Only store script if it ended with endp */
               num_scripts++;
               script = realloc(script, num_scripts*sizeof *script);
               script[num_scripts-1] = test;
            }
         } while(r!=-1);

         /* Reset the pass flag so subsequent scripts can be read */
         pass = 1;
         break;
      default:
         output_msg("Invalid parser pass (internal error/corrupted pointer\n");
         break;

   }

   fclose(f);

   return TRUE;
}

void unload_scripts(void)
{
   int c;
   clear_label_stack();

   for (c=0; c<num_scripts; c++) {
      free(script[c].script_name);
      free(script[c].script);
   }
   free(script);
   script = NULL;
   num_scripts = 0;

}

SCRIPT *find_script(char *name)
{
   int n;

   for(n=0; n<num_scripts; n++) {
      if (streq(script[n].script_name, name))
         return &(script[n]);
   }

   return NULL;
}
