/**********************************************/
/* Rise of the Tribes                         */
/* C version, Take 2                          */
/* Evert Glebbeek 1998, 2003                  */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include <math.h>
#include "opcodes.h"
#include "dialog.h"
#include "playgame.h"
#include "map.h"
#include "maptile.h"
#include "unit.h"
#include "ugrp.h"
#include "script.h"
#include "utasks.h"
#include "uactions.h"
#include "skeyword.h"
#include "udraw.h"
#include "global.h"
#include "unitmap.h"
#include "gamedraw.h"
#include "gamemsg.h"
#include "cheats.h"
#include "path.h"
#include "vm.h"

/********************/
/* Helper functions */
/********************/
static inline int pop_script_stack(SCRIPT *script)
{
   ASSERT(script->sp<SCRIPT_STKSIZE);

   return script->ss[script->sp++];
}

static inline void push_script_stack(SCRIPT *script, int n)
{
   ASSERT(script->sp);

   script->ss[--script->sp] = n;
}

static inline int read_script_int(SCRIPT *script, int offset)
{
   return script->script[offset] | script->script[offset + 1] << 8 |
           script->script[offset + 2] << 16 | script->script[offset + 3] << 24;
}



/**************************/
/* Opcode implementations */
/**************************/

/* Mathematical functions */
static int opcode_and(void *obj, SCRIPT *script, int *loop)
{
   int n;

   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   n = pop_script_stack(script) & pop_script_stack(script);

   if (n==0)
      script->flags |= SCRIPT_ZF;
   if (n<0)
      script->flags |= SCRIPT_SF;
   push_script_stack(script, n);

   script->ip++;
   return 0;
}

static int opcode_or(void *obj, SCRIPT *script, int *loop)
{
   int n;

   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   n = pop_script_stack(script) | pop_script_stack(script);

   if (n==0)
      script->flags |= SCRIPT_ZF;
   if (n<0)
      script->flags |= SCRIPT_SF;
   push_script_stack(script, n);

   script->ip++;
   return 0;
}

static int opcode_xor(void *obj, SCRIPT *script, int *loop)
{
   int n;

   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   n = pop_script_stack(script) ^ pop_script_stack(script);

   if (n==0)
      script->flags |= SCRIPT_ZF;
   if (n<0)
      script->flags |= SCRIPT_SF;
   push_script_stack(script, n);

   script->ip++;
   return 0;
}

static int opcode_not(void *obj, SCRIPT *script, int *loop)
{
   int n;

   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   n = ~pop_script_stack(script);

   if (n==0)
      script->flags |= SCRIPT_ZF;
   if (n<0)
      script->flags |= SCRIPT_SF;
   push_script_stack(script, n);

   script->ip++;
   return 0;
}

static int opcode_test(void *obj, SCRIPT *script, int *loop)
{
   int n;

   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   n = pop_script_stack(script) & pop_script_stack(script);

   if (n==0)
      script->flags |= SCRIPT_ZF;
   if (n<0)
      script->flags |= SCRIPT_SF;

   script->ip++;
   return 0;
}

static int opcode_cmp(void *obj, SCRIPT *script, int *loop)
{
   int n1, n2;
   
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   n1 = pop_script_stack(script);
   n2 = pop_script_stack(script);
   
   if (n1==n2)
      script->flags|=SCRIPT_ZF;

   script->ip++;
   return 0;
}

static int opcode_itof(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   script->ss[script->sp] = itofix(script->ss[script->sp]);

   script->ip++;
   return 0;
}

static int opcode_ftoi(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   script->ss[script->sp] = fixtoi(script->ss[script->sp]);

   script->ip++;
   return 0;
}

static int opcode_sin(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   script->ss[script->sp] = fixsin(script->ss[script->sp]);

   script->ip++;
   return 0;
}

static int opcode_cos(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   script->ss[script->sp] = fixcos(script->ss[script->sp]);

   script->ip++;
   return 0;
}


static int opcode_tan(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   script->ss[script->sp] = fixtan(script->ss[script->sp]);

   script->ip++;
   return 0;
}


static int opcode_asin(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   script->ss[script->sp] = (360*fixasin(script->ss[script->sp]))/256;

   script->ip++;
   return 0;
}


static int opcode_acos(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   script->ss[script->sp] = (360*fixacos(script->ss[script->sp]))/256;

   script->ip++;
   return 0;
}


static int opcode_atan(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   script->ss[script->sp] = (360*fixatan(script->ss[script->sp]))/256;

   script->ip++;
   return 0;
}


static int opcode_atan2(void *obj, SCRIPT *script, int *loop)
{
   float f1, f2;
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   f1 = pop_script_stack(script);
   f2 = pop_script_stack(script);

   push_script_stack(script, fixatan2(f1, f2));

   script->ip++;
   return 0;
}


static int opcode_exp(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   script->ss[script->sp] = ftofix(exp(fixtof(script->ss[script->sp])));

   script->ip++;
   return 0;
}

static int opcode_log(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   script->ss[script->sp] = ftofix(log(fixtof(script->ss[script->sp])));

   script->ip++;
   return 0;
}

static int opcode_add(void *obj, SCRIPT *script, int *loop)
{
   int n1, n2;
   float f1, f2;
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   if(script->flags & SCRIPT_FLOAT) {
      f1 = fixtof(pop_script_stack(script));
      f2 = fixtof(pop_script_stack(script));

      push_script_stack(script, ftofix(f1+f2));
   } else {
      n1 = pop_script_stack(script);
      n2 = pop_script_stack(script);

      push_script_stack(script, n1+n2);
   }

   script->ip++;
   return 0;
}

static int opcode_sub(void *obj, SCRIPT *script, int *loop)
{
   int n1, n2;
   float f1, f2;
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   if(script->flags & SCRIPT_FLOAT) {
      f1 = fixtof(pop_script_stack(script));
      f2 = fixtof(pop_script_stack(script));

      push_script_stack(script, ftofix(f2-f1));
   } else {
      n1 = pop_script_stack(script);
      n2 = pop_script_stack(script);

      push_script_stack(script, n2-n1);
   }

   script->ip++;
   return 0;
}

static int opcode_mul(void *obj, SCRIPT *script, int *loop)
{
   int n1, n2;
   float f1, f2;
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   if(script->flags & SCRIPT_FLOAT) {
      f1 = fixtof(pop_script_stack(script));
      f2 = fixtof(pop_script_stack(script));

      push_script_stack(script, ftofix(f1*f2));
   } else {
      n1 = pop_script_stack(script);
      n2 = pop_script_stack(script);

      push_script_stack(script, n1*n2);
   }

   script->ip++;
   return 0;
}

static int opcode_div(void *obj, SCRIPT *script, int *loop)
{
   int n1, n2;
   float f1, f2;
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   if(script->flags & SCRIPT_FLOAT) {
      f1 = fixtof(pop_script_stack(script));
      f2 = fixtof(pop_script_stack(script));

      push_script_stack(script, ftofix(f1/f2));
   } else {
      n1 = pop_script_stack(script);
      n2 = pop_script_stack(script);

      push_script_stack(script, n1/n2);
   }

   script->ip++;
   return 0;
}

static int opcode_inc(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   script->ss[script->sp]++;

   script->ip++;
   return 0;
}

static int opcode_dec(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   script->ss[script->sp]--;

   script->ip++;
   return 0;
}

static int opcode_neg(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(script);
   ASSERT(obj);
   ASSERT(loop);

   script->ss[script->sp] = -script->ss[script->sp];

   script->ip++;
   return 0;
}

/* General operators */
static int opcode_nop(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   script->ip++;
   return 0;
}

/* update unit status and orders */
static int opcode_reset(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   int n;

   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);
   
   /* Don't do anything if the units script processor is locked */
   if (c->flags & CF_LOCKED) {
      (*loop) = 0;
      return 0;
   }
   
   c->script_target = c;
   c->flags |= CF_LOCKED;
   c->progress = 0;
   script->flags &= ~(SCRIPT_CANCEL|SCRIPT_SIGNAL);
   c->flags &= ~(CF_INVISIBLE | CF_PROGRESS);
   if (c->draw_dir != c->direction) {
      unregister_unit(c);
      c->draw_dir = c->direction;
      register_unit(c);
      mark_unit(c);
      set_screen_changed();
   }
   /* Building sites should take their icon/hit points/etc from what they'll */
   /*  become when they're done. The end product is in the arg1 variable */
   if (c->flags&CC_BUILDSITE) {
      c->icon = get_char_icon(script->argv[1]);
      c->name = get_char_name(script->argv[1]);
      c->maxhp = get_char_maxhp(script->argv[1]);
      c->maxmp = get_char_maxmp(script->argv[1]);
      for (n=0; n<8; n++) {
         int r = get_char_resource_cost(script->argv[1], n);
         if (r>=0) {
            c->resource_cost[n] = r;
            mark_interface_area(INT_RESOURCE);
         }
      }
   }
   script->sp = 8;
   script->ip++;
   (*loop)--;
   
   return 0;
}

/* make unit invisible; atomic */
static int opcode_invsbl(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;

   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   c->flags |= CF_INVISIBLE;
   script->ip++;
   (*loop)--;
   return 0;
}

/* Treat a wait command as a nop command (that is, ignore arguments) */
static int opcode_wait_is_nop(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   pop_script_stack(script);
   *loop = 0;
   script->ip++;
   return 0;
}

/* Take the time needed to execute the 'wait' command into account */
static int opcode_wait1(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   script->wait = pop_script_stack(script) - 1;
   script->ip ++;
   *loop = 0;
   return 0;
}

/* Take the time needed to execute the 'wait' command into account */
/* This version of the WAIT command adjusts the wait time to the unit's */
/*  speed characteristic */
static int opcode_wait3(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;

   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   script->wait = pop_script_stack(script) * 10 / c->speed;
   if (script->wait)
      script->wait--;
   script->ip ++;
   *loop = 0;
   return 0;
}

static int opcode_showf(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;

   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   /* Is this a normal object or a map object? */
   if (is_mapobject(c->script_target)) {
      /* We store the tile pointer in the gfx structure */
      /* The base tile (the bottom layer, so to speak, is in the */
      /* c->tiles structure */
      /*
         MAP_TILE *t = (MAP_TILE *)c->script_target->icon;
         t->gfx_index = prog->prog[script->ip+1];
         t->gfx = get_tile_bitmap(t->gfx_index);
         mark_tile (c->script_target->tiles->tile);
       */
   } else {
      /* parameter: frame number */
      set_screen_changed();
      unregister_unit(c->script_target);
      c->script_target->current_frame = pop_script_stack(script);
      c->script_target->flags &= ~CF_INVISIBLE;
      register_unit(c->script_target);
      mark_unit(c);
   }
   script->ip++;
   return 0;
}

static int opcode_move(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   int x, y;
   int n = pop_script_stack(script);

   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   /* parameter: number of pixels */
   set_screen_changed();
   x = c->script_target->x +
      dir_to_dx(c->script_target->draw_dir) * n;
   y = c->script_target->y -
      dir_to_dy(c->script_target->draw_dir) * n;
   if ((x >= 0) && (y >= 0) && (x < TILE_WIDTH * (get_map_width() - 1)) &&
       (y < TILE_HEIGHT * (get_map_height() - 1))) {
      unregister_unit(c->script_target);
      c->script_target->x = x;
      c->script_target->y = y;
      register_unit(c->script_target);
      mark_unit(c);
   }
   script->ip++;
   return 0;
}

static int opcode_chdir(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   int n = pop_script_stack(script);

   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   if (n <= SCRA_NORTH_WEST) {
      c->script_target->direction = n;
   } else {
      switch (n) {
         case SCRA_RANDOM:
            break;
         case SCRA_LEFT:
            c->script_target->direction -= 2;
            c->script_target->direction %= 8;
            break;
         case SCRA_RIGHT:
            c->script_target->direction += 2;
            c->script_target->direction %= 8;
            break;
         case SCRA_BACKWARD:
            c->script_target->direction += 4;
            c->script_target->direction %= 8;
            break;
         default:
      }                         /* End of switch */
   }
   script->ip++;
   return 0;
}

static int opcode_alert(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   popup_message("Alert!");
   script->ip++;
   
   TRACE("script %p\n", script);
   TRACE("ip = %02x\n", script->ip);
   TRACE("sp = %02x\n", script->sp);
   TRACE("zf = %d sf = %d lf = %d cf = %d\n", script->flags&SCRIPT_ZF?1:0,script->flags&SCRIPT_SF?1:0,script->flags&SCRIPT_LOOP?1:0,script->flags&SCRIPT_CANCEL?1:0);
   TRACE("a1 = %d, a2 = %d, a3 = %d, ac = %d\n", script->argv[0], script->argv[1], script->argv[2], script->argc);
   TRACE("---\n");
   return 0;
}

static int opcode_return(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);
   
   /* Remove current control program, prepare for next one */
   c->flags &= ~CF_LOCKED;

   //printf ("Return from %s... script %p\n", c->name, script);

   if (script == c->ctask) {
      c->ctask = NULL;
      pop_task(c);
   }

   if (script == c->caction) {
      //printf ("Poping next action\n");
      c->caction = NULL;
      pop_action(c);
   }

   /*
   if (script == c->task[c->queuestart%c->queuesize].prog) {
      pop_task(c);
   }
   */

   /*
   if (c->action && script == c->action->data) {
      printf ("Poping next action\n");
      pop_action(c);
   }
   */

   //release_command_head(c);
   halt_script(c, script);
   *loop = 0;                   // We're absolutely really done now
   return 0;
}

static int opcode_jmp(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   script->ip = read_script_int(script, script->ip + 1);
   (*loop)--;                   // Don't return yet
   return 0;
}

/* Jump if Cancel flag is set */
static int opcode_jcf(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   /* Conditional jump (CF=1) */
   if (script->flags & SCRIPT_CANCEL) {
      script->ip = read_script_int(script, script->ip + 1);
   } else {
      script->ip += 5;
   }
   (*loop)--;                   // Don't return yet
   return 0;
}

static int opcode_je(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   /* Conditional jump (ZF=1) */
   if (script->flags & SCRIPT_ZF) {
      script->ip = read_script_int(script, script->ip + 1);
   } else {
      script->ip += 5;
   }
   (*loop)--;                   // Don't return yet
   return 0;
}

static int opcode_jne(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   /* Conditional jump (ZF=0) */
   if (script->flags & SCRIPT_ZF) {
      script->ip = read_script_int(script, script->ip + 1);
   } else {
      script->ip += 5;
   }
   (*loop)--;                   // Don't return yet
   return 0;
}

static int opcode_ja(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   /* Conditional jump (SF=0, ZF=0) */
   if (!(script->flags & (SCRIPT_ZF + SCRIPT_SF))) {
      script->ip = read_script_int(script, script->ip + 1);
   } else {
      script->ip += 5;
   }
   (*loop)--;                   // Don't return yet
   return 0;
}

static int opcode_jb(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   /* Conditional jump (SF=1, ZF=0) */
   if (script->flags & SCRIPT_ZF && !(script->flags & SCRIPT_SF)) {
      script->ip = read_script_int(script, script->ip + 1);
   } else {
      script->ip += 5;
   }
   (*loop)--;                   // Don't return yet
   return 0;
}

static int opcode_jae(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   /* Conditional jump (SF=0 or ZF=1) */
   if (!(script->flags & SCRIPT_SF) || (script->flags & SCRIPT_ZF)) {
      script->ip = read_script_int(script, script->ip + 1);
   } else {
      script->ip += 5;
   }
   (*loop)--;                   // Don't return yet
   return 0;
}

static int opcode_jbe(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   /* Conditional jump (SF=1 or ZF=1) */
   if (script->flags & (SCRIPT_ZF + SCRIPT_SF)) {
      script->ip = read_script_int(script, script->ip + 1);
   } else {
      script->ip += 5;
   }
   (*loop)--;                   // Don't return yet
   return 0;
}

static int opcode_clrzf(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);
   
   script->flags &= ~SCRIPT_ZF;
   script->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

static int opcode_chkd(void *obj, SCRIPT *script, int *loop)
{
   LINK *chr;
   UNITDATA *cdta;
   int dx, dy;
   unsigned int tx, ty;
   unsigned int x, y;
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   dx = dir_to_dx(c->script_target->draw_dir);
   dy = dir_to_dy(c->script_target->draw_dir);

   tx = c->script_target->x / TILE_WIDTH;
   ty = c->script_target->y / TILE_HEIGHT;
   x = tx + dx;
   y = ty - dy;

   if (tile_flags(x, y, c->script_target->layer) & TILE_BARRIER) {
      script->flags |= SCRIPT_ZF;
   } else {
      chr = get_block_char(x, y);
      while (chr) {
         cdta = chr->data;
         if ((cdta != c) && !(cdta->flags & CF_NOBLOCK) &&
             (cdta->layer == c->script_target->layer)) {
            script->flags |= SCRIPT_ZF;
            break;
         }
         chr = chr->next;
      }
   }


   script->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

static int opcode_chkc(void *obj, SCRIPT *script, int *loop)
{
   unsigned int tx, ty;
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   tx = c->script_target->x / TILE_WIDTH;
   ty = c->script_target->y / TILE_HEIGHT;

   if (tile_blocked(tx, ty, c->script_target->layer, NULL))
      script->flags |= SCRIPT_ZF;

   script->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

/* Clear current tile occupied flag */
static int opcode_clrc(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   clear_unit_block(c);

   script->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

/* Set current tile occupied flag */
static int opcode_setc(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   set_unit_block(c);

   script->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

/* Clear destination tile occupied flag */
static int opcode_clrd(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   clear_unit_dest_block(c);

   script->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

/* Set destination tile occupied flag */
static int opcode_setd(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   set_unit_dest_block(c);

   script->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

/* Check the progress indicator */
static int opcode_chkcounter(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   int n;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   n = pop_script_stack(script);

   if (c->progress >= n)
      script->flags |= SCRIPT_ZF;

   script->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

static int opcode_remove(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   c->flags &= ~CF_LOCKED;

   release_first_action(c);
   halt_script(c, script);
   remove_from_all_selections(c);
   mark_unit(c);
   unregister_unit(c);
   clear_all_actions(c);
   set_command_panel();
   release_unit(take_from_player(c->player, c));
   
   set_screen_changed();
   script->ip++;
   return 1;
}


/* Check if enough resources to create a certain unit */
/* Set ZF if ok */
static int opcode_chkres(void *obj, SCRIPT *script, int *loop)
{
   int n;
   int r;
   int rc;
   int ok = TRUE;
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);
   
   n = pop_script_stack(script);

   if (!(cheat_flags & CHEAT_CONTROL)) {
      for (r=1; r<3; r++) {

         rc = get_char_resource_cost(n, r);
         
         if ((rc>0) && (rc+get_player_upkeep(c->player, r) > get_player_provision(c->player, r))) {
            if (!(script->flags&SCRIPT_SIGNAL)) {
               broadcast_message(GAMEMSG_NOR0+r,c->player,c->player);
               script->flags|=SCRIPT_SIGNAL;
               set_screen_changed();
            }
            ok = FALSE;
         }
      }
      /* Check resource 0 (morale) */
      if ((get_char_resource_cost(n, 0)>0) && (get_char_resource_cost(n, 0)+get_player_upkeep(c->player, 0) > get_player_provision(c->player, 0))) {
         if (!(script->flags&SCRIPT_SIGNAL) && ok) {
            broadcast_message(GAMEMSG_NOR0+0,c->player,c->player);
            script->flags|=SCRIPT_SIGNAL;
            set_screen_changed();
         }
      }
   }
   
   if (ok)
      script->flags |= SCRIPT_ZF;

   script->ip ++;
   (*loop)--;
   return 0;
}

/* Create a child unit loaded into the current structure, or a structure with */
/*  the current unit loaded inside it */
static int opcode_spawn(void *obj, SCRIPT *script, int *loop)
{
   LINK *u;
   unsigned int n = 0;
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   /* Check cancel flag */
   /* We check that here because this instruction can wait until upkeep is */
   /*  available for the unit to build */
   if (script->flags&SCRIPT_CANCEL) {
      script->ip++;
      return 0;
   }

   /* Get unit type (first argument) */
   n = pop_script_stack(script);
   
   //popup_message("unit-type = %d", n);

   u = create_unit(n);

   /* If there is still data on the stack, we pass this as arguments to the */
   /*  init script */
   for (n=script->sp; n<8; n++) {
      pass_action_script_argument(u->data, script->ss[n]);
      //popup_message("argv[%d] = %d", 7-n, c->ss[n]);
   }

   /* Now we need to know what to do next: if the unit we just `spawned' is */
   /*  in fact a unit, then load it into the parent. */
   /* Otherwise, if the unit is a structure, then load the parent into the */
   /*  structure. */
   if (is_structure(u->data)) {
      /* Set coordinates of structure: here */
      set_unit_coors(u->data, c->x, c->y);

      if (c->flags & CF_SELECTED)
         mark_interface_area(INT_COMMANDS|INT_GROUP|INT_CAPTAIN|INT_INFO|INT_QUEUE);

      /* Remove worker from the map and put him inside the structure */
      mark_unit(c);
      unregister_unit(c);
      clear_unit_block(c);
      load_unit(c, u->data);
      c->flags |= (CF_NOORDERS + CF_NOSELECT);
      remove_from_active_selections(c);
      set_command_panel();
      free_link(take_from_player(c->player, c));

      /* Give the new structure to the player and register it */
      give_to_player(c->player, u);
      mark_unit(u->data);
      register_unit(u->data);

      set_screen_changed();
   } else {
      load_unit(u->data, c);
      free_link(u);
   }

   if (c->player==get_local_player()) {
      mark_interface_area(INT_RESOURCE);
      set_screen_changed();
   }

   script->ip++;
   return 0;
}

/* Unload the first of the loaded_units() */
static int opcode_unload(void *obj, SCRIPT *script, int *loop)
{
   LINK *u;
   int dx;
   int dy;
   int r = -1;
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   u = c->loaded_units;
   /* Remove unit from the list of loaded units */
   if (u) {
      c->loaded_units = c->loaded_units->next;
      if (c->loaded_units)
         c->loaded_units->prev = NULL;
      u->prev = u->next = NULL;
   } else {
      script->ip++;
      return 0;
   }

   for (r = 0; r < 12; r++)
      for (dy = -r; dy < r + c->gfx_y_size; dy++)
         for (dx = -r; dx < r + c->gfx_x_size; dx++) {
            if (in_rect
                (c->script_target->x / TILE_WIDTH + dx,
                 c->script_target->y / TILE_HEIGHT + dy, 0, 0,
                 get_map_width() - 1, get_map_height() - 1))
               if (!tile_blocked
                   (c->script_target->x / TILE_WIDTH + dx,
                    c->script_target->y / TILE_HEIGHT + dy,
                    c->script_target->layer, NULL)) {
                  set_unit_coors(u->data,
                                 c->script_target->x + dx * TILE_WIDTH,
                                 c->script_target->y + dy * TILE_HEIGHT);
                  r = -1;
                  goto unit_placed;
               }
         }
 unit_placed:

   if (r < 0) {
      /* Check rally points */
      /*
         if (is_structure(c->script_target)) {
         u->data->ai_target.x = c->script_target->ai_target.x;
         u->data->ai_target.y = c->script_target->ai_target.y;
         u->data->ai_target.unit = c->script_target->ai_target.unit;
         }
       */

      register_unit(u->data);
      mark_unit(u->data);
      set_unit_block(u->data);
      c = u->data;
      c->flags &= ~(CF_NOORDERS + CF_NOSELECT);
      give_to_player(c->script_target->player, u);
      set_screen_changed();
   } else {
      release_unit(u);
   }

   script->ip++;
   return 0;
}

static int opcode_cmdoff(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   c->flags |= CF_NOORDERS;
   script->ip++;
   return 0;
}

static int opcode_cmdon(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   c->flags &= ~CF_NOORDERS;
   script->ip++;
   return 0;
}

static int opcode_target(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   switch (pop_script_stack(script)) {
      case SCRA_AITARGT:       /* Ai-target */
         if (c->ai_target.unit)
            c->script_target = c->ai_target.unit;
         break;
      case SCRA_ATTCKER:       /* Attacker */
         if (c->attacker)
            c->script_target = c->attacker;
         break;
      case SCRA_PARENT:        /* Parent */
         if (c->parent)
            c->script_target = c->parent;
         break;
      case SCRA_SELF:          /* Self */
      default:                 /* Assume self */
         c->script_target = c;
         break;
   }                            /* end of switch */
   script->ip++;
   return 0;
}

static int opcode_change(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *cd = NULL;
   int n;
   int w;
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   w = pop_script_stack(script);
   n = pop_script_stack(script);

   switch (w) {
      case SCRA_AITARGT:       /* Ai-target */
         if (c->ai_target.unit)
            cd = c->ai_target.unit;
         break;
      case SCRA_ATTCKER:       /* Attacker */
         if (c->attacker)
            cd = c->attacker;
         break;
      case SCRA_PARENT:        /* Parent */
         if (c->parent)
            cd = c->parent;
         break;
      case SCRA_SELF:          /* Self */
      default:                 /* Assume self */
         cd = c;
         break;
   }                            /* end of switch */

   if (cd) {
      unregister_unit(cd);
      change_unit(cd, n);
      set_command_panel();
      if (c->flags & CF_SELECTED)
         mark_interface_area(INT_COMMANDS|INT_GROUP|INT_CAPTAIN|INT_INFO|INT_QUEUE);
      register_unit(cd);
      mark_unit(cd);
      set_screen_changed();
   }

   if (c->player==get_local_player()) {
      mark_interface_area(INT_RESOURCE);
      set_screen_changed();
   }

   script->ip++;
   return 0;
}

static int opcode_atomic(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   script->ip++;
   (*loop) -= 2;                // Don't return yet, even after next command
   return 0;
}

/* Advance progress indicator in unit construction */
static int opcode_advcounter(void *obj, SCRIPT *script, int *loop)
{
   int n;
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   n = pop_script_stack(script);

   c->progress += n;

   script->ip++;
   (*loop)--;
   return 0;
}

/* Advance progress indicator in unit construction */
static int opcode_showcounter(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   c->flags |= CF_PROGRESS;

   if (c->flags & CF_SELECTED) {
      mark_interface_area(INT_QUEUE);
      set_screen_changed();
   }
   script->ip++;
   (*loop)--;
   return 0;
}
/* Advance progress indicator in unit construction */
static int opcode_hidecounter(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   if (get_task_queue_count(c)<=1) {
      c->flags &= ~CF_PROGRESS;

      if (c->flags & CF_SELECTED) {
         mark_interface_area(INT_QUEUE);
         set_screen_changed();
      }
   }
   script->ip ++;
   (*loop)--;
   return 0;
}

/* Elementaty pushn and popn instructions */
/* These are (supposedly!) the only instructions that have their immediate */
/*  arguments inlined in the program code */

/* Push an immediate value onto the stack */
static int opcode_pushn(void *obj, SCRIPT *script, int *loop)
{
   int n;
   
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   n = read_script_int(script, script->ip + 1);

   push_script_stack(script, n);

   script->ip += 5;
   (*loop)--;
   return 0;
}

/* Pop a number off the stack (and throw it away) */
static int opcode_popn(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   pop_script_stack(script);

   script->ip ++;
   (*loop)--;
   return 0;
}

/* Store a game-state variable on the stack */
static int opcode_load(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   int n = read_script_int(script, script->ip + 1);

   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);
   
   switch(n) {
      case 0:
      case 1:
      case 2:
         push_script_stack(script, script->argv[n]);
         break;
      case 3:
         push_script_stack(script, script->argc);
         break;
      case SCRA_PATH:
         push_script_stack(script, (c->path!=NULL));
         break;
      default:
         break;
   }
   
   script->ip+=5;
   (*loop)--;
   return 0;
}

/* Retrieve a game-state variable off the stack */
static int opcode_store(void *obj, SCRIPT *script, int *loop)
{
   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);
   
   switch (read_script_int(script, script->ip + 1)) {
      case SCRA_ARG0:
         script->argv[0] = pop_script_stack(script);
         break;
      case SCRA_ARG1:
         script->argv[1] = pop_script_stack(script);
         break;
      case SCRA_ARG2:
         script->argv[2] = pop_script_stack(script);
         break;
      case SCRA_ARGC:
         script->argc = pop_script_stack(script);
         break;
      case SCRA_PATH:
         TRACE("Attempt to write to read-only variable: unit->path!\n");
         pop_script_stack(script);
         break;
      default:
         break;
   }
   
   script->ip+=5;
   (*loop)--;
   return 0;
}

static int opcode_chk_task_target(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;
   TARGET t1;
   TARGET t2;

   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);
   
   t1.unit = c;
   t1.x = c->x/TILE_WIDTH;
   t1.y = c->y/TILE_HEIGHT;
   t2 = c->task[c->queuestart%c->queuesize].target;
   if (targets_are_equal(t1, t2)) {
      script->flags|=SCRIPT_ZF;
      printf("Targets equal!\n");
   }

   printf("checktarget: %s: target %p (%d, %d) @ target(%d, %d)\n", c->name, t2.unit, t2.x, t2.y, t1.x, t1.y);
   
   script->ip++;
   return 0;
}

static int opcode_move_task_target(void *obj, SCRIPT *script, int *loop)
{
   UNITDATA *c = obj;

   ASSERT(obj);
   ASSERT(script);
   ASSERT(loop);

   set_unit_target(c, c->task[c->queuestart%c->queuesize].target);
   
   printf("(%d, %d) -> (%d, %d)\n", c->x / TILE_WIDTH, c->y / TILE_HEIGHT,
                   c->ai_target.x, c->ai_target.y);
   
   if (path_exists(c->x / TILE_WIDTH, c->y / TILE_HEIGHT,
                   c->ai_target.x, c->ai_target.y, c->layer)) {
      set_path(c, create_path(c->x / TILE_WIDTH, c->y / TILE_HEIGHT,
                              c->ai_target.x, c->ai_target.y, c->layer));
   }
                              
   /* If no path exists to the target, make a path to the closest target */
   #warning Incomplete: add closest point to path gen.

   script->ip++;
   return 0;
}

void register_opcodes(void)
{
   register_opcode(opcode_and, CMD_AND);
   register_opcode(opcode_or, CMD_OR);
   register_opcode(opcode_xor, CMD_XOR);
   register_opcode(opcode_not, CMD_NOT);
   register_opcode(opcode_test, CMD_TEST);
   register_opcode(opcode_cmp, CMD_CMP);
   register_opcode(opcode_itof,  CMD_ITOF);
   register_opcode(opcode_ftoi, CMD_FTOI);
   register_opcode(opcode_sin, CMD_SIN);
   register_opcode(opcode_cos, CMD_COS);
   register_opcode(opcode_tan, CMD_TAN);
   register_opcode(opcode_asin, CMD_ASIN);
   register_opcode(opcode_acos, CMD_ACOS);
   register_opcode(opcode_atan, CMD_ATAN);
   register_opcode(opcode_atan2, CMD_ATAN2);
   register_opcode(opcode_exp, CMD_EXP);
   register_opcode(opcode_log, CMD_LOG);
   register_opcode(opcode_add, CMD_ADD);
   register_opcode(opcode_sub, CMD_SUB);
   register_opcode(opcode_mul, CMD_MUL);
   register_opcode(opcode_div, CMD_DIV);
   register_opcode(opcode_inc, CMD_INC);
   register_opcode(opcode_dec, CMD_DEC);
   register_opcode(opcode_neg, CMD_NEG);

   register_opcode(opcode_reset, CMD_RESET);
   register_opcode(opcode_wait1, CMD_WAIT1);
   register_opcode(opcode_wait1, CMD_WAIT2);
   register_opcode(opcode_wait3, CMD_WAIT3);
   register_opcode(opcode_wait1, CMD_WAIT4);
   register_opcode(opcode_showf, CMD_SHOWF);
   register_opcode(opcode_move, CMD_MOVE);
   register_opcode(opcode_chdir, CMD_CHDIR);
   register_opcode(opcode_alert, CMD_ALERT);
   register_opcode(opcode_return, CMD_RETURN);
   register_opcode(opcode_nop, CMD_NOP);
//   register_opcode(opcode_invsbl, CMD_INVSBL);
   register_opcode(opcode_clrzf, CMD_CLRZF);
   register_opcode(opcode_jmp, CMD_JMP);
   register_opcode(opcode_jbe, CMD_JBE);
   register_opcode(opcode_jae, CMD_JAE);
   register_opcode(opcode_jb, CMD_JB);
   register_opcode(opcode_ja, CMD_JA);
   register_opcode(opcode_jne, CMD_JNE);
   register_opcode(opcode_je, CMD_JE);
   register_opcode(opcode_jcf, CMD_JCF);
   register_opcode(opcode_chkd, CMD_CHKD);
   register_opcode(opcode_clrc, CMD_CLRC);
   register_opcode(opcode_setc, CMD_SETC);
   register_opcode(opcode_setd, CMD_SETD);
   register_opcode(opcode_clrd, CMD_CLRD);
   register_opcode(opcode_chkc, CMD_CHKC);
   register_opcode(opcode_spawn, CMD_SPAWN);
   register_opcode(opcode_unload, CMD_UNLOAD);
//   register_opcode(opcode_cmdon, CMD_CMDON);
//   register_opcode(opcode_cmdoff, CMD_CMDOFF);
   register_opcode(opcode_remove, CMD_REMOVE);
   register_opcode(opcode_target, CMD_TARGET);
   register_opcode(opcode_cmdoff, CMD_CHANGE);
   register_opcode(opcode_atomic, CMD_ATOMIC);
   register_opcode(opcode_advcounter, CMD_ADVCOUNTER);
   register_opcode(opcode_chkcounter, CMD_CHKCOUNTER);
   register_opcode(opcode_showcounter, CMD_SHOWCNTER);
   register_opcode(opcode_hidecounter, CMD_HIDECNTER);
   register_opcode(opcode_pushn, CMD_PUSHN);
   register_opcode(opcode_popn, CMD_POPN);
   register_opcode(opcode_store, CMD_STORE);
   register_opcode(opcode_load, CMD_LOAD);
   register_opcode(opcode_chkres, CMD_CHKRES);
   register_opcode(opcode_move_task_target, CMD_MAKEPATH);
   register_opcode(opcode_chk_task_target, CMD_CHKTARGET);
}

void disable_waittimes(void)
{
   register_opcode(opcode_wait_is_nop, CMD_WAIT1);
   register_opcode(opcode_wait_is_nop, CMD_WAIT2);
   register_opcode(opcode_wait_is_nop, CMD_WAIT3);
   register_opcode(opcode_wait_is_nop, CMD_WAIT4);
}
