/**********************************************/
/* Evert Glebbeek 2003, 2005                  */
/* eglebbk@dds.nl                             */
/**********************************************/
#include <allegro.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include "assert.h"
#include "genrand.h"
#include "dialog.h"
#include "map.h"
#include "maptile.h"
#include "gamegfx.h"
#include "global.h"
#include "player.h"
#include "linked.h"
#include "gfx.h"
#include "str.h"
#include "damage.h"
#include "monster.h"
#include "items.h"
#include "particle.h"
#include "bbox.h"

#define MAX_ROOMS    6
#define MAX_PORTALS  128

/* Struct used to store pre-placed items/monsters */
typedef struct INIT_OBJECTS {
   int type;
   int drop;
   unsigned int location;
} INIT_OBJECTS;

/* Doors */
typedef struct PORTAL {
   int x;
   int y;
   int room;

   int target_x;
   int target_y;
   int target_room;
} PORTAL;

typedef struct LEVEL_LAYOUT {
   int num_rooms;
   int room_layout[MAX_ROOMS];

   INIT_OBJECTS init_items[16];
   INIT_OBJECTS init_monster[32];

   int start_room;
   int player_x, player_y;
} LEVEL_LAYOUT;

/* Room layout (section of 1x4 screens) */
typedef struct ROOM {
   unsigned int layout;                   /* Room layout (packed) */
   int tilemap[MAP_WIDTH*MAP_HEIGHT];     /* Tilemap - for speed  */
   LINK *portal;
   LINK *monster;
   LINK *item;
   LINK *projectile;
   LINK *particle;
   LINK *geometry_bbox;
} ROOM;

typedef struct LEVEL {
   ROOM rooms[MAX_ROOMS];
   unsigned int doors[MAX_ROOMS][4];   /* Status of each door connecting rooms */
   int num_rooms;

   int switch_state;
   int start_room;
   int player_x, player_y;
} LEVEL;

static PORTAL *portal_chunk = NULL;
static PORTAL **portal_stack = NULL;
static int portal_stack_counter = 0;

/* Object x and y coordinate locations */
static const int obj_xs[] = {3, 6,  9, 12, 15, 18, 21, /* exit x */ 1, 23};
static const int obj_ys[] = {5, 8, 12, 20, 23, 27, 35, 38, 42, 50, 53, 57};

/* Map room descritions - internal walls                                */
/* Horizontals: 3x3 = 9 bits needed -> make into 3x4 = 16 bits, easier  */
/*  Bit 0 - left, bit 1 - middle, bit 2 - right                         */
/*  Store in lower word                                                 */
/* Verticals:   4x3 = 12 bits needed -> make into 3x4 = 16 bits, easier */
/*  Bit 0 - left, bit 1 - middle, bit 2 - right                         */
/*  Store in upper word                                                 */

static LEVEL level;

static PORTAL *alloc_portal(void)
{
   ASSERT(portal_stack_counter<MAX_PORTALS);
   ASSERT(portal_stack);
   
   /* return the address of the last free item in the list */
   return portal_stack[portal_stack_counter++];
}

static void free_portal(PORTAL *l)
{
   ASSERT(portal_stack_counter);
   ASSERT(portal_stack);
   
   portal_stack[--portal_stack_counter]=l;
}

static int get_portal_count(void)
{
   return portal_stack_counter;
}

void map_init(void)
{
   int c;
   portal_chunk = realloc(portal_chunk, sizeof *portal_chunk * MAX_PORTALS);
   portal_stack = realloc(portal_stack, sizeof *portal_stack * MAX_PORTALS);
   
   portal_stack_counter = 0;
   for (c=0; c<MAX_PORTALS; c++)
      portal_stack[c] = &(portal_chunk[c]);
   level.num_rooms = 0;
}

void unload_map(void)
{
   int c;

   for (c=0; c<level.num_rooms; c++) {
      destroy_list(level.rooms[c].portal, (void *)free_portal);
      destroy_list(level.rooms[c].geometry_bbox, (void *)free_bshape);
      destroy_list(level.rooms[c].particle, (void *)free_particle);
      destroy_list(level.rooms[c].projectile, (void *)free_projectile);
      destroy_list(level.rooms[c].monster, (void *)free_monster);
      destroy_list(level.rooms[c].item, (void *)free_item);
   }
   level.num_rooms = 0;
}

/* Laydown the tilemap for the current level */
static void laydown_tiles(void)
{
   int n;
   int c;
   int x, y;
   int k;
   int num;

   for(n=0; n<level.num_rooms; n++) {
      for(c=0; c<MAP_WIDTH*MAP_HEIGHT; c++) {
         level.rooms[n].tilemap[c] = MAP_FLOOR;
      }

      num = level.rooms[n].layout;

      /* Draw horizontal lines */
      for (c=1; c<4; c++) {
         k = num>>(4*(c-1)) & 0x0F;

         /* The cth horizontal is at y = 15*c+1 */
         if (k&2)
            for(x=5; x<MAP_WIDTH-5; x++) {
               level.rooms[n].tilemap[TILE_INDEX(x,1+15*c-1)] = MAP_WALL;
               level.rooms[n].tilemap[TILE_INDEX(x,1+15*c)] = MAP_WALL;
               level.rooms[n].tilemap[TILE_INDEX(x,1+15*c+1)] = MAP_WALL;
            }

         if (k&1)
            for(x=2; x<8; x++) {
               level.rooms[n].tilemap[TILE_INDEX(x,1+15*c-1)] = MAP_WALL;
               level.rooms[n].tilemap[TILE_INDEX(x,1+15*c)] = MAP_WALL;
               level.rooms[n].tilemap[TILE_INDEX(x,1+15*c+1)] = MAP_WALL;
            }


         if (k&4)
            for(x=2; x<8; x++) {
               level.rooms[n].tilemap[TILE_INDEX(MAP_WIDTH-1-x,1+15*c-1)] = MAP_WALL;
               level.rooms[n].tilemap[TILE_INDEX(MAP_WIDTH-1-x,1+15*c)] = MAP_WALL;
               level.rooms[n].tilemap[TILE_INDEX(MAP_WIDTH-1-x,1+15*c+1)] = MAP_WALL;
            }
      }


      /* Draw vertical lines */
      /* The jth vertical is at x = 6*j, and if it is in the k-th segment, */
      /*  it runs from y = 15*(k-1)+1 to y=15*k+1 */
      for (c = 0; c<4; c++) {
         k = num>>(16+4*c) & 0x0F;
         if (k&2)
            for(y=15*c; y<15*(c+1)+1; y++) {
               level.rooms[n].tilemap[TILE_INDEX(6*2-1,y)] = MAP_WALL;
               level.rooms[n].tilemap[TILE_INDEX(6*2,y)] = MAP_WALL;
               level.rooms[n].tilemap[TILE_INDEX(6*2+1,y)] = MAP_WALL;
            }

         if (k&1)
            for(y=15*c; y<15*(c+1)+1; y++) {
               level.rooms[n].tilemap[TILE_INDEX(6*1-1,y)] = MAP_WALL;
               level.rooms[n].tilemap[TILE_INDEX(6*1,y)] = MAP_WALL;
               level.rooms[n].tilemap[TILE_INDEX(6*1+1,y)] = MAP_WALL;
            }

         if (k&4)
            for(y=15*c; y<15*(c+1)+1; y++) {
               level.rooms[n].tilemap[TILE_INDEX(6*(1+2)-1,y)] = MAP_WALL;
               level.rooms[n].tilemap[TILE_INDEX(6*(1+2),y)] = MAP_WALL;
               level.rooms[n].tilemap[TILE_INDEX(6*(1+2)+1,y)] = MAP_WALL;
            }
      }

      /* Mark map edges */
      for(x=0; x<MAP_WIDTH; x++) {
         level.rooms[n].tilemap[TILE_INDEX(x,1)] = MAP_WALL;
         level.rooms[n].tilemap[TILE_INDEX(x,MAP_HEIGHT-2)] = MAP_WALL;
      }
      for(y=0; y<MAP_HEIGHT; y++) {
         level.rooms[n].tilemap[TILE_INDEX(1,y)] = MAP_WALL;
         level.rooms[n].tilemap[TILE_INDEX(MAP_WIDTH-2,y)] = MAP_WALL;
      }
      for(x=0; x<MAP_WIDTH; x++) {
         level.rooms[n].tilemap[TILE_INDEX(x,0)] = MAP_CEILING;
         level.rooms[n].tilemap[TILE_INDEX(x,MAP_HEIGHT-1)] = MAP_CEILING;
      }
      for(y=0; y<MAP_HEIGHT; y++) {
         level.rooms[n].tilemap[TILE_INDEX(0,y)] = MAP_CEILING;
         level.rooms[n].tilemap[TILE_INDEX(MAP_WIDTH-1,y)] = MAP_CEILING;
      }

      /* Place doors left and right */
      for (c=0; c<4; c++) {
         level.rooms[n].tilemap[TILE_INDEX(1,8+c*15)] =
            level.doors[n][c]!=MAP_DOOR_NONE?MAP_WALL+MAP_DOOR:MAP_WALL;
         level.rooms[n].tilemap[TILE_INDEX(MAP_WIDTH-2,8+c*15)] =
            level.doors[(n+1)%level.num_rooms][c]!=MAP_DOOR_NONE?MAP_WALL+MAP_DOOR:MAP_WALL;
      }
      //level.rooms[n].tilemap[TILE_INDEX(1,8+1*15)] = level.doors[n][0]!=MAP_DOOR_NONE?MAP_WALL+MAP_DOOR:MAP_WALL;
      //level.rooms[n].tilemap[TILE_INDEX(1,8+2*15)] = level.doors[n][0]!=MAP_DOOR_NONE?MAP_WALL+MAP_DOOR:MAP_WALL;
      //level.rooms[n].tilemap[TILE_INDEX(1,8+3*15)] = level.doors[n][0]!=MAP_DOOR_NONE?MAP_WALL+MAP_DOOR:MAP_WALL;

      /* Place doors right */
      //level.rooms[n].tilemap[TILE_INDEX(MAP_WIDTH-2,8+0*15)] = MAP_WALL+MAP_DOOR;
      //level.rooms[n].tilemap[TILE_INDEX(MAP_WIDTH-2,8+1*15)] = MAP_WALL+MAP_DOOR;
      //level.rooms[n].tilemap[TILE_INDEX(MAP_WIDTH-2,8+2*15)] = MAP_WALL+MAP_DOOR;
      //level.rooms[n].tilemap[TILE_INDEX(MAP_WIDTH-2,8+3*15)] = MAP_WALL+MAP_DOOR;

      /* Scan for embedded walls */
      for(y=1; y<MAP_HEIGHT-1; y++) {
         for(x=1; x<MAP_WIDTH-1; x++) {
            c = 0;
            if (level.rooms[n].tilemap[TILE_INDEX(x-1,y-1)]&(MAP_WALL|MAP_CEILING))
               c|=1;
            if (level.rooms[n].tilemap[TILE_INDEX(x+1,y-1)]&(MAP_WALL|MAP_CEILING))
               c|=2;
            if (level.rooms[n].tilemap[TILE_INDEX(x+1,y+1)]&(MAP_WALL|MAP_CEILING))
               c|=4;
            if (level.rooms[n].tilemap[TILE_INDEX(x-1,y+1)]&(MAP_WALL|MAP_CEILING))
               c|=8;
            if (c==15) {
               level.rooms[n].tilemap[TILE_INDEX(x,y)]=MAP_CEILING;
            }
         }
      }
   }
}

/* Place portals (doors) in the current level */
static void place_portals(void)
{
   int n;
   
//&(level.doors[n][0]), &(level.doors[n][1]),
//&(level.doors[n][2]), &(level.doors[n][3]));

}

static ITEM *_add_init_item(int room, int x, int y, int type, int child)
{
   ITEM *item;
   
   item = alloc_item();
   item->x = x*TILE_WIDTH;
   item->y = y*TILE_HEIGHT;
   item->type = type;
   item->child = NULL;
   item->anim = 0;
   item->lifetime = 0;
   
   if (child != ITEM_NONE) {
      item->child = alloc_item();
      item->child->x = x*TILE_WIDTH;
      item->child->y = y*TILE_HEIGHT;
      item->child->type = child;
      item->child->child = NULL;
      item->child->anim = 0;
      item->child->lifetime = 0;
   }
   set_item_list(room, add_to_list(item, get_item_list(room)));
   
   return item;
}

static void _add_init_monster(int room, int x, int y, int type, int item)
{
   MONSTER *monster=alloc_monster();
   monster->x = x*TILE_WIDTH;
   monster->y = y*TILE_HEIGHT;
   monster->type = type;
   monster->move_delay = 0;
   monster->health = get_monster_hp(monster->type);
   monster->def = get_monster_defence(type);
   monster->atk = get_monster_attack(type);
   monster->dropped_item = item;
   monster->angle = -M_PI/2;
   monster->ai_timer = 1;
   monster->velocity = 0.1;
   monster->hitc = 0;
   monster->ability = get_monster_ability(type);
   monster->prev = NULL;
   monster->next = NULL;
   set_monster_list(room, add_to_list(monster, get_monster_list(room)));
}

static void place_boss(int num)
{
   MONSTER *monster;
   LINK *l;

   if (num == -1) {                 /* Serpentile */
      int c;
      int n = 128;
      
      for (c=0; c<n; c++) {
         monster=alloc_monster();

         monster->x = obj_xs[3]*TILE_WIDTH + (c-n/2)*2;
         monster->y = obj_ys[0]*TILE_WIDTH;

         monster->type = MONSTER_BOSS1;

         monster->move_delay = 0;
         monster->health = 10;
         monster->def = 0xFF602020;
         monster->atk = 0x00240024;
         monster->dropped_item = ITEM_NONE;
         monster->angle = -M_PI/2;
         monster->ai_timer = 0;
         monster->velocity = 1;
         monster->hitc = 0;
         monster->angle = 0;
         monster->order = c/4;
         set_monster_list(0, add_to_list(monster, get_monster_list(0)));
      }

      for (l = get_monster_list(0); l; l=l->next) {
         monster = l->data;
         if (l->prev)
            monster->prev = l->prev->data;
         else
            monster->prev = NULL;
         if (l->next)
            monster->next = l->next->data;
         else
            monster->next = NULL;
      }
   }

   if (num == -2) {
      monster=alloc_monster();

      monster->x = obj_xs[3]*TILE_WIDTH;
      monster->y = obj_ys[0]*TILE_WIDTH;
      monster->dy = 0;
      monster->dx = 0;

      monster->type = MONSTER_BOSS2;

      monster->move_delay = 0;
      monster->health = 350;
      monster->def = 0xC0FFD090;
      monster->atk = 0x00414261;
      monster->ability = MONSTER_ABILITY_FLY|MONSTER_ABILITY_VAMPIRE;
      monster->dropped_item = ITEM_NONE;
      monster->angle = -M_PI/2;
      monster->ai_timer = 40;
      monster->velocity = 0;
      monster->hitc = 0;
      monster->angle = 0;
      monster->order = 0;
      monster->prev = NULL;
      monster->next = NULL;
      set_monster_list(0, add_to_list(monster, get_monster_list(0)));
   }
}

/* Load level data from the file levels.gme */
static void load_initlevel(int num)
{
   char *filename;
   char start_token[128];
   char end_token[128];
   char s[256];
   char *eol;
   char *name;
   char *name_end;
   int section = -1;
   FILE *f;
   int n;
   int c;
   int x, y, room;
   int ctype = -1, itype = -1;


   /* First clear all data */
   for(n=0; n<MAX_ROOMS; n++) {
      level.rooms[n].portal = NULL;
      level.rooms[n].monster = NULL;
      level.rooms[n].item = NULL;
      level.rooms[n].projectile = NULL;
      level.rooms[n].particle = NULL;
      level.rooms[n].geometry_bbox = NULL;

      for (c=0; c<4; c++) {
         level.doors[n][c] = MAP_DOOR_OPEN;
      }
   }
   level.num_rooms = 0;

   sprintf(start_token, "[Begin level %c]", 'A'+num);
   sprintf(end_token, "[End level %c]", 'A'+num);
   
   filename = get_game_path("levels.gme");
   f = fopen(filename, "r");
   if (!f) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Cannot load %s!\n", filename);
      exit(EXIT_FAILURE);
   }

   while(fgets(s, 256, f)) {
      /* Skip comments */
      eol = strstr(s, ";");
      if (eol)
         eol[0] = '\0';

      /* Clear end-of-line marker if it is present */
      eol = strstr(s, "\n");
      if (eol)
         eol[0] = '\0';

      /* Replace tab characters by spaces */
      for (eol = strstr(s, "\t"); eol; eol = strstr(eol, "\t"))
         eol[0]=' ';

      /* clear spaces from the beginning and end of the string */
      strflushl(s);
      strflushr(s);

      /* Now see if we have anything left to parse */
      if (s[0]) {
         if (s[0] == '[') {  /* new section */
            if (!strcmp(s, start_token)) {
               section = 0;
            } else if (!strcmp(s, end_token)) {
               section = -1;
               break;
            } else if (!strcmp(s, "[Layout]") && section>=0) {
               section = 1;
            } else if (!strcmp(s, "[Doors]") && section>=0) {
               section = 2;
               n = 0;
            } else if (!strcmp(s, "[Items]") && section>=0) {
               section = 3;
            } else if (!strcmp(s, "[Creatures]") && section>=0) {
               section = 4;
            } else if (!strcmp(s, "[Start location]") && section>=0) {
               section = 5;
            }
         } else {
            switch (section) {
               case 1:     /* Layout */
                  sscanf(s, "%x", &(level.rooms[level.num_rooms].layout));
                  level.num_rooms++;
                  break;
               case 2:     /* Doors */
                  sscanf(s, "%x\t%x\t%x\t%x\t",
                         &(level.doors[n][0]), &(level.doors[n][1]),
                         &(level.doors[n][2]), &(level.doors[n][3]));
                  n++;
                  break;
               case 3:     /* Items */
                  sscanf(s, "Room %d (%d,%d):\n", &room, &x, &y);
                  name = strstr(s, ":");
                  if (name) {
                     while(name[0] == ':' || name[0] == ' ') name++;
                     
                     /* Child items */
                     name_end = strstr(name, ",");
                     if (name_end) {
                        name_end[0] = '\0';
                        name_end++;
                        while(name_end[0] == ' ') name_end++;
                     } else {
                        name_end = "None";
                     }
                     itype = get_item_id(name);
                     ctype = get_item_id(name_end);
                  }
                  _add_init_item(room, x, y, itype, ctype);
                  break;
               case 4:     /* Creatures */
                  sscanf(s, "Room %d (%d,%d):\n", &room, &x, &y);
                  name = strstr(s, ":");
                  if (name) {
                     name += 2;
                     name_end = strstr(name, ",");
                     name_end[0] = '\0';
                     ctype = get_monster_id(name);
                     name = name_end+2;
                     itype = get_item_id(name);
                  }
                  
                  _add_init_monster(room, x, y, ctype, itype);
                  break;
               case 5:     /* Start location */
                  sscanf(s, "Room %d (%d, %d)\n", &level.start_room, &level.player_x, &level.player_y);
                  break;
               default:                  
                  break;
            }  /* end switch */
         }
      } /* end if*/
   }
   fclose(f);
   
   /* Place teleporters in doorways */
   for(n=0; n<level.num_rooms; n++) {
      for(c=0; c<4; c++) {
         if (level.doors[n][c] != MAP_DOOR_NONE) {
            ITEM *item;
            
            /* Move left through a door */
            if (level.doors[n][c] == MAP_DOOR_LOCKED)
               itype = ITEM_LOCKEDTELEP;
            else
               itype = ITEM_TELEPORTER;

            item = _add_init_item(n, 1, c*15+8, itype, ITEM_NONE);
            if (itype == ITEM_TELEPORTER)
               item->x -= 16;
            item->tx = (MAP_WIDTH-3) * TILE_SIZE + 8;
            item->ty = (c*15+8) * TILE_SIZE;
            item->troom = (n+level.num_rooms-1)%level.num_rooms;
            
            /* Move right though the same door from the previous room */
            item = _add_init_item((n+level.num_rooms-1)%level.num_rooms, MAP_WIDTH-2, c*15+8, itype, ITEM_NONE);
            if (itype == ITEM_TELEPORTER)
               item->x += 16;
            item->tx = 2*TILE_SIZE - 8;
            item->ty = (c*15+8) * TILE_SIZE;
            item->troom = n;
         }
      }
   }

   /*
   fprintf(f, "[Layout]\n");
   for(n=0; n<gamelevels[num].num_rooms; n++) {
      fprintf(f, "0x%08x\n", gamelevels[num].room_layout[n]);
   }
   fprintf(f, "[Doors]\n");
   for(n=0; n<gamelevels[num].num_rooms; n++) {
      for(c=0; c<4; c++)
         fprintf(f, "0x%02x\t", gamelevels[num].door_states[n][c]);
      fprintf(f, "\n");
   }
   fprintf(f, "[Items]\n");
   for(n=0; n<16; n++)
      if (gamelevels[num].init_items[n].type!=ITEM_NONE) {
         int x = obj_xs[gamelevels[num].init_items[n].location&0x0F];
         int y = obj_ys[(gamelevels[num].init_items[n].location>>4)&0x0F];
         int room = (gamelevels[num].init_items[n].location>>8)&0x0F;

         fprintf(f, "Room %d (%2d,%2d): 0x%08x\n", room, x, y, gamelevels[num].init_items[n].type);
      }
   fprintf(f, "[Creatures]\n");
   for(n=0; n<32; n++)
      if (gamelevels[num].init_monster[n].type!=MONSTER_NONE){
         int x = obj_xs[gamelevels[num].init_monster[n].location&0x0F];
         int y = obj_ys[(gamelevels[num].init_monster[n].location>>4)&0x0F];
         int room = (gamelevels[num].init_monster[n].location>>8)&0x0F;

         fprintf(f, "Room %d (%2d,%2d): 0x%08x, 0x%08x\n", room, x, y, gamelevels[num].init_monster[n].type, gamelevels[num].init_monster[n].drop);
      }
   fprintf(f, "[Start location]\n");
   fprintf(f, "Room %d (%2d, %2d)\n", gamelevels[num].start_room, gamelevels[num].player_x, gamelevels[num].player_y);
   fprintf(f, "[End level %c]\n\n", 'A'+num);
   */
}

#ifdef LEV_TO_GIF
#include "sgif.h"
static void draw_item(BITMAP *bmp, ITEM *item, int room)
{
   int x, y, col;
   
   x = room*MAP_WIDTH + item->x/TILE_SIZE;
   y = item->y/TILE_SIZE;
   
   col = 0;
   if (item->type == ITEM_TELEPORTER || item->type == ITEM_LOCKEDTELEP) {
      col = 0;
   } else {
      if (!is_power_of_two(item->type) && get_item_name(item->type)) {
         col = 7+item->type*4;
      } else if (get_item_name(item->type)) {
         col = 4+(log(item->type)/log(2)+1)*8;
      }
   }
   
   putpixel(bmp, x, y, col);
}

static void draw_monster(BITMAP *bmp, MONSTER *m, int room)
{
   int x, y, col;
   
   x = room*MAP_WIDTH + m->x/TILE_SIZE;
   y = m->y/TILE_SIZE;
   
   //col = 128+m->type;
   col = 7+m->type*8;
   
   putpixel(bmp, x, y, col);
}

void level_to_gif(int num)
{
   char filename[256];
   char *doorname;
   LINK *l;
   MONSTER *m;
   ITEM *item;
   PALETTE pal;
   BITMAP *legend;
   BITMAP *layout;
   BITMAP *blocks;
   BITMAP *items;
   BITMAP *drops;
   BITMAP *monsters;
   BITMAP *doors;
   GIF_ANIM gif;
   GIF_FRAME frame[6];
   int c, col, n;
   int x, y, y2, y3;
   
   sprintf(filename, "gif/%c.gif", 'a'+num);

   load_initlevel(num);
   laydown_tiles();
   place_portals();
   
   /* Palette colours */
   for (c=0; c<256; c++) {
      pal[c].r = pal[c].g = pal[c].b = 0;
   }
   
   for (c=0; c<16; c++) {
      pal[c].r = pal[c].g = pal[c].b = MIN(c*4, 63);
      
      pal[16+c].r = 15+c*3;
      pal[16+c].g = pal[16+c].b = 0;

      pal[32+c].g = 15+c*3;
      pal[32+c].r = pal[32+c].b = 0;

      pal[48+c].b = 15+c*3;
      pal[48+c].r = pal[48+c].g = 0;

      pal[64+c].b = 0;
      pal[64+c].r = pal[64+c].g = 15+c*3;

      pal[80+c].r = 0;
      pal[80+c].g = pal[80+c].b = 15+c*3;

      pal[96+c].g = 0;
      pal[96+c].r = pal[96+c].b = 15+c*3;

      pal[112+c].r = MIN((16+c*4), 63);
      pal[112+c].g = MIN((8+c*4), 63);
      pal[112+c].b = MIN(c*4, 63);

      pal[128+c].r = MIN((8+c*4), 63);
      pal[128+c].g = MIN(c*4, 63);
      pal[128+c].b = MIN((16+c*4), 63);

      pal[144+c].r = MIN(c*4, 63);
      pal[144+c].g = MIN((8+c*4), 63);
      pal[144+c].b = MIN((16+c*4), 63);

      pal[160+c].r = MIN(48-c*2, 63);
      pal[160+c].g = MIN((8+c*4), 63);
      pal[160+c].b = MIN((16+c*4), 63);

      pal[176+c].r = MIN((16+c*4), 63);
      pal[176+c].g = MIN(48-c*2, 63);
      pal[176+c].b = MIN((8+c*4), 63);

      pal[192+c].r = MIN((8+c*4), 63);
      pal[192+c].g = MIN((16+c*4), 63);
      pal[192+c].b = MIN(48-c*2, 63);
   }
   pal[255].r = pal[255].g = pal[255].b = 0;

   /* Animated gif settings */
   gif_a_settings = 0;
   GIF_SETBACKGROUND(0);

   legend = create_bitmap_ex(8, 480, 320);
   layout = create_bitmap_ex(8, level.num_rooms*MAP_WIDTH, MAP_HEIGHT);
   doors = create_bitmap_ex(8, level.num_rooms*MAP_WIDTH, MAP_HEIGHT);
   items = create_bitmap_ex(8, level.num_rooms*MAP_WIDTH, MAP_HEIGHT);
   blocks = create_bitmap_ex(8, level.num_rooms*MAP_WIDTH, MAP_HEIGHT);
   monsters = create_bitmap_ex(8, level.num_rooms*MAP_WIDTH, MAP_HEIGHT);
   drops = create_bitmap_ex(8, level.num_rooms*MAP_WIDTH, MAP_HEIGHT);

   clear_bitmap(layout);
   clear_bitmap(blocks);
   clear_bitmap(items);
   clear_bitmap(drops);
   clear_bitmap(monsters);
   clear_bitmap(doors);
   
   for(y=0; y<MAP_HEIGHT; y++) {
      for(c=0; c<level.num_rooms; c++) {
         for(x=0; x<MAP_WIDTH; x++) {
            switch(level.rooms[c].tilemap[TILE_INDEX(x,y)]) {
               case MAP_FLOOR:
                  putpixel(layout, x+c*MAP_WIDTH, y, 7);
                  break;
               case MAP_WALL:
                  putpixel(layout, x+c*MAP_WIDTH, y, 15);
                  break;
               case MAP_CEILING:
                  putpixel(layout, x+c*MAP_WIDTH, y, 255);
                  break;
               case MAP_DOOR:
                  break;
            }
         }
      }
   }
   
   for(c=0; c<level.num_rooms; c++) {
      for(y=0; y<4; y++) {
         switch (level.doors[c][y]) {
            case MAP_DOOR_OPEN:
               col = 7;
               break;
            case MAP_DOOR_LOCKED:
               col = 79;
               break;
            case MAP_DOOR_BLOCKED:
               col = 47;
               break;
            case MAP_DOOR_SWITCH:
               col = 31;
               break;
            case MAP_DOOR_NONE:
               col = 15;
               break;
            case MAP_DOOR_SWITCH2:
               col = 23;
               break;
            default:
               col = 64;
               if (level.doors[c][y] & MAP_DOOR_CREATUREL)
                  col |= 8;
               if (level.doors[c][y] & MAP_DOOR_CREATURER)
                  col |= 4;
               if ((level.doors[c][y] & MAP_DOOR_CREATURE) == MAP_DOOR_CREATURE)
                  col |= 14;
               break;
         }
         putpixel(doors, 1+c*MAP_WIDTH, 8+y*15, col);
         putpixel(doors, (1+c*MAP_WIDTH+level.num_rooms*MAP_WIDTH-3)%(level.num_rooms*MAP_WIDTH), 8+y*15, col);
      }
   }
   
   /* Starting location */
   putpixel(doors, level.player_x+level.start_room*MAP_WIDTH, level.player_y, 24);
   
   /* Items & monsters */
   for (c=0; c<level.num_rooms; c++) {
      l = get_monster_list(c);
      while(l) {
         m = l->data;

         /* Move the monster around if there is already another monster at the */
         /* same location */
         while (getpixel(monsters, c*MAP_WIDTH+m->x/TILE_SIZE, m->y/TILE_SIZE)!=0
                ||  
                getpixel(blocks, c*MAP_WIDTH+m->x/TILE_SIZE, m->y/TILE_SIZE)) {
            n = genrandui();
            if ((n&3) == 1)
               m->x += TILE_SIZE;
            else if ((n&3) == 2)
               m->x -= TILE_SIZE;
            
            n = genrandui();
            if ((n&3) == 1)
               m->y += TILE_SIZE;
            else if ((n&3) == 2)
               m->y -= TILE_SIZE;
            
            if (m->x < 3*TILE_SIZE) m->x = 3*TILE_SIZE;
            if (m->y < 3*TILE_SIZE) m->y = 3*TILE_SIZE;
            if (m->x > (MAP_WIDTH-3)*TILE_SIZE) m->x = (MAP_WIDTH-3)*TILE_SIZE;
            if (m->y > (MAP_HEIGHT-3)*TILE_SIZE) m->y = (MAP_HEIGHT-3)*TILE_SIZE;
         }
         draw_monster(monsters, m, c);
         if (m->dropped_item != ITEM_NONE) {
            ITEM itm;
            itm.type = m->dropped_item;
            itm.x = m->x;
            itm.y = m->y;
            draw_item(drops, &itm, c);
         }
                  
         l = l->next;
      }
      
      l = get_item_list(c);
      while(l) {
         item = l->data;
         if (item->child) {
            draw_item(items, item->child, c);
            draw_item(blocks, item, c);
         } else {
            if ((item->type == ITEM_CHESTOPEN) ||
                (item->type == ITEM_CHESTCLOSED) ||
                (item->type == ITEM_BLOCK) ||
                (item->type == ITEM_BLOCKCRACK) ||
                (item->type == ITEM_BLOCKLOCKED)) {
               draw_item(blocks, item, c);
            } else {         
               draw_item(items, item, c);
            }
         }
         l = l->next;
      }
   }
   
   /* Legend */
   clear_to_color(legend, 10);

   y2 = y3 = y = 0;
   for (c=0; c<38; c++) {
      if (!is_power_of_two(c) && get_item_name(c)) {
         //col = 112+c;
         col = 7+c*4;
         textprintf_ex(legend, get_small_font(), 2, y, col, -1, "%s %d", get_item_name(c), col);
         y+=text_height(get_small_font());
      }

      if (get_item_name(1<<c)) {
         col = 4+(c+1)*8;
         //col = 80+c;
         textprintf_ex(legend, get_small_font(), 120, y2, col, -1, "%s %d", get_item_name(1<<c), col);
         y2+=text_height(get_small_font());
      }

      if (get_monster_name(c)) {
         col = 7+c*8;
         textprintf_ex(legend, get_small_font(), 238, y3, col, -1, "%s %d", get_monster_name(c), col);
         y3+=text_height(get_small_font());
      }
   }
   
   /* Doors */
   y = 0;
   for(c=0; c<32; c++) {
      col = 0;
      doorname = NULL;
      switch (c) {
         case MAP_DOOR_OPEN:
            col = 7;
            doorname = "open door";
            break;
         case MAP_DOOR_LOCKED:
            col = 79;
            doorname = "locked door";
            break;
         case MAP_DOOR_BLOCKED:
            col = 47;
            doorname = "blocked door";
            break;
         case MAP_DOOR_SWITCH:
            col = 31;
            doorname = "switch(dn) door";
            break;
         case MAP_DOOR_NONE:
            col = 15;
            doorname = "no door";
            break;
         case MAP_DOOR_SWITCH2:
            col = 23;
            doorname = "switch(up) door";
            break;
         default:
            col = 64;
            if (c == MAP_DOOR_CREATUREL) {
               col |= 8;
               doorname = "creature >L < door";
            }
            if (c == MAP_DOOR_CREATURER) {
               col |= 4;
               doorname = "creature > R< door";
            }
            if (c == MAP_DOOR_CREATURE) {
               col |= 14;
               doorname = "creature >LR< door";
            }
            break;
      }
      
      if (col && doorname) {
         textprintf_ex(legend, get_small_font(), 356, y, col, -1, "%s %d", doorname, col);
         y+=text_height(get_small_font());
      }
   }
   
   /* Initialise animated gif structure */
   gif.frame_color = 0;
   gif.frame_width = layout->w;
   gif.frame_height = layout->h;
   gif.frames = frame;
   
   for(c=1; c<6; c++) {
      frame[c-1].next = &(frame[c]);
      frame[c].next = NULL;
   }

   for(c=0; c<6; c++) {
      memcpy(frame[c].pal, pal, sizeof(PALETTE));
   }
   
   frame[0].image = layout;
   frame[1].image = doors;
   frame[2].image = items;
   frame[3].image = blocks;
   frame[4].image = drops;
   frame[5].image = monsters;
   
   //save_gif(filename, doors, pal);
   save_gif_anim(filename, &gif);
   save_bitmap("gif/legend.pcx", legend, pal);

   set_palette(pal);   
   for(c=0; c<6; c++) {
      blit(frame[c].image, screen, 0,0, c*frame[c].image->w,c*frame[c].image->h, frame[c].image->w,frame[c].image->h);
   }
   
   destroy_bitmap(layout);
   destroy_bitmap(blocks);
   destroy_bitmap(items);
   destroy_bitmap(drops);
   destroy_bitmap(monsters);
   destroy_bitmap(doors);
}

void gif_to_level(int num)
{
   char filename[256];
   char *doorname;
   LINK *l;
   FILE *f;
   MONSTER *m;
   ITEM *item;
   PALETTE pal;
   BITMAP *legend;
   BITMAP *layout;
   BITMAP *blocks;
   BITMAP *items;
   BITMAP *drops;
   BITMAP *monsters;
   BITMAP *doors;
   GIF_ANIM *gif;
   int c, col, n;
   int x, y, y2, y3;
   int num_rooms;
   int coded_layout;
   
   sprintf(filename, "gif/%c.gif", 'a'+num);
   
   //gif = load_gif_animated(filename);

   set_color_conversion(COLORCONV_NONE);
   layout = load_gif_animated_frame(filename, pal, 1);
   doors = load_gif_animated_frame(filename, pal, 2);
   items = load_gif_animated_frame(filename, pal, 3);
   blocks = load_gif_animated_frame(filename, pal, 4);
   drops = load_gif_animated_frame(filename, pal, 5);
   monsters = load_gif_animated_frame(filename, pal, 6);

   sprintf(filename, "gif/%c.txt", 'a'+num);
   f = stdout;
   f = fopen(filename, "w");
   
   fprintf(f, "[Begin level %c]\n", 'A' + num);

   /* Map layout */
   fprintf(f, "[Layout]\n");
   num_rooms = layout->w / MAP_WIDTH;
   for(n=0; n<num_rooms; n++) {
      coded_layout = 0;
      for (c=0; c<4; c++) {
         if (getpixel(layout, n * MAP_WIDTH + 6, 8+c*15) != 7)
            coded_layout |= 1<<(16 + 4*c + 0);
         if (getpixel(layout, n * MAP_WIDTH + 12, 8+c*15) != 7)
            coded_layout |= 1<<(16 + 4*c + 1);
         if (getpixel(layout, n * MAP_WIDTH + 18, 8+c*15) != 7)
            coded_layout |= 1<<(16 + 4*c + 2);
      }

      for (c=0; c<3; c++) {
         if (getpixel(layout, n * MAP_WIDTH + 3, 16+c*15) != 7)
            coded_layout |= 1<<(4*c + 0);
         if (getpixel(layout, n * MAP_WIDTH + 9, 16+c*15) != 7)
            coded_layout |= 1<<(4*c + 1);
         if (getpixel(layout, n * MAP_WIDTH + 21, 16+c*15) != 7)
            coded_layout |= 1<<(4*c + 2);
      }
      fprintf(f, "0x%08x\n", coded_layout);
   }
   /*
   for(n=0; n<gamelevels[num].num_rooms; n++) {
      fprintf(f, "0x%08x\n", gamelevels[num].room_layout[n]);
   }
   */
   fprintf(f, "[Doors]\n");
   for(n=0; n<num_rooms; n++) {
      for (y=0; y<4; y++) {
         coded_layout = 0;
         c = getpixel(doors, n * MAP_WIDTH + 1, 8+y*15);
         if (c==7)
            coded_layout = MAP_DOOR_OPEN;
         if (c==79)
            coded_layout = MAP_DOOR_LOCKED;
         if (c==47)
            coded_layout = MAP_DOOR_BLOCKED;
         if (c==31)
            coded_layout = MAP_DOOR_SWITCH;
         if (c==23)
            coded_layout = MAP_DOOR_SWITCH2;
         if (c==15)
            coded_layout = MAP_DOOR_NONE;

         if (c==72)
            coded_layout |= MAP_DOOR_CREATUREL;
         if (c==68)
            coded_layout |= MAP_DOOR_CREATURER;
         if (c==78)
            coded_layout |= MAP_DOOR_CREATURE;
         fprintf(f, "0x%02x\t", coded_layout);
      }
      fprintf(f, "\n");
   }
   
   fprintf(f, "[Items]\n");
   for(n=0; n<num_rooms; n++) {
      for (y=0; y<MAP_HEIGHT; y++) {
         for (x=0; x<MAP_WIDTH; x++) {
            col = getpixel(blocks, n * MAP_WIDTH + x, y);
            if (col) {
               if ( (col-7) % 4 == 0) {
                  c = (col - 7)/4;
                  fprintf(f, "Room %d (%2d,%2d): %s", n, x, y, get_item_name(c));
               }
               col = getpixel(items, n * MAP_WIDTH + x, y);
               if (col) {
                  char *s = ", (null)";
                  if ( (col-4) % 8 == 0) {
                     c = (col - 4)/8 - 1;
                     s = get_item_name(1<<c);
                  }
                  if ( (col-7) % 4 == 0) {
                     c = (col - 7)/4;
                     s = get_item_name(c);
                  }
                  fprintf(f, ", %s", s);
               }
               fprintf(f, "\n");
               putpixel(items, n * MAP_WIDTH + x, y, 0);
            }

            col = getpixel(items, n * MAP_WIDTH + x, y);
            if (col) {
               if ( (col-4) % 8 == 0) {
                  c = (col - 4)/8 - 1;
                  fprintf(f, "Room %d (%2d,%2d): %s\n", n, x, y, get_item_name(1<<c));
               }
               if ( (col-7) % 4 == 0) {
                  c = (col - 7)/4;
                  fprintf(f, "Room %d (%2d,%2d): %s\n", n, x, y, get_item_name(c));
               }
            }
         }
      }
   }
   fprintf(f, "[Creatures]\n");
   for(n=0; n<num_rooms; n++) {
      for (y=0; y<MAP_HEIGHT; y++) {
         for (x=0; x<MAP_WIDTH; x++) {
            col = getpixel(monsters, n * MAP_WIDTH + x, y);
            if (col) {
               char *s = "(null)";
               c = (col - 7)/8;
               fprintf(f, "Room %d (%2d,%2d): %s", n, x, y, get_monster_name(c));

               col = getpixel(drops, n * MAP_WIDTH + x, y);
               if (col) {
                  if ( (col-4) % 8 == 0) {
                     c = (col - 4)/8 - 1;
                     s = get_item_name(1<<c);
                  }
                  if ( (col-7) % 4 == 0) {
                     c = (col - 7)/4;
                     s = get_item_name(c);
                  }
               }
               fprintf(f, ", %s\n", s);
               putpixel(drops, n * MAP_WIDTH + x, y, 0);
            }
         }
      }
   }
   
   fprintf(f, "[Start location]\n");
   for(n=0; n<num_rooms; n++) {
      for (y=0; y<MAP_HEIGHT; y++) {
         for (x=0; x<MAP_WIDTH; x++) {
            col = getpixel(doors, n * MAP_WIDTH + x, y);
            if (col == 24)
               fprintf(f, "Room %d (%2d, %2d)\n", n, x, y);
         }
      }
   }
   fprintf(f, "[End level %c]\n", 'A' + num);
   
   if (f!=stdout) fclose(f);

   /* Cleanup */   
   destroy_bitmap(layout);
   destroy_bitmap(blocks);
   destroy_bitmap(items);
   destroy_bitmap(drops);
   destroy_bitmap(monsters);
   destroy_bitmap(doors);
   //free_gifa(gif);
}

#endif

void load_level(int num)
{
   int n;
   int c;

   /* First clear all data */
   for(n=0; n<6; n++) {
      level.rooms[n].portal = NULL;
      level.rooms[n].monster = NULL;
      level.rooms[n].item = NULL;
      level.rooms[n].projectile = NULL;
      level.rooms[n].particle = NULL;
      level.rooms[n].geometry_bbox = NULL;

      for (c=0; c<4; c++) {
         level.doors[n][c] = MAP_DOOR_OPEN;
      }
   }

   /* Number larger than or equal to 0 - load normal level */
   if (num>=0) {
      /* Load level-specific data */
      load_initlevel(num);
      laydown_tiles();
      place_portals();
   } else {
      /* Negative number - load special level (ie, boss level) */

      level.num_rooms = 1;
      for(n=0; n<level.num_rooms; n++) {
         level.rooms[n].layout = 0x00000007;
         for (c=0; c<4; c++)
            level.doors[n][c] = MAP_DOOR_NONE;

         level.start_room = 0;
         level.player_x = obj_xs[3];
         level.player_y = obj_ys[2];
      }

      place_boss(num);

      laydown_tiles();
      place_portals();
      //level.rooms[0].tilemap[TILE_INDEX(obj_xs[3], obj_ys[6]-3)] = MAP_WALL|MAP_DOOR;
   }

   /* Calculate the level collision boundingbox */
   for(n=0; n<level.num_rooms; n++) {
      calculate_geometry_bbox(n);
   }
   level.switch_state = 0;
}

static void flood_fill_connect(BITMAP *bmp, const int room, const int x, const int y, const int colour)
{
   int mask;

   if (getpixel(bmp, x, y)!=0)
      return;
   putpixel(bmp, x, y, colour);
   /* Check left */
   if (x>1) {
      mask = ((1<<(x-2))<<(4*y-4))<<16;
      if (!(level.rooms[room].layout & mask))
         flood_fill_connect(bmp, room, x-1, y, colour);
   }

   /* Check right */
   if (x<4) {
      mask = ((1<<(x-1))<<(4*y-4))<<16;
      if (!(level.rooms[room].layout & mask))
         flood_fill_connect(bmp, room, x+1, y, colour);
   }

   /* Check up */
   if (y>1) {
      /* Round x to the next lower power of two if it is not a power of two */
      /* That is to say, round x-1 to the next higher power of two if it is */
      /* not a power of two... */
      if (x==2 || x==3)
         mask = ((2)<<(4*y-8));
      else
         mask = ((x)<<(4*y-8));
      if (!(level.rooms[room].layout & mask))
         flood_fill_connect(bmp, room, x, y-1, colour);
   }

   /* Check down */
   if (y<4) {
      if (x==2 || x==3) {
         mask = (2<<(4*y-4));
      } else {
         mask = (x)<<(4*y-4);
      }
      if (!(level.rooms[room].layout & mask))
         flood_fill_connect(bmp, room, x, y+1, colour);
   }
}

void get_connected_mask(BITMAP *bmp, const int room)
{
   int c;

   ASSERT(bmp->w>=6);
   ASSERT(bmp->h>=6);

   clear_bitmap(bmp);
   rect(bmp, 0,0,5,5,white);

   for (c=0; c<16; c++) {
      flood_fill_connect(bmp, room, c%4+1,c/4+1, makecol(16*c, 255-(32*c)%256, 255-16*c));
   }
}

void get_connected_map(BITMAP *bmp, const int room, int x, int y)
{
   ASSERT(bmp->w>=6);
   ASSERT(bmp->h>=6);
   clear_bitmap(bmp);
   rect(bmp, 0,0,5,5,white);

   flood_fill_connect(bmp, room, x+1, y+1, white);
   rect(bmp, 0,0,5,5,blue);
}

int get_start_x(void)
{
   return level.player_x;
}

int get_start_y(void)
{
   return level.player_y;
}

int get_start_room(void)
{
   return level.start_room;
}

int *get_tilemap(int roomnumber)
{
   return level.rooms[roomnumber].tilemap;
}

int get_num_rooms(void)
{
   return level.num_rooms;
}

int *get_left_doors(int roomnumber)
{
   return level.doors[roomnumber];
}

int *get_right_doors(int roomnumber)
{
   return level.doors[(roomnumber+1)%level.num_rooms];
}

/* Returns TRUE if the tile the player is on is a door of some sort */
int tile_is_door(const int room, const int x, const int y)
{
   return level.rooms[room].tilemap[TILE_INDEX(x/TILE_WIDTH, y/TILE_HEIGHT)] & MAP_DOOR;
}

/* x and y in world coordinates, check if the door is open */
int tile_passable(const int room, const int x, const int y)
{
   int c;
   int *doors;
   
   
   if (!((x>=0) && (x/TILE_WIDTH<MAP_WIDTH) && (y>=0) && (y/TILE_HEIGHT<MAP_HEIGHT)))
      return FALSE;

   c = level.rooms[room].tilemap[TILE_INDEX((x+TILE_SIZE/2)/TILE_SIZE, (y+TILE_SIZE/2)/TILE_SIZE)];
   if (c & MAP_DOOR) {
      if (x>3*MAP_WIDTH*TILE_WIDTH/4) {
         doors = get_right_doors(room);
      } else if (x<MAP_WIDTH*TILE_WIDTH/4) {
         doors = get_left_doors(room);
      } else if (x/TILE_WIDTH==obj_xs[3]) {
         return TRUE;
      } else {
         return TRUE;
      }
      c = (y/TILE_HEIGHT-8)/15;
      /* We can pass the door if it is open, or at least not blocked on this */
      /* side. Locked doors are handled by item routines */
      return (doors[c] == MAP_DOOR_OPEN)                                   ||
             (doors[c] == MAP_DOOR_LOCKED)                                 ||
             ((doors[c]&MAP_DOOR_CREATUREL) && x>3*MAP_WIDTH*TILE_WIDTH/4) ||
             ((doors[c]&MAP_DOOR_CREATURER) && x<MAP_WIDTH*TILE_WIDTH/4)   ||
             ((doors[c]==MAP_DOOR_SWITCH2)&&(~level.switch_state))          ||
             ((doors[c]==MAP_DOOR_SWITCH)&&(level.switch_state));
   } else {
      return c == MAP_FLOOR;
   }
}

int tile_is_locked_door(const int room, const int x, const int y)
{
   int c;
   int *doors;
   ASSERT(x>=0);
   ASSERT(x<MAP_WIDTH);
   ASSERT(y>=0);
   ASSERT(y<MAP_HEIGHT);

   c = level.rooms[room].tilemap[TILE_INDEX(x/TILE_WIDTH, y/TILE_HEIGHT)];
   if (c & MAP_DOOR) {
      if (x>MAP_WIDTH*TILE_WIDTH/2) {
         doors = get_right_doors(room);
      } else {
         doors = get_left_doors(room);
      }
      return (doors[(y/TILE_HEIGHT-8)/15]&MAP_DOOR_LOCKED) == MAP_DOOR_LOCKED;
   } else {
      return FALSE;
   }
}

LINK *get_monster_list(const int room)
{
   return level.rooms[room].monster;
}

void set_monster_list(const int room, LINK *list)
{
   level.rooms[room].monster = list;
}

LINK *get_item_list(const int room)
{
   return level.rooms[room].item;
}

void set_item_list(const int room, LINK *list)
{
   level.rooms[room].item = list;
}

LINK *get_projectile_list(const int room)
{
   return level.rooms[room].projectile;
}

void set_projectile_list(const int room, LINK *list)
{
   level.rooms[room].projectile = list;
}

LINK *get_particle_list(const int room)
{
   return level.rooms[room].particle;
}

void set_particle_list(const int room, LINK *list)
{
   level.rooms[room].particle = list;
}

/* Calculate a set of bounding boxes for the map geometry */
void calculate_geometry_bbox(int room)
{
   LINK *l;
   int x, y;
   room = (room+level.num_rooms) % level.num_rooms;

   destroy_list(level.rooms[room].geometry_bbox, (void *)free_bshape);
   level.rooms[room].geometry_bbox = NULL;
   for (y=0; y<MAP_HEIGHT; y++) {
      for (x=0; x<MAP_WIDTH; x++) {
         if (!tile_passable(room, x*TILE_SIZE, y*TILE_SIZE)) {
            BSHAPE *bbox = alloc_bshape();
            assert(bbox);
            
            bbox->dx = x*TILE_SIZE;
            bbox->dy = y*TILE_SIZE;
            bbox->w = bbox->h = TILE_SIZE;
            bbox->r = TILE_SIZE/2;
            /* Try to merge as many bounding boxes as possible */
            for (l = level.rooms[room].geometry_bbox; l && bbox; l = l->next) {
               BSHAPE *bb2 = l->data;
               if (bb2->dy == bbox->dy && bb2->h == bbox->h && abs(bb2->dx - bbox->dx)==TILE_SIZE) {
                  bb2->dx = MIN(bb2->dx, bbox->dx);
                  bb2->w = bb2->w + bbox->w;
                  free_bshape(bbox);
                  bbox = NULL;
               } else if (bb2->dx == bbox->dx && bb2->w == bbox->w && abs(bb2->dy - bbox->dy)==TILE_SIZE) {
                  bb2->dy = MIN(bb2->dy, bbox->dy);
                  bb2->h = bb2->h + bbox->h;
                  free_bshape(bbox);
                  bbox = NULL;
               }
            }
            if (bbox) {
               level.rooms[room].geometry_bbox = append_to_list(bbox, level.rooms[room].geometry_bbox);
            }
         }
      }
   }
}

LINK *get_geometry_boundingbox(int room)
{   
   return level.rooms[room].geometry_bbox;
}

int collide_geometry(int room, int x, int y, BSHAPE *bbox)
{
   LINK *l;
   
   for(l=level.rooms[room].geometry_bbox; l; l=l->next) {
      if (collide_box(x, y, bbox, 0, 0, l->data)) return TRUE;
   }
   return FALSE;
}

/* return the global level switch state */
int get_switch_state(void)
{
   return level.switch_state;
}

/* toggle the switch state */
/* when toggling the switch state, all unpressed switches are pressed, */
/*  depressed are unpressed */
void toggle_switch_state(void)
{
   int c;
   LINK *l;
   ITEM *item;

   /* Toggle all switches */
   for (c=0; c<level.num_rooms; c++)
      for(l = get_item_list(c); l; l=l->next) {
         item = l->data;
         if (item->type == ITEM_SWITCHUP) {
            item->type = ITEM_SWITCHDN;
         } else if (item->type == ITEM_SWITCHDN) {
            item->type = ITEM_SWITCHUP;
         }
      }

   level.switch_state^=1;

   /* Calculate the level collision boundingbox */
   for(c=0; c<level.num_rooms; c++) {
      calculate_geometry_bbox(c);
   }
}
