/**********************************************/
/* Multi-controller game input routines       */
/* Evert Glebbeek 2002, 2003                  */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include <string.h>
#include "input.h"

/* Number of input devices per controller */
#define INPUT_DEV_PER_CONTROLLER 3
/* Number of controllers */
#define NUMBER_OF_CONTROLLERS    2

/* Game controller data structures */
typedef struct KEY_MAPPING {
   /* Gamepad mappings */
   /* Joystick number */
   int joy_n;
   /* The stick to use for up/down and l/r movement */
   int joy_ud_stick;
   int joy_lr_stick;
   /* Axes to use for up/down and l/r movement */
   int joy_ud_axis;
   int joy_lr_axis;
   /* Game buttons to set on up/down and left/right movement */
   int joy_udlr_buttons[4];
   /* Gamepad button map. We don't know the number of gamepad buttons in advance */
   int *joy_button_map;

   /* Keyboard mappings */
   int keyboard_mapping[NUM_GAMEKEYS];

   /* Mouse */
   /* Buttons to set on mickey counts */
   int mouse_udlr_buttons[4];
   int mouse_buttons[3];
   /* Buttons to set on wheel `mickeys' */
   int mouse_mzp;
   int mouse_mzn;
} KEY_MAPPING;

/* Prototype for game controller input routine */
/* It takes a mapping and an array of gamekeys */
/* Returns TRUE if a gamekey was pressed */
typedef int (GAME_CTRL_INPUT)(KEY_MAPPING keymap, int *gamekey);

/* Each controller has two possible input devices (to allow for multiple */
/*  input sources/key bindings) and one gamekey array */
typedef struct CONTROLLER {
   GAME_CTRL_INPUT *input[INPUT_DEV_PER_CONTROLLER];
   KEY_MAPPING keymap[INPUT_DEV_PER_CONTROLLER];
   int relax[INPUT_DEV_PER_CONTROLLER];
   int relax_time[INPUT_DEV_PER_CONTROLLER];
   int dev_type[INPUT_DEV_PER_CONTROLLER];

   int old_gamekey[INPUT_DEV_PER_CONTROLLER][NUM_GAMEKEYS];
   int gamekey[INPUT_DEV_PER_CONTROLLER][NUM_GAMEKEYS];
   int gamekey_pressed[INPUT_DEV_PER_CONTROLLER];

   int *gamekey_buffer;
   int gamekey_sp;
   int gamekey_sb;
} CONTROLLER;

static CONTROLLER ctrl[NUMBER_OF_CONTROLLERS];
static int init = FALSE;
static int timer_installed = FALSE;
static int gamekey_buffer_size = 16;

/* Timer routine to keep track of joystick relaxation time */
static void controller_relax(void)
{
   int c;
   int n;
   for (n=0; n<NUMBER_OF_CONTROLLERS; n++)
      for (c=0; c<INPUT_DEV_PER_CONTROLLER; c++) {
         if (ctrl[n].relax[c])
            ctrl[n].relax[c]--;
      }
}
END_OF_STATIC_FUNCTION(controller_relax);

/* Input devices */

/* Dummy input driver: always returns FALSE */
static int dummy_controller(KEY_MAPPING keymap, int *gamekey)
{
   return FALSE;
}

/* Keyboard driver */
static int keyboard_controller(KEY_MAPPING keymap, int *gamekey)
{
   int state = FALSE;
   int c;

   ASSERT(gamekey);

   for(c=0; c<NUM_GAMEKEYS; c++) {
      if (key[keymap.keyboard_mapping[c]]) {
         gamekey[c] = TRUE;
         state = TRUE;
      }
   }

   return state;
}

/* Gamepad driver */
static int joystick_controller(KEY_MAPPING keymap, int *gamekey)
{
   int state = FALSE;
   int c;

   poll_joystick();

   /* Up */
   if (joy[keymap.joy_n].stick[keymap.joy_ud_stick].axis[keymap.joy_ud_axis].d1 && keymap.joy_udlr_buttons[0]) {
      gamekey[keymap.joy_udlr_buttons[0]] = TRUE;
      state = TRUE;
   }

   /* Down */
   if (joy[keymap.joy_n].stick[keymap.joy_ud_stick].axis[keymap.joy_ud_axis].d2 && keymap.joy_udlr_buttons[1]) {
      gamekey[keymap.joy_udlr_buttons[1]] = TRUE;
      state = TRUE;
   }

   /* Left */
   if (joy[keymap.joy_n].stick[keymap.joy_lr_stick].axis[keymap.joy_lr_axis].d1 && keymap.joy_udlr_buttons[2]) {
      gamekey[keymap.joy_udlr_buttons[2]] = TRUE;
      state = TRUE;
   }

   /* Right */
   if (joy[keymap.joy_n].stick[keymap.joy_lr_stick].axis[keymap.joy_lr_axis].d2 && keymap.joy_udlr_buttons[3]) {
      gamekey[keymap.joy_udlr_buttons[3]] = TRUE;
      state = TRUE;
   }

   for (c=0; c<joy[keymap.joy_n].num_buttons; c++) {
      if (joy[keymap.joy_n].button[c].b && (keymap.joy_button_map[c]!=GAMEKEY_NONE)) {
         gamekey[keymap.joy_button_map[c]] = TRUE;
         state = TRUE;
      }
   }

   return state;
}

/* Mouse driver */
static int mouse_controller(KEY_MAPPING keymap, int *gamekey)
{
   int state = FALSE;
   int dx, dy, dz;
   int b;

   dz = mouse_z;
   b = mouse_b;
   get_mouse_mickeys(&dx, &dy);

   if (dx) {
      if (dx>0) {
         gamekey[keymap.mouse_udlr_buttons[0]] = TRUE;
         if (keymap.mouse_udlr_buttons[0])
            state = TRUE;
      } else {
         gamekey[keymap.mouse_udlr_buttons[1]] = TRUE;
         if (keymap.mouse_udlr_buttons[1])
            state = TRUE;
      }
   }

   if (dy) {
      if (dy>0) {
         gamekey[keymap.mouse_udlr_buttons[3]] = TRUE;
         if (keymap.mouse_udlr_buttons[3])
            state = TRUE;
      } else {
         gamekey[keymap.mouse_udlr_buttons[2]] = TRUE;
         if (keymap.mouse_udlr_buttons[2])
            state = TRUE;
      }
   }

   if (dz) {
      if (dz>0) {
         gamekey[keymap.mouse_mzp] = TRUE;
         if (keymap.mouse_mzp)
            state = TRUE;
      } else {
         gamekey[keymap.mouse_mzn] = TRUE;
         if (keymap.mouse_mzn)
            state = TRUE;
      }
   }

   if (b&1) {
      gamekey[keymap.mouse_buttons[0]] = TRUE;
      if (keymap.mouse_buttons[0])
         state = TRUE;
   }

   if (b&2) {
      gamekey[keymap.mouse_buttons[1]] = TRUE;
      if (keymap.mouse_buttons[1])
         state = TRUE;
   }

   if (b&4) {
      gamekey[keymap.mouse_buttons[2]] = TRUE;
      if (keymap.mouse_buttons[2])
         state = TRUE;
   }

   position_mouse_z(0);

   return state;
}

void initialize_input(void)
{
   int c;
   int n;

   /* Initialize joystick relaxation time counter */
   if (!init) {
      LOCK_FUNCTION(controller_relax);
      LOCK_VARIABLE(ctrl);
      for (n=0; n<NUMBER_OF_CONTROLLERS; n++)
         for (c=0; c<INPUT_DEV_PER_CONTROLLER; c++) {
            LOCK_VARIABLE(ctrl[n]);
            LOCK_VARIABLE(ctrl[n].relax[c]);
         }
   }
   if (timer_installed) {
      remove_int(controller_relax);
      timer_installed = FALSE;
   }

   if (install_int(controller_relax, 10)) {
      allegro_message("Failed to install input timer!\n");
      exit(EXIT_FAILURE);
   }

   timer_installed = TRUE;

   for (n=0; n<NUMBER_OF_CONTROLLERS; n++) {
      for (c=0; c<INPUT_DEV_PER_CONTROLLER; c++) {
         ctrl[n].input[c] = dummy_controller;
         ctrl[n].relax_time[c] = 0;
         ctrl[n].dev_type[c] = CTRL_INPUT_NONE;
         ctrl[n].keymap[c].joy_n = 0;
         ctrl[n].keymap[c].joy_ud_stick = 0;
         ctrl[n].keymap[c].joy_lr_stick = 0;
         ctrl[n].keymap[c].joy_ud_axis = 1;
         ctrl[n].keymap[c].joy_lr_axis = 0;
         ctrl[n].keymap[c].joy_button_map = NULL;
      }
      ctrl[n].gamekey_buffer = malloc(gamekey_buffer_size*sizeof(int));
      ctrl[n].gamekey_sp = 0;
      ctrl[n].gamekey_sb = 0;
   }
}

void shutdown_input(void)
{
   int n;
   int c;

   remove_int(controller_relax);
   timer_installed = FALSE;

   for (n=0; n<NUMBER_OF_CONTROLLERS; n++) {
      for (c=0; c<INPUT_DEV_PER_CONTROLLER; c++) {
         ctrl[n].input[c] = dummy_controller;
         ctrl[n].relax_time[c] = 0;
         ctrl[n].keymap[c].joy_n = 0;
         ctrl[n].keymap[c].joy_ud_stick = 0;
         ctrl[n].keymap[c].joy_lr_stick = 0;
         ctrl[n].keymap[c].joy_ud_axis = 1;
         ctrl[n].keymap[c].joy_lr_axis = 0;
         free(ctrl[n].keymap[c].joy_button_map);
         ctrl[n].keymap[c].joy_button_map = NULL;
      }
      free(ctrl[n].gamekey_buffer);
   }
}

/* Assigns an input device to a logical game controller */
/*   ctrl_num  is the number of the logical controller */
/*   device    is a source ID for the controller (multiple) */
/*   dev_id    is the type of input device */
/* Returns TRUE on succes, FALSE on failure */
int set_controller_device(int ctrl_num, int device, int dev_id)
{
   int joy_n = -1;
   int c;

   ASSERT(ctrl_num<NUMBER_OF_CONTROLLERS);
   ASSERT(device<INPUT_DEV_PER_CONTROLLER);
   ASSERT(dev_id<=CTRL_INPUT_MOUSE);

   switch (dev_id) {
      case CTRL_INPUT_NONE:
         ctrl[ctrl_num].input[device] = dummy_controller;
         ctrl[ctrl_num].dev_type[device] = dev_id;
         return TRUE;
      case CTRL_INPUT_KEYB:
         ctrl[ctrl_num].input[device] = keyboard_controller;
         ctrl[ctrl_num].dev_type[device] = dev_id;
         return TRUE;
      case CTRL_INPUT_JOY1:
      case CTRL_INPUT_JOY2:
      case CTRL_INPUT_JOY3:
      case CTRL_INPUT_JOY4:
         switch (dev_id) {
            case CTRL_INPUT_JOY1:
               joy_n = 0;
               break;
            case CTRL_INPUT_JOY2:
               joy_n = 1;
               break;
            case CTRL_INPUT_JOY3:
               joy_n = 2;
               break;
            case CTRL_INPUT_JOY4:
               joy_n = 3;
               break;
         }
         if (joy_n>=num_joysticks)
            return FALSE;

         ctrl[ctrl_num].input[device] = joystick_controller;
         ctrl[ctrl_num].keymap[device].joy_button_map = realloc(ctrl[ctrl_num].keymap[device].joy_button_map, joy[joy_n].num_buttons * sizeof(int));
         for (c=0; c<joy[joy_n].num_buttons; c++) {
            ctrl[ctrl_num].keymap[device].joy_button_map[c] = GAMEKEY_NONE;
         }
         ctrl[ctrl_num].keymap[device].joy_n = joy_n;
         /* The stick to use for up/down and l/r movement */
         ctrl[ctrl_num].keymap[device].joy_ud_stick = 0;
         ctrl[ctrl_num].keymap[device].joy_lr_stick = 0;
         /* Axes to use for up/down and l/r movement */
         ctrl[ctrl_num].keymap[device].joy_ud_axis = 1;
         ctrl[ctrl_num].keymap[device].joy_lr_axis = 0;

         ctrl[ctrl_num].dev_type[device] = CTRL_INPUT_JOY;

         return TRUE;
      case CTRL_INPUT_MOUSE:
         ctrl[ctrl_num].input[device] = mouse_controller;
         ctrl[ctrl_num].dev_type[device] = dev_id;
         return TRUE;
      default:
         return FALSE;
   }
}

/* Get driver ID for a specific input device */
int get_controller_device(int ctrl_num, int device)
{
   ASSERT(ctrl_num<NUMBER_OF_CONTROLLERS);
   ASSERT(device<INPUT_DEV_PER_CONTROLLER);
   return ctrl[ctrl_num].dev_type[device];
}

/* Set controller relax time (in units of 10 ms) */
void set_controller_device_relax_time(int ctrl_num, int device, int relax)
{
   ASSERT(ctrl_num<NUMBER_OF_CONTROLLERS);
   ASSERT(device<INPUT_DEV_PER_CONTROLLER);

   ctrl[ctrl_num].relax_time[device] = relax;
   ctrl[ctrl_num].relax[device] = relax;
}

/* Map a controller stick and axis to a button */
/* On a game pad, stick 0, axis 0 is left/right, stick 0, axis 1 is up/down */
/* On a mouse, stick 0 is x/y movement, axis 0 is left/right, 1 is up/down */
/*             stick 1, axis 0 is the mouse wheel */
/* Has no meaning for the keyboard driver */
void map_controller_axis(int ctrl_num, int device, int stick, int axis, int sign, int game_key)
{
   int dummy;

   ASSERT(ctrl_num<NUMBER_OF_CONTROLLERS);
   ASSERT(device<INPUT_DEV_PER_CONTROLLER);

   switch(ctrl[ctrl_num].dev_type[device]) {
      case CTRL_INPUT_JOY:
         if (axis==1) {
            if (sign>0) {
               ctrl[ctrl_num].keymap[device].joy_udlr_buttons[0] = game_key;
            } else {
               ctrl[ctrl_num].keymap[device].joy_udlr_buttons[1] = game_key;
            }
         } else if (axis==0) {
            if (sign>0) {
               ctrl[ctrl_num].keymap[device].joy_udlr_buttons[3] = game_key;
            } else {
               ctrl[ctrl_num].keymap[device].joy_udlr_buttons[2] = game_key;
            }
         }
         break;
      case CTRL_INPUT_MOUSE:
         if (stick==0) {   /* Normal x/y movement */
            if (axis==1) {
               if (sign>0) {
                  ctrl[ctrl_num].keymap[device].mouse_udlr_buttons[0] = game_key;
               } else {
                  ctrl[ctrl_num].keymap[device].mouse_udlr_buttons[1] = game_key;
               }
            } else if (axis==0) {
               if (sign>0) {
                  ctrl[ctrl_num].keymap[device].mouse_udlr_buttons[3] = game_key;
               } else {
                  ctrl[ctrl_num].keymap[device].mouse_udlr_buttons[2] = game_key;
               }
            }
         } else {          /* mouse wheel */
            if (axis==0) {
               if (sign>0) {
                  ctrl[ctrl_num].keymap[device].mouse_mzp = game_key;
               } else {
                  ctrl[ctrl_num].keymap[device].mouse_mzn = game_key;
               }
            }
         }
         position_mouse_z(0);
         get_mouse_mickeys(&dummy, &dummy);
         break;
      default:
         return;
   }
}

void map_controller_button(int ctrl_num, int device, int button, int game_key)
{
   ASSERT(ctrl_num<NUMBER_OF_CONTROLLERS);
   ASSERT(device<INPUT_DEV_PER_CONTROLLER);

   switch(ctrl[ctrl_num].dev_type[device]) {
      case CTRL_INPUT_KEYB:
         ctrl[ctrl_num].keymap[device].keyboard_mapping[game_key] = button;
         break;
      case CTRL_INPUT_JOY:
         ASSERT(button<=joy[ctrl[ctrl_num].keymap[device].joy_n].num_buttons);
         ctrl[ctrl_num].keymap[device].joy_button_map[button] = game_key;
         break;
      case CTRL_INPUT_MOUSE:
         ASSERT(button<=3);
         ctrl[ctrl_num].keymap[device].mouse_buttons[button] = game_key;
         break;
      default:
         break;
   }

}

/* Polls the controller for input */
int poll_controller(int ctrl_num)
{
   int dev;
   int p=FALSE;

   ASSERT(ctrl_num<NUMBER_OF_CONTROLLERS);


   for (dev=0; dev<INPUT_DEV_PER_CONTROLLER; dev++)
      if (!ctrl[ctrl_num].relax[dev] && ctrl[ctrl_num].dev_type[dev]) {
         memset(ctrl[ctrl_num].gamekey[dev], FALSE, NUM_GAMEKEYS*sizeof *(ctrl[ctrl_num].gamekey[dev]));
         ctrl[ctrl_num].gamekey_pressed[dev] = FALSE;
         if (ctrl[ctrl_num].input[dev](ctrl[ctrl_num].keymap[dev], ctrl[ctrl_num].gamekey[dev])) {
            ctrl[ctrl_num].gamekey_pressed[dev] = TRUE;
            ctrl[ctrl_num].relax[dev] = ctrl[ctrl_num].relax_time[dev];
            p=TRUE;
         }
      }

   return p;
}

/* Poll all controllers for input */
int poll_controllers(void)
{
   int state = FALSE;
   int c;

   for (c=0; c<NUMBER_OF_CONTROLLERS; c++)
      state = state || poll_controller(c);

   return state;
}

/* Store current press/release state for one controller */
/* This information can be used by the gamekey_released function: */
/*  it will return TRUE if a gamekey was released between the last call to */
/*  push_gamekey_state() and poll_controller() */
void push_gamekey_state(int ctrl_num)
{
   int dev;

   ASSERT(ctrl_num<NUMBER_OF_CONTROLLERS);


   for (dev=0; dev<INPUT_DEV_PER_CONTROLLER; dev++)
      if (!ctrl[ctrl_num].relax[dev] && ctrl[ctrl_num].dev_type[dev]) {
         /* Save old state */
         memcpy(ctrl[ctrl_num].old_gamekey[dev], ctrl[ctrl_num].gamekey[dev], NUM_GAMEKEYS*sizeof *(ctrl[ctrl_num].gamekey[dev]));
      }
}

/* Store current press/release state for all controllers */
void push_gamekey_states(void)
{
   int c;

   for (c=0; c<NUMBER_OF_CONTROLLERS; c++)
      push_gamekey_state(c);
}

/* Functions similar to Allegro's key_pressed() function and key[] array */

/* Returns TRUE if a key is currently pressed on the controller */
int gamekey_pressed(int ctrl_num)
{
   int dev;

   ASSERT(ctrl_num<NUMBER_OF_CONTROLLERS);

   for (dev=0; dev<INPUT_DEV_PER_CONTROLLER; dev++)
      if (ctrl[ctrl_num].dev_type[dev] && ctrl[ctrl_num].gamekey_pressed[dev])
         return TRUE;

   return FALSE;
}

/* Returns TRUE if key is currently pressed down */
int gamekey(int ctrl_num, int key)
{
   int dev;

   ASSERT(ctrl_num<NUMBER_OF_CONTROLLERS);
   ASSERT(key<NUM_GAMEKEYS);

   for (dev=0; dev<INPUT_DEV_PER_CONTROLLER; dev++)
      if (ctrl[ctrl_num].dev_type[dev] && ctrl[ctrl_num].gamekey[dev][key])
         return TRUE;
   return FALSE;
}

/* Returns TRUE if key was pressed at the last call of push_gamekey_state(), */
/*  but is not pressed down now */
int gamekey_released(int ctrl_num, int key)
{
   int dev;

   ASSERT(ctrl_num<NUMBER_OF_CONTROLLERS);
   ASSERT(key<NUM_GAMEKEYS);

   for (dev=0; dev<INPUT_DEV_PER_CONTROLLER; dev++)
      if (ctrl[ctrl_num].dev_type[dev] && (ctrl[ctrl_num].old_gamekey[dev][key] && !ctrl[ctrl_num].gamekey[dev][key]))
         return TRUE;
   return FALSE;
}

/* Buffering routines */
/* Set size of the gamekey buffer. Unsafe to call after initialize_input */
void set_gamekey_buffer_size(int size)
{
   ASSERT(!timer_installed);

   gamekey_buffer_size = size;
}

/* Push gamekey */
void push_gamekey(int ctrl_num, int k)
{
   ASSERT(ctrl_num<NUMBER_OF_CONTROLLERS);
   ASSERT(k<NUM_GAMEKEYS);

   /* Trap buffer overflows */
   if ((ctrl[ctrl_num].gamekey_sp+1)%gamekey_buffer_size == ctrl[ctrl_num].gamekey_sb) {
      //TRACE("push_gamekey: input buffer overflow!\n");
      return;
   }

   ctrl[ctrl_num].gamekey_buffer[ctrl[ctrl_num].gamekey_sp] = k;
   ctrl[ctrl_num].gamekey_sp = (ctrl[ctrl_num].gamekey_sp+1)%gamekey_buffer_size;
}

/* Pop gamekey */
int pop_gamekey(int ctrl_num)
{
   int k;
   ASSERT(ctrl_num<NUMBER_OF_CONTROLLERS);

   k = ctrl[ctrl_num].gamekey_buffer[ctrl[ctrl_num].gamekey_sb];
   ctrl[ctrl_num].gamekey_sb = (ctrl[ctrl_num].gamekey_sb+1)%gamekey_buffer_size;
   return k;
}

/* Returns TRUE if there are keys waiting in the buffer */
int gamekey_in_buffer(int ctrl_num)
{
   ASSERT(ctrl_num<NUMBER_OF_CONTROLLERS);
   return ctrl[ctrl_num].gamekey_sb!=ctrl[ctrl_num].gamekey_sp;
}

/* Flush the buffer */
void flush_gamekey_buffer(int ctrl_num)
{
   ASSERT(ctrl_num<NUMBER_OF_CONTROLLERS);
   ctrl[ctrl_num].gamekey_sb=ctrl[ctrl_num].gamekey_sp;
}
