/**********************************************/
/* Reusable floating window routines          */
/* Evert Glebbeek 2002, 2003                  */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include <string.h>
#include <stdio.h>
#include "floatwin.h"
#include "gfx.h"

/* Floating window frame set */
#define FRAME_BOTTOM   0
#define FRAME_LEFT     1
#define FRAME_LL       2
#define FRAME_LR       3
#define FRAME_RIGHT    4
#define FRAME_TOP      5
#define FRAME_UL       6
#define FRAME_UR       7

FLOATWIN_LIST *flwin = NULL;

static BITMAP **flwin_frame = NULL;

/****************************/
/* Text printing routines   */
/****************************/

/* Set border bitmaps for floating windows, This is a list of 8 bitmaps */
/*  representing the edges and corners of the frame */
void set_flwin_border(BITMAP **blst)
{
   flwin_frame = blst;
}

BITMAP *text_boxed(int maxw, int bgcolor, int textcolor,
                   const FONT *textfont, int center, char *msg, ...)
{
   char *s = malloc(4096);
   char *l = s;
   char *lastline = l;
   char *space = l;
   char *lastspace = space;
   BITMAP *bmp;
   BITMAP *textbox;
   int numlines = 1;
   int c;
   int w = maxw;
   int h;


   /* Expand the parameter into a full-length string */
   va_list ap;

   va_start(ap, msg);
   uvsprintf(s, msg, ap);
   va_end(ap);

   /* We need to split the text in a number of lines */
   /* We do this by finding spaces. We replace the space with a '\0' and */
   /*  check to make sure the string isn't too long. We replace it again */
   /*  with a space and look for the next. Once the line is too long,    */
   /*  we insert a new-line at the location of the previous string */

   while (space) {
      lastspace = space;
      /* find the next space in l */
      space = strstr(space, " ");
      if (space) {
         space[0] = '\0';
         /* Check if the line is now longer than the width of the window */
         if (text_length(textfont, lastline) > (w - 2 * 12)) {
            /* back-track to previous space and insert space */
            space[0] = ' ';
            space++;
            /* Check if there is a line-break in the string */
            if (strstr(lastline, "\n")) {
               /* Reset pointers to the line break and dont break the line */
               /*  at this space */
               lastline = strstr(lastline, "\n") + 1;
               lastspace = space = lastline;
            } else {
               lastspace--;
               lastspace[0] = '\n';
               /* Set pointer to the start of the new line */
               lastline = lastspace + 1;
            }
         } else {
            /* Restore the space */
            space[0] = ' ';
            space++;
         }
      }
   }
   /* Add a newline to the end of the string */
   l[strlen(l) + 1] = '\0';
   l[strlen(l)] = '\n';

   /* count number of newlines in string */
   /* This will also count the newlines inserted by the user */
   lastline = l;
   numlines = 0;
   do {
      lastline = strstr(lastline, "\n");
      if (lastline) {
         numlines++;
         lastline++;
      }
   } while (lastline);

   /* Now, l points to the parsed string with appropriate line-breaks and */
   /*  numlines is the number of text rows. Convert this to the height    */
   /*  for the window in pixels */

   h = numlines * text_height(textfont) + 3 * 8;

   textbox = create_bitmap(w, h);

   /* Clear the bitmap */
   clear_to_color(textbox, bgcolor);

   /* Create the text border */
   if (flwin_frame) {
      bmp = flwin_frame[FRAME_TOP];
      for (c = 0; c < w + bmp->w; c += bmp->w)
         draw_sprite(textbox, bmp, c, 0);
      bmp = flwin_frame[FRAME_BOTTOM];
      for (c = 0; c < w + bmp->w; c += bmp->w)
         draw_sprite(textbox, bmp, c, h - bmp->h);
      bmp = flwin_frame[FRAME_LEFT];
      for (c = 0; c < h + bmp->h; c += bmp->h)
         draw_sprite(textbox, bmp, 0, c);
      bmp = flwin_frame[FRAME_RIGHT];
      for (c = 0; c < h + bmp->h; c += bmp->h)
         draw_sprite(textbox, bmp, w - bmp->w, c);
      bmp = flwin_frame[FRAME_UL];
      draw_sprite(textbox, bmp, 0, 0);
      bmp = flwin_frame[FRAME_UR];
      draw_sprite(textbox, bmp, w - bmp->w, 0);
      bmp = flwin_frame[FRAME_LL];
      draw_sprite(textbox, bmp, 0, h - bmp->h);
      bmp = flwin_frame[FRAME_LR];
      draw_sprite(textbox, bmp, w - bmp->w, h - bmp->h);
   }
   /* Now, begin writing the text lines */
   text_mode(bgcolor);
   for (c = 0; c < numlines; c++) {
      lastline = strstr(l, "\n");
      if (lastline) {
         lastline[0] = '\0';
         if (center)
            textout_centre(textbox, textfont, l,
                           w / 2, 12 + c * text_height(textfont), textcolor);
         else
            textout_justify(textbox, textfont, l,
                            12, w - 12, 12 + c * text_height(textfont),
                            5 * text_length(textfont, " "), textcolor);
         l = lastline + 1;
      }
   }
   /* We've created the window, now let's draw it! */

   /* And we're done */
   free(s);

   return textbox;
}


/****************************/
/* Floating windows         */
/****************************/

/* Add a floating window to the list of floating windows */
void push_floatwin(FLOATWIN *flw)
{
   FLOATWIN_LIST *flwl;

   /* Setup the structure to store this one in the floating window list */
   flwl = malloc(sizeof(FLOATWIN_LIST));
   flwl->win = flw;
   flwl->next = flwin;
   flwl->prev = NULL;

   /* Add it to the list */
   if (flwin)
      flwin->prev = flwl;
   flwin = flwl;

   if (!(flw->flags & FWIN_STICKMAP)) {
      /* Finally, mark the floating window area as `dirty' so it gets drawn next time */
      mark_rect(flw->x, flw->y, flw->w, flw->h);
   } else {
      /* Finally, mark the floating window area as `dirty' so it gets drawn next time */
      mark_rect(flw->x, flw->y, flw->w, flw->h);
   }
}

/* Removes a floating window from the list */
void destroy_floatwin(FLOATWIN *flw)
{
   FLOATWIN_LIST *flwl = flwin;

   while (flwl) {
      /* If we've found the floating window, remove it */
      if (flwl->win == flw) {
         FLOATWIN_LIST *flwldummy = flwl->next;

         /* Free window memory that was allocated by the floatwin manager */
         if (flw->bg)
            destroy_bitmap(flw->bg);
         if (flw->content)
            destroy_bitmap(flw->content);
         if (flw->bbuffer)
            destroy_bitmap(flw->bbuffer);
         free(flw);
         flw = NULL;

         /* Update linked list */
         if (flwl->next)
            flwl->next->prev = flwl->prev;
         if (flwl->prev)
            flwl->prev->next = flwl->next;

         /* Make sure the head of the list remains valid */
         if (flwin == flwl)
            flwin = flwl->next;

         free(flwl);
         /* proceed with next entry */
         flwl = flwldummy;
      } else {
         flwl = flwl->next;
      }
   }
}

/* Removes all floating windows from the list */
void destroy_all_floatwins(void)
{
   while (flwin) {
      destroy_floatwin(flwin->win);
   }
}

/* General floating-window function */
FLOATWIN *make_floatwin(int x, int y, int w, int h, int flags, int lifetime, BITMAP *bg, BITMAP *content)
{
   FLOATWIN *flw = malloc(sizeof(FLOATWIN));

   /* Create floating window */
   flw->bg = bg;
   flw->bbuffer = NULL;
   flw->content = content;
   flw->x = x;
   flw->y = y;
   flw->w = w;
   flw->h = h;
   flw->flags = flags|FWIN_DIRTY;
   flw->timeleft = lifetime;

   /* We've created the window, now let's draw it! */
   push_floatwin(flw);

   return flw;
}

/* Remove a floating window from the list and mark all tiles under it as */
/*  `dirty' */
void kill_floatwin(FLOATWIN *flw)
{
   undraw_floatwin(flw);
   destroy_floatwin(flw);
}

/* Create a floating text at the (x, y) screen location */
/* This essentially makes a borderless floatwin */
FLOATWIN *make_floattext(int x, int y, int bgcolor, int fgcolor, int flags,
                         int lifetime, const FONT *textfont, char *msg, ...)
{
   char *s = malloc(4096);
   FLOATWIN *flw;
   int w;
   int h;


   /* Expand the parameter into a full-length string */
   va_list ap;

   va_start(ap, msg);
   uvsprintf(s, msg, ap);
   va_end(ap);

   w = text_length(textfont, s) + 2;
   h = text_height(textfont) + 2;

   /* Now, create the window */
   flw =
      make_floatwin(x, y, w, h, flags, lifetime, NULL, create_bitmap(w, h));

   /* Clear the bitmap */
   rectfill(flw->content, 0, 0, w, h, bitmap_mask_color(flw->content));

   /* Now, begin writing the text lines */
   text_mode(-1);

   textout(flw->content, textfont, s, 0, 0, bgcolor);
   textout(flw->content, textfont, s, 2, 2, bgcolor);
   textout(flw->content, textfont, s, 1, 1, fgcolor);

   /* And we're done */
   free(s);

   return flw;
}

/* Create a floating message box and add it to the list */
/* If center is 1, then the text will be centered. Otherwise it will be */
/* justified */
/* The window frame has thickness 12; the border GFX have thickness 8 */
FLOATWIN *make_floatmsg(int bgcolor, int textcolor, const FONT *textfont,
                        int center, int lifetime, char *msg, ...)
{
   char *s = malloc(4096);
   BITMAP *bmp;
   FLOATWIN *flw;

   /* Expand the parameter into a full-length string */
   va_list ap;

   va_start(ap, msg);
   uvsprintf(s, msg, ap);
   va_end(ap);

   /* print the text */
   bmp =
      text_boxed(SCREEN_W * 8 / 10, bgcolor, textcolor, textfont, center, s);
   /* Now, create the window */
   flw =
      make_floatwin((SCREEN_W - bmp->w) / 2, 16, bmp->w, bmp->h, 0, lifetime,
                    NULL, bmp);

   /* And we're done */
   free(s);

   return flw;
}

/* Create a floating message box and add it to the list */
/* If center is 1, then the text will be centered. Otherwise it will be */
/* justified */
/* The window frame has thickness 12; the border GFX have thickness 8 */
FLOATWIN *make_floatmsg_fit(int bgcolor, int textcolor, const FONT *textfont,
                            int center, int lifetime, char *msg, ...)
{
   char *s = malloc(4096);
   FLOATWIN *flw;
   BITMAP *bmp = NULL;

   /* Expand the parameter into a full-length string */
   va_list ap;

   va_start(ap, msg);
   uvsprintf(s, msg, ap);
   va_end(ap);

   /* print the text */
   bmp =
      text_boxed(text_length(textfont, s) + 2 * 8, bgcolor, textcolor,
                 textfont, center, s);
   /* create the window */
   flw =
      make_floatwin((SCREEN_W - bmp->w) / 2, 16, bmp->w, bmp->h, 0, lifetime,
                    NULL, bmp);

   /* And we're done */
   free(s);

   return flw;
}

/* Find a floatwin that has the specified flag */
FLOATWIN *find_floatwin(int flag)
{
   FLOATWIN_LIST *flwl = flwin;

   while (flwl) {
      /* If we've found the floating window, remove it */
      if ((flwl->win->flags & flag) == flag) {
         return flwl->win;
      }
      flwl = flwl->next;
   }

   return NULL;
}

void set_lifetime(FLOATWIN *flw, int lifetime)
{
   if (flw) {
      flw->timeleft = lifetime;
   }
}

int get_lifetime(FLOATWIN *flw)
{
   return flw->timeleft;
}

/****************************/
/* Floating windows         */
/****************************/
/* Draw (ie, blit) a floating window to the screen */
void draw_floatwin(FLOATWIN *flw)
{
   BITMAP *bmp;
   int x, y;

   if (flw->flags & FWIN_HIDDEN)
      return;

   if ((flw->flags & FWIN_DIRTY)) {
      mark_rect(flw->x, flw->y, flw->w, flw->h);
      flw->flags &= ~FWIN_DIRTY;
   }

   x = flw->x;
   y = flw->y;
   bmp = get_screen_bmp();

   /* Store backbuffer */
   flw->bbuffer = create_bitmap(flw->w, flw->y);
   if (flw->bbuffer)
      blit(bmp, flw->bbuffer, x, y, 0,0, flw->w, flw->y);

   if (flw->flags & FWIN_FADETPDN) {
      draw_gouraud_sprite(bmp, flw->content, x, y, 64,
                          64, 255, 255);
   } else if (flw->flags & FWIN_TRANS) {
      draw_trans_sprite(bmp, flw->content, x, y);
   } else {
      draw_sprite(bmp, flw->content, x, y);
   }
}

/* Draw (ie, blit) all floating windows to the screen */
void draw_floatwins(void)
{
   FLOATWIN_LIST *flwl = flwin;

   while (flwl) {
      draw_floatwin (flwl->win);
      flwl = flwl->next;
   }
}

/* Draw the backbuffer over the floatwin */
void undraw_floatwin(FLOATWIN *flw)
{
   if (flw->bbuffer) {
      blit(flw->bbuffer, get_screen_bmp(), 0,0, flw->x, flw->y, flw->w, flw->y);
      destroy_bitmap(flw->bbuffer);
      flw->bbuffer = NULL;
   }
}

void undraw_floatwins(void)
{
   FLOATWIN_LIST *flwl = flwin;

   while (flwl) {
      undraw_floatwin (flwl->win);
      flwl = flwl->next;
   }
}
