/**********************************************/
/* Dungeon of Darkness                        */
/* Evert Glebbeek 2003                        */
/* eglebbk@dds.nl                             */
/**********************************************/
#include <allegro.h>
#include "gamespri.h"
#include "sprigfx.h"
#include "global.h"
#include "items.h"
#include "gfx.h"

/* Datafile holding all sprites */
static DATAFILE *sprites_dat = NULL;

/* Specific sub-datafiles, holding monster/character sprites */
static SPRITE batspr[BAT_DAT_COUNT];
static SPRITE vambatspr[BAT_DAT_COUNT];
static SPRITE ghostspr[GHOST_DAT_COUNT];
static SPRITE ghoulspr[GHOULSPR_DAT_COUNT];
static SPRITE herospr[HEROSPR_DAT_COUNT];
static SPRITE skelspr[SKELSPR_DAT_COUNT];
static SPRITE spawnspr[SPAWNSPR_DAT_COUNT];
static SPRITE aspawnspr[ASPAWNSPR_DAT_COUNT];
static SPRITE zomspr[ZOMSPR_DAT_COUNT];
static SPRITE armzomspr[ZOMSPR2_DAT_COUNT];
static SPRITE daemonspr[DAEMON_DAT_COUNT];
static SPRITE wizspr[WIZSPR_DAT_COUNT];
static SPRITE targetspr[TARGET_DAT_COUNT];
static SPRITE itemsspr[ITEMS_COUNT];
static SPRITE projectilespr[PROSPR_DAT_COUNT];
static SPRITE serpentspr[SERPENT_DAT_COUNT];
static SPRITE reaperspr[REAPSPR_DAT_COUNT];
static SPRITE shieldspr[SHIELDSPR_DAT_COUNT];
static SPRITE clubspr[CLUBSPR_DAT_COUNT];
static SPRITE club2spr[CLUB2SPR_DAT_COUNT];
static SPRITE daggerspr[DAGGERSPR_DAT_COUNT];
static SPRITE dagger2spr[DAGGER2SPR_DAT_COUNT];
static SPRITE swordspr[SWORDSPR_DAT_COUNT];
static SPRITE sword2spr[SWORDSPR2_DAT_COUNT];
static SPRITE kingbatspr[KINGBAT_DAT_COUNT];

#define MAX_RENDER_SPRITES    (512*1024)

static RENDER_SPRITE *render_sprite_chunk = NULL;
static RENDER_SPRITE **render_sprite_stack = NULL;
static int render_sprite_stack_counter = 0;

RENDER_SPRITE *alloc_render_sprite(void)
{
   /* return the address of the last free render_sprite in the list */
   return render_sprite_stack[render_sprite_stack_counter++];
}

void free_render_sprite(RENDER_SPRITE *l)
{
   render_sprite_stack[--render_sprite_stack_counter]=l;
}

int get_render_sprite_count(void)
{
   return render_sprite_stack_counter;
}

int init_render_sprites(void)
{
   int c;

   render_sprite_chunk = realloc(render_sprite_chunk, sizeof *render_sprite_chunk *MAX_RENDER_SPRITES);
   render_sprite_stack = realloc(render_sprite_stack, sizeof *render_sprite_stack *MAX_RENDER_SPRITES);
   
   render_sprite_stack_counter = 0;
   for (c=0; c<MAX_RENDER_SPRITES; c++)
      render_sprite_stack[c] = &(render_sprite_chunk[c]);

   return (sizeof(RENDER_SPRITE)+sizeof(RENDER_SPRITE *))*MAX_RENDER_SPRITES;
}

void free_render_sprites(void)
{
   free(render_sprite_chunk);
   free(render_sprite_stack);
   
   render_sprite_chunk = NULL;
   render_sprite_stack = NULL;
   render_sprite_stack_counter = 0;
}

/* Load and convert graphics */
static void load_and_convert(SPRITE *spr, int count, DATAFILE *dat, int allow_vbmp)
{
   int c;

   for (c=0; c<count; c++) {
      spr[c].rle = dat[c].dat;
      if ((settings.flags & USE_BMPS) || !(settings.flags & USE_RLES)) {
         if (allow_vbmp)
            spr[c].bmp = create_optimized_mask_bitmap(spr[c].rle->w, spr[c].rle->h);
         else
            spr[c].bmp = create_bitmap(spr[c].rle->w, spr[c].rle->h);
      } else {
         spr[c].bmp = NULL;
      }
      if (spr[c].bmp) {
         clear_to_color(spr[c].bmp, bitmap_mask_color(spr[c].bmp));
         draw_rle_sprite(spr[c].bmp, spr[c].rle, 0,0);
      }
   }
}

/* Destroy sprites */
static void destroy_sprites(SPRITE *spr, int count)
{
   int c;

   for (c=0; c<count; c++) {
      /* The RLE sprite is taken from the datafile */
      destroy_bitmap(spr[c].bmp);
      spr[c].rle = NULL;
      spr[c].bmp = NULL;
   }
}

/* Load sprite datafile */
/* Returns 0 on succes, -1 on failure */
int load_sprites(const char *filename)
{
   if (sprites_dat) {
      unload_datafile(sprites_dat);
   }

   sprites_dat = load_datafile(get_game_path(filename));
   if (!sprites_dat)
      return -1;
   
   /* Convert graphics from teh datafile. Disallow video bitmaps for sprites that need to be rotated */
   load_and_convert(batspr, BAT_DAT_COUNT, sprites_dat[BAT_DAT].dat, 1);
   load_and_convert(vambatspr, VAMBAT_DAT_COUNT, sprites_dat[VAMBAT_DAT].dat, 1);
   load_and_convert(ghostspr, GHOST_DAT_COUNT, sprites_dat[GHOST_DAT].dat, 1);
   load_and_convert(ghoulspr, GHOULSPR_DAT_COUNT, sprites_dat[GHOULSPR_DAT].dat, 1);
   load_and_convert(herospr, HEROSPR_DAT_COUNT, sprites_dat[HEROSPR_DAT].dat, 1);
   load_and_convert(skelspr, SKELSPR_DAT_COUNT, sprites_dat[SKELSPR_DAT].dat, 1);
   load_and_convert(spawnspr, SPAWNSPR_DAT_COUNT, sprites_dat[SPAWNSPR_DAT].dat, 1);
   load_and_convert(aspawnspr, ASPAWNSPR_DAT_COUNT, sprites_dat[ASPAWNSPR_DAT].dat, 1);
   load_and_convert(zomspr, ZOMSPR_DAT_COUNT, sprites_dat[ZOMSPR_DAT].dat, 1);
   load_and_convert(armzomspr, ZOMSPR2_DAT_COUNT, sprites_dat[ZOMSPR2_DAT].dat, 1);
   load_and_convert(daemonspr, DAEMON_DAT_COUNT, sprites_dat[DAEMON_DAT].dat, 1);
   load_and_convert(wizspr, WIZSPR_DAT_COUNT, sprites_dat[WIZSPR_DAT].dat, 1);
   load_and_convert(targetspr, TARGET_DAT_COUNT, sprites_dat[TARGET_DAT].dat, 1);
   load_and_convert(itemsspr, ITEMS_COUNT, sprites_dat[ITEMS].dat, 1);
   load_and_convert(projectilespr, PROSPR_DAT_COUNT, sprites_dat[PROSPR_DAT].dat, 0);
   load_and_convert(serpentspr, SERPENT_DAT_COUNT, sprites_dat[SERPENT_DAT].dat, 0);
   load_and_convert(reaperspr, REAPSPR_DAT_COUNT, sprites_dat[REAPSPR_DAT].dat, 1);
   load_and_convert(shieldspr, SHIELDSPR_DAT_COUNT, sprites_dat[SHIELDSPR_DAT].dat, 1);
   load_and_convert(clubspr, CLUBSPR_DAT_COUNT, sprites_dat[CLUBSPR_DAT].dat, 1);
   load_and_convert(club2spr, CLUB2SPR_DAT_COUNT, sprites_dat[CLUB2SPR_DAT].dat, 1);
   load_and_convert(daggerspr, DAGGER2SPR_DAT_COUNT, sprites_dat[DAGGERSPR_DAT].dat, 1);
   load_and_convert(dagger2spr, DAGGER2SPR_DAT_COUNT, sprites_dat[DAGGER2SPR_DAT].dat, 1);
   load_and_convert(swordspr, SWORDSPR_DAT_COUNT, sprites_dat[SWORDSPR_DAT].dat, 1);
   load_and_convert(sword2spr, SWORDSPR2_DAT_COUNT, sprites_dat[SWORDSPR2_DAT].dat, 1);
   load_and_convert(kingbatspr, KINGBAT_DAT_COUNT, sprites_dat[KINGBAT_DAT].dat, 1);
   return 0;
}

void unload_sprites(void)
{
   if (sprites_dat) {
      unload_datafile(sprites_dat);

      sprites_dat = NULL;

      destroy_sprites(batspr, BAT_DAT_COUNT);
      destroy_sprites(vambatspr, VAMBAT_DAT_COUNT);
      destroy_sprites(ghostspr, GHOST_DAT_COUNT);
      destroy_sprites(ghoulspr, GHOULSPR_DAT_COUNT);
      destroy_sprites(herospr, HEROSPR_DAT_COUNT);
      destroy_sprites(skelspr, SKELSPR_DAT_COUNT);
      destroy_sprites(spawnspr, SPAWNSPR_DAT_COUNT);
      destroy_sprites(aspawnspr, ASPAWNSPR_DAT_COUNT);
      destroy_sprites(zomspr, ZOMSPR_DAT_COUNT);
      destroy_sprites(armzomspr, ZOMSPR2_DAT_COUNT);
      destroy_sprites(daemonspr, DAEMON_DAT_COUNT);
      destroy_sprites(wizspr, WIZSPR_DAT_COUNT);
      destroy_sprites(targetspr, TARGET_DAT_COUNT);
      destroy_sprites(itemsspr, ITEMS_COUNT);
      destroy_sprites(projectilespr, PROSPR_DAT_COUNT);
      destroy_sprites(serpentspr, SERPENT_DAT_COUNT);
      destroy_sprites(reaperspr, REAPSPR_DAT_COUNT);
      destroy_sprites(shieldspr, SHIELDSPR_DAT_COUNT);
      destroy_sprites(clubspr, CLUBSPR_DAT_COUNT);
      destroy_sprites(club2spr, CLUB2SPR_DAT_COUNT);
      destroy_sprites(daggerspr, DAGGERSPR_DAT_COUNT);
      destroy_sprites(dagger2spr, DAGGER2SPR_DAT_COUNT);
      destroy_sprites(swordspr, SWORDSPR_DAT_COUNT);
      destroy_sprites(sword2spr, SWORDSPR2_DAT_COUNT);
      destroy_sprites(kingbatspr, KINGBAT_DAT_COUNT);
   }
}

/* Return a projectile sprite */
SPRITE *get_projectile_sprite(const int projectile)
{
   ASSERT(sprites_dat);
   ASSERT(projectilespr);

   return projectilespr+projectile;
}

/* Helper function: get animation for a character's walking animation */
static inline SPRITE *get_char_sprite(SPRITE *df, const int dir, const int anim)
{
   int c;

   ASSERT(dir>=0);
   ASSERT(dir<4);

   c = dir;

   switch (anim%4) {
      case 0:        /* Standing still */
      case 2:        /* In between motion */
         break;
      case 1:        /* Right foot forward */
         c+=4;
         break;
      case 3:        /* Left foot forward */
         c+=8;
         break;
      default:
         break;
   }
   c += 12*(anim/12);

   return df+c;
}

#define GET_XXXX_SPRITE(str)\
SPRITE *get_##str##_sprite(const int dir, const int action, const int anim)\
{\
   ASSERT(sprites_dat);\
   ASSERT(str##spr);\
   \
   return get_char_sprite(str##spr, dir, anim+12*action);\
}

GET_XXXX_SPRITE(hero);
GET_XXXX_SPRITE(ghoul);
GET_XXXX_SPRITE(skel);
GET_XXXX_SPRITE(zom);
GET_XXXX_SPRITE(armzom);
GET_XXXX_SPRITE(daemon);
GET_XXXX_SPRITE(wiz);
GET_XXXX_SPRITE(reaper);
GET_XXXX_SPRITE(shield);
GET_XXXX_SPRITE(club);
GET_XXXX_SPRITE(club2);
GET_XXXX_SPRITE(dagger);
GET_XXXX_SPRITE(dagger2);
GET_XXXX_SPRITE(sword);
GET_XXXX_SPRITE(sword2);

SPRITE *get_spawn_sprite(const int dir, const int action, const int anim)
{
   ASSERT(sprites_dat);
   
   (void)action;

   return spawnspr+dir+4*(anim&1);
}

SPRITE *get_aspawn_sprite(const int dir, const int action, const int anim)
{
   ASSERT(sprites_dat);
   
   (void)action;

   return aspawnspr+dir+4*(anim&1);
}

SPRITE *get_bat_sprite(const int dir, const int action, const int anim)
{
   ASSERT(sprites_dat);
   
   (void)action;
   (void)dir;

   return batspr+anim;
}

SPRITE *get_vambat_sprite(const int dir, const int action, const int anim)
{
   ASSERT(sprites_dat);
   
   (void)action;
   (void)dir;

   return vambatspr+anim;
}

SPRITE *get_kingbat_sprite(const int dir, const int action, const int anim)
{
   ASSERT(sprites_dat);
   
   (void)action;
   (void)dir;

   return kingbatspr+anim;
}

SPRITE *get_ghost_sprite(const int dir, const int action, const int anim)
{
   ASSERT(sprites_dat);
   (void)action;
   (void)dir;

   return ghostspr+anim;
}

/* Return a segment of the serpent */
SPRITE *get_serpent_sprite(const int segment)
{
   ASSERT(sprites_dat);
   ASSERT(serpentspr);

   return serpentspr+segment;
}


int get_item_gfx_index(int type)
{
   switch (type) {
      case ITEM_BOW:
         return ITEMS_BOW_PCX;
      case ITEM_CROSSBOW:
         return ITEMS_CRBOW_PCX;
      case ITEM_CLUB:
         return ITEMS_CLUB1_PCX;
      case ITEM_MACE:
         return ITEMS_MACE1_PCX;
      case ITEM_DAGGER:
         return ITEMS_DAGGER1_PCX;
      case ITEM_DAGGER2:
         return ITEMS_DAGGER2_PCX;
      case ITEM_SWORD:
         return ITEMS_SWORD1_PCX;
      case ITEM_SWORD2:
         return ITEMS_SWORD2_PCX;
      case ITEM_SHIELD:
         return ITEMS_LSHIELD_PCX;
      case ITEM_OWLFIG:
         return ITEMS_OWLFIG_PCX;
      case ITEM_CATFIG:
         return ITEMS_CATFIG_PCX;
      case ITEM_SNAKEFIG:
         return ITEMS_COBRAFIG_PCX;
      case ITEM_CHAINMAIL:
         return ITEMS_CHAINMAIL_PCX;
      case ITEM_HELMET:
         return ITEMS_HELMET_PCX;

      case ITEM_ARROW:
         return ITEMS_ARROWS_PCX;
      case ITEM_SARROW:
         return ITEMS_SARROWS_PCX;
      case ITEM_FARROW:
         return ITEMS_FARROWS_PCX;
      case ITEM_BOMB:
         return ITEMS_BOMB_PCX;
      case ITEM_QHEART:
         return ITEMS_HEARTC_PCX;
      case ITEM_HEARTC:
         return ITEMS_HEARTC4Q_PCX;
      case ITEM_KEY:
         return ITEMS_KEY_PCX;
      case ITEM_POTION:
         return ITEMS_POTION_PCX;
      case ITEM_HEART:
         return ITEMS_HEARTSS1_PCX;
      case ITEM_BOSSGATE:
         return ITEMS_EXIT2_PCX;
      case ITEM_BOSSGATETOP:
         return ITEMS_EXIT2A_PCX;
      case ITEM_EXIT:
      case ITEM_BOSSEXIT:
         return ITEMS_EXIT_PCX;
      case ITEM_CANDLEHOLD:
         return ITEMS_CANHOLD_PCX;
      case ITEM_CHALICE:
         return ITEMS_CHALICE_PCX;
      case ITEM_GEM:
         return ITEMS_GEM_PCX;
      case ITEM_POISON:
         return ITEMS_POISON_PCX;
      case ITEM_TRAP:
         return ITEMS_TRAP_PCX;
      case ITEM_SWITCHWGHTU:
      case ITEM_SWITCHUP:
         return ITEMS_SWITCHUP_PCX;
      case ITEM_SWITCHWGHTD:
      case ITEM_SWITCHDN:
         return ITEMS_SWITCHDN_PCX;
      case ITEM_RINGREGEN:
         return ITEMS_RING2_PCX;
      case ITEM_TELLRING:
         return ITEMS_RING0_PCX;
      case ITEM_CHESTOPEN:
         return ITEMS_CHEST2_PCX;
      case ITEM_CHESTCLOSED:
         return ITEMS_CHEST1_PCX;
      case ITEM_BLOCK:
         return ITEMS_BLOCK2_PCX;
      case ITEM_BLOCKCRACK:
         return ITEMS_BLOCK3_PCX;
      case ITEM_BLOCKLOCKED:
         return ITEMS_BLOCK4_PCX;
      case ITEM_LOCKEDTELEP:
      case ITEM_TELEPORTER:
         return ITEMS_TELEPORT_PCX;
      case ITEM_STATUE:
         return ITEMS_STATUE_PCX;
      default:
         ASSERT(0);
         return 0;
   }
}

/* Get item sprites */
SPRITE *get_item_sprite(const int gfx)
{
   ASSERT(sprites_dat);
   ASSERT(gfx>=0);
   ASSERT(gfx<ITEMS_COUNT);

   return itemsspr+gfx;
}

/* Small heart - state=0 means filled, state=1 means empty */
SPRITE *get_heart_item_sprite(const int state)
{
   ASSERT(state>=0);
   ASSERT(state<=2);

   return get_item_sprite(ITEMS_HEARTSS1_PCX+state);
}

/* Heart container, empty, quarter, half, three quarter or full */
SPRITE *get_heart_container_item_sprite(const int filled)
{
   ASSERT(filled>=0)
   ASSERT(filled<4)
   return get_item_sprite(ITEMS_HEARTC0Q_PCX+filled);
}

SPRITE *get_bow_item_sprite(const int state)
{
   if (!state)
      return get_item_sprite(ITEMS_BOW_PCX);
   else
      return get_item_sprite(ITEMS_BOW2_PCX);
}

SPRITE *get_crossbow_item_sprite(const int state)
{
   if (!state)
      return get_item_sprite(ITEMS_CRBOW_PCX);
   else
      return get_item_sprite(ITEMS_CRBOW2_PCX);
}

SPRITE *get_arrows_item_sprite(void)
{
   return get_item_sprite(ITEMS_ARROWS_PCX);
}

SPRITE *get_bomb_item_sprite(void)
{
   return get_item_sprite(ITEMS_BOMB_PCX);
}

SPRITE *get_key_item_sprite(void)
{
   return get_item_sprite(ITEMS_KEY_PCX);
}

SPRITE *get_arrow_sprite(const int anim)
{
   ASSERT(sprites_dat);
   (void)anim;
   
   return get_projectile_sprite(PROSPR_DAT_ARROWN_PCX);
}

SPRITE *get_swordpr_sprite(const int anim)
{
   ASSERT(sprites_dat);
   (void)anim;

   return get_projectile_sprite(PROSPR_DAT_SWORD1_PCX);
}

SPRITE *get_dagger1pr_sprite(const int anim)
{
   ASSERT(sprites_dat);
   (void)anim;

   return get_projectile_sprite(PROSPR_DAT_DAGGER1_PCX);
}

SPRITE *get_mace_sprite(const int anim)
{
   ASSERT(sprites_dat);
   (void)anim;

   return get_projectile_sprite(PROSPR_DAT_MACE_PCX);
}

SPRITE *get_death_sprite(const int anim)
{
   ASSERT(sprites_dat);
   (void)anim;

   return get_projectile_sprite(PROSPR_DAT_DEATH0_PCX+anim);
}

SPRITE *get_bomb_sprite(const int anim)
{
   ASSERT(sprites_dat);

   if (anim < 5) {
      return get_projectile_sprite(PROSPR_DAT_BOMB1_PCX+anim);
   } else {
      return get_projectile_sprite(PROSPR_DAT_BOMB1_PCX+3+((anim-5)&1));
   }
}

SPRITE *get_explosion_sprite(const int anim)
{
   ASSERT(sprites_dat);

   return get_projectile_sprite(PROSPR_DAT_EXPLO1_PCX+anim);
}

SPRITE *get_fireball_sprite(const int anim)
{
   ASSERT(sprites_dat);

   return get_projectile_sprite(PROSPR_DAT_FIREBALL0_PCX+(anim&1));
}


SPRITE *get_target_sprite(const int anim)
{
   ASSERT(sprites_dat);

   return targetspr+(anim % TARGET_DAT_COUNT);
}

void game_draw_sprite(BITMAP *bmp, SPRITE *sprite, const int x, const int y)
{
   /* Always use bitmaps if RLE sprites are disabled and BITMAPS are not */
   if (((settings.flags&(USE_BMPS|USE_RLES)) == USE_BMPS) && sprite->bmp) {
      draw_sprite(bmp, sprite->bmp, x, y);
   /* If both are enabled, use BMP's if they are hardware accelerated */
   } else if (((settings.flags&(USE_BMPS|USE_RLES))==(USE_BMPS|USE_RLES)) && sprite->bmp && ((gfx_capabilities & GFX_HW_SYS_TO_VRAM_BLIT_MASKED) || (gfx_capabilities & GFX_HW_VRAM_BLIT_MASKED))) {
      draw_sprite(bmp, sprite->bmp, x, y);
   /* In all other cases, use RLE sprites */
   } else {
      draw_rle_sprite(bmp, sprite->rle, x, y);
   }
}

void game_centre_sprite(BITMAP *bmp, SPRITE *sprite, const int x, const int y)
{
   game_draw_sprite(bmp, sprite, x-sprite->rle->w/2, y-sprite->rle->h/2);
}

void game_draw_lit_sprite(BITMAP *bmp, SPRITE *sprite, const int x, const int y, const int colour)
{
   draw_lit_rle_sprite(bmp, sprite->rle, x, y, colour);
}

void game_draw_trans_sprite(BITMAP *bmp, SPRITE *sprite, const int x, const int y)
{
   draw_trans_rle_sprite(bmp, sprite->rle, x, y);
}

void game_rotate_sprite(BITMAP *bmp, SPRITE *sprite, const int x, const int y, fixed angle)
{
   if (!sprite->bmp) {
      BITMAP *bmp2;

      bmp2 = create_bitmap(sprite->rle->w, sprite->rle->h);
      clear_to_color(bmp2, bitmap_mask_color(bmp2));
      draw_rle_sprite(bmp2, sprite->rle, 0, 0);
      rotate_sprite (bmp, bmp2, x, y, angle);
      destroy_bitmap(bmp2);
   } else {
      rotate_sprite (bmp, sprite->bmp, x, y, angle);
   }
}

void game_render_sprite(BITMAP *bmp, RENDER_SPRITE *sprite)
{
   ASSERT(bmp);
   ASSERT(sprite);

   if (sprite->render_flags == RENDER_NORMAL) {
      game_draw_sprite(bmp, sprite->sprite, sprite->x, sprite->y);
   }
   if (sprite->render_flags & RENDER_TRANS) {
      game_draw_trans_sprite(bmp, sprite->sprite, sprite->x, sprite->y);
   }
   if (sprite->render_flags & RENDER_LIT) {
      game_draw_lit_sprite(bmp, sprite->sprite, sprite->x, sprite->y, 128);
   }
   if (sprite->render_flags & RENDER_ROTATE) {
      game_rotate_sprite(bmp, sprite->sprite, sprite->x, sprite->y, sprite->angle);
   }
   if (sprite->render_flags & RENDER_CENTRE) {
      game_centre_sprite(bmp, sprite->sprite, sprite->x, sprite->y);
   }
}
