/**********************************************/
/* Dungeon of Darkness                        */
/* Evert Glebbeek 2003, 2005                  */
/* eglebbk@dds.nl                             */
/**********************************************/
#include <allegro.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include "dark.h"
#include "playgame.h"
#include "highscor.h"
#include "dialog.h"
#include "global.h"
#include "sound.h"
#include "gfx.h"
#include "game.h"
#include "gamegfx.h"
#include "alintro.h"
#include "gmeparse.h"
#include "gmemusic.h"
#include "srun.h"
#include "input.h"
#include "linked.h"
#include "gamespri.h"
#include "maptile.h"
#include "player.h"
#include "mapdraw.h"
#include "map.h"
#include "monster.h"
#include "genrand.h"
#include "password.h"
#include "damage.h"
#include "items.h"
#include "particle.h"
#include "droptab.h"
#include "palette.h"
#include "keymap.h"
#include "bbox.h"

#define STARTLEVEL      'A'   // M
#define MAX_ARROWS      127
#define MAX_BOMBS        31
#define MAX_KEYS         31

/* Gamekey associations */
#define GAMEKEY_ATTACK GAMEKEY_BUTTON1
#define GAMEKEY_ACT    GAMEKEY_BUTTON2
#define GAMEKEY_BOMB   GAMEKEY_BUTTON3
#define GAMEKEY_CENTRE GAMEKEY_BUTTON4
#define GAMEKEY_SELECT GAMEKEY_BUTTON8
#define GAMEKEY_START  GAMEKEY_BUTTON7
#define GAMEKEY_SELECT GAMEKEY_BUTTON8
#define GAMEKEY_PREV   GAMEKEY_BUTTON5
#define GAMEKEY_NEXT   GAMEKEY_BUTTON6

/* Boss music playlist */
static int boss_playlist[] = { 2,  /* Number of items in the play-list */
   6, 7
};

/* Game-over music playlist */
static int game_over_playlist[] = { 1,  /* Number of items in the play-list */
   5
};

/* Game music playlist */
static int game_playlist[] = { 4,  /* Number of items in the play-list */
   1, 2, 3, 4
};

/* Titlescreen playlist */
static int title_playlist[] = { 1,  /* Number of items in the play-list */
   0
};

static HERO hero;

static int redraw_tilemap = FALSE;
static int load_boss_level = FALSE;
static int is_boss_level = FALSE;
static int boss_level_id;

/* Current game state */
#define GAME_STATE_NORMAL        0
#define GAME_STATE_NEW_LEVEL     1
#define GAME_STATE_GAME_OVER     2
#define GAME_STATE_WIN           4
#define GAME_STATE_PASSWORD      8
#define GAME_STATE_PAUSEGAME     16
#define GAME_STATE_EXITMENU      32
#define GAME_STATE_ITEMMENU      64
#define GAME_STATE_GET_HIGHSCORE 128
#define GAME_STATE_SHOW_SCORES   256

static int game_state = GAME_STATE_NORMAL;
static int game_state_counter = 0;
static int game_state_options = 0;

static int top_x = -1;
static int top_y = -1;

static int new_game = TRUE;   /* True if the level was started from the game menu */

static BITMAP *statusbar = NULL;

/* To enable cheat keys (F-keys), uncomment the following line and recompile */
//#define ENALE_CHEAT_KEYS

/* To see the bounding boxes during play, uncomment the following and recompile */
//#define SHOW_BBOXES
#ifdef SHOW_BBOXES
static BITMAP *bbox = NULL;
#endif

static char *password = NULL;
static char *last_password = NULL;

static LINK *render_list = NULL;

static void display_switchin(void)
{
   redraw_tilemap = TRUE;
   set_screen_changed();
}

/* returns 0 when `exit' is chosen, or non-zero if gameplay can begin */
static int game_menu(void)
{
   BITMAP *bmp;
   PALETTE pal;
   FONT *f;
   int focus;
   int res;
   int mv;
   int v;

   /* Load title background */
   bmp = get_title_screen(pal);
   set_palette(*get_game_palette());
   select_palette(*get_game_palette());

   /* Select menu font */
   f = font;
   font = (FONT *)get_menu_font();

   /* Set the transparent blender */
   set_trans_blender(0, 0, 0, 128);

   /* Show the background */
   prepare_screen_update();
   stretch_blit(bmp, get_screen_bmp(), 0, 0, bmp->w, bmp->h, 0, 0, SCREEN_W,
                SCREEN_H);
   font = (FONT *)get_small_font();
   textprintf(get_screen_bmp(), font, 2, SCREEN_H - text_height(font), white,
              GAME_VERSTR);
   mark_rect(0, 0, SCREEN_W, SCREEN_H);
   screen_update_done();

   font = (FONT *)get_menu_font();

   /* Start title music */
   stop_music();
   mv = get_music_volume();
   set_music_playlist(title_playlist);

   if ((focus=find_dialog_focus(main_menu_dialog)) == -1) {
      focus = 1;
   }

   select_mouse_pointer(-1);
   do {
      show_mouse(get_monitor_bmp());

      game_popup_dialog(main_menu_dialog, focus, FALSE);
      res = 0;
      while (dialog_open()) {
         /* Simulate keypresses from alternate input sources */
         res = game_update_dialog();
      }
      switch (res) {
         case 1:               /* Play game */
            do {
               game_popup_dialog(game_menu_dialog, 1, FALSE);
               res = 0;
               while (dialog_open()) {
                  /* Simulate keypresses from alternate input sources */
                  res = game_update_dialog();
               }
               switch (res) {
                  case 1:        /* New game */
                     show_mouse(NULL);
                     stop_music();
                     set_music_volume(mv);
                     font = f;
                     hero.level = -1;   /* Start at level 0 */
                     //hero.level = 'H' - 'A' - 1;
                     hero.level = STARTLEVEL - 'A' - 1;
                     return -1;
                  case 2:               /* password */
                     password = malloc(24);
                     password[0] = '\0';
                     /* Try to read back last password from the game config system */
                     if (!last_password) {
                        const char *s = get_config_string("game", "password", NULL);
                        if (s) last_password = strdup(s);
                     }
                     /* Make the last password the default password */
                     if (last_password)
                        sprintf(password, "%s", last_password);
                     show_mouse(NULL);
                     if (get_password(password)) {
                        if (decode_password(password,&hero)) {
                           return -1;
                        } else {
                           free(password);
                           password = NULL;
                           start_sample(12, FALSE);
                           game_message("Incorrect password!");
                        }
                     } else {
                        free(password);
                        password = NULL;
                     }
                     break;
                  case 3:                 /* Difficulty settings */
                     res = popup_dialog(game_difficulty_dialog, -1)-2;
                     if (res>=0)
                        settings.difficulty = res;
                     res = 0;
                     break;
                  default:
                     break;
               }                 /* End of switch */
            } while (res!=5 && res != -1 && settings.window_close_button == 0);
            res = 0;
            break;
         case 3:               /* options */
            popup_options();
            mv = get_music_volume();
            break;
         case 4:               /* high score */
            /* Clear buffers */
            while (joystick_button());
            while(keypressed())
               readkey();
             
            select_palette((*get_font_palette()));
            bmp = create_bitmap(SCREEN_W, SCREEN_H);
            scare_mouse();
            blit(get_monitor_bmp(), bmp, 0,0, 0,0, SCREEN_W, SCREEN_H);
            show_highscore_list(get_monitor_bmp(), get_colour_font(), -1);
            unscare_mouse();
            while (!(keypressed() || mouse_b || joystick_button()));
            /* Clear buffers */
            while(joystick_button());
            while(mouse_b);
            while(keypressed())
               readkey();

            scare_mouse();
            blit(bmp, get_monitor_bmp(), 0,0, 0,0, SCREEN_W, SCREEN_H);
            unscare_mouse();
            destroy_bitmap(bmp);
            
            break;
         default:
            break;
      }                         /* end of switch */
      show_mouse(NULL);
   } while (res != 5 && res != -1 && settings.window_close_button == 0);

   /* Fade out title music and stop */
   while ((v = get_music_volume())>0) {
      set_music_volume(--v);
   }
   stop_music();
   /* Restore original settings */
   set_music_volume(mv);
   font = f;
   return 0;
}

static void load_controller_settings(const int player, const int device)
{
   int map[NUM_GAMEKEYS];
   char s[128];
   int key;
   int n;
   int c;

   /* First read default settings */
   load_default_mapping(player, get_controller_device(player, device), map);
   for (c=1; c<NUM_GAMEKEYS; c++) {
      if (map[c]!=-1) {
         //popup_message("Player %d device %d (DevDriver %d) button %d mapped to key %d", player, device, get_controller_device(player, device), c, map[c]);
         map_controller_button(player, device, map[c], c);
      }
   }

   /* Set relaxation time */
   set_controller_device_relax_time(player, device, 1);

   /* Map gamepad directional pad to up/down/left/right */
   if (get_controller_device(player, device)>=CTRL_INPUT_JOY1) {
      map_controller_axis(0,1, 0,1,  1, GAMEKEY_UP);
      map_controller_axis(0,1, 0,1, -1, GAMEKEY_DOWN);
      map_controller_axis(0,1, 0,0, -1, GAMEKEY_LEFT);
      map_controller_axis(0,1, 0,0,  1, GAMEKEY_RIGHT);
   }

   /* Allow override via config file */
   sprintf (s, "controller_%d_%d", player, device);
   n = get_config_int("controller", s, -1);
   if (n!=-1) {
      //popup_message("Player %d device %d set to DevDriver %d (was %d)", player, device, n, get_controller_device(player, device));
      set_controller_device(player, device, n);
   }

   /* Gamepad or mouse */
   if (n>=CTRL_INPUT_JOY1) {
      map_controller_axis(player, device, 0,1,  1, GAMEKEY_UP);
      map_controller_axis(player, device, 0,1, -1, GAMEKEY_DOWN);
      map_controller_axis(player, device, 0,0, -1, GAMEKEY_LEFT);
      map_controller_axis(player, device, 0,0,  1, GAMEKEY_RIGHT);
   }

   if (n) {
      for (c=1; c<NUM_GAMEKEYS; c++) {
         sprintf (s, "key_%d_%d_%d", player, device, c);
         key = get_config_int("controller", s, -1);
         if (key!=-1) {
            map_controller_button(player, device, key, c);
         }
      }
   }

}

/* Get sprite direction, 0, 1, 2, 3, 4, 5, 6  or 7 from a facing angle a */
static int get_sprite_direction(double a)
{
   static int compass[] = { 7, 0, 1, 
                            6, 0, 2, 
                            5, 4, 3 };
   double x = cos(a);
   double y = sin(a);
   int ix, iy;
   ix = sgn( (int)(x*2) ) + 1;
   iy = sgn( (int)(-y*2) ) + 1;
   
   return compass[ix+3*iy];
}

static double get_direction_angle(int d)
{
   static double a[] = { M_PI/2, M_PI/4, 0, -M_PI/4,  
                        -M_PI/2, -3*M_PI/4, M_PI, 3*M_PI/4 };
   return a[d];
}

/* Returns TRUE if the monster faces the player and is close enough */
static int monster_face_player(MONSTER *m, HERO *player, int max_distance)
{
   double inprod = (player->x - m->x)*cos(m->angle) + (player->y - m->y)*sin(m->angle);
   
   return inprod>0 && distsqr(player->x, player->y, m->x, m->y)<max_distance;
}

static void calc_hero_defence(HERO *player)
{
   /* No damage from elemental attacks (that saves most bomb damage) */
   player->def = st_dmg_elem(255);
   /* Calculate additional armour from inventory */
   if (player->inventory&ITEM_SHIELD)
      player->def |= st_dmg_impact(128);
   if (player->inventory&ITEM_CHAINMAIL)
      player->def |= st_dmg_cut(128);
   if (player->inventory&ITEM_HELMET)
      player->def |= st_dmg_cut(32)|st_dmg_impact(32)|st_dmg_pierce(32);
}

static int calc_weapon_damage(int inventory, int weapon)
{
   int dmg = 0;
   
   switch (weapon) {
      case ITEM_BOW:
         dmg = st_dmg_impact(pack_dmg(1,4));
         if (inventory & ITEM_SARROW)
           dmg |= st_dmg_cut(pack_dmg(2,2))|st_dmg_pierce(pack_dmg(3,4));
         else
           dmg |= st_dmg_cut(pack_dmg(1,3))|st_dmg_pierce(pack_dmg(2,6));
         //dmg = st_dmg_impact(pack_dmg(2,2))|st_dmg_cut(pack_dmg(2,2))|st_dmg_pierce(pack_dmg(4,4));
         //dmg = st_dmg_impact(pack_dmg(4,2))|st_dmg_cut(pack_dmg(4,2))|st_dmg_pierce(pack_dmg(8,2));
         break;

      case ITEM_CROSSBOW:
      case ITEM_CROSSBOW|ITEM_BOW:
         dmg = st_dmg_impact(pack_dmg(2,3));
         if (inventory & ITEM_SARROW)
           dmg |= st_dmg_cut(pack_dmg(2,2))|st_dmg_pierce(pack_dmg(3,6));
         else
           dmg |= st_dmg_cut(pack_dmg(1,3))|st_dmg_pierce(pack_dmg(2,8));

         //dmg = st_dmg_impact(pack_dmg(2,2))|st_dmg_cut(pack_dmg(2,2))|st_dmg_pierce(pack_dmg(4,4));
         //dmg = st_dmg_impact(pack_dmg(4,2))|st_dmg_cut(pack_dmg(4,2))|st_dmg_pierce(pack_dmg(8,2));
         break;

      case ITEM_CLUB:
         dmg = st_dmg_impact(pack_dmg(3,6));
         break;

      case ITEM_MACE:
      case ITEM_MACE|ITEM_CLUB:
         dmg = st_dmg_impact(pack_dmg(3,8));
         break;

      case ITEM_DAGGER:
         dmg = st_dmg_cut(pack_dmg(1,3))|st_dmg_pierce(pack_dmg(2,8));
         break;

      case ITEM_DAGGER2:
      case ITEM_DAGGER2|ITEM_DAGGER:
         dmg = st_dmg_cut(pack_dmg(1,4))|st_dmg_pierce(pack_dmg(3,6));
         break;

      case ITEM_SWORD:
         dmg = st_dmg_impact(pack_dmg(1,3))|st_dmg_cut(pack_dmg(2,8));
         break;

      case ITEM_SWORD2:
      case ITEM_SWORD2|ITEM_SWORD:
         dmg = st_dmg_impact(pack_dmg(1,4))|st_dmg_cut(pack_dmg(3,6));
         break;
      default:
         break;
   } /* end switch */
   
   return dmg;
}

static void next_level(void)
{
   game_state = GAME_STATE_NEW_LEVEL;
   load_boss_level = FALSE;
   is_boss_level = FALSE;
   game_state_counter = 60;

   /* Initialize hero state variables */
   hero.target = NULL;
   hero.target_x = 0;
   hero.target_y = 0;
   hero.next_level = FALSE;
   hero.select_timeout = 0;
   hero.weapon_cooldown = 0;
   hero.max_cooldown = 0;

   hero.direction = 0;
   hero.angle = M_PI/2;
   hero.dx = 0;
   hero.dy = 0;
   hero.anim = 0;
   hero.invc = 0;

   hero.action = ACTION_NONE;

   /* Decode password if we started from one */
   if (password && new_game) {
      if (!decode_password(password,&hero)) {
         //printf ("CRC error decoding password!\n");
         set_game_done();
      }
   } else {
      hero.level++;

      /* If this is the first level, enter some other stuff as well */
      if (hero.level==0) {
         hero.max_health=3*8;                  /* 8 energy units -> one heart */
         hero.arrows=0;
         hero.bombs=0;
         hero.keys=0;
         hero.inventory=0;
         hero.weapon=0;
      }
      /* Give 15 bonus arrows at the start of each new level, 5 more at higher difficulty levels */
      /* (You need them) */
      hero.arrows+=15+settings.difficulty*5;
      if (hero.arrows>127)
         hero.arrows=127;
   }

   /* Select weapon (if any) */
   if (hero.inventory&(ITEM_BOW|ITEM_CLUB|ITEM_DAGGER|ITEM_SWORD|ITEM_CROSSBOW|ITEM_MACE|ITEM_DAGGER2|ITEM_SWORD)) {
      if (hero.inventory&(ITEM_BOW|ITEM_CROSSBOW))
         hero.weapon = hero.inventory&(ITEM_BOW|ITEM_CROSSBOW);
      else if (hero.inventory&(ITEM_SWORD|ITEM_SWORD2))
         hero.weapon = hero.inventory&(ITEM_SWORD|ITEM_SWORD2);
      else if (hero.inventory&(ITEM_DAGGER|ITEM_DAGGER2))
         hero.weapon = hero.inventory&(ITEM_DAGGER|ITEM_DAGGER2);
      else if (hero.inventory&(ITEM_MACE|ITEM_CLUB))
         hero.weapon = hero.inventory&(ITEM_MACE|ITEM_CLUB);
      /* Deselect weakest weapon */
      if (!is_power_of_two(hero.weapon))
         hero.weapon &= (ITEM_CROSSBOW|ITEM_MACE|ITEM_DAGGER2|ITEM_SWORD2);
   }


   if (new_game) {
      hero.health = (3+(hero.max_health/8-2)/2)*8;  /* 8 energy units -> one heart */
      hero.score=0;
      new_game = FALSE;
   }

   free(password);
   free(last_password);
   password = generate_password(&hero);
   last_password = strdup(password);
   set_config_string("game", "password", last_password);

   {
      FILE *f = fopen(get_game_path("../passwrds.txt"), "a");
      fprintf(f, "Level %c [Bombs: %d, Arrows %d, Hearts %d]\t",
                 hero.level+'A', hero.bombs, hero.arrows, hero.max_health/8);
      fprintf(f, "Password: %s\n", password);
      fclose(f);
   }
   /* Calculate defence from inventory */
   calc_hero_defence(&hero);

   /*
   {
      printf ("%d\n", hero.arrows);
      if (!decode_password(password,&hero)) {
         printf ("CRC error decoding password %s!\n", password);
      }
      printf ("%d\n", hero.arrows);
   }
   */
   /* Read actual level data */
   unload_map();
   load_level(hero.level);
   hero.current_room = get_start_room();
   hero.x = TILE_WIDTH*get_start_x();
   hero.y = TILE_HEIGHT*get_start_y();
   redraw_tilemap = TRUE;
   stop_music();
}

inline void playgame_preamble(void)
{
   /* This game was just started */
   new_game = TRUE;

   game_state = GAME_STATE_NORMAL;
   game_state_counter = 0;
   game_state_options = 0;

   /* Clear the screen */
   clear_bitmap(screen);
   statusbar = create_bitmap(SCREEN_W, TILE_SIZE);

   /* Load graphics */
   if (load_tileset("tiles.dat")!=0) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Unable to load the tileset!\n");
      exit(EXIT_FAILURE);
   }

   if (load_sprites("sprites.dat")!=0) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Unable to load the sprites!\n");
      exit(EXIT_FAILURE);
   }

   set_trans_blender(128,0,0,128);

   /* Initialise input devices */
   set_gamekey_buffer_size(settings.gamekey_buffer_size);
   initialize_input();

   /* Load default controller settings */

   /* Set default player 1 controller */
   set_controller_device(0, 0, CTRL_INPUT_KEYB);
   set_controller_device(0, 1, CTRL_INPUT_JOY1);

   /* Load controller key bindings/config file override */
   load_controller_settings(0, 0);
   load_controller_settings(0, 1);
   load_controller_settings(0, 2);

   /* Initialize general linked list structure */
   init_lists();

   /* Reset random number generator */
   sgenrand(time(NULL));

   /* Load game map */
   map_init();

   /* Initialise memory pools */
   init_objects();
   init_particles();
   init_monsters();
   init_render_sprites();
   init_boxes();

   map_draw_initialize();

   next_level();

   /* Initialise script virtual machine and read game scripts */
   //register_opcodes();

   /* Set the music we want */
   set_music_playlist(game_playlist);

   /* Clear input state */
   poll_controllers();
   push_gamekey_states();
}

inline void playgame_closing(void)
{
   prepare_screen_update();
   clear_bitmap(get_screen_bmp());
   mark_rect(0, 0, SCREEN_W, SCREEN_H);
   screen_update_done();

   destroy_bitmap(statusbar);
   statusbar = NULL;

   shutdown_input();

   map_draw_shutdown();

   free_objects();
   free_particles();
   free_monsters();
   free_render_sprites();
   free_boxes();
   free_lists();
   unload_tileset();
   unload_sprites();

   free(password);
   password = NULL;

   while (keypressed())
      readkey();
}

inline void game_gfx_update_preamble(void)
{
}

static int render_sort(const void *p1, const void *p2)
{
   const RENDER_SPRITE *spr1 = p1;
   const RENDER_SPRITE *spr2 = p2;

   if (spr1->sort_flags == spr2->sort_flags)
      return spr1->y - spr2->y;
   else
      return spr1->sort_flags - spr2->sort_flags;
}

static void draw_monsters(void)
{
   int dx;
   int dy;
   int n;
   int direction;
   SPRITE *sprite = NULL;
   LINK *l;
   RENDER_SPRITE *rspr;

   n = 0;
   for (l=get_monster_list(hero.current_room); l; l=l->next) {
      MONSTER *m = l->data;
      dy = 0;
      dx = 0;

      if (tile_is_visible(m->x/TILE_WIDTH, m->y/TILE_HEIGHT)) {
         direction = get_sprite_direction(m->angle);
         /* Get monster sprite */
         switch (m->type) {
            case MONSTER_SPAWN:
               if (m->x < 2*TILE_WIDTH) {
                  sprite = get_spawn_sprite(1,0,(m->anim/5)&1);
                  dx = 8;
               } else if (m->x > (MAP_WIDTH-3)*TILE_WIDTH) {
                  sprite = get_spawn_sprite(3,0,(m->anim/5)&1);
                  dx = -8;
               } else {
                  sprite = get_spawn_sprite(0,0,(m->anim/5)&1);
               }
               break;
            case MONSTER_ASPAWN:
               if (m->x < 2*TILE_WIDTH) {
                  sprite = get_aspawn_sprite(1,0,(m->anim/5)&1);
                  dx = 8;
               } else if (m->x > (MAP_WIDTH-3)*TILE_WIDTH) {
                  sprite = get_aspawn_sprite(3,0,(m->anim/5)&1);
                  dx = -8;
               } else {
                  sprite = get_aspawn_sprite(0,0,(m->anim/5)&1);
               }
               break;
            case MONSTER_ZOMBIE:
               sprite = get_zom_sprite(direction/2,0,(m->anim/5)&3);
               dy = -16;
               break;
            case MONSTER_ARMEDZOMBIE:
               sprite = get_armzom_sprite(direction/2,0,(m->anim/5)&3);
               dy = -16;
               break;
            case MONSTER_SKELETON:
               sprite = get_skel_sprite(direction/2,0,(m->anim/5)&3);
               dy = -16;
               break;
            case MONSTER_GHOUL:
               sprite = get_ghoul_sprite(direction/2,0,(m->anim/5)&3);
               dy = -16;
               break;
            case MONSTER_BAT:
               sprite = get_bat_sprite(0,0,(m->anim/5)&1);
               dy = -16;
               break;
            case MONSTER_BOSS2:
               sprite = get_kingbat_sprite(0,0,(m->anim/5)&1);
               dy = -16;
               break;
            case MONSTER_VAMPIREBAT:
               sprite = get_vambat_sprite(0,0,(m->anim/5)&1);
               dy = -16;
               break;
            case MONSTER_DAEMON:
               sprite = get_daemon_sprite(direction/2,0,(m->anim/5)&3);
               dy = -16;
               break;
            case MONSTER_GHOST:
               sprite = get_ghost_sprite(0,0,(m->anim/5)&1);
               dy = 0;
               break;
            case MONSTER_WIZARD:
               if (m->action & ACTION_SPELL)
                  sprite = get_wiz_sprite(direction/2,1,(m->anim/10)&3);
               else
                  sprite = get_wiz_sprite(direction/2,0,(m->anim/5)&3);
               dy = -16;
               break;
            case MONSTER_GRIMREAPER:
               sprite = get_reaper_sprite(direction/2,0,(m->anim/5)&3);
               dy = 0;
               break;
            case MONSTER_BOSS1:
               if (!m->prev) {
                  /* No previous one - head */
                  sprite = get_serpent_sprite(SERPENT_HEAD);
                  n = 0;
               } else if (!m->next) {
                  /* No next one - tail */
                  sprite = get_serpent_sprite(SERPENT_TAIL);
                  n = 0;
               } else {
                  /* Body segment */
                  sprite = get_serpent_sprite(SERPENT_BODY);
                  n++;
               }
               dx = dy = 0;
               break;
         }

         /* How the monster is drawn depends on wether it is a normal monster */
         /*  or a boss */
         rspr = alloc_render_sprite();
         ASSERT(rspr);

         rspr->sprite = sprite;
         rspr->render_flags = RENDER_NORMAL;
         rspr->sort_flags = 0;
         rspr->x = m->x + dx - top_x;
         rspr->y = m->y + dy - top_y;
         
         if (m->ability & MONSTER_ABILITY_FLY) {
            rspr->sort_flags = 100;
         }
         if (m->ability & MONSTER_ABILITY_SPAWN) {
            rspr->sort_flags = -100;
         }

         if (m->type!=MONSTER_BOSS1) {
            /* Draw normal monster */
            ASSERT(sprite);
            
            if ((m->hitc&3)) {
               rspr->render_flags = RENDER_LIT;
            } else {
               if (m->ability & MONSTER_ABILITY_ETHEREAL) {
                  rspr->render_flags = RENDER_TRANS;
               }
            }
         } else if (m->type==MONSTER_BOSS1 && (n%4==0)) {
            /* Draw boss - Serpentile */
            rspr->angle = itofix(64+m->angle*127/M_PI);
            rspr->render_flags = RENDER_ROTATE;
         } else {
            rspr->sprite = NULL;
         }
         
         if (rspr->sprite) {
            LINK *rl = alloc_link();
            ASSERT(rl);
            
            rl->next = rl->prev = NULL;
            rl->data = rspr;
            render_list = merge_list(rl, render_list, render_sort);

            #ifdef SHOW_BBOXES
            BSHAPE *bb = get_monster_move_bbox(m->type);
            rect(bbox, rspr->x-dx+bb->dx, 
                       rspr->y-dy+bb->dy,
                       rspr->x-dx+bb->dx+bb->w, 
                       rspr->y-dy+bb->dy+bb->h, blue);
            bb = get_monster_hit_bbox(m->type);
            rect(bbox, rspr->x-dx+bb->dx, 
                       rspr->y-dy+bb->dy,
                       rspr->x-dx+bb->dx+bb->w, 
                       rspr->y-dy+bb->dy+bb->h, red);
            #endif
         } else {
            free_render_sprite(rspr);
         }
      }
   }
}

static void do_draw_gamescreen(void)
{
   int arrowtype;
   int c;
   int x;
   LINK *l;
   LINK *ll;
   RENDER_SPRITE *rspr;
   
   destroy_list(render_list, (void *)free_render_sprite);
   render_list = NULL;

   #ifdef SHOW_BBOXES
   bbox = create_bitmap(SCREEN_W, SCREEN_H);
   clear_to_color(bbox, bitmap_mask_color(bbox));
   #endif

   if (redraw_tilemap) {
      get_connected_map(get_connect_info_bitmap(), hero.current_room, hero.x/(MAP_WIDTH*TILE_WIDTH/4), hero.y/(MAP_HEIGHT*TILE_HEIGHT/4));
      render_tilemap(get_tilemap(hero.current_room), get_left_doors(hero.current_room), get_right_doors(hero.current_room));
      redraw_tilemap = FALSE;
   }

   /* Don't scroll boss level */
   if (!is_boss_level) {
      top_x = clip_num(hero.x-SCREEN_W/2,0,MAP_WIDTH*TILE_WIDTH-SCREEN_W);
      top_y = clip_num(hero.y-SCREEN_H/2,0,MAP_HEIGHT*TILE_HEIGHT-SCREEN_H);
   } else {
      top_x = clip_num(hero.x-SCREEN_W/2,0,MAP_WIDTH*TILE_WIDTH-SCREEN_W);
      top_y = (17*TILE_HEIGHT - SCREEN_H)/2;
   }

   #ifdef SHOW_BBOXES
   for (l=get_geometry_boundingbox(hero.current_room); l; l = l->next) {
      BSHAPE *bb = l->data;
      rect(bbox, -top_x+bb->dx, 
                 -top_y+bb->dy,
                 -top_x+bb->dx+bb->w, 
                 -top_y+bb->dy+bb->h, yellow);
   }
   #endif

   rectfill(get_screen_bmp(), 0, 8, SCREEN_W, 8+16, black);
   blit_backbuffer(get_screen_bmp(), 0,0, top_x,top_y, SCREEN_W,SCREEN_H);

   /* Draw items - lower*/
   for (l=get_item_list(hero.current_room); l; l=l->next) {
      ITEM *item = l->data;
      if ((!item->lifetime || (item->lifetime>40) || (item->lifetime&2)) &&
          tile_is_visible(item->x/TILE_WIDTH, item->y/TILE_HEIGHT)) {
         rspr = alloc_render_sprite();
         ASSERT(rspr);
         
         if (item->type == ITEM_ARROW && hero.inventory&ITEM_SARROW)
            rspr->sprite = get_item_sprite(get_item_gfx_index(ITEM_SARROW));
         else
            rspr->sprite = get_item_sprite(get_item_gfx_index(item->type));
         rspr->render_flags = RENDER_NORMAL;
         rspr->sort_flags = -1;
         rspr->x = item->x-top_x;
         rspr->y = item->y-top_y;

         switch (item->type) {
            case ITEM_CHESTOPEN:
            case ITEM_CHESTCLOSED:
            case ITEM_BLOCK:
            case ITEM_STATUE:
            case ITEM_BLOCKCRACK:
            case ITEM_BLOCKLOCKED:
            case ITEM_BOSSGATE:
            case ITEM_BOSSEXIT:
            case ITEM_EXIT:
               rspr->sort_flags = -200;
               break;
            default:
               break;
         }
         #ifdef SHOW_BBOXES
         {
            BSHAPE *bb = get_item_bbox(item->type);
            rect(bbox, rspr->x+bb->dx, 
                       rspr->y+bb->dy,
                       rspr->x+bb->dx+bb->w, 
                       rspr->y+bb->dy+bb->h, green);
         }
         #endif
         if (rspr->sprite) {
            ll = alloc_link();
            ASSERT(ll);
            
            ll->next = ll->prev = NULL;
            ll->data = rspr;
            render_list = merge_list(ll, render_list, render_sort);
         } else {
            free_render_sprite(rspr);
         }
      }
   }

   /* Draw projectiles (that appear below other sprites) */
   /* Draw them centered */
   for (l=get_projectile_list(hero.current_room); l; l=l->next) {
      PROJECTILE *projectile = l->data;

      rspr = alloc_render_sprite();
      ASSERT(rspr);
      rspr->sort_flags = 1;
      rspr->sprite = NULL;

      switch (projectile->type) {
         case PROJECTILE_ARROW:
            rspr->sprite = get_arrow_sprite(0);
            rspr->render_flags = RENDER_ROTATE;
            rspr->x = projectile->x-top_x+8;
            rspr->y = projectile->y-top_y+8;
            rspr->angle = itofix(64+projectile->angle*127/M_PI);
            #ifdef SHOW_BBOXES
            {
               BSHAPE *bb = &arrow_bb;
               rect(bbox, projectile->x-top_x+8+bb->dx, 
                          projectile->y-top_y+8+bb->dy,
                          projectile->x-top_x+8+bb->dx+bb->w, 
                          projectile->y-top_y+8+bb->dy+bb->h, green);
            }
            #endif
            break;
         case PROJECTILE_SWORD:
            /*
            if (projectile->angle<0 || projectile->angle>3*M_PI/2) {
               rspr->sort_flags = -1;
            }
            rspr->sprite = get_swordpr_sprite(0);
            rspr->render_flags = RENDER_ROTATE;
            rspr->x = projectile->x-top_x+8;
            rspr->y = projectile->y-top_y+8;
            rspr->angle = itofix(64+projectile->angle*127/M_PI);
            */
            rspr->sprite = NULL;
            #ifdef SHOW_BBOXES
            {
               BSHAPE *bb = &melee_bb;
               rect(bbox, projectile->x-top_x+8+bb->dx, 
                          projectile->y-top_y+8+bb->dy,
                          projectile->x-top_x+8+bb->dx+bb->w, 
                          projectile->y-top_y+8+bb->dy+bb->h, green);
            }
            #endif
            break;
         case PROJECTILE_DAGGER:
            /*
            if (projectile->angle<0 || projectile->angle>3*M_PI/2) {
               rspr->sort_flags = -1;
            }
            rspr->sprite = get_dagger1pr_sprite(0);
            rspr->render_flags = RENDER_ROTATE;
            rspr->x = projectile->x-top_x+8;
            rspr->y = projectile->y-top_y+8;
            rspr->angle = itofix(64+projectile->angle*127/M_PI);
            */
            rspr->sprite = NULL;
            #ifdef SHOW_BBOXES
            {
               BSHAPE *bb = &melee_bb;
               rect(bbox, projectile->x-top_x+8+bb->dx, 
                          projectile->y-top_y+8+bb->dy,
                          projectile->x-top_x+8+bb->dx+bb->w, 
                          projectile->y-top_y+8+bb->dy+bb->h, green);
            }
            #endif
            break;
         case PROJECTILE_CLUB:
            /*
            if (projectile->angle<0 || projectile->angle>3*M_PI/2) {
               rspr->sort_flags = -1;
            }
            rspr->sprite = get_club_sprite(0);
            rspr->sprite = NULL;
            rspr->render_flags = RENDER_ROTATE;
            rspr->x = projectile->x-top_x+8;
            rspr->y = projectile->y-top_y+8;
            rspr->angle = itofix(64+projectile->angle*127/M_PI);
            */
            #ifdef SHOW_BBOXES
            {
               BSHAPE *bb = &melee_bb;
               rect(bbox, projectile->x-top_x+8+bb->dx, 
                          projectile->y-top_y+8+bb->dy,
                          projectile->x-top_x+8+bb->dx+bb->w, 
                          projectile->y-top_y+8+bb->dy+bb->h, green);
            }
            #endif
            rspr->sprite = NULL;
            break;
         case PROJECTILE_BOMB:
            rspr->sprite = get_bomb_sprite(projectile->anim/6);
            rspr->render_flags = RENDER_NORMAL;
            rspr->x = projectile->x-top_x;
            rspr->y = projectile->y-top_y;
            break;

         case PROJECTILE_E_FIREBALL:
            rspr->sprite = get_fireball_sprite(projectile->anim/2);
            rspr->render_flags = RENDER_ROTATE;
            rspr->x = projectile->x-top_x+16;
            rspr->y = projectile->y-top_y+16;
            rspr->angle = itofix(projectile->anim*8);
            //game_rotate_sprite(get_screen_bmp(), get_fireball_sprite(projectile->anim/2), projectile->x-top_x+16, projectile->y-top_y+16, itofix(projectile->anim*8));
            break;
         case PROJECTILE_EXPLOSION:
            rspr->sprite = get_explosion_sprite(projectile->anim/5);
            rspr->render_flags = RENDER_NORMAL;
            rspr->x = projectile->x-50-top_x+16;
            rspr->y = projectile->y-50-top_y+16;
            rspr->sort_flags = 100;
            //game_draw_sprite(get_screen_bmp(), get_explosion_sprite(projectile->anim/5), projectile->x-50-top_x+16, projectile->y-50-top_y+16);
            break;
      }


      if (rspr->sprite) {
         ll = alloc_link();
         ASSERT(ll);

         ll->next = ll->prev = NULL;
         ll->data = rspr;
         render_list = merge_list(ll, render_list, render_sort);
      } else {
         free_render_sprite(rspr);
      }
   }

   /* Draw particles (that appear below other sprites) */
   /* Draw them centered */
   for (l=get_particle_list(hero.current_room); l; l=l->next) {
      PARTICLE *particle = l->data;

      rspr = alloc_render_sprite();
      ASSERT(rspr);

      rspr->sort_flags = 0;
      rspr->render_flags = RENDER_NORMAL;
      rspr->sprite = NULL;
      rspr->x = hero.x-top_x;
      rspr->y = hero.y-16-top_y;

      switch (particle->type) {
         case PROJECTILE_ITEM:
            rspr->render_flags = RENDER_CENTRE;
            rspr->sprite = get_item_sprite(get_item_gfx_index(particle->sub_type));
            rspr->x = particle->x-top_x+8;
            rspr->y = particle->y-top_y+8;
            break;
         case PROJECTILE_DEATH:
            rspr->sprite = get_death_sprite(particle->anim/2);
            rspr->x = particle->x-top_x;
            rspr->y = particle->y-top_y;
            break;
      }

      if (rspr->sprite) {
         ll = alloc_link();
         ASSERT(ll);
            
         ll->next = ll->prev = NULL;
         ll->data = rspr;
         render_list = merge_list(ll, render_list, render_sort);
      } else {
         free_render_sprite(rspr);
      }
   }

   draw_monsters();

   if (hero.health>0 && !(hero.invc&1)) {
      rspr = alloc_render_sprite();
      ASSERT(rspr);

      rspr->sprite = get_hero_sprite(hero.direction, 0, hero.anim);
      rspr->render_flags = RENDER_NORMAL;
      rspr->sort_flags = 0;
      rspr->x = hero.x-top_x;
      rspr->y = hero.y-16-top_y;

      #ifdef SHOW_BBOXES
      rect(bbox, hero.x-top_x+player_bb.dx, 
                 hero.y-top_y+player_bb.dy,
                 hero.x-top_x+player_bb.dx+player_bb.w, 
                 hero.y-top_y+player_bb.dy+player_bb.h, blue);
      rect(bbox, hero.x-top_x+hit_bb.dx, 
                 hero.y-top_y+hit_bb.dy, 
                 hero.x-top_x+hit_bb.dx+hit_bb.w, 
                 hero.y-top_y+hit_bb.dy+hit_bb.h, red);
      rect(bbox, hero.x-top_x+pickup_bb.dx, 
                 hero.y-top_y+pickup_bb.dy, 
                 hero.x-top_x+pickup_bb.dx+pickup_bb.w, 
                 hero.y-top_y+pickup_bb.dy+pickup_bb.h, green);
      #endif
      l = alloc_link();
      ASSERT(l);
            
      l->next = l->prev = NULL;
      l->data = rspr;
      render_list = merge_list(l, render_list, render_sort);

      /* Draw the shield */
      if (hero.inventory & ITEM_SHIELD) {
         rspr = alloc_render_sprite();
         ASSERT(rspr);

         rspr->sprite = get_shield_sprite(hero.direction, 0, hero.anim);
         rspr->render_flags = RENDER_NORMAL;
         rspr->sort_flags = 1;
         rspr->x = hero.x-top_x;
         rspr->y = hero.y-16-top_y;

         l = alloc_link();
         ASSERT(l);
            
         l->next = l->prev = NULL;
         l->data = rspr;
         render_list = merge_list(l, render_list, render_sort);
      }

      /* Draw the weapon the player is carrying */
      if (hero.weapon & (ITEM_SWORD | ITEM_SWORD2)) {
         rspr = alloc_render_sprite();
         ASSERT(rspr);

         rspr->render_flags = RENDER_NORMAL;
         if (hero.direction == 3 || hero.direction == 2) {
            rspr->sort_flags = 0;
         } else {
            rspr->sort_flags = 2;
         }
         
         if (hero.action&ACTION_PARRY) {
            rspr->sprite = get_sword2_sprite(hero.direction, 0, hero.anim);
         } else {
            rspr->sprite = get_sword_sprite(hero.direction, 0, hero.anim);
         }
         rspr->x = hero.x-top_x;
         rspr->y = hero.y-16-top_y;

         l = alloc_link();
         ASSERT(l);
            
         l->next = l->prev = NULL;
         l->data = rspr;
         render_list = merge_list(l, render_list, render_sort);
      } else if (hero.weapon & (ITEM_CLUB | ITEM_MACE)) {
         rspr = alloc_render_sprite();
         ASSERT(rspr);

         rspr->render_flags = RENDER_NORMAL;
         if (hero.direction == 3) {
            rspr->sort_flags = 0;
         } else {
            rspr->sort_flags = 2;
         }
         
         if (hero.action&ACTION_TARGET) {
            rspr->sprite = get_club2_sprite(hero.direction, 0, hero.anim);
         } else {
            rspr->sprite = get_club_sprite(hero.direction, 0, hero.anim);
         }
         rspr->x = hero.x-top_x;
         rspr->y = hero.y-16-top_y;

         l = alloc_link();
         ASSERT(l);
            
         l->next = l->prev = NULL;
         l->data = rspr;
         render_list = merge_list(l, render_list, render_sort);
      } else if (hero.weapon & (ITEM_DAGGER | ITEM_DAGGER2)) {
         rspr = alloc_render_sprite();
         ASSERT(rspr);

         rspr->render_flags = RENDER_NORMAL;
         if (hero.direction == 3) {
            rspr->sort_flags = 0;
         } else {
            rspr->sort_flags = 2;
         }
         
         if (hero.action&ACTION_TARGET) {
            rspr->sprite = get_dagger2_sprite(hero.direction, 0, hero.anim);
         } else {
            rspr->sprite = get_dagger_sprite(hero.direction, 0, hero.anim);
         }
         rspr->x = hero.x-top_x;
         rspr->y = hero.y-16-top_y;

         l = alloc_link();
         ASSERT(l);
            
         l->next = l->prev = NULL;
         l->data = rspr;
         render_list = merge_list(l, render_list, render_sort);
      } 

   }

   for (l=render_list; l; l=l->next) {
      rspr = l->data;
      game_render_sprite(get_screen_bmp(), rspr);
   }
   destroy_list(render_list, (void *)free_render_sprite);
   render_list = NULL;


   /* Draw particles (that appear above other sprites) */
   for (l=get_particle_list(hero.current_room); l; l=l->next) {
      PARTICLE *particle = l->data;
      switch (particle->type) {
         case PROJECTILE_BOSS_DEATH:
            game_draw_sprite(get_screen_bmp(), get_explosion_sprite(particle->anim/5), particle->x-50-top_x+16, particle->y-50-top_y+16);
            break;
      }
   }

   if (hero.action & ACTION_TARGET && hero.weapon & (ITEM_BOW|ITEM_CROSSBOW)) {
      game_draw_sprite(get_screen_bmp(), get_target_sprite(global_cycles_passed/10), hero.x + hero.target_x-top_x, hero.y + hero.target_y-top_y);
   }

   /* Draw items - upper */
   for (l=get_item_list(hero.current_room); l; l=l->next) {
      ITEM *item = l->data;
      if (tile_is_visible(item->x/TILE_WIDTH, item->y/TILE_HEIGHT)) {
         switch (item->type) {
            case ITEM_BOSSGATE:
               c = get_item_gfx_index(ITEM_BOSSGATETOP);
               game_draw_sprite(get_screen_bmp(), get_item_sprite(c), item->x-top_x, item->y-top_y);
               break;
            default:
               break;
         }
      }
   }

   /* Draw status bar */
   clear_to_color(statusbar, bitmap_mask_color(statusbar));
   textprintf_centre(statusbar, get_menu_bold_font(),
                     SCREEN_W-(10*10+80)/3, 0, white, "life");
   for(c=0; c<hero.max_health/8; c++) {
      if (hero.health>=c*8) {
         game_draw_sprite(statusbar, get_heart_item_sprite(0), SCREEN_W-10*10+(c%10)*8, 12+(c/10)*10);
      } else {
         game_draw_sprite(statusbar, get_heart_item_sprite(1), SCREEN_W-10*10+(c%10)*8, 12+(c/10)*10);
      }
   }
   if (hero.health>0 && hero.health&7) {
      game_draw_sprite(statusbar, get_heart_item_sprite(2), SCREEN_W-10*10+((hero.health/8)%10)*8, 12+((hero.health/8)/10)*10);
   }
   
   /* Weapon cooldown metre */
   if (hero.weapon_cooldown == 0)
      c = green;
   else if (hero.weapon_cooldown < hero.max_cooldown/2)
      c = yellow;
   else
      c = red;

   x = 0;
   if (hero.weapon_cooldown != 0)
      x = 78 - 78 * (hero.max_cooldown - hero.weapon_cooldown) / (hero.max_cooldown+1);
   rect(statusbar, SCREEN_W-10*10, 30, SCREEN_W-2*10, 32, black);
   hline(statusbar, SCREEN_W-10*10+1, 31, SCREEN_W-2*10-1-x, c);

   if (hero.weapon_buildup != 0 && hero.weapon & (ITEM_MACE|ITEM_CLUB)) {
      x = 78 * (hero.max_cooldown - hero.weapon_buildup/2) / (hero.max_cooldown+1);
      hline(statusbar, SCREEN_W-10*10+1, 31, SCREEN_W-2*10-1-x, white);
   }

   arrowtype = get_item_gfx_index((hero.inventory&ITEM_SARROW) ? ITEM_SARROW : ITEM_ARROW);
   
   x = SCREEN_W-13*10;
   game_draw_sprite(statusbar, get_key_item_sprite(), x, 8);
   textprintf_centre(statusbar, get_menu_bold_font(), x+9, 12, white, "%03d", hero.keys);
   x -= 30;
   game_draw_sprite(statusbar, get_item_sprite(arrowtype), x, 4);
   textprintf_centre(statusbar, get_menu_bold_font(), x+9, 12, white, "%03d", hero.arrows);
   x -= 30;
   game_draw_sprite(statusbar, get_bomb_item_sprite(), x, 4);
   textprintf_centre(statusbar, get_menu_bold_font(), x+9, 12, white, "%03d", hero.bombs);

   /* Active weapon */
   x = SCREEN_W-22*10;
   rectfill(statusbar, SCREEN_W-22*10, 8, x+16, 8+16, black);
   switch (hero.weapon) {
      case ITEM_BOW:
         game_draw_sprite(statusbar, get_bow_item_sprite(hero.arrows), x, 8);

         if (hero.arrows) {
            game_draw_sprite(statusbar, get_item_sprite(arrowtype), x, 8);
         }
         break;
      case ITEM_CROSSBOW:
         game_draw_sprite(statusbar, get_bow_item_sprite(hero.arrows), x, 8);

         if (hero.arrows) {
            game_draw_sprite(statusbar, get_item_sprite(arrowtype), x, 8);
         }
         break;
      default:
         if (hero.weapon) {
            game_draw_sprite(statusbar, get_item_sprite(get_item_gfx_index(hero.weapon)), x, 8);
         }
   }
   rect(statusbar, x, 8, x+16, 8+16, makecol(255,206,65));

   x = SCREEN_W-24*10;
   textprintf_right(statusbar, get_menu_bold_font(), x, 12, white, "%d", hero.score);

   masked_blit(statusbar, get_screen_bmp(), 0,0,0,0,SCREEN_W,TILE_SIZE);
   mark_rect(0, 0, SCREEN_W, SCREEN_H);

   #ifdef SHOW_BBOXES
   masked_blit(bbox, get_screen_bmp(), 0,0,0,0,SCREEN_W,SCREEN_H);
   destroy_bitmap(bbox);
   #endif
}

inline void game_gfx_update(void)
{
   PALETTE p;
   int c;
   int x, y;
   int dmg;

   TRACE("Starting graphical update [game_state = %d, global_cycle = %d]\n\t", game_state, global_cycles_passed);
   prepare_screen_update();

   switch (game_state) {
      case GAME_STATE_GET_HIGHSCORE:
         clear_bitmap(get_screen_bmp());

         shift_palette_hue((*get_font_palette()), p, 35.00f);
         select_palette(p);

         textprintf_centre(get_screen_bmp(), get_title_font(), SCREEN_W/2, 64, white, "Highscore!");
         textprintf_centre(get_screen_bmp(), get_colour_font(), SCREEN_W/2, SCREEN_H/2+20, -1, "Your score has reached the highscore list!");
         mark_rect(0, 0, SCREEN_W, SCREEN_H);
         break;
      case GAME_STATE_SHOW_SCORES:
         clear_bitmap(get_screen_bmp());
         select_palette((*get_font_palette()));
         show_highscore_list(get_screen_bmp(), get_colour_font(), -1);
         mark_rect(0, 0, SCREEN_W, SCREEN_H);
         break;
      case GAME_STATE_NORMAL:
         do_draw_gamescreen();
         break;
      case GAME_STATE_NORMAL|GAME_STATE_PAUSEGAME|GAME_STATE_EXITMENU:

         x = SCREEN_W/2-160;
         y = SCREEN_H/2-120;
         masked_blit(get_alt_dframe_bitmap(), get_screen_bmp(), 0,0, x, y, 320,240);

         /* Display fancy `Game Paused' text */
         shift_palette_hue((*get_font_palette()), p, 2*(global_cycles_passed%180));
         select_palette(p);
         textprintf_centre(get_screen_bmp(), get_password_font(), SCREEN_W/2, SCREEN_H/2-text_height(get_title_font()), -1, "Game Paused");

         /* Display options: resume, quit */
         shift_palette_hue((*get_font_palette()), p, 45.00f);
         select_palette(game_state_options?(*get_font_palette()):p);
         textprintf_centre(get_screen_bmp(), get_colour_font(), x+96, y+160, -1, "Resume");
         select_palette(game_state_options?p:(*get_font_palette()));
         textprintf_centre(get_screen_bmp(), get_colour_font(), x+320-96, y+160, -1, "Quit");

         mark_rect(0, 0, SCREEN_W, SCREEN_H);
         break;
      case GAME_STATE_NORMAL|GAME_STATE_PAUSEGAME|GAME_STATE_ITEMMENU:
         x = SCREEN_W/2-160;
         y = SCREEN_H/2-120;
         masked_blit(get_alt_dframe_bitmap(), get_screen_bmp(), 0,0, x, y, 320,240);

         /* Draw inventory */
         if(game_state_options>=0)
            c = game_state_options%24;
         else
            c = (24+game_state_options)%24;

         if (hero.inventory&(ITEM_BOW|ITEM_CROSSBOW)) {
            int arrowtype = get_item_gfx_index((hero.inventory&ITEM_SARROW) ? ITEM_SARROW : ITEM_ARROW);

            game_centre_sprite(get_screen_bmp(), get_item_sprite(get_item_gfx_index(hero.inventory&(ITEM_CROSSBOW)?ITEM_CROSSBOW:ITEM_BOW)-(hero.arrows>0)),
                             x+TILE_SIZE+TILE_SIZE/2, y+TILE_SIZE+TILE_SIZE/2);
            if (hero.arrows) {
               game_centre_sprite(get_screen_bmp(), get_item_sprite(arrowtype), x+TILE_SIZE+TILE_SIZE/2, y+TILE_SIZE+TILE_SIZE/2);
            }
            if (c==0) {
               dmg = calc_weapon_damage(hero.inventory, hero.inventory&(ITEM_BOW|ITEM_CROSSBOW));

               textprintf (get_screen_bmp(), get_menu_bold_font(), 
                              x+TILE_SIZE, y+240-64, white, "%s%s%s%s",
                              hero.inventory&ITEM_CROSSBOW?"Crossbow":"Bow", 
                              hero.arrows?" & ":"",
                              (hero.arrows && hero.inventory&ITEM_SARROW)?"steel-tipped ":"",
                              hero.arrows?"arrows":"");
               /* Show damage output (hardcoded for now) */
               textprintf (get_screen_bmp(), get_menu_font(), x+TILE_SIZE, y+240-64+16, white,
               get_item_desc(hero.inventory&(ITEM_BOW|ITEM_CROSSBOW)), 
               (dmg >> 24)&0x0000000F, (dmg >> 28)&0x0000000F,
               (dmg >> 16)&0x0000000F, (dmg >> 20)&0x0000000F,
               (dmg >>  8)&0x0000000F, (dmg >> 12)&0x0000000F,
               (dmg >>  0)&0x0000000F, (dmg >>  4)&0x0000000F);
            }
         }
         if (hero.inventory&(ITEM_SWORD|ITEM_SWORD2)) {
            dmg = calc_weapon_damage(hero.inventory, hero.inventory&(ITEM_SWORD|ITEM_SWORD2));
            
            game_centre_sprite(get_screen_bmp(), get_item_sprite(get_item_gfx_index(hero.inventory&(ITEM_SWORD2)?ITEM_SWORD2:ITEM_SWORD)),
                             x+2*TILE_SIZE+TILE_SIZE/2, y+TILE_SIZE+TILE_SIZE/2);
            if (c==1) {
               textprintf (get_screen_bmp(), get_menu_bold_font(), x+TILE_SIZE, y+240-64, white, "Sword");
               /* Show damage output (hardcoded for now) */
               textprintf (get_screen_bmp(), get_menu_font(), x+TILE_SIZE, y+240-64+16, white,
               get_item_desc(hero.inventory&(ITEM_SWORD|ITEM_SWORD2)), 
               (dmg >> 24)&0x0000000F, (dmg >> 28)&0x0000000F,
               (dmg >> 16)&0x0000000F, (dmg >> 20)&0x0000000F,
               (dmg >>  8)&0x0000000F, (dmg >> 12)&0x0000000F,
               (dmg >>  0)&0x0000000F, (dmg >>  4)&0x0000000F);
            }
         }
         if (hero.inventory&(ITEM_DAGGER|ITEM_DAGGER2)) {
            dmg = calc_weapon_damage(hero.inventory, hero.inventory&(ITEM_DAGGER|ITEM_DAGGER2));
            
            game_centre_sprite(get_screen_bmp(), get_item_sprite(get_item_gfx_index(hero.inventory&(ITEM_DAGGER2)?ITEM_DAGGER2:ITEM_DAGGER)),
                             x+3*TILE_SIZE+TILE_SIZE/2, y+TILE_SIZE+TILE_SIZE/2);
            if (c==2) {
               textprintf (get_screen_bmp(), get_menu_bold_font(), x+TILE_SIZE, y+240-64, white, "Dagger");
               /* Show damage output (hardcoded for now) */
               textprintf (get_screen_bmp(), get_menu_font(), x+TILE_SIZE, y+240-64+16, white,
               get_item_desc(hero.inventory&(ITEM_DAGGER|ITEM_DAGGER2)), 
               (dmg >> 24)&0x0000000F, (dmg >> 28)&0x0000000F,
               (dmg >> 16)&0x0000000F, (dmg >> 20)&0x0000000F,
               (dmg >>  8)&0x0000000F, (dmg >> 12)&0x0000000F,
               (dmg >>  0)&0x0000000F, (dmg >>  4)&0x0000000F);
            }
         }
         if (hero.inventory&(ITEM_CLUB|ITEM_MACE)) {
            dmg = calc_weapon_damage(hero.inventory, hero.inventory&(ITEM_CLUB|ITEM_MACE));
            
            game_centre_sprite(get_screen_bmp(), get_item_sprite(get_item_gfx_index(hero.inventory&(ITEM_MACE)?ITEM_MACE:ITEM_CLUB)),
                             x+4*TILE_SIZE+TILE_SIZE/2, y+TILE_SIZE+TILE_SIZE/2);
            if (c==3) {
               textprintf (get_screen_bmp(), get_menu_bold_font(), x+TILE_SIZE, y+240-64, white, "%s", hero.inventory&ITEM_MACE?"Iron mace":"Wooden club");
               /* Show damage output (hardcoded for now) */
               textprintf (get_screen_bmp(), get_menu_font(), x+TILE_SIZE, y+240-64+16, white,
               get_item_desc(hero.inventory&(ITEM_DAGGER|ITEM_MACE)),
               (dmg >> 24)&0x0000000F, (dmg >> 28)&0x0000000F,
               (dmg >> 16)&0x0000000F, (dmg >> 20)&0x0000000F,
               (dmg >>  8)&0x0000000F, (dmg >> 12)&0x0000000F,
               (dmg >>  0)&0x0000000F, (dmg >>  4)&0x0000000F);
            }
         }
         if (hero.inventory&(ITEM_SHIELD)) {
            game_centre_sprite(get_screen_bmp(), get_item_sprite(get_item_gfx_index(ITEM_SHIELD)),
                             x+TILE_SIZE+TILE_SIZE/2, y+2*TILE_SIZE+TILE_SIZE/2);
            if (c==6) {
               textprintf (get_screen_bmp(), get_menu_bold_font(), x+TILE_SIZE, y+240-64, white, get_item_name(ITEM_SHIELD));
               textprintf (get_screen_bmp(), get_menu_font(), x+TILE_SIZE, y+240-64+16, white,
               get_item_desc(hero.inventory&ITEM_SHIELD));
            }
         }
         if (hero.inventory&(ITEM_CHAINMAIL)) {
            game_centre_sprite(get_screen_bmp(), get_item_sprite(get_item_gfx_index(ITEM_CHAINMAIL)),
                             x+2*TILE_SIZE+TILE_SIZE/2, y+2*TILE_SIZE+TILE_SIZE/2);
            if (c==7) {
               textprintf (get_screen_bmp(), get_menu_bold_font(), x+TILE_SIZE, y+240-64, white, get_item_name(ITEM_CHAINMAIL));
               textprintf (get_screen_bmp(), get_menu_font(), x+TILE_SIZE, y+240-64+16, white,
               get_item_desc(hero.inventory&ITEM_CHAINMAIL));
            }
         }
         if (hero.inventory&(ITEM_HELMET)) {
            game_centre_sprite(get_screen_bmp(), get_item_sprite(get_item_gfx_index(ITEM_HELMET)),
                             x+3*TILE_SIZE+TILE_SIZE/2, y+2*TILE_SIZE+TILE_SIZE/2);
            if (c==8) {
               textprintf (get_screen_bmp(), get_menu_bold_font(), x+TILE_SIZE, y+240-64, white, get_item_name(ITEM_HELMET));
               textprintf (get_screen_bmp(), get_menu_font(), x+TILE_SIZE, y+240-64+16, white,
               get_item_desc(hero.inventory&ITEM_HELMET));
            }
         }
         if (hero.inventory&(ITEM_RINGREGEN)) {
            game_centre_sprite(get_screen_bmp(), get_item_sprite(get_item_gfx_index(ITEM_RINGREGEN)),
                             x+5*TILE_SIZE+TILE_SIZE/2, y+TILE_SIZE+3*TILE_SIZE+TILE_SIZE/2);
            if (c==22) {
               textprintf (get_screen_bmp(), get_menu_bold_font(), x+TILE_SIZE, y+240-64, white, get_item_name(ITEM_RINGREGEN));
               textprintf (get_screen_bmp(), get_menu_font(), x+TILE_SIZE, y+240-64+16, white,
               get_item_desc(hero.inventory&ITEM_RINGREGEN));
            }
         }
         if (hero.inventory&(ITEM_TELLRING)) {
            game_centre_sprite(get_screen_bmp(), get_item_sprite(get_item_gfx_index(ITEM_TELLRING)),
                             x+6*TILE_SIZE+TILE_SIZE/2, y+TILE_SIZE+3*TILE_SIZE+TILE_SIZE/2);
            if (c==23) {
               textprintf (get_screen_bmp(), get_menu_bold_font(), x+TILE_SIZE, y+240-64, white, get_item_name(ITEM_TELLRING));
               textprintf (get_screen_bmp(), get_menu_font(), x+TILE_SIZE, y+240-64+16, white,
               get_item_desc(hero.inventory&ITEM_TELLRING));
            }
         }
         if (hero.inventory&(ITEM_OWLFIG)) {
            game_centre_sprite(get_screen_bmp(), get_item_sprite(get_item_gfx_index(ITEM_OWLFIG)),
                             x+TILE_SIZE+TILE_SIZE/2, y+3*TILE_SIZE+TILE_SIZE/2);
            if (c==12) {
               textprintf (get_screen_bmp(), get_menu_bold_font(), x+TILE_SIZE, y+240-64, white, get_item_name(ITEM_OWLFIG));
               textprintf (get_screen_bmp(), get_menu_font(), x+TILE_SIZE, y+240-64+16, white,
               get_item_desc(hero.inventory&ITEM_OWLFIG));
            }
         }
         if (hero.inventory&(ITEM_CATFIG)) {
            game_centre_sprite(get_screen_bmp(), get_item_sprite(get_item_gfx_index(ITEM_CATFIG)),
                             x+2*TILE_SIZE+TILE_SIZE/2, y+3*TILE_SIZE+TILE_SIZE/2);
            if (c==13) {
               textprintf (get_screen_bmp(), get_menu_bold_font(), x+TILE_SIZE, y+240-64, white, get_item_name(ITEM_CATFIG));
               textprintf (get_screen_bmp(), get_menu_font(), x+TILE_SIZE, y+240-64+16, white,
               get_item_desc(hero.inventory&ITEM_CATFIG));
            }
         }
         if (hero.inventory&(ITEM_SNAKEFIG)) {
            game_centre_sprite(get_screen_bmp(), get_item_sprite(get_item_gfx_index(ITEM_SNAKEFIG)),
                             x+3*TILE_SIZE+TILE_SIZE/2, y+3*TILE_SIZE+TILE_SIZE/2);
            if (c==14) {
               textprintf (get_screen_bmp(), get_menu_bold_font(), x+TILE_SIZE, y+240-64, white, get_item_name(ITEM_SNAKEFIG));
               textprintf (get_screen_bmp(), get_menu_font(), x+TILE_SIZE, y+240-64+16, white,
               get_item_desc(hero.inventory&ITEM_SNAKEFIG));
            }
         }
         masked_blit(get_cursor_bitmap(), get_screen_bmp(), 0,0, x+TILE_SIZE+TILE_SIZE*(c%6), y+TILE_SIZE+(c/6)*TILE_SIZE, 320,240);

         /* Draw heart status */
         textprintf_centre (get_screen_bmp(), get_menu_bold_font(), x+320-80+16, y+TILE_SIZE, white, "Heart" );
         textprintf_centre (get_screen_bmp(), get_menu_bold_font(), x+320-80+16, y+44, white, "status" );
         game_draw_sprite(get_screen_bmp(), get_heart_container_item_sprite((hero.max_health&7)/2), x+320-80, y+54);
         mark_rect(0, 0, SCREEN_W, SCREEN_H);
         break;
      case GAME_STATE_GAME_OVER:
         if (game_state_counter) {
            do_draw_gamescreen();
         } else {
            clear_bitmap(get_screen_bmp());
            textprintf_centre(get_screen_bmp(), get_title_font(), SCREEN_W/2, SCREEN_H/2-text_height(get_title_font()), white, "Game Over");
            textprintf_right(get_screen_bmp(), get_menu_bold_font(), SCREEN_W*2/3, SCREEN_H*3/4, white, "Score: %d", hero.score);
            mark_rect(0, 0, SCREEN_W, SCREEN_H);
         }
         break;
      case GAME_STATE_PASSWORD:
         clear_bitmap(get_screen_bmp());
         select_palette(*get_font_palette());
         textprintf_centre(get_screen_bmp(), get_alt_colour_font(), SCREEN_W/2, SCREEN_H/2-text_height(get_title_font()), -1, "Password:");
         textprintf_centre(get_screen_bmp(), get_password_font(), SCREEN_W/2, SCREEN_H/2, -1, "%s", password);
         unselect_palette();
         mark_rect(0, 0, SCREEN_W, SCREEN_H);
         break;
      case GAME_STATE_NEW_LEVEL:
         clear_bitmap(get_screen_bmp());
         textprintf_centre(get_screen_bmp(), get_title_font(), SCREEN_W/2, SCREEN_H/2-text_height(get_title_font()), white, "Entering");
         if (settings.difficulty == 0)
            textprintf_centre(get_screen_bmp(), get_title_font(), SCREEN_W/2, SCREEN_H/2, white, "level %c", 'A'+hero.level);
         else
            textprintf_centre(get_screen_bmp(), get_title_font(), SCREEN_W/2, SCREEN_H/2, white, "%s level %c", settings.difficulty == 1?"Hard":"Hell", 'A'+hero.level);
         mark_rect(0, 0, SCREEN_W, SCREEN_H);
         break;
      case GAME_STATE_WIN:
         clear_bitmap(get_screen_bmp());

         shift_palette_hue((*get_font_palette()), p, 35.00f);
         select_palette(p);

         textprintf_centre(get_screen_bmp(), get_title_font(), SCREEN_W/2, SCREEN_H/2-text_height(get_title_font()), white, "Game Completed");
         textprintf_centre(get_screen_bmp(), get_colour_font(), SCREEN_W/2, SCREEN_H/2+20, -1, "Congratulations on completing this demo");
         textprintf_centre(get_screen_bmp(), get_colour_font(), SCREEN_W/2, SCREEN_H/2+50, -1, "version of 'Dungeon of Darkness' and");
         textprintf_centre(get_screen_bmp(), get_colour_font(), SCREEN_W/2, SCREEN_H/2+80, -1, "thank you for playing!");

         textprintf_centre(get_screen_bmp(), get_colour_font(), SCREEN_W/2, SCREEN_H/2+120, -1, "Please send me feedback and bugreports");
         textprintf_centre(get_screen_bmp(), get_colour_font(), SCREEN_W/2, SCREEN_H/2+140, -1, "so that I can improve this game!");
         textprintf_centre(get_screen_bmp(), get_colour_font(), SCREEN_W/2, SCREEN_H/2+160, -1, "See the README file for contact info.");
         mark_rect(0, 0, SCREEN_W, SCREEN_H);
         break;
   }

   TRACE("\t");
   screen_update_done();
   TRACE("Graphic update cycle completed\n");
}

inline void game_gfx_skip(void)
{
}

/* Returns TRUE if the item is touched by a creature standing at (x, y) */
static int touch_item(int x, int y, ITEM *item)
{
   //return rect_overlap(x, y+pickup_bb.dy, 8, 8, item->x, item->y, TILE_SIZE, TILE_SIZE);
   return collide_box(x, y, &pickup_bb, item->x, item->y, get_item_bbox(item->type));
}

/* True if hero is touching an item (doesn't mean he can actually pick it up!)*/
static int hero_touches_item(HERO *player, ITEM *item)
{
   //return touch_item(player->x+player->dx+12, player->y+player->dy, item);
   return touch_item(player->x+player->dx, player->y+player->dy, item);
}

/* True if a monster is touching an item */
static int monster_touches_item(MONSTER *monster, ITEM *item)
{
   return touch_item(monster->x+monster->dx, monster->y+monster->dy, item);

   //return touch_item(monster->x+monster->dx+12, monster->y+monster->dy+16, item);
}

/* Similar to tile_passable but logically reversed and for items */
/* Returns TRUE if any item (say, a block) hinders movement to (x, y) */
static int blocked_by_item(int room, int x, int y)
{
   LINK *l;

   l = get_item_list(room);
   while (l) {
      ITEM *item = l->data;
      if (item_blocks(item->type) && touch_item(x, y, item)) 
         return TRUE;
      l = l->next;
   }
   
   return FALSE;
}

/* True if the hero can open a chest */
static int hero_can_open_chest(HERO *player, ITEM *item)
{
   if (item->type != ITEM_CHESTCLOSED)
      return FALSE;
   if (player->y+player->dy > item->y && player->direction == 2)
      return TRUE;
   return FALSE;
}

/* Attempt to pickup/activate an item; returns TRUE on succes */
int could_pickup_item(HERO *player, int item_id)
{
   switch (item_id) {
      case ITEM_LOCKEDTELEP:
      case ITEM_TELEPORTER:
         return FALSE;
      case ITEM_CHESTOPEN:
      case ITEM_CHESTCLOSED:
      case ITEM_STATUE:
      case ITEM_BLOCK:
      case ITEM_BLOCKCRACK:
      case ITEM_BLOCKLOCKED:
         return FALSE;
      case ITEM_SARROW:
         player->inventory|=item_id;
      case ITEM_FARROW:
      case ITEM_ARROW:
         /* Award more arrows at higher difficulty */
         player->arrows+=15 + settings.difficulty*5;
         if (player->arrows>MAX_ARROWS)
            player->arrows = MAX_ARROWS;
         break;
      case ITEM_BOMB:
         player->bombs+=3 + settings.difficulty;
         if (player->bombs>MAX_BOMBS)
            player->bombs = MAX_BOMBS;
         break;
      case ITEM_BOW:
      case ITEM_CROSSBOW:
      case ITEM_CLUB:
      case ITEM_MACE:
      case ITEM_DAGGER:
      case ITEM_DAGGER2:
      case ITEM_SWORD:
      case ITEM_SWORD2:
         player->inventory|=item_id;
         if (player->weapon==0)
            player->weapon|=item_id;
         break;
      case ITEM_SHIELD:
      case ITEM_CHAINMAIL:
      case ITEM_HELMET:
         player->inventory|=item_id;
         calc_hero_defence(player);
         break;
      case ITEM_OWLFIG:
      case ITEM_CATFIG:
      case ITEM_SNAKEFIG:
      case ITEM_RINGREGEN:
      case ITEM_TELLRING:
         player->inventory|=item_id;
         break;
      case ITEM_HEARTC:
         player->max_health+=8;
         player->health+=8;
         break;
      case ITEM_QHEART:
         player->max_health+=2;
         if (!(player->max_health&7)) {
            player->health+=8;
            start_sample(get_item_sound(ITEM_HEARTC), FALSE);
         }
         break;
      case ITEM_HEART:
         player->health+=8;
         if (player->health>8*(player->max_health/8))
            player->health = player->max_health&(~7);
         break;
      case ITEM_KEY:
         player->keys++;
         if (player->keys>MAX_KEYS)
            player->keys = MAX_KEYS;
         break;
      case ITEM_POTION:
         player->health+=8*5;
         if (player->health>8*(player->max_health/8))
            player->health = player->max_health&(~7);
         break;
      case ITEM_CANDLEHOLD:
         /* Treasure - earn points */
         player->score += 1000;
         break;
      case ITEM_CHALICE:
         /* Treasure - earn points */
         player->score += 2500;
         break;
      case ITEM_GEM:
         /* Treasure - earn points */
         player->score += 1500;
         break;
      case ITEM_POISON:
         /* Lose health */
         player->health -= 16;
         player->invc = 30;
         break;
      case ITEM_TRAP:
         /* Lose health */
         player->health -= 18;
         player->invc = 30;
         break;
      case ITEM_SWITCHWGHTU:
      case ITEM_SWITCHUP:
         /* Depress a floor switch */
         toggle_switch_state();
         start_sample(get_item_sound(item_id), FALSE);
         /* Commit the changes to the tilemap */
         redraw_tilemap = TRUE;
         /* Don't pick up floor switch! */
         return FALSE;
      case ITEM_SWITCHWGHTD:
      case ITEM_SWITCHDN:
         /* Don't pick up floor switch! */
         return FALSE;
         
      case ITEM_BOSSEXIT:
         if (player->direction == 2) {
            game_state = GAME_STATE_WIN;
            game_state_counter = 30;
            return TRUE;
         }
         /* Don't pick up the exit! */
         return FALSE;
      case ITEM_BOSSGATE:
         if (player->direction == 2) {
            load_boss_level = TRUE;
            return TRUE;
         }
         /* Don't pick up the exit! */
         return FALSE;
      case ITEM_EXIT:
         if (player->direction == 2) {
            player->next_level = TRUE;
            set_music_playlist(game_playlist);
            return TRUE;
         }
         /* Don't pick up the exit! */
         return FALSE;
   }
   
   start_sample(get_item_sound(item_id), FALSE);
   return TRUE;
}

static PROJECTILE *fire_projectile(int type, int damage, int fromx, int fromy, int tox, int toy)
{
   PROJECTILE *projectile;

   projectile = alloc_projectile();
   projectile->x = fromx;
   projectile->y = fromy;
   projectile->type=type;
   projectile->sub_type = 0;
   projectile->anim=0;
   projectile->damage=damage;
   projectile->angle = atan2(toy-fromy, tox-fromx);

   switch (type) {
      case PROJECTILE_E_FIREBALL:
         projectile->speed=10;
         break;
      case PROJECTILE_CROSSARROW:
         projectile->type=PROJECTILE_ARROW;
         projectile->speed=10;
         break;
      default:
         projectile->speed=8;
         break;
   }

   return projectile;
}

static PARTICLE *create_particle(int type, int fromx, int fromy, int tox, int toy)
{
   PARTICLE *particle;

   particle = alloc_particle();
   particle->x = fromx;
   particle->y = fromy;
   particle->type=type;
   particle->sub_type = 0;
   particle->anim=0;
   particle->angle = atan2(toy-fromy, tox-fromx);

   switch (type) {
      default:
         particle->speed=8;
         break;
   }

   return particle;
}

/* Check for collisions between the player and a monster */
static int collide_monster(HERO *player)
{
   LINK *l;
   MONSTER *m;
   int dmg;
   int collide = FALSE;
   int px, py;

   l = get_monster_list(player->current_room);

   while (l) {
      BSHAPE *bbox;
      m = l->data;
      
      /* Monster collision bounding box */
      bbox = get_monster_hit_bbox(m->type);
      
      px = player->x+player->dx;
      py = player->y+player->dy;

      /* Check for central collisions */
      if (collide_box(px, py, &hit_bb, m->x, m->y, bbox)) {
         int n = 0;
         while (collide_box(px, py, &hit_bb, m->x, m->y, bbox) && n<2) {
            double angle = atan2(m->y-py, m->x-px);
            int dx, dy;
            int x, y;

            n++;

            dx = (2*player->velocity+1)*cos(angle);
            dy = (2*player->velocity+1)*sin(angle);
            if (px - dx < 2*TILE_SIZE) {
               dx = px - 2*TILE_SIZE;
            }
            if (px - dx > (MAP_WIDTH-3)*TILE_SIZE) {
               dx = px - (MAP_WIDTH-3)*TILE_SIZE;
            }
            /* backlash */
            x = px-dx+(1+sgn(dx))*TILE_SIZE/2;
            y = py-dy+(1-sgn(dy))*TILE_SIZE/2;
            
            if (!collide_geometry(player->current_room, x, y, &move_bb) &&
                  !blocked_by_item(player->current_room, px-dx, py-dy) ) {
               player->dx -= dx;
               player->dy -= dy;
               m->dx += dx;
               m->dy += dy;
            }

            px = player->x+player->dx;
            py = player->y+player->dy;
         }
         player->velocity = 1;

         if (!player->invc) {
            /* Deal damage to the player */
            dmg = calc_damage(throw_attack_dice(m->atk), player->def,  settings.difficulty>>1, -settings.difficulty);
            player->health -= dmg;
            player->invc = 30;
            start_sample(6, FALSE);
            if (get_monster_ability(m->type) & MONSTER_ABILITY_VAMPIRE) {
               /* Vampires gain health by draining their victim */
               /* They can in principle aquire infinite health in this way! */
               m->health += dmg * (1+player->level/6 + settings.difficulty);
            }
            collide = TRUE;
         }
      }
      l = l->next;
   }

   if (player->invc)
      player->invc--;

   return collide;
}

/* Collides a projectile with other monsters (or the player!) */
static int collide_projectile(PROJECTILE *projectile)
{
   LINK *l;
   LINK *ll;
   MONSTER *m;
   int c;
   int x, y;
   int dmg;
   int def;

   l = ll = get_monster_list(hero.current_room);

   /* Check for collisions with monsters/player */
   switch (projectile->type) {
      case PROJECTILE_E_FIREBALL:
         /* Check along the path of the fireball */
         for (c=0; c<projectile->speed; c++) {
            x = projectile->x+c*cos(projectile->angle);
            y = projectile->y+c*sin(projectile->angle);

            /* Check bounding circle */
            //if (distsqr(x+16,y+16,hero.x+16,hero.y+16)<256) {
            if (collide_box(x,y,&tile_bb, hero.x,hero.y,&hit_bb)) {
               /* Deal damage to the hero, remove the projectile */
               dmg = calc_damage(throw_attack_dice(projectile->damage), hero.def, settings.difficulty, -settings.difficulty);
               hero.health-=dmg;
               return TRUE;
            }
         }
         break;
      case PROJECTILE_ARROW:
         /* Check along the path of the arrow */
         for (c=0; c<projectile->speed; c++) {
            x = projectile->x+c*cos(projectile->angle);
            y = projectile->y+c*sin(projectile->angle);

            l=ll;
            while (l) {
               m = l->data;
               /* Check for central collisions */
               //if (!(m->ability & MONSTER_ABILITY_INVULNERABLE)
               //      && distsqr(x+16+c*cos(projectile->angle),y+c*sin(projectile->angle),m->x+16,m->y)<256) {
               if (!(m->ability & MONSTER_ABILITY_INVULNERABLE)
                     && collide_box(x+16+c*cos(projectile->angle),y+c*sin(projectile->angle),&arrow_bb, 
                                    m->x,m->y, get_monster_hit_bbox(m->type))) {
                  /* Deal damage to the monster, remove the projectile */
                  dmg = calc_damage(throw_attack_dice(projectile->damage), m->def, 0, settings.difficulty);
                  if (dmg && !m->hitc) {
                     m->health -= dmg;
                     m->hitc = 15;
                     /* Remember that the player hit this monster last */
                     //hero.target = m;
                  }
                  /* Knockback by fast arrows */
                  if (projectile->speed > 8) {
                     double angle = atan2(projectile->y - m->y, projectile->x - m->x);
                     m->dx -= cos(angle) * (projectile->speed-2);
                     m->dy -= sin(angle) * (projectile->speed-2);
                     
                     /* Interrupt spellcasting */
                     m->action &= ~ACTION_SPELL;
                     m->ai_timer = 1;
                  }
                  return TRUE;
               }
               //printf ("(%d, %d)/(%d, %d): %d\n", x,y,m->x+16, m->y+16,distsqr(x,y,m->x+16,m->y+16));
               l = l->next;
            }
         }
         break;
      case PROJECTILE_SWORD:
      case PROJECTILE_DAGGER:
      case PROJECTILE_CLUB:
         l=ll;
         while (l) {
            double angle;
            m = l->data;
            /* Check for central collisions */
            def = m->def;
            if (m->ability & MONSTER_ABILITY_INVULNERABLE) {
               /* Completely invulnerable to physical attacks */
               def |= st_dmg_pierce(0xFF)|st_dmg_cut(0xFF)|st_dmg_impact(0xFF);
            }

            //if (distsqr(projectile->x+16+8*cos(projectile->angle), projectile->y+16+16*sin(projectile->angle),m->x+16,m->y)<576) {
            if (collide_box(projectile->x+8*cos(projectile->angle),
                            projectile->y+8*sin(projectile->angle),
                            &melee_bb, m->x,m->y, get_monster_hit_bbox(m->type))) {
               /* Deal damage to the monster, remove the projectile */
               dmg = calc_damage(throw_attack_dice(projectile->damage), def, 0, settings.difficulty);
               if (dmg && !m->hitc) {
                  m->health -= dmg;
                  m->hitc = 15;
               }
               /* Knockback */
               if (!(m->ability&MONSTER_ABILITY_ETHEREAL) && projectile->type != PROJECTILE_DAGGER) {
                  angle = atan2(hero.y - m->y, hero.x - m->x);
                  if (projectile->type == PROJECTILE_CLUB) {
                     m->dx -= cos(angle) * 16;
                     m->dy -= sin(angle) * 16;
                  } else {
                     m->dx -= cos(angle) * 12;
                     m->dy -= sin(angle) * 12;
                  }
                  m->velocity = 0;
                  m->ai_timer = 2;
                  m->action &= ~ACTION_SPELL;
               }
               hero.action &= ~ACTION_PARRY;
               start_sample(18, FALSE);
               return TRUE;
            }
            //printf ("(%d, %d)/(%d, %d): %d\n", x,y,m->x+16, m->y+16,distsqr(x,y,m->x+16,m->y+16));
            l = l->next;
         }
         break;
      case PROJECTILE_EXPLOSION:
         l=ll;
         while (l) {
            m = l->data;
            /* Check for central collisions */
            if (distsqr(projectile->x+16,projectile->y+16,m->x+16,m->y+16)<2500) {
               /* Deal damage to the monster, remove the projectile */
               dmg = calc_damage(throw_attack_dice(projectile->damage), m->def, 0, settings.difficulty);
               if (m->type==MONSTER_BOSS1) {
                  if (projectile->anim & 1)
                     dmg/=(2+settings.difficulty);
                  else
                     dmg=0;
               }
               m->health-=dmg;
               /* Knockback */
               if (!(m->ability&MONSTER_ABILITY_ETHEREAL) ) {
                  double angle = atan2(projectile->y - m->y, projectile->x - m->x);
                  m->dx -= cos(angle);
                  m->dy -= sin(angle);
               }
            }
            l = l->next;
         }
         if (distsqr(projectile->x+16,projectile->y+16,hero.x+16,hero.y+16)<2500) {
            double angle;
            /* Deal damage to the hero, remove the projectile */
            dmg = calc_damage(throw_attack_dice(projectile->damage), hero.def, settings.difficulty, -settings.difficulty);
            hero.health-=dmg;
            
            /* Knockback */
            angle = atan2(projectile->y - hero.y, projectile->x - hero.x);
            hero.dx -= cos(angle);
            hero.dy -= sin(angle);
         }
   }

   return FALSE;
}

static void move_monster(MONSTER *m)
{
   if (m->move_delay) {
      m->move_delay--;
      return;
   }

   switch (m->type) {
      case MONSTER_BAT:
      case MONSTER_VAMPIREBAT:
      case MONSTER_BOSS2:
         break;
      default:
         m->move_delay = 1;
   }
   
   m->dy += sin(m->angle)*m->velocity;
   m->dx += cos(m->angle)*m->velocity;
   
   if ((m->dx<0) && (m->x < 2*TILE_SIZE)) {
      m->dx = -m->dx;
      m->angle = atan2(m->dy, m->dx);
   }

   if ((m->dx>0) && (m->x > (MAP_WIDTH - 2)*TILE_SIZE)) {
      m->dx = -m->dx;
      m->angle = atan2(m->dy, m->dx);
   }

   if (!collide_geometry(hero.current_room, m->x+m->dx, m->y+m->dy, &move_bb)) {
      /* For monsters that are not ethereal and do not fly, check for */
      /*  blocking items. */
      if ( !(get_monster_ability(m->type) & 
               (MONSTER_ABILITY_ETHEREAL|MONSTER_ABILITY_FLY) )) {
         LINK *l = get_item_list(hero.current_room);
         while (l) {
            ITEM *item = l->data;
            
            if (item_blocks(item->type) && monster_touches_item(m, item)) {
               double tdx = m->dx;
               double tdy = m->dy;

               m->dx = 0;
               if (!monster_touches_item(m, item)) {
                  tdx = 0;
               } else {
                  tdy = 0;
               }
               m->dx = tdx;
               m->dy = tdy;
            }
            l = l->next;
         }
      }
      m->x+=m->dx;
      m->y+=m->dy;
   } else {
      /* Time to think of something else */
      m->ai_timer = 0;
   }
   m->dy = 0;
   m->dx = 0;
}

static void update_monsters(void)
{
   LINK *l;
   LINK *ll;
   LINK *lnew;
   ITEM *item;
   MONSTER *m;
   MONSTER *newm;
   int *doors;
   int inroom;
   int spawners;
   int remove;
   int area;
   int x, y;
   int n;
   int d;

   lnew = NULL;
   l = ll = get_monster_list(hero.current_room);

   n = get_list_length(ll);

   /* Count number of monsters in this area */
   inroom = 0;
   spawners = 1;
   while (l) {
      m = l->data;
      if ((m->health>=0) && tile_is_visible(m->x/TILE_WIDTH, m->y/TILE_HEIGHT))
         inroom++;
      if ((get_monster_ability(m->type) & MONSTER_ABILITY_SPAWN) && tile_is_visible(m->x/TILE_WIDTH, m->y/TILE_HEIGHT))
         spawners++;
      l = l->next;
   }
   l = ll;
   
   /* Count area size */
   area = 0;
   for(y=0; y<MAP_HEIGHT; y++) {
      for(x=0; x<MAP_WIDTH; x++) {
         if (tile_is_visible(x, y))
            area++;
      }
   }

   while (l) {
      m = l->data;
      remove = 0;
      if (m->hitc)
         m->hitc--;
      if (m->health>=0) {

         /* Special abilities */
         /* Spawn new monsters - only if not too many in room yet! */
         if ((get_monster_ability(m->type) & MONSTER_ABILITY_SPAWN) && 
             !m->ai_timer && 
             (2*inroom/spawners)<MIN(MIN(16, area/15),8+hero.level+6*(hero.level/6)+settings.difficulty)) {
            newm = alloc_monster();
            m->ai_timer=1;
            do {
               newm->type = spawn_monster(hero.level);
            } while (newm->type == MONSTER_NONE);
            /* Delay until next monster spawning - depends on how powerful */
            /*  the last monster spawned is */
            switch (newm->type) {
               case MONSTER_ARMEDZOMBIE:
               case MONSTER_ZOMBIE:
                  m->ai_timer = 350+genrandf()*200 - 50*settings.difficulty;
                  break;
               case MONSTER_SKELETON:
                  m->ai_timer = 400+genrandf()*300 - 50*settings.difficulty;
                  break;
               case MONSTER_GHOUL:
               case MONSTER_DAEMON:
                  m->ai_timer = 600+genrandf()*200 - 50*settings.difficulty;
                  break;
               case MONSTER_WIZARD:
                  m->ai_timer = 600+genrandf()*400 - 50*settings.difficulty;
                  break;
               case MONSTER_GRIMREAPER:
                  m->ai_timer = 800+genrandf()*300 - 50*settings.difficulty;
                  break;
               case MONSTER_GHOST:
               case MONSTER_BAT:
                  m->ai_timer = 240+genrandf()*240 - 50*settings.difficulty;
                  break;
               case MONSTER_VAMPIREBAT:
                  m->ai_timer = 240+genrandf()*300 - 50*settings.difficulty;
                  break;
            }
            newm->x = m->x;
            newm->y = m->y;
            newm->dy = 0;
            newm->dx = 0;
            if (m->x < 2*TILE_WIDTH) {
               newm->x += TILE_WIDTH;
            } else if (m->x > (MAP_WIDTH-2)*TILE_WIDTH) {
               newm->x -= TILE_WIDTH;
            } else {
               newm->x += 4-8*genrandf();
               newm->y += 4-8*genrandf();
            }
            newm->dropped_item = drop_item(hero.level);
            newm->health = get_monster_hp(newm->type);
            newm->def = get_monster_defence(newm->type);
            newm->atk = get_monster_attack(newm->type);
            newm->velocity = 1;
            newm->angle = -M_PI/2;
            newm->ai_timer = 1;
            newm->move_delay = 0;
            newm->hitc = 0;
            newm->ability = get_monster_ability(newm->type);
            newm->prev = NULL;
            newm->next = NULL;
            lnew = add_to_list(newm, lnew);
         }
         
         if ( get_monster_ability(m->type) & MONSTER_ABILITY_SPELLS && 
             !m->ai_timer){

            if (m->action&ACTION_SPELL) {
               /* Clear ability flags */
               m->ability &= ~(MONSTER_ABILITY_ETHEREAL|MONSTER_ABILITY_INVULNERABLE);

               /* Pick spell efect */
               if (genrandf()<0.2) {
                  /* Fireball */
                  set_projectile_list(hero.current_room, add_to_list(fire_projectile(PROJECTILE_E_FIREBALL, st_dmg_impact(pack_dmg(4,6)), m->x, m->y, hero.x,hero.y), get_projectile_list(hero.current_room)));
                  m->ai_timer += 350;
               } else if (genrandf()<0.4) {
                  /* Invulnerable/ethereal */
                  m->ability |= MONSTER_ABILITY_ETHEREAL;
                  m->ability |= MONSTER_ABILITY_INVULNERABLE;
                  m->ai_timer += 200;
               }
               m->action &= ~ACTION_SPELL;
            } else {
               m->ability &= ~(MONSTER_ABILITY_ETHEREAL|MONSTER_ABILITY_INVULNERABLE);
               
               /* Wizards don't cast spells at random */
               if ((m->type!=MONSTER_WIZARD) && (genrandf() < 0.1)) {
                  m->action |= ACTION_SPELL;
                  m->ai_timer = 10;
               }
            }
         }

         switch (m->type) {
            case MONSTER_ASPAWN:
               /* Fire projectile? */
               if (tile_is_visible(m->x/TILE_SIZE, m->y/TILE_SIZE) &&
                   m->ai_timer == 40 && genrandf()<0.65 &&
                   distsqr(hero.x, hero.y, m->x, m->y)<=49*TILE_SIZE*TILE_SIZE) {
                  set_projectile_list(hero.current_room, add_to_list(fire_projectile(PROJECTILE_E_FIREBALL, st_dmg_impact(pack_dmg(4,6)), m->x, m->y, hero.x,hero.y), get_projectile_list(hero.current_room)));
               }
            case MONSTER_SPAWN:
               if (m->health>=0)
                  m->anim++;
               /* Check ai */
               if (m->ai_timer) {
                  m->ai_timer--;
               }
               break;
            case MONSTER_ARMEDZOMBIE:
               if (!m->ai_timer && 
                     tile_is_visible(m->x/TILE_SIZE, m->y/TILE_SIZE) && 
                     monster_face_player(m, &hero, 8*8*TILE_SIZE*TILE_SIZE)) {
                  m->angle = atan2(hero.y - m->y, hero.x - m->x);
                  m->ai_timer = 25*genrandf()+1;
                  m->velocity = 4 + settings.difficulty/2;
               }
            case MONSTER_ZOMBIE:
               if (m->health>=0)
                  m->anim++;
               /* Check ai */
               if (m->ai_timer) {
                  m->ai_timer--;
                  move_monster(m);
               } else {
                  m->angle = ((int)(4*genrandf())) * M_PI/2;
                  m->ai_timer = 10+20*genrandf();
                  m->velocity = 1 + settings.difficulty;
               }
               break;
            case MONSTER_SKELETON:
               if (m->health>=0)
                  m->anim++;
               /* Check ai */
               if (m->ai_timer) {
                  m->ai_timer--;
                  move_monster(m);
               } else {
                  m->angle = ((int)(4*genrandf())) * M_PI/2;
                  m->ai_timer = 10+20*genrandf();
                  m->velocity = 1 + settings.difficulty;
               }
               break;
            case MONSTER_GHOUL:
               if (m->health>=0)
                  m->anim++;
               /* Check ai */
               if (m->ai_timer) {
                  m->ai_timer--;
                  move_monster(m);
               } else {
                  /* Move towards hero */
                  if (tile_is_visible(m->x/TILE_SIZE, m->y/TILE_SIZE) && 
                      monster_face_player(m, &hero, 16*16*TILE_SIZE*TILE_SIZE))
                     m->angle = atan2(hero.y - m->y, hero.x - m->x);
                  else
                     m->angle = ((int)(4*genrandf())) * M_PI/2;
                  m->ai_timer = 10+20*genrandf();
                  m->velocity = 2 + MAX(hero.level/5, 2) + settings.difficulty;
               }
               break;
            case MONSTER_GRIMREAPER:
               if (m->health>=0)
                  m->anim++;
               /* Check ai */
               if (m->ai_timer) {
                  m->ai_timer--;
                  m->ability |= MONSTER_ABILITY_ETHEREAL;
                  move_monster(m);
               } else {
                  /* Move towards hero */
                  if (tile_is_visible(m->x/TILE_SIZE, m->y/TILE_SIZE) &&
                      monster_face_player(m, &hero, 4*16*16*TILE_SIZE*TILE_SIZE)) {
                     m->angle = atan2(hero.y - m->y, hero.x - m->x);
                     m->ability &= ~MONSTER_ABILITY_ETHEREAL;
                  } else {
                     m->angle = ((int)(4*genrandf())) * M_PI/2;
                  }
                  m->ai_timer = 20+20*genrandf();
                  m->velocity = 3+3*genrandf() + settings.difficulty;
               }
               break;
            case MONSTER_DAEMON:
               if (m->health>=0)
                  m->anim++;
               /* Check ai */
               if (m->ai_timer) {
                  m->ai_timer--;
                  move_monster(m);
               } else {
                  /* Move towards hero */
                  if (tile_is_visible(m->x/TILE_SIZE, m->y/TILE_SIZE) &&
                      monster_face_player(m, &hero, 2*16*16*TILE_SIZE*TILE_SIZE))
                     m->angle = atan2(hero.y - m->y, hero.x - m->x);
                  else
                     m->angle = ((int)(4*genrandf())) * M_PI/2;
                  m->ai_timer = 30*genrandf();
                  m->velocity = 2+2*genrandf() + settings.difficulty;
                  
                  /* Call forth a bat or (at higher levels) a vampire bat */
                  /* but only if the room isn't too crowded yet and there's */
                  /* free memory */
                  if (genrandf()<0.05 && inroom<15 && (newm = alloc_monster())) {
                     newm->type = MONSTER_BAT;
                     if (genrandf() < 0.01*(hero.level/6))
                        newm->type = MONSTER_VAMPIREBAT;
                     newm->x = m->x;
                     newm->y = m->y;
                     newm->dy = 0;
                     newm->dx = 0;
                     if (m->x < 2*TILE_WIDTH) {
                        newm->x += TILE_WIDTH;
                     } else if (m->x > (MAP_WIDTH-2)*TILE_WIDTH) {
                        newm->x -= TILE_WIDTH;
                     } else {
                        newm->x += 4-8*genrandf();
                        newm->y += 4-8*genrandf();
                     }
                     newm->dropped_item = drop_item(hero.level);
                     newm->health = get_monster_hp(newm->type);
                     newm->def = get_monster_defence(newm->type);
                     newm->atk = get_monster_attack(newm->type);
                     newm->velocity = 1;
                     m->angle = -M_PI/2;
                     newm->ai_timer = 1;
                     newm->move_delay = 0;
                     newm->hitc = 0;
                     newm->ability = get_monster_ability(newm->type);
                     newm->prev = m;         /* Set parent pointer */
                     newm->next = NULL;
                     lnew = add_to_list(newm, lnew);
                  }
               }
               break;
            case MONSTER_WIZARD:
               if (m->health>=0)
                  m->anim++;

               /* Retaliate */
               if (m->hitc && !(m->action&ACTION_SPELL)) {
                  m->ai_timer = 20;
                  m->action |= ACTION_SPELL;
               }
               
               if (m->ai_timer) {
                  m->ai_timer--;
                  //if (!(m->action&ACTION_SPELL))
                  move_monster(m);
               } else {
                  /* Do something special if facing hero */
                  if (!(m->action&ACTION_SPELL) && 
                     tile_is_visible(m->x/TILE_SIZE, m->y/TILE_SIZE) && 
                     monster_face_player(m, &hero, 8*8*TILE_SIZE*TILE_SIZE) && 
                     genrandf()<0.3) {
                     m->action |= ACTION_SPELL;
                     m->ai_timer = 30;
                     m->velocity = 0;
                  } else {
                     m->angle = ((int)(4*genrandf())) * M_PI/2;
                     m->ai_timer = 30+10*genrandf();
                     m->velocity = 1+genrandf() + settings.difficulty;
                  }
                     
                  /* cast a spell */
               }
               break;
            case MONSTER_BOSS2:
               /* King Vampire */
               if (!m->ai_timer && m->velocity < 1.0e-3) {
                  m->ai_timer = 25 + 10*genrandf();
                  m->velocity = 9+2*genrandf() + (3*settings.difficulty)/2;
               } else if (m->ai_timer == 5 && m->velocity < 1.0e-3) {
                  m->angle = atan2(hero.y-m->y, hero.x-m->x);
               }

               if (m->health>=0)
                  m->anim++;
               /* Check ai */
               move_monster(m);
               if (m->ai_timer) {
                  m->ai_timer--;
               } else if (genrandf()<0.09) {
                  m->velocity = 0;
                  m->ai_timer = 30;
               } else if (genrandf()<0.05 && inroom<5 && (newm = alloc_monster())){
                  /* Call forth a bat or (at higher levels) a vampire bat */
                  /* but only if the room isn't too crowded yet and there's */
                  /* free memory */
                  m->ai_timer = 20;
                  newm->type = MONSTER_BAT;
                  if (genrandf() < 0.2)
                     newm->type = MONSTER_VAMPIREBAT;
                  newm->x = m->x;
                  newm->y = m->y;
                  newm->dy = 0;
                  newm->dx = 0;
                  if (m->x < 2*TILE_WIDTH) {
                     newm->x += TILE_WIDTH;
                  } else if (m->x > (MAP_WIDTH-2)*TILE_WIDTH) {
                     newm->x -= TILE_WIDTH;
                  } else {
                     newm->x += 4-8*genrandf();
                     newm->y += 4-8*genrandf();
                  }
                  newm->dropped_item = drop_item(hero.level);
                  newm->health = get_monster_hp(newm->type);
                  newm->def = get_monster_defence(newm->type);
                  newm->atk = get_monster_attack(newm->type);
                  newm->velocity = 1;
                  m->angle = -M_PI/2;
                  newm->ai_timer = 1;
                  newm->move_delay = 0;
                  newm->hitc = 0;
                  newm->ability = get_monster_ability(newm->type);
                  newm->prev = m;         /* Set parent pointer */
                  newm->next = NULL;
                  lnew = add_to_list(newm, lnew);
               } else if (genrandf()<0.02) {
                  set_projectile_list(hero.current_room, add_to_list(fire_projectile(PROJECTILE_E_FIREBALL, st_dmg_impact(pack_dmg(4,6)), m->x, m->y, hero.x,hero.y), get_projectile_list(hero.current_room)));
                  m->ai_timer = 20*genrandf();
               } else {
                  /* Move in a random direction */
                  m->angle = 2*genrandf() * M_PI;
                  if (genrandf()<0.1) {
                     m->ai_timer = 30*genrandf();
                     m->velocity = 2+3*genrandf() + settings.difficulty;
                  } else {
                     m->ai_timer = 3*genrandf();
                     m->velocity = 1+2*genrandf() + settings.difficulty;
                  }
               }
               break;
            case MONSTER_VAMPIREBAT:
               /* Vampire bats sometimes dive at the player */
               if (!m->ai_timer && 
                  tile_is_visible(m->x/TILE_SIZE, m->y/TILE_SIZE) && 
                  genrandf()<0.25) {
                  m->angle = atan2(hero.y-m->y, hero.x-m->x);
                  m->ai_timer = 24*genrandf();
                  m->velocity = 2.5+1.5*genrandf() + (3*settings.difficulty)/2;
               }
               if (m->health>=0)
                  m->anim++;
               /* Check ai */
               if (m->ai_timer) {
                  m->ai_timer--;
                  move_monster(m);
               } else {
                  if (m->prev == NULL) {
                     /* This is a `free' bat, it doesn't follow a daemon */
                     m->angle = 2*genrandf() * M_PI;
                     if (genrandf()<0.1) {
                        m->ai_timer = 20*genrandf();
                        m->velocity = 2+3*genrandf() + settings.difficulty;
                     } else {
                        m->ai_timer = 3*genrandf();
                        m->velocity = 3*genrandf() + settings.difficulty;
                     }
                  } else {
                     /* Swarm around parent */
                     if (genrandf()<0.2) {
                        m->angle = atan2(m->prev->y-m->y, m->prev->x-m->x);
                        m->ai_timer = 20*genrandf();
                        m->velocity = 4+3*genrandf();
                     } else {
                        m->angle = 2*genrandf() * M_PI;
                        m->ai_timer = 7*genrandf();
                        m->velocity = 2+2*genrandf() + settings.difficulty;
                     }
                     if (m->prev->health <= 0) {
                        /* Become free if the daemon that summoned it dies */
                        /* Die if the summoner was the King vampire */
                        if (m->prev->type != MONSTER_BOSS2)
                           m->prev = NULL;
                        else
                           m->health = -1;
                     }
                  }
               }

            case MONSTER_BAT:
               if (m->health>=0)
                  m->anim++;
               /* Check ai */
               if (m->ai_timer) {
                  m->ai_timer--;
                  move_monster(m);
               } else {
                  if (m->prev == NULL) {
                     /* This is a `free' bat, it doesn't follow a daemon */
                     m->angle = 2*genrandf() * M_PI;
                     if (genrandf()<0.1) {
                        m->ai_timer = 20*genrandf();
                        m->velocity = 2+3*genrandf() + settings.difficulty;
                     } else {
                        m->ai_timer = 3*genrandf();
                        m->velocity = 3*genrandf() + settings.difficulty;
                     }
                  } else {
                     /* Swarm around parent */
                     if (genrandf()<0.2) {
                        m->angle = atan2(m->prev->y-m->y, m->prev->x-m->x);
                        m->ai_timer = 20*genrandf();
                        m->velocity = 4+3*genrandf();
                     } else {
                        m->angle = 2*genrandf() * M_PI;
                        m->ai_timer = 7*genrandf();
                        m->velocity = 2+2*genrandf() + settings.difficulty;
                     }
                     if (m->prev->health <= 0) {
                        /* Become free if the daemon that summoned it dies */
                        /* Die if the summoner was the King vampire */
                        if (m->prev->type != MONSTER_BOSS2)
                           m->prev = NULL;
                        else
                           m->health = -1;
                     }
                  }
               }
               break;
            case MONSTER_GHOST:
               if (m->health>=0)
                  m->anim++;
               /* Check ai */
               if (m->ai_timer) {
                  m->ai_timer--;
                  move_monster(m);
               } else {
                  /* At higher levels, ghosts move towards the player, but */
                  /*  slowly */
                  if (hero.level>(8-settings.difficulty) &&
                      genrandf() < (0.3 + settings.difficulty*0.3) &&
                      tile_is_visible(m->x/TILE_SIZE, m->y/TILE_SIZE) &&
                      monster_face_player(m, &hero, 4*16*16*TILE_SIZE*TILE_SIZE)) {
                     m->angle = atan2(hero.y - m->y, hero.x - m->x);
                  } else {
                     m->angle = ((int)(4*genrandf())) * M_PI/2;
                  }
                  if (genrandf()<0.1) {
                     m->ai_timer = 30*genrandf();
                     m->velocity = 1+2*genrandf() + settings.difficulty;
                  } else {
                     m->ai_timer = 3*genrandf();
                     m->velocity = 2*genrandf() + settings.difficulty;
                  }
               }
               break;
            case MONSTER_BOSS1:
               break;
            //case MONSTER_BOSS2:
            //   break;
         }
      } else {
         if (m->type == MONSTER_ASPAWN) {
            /* Attacking spawners first revert to non-attacking spawners */
            m->type = MONSTER_SPAWN;
            m->health = get_monster_hp(m->type);
            m->hitc = 30;
         } else if (m->type == MONSTER_BOSS2) {
            set_particle_list(hero.current_room, add_to_list(create_particle(PROJECTILE_BOSS_EXPLO, m->x, m->y, 0,0), get_particle_list(hero.current_room)));
            remove = 1;
         } else if (m->action&ACTION_DIE) {
               if (hero.target == m) {
                  hero.target = NULL;
                  hero.target_x = 0;
                  hero.target_y = 0;
               }
               /* Create death animation */
               set_particle_list(hero.current_room, add_to_list(create_particle(PROJECTILE_DEATH, m->x, m->y, 0,0), get_particle_list(hero.current_room)));
               start_sample(1, FALSE);
               remove = 1;
               /* Drop item */
               switch (m->type) {
                  case MONSTER_BOSS2:
                     hero.score+=10000;
                  case MONSTER_BOSS1:
                     hero.score+=10000;
                     break;
                  case MONSTER_ASPAWN:
                  case MONSTER_SPAWN:
                     hero.score+=100;
                     break;
                  case MONSTER_ARMEDZOMBIE:
                  case MONSTER_ZOMBIE:
                     hero.score+=10;
                     break;
                  case MONSTER_SKELETON:
                     hero.score+=30;
                     break;
                  case MONSTER_GHOUL:
                     hero.score+=60;
                     break;
                  case MONSTER_DAEMON:
                     hero.score+=80;
                     break;
                  case MONSTER_WIZARD:
                     hero.score+=80;
                     break;
                  case MONSTER_GRIMREAPER:
                     hero.score+=100;
                     break;
                  case MONSTER_VAMPIREBAT:
                     hero.score+=50;
                     break;
                  case MONSTER_BAT:
                     hero.score+=5;
                     break;
                  case MONSTER_GHOST:
                     hero.score+=10;
                     break;
               }
               if (m->dropped_item!=ITEM_NONE) {
                  int iter;
                  item = alloc_item();
                  item->x = m->x+4-8*genrandf();
                  item->y = m->y+4-8*genrandf();
                  iter = 0;
                  while (collide_geometry(hero.current_room, item->x, item->y, &item_bb)) {
                     item->x = m->x+4-12*genrandf();
                     item->y = m->y+4-12*genrandf();
                     iter++;
                     if (iter>5)
                        break;
                  }
                  item->type = m->dropped_item;
                  item->child = NULL;
                  item->lifetime = get_item_lifetime(item->type);
                  item->anim = 0;
                  item->lifetime = get_item_lifetime(item->type);
                  set_item_list(hero.current_room, add_to_list(item, get_item_list(hero.current_room)));
               }
         } else {
            m->action|=ACTION_DIE;
            if (hero.target == m) {
               hero.target = NULL;
               hero.target_x = 0;
               hero.target_y = 0;
            }
         }
      }

      if (remove) {
         set_screen_changed();
         if (l==ll) {
            l = ll = remove_from_list(m, l);
         } else {
            l = remove_from_list(m, l);
         }
         free_monster(m);
      } else {
         l = l->next;
      }
   }

   /* Add new monsters to the list */
   l=lnew;
   while(l) {
      ll = add_to_list(l->data, ll);
      l = l->next;
   }

   /* Free memory used by the temporary list */
   destroy_list(lnew, NULL);

   /* Store the updated list */
   set_monster_list(hero.current_room, ll);
   
   /* Check special doors that close if there are monsters in the room */
   if (inroom == 0) {
      int play = FALSE;
      doors = get_right_doors(hero.current_room);
      for (n=0; n<4; n++) {
         if (tile_is_visible(MAP_WIDTH-2, n*15+8) && (doors[n] & MAP_DOOR_CREATURER)) {
            doors[n] ^= MAP_DOOR_CREATURER;
            calculate_geometry_bbox(hero.current_room);
            redraw_tilemap = TRUE;
            play = TRUE;
         }
      }

      doors = get_left_doors(hero.current_room);
      for (n=0; n<4; n++) {
         if (tile_is_visible(1, n*15+8) && (doors[n] & MAP_DOOR_CREATUREL)) {
            doors[n] ^= MAP_DOOR_CREATUREL;
            calculate_geometry_bbox(hero.current_room);
            redraw_tilemap = TRUE;
            play = TRUE;
         }
      }
      if (play) start_sample(17, FALSE);
   }
}

/* Update boss (just Serpentile for now) */
static void update_boss(void)
{
   int x=0, y=0;
   LINK *l;
   LINK *ll;
   MONSTER *m;

   ll = get_monster_list(hero.current_room);

   if (get_list_length(ll)==0)
      return;

   m = ll->data;
   if (m->type == MONSTER_BOSS1) {
      MONSTER *mp;
      MONSTER *head;
      double angle;
      double da;
      int vx, vy;
      int remove;
      int n;
      int k;

      vx = 0;
      vy = 0;

      for (l=ll; l; l=l->next) {
         head=l->data;
         /* If it is really a head... */
         if (head->prev==NULL) {
            /* Store these (we may need them later) */
            x = head->x;
            y = head->y;

            /* Update all segments, starting at the tail */
            /* Relabel while we're reversing the list */
            n = 0;
            for (mp=head; mp->next; mp=mp->next) {
               mp->order = n/4;
               n++;
            }
            mp->order = n/4;

            /* Velocity depends on the length of the snake */
            head->velocity = 1+15/MAX(1,n+1-settings.difficulty);

            /* Very short segments die */
            if (n<8) {
               for (mp=head; mp; mp=mp->next) {
                  mp->health = -1;
               }
               set_particle_list(hero.current_room, add_to_list(create_particle(PROJECTILE_BOSS_DEATH, head->x, head->y, 0,0), get_particle_list(hero.current_room)));
            } else {
               for (k=0; k<head->velocity; k++) {
                  for (m=mp; m->prev; m=m->prev) {
                     /* Propagate snake movement */
                     m->x = m->prev->x;
                     m->y = m->prev->y;
                     m->angle = m->prev->angle;
                  }

                  angle=(1.0-2.0*genrandf())/10.0;
                  /* Update the head, the action depends on the length of the snake */
                  da = (atan2(hero.y-m->y, hero.x-m->x)-head->angle);
                  while (da<-M_PI)
                     da+=2*M_PI;
                  if (n>=96) {
                     angle+=da/30.0;
                  } else if (n>=48 && n<96) {
                     angle+=da/20.0;
                  } else if (n>=32 && n<48) {
                     angle+=da/50.0;
                     angle+=1.0*(1.0+settings.difficulty)*(1.0-2.0*genrandf())/10.0;
                  } else if (n>=16 && n<32) {
                     angle+=da/50.0;
                     angle+=2.0*(1.0+settings.difficulty)*(1.0-2.0*genrandf())/10.0;
                  } else if (n>=8 && n<16) {
                     angle+=3.0*(1.0+settings.difficulty)*(1.0-2.0*genrandf())/10.0;
                  }

                  head->angle+=angle;

                  vx = 10000*cos(head->angle);
                  vy = 10000*sin(head->angle);

                  if (n>64) {
                     vy += 1000*sgn(hero.y-m->y);
                     vx += 1000*sgn(hero.x-m->x);
                  }

                  if (head->x<2*TILE_WIDTH && vx<0) {
                     vx = -vx;
                  }
                  if (head->x>(MAP_WIDTH-3)*TILE_WIDTH && vx>0) {
                     vx = -vx;
                  }
                  if (head->y<2*TILE_HEIGHT && vy<0) {
                     vy = -vy;
                  }
                  if (head->y>14*TILE_WIDTH && vy>0) {
                     vy = -vy;
                  }

                  head->angle = atan2(vy, vx);

                  head->x+=vx/2500;
                  head->y+=vy/2500;
               }
            }
         }
      }

      /* Now remove the dead wood */
      l = ll = get_monster_list(hero.current_room);
      /* We need to treat all nodes within one snake of the same order the same! */
      k = -1;
      while (l) {
         m = l->data;
         remove = 0;
         if (m->health>=0) {
            if ((!m->prev) && (!m->next))
               m->health=-1;
            /* If at a head or tail, stop the killing! */
            if (!m->next)
               k=-1;
            if (!m->prev)
               k=-1;
            /* If any of its children are dead, kill this segment as well */
            for (mp=m->next; mp && mp->order==m->order; mp=mp->next) {
               if (mp->health<0) {
                  remove = 1;
               }
            }
            /* If of the same order as the last one killed, kill this one as well */
            if (m->order==k) {
               remove = 1;
               if (hero.target == m) {
                  hero.target = NULL;
                  hero.target_x = 0;
                  hero.target_y = 0;
               }
               if (m->prev) {
                  m->prev->next = m->next;
               }
               if (m->next) {
                  m->next->prev = m->prev;
               }
            }
         } else {
            if (m->action&ACTION_DIE) {
               set_particle_list(hero.current_room, add_to_list(create_particle(PROJECTILE_DEATH, m->x, m->y, 0,0), get_particle_list(hero.current_room)));
               start_sample(1, FALSE);
               remove = 1;
               if (m->prev) {
                  m->prev->next = NULL;
               }
               if (m->next) {
                  m->next->prev = NULL;
               }
               k = m->order;
            } else {
               m->action|=ACTION_DIE;
               if (hero.target == m) {
                  hero.target = NULL;
                  hero.target_x = 0;
                  hero.target_y = 0;
               }
            }
         }

         if (remove) {
            set_screen_changed();
            if (l==ll) {
               l = ll = remove_from_list(m, l);
            } else {
               l = remove_from_list(m, l);
            }
            free_monster(m);
         } else {
            l = l->next;
         }
      }

      for (l=ll; l; l=l->next) {
         head=l->data;
         /* If it is really a head... */
         if (head->prev==NULL) {
            /* Update all segments, starting at the tail */

            /* Relabel while we're reversing the list */
            n = 0;
            for (mp=head; mp->next; mp=mp->next) {
               mp->order = n/4;
               n++;
            }
            mp->order = n/4+1;
         }
      }
      
      /* Store the updated list */
      set_monster_list(hero.current_room, ll);
      if (get_list_length(ll)==0) {
         set_particle_list(hero.current_room, add_to_list(create_particle(PROJECTILE_BOSS_EXPLO, x, y, 0,0), get_particle_list(hero.current_room)));
         hero.score+=10000;
      }
   } else { //if (m->type == MONSTER_BOSS2) {
      update_monsters();
   }
   

   ll = get_monster_list(hero.current_room);

   /* Victory */
   if (get_list_length(ll)==0) {
      start_sample(11, 0);
      stop_music();
      start_music(8, FALSE);
      set_music_playlist(NULL);
   }
}

static void update_projectiles(void)
{
   LINK *l;
   LINK *ll;
   LINK *il, *ill;
   LINK *lnew;
   PROJECTILE *projectile;
   int remove;
   int c;
   int *ldoor;
   int *rdoor;

   lnew = NULL;
   l = ll = get_projectile_list(hero.current_room);
   while (l) {
      projectile = l->data;
      remove = 0;
      switch (projectile->type) {
         case PROJECTILE_SWORD:
            projectile->x=hero.x+16*cos(projectile->angle);
            projectile->y=hero.y+16*sin(projectile->angle);
            switch (hero.direction) {
               case 0:
                  projectile->angle = M_PI/2;
                  break;
               case 1:
                  projectile->angle = 0;
                  break;
               case 2:
                  projectile->angle = -M_PI/2;
                  break;
               case 3:
                  projectile->angle = M_PI;
                  break;
            }

            if (collide_projectile(projectile)||projectile->anim) {
               //start_sample(16, FALSE);
               remove = 1;
            }
            break;
         case PROJECTILE_DAGGER:
         case PROJECTILE_CLUB:
            projectile->x=hero.x+16*cos(projectile->angle);
            projectile->y=hero.y+16*sin(projectile->angle);

            projectile->anim++;
            if (collide_projectile(projectile)||projectile->anim>24) {
               remove = 1;
            }
            break;
         case PROJECTILE_ARROW:
            if (collide_projectile(projectile)) {  /* Impact */
               start_sample(0, FALSE);
               remove = 1;
            } else {                               /* Continue forward */
               projectile->x+=projectile->speed*cos(projectile->angle);
               projectile->y+=projectile->speed*sin(projectile->angle);
               if (!tile_passable(hero.current_room, projectile->x, projectile->y) && !tile_is_door(hero.current_room, projectile->x, projectile->y)) {
                  start_sample(0, FALSE);
                  remove = 1;
               }
               if (((projectile->x-top_x) < -10) || ((projectile->x-top_x) > SCREEN_W+10) || ((projectile->y-top_y) < -10) || ((projectile->y-top_y) > SCREEN_H+10)) {
                  remove = 1;
               }
            }
            break;
         case PROJECTILE_E_FIREBALL:
            projectile->anim++;
            if (collide_projectile(projectile)) {  /* Impact */
               start_sample(10, FALSE);
               remove = 1;
            } else {                               /* Continue forward */
               if (projectile->anim>30) {
                  projectile->x+=projectile->speed*cos(projectile->angle);
                  projectile->y+=projectile->speed*sin(projectile->angle);
                  if (!tile_passable(hero.current_room, projectile->x, projectile->y)) {
                     start_sample(10, FALSE);
                     remove = 1;
                  }
                  if (((projectile->x-top_x) < -10) || ((projectile->x-top_x) > SCREEN_W+10) || ((projectile->y-top_y) < -10) || ((projectile->y-top_y) > SCREEN_H+10)) {
                     remove = 1;
                  }
               }
               /* Adjust path of projectile in its initial stage */
               if (projectile->anim<60) {
                  projectile->angle += (atan2(hero.y-projectile->y, hero.x-projectile->x)-projectile->angle)/10;
               }
            }
            break;
         case PROJECTILE_BOMB:
            projectile->anim++;
            if (projectile->anim>50) {
               start_sample(10, FALSE);
               remove = 1;
               lnew = add_to_list(fire_projectile(PROJECTILE_EXPLOSION, projectile->damage, projectile->x, projectile->y, 0,0), lnew);
            }
            break;
         case PROJECTILE_EXPLOSION:
            projectile->anim++;
            collide_projectile(projectile);
            if (projectile->anim>24) {
               remove = 1;
               /* Check for things that were opened by this explosion */
               ldoor = get_left_doors(hero.current_room);
               rdoor = get_right_doors(hero.current_room);
               for (c=0; c<4; c++) {
                  /* Open a cracked wall with a bomb */
                  if (distsqr(projectile->x, projectile->y, 1*TILE_WIDTH, (8+c*15)*TILE_WIDTH)<2500) {
                     if (ldoor[c]==MAP_DOOR_BLOCKED) {
                        ldoor[c] = MAP_DOOR_OPEN;
                        /* Commit the changes to the tilemap */
                        redraw_tilemap = TRUE;
                        calculate_geometry_bbox(hero.current_room);
                        calculate_geometry_bbox(hero.current_room-1);
                     }
                  }
                  if (distsqr(projectile->x, projectile->y, (MAP_WIDTH-2)*TILE_WIDTH, (8+c*15)*TILE_WIDTH)<2500) {
                     if (rdoor[c]==MAP_DOOR_BLOCKED) {
                        rdoor[c] = MAP_DOOR_OPEN;
                        /* Commit the changes to the tilemap */
                        redraw_tilemap = TRUE;
                        calculate_geometry_bbox(hero.current_room);
                        calculate_geometry_bbox(hero.current_room+1);
                     }
                  }
               }
               /* Cracked blocks */
               ill = il = get_item_list(hero.current_room);
               while (il) {
                  ITEM *item = il->data;
                  if (item->type == ITEM_BLOCKCRACK && 
                        distsqr(projectile->x, projectile->y, 
                                item->x, item->y)<2500) {
                     if (item->child) {
                        il->data = item->child;
                        item->child = NULL;
                        free_item(item);
                        il = il->next;
                     } else {
                        if (il==ill) {
                           il = ill = remove_from_list(item, il);
                        } else {
                           il = remove_from_list(item, il);
                        }
                        free_item(item);
                     }
                  } else {
                     il = il->next;
                  }
               }
               set_item_list(hero.current_room, ill);
            }
         break;
      }

      if (remove) {
         if (projectile->type == PROJECTILE_SWORD) {
            hero.weapon_buildup = 0;
         } else if (projectile->type == PROJECTILE_DAGGER) {
            hero.weapon_buildup = 0;
         } else if (projectile->type == PROJECTILE_CLUB) {
            hero.weapon_buildup = 0;
         }

         set_screen_changed();
         if (l==ll) {
            l = ll = remove_from_list(projectile, l);
         } else {
            l = remove_from_list(projectile, l);
         }
         free_projectile(projectile);
      } else {
         l = l->next;
      }
   }

   /* Add new items to the list */
   l=lnew;
   while(l) {
      ll = add_to_list(l->data, ll);
      l = l->next;
   }

   /* Free memory used by the temporary list */
   destroy_list(lnew, NULL);

   /* Store the updated list */
   set_projectile_list(hero.current_room, ll);
}

static void update_particles(void)
{
   LINK *l;
   LINK *ll;
   LINK *il, *ill;
   LINK *lnew;
   PARTICLE *particle;
   int remove;
   int c;
   int *ldoor;
   int *rdoor;

   lnew = NULL;
   l = ll = get_particle_list(hero.current_room);
   while (l) {
      particle = l->data;
      remove = 0;
      switch (particle->type) {
         case PROJECTILE_ITEM:
            particle->anim++;
            if (particle->anim<12 && particle->anim&1)
               particle->y--;
            if (particle->anim>=40)
               remove = 1;
            break;
         case PROJECTILE_DEATH:
            particle->anim++;
            if (particle->anim>9) {
               remove = 1;
            }
            break;
         case PROJECTILE_BOSS_DEATH:
            particle->anim++;
            if (particle->anim>24) {
               remove = 1;
            }
            break;
         case PROJECTILE_BOSS_EXPLO:
            /* Create a spectacular explosion centered around the former boss' demise */
            particle->anim++;
            if (particle->anim == 30) {
               ITEM *item;
               
               /* Create heartcontainer */
               item = alloc_item();
               item->x = particle->x;
               item->y = particle->y;
               item->type = ITEM_HEARTC;
               item->lifetime = get_item_lifetime(item->type);
               item->anim = 0;

               /* Create child item: boss exit */
               /* This will then be automatically spawned when the container */
               /*  is picked up */
               item->child = alloc_item();
               item->child->x = 12*TILE_WIDTH;
               item->child->y = 5*TILE_HEIGHT;
               item->child->type = ITEM_BOSSEXIT;
               if (hero.level < 10) item->child->type = ITEM_EXIT;
               item->child->child = NULL;
               item->lifetime = get_item_lifetime(item->type);
               item->child->anim = 0;

               set_item_list(hero.current_room, add_to_list(item, get_item_list(hero.current_room)));
            }
            if (particle->anim<32 && (particle->anim&7)) {
               start_sample(10, FALSE);
               lnew = add_to_list(create_particle(PROJECTILE_BOSS_DEATH, particle->x+25-50*genrandf(), particle->y+25-50*genrandf(), 0,0), lnew);
            } else if (particle->anim>=32) {
               remove = 1;
            }
            break;
      }

      if (remove) {
         set_screen_changed();
         if (l==ll) {
            l = ll = remove_from_list(particle, l);
         } else {
            l = remove_from_list(particle, l);
         }
         free_particle(particle);
      } else {
         l = l->next;
      }
   }

   /* Add new items to the list */
   l=lnew;
   while(l) {
      ll = add_to_list(l->data, ll);
      l = l->next;
   }

   /* Free memory used by the temporary list */
   destroy_list(lnew, NULL);

   /* Store the updated list */
   set_particle_list(hero.current_room, ll);
}

static void update_items(void)
{
   LINK *l, *ll;
   ITEM *item;

   ll = l = get_item_list(hero.current_room);
   while (l) {
      item = l->data;
      item->anim++;
      if (item->lifetime) {
         item->lifetime--;

         if (item->lifetime == 0) {
            if (item->child) {
               /* This item had a child item - spawn that (reuse current */
               /*  item slot) */
               l->data = item->child;
               item->child = NULL;
               free_item(item);
               l = l->next;
            } else {
               if (l==ll) {
                  l = ll = remove_from_list(item, l);
               } else {
                  l = remove_from_list(item, l);
               }
               free_item(item);
            }
         } else {
            l = l->next;
         }
      } else {
         l = l->next;
      }
   }
   set_item_list(hero.current_room, ll);
}


#define select_next_weapon(player) _select_new_weapon(player, 1)
#define select_prev_weapon(player) _select_new_weapon(player, 3)

static int _select_new_weapon(HERO *player, int d)
{
   int weapons[4] = {ITEM_BOW|ITEM_CROSSBOW, ITEM_CLUB|ITEM_MACE, ITEM_DAGGER|ITEM_DAGGER2, ITEM_SWORD|ITEM_SWORD2};
   int c;
   int n;
   int res = FALSE;

   c = 0;

   if (player->weapon) {
      /* Find current */
      for (n=0; n<4; n++) {
         if (player->weapon&weapons[n])
            c = n;
      }

      /* Find new */
      for (n=0; n<4; n++) {
         if (player->inventory&weapons[(n+d+c)%4] && ((n+d+c)%4)!=c) {
            player->weapon = player->inventory&weapons[(n+d+c)%4];
            res = TRUE;
            break;
         }
      }
   } else {
      /* Find new */
      for (n=0; n<4; n++) {
         if (player->inventory&weapons[(n+d)%4]) {
            player->weapon = player->inventory&weapons[(n+d+c)%4];
            res = TRUE;
            break;
         }
      }
   }
   /* If not a power of two, deselect the weakest weapon */
   if (!is_power_of_two(player->weapon)) {
      player->weapon &= (ITEM_CROSSBOW|ITEM_MACE|ITEM_DAGGER2|ITEM_SWORD2);
   }

   return res;
}

static void update_hero(void)
{
   LINK *l;
   LINK *ll;
   ITEM *item;
   ITEM *teleporter;
   int *doors = NULL;
   PROJECTILE *pr;
   unsigned int old_dir;
   int old_action;
   int dmg;

   if (hero.select_timeout)
      hero.select_timeout--;

   if (hero.weapon_cooldown)
      hero.weapon_cooldown--;

   old_action = hero.action;
   old_dir = hero.direction;

   /* Mask actions we only want to execute if the key is pressed AND released */
   hero.action &= ~(ACTION_WALK|ACTION_BOMB|ACTION_DASH|ACTION_ACTIVATE);

   if (gamekey(0, GAMEKEY_LEFT)) {
      hero.direction = 3;
      hero.action |= ACTION_WALK;
   }
   if (gamekey(0, GAMEKEY_RIGHT)) {
      hero.direction = 1;
      hero.action |= ACTION_WALK;
   }
   if (gamekey(0, GAMEKEY_UP)) {
      hero.direction = 2;
      hero.action |= ACTION_WALK;
   }
   if (gamekey(0, GAMEKEY_DOWN)) {
      hero.direction = 0;
      hero.action |= ACTION_WALK;
   }

   if (gamekey(0, GAMEKEY_ATTACK) && hero.weapon_buildup<2*hero.max_cooldown && !hero.weapon_cooldown) {
       hero.weapon_buildup++;
   }

   if (!hero.weapon_cooldown) {
      if (gamekey(0, GAMEKEY_ATTACK)) {
         hero.action |= ACTION_TARGET;
         hero.direction = old_dir;
      } else if (hero.action & ACTION_TARGET) {
         hero.action &= ~(ACTION_TARGET|ACTION_PARRY);
         hero.direction = old_dir;
         flush_gamekey_buffer(0);
      }
   }

   if (gamekey(0, GAMEKEY_CENTRE)) {
      hero.target_x = hero.target_y = 0;
      hero.target = NULL;
   }

   if (gamekey(0, GAMEKEY_BOMB) && hero.bombs) {
      hero.action|=ACTION_BOMB;
   }

   if (gamekey(0, GAMEKEY_ACT)) {
      hero.action|=ACTION_DASH;
      hero.action|=ACTION_ACTIVATE;
   }

   /* Fire projectile */
   /* Each weapon is treated differntly: */
   /*  The bow will fire only when the target is set and the button released */
   /*  The club will swing when the button is released; its power depends on how long the button was held down */
   /*  The dagger will strike immediately and only once */
   /*  The sword will be held ready while the button is pressed (a la Zelda) */

   /* Was pressed, released now */
   if (old_action&ACTION_TARGET && !(hero.action&ACTION_TARGET)) {
      /* Fire bow, swing mace or put down sword */
      if (hero.weapon & (ITEM_BOW|ITEM_CROSSBOW)) {
         if (hero.arrows) {
            int item;
            /* Calculate potential damage of the projectile (depends on upgrades) */
            dmg = calc_weapon_damage(hero.inventory, hero.weapon);
            /* Arrows fired from a crossbow are faster */
            if (hero.weapon & ITEM_CROSSBOW)
               item = PROJECTILE_CROSSARROW;
            else
               item = PROJECTILE_ARROW;
               
            hero.weapon_cooldown = BOW_COOLDOWN;
            /* Offset slightly so that the arrow will fly in the direction the hero */
            /* is facing if the target is centered */
            pr = fire_projectile(item, dmg, hero.x+(hero.direction&1?hero.direction-2:0), hero.y+(hero.direction&1?0:hero.direction-1), hero.x+hero.target_x, hero.y+hero.target_y);
            set_projectile_list(hero.current_room, add_to_list(pr, get_projectile_list(hero.current_room)));
            hero.arrows--;
            start_sample(2, FALSE);
         } else {
            start_sample(12, FALSE);
         }
      } else if (hero.weapon & (ITEM_MACE|ITEM_CLUB)) {
         hero.weapon_cooldown = MACE_COOLDOWN;
         dmg = calc_weapon_damage(hero.inventory, hero.weapon);
         pr = fire_projectile(PROJECTILE_CLUB, dmg, 
                                 hero.x+(hero.direction&1?hero.direction-2:0), 
                                 hero.y+(hero.direction&1?0:hero.direction-1), 
                                 hero.x+hero.target_x, hero.y+hero.target_y);
         set_projectile_list(hero.current_room, add_to_list(pr, get_projectile_list(hero.current_room)));
      } else if (hero.weapon & (ITEM_SWORD|ITEM_SWORD2)) {
         /* Remove sword projectile from the list */
         hero.weapon_cooldown = SWORD_COOLDOWN;
         hero.action &= ~ACTION_PARRY;
         for (l = get_projectile_list(hero.current_room); l; l = l->next) {
            pr = l->data;
            if (pr->type == PROJECTILE_SWORD)
               pr->anim = 1;
         }
      }
   }

   /* Pressed now, but not previously */
   if (!(old_action&ACTION_TARGET) && (hero.action&ACTION_TARGET)) {
      /* Swing sword or stab with dagger */
      if (hero.weapon & (ITEM_SWORD|ITEM_SWORD2)) {
         hero.action |= ACTION_PARRY;
         dmg = calc_weapon_damage(hero.inventory, hero.weapon);
         pr = fire_projectile(PROJECTILE_SWORD, dmg, 
                                 hero.x+(hero.direction&1?hero.direction-2:0), 
                                 hero.y+(hero.direction&1?0:hero.direction-1),
                                 hero.x, hero.y);
         set_projectile_list(hero.current_room, add_to_list(pr, get_projectile_list(hero.current_room)));
         start_sample(15, FALSE);
      } else if (hero.weapon & (ITEM_DAGGER|ITEM_DAGGER2)) {
         hero.weapon_cooldown = DAGGER_COOLDOWN;
         dmg = calc_weapon_damage(hero.inventory, hero.weapon);
         pr = fire_projectile(PROJECTILE_DAGGER, dmg, 
                                 hero.x+(hero.direction&1?hero.direction-2:0),
                                 hero.y+(hero.direction&1?0:hero.direction-1),
                                 hero.x+hero.target_x, hero.y+hero.target_y);
         set_projectile_list(hero.current_room, add_to_list(pr, get_projectile_list(hero.current_room)));
      }
   }

   if (hero.action&ACTION_TARGET) {
      if (hero.weapon & (ITEM_BOW|ITEM_CROSSBOW)) {
         /* Use Tell's Ring if appropriate */
         if (hero.target) {
            hero.target_y = hero.target->y-hero.y;
            hero.target_x = hero.target->x-hero.x;
         }
      }
   }
   /* Set bomb */
   if (old_action&ACTION_BOMB && !(hero.action&ACTION_BOMB) && hero.bombs) {
      /* Calculate damage dealt by the resulting explosion */
      dmg = st_dmg_impact(pack_dmg(1,2))|st_dmg_pierce(pack_dmg(1,2))|st_dmg_elem(pack_dmg(1,4));
      pr = fire_projectile(PROJECTILE_BOMB, dmg, hero.x+8, hero.y+12, 0, 0);
      set_projectile_list(hero.current_room, add_to_list(pr, get_projectile_list(hero.current_room)));
      hero.bombs--;
   }

   if (hero.action & ACTION_WALK) {
      if (old_dir == hero.direction) {
         /* Walk more slowly when firing a bow */
         if (hero.action & ACTION_DASH && !(hero.action & ACTION_TARGET && hero.weapon & (ITEM_BOW|ITEM_CROSSBOW))) {
            if (hero.velocity<5)
               hero.velocity++;
         } else {
            if (hero.velocity>3)
               hero.velocity--;
         }
         if (hero.velocity<3)
            hero.velocity++;
         if (hero.velocity<2)
            hero.velocity++;
         if (hero.velocity<1.25)
            hero.velocity++;

         doors = NULL;

         if (gamekey(0, GAMEKEY_DOWN)) {
            hero.dy += hero.velocity;
         }
         if (gamekey(0, GAMEKEY_RIGHT)) {
            hero.dx += hero.velocity;
            doors = get_right_doors(hero.current_room);
         }
         if (gamekey(0, GAMEKEY_UP)) {
            hero.dy += -hero.velocity;
         }
         if (gamekey(0, GAMEKEY_LEFT)) {
            hero.dx += -hero.velocity;
            doors = get_left_doors(hero.current_room);
         }
         
      }
   } else {
      hero.velocity = 0.25;
   }

   /* Check for unpassable tiles */   
   /*
   if (!tile_passable(hero.current_room, hero.x + hero.dx, hero.y + hero.dy)) {
      if (tile_passable(hero.current_room, hero.x, hero.y + hero.dy)) hero.dx = 0;
      if (tile_passable(hero.current_room, hero.x + hero.dx, hero.y)) hero.dy = 0;
      if (!tile_passable(hero.current_room, hero.x + hero.dx, hero.y + hero.dy)) {
         hero.dx = 0;
         hero.dy = 0;
      }
   }
   */
   if (collide_geometry(hero.current_room, hero.x+hero.dx, hero.y+hero.dy, &move_bb)) {
      if (!collide_geometry(hero.current_room, hero.x, hero.y+hero.dy, &move_bb)) hero.dx = 0;
      if (!collide_geometry(hero.current_room, hero.x+hero.dx, hero.y, &move_bb)) hero.dy = 0;
      if (collide_geometry(hero.current_room, hero.x+hero.dx, hero.y+hero.dy, &move_bb)) {
         hero.dx = 0;
         hero.dy = 0;
      }
   }

   if (global_cycles_passed%5==0) {
      if (hero.action & ACTION_WALK) {
         hero.anim++;
         hero.anim%=4;
      } else {
         hero.anim=0;
      }
   }

   /* Check for overlap with items */
   teleporter = NULL;
   l = ll = get_item_list(hero.current_room);
   while (l) {
      item = l->data;
      
      if (hero_touches_item(&hero, item)) {
         /* Check for item types that block */
         if (item->type == ITEM_CHESTCLOSED) {
            int tdx = 0, tdy = 0;
            
            /* Open locked chest */
            if (hero_can_open_chest(&hero, item) && 
                hero.action & ACTION_ACTIVATE) {
               ITEM *child = item->child;
               PARTICLE *particle;
               
               start_sample(get_item_sound(item->type), FALSE);
               item->child = NULL;
               item->type = ITEM_CHESTOPEN;

               /* Display the item as a particle over the player's head */
               particle = create_particle(PROJECTILE_ITEM,
                  item->x + TILE_SIZE/3, item->y - TILE_SIZE/2, 
                  0,0);
               particle->sub_type = child->type;
               set_particle_list(hero.current_room, add_to_list(particle, get_particle_list(hero.current_room)));
               
               set_screen_changed();
               could_pickup_item(&hero, child->type);
               free_item(child);
            }

            tdx = hero.dx;
            tdy = hero.dy;

            hero.dx = 0;
            if (!hero_touches_item(&hero, item)) {
               tdx = 0;
            } else {
               tdy = 0;
            }
            hero.dx = tdx;
            hero.dy = tdy;

            l = l->next;
         } else if (item->type == ITEM_EXIT || item->type == ITEM_BOSSEXIT || item->type == ITEM_BOSSGATE) {
            if (hero.y<=item->y || !could_pickup_item(&hero, item->type)) {
               hero.dx = 0;
               hero.dy = 0;
            }
            l = l->next;
         } else if (item->type == ITEM_LOCKEDTELEP && doors) {
            int y = ((hero.y+TILE_HEIGHT/2+12)/TILE_HEIGHT-8)/15;
            
            if (doors[y] == MAP_DOOR_OPEN) {
               item->type = ITEM_TELEPORTER;
            } else {
               hero.dx = 0;
               hero.dy = 0;
               hero.velocity = 0;
            
               /* Open the locked door */
               if (hero.action & ACTION_ACTIVATE && hero.keys) {
               
                  doors[y] = MAP_DOOR_OPEN;
                  redraw_tilemap = TRUE;
                  hero.keys--;
                  start_sample(7, FALSE);
               
                  item->type = ITEM_TELEPORTER;
               }
            }
            l = l->next;
         } else if (item->type == ITEM_TELEPORTER) {
            teleporter = item;
            l = l->next;
         } else if (item->type == ITEM_BLOCKLOCKED) {
            int tdx = hero.dx;
            int tdy = hero.dy;

            hero.dx = 0;
            if (!hero_touches_item(&hero, item)) {
               tdx = 0;
            } else {
               tdy = 0;
            }
            hero.dx = tdx;
            hero.dy = tdy;

            if (hero.action & ACTION_ACTIVATE && hero.keys) {
               start_sample(get_item_sound(item->type), FALSE);
               
               if (item->child) {
                  l->data = item->child;
                  l = l->next;
                  item->child = NULL;
               } else {
                  if (l==ll) {
                     l = ll = remove_from_list(item, l);
                  } else {
                     l = remove_from_list(item, l);
                  }
               }
               free_item(item);
               hero.keys--;
            }
         } else if (item_blocks(item->type)) {
            int tdx = hero.dx;
            int tdy = hero.dy;

            hero.dx = 0;
            if (!hero_touches_item(&hero, item)) {
               tdx = 0;
            } else {
               tdy = 0;
            }
            hero.dx = tdx;
            hero.dy = tdy;
            l = l->next;
         //} else if (distsqr(hero.x, hero.y, item->x, item->y)<TILE_SIZE*TILE_SIZE/4 && 
         //           could_pickup_item(&hero, item->type)) {
         } else if (collide_box(hero.x, hero.y, &pickup_bb, item->x, item->y,
                                 get_item_bbox(item->type)) && 
                    could_pickup_item(&hero, item->type)) {
            set_screen_changed();
            if (item->child) {
               /* This item had a child item - spawn that (reuse current */
               /*  item slot) */
               l->data = item->child;
               item->child = NULL;
               free_item(item);
               l = l->next;
            } else {
               if (l==ll) {
                  l = ll = remove_from_list(item, l);
               } else {
                  l = remove_from_list(item, l);
               }
               free_item(item);
            }
         } else {
            if (item->type == ITEM_SWITCHWGHTU) {
               could_pickup_item(&hero, item->type);
               item->type = ITEM_SWITCHWGHTD;
            }
            l = l->next;
         }
      } else {
         if (item->type == ITEM_SWITCHWGHTD) {
            item->type = ITEM_SWITCHWGHTU;
            could_pickup_item(&hero, item->type);
         }
         l = l->next;
      }
   }
   set_item_list(hero.current_room, ll);

   /* Update position and movement angle */
   hero.x += hero.dx;
   hero.y += hero.dy;
   if (hero.dx || hero.dy)
      hero.angle = atan2(hero.dy, hero.dx);

   if (teleporter) {   
      /* Pass through a teleporter */
      /* Remove projectiles from the room */
      destroy_list(get_projectile_list(hero.current_room), (void*)free_projectile);
      set_projectile_list(hero.current_room, NULL);
      
      destroy_list(get_particle_list(hero.current_room), (void*)free_particle);
      set_particle_list(hero.current_room, NULL);

      hero.target = NULL;
      hero.target_x = 0;
      hero.target_y = 0;

      hero.current_room = teleporter->troom;
      hero.x = teleporter->tx;
      hero.y = teleporter->ty;

      redraw_tilemap = TRUE;
      teleporter = NULL;
   }
   
   if (hero.next_level) {
      next_level();
      game_state = GAME_STATE_PASSWORD;
   }

   if (load_boss_level) {
      load_boss_level = FALSE;
      is_boss_level = TRUE;
      hero.target_x = 0;
      hero.target_y = 0;
      hero.target = NULL;
      hero.next_level = FALSE;
      hero.select_timeout = 0;
      hero.weapon_cooldown = 0;
      hero.max_cooldown = 0;

      hero.direction = 0;
      hero.angle = M_PI/2;
      hero.anim = 0;
      hero.invc = 0;

      unload_map();
      boss_level_id = -(int)(hero.level)/8 - 1;
      load_level(boss_level_id);
      hero.current_room = get_start_room();
      hero.x = TILE_WIDTH*get_start_x();
      hero.y = TILE_HEIGHT*get_start_y();
      hero.dx = 0;
      hero.dy = 0;
      redraw_tilemap = TRUE;
      stop_music();
      set_music_playlist(boss_playlist);
   }

   /* Reset hero dx/dy to 0 */
   hero.dx = 0;
   hero.dy = 0;
}

static void game_logic_update(void)
{
   /* Move projectiles */
   update_projectiles();
   update_particles();
   
   /* Graphical effects on items */
   update_items();

   /* Move monsters */
   if (!is_boss_level)
      update_monsters();
   else
      update_boss();

   /* Check for player/monster collisions */
   collide_monster(&hero);

   /* Handle the hero */
   if (!(game_state&(GAME_STATE_GAME_OVER|GAME_STATE_PAUSEGAME)))
      update_hero();

   /* And check for game-over */
   if (hero.health<=0) {
      if (hero.inventory&ITEM_RINGREGEN) {
         hero.health = hero.max_health&(~7);
         hero.inventory^=ITEM_RINGREGEN;
         hero.invc = 45;
         start_sample(9, FALSE);
      } else if  (game_state!=GAME_STATE_GAME_OVER) {
         game_state=GAME_STATE_GAME_OVER;
         game_state_counter=30;
         start_sample(10, FALSE);
         set_particle_list(hero.current_room, add_to_list(create_particle(PROJECTILE_DEATH, hero.x, hero.y, 0,0), get_particle_list(hero.current_room)));
      }
   }

   set_screen_changed();
}

inline void game_state_update(void)
{
   TRACE("Updating gamestate [statecounter = %d, state = %d]...\n", game_state_counter, game_state);

   /*
   printf("%8d%8d%8d%8d%8d%8d%8d\n", get_bshape_count(),
         get_render_sprite_count(), get_item_count(), get_projectile_count(),
         get_monster_count(), get_particle_count(), get_link_count());
   */

   /* What we should do during an update cycle depends on the state the game */
   /*  is currently in */
   if (game_state == GAME_STATE_NORMAL) {
      TRACE("\tCycling game music...\n");
      cycle_game_music();
      TRACE("\tUpdating logic...\n");
      game_logic_update();
   } else if (game_state == GAME_STATE_GAME_OVER) {
      cycle_game_music();
      if (game_state_counter>0) {
         game_logic_update();
         game_state_counter--;
         if (!game_state_counter) {
            stop_music();
            set_music_playlist(game_over_playlist);
         }
      }
   } else if (game_state & (GAME_STATE_PASSWORD|GAME_STATE_SHOW_SCORES)){
      if (game_state_counter) {
         game_state_counter--;
      }
   } else if (game_state == GAME_STATE_GET_HIGHSCORE) {
      if (game_state_counter>1) {
         game_state_counter--;
      }
      if (!game_state_counter) {
         char name[128];
         char s[256];

         clear_bitmap(get_monitor_bmp());
         select_palette((*get_font_palette()));
         textprintf_centre(get_monitor_bmp(), get_colour_font(), SCREEN_W/2, 64, -1, "Enter your name");
         get_name(name);
         if (settings.difficulty) {
            snprintf(s, 256, "%s (%s)", name, (settings.difficulty==1)?"Hard":"Hell");
         } else {
            snprintf(s, 256, "%s", name);
         }
         add_highscore_list(s, hero.score);

         game_state = GAME_STATE_SHOW_SCORES;
         game_state_counter = 40;
         set_screen_changed();
         synchronize();
      }
   } else if (game_state & GAME_STATE_PAUSEGAME) {
      cycle_game_music();
      set_screen_changed();
   } else {
      if (game_state_counter) {
         game_state_counter--;
      }
      if (!game_state_counter && (game_state == GAME_STATE_NEW_LEVEL)) {
         game_state = GAME_STATE_NORMAL;
         TRACE("\tMenu wait time over, starting normal play\n");
      }
   }

   if (game_state_counter<0)
      game_state_counter = 0;

   TRACE("Updatecycle completed [statecounter = %d, state = %d]...\n", game_state_counter, game_state);
}

inline void game_input(void)
{
   TRACE("Starting input cycle.\n");
   TRACE("Reading controller input...\n");
   poll_controller(0);

   TRACE("\tTracking input [game_state = %d]\n", game_state);
   /* The way the game responds to the controller depends on the game state */
   if (game_state == GAME_STATE_NORMAL) {
      if (gamekey_pressed(0)) {
         if (gamekey(0, GAMEKEY_NEXT) && !hero.select_timeout) {
            if (hero.action & ACTION_TARGET){
               if (hero.weapon & (ITEM_BOW|ITEM_CROSSBOW) && hero.inventory&(ITEM_TELLRING)) {
                  LINK *l;
                  MONSTER *m;
                  
                  if (hero.target) {
                     for (l=get_monster_list(hero.current_room);l;l=l->next) {
                        if (l->data==hero.target)
                           break;
                     }
                     if (l && l->next) {
                        for (l=l->next; l; l=l->next) {
                           m = l->data;
                           if (in_rect(m->x, m->y, top_x, top_y, top_x+SCREEN_W, top_y+SCREEN_W)) {
                              hero.target = m;
                              hero.target_x = m->x-hero.x;
                              hero.target_y = m->y-hero.y;
                              break;
                           }
                        }
                     } else {
                        hero.target = NULL;
                        hero.target_x = 0;
                        hero.target_y = 0;
                     }
                  } else {
                     for (l=get_monster_list(hero.current_room); l; l=l->next) {
                        m = l->data;
                        if (in_rect(m->x, m->y, top_x, top_y, top_x+SCREEN_W, top_y+SCREEN_W)) {
                           hero.target = m;
                           hero.target_x = m->x-hero.x;
                           hero.target_y = m->y-hero.y;
                           break;
                        }
                     }
                  }
               }
            } else if (!select_next_weapon(&hero)) {
               start_sample(12, FALSE);
            }
            hero.select_timeout = 15;
         }

         if (gamekey(0, GAMEKEY_PREV) && !hero.select_timeout) {
            if (hero.action & ACTION_TARGET){
               if (hero.weapon & (ITEM_BOW|ITEM_CROSSBOW) && hero.inventory&(ITEM_TELLRING)) {
                  LINK *l;
                  MONSTER *m;
                  for (l=get_monster_list(hero.current_room);l;l=l->next) {
                     m = l->data;
                     if (l->data==hero.target) {
                        break;
                     }
                     if (in_rect(m->x, m->y, top_x, top_y, top_x+SCREEN_W, top_y+SCREEN_W)) {
                        hero.target = m;
                        hero.target_x = m->x-hero.x;
                        hero.target_y = m->y-hero.y;
                     }
                  }
               }
            } else if (!select_prev_weapon(&hero)) {
               start_sample(12, FALSE);
            }
            hero.select_timeout = 15;
         }

         if (hero.weapon &(ITEM_BOW|ITEM_CROSSBOW))
            hero.max_cooldown = BOW_COOLDOWN;
         if (hero.weapon &(ITEM_SWORD|ITEM_SWORD2))
            hero.max_cooldown = SWORD_COOLDOWN;
         if (hero.weapon &(ITEM_DAGGER|ITEM_DAGGER2))
            hero.max_cooldown = DAGGER_COOLDOWN;
         if (hero.weapon &(ITEM_MACE|ITEM_CLUB))
            hero.max_cooldown = MACE_COOLDOWN;

         if (gamekey(0, GAMEKEY_ATTACK))
            push_gamekey(0, GAMEKEY_ATTACK);
      }

      if (gamekey_released(0, GAMEKEY_SELECT)) {
         game_state|=GAME_STATE_PAUSEGAME|GAME_STATE_EXITMENU;
         set_screen_changed();
         synchronize();
      }
      if (gamekey_released(0, GAMEKEY_START)) {
         game_state|=GAME_STATE_PAUSEGAME|GAME_STATE_ITEMMENU;
         set_screen_changed();
         synchronize();
      }
   } else if (game_state&GAME_STATE_PAUSEGAME) {

      if (gamekey_released(0, GAMEKEY_LEFT)) {
         game_state_options = (game_state_options+23) % 24;
      }
      if (gamekey_released(0, GAMEKEY_RIGHT)) {
         game_state_options = (game_state_options+1) % 24;
      }
      if (gamekey_released(0, GAMEKEY_UP)) {
         game_state_options = (game_state_options+18) % 24;
      }
      if (gamekey_released(0, GAMEKEY_DOWN)) {
         game_state_options = (game_state_options+6) % 24;
      }

      /* Actions depend on why the game was paused */
      if (game_state&GAME_STATE_EXITMENU) {
         /* Exit the game? */
         if ((gamekey_released(0, GAMEKEY_START)||gamekey_released(0, GAMEKEY_ATTACK))) {
            game_state^=GAME_STATE_PAUSEGAME|GAME_STATE_EXITMENU;
            set_screen_changed();
            /* Exit if the option `quit' was chosen */
            if (game_state_options&1)
               set_game_done();
         }
         if (gamekey_released(0, GAMEKEY_SELECT)) {
            game_state^=GAME_STATE_PAUSEGAME|GAME_STATE_EXITMENU;
            set_screen_changed();
            /* Cancel */
            game_state_options=0;
         }
         game_state_options &= 1;

      } else if (game_state&GAME_STATE_ITEMMENU) {

         /* Show item info */
         if (gamekey_released(0, GAMEKEY_START)) {
            game_state^=GAME_STATE_PAUSEGAME|GAME_STATE_ITEMMENU;
            set_screen_changed();
            /* Cancel */
            game_state_options=0;
         }
      }
   } else if (game_state == GAME_STATE_GAME_OVER && !game_state_counter) {
      if (gamekey_pressed(0)) {
         //popup_game_save_password_dialog();
         if (is_highscore(hero.score)) {
            game_state = GAME_STATE_GET_HIGHSCORE;
            game_state_counter = 40;
            set_screen_changed();
         } else {
            set_game_done();
         }
      }
   } else if (game_state == GAME_STATE_WIN && !game_state_counter) {
      if (gamekey_pressed(0)) {
         if (is_highscore(hero.score)) {
            game_state = GAME_STATE_GET_HIGHSCORE;
            game_state_counter = 40;
            set_screen_changed();
         } else {
            set_game_done();
         }
      }
   } else if (game_state == GAME_STATE_GET_HIGHSCORE) {
      if (gamekey_pressed(0)  && game_state_counter==1) {
         game_state_counter = 0;
      }
   } else if (game_state == GAME_STATE_SHOW_SCORES && !game_state_counter) {
      if (gamekey_pressed(0)) {
         set_game_done();
      }
   } else if (game_state == GAME_STATE_PASSWORD) {
      if (gamekey_pressed(0) && !game_state_counter) {
         game_state = GAME_STATE_NEW_LEVEL;
         game_state_counter = 60;
         set_screen_changed();
      }
   }

   /* Save screen shot */
   if (key[KEY_PRTSCR]||key[KEY_F12])
      screenshot();

   /* Keep <esc> as a general exit key, just to be save */
   if (key[KEY_ESC]) {
      set_game_done();
      TRACE("\t<ESC> - Shutting down the game, returning to menu\n");
   }

#ifdef ENALE_CHEAT_KEYS
   if (key[KEY_F11]) {
      load_boss_level = TRUE;
   }

   if (key[KEY_F10]) {
      hero.bombs = 12;
      hero.arrows = MAX_ARROWS;
      hero.keys = 4;
      hero.health = hero.max_health = 3*8 + (hero.level*2)*2;
      hero.inventory = -1;
   }

   if (key[KEY_F9]) {
      hero.score += 100;
      set_screen_changed();
   }

   if (key[KEY_F8]) {
      toggle_switch_state();
      redraw_tilemap = TRUE;
   }

   if (key[KEY_F7]) {
      hero.def = -1;
   }
#endif /* ENALE_CHEAT_KEYS */

   /* Check the closebutton */
   if (settings.window_close_button) {
      set_game_done();
   }
   
   /* Save current state for later comparison */
   push_gamekey_states();
   TRACE("Input update completed\n");
}

void start_game(void)
{
   srand(time(NULL));

   show_opening_animation();
   
   set_display_switch_callback(SWITCH_IN, display_switchin);

   while(game_menu()) {
      playgame();
   }

   remove_display_switch_callback(display_switchin);
}
