/*  TA3D, a remake of Total Annihilation
    Copyright (C) 2005   Roland BROCHARD

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA*/

/*----------------------------------------------------------\
|                      EngineClass.cpp                      |
|    Contient toutes les classes nécessaires au moteur 3D   |
| et au moteur physique.                                    |
|                                                           |
\----------------------------------------------------------*/

#include <allegro.h>
#include <alleggl.h>
#include <GL/glu.h>
#include "matrix.h"
#include "ta3dbase.h"
#include "3do.h"					// Pour la lecture des fichiers 3D
#include "cob.h"					// Pour la lecture et l'éxecution des scripts
#include "tdf.h"					// Pour la gestion des éléments du jeu
#include "weapons.h"				// Pour la gestion des armes
#include "fbi.h"					// Pour la gestion des unités
#include "EngineClass.h"

float	player_color[30]={	0.0f,	0.0f,	1.0f,
							1.0f,	0.0f,	0.0f,
							0.0f,	1.0f,	0.0f,
							1.0f,	1.0f,	0.0f,
							1.0f,	0.0f,	1.0f,
							0.5f,	0.5f,	0.5f,
							0.5f,	0.5f,	1.0f,
							1.0f,	1.0f,	1.0f,
							0.0f,	0.0f,	0.0f,
							0.0f,	1.0f,	1.0f };
INGAME_UNITS units;

int NB_PLAYERS;

	void MAP::draw_mini(int x1,int y1,int w,int h,CAMERA *cam)			// Dessine la mini-carte
	{
		if(!mini) return;		// Vérifie si la mini-carte existe
		int rw=w*mini_w/252;
		int rh=h*mini_h/252;
		x1+=w-rw>>1;
		y1+=h-rh>>1;
		float lw=mini_w/252.0f;
		float lh=mini_h/252.0f;
		glEnable(GL_TEXTURE_2D);
		glBindTexture(GL_TEXTURE_2D,glmini);
		glBegin(GL_QUADS);
			glTexCoord2f(0.0f,0.0f);	glVertex2f(x1,y1);
			glTexCoord2f(lw,0.0f);		glVertex2f(x1+rw,y1);
			glTexCoord2f(lw,lh);		glVertex2f(x1+rw,y1+rh);
			glTexCoord2f(0.0f,lh);		glVertex2f(x1,y1+rh);
		glEnd();

		if(!cam) return;

		VECTOR A,B,C,D,P;
		A=cam->Dir+1.0f*(-cam->Side-0.75f*cam->Up);
		B=cam->Dir+1.0f*(cam->Side-0.75f*cam->Up);
		C=cam->Dir+1.0f*(cam->Side+0.75f*cam->Up);
		D=cam->Dir+1.0f*(-cam->Side+0.75f*cam->Up);
		const int nmax=64;
		float cx[4*nmax+4],cy[4*nmax+4];
		if(A.y<0.0f) {
			P=cam->Pos+cam->Pos.y/fabs(A.y)*A;	cx[0]=P.x;	cy[0]=P.z; }
		else {
			P=cam->Pos+10000.0f*A;	cx[0]=P.x;	cy[0]=P.z; }
		if(B.y<0.0f) {
			P=cam->Pos+cam->Pos.y/fabs(B.y)*B;	cx[1]=P.x;	cy[1]=P.z; }
		else {
			P=cam->Pos+10000.0f*B;	cx[1]=P.x;	cy[1]=P.z; }
		if(C.y<0.0f) {
			P=cam->Pos+cam->Pos.y/fabs(C.y)*C;	cx[2]=P.x;	cy[2]=P.z; }
		else {
			P=cam->Pos+10000.0f*C;	cx[2]=P.x;	cy[2]=P.z; }
		if(D.y<0.0f) {
			P=cam->Pos+cam->Pos.y/fabs(D.y)*D;	cx[3]=P.x;	cy[3]=P.z; }
		else {
			P=cam->Pos+10000.0f*D;	cx[3]=P.x;	cy[3]=P.z; }

		int i;
		for(i=0;i<4;i++) {
			cx[i]=(cx[i]+0.5f*map_w)*rw/map_w;
			cy[i]=(cy[i]+0.5f*map_h)*rh/map_h;
			}
		for(i=0;i<4;i++) {
			for(int e=0;e<nmax;e++) {
				cx[i*nmax+e+4]=(cx[i]*(nmax-e)+cx[(i+1)%4]*(e+1))/(nmax+1);
				cy[i*nmax+e+4]=(cy[i]*(nmax-e)+cy[(i+1)%4]*(e+1))/(nmax+1);
				}
			}
		for(i=0;i<4+(nmax<<2);i++) {
			if(cx[i]<0.0f) cx[i]=0.0f;
			else if(cx[i]>rw) cx[i]=rw;
			if(cy[i]<0.0f) cy[i]=0.0f;
			else if(cy[i]>rh) cy[i]=rh;
			}

		glDisable(GL_TEXTURE_2D);
		glColor3f(1.0f,1.0f,1.0f);
		glBegin(GL_LINE_STRIP);
			for(i=0;i<4;i++) {
				glVertex2f(cx[i]+x1,cy[i]+y1);
				for(int e=0;e<nmax;e++)
					glVertex2f(x1+cx[i*nmax+e+4],y1+cy[i*nmax+e+4]);
				}
			glVertex2f(cx[0]+x1,cy[0]+y1);
		glEnd();
		glEnable(GL_TEXTURE_2D);
	}

	void MAP::draw(CAMERA *cam,bool FLAT,float niv,float t,float dt)
	{
		cam->SetView();
		if(FLAT)
			glTranslatef(0.0f,0.0f,sea_dec);
		int i,x,y;
		glDisable(GL_CULL_FACE);
		glDisable(GL_LIGHTING);
		if(ntex>0) {
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
			for(i=0;i<8;i++) {
				glActiveTextureARB(GL_TEXTURE0_ARB+i);
				ReInitTexSys();
				glDisable(GL_TEXTURE_2D);
				}
			}
		else
			glDisableClientState(GL_TEXTURE_COORD_ARRAY);
		VECTOR A,B,C,D,P;
		A=cam->Dir+1.0f*(-cam->Side-0.75f*cam->Up);
		B=cam->Dir+1.0f*(cam->Side-0.75f*cam->Up);
		C=cam->Dir+1.0f*(cam->Side+0.75f*cam->Up);
		D=cam->Dir+1.0f*(-cam->Side+0.75f*cam->Up);
		float cx[4],cy[4];
		if(A.y<0.0f) {
			P=cam->Pos+cam->Pos.y/fabs(A.y)*A;	cx[0]=P.x;	cy[0]=P.z; }
		else {
			P=cam->Pos+10000.0f*A;	cx[0]=P.x;	cy[0]=P.z; }
		if(B.y<0.0f) {
			P=cam->Pos+cam->Pos.y/fabs(B.y)*B;	cx[1]=P.x;	cy[1]=P.z; }
		else {
			P=cam->Pos+10000.0f*B;	cx[1]=P.x;	cy[1]=P.z; }
		if(C.y<0.0f) {
			P=cam->Pos+cam->Pos.y/fabs(C.y)*C;	cx[2]=P.x;	cy[2]=P.z; }
		else {
			P=cam->Pos+10000.0f*C;	cx[2]=P.x;	cy[2]=P.z; }
		if(D.y<0.0f) {
			P=cam->Pos+cam->Pos.y/fabs(D.y)*D;	cx[3]=P.x;	cy[3]=P.z; }
		else {
			P=cam->Pos+10000.0f*D;	cx[3]=P.x;	cy[3]=P.z; }

		int minx=cx[0],maxx=cx[0];
		int miny=cy[0],maxy=cy[0];
		for(i=0;i<4;i++) {
			cx[i]=(cx[i]+0.5f*map_w)/16.0f;
			cy[i]=(cy[i]+0.5f*map_h)/16.0f;
			if(cx[i]<minx) minx=cx[i];
			if(cx[i]>maxx) maxx=cx[i];
			if(cy[i]<miny) miny=cy[i];
			if(cy[i]>maxy) maxy=cy[i];
			}
		int y1=bloc_h,y2=0,x1=bloc_w,x2=0,mx,my;
		float limit=cam->zfar*sqrt(2.0f);
		x1=(cam->Pos.x+0.5f*map_w-limit)/16.0f;
		y1=(cam->Pos.z+0.5f*map_h-limit)/16.0f;
		x2=(cam->Pos.x+0.5f*map_w+limit)/16.0f;
		y2=(cam->Pos.z+0.5f*map_h+limit)/16.0f;
		mx=(cam->Pos.x+0.5f*map_w)/16.0f;
		my=(cam->Pos.z+0.5f*map_h)/16.0f;
		if(x1<minx)
			x1=minx;
		if(x2>maxx)
			x2=maxx;
		if(y1<miny)
			y1=miny;
		if(y2>maxy)
			y2=maxy;
		if(x1>mx) x1=mx;
		if(y1>my) y1=my;
		if(x2<mx) x2=mx;
		if(y2<my) y2=my;
		if(y1<0) y1=0;
		if(y2>=bloc_h) y2=bloc_h-1;
		if(x1<0) x1=0;
		if(x2>=bloc_w) x2=bloc_w-1;

		A=(cam->Dir+1.0f*(0.75f*cam->Up-cam->Side));	A.Unit();
		float ref=0.8f*(A%cam->Dir);
		float dhm=0.5f*map_h;
		float dwm=0.5f*map_w;

		if(!FLAT)
			glColor4f(1.0f,1.0f,1.0f,1.0f);

		POINTF flat[9];
		if(FLAT) {
			flat[0].x=0.0f;		flat[0].y=niv+cos(t)*0.5f;					flat[0].z=0.0f;
			flat[1].x=8.0f;		flat[1].y=niv+cos(t+1.0f)*0.5f;				flat[1].z=0.0f;
			flat[2].x=16.0f;	flat[2].y=flat[0].y;						flat[2].z=0.0f;
			flat[3].x=0.0f;		flat[3].y=niv+cos(t+1.5f)*0.5f;				flat[3].z=8.0f;
			flat[4].x=8.0f;		flat[4].y=niv+cos(t+2.5f)*0.5f;				flat[4].z=8.0f;
			flat[5].x=16.0f;	flat[5].y=flat[3].y;						flat[5].z=8.0f;
			flat[6].x=0.0f;		flat[6].y=flat[0].y;						flat[6].z=16.0f;
			flat[7].x=8.0f;		flat[7].y=flat[1].y;						flat[7].z=16.0f;
			flat[8].x=16.0f;	flat[8].y=flat[0].y;						flat[8].z=16.0f;
			}

		if(ntex>0) {
			glActiveTextureARB(GL_TEXTURE0_ARB);
			glEnable(GL_TEXTURE_2D);
			glClientActiveTextureARB(GL_TEXTURE0_ARB);
			}
		glEnableClientState(GL_VERTEX_ARRAY);		// Les sommets

		if(FLAT) {
			glTranslatef(cos(t),0.0f,sin(t));
			glVertexPointer( 3, GL_FLOAT, 0, flat);
			}
		int old_i=-1;
		GLuint old_tex=bloc[0].tex;
		glBindTexture(GL_TEXTURE_2D,old_tex);
		for(y=y1;y<=y2;y++) {
			int pre_y=y<<4;
			int Y=y<<1;
			int pre_y2=y*bloc_w;
			glPushMatrix();
			glTranslatef((x1<<4)-dwm,0.0f,pre_y-dhm);
			for(x=x1;x<=x2;x++) {
				if(!FLAT) {
					VECTOR	V;
					V.x=(x<<4)-dwm;
					V.y=h_map[Y+1][(x<<1)+1];
					V.z=pre_y-dhm;
					V=V-cam->Pos;
					if(fabs(V%cam->Dir)>cam->zfar) {
						bmap[y][x]=-abs(bmap[y][x]);
						glTranslatef(16.0f,0.0f,0.0f);
						continue;
						}
					float d=V.Sq();
					if(d>16384.0f)
						if((V%cam->Dir)/sqrt(d)<ref) {
							bmap[y][x]=-abs(bmap[y][x]);
							glTranslatef(16.0f,0.0f,0.0f);
							continue;
							}
					bmap[y][x]=abs(bmap[y][x]);
					}
				else
					if(bmap[y][x]<0) {
						glTranslatef(16.0f,0.0f,0.0f);
						continue;
						}
				i=bmap[y][x];
				int X=x<<1;
				if(FLAT) {
					bloc[i].point=lvl[pre_y2+x];
					if(bloc[i].point==NULL || bloc[i].point[0].y<niv || bloc[i].point[1].y<niv || bloc[i].point[2].y<niv ||
						bloc[i].point[3].y<niv || bloc[i].point[4].y<niv || bloc[i].point[5].y<niv ||
						bloc[i].point[6].y<niv || bloc[i].point[7].y<niv || bloc[i].point[8].y<niv)
						bloc[i].point=flat;
					else {
						glTranslatef(16.0f,0.0f,0.0f);
						continue;
						}
					if(bloc[i].lava && (rand()%1000000)<=10000*dt) {
						POINTF POS;
						VECTOR V;
						POS.x=(x<<4)-dwm+8.0f;
						POS.z=pre_y-dhm+8.0f;
						POS.y=get_unit_h(POS.x,POS.y)-10.0f;
						V.x=((rand()%201)-100)*0.01f;
						V.y=((rand()%51)+50)*0.01f;
						V.z=((rand()%201)-100)*0.01f;
						V.Unit();
						particle_engine.emit_lava(POS,V,1,100,(rand()%1000)*0.01f+30.0f);
						}
					}
				else {
					bloc[i].point=lvl[pre_y2+x];
					if(bloc[i].point==NULL) {
						lvl[pre_y2+x]=bloc[i].point=(POINTF*) malloc(sizeof(POINTF)*9);
						if(tnt) {
							bloc[i].point[0].x=0.0f;	bloc[i].point[0].z=get_zdec(X,Y);
							bloc[i].point[1].x=8.0f;	bloc[i].point[1].z=get_zdec(X+1,Y);
							bloc[i].point[2].x=16.0f;	bloc[i].point[2].z=get_zdec(X+2,Y);
							bloc[i].point[3].x=0.0f;	bloc[i].point[3].z=8.0f+get_zdec(X,Y+1);
							bloc[i].point[4].x=8.0f;	bloc[i].point[4].z=8.0f+get_zdec(X+1,Y+1);
							bloc[i].point[5].x=16.0f;	bloc[i].point[5].z=8.0f+get_zdec(X+2,Y+1);
							bloc[i].point[6].x=0.0f;	bloc[i].point[6].z=16.0f+get_zdec(X,Y+2);
							bloc[i].point[7].x=8.0f;	bloc[i].point[7].z=16.0f+get_zdec(X+1,Y+2);
							bloc[i].point[8].x=16.0f;	bloc[i].point[8].z=16.0f+get_zdec(X+2,Y+2);
							bloc[i].point[0].y=get_nh(X,Y);
							bloc[i].point[1].y=get_nh(X+1,Y);
							bloc[i].point[2].y=get_nh(X+2,Y);
							bloc[i].point[3].y=get_nh(X,Y+1);
							bloc[i].point[4].y=get_nh(X+1,Y+1);
							bloc[i].point[5].y=get_nh(X+2,Y+1);
							bloc[i].point[6].y=get_nh(X,Y+2);
							bloc[i].point[7].y=get_nh(X+1,Y+2);
							bloc[i].point[8].y=get_nh(X+2,Y+2);
							}
						else {
							bloc[i].point[0].x=0.0f;	bloc[i].point[0].z=0.0f;
							bloc[i].point[1].x=8.0f;	bloc[i].point[1].z=0.0f;
							bloc[i].point[2].x=16.0f;	bloc[i].point[2].z=0.0f;
							bloc[i].point[3].x=0.0f;	bloc[i].point[3].z=8.0f;
							bloc[i].point[4].x=8.0f;	bloc[i].point[4].z=8.0f;
							bloc[i].point[5].x=16.0f;	bloc[i].point[5].z=8.0f;
							bloc[i].point[6].x=0.0f;	bloc[i].point[6].z=16.0f;
							bloc[i].point[7].x=8.0f;	bloc[i].point[7].z=16.0f;
							bloc[i].point[8].x=16.0f;	bloc[i].point[8].z=16.0f;
							bloc[i].point[0].y=get_h(X,Y);
							bloc[i].point[1].y=get_h(X+1,Y);
							bloc[i].point[2].y=get_h(X+2,Y);
							bloc[i].point[3].y=get_h(X,Y+1);
							bloc[i].point[4].y=get_h(X+1,Y+1);
							bloc[i].point[5].y=get_h(X+2,Y+1);
							bloc[i].point[6].y=get_h(X,Y+2);
							bloc[i].point[7].y=get_h(X+1,Y+2);
							bloc[i].point[8].y=get_h(X+2,Y+2);
							}
						}
					}
				if(!FLAT)
					glVertexPointer( 3, GL_FLOAT, 0, bloc[i].point);

				if(i!=old_i && ntex>0) {
					if(bloc[i].tex!=old_tex) {
						glBindTexture(GL_TEXTURE_2D,bloc[i].tex);
						old_tex=bloc[i].tex;
						}
					glTexCoordPointer(2, GL_FLOAT, 0, bloc[i].texcoord);
					old_i=i;
					}

				glDrawElements(GL_TRIANGLE_STRIP, bloc[i].nbindex,GL_UNSIGNED_INT,bloc[i].index);		// dessine le tout
				glTranslatef(16.0f,0.0f,0.0f);
				}
			glPopMatrix();
			}
		if(!FLAT && water) {
			glColor4f(1.0f,1.0f,1.0f,0.75f);
			glEnable(GL_BLEND);
			glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
			draw(cam,true,sealvl,t,dt);
			glDisable(GL_BLEND);
			}
	}

	VECTOR MAP::hit(VECTOR Pos,VECTOR Dir)			// Calcule l'intersection d'un rayon avec la carte(le rayon partant du dessus de la carte)
	{
		if(Dir.x==0.0f && Dir.z==0.0f)			// Solution triviale
			return Pos;
		if(get_unit_h(Pos.x,Pos.z)>Pos.y)		// Cas non traité
			return Pos;
		float step=1.0f;
		if(Dir.x!=0.0f && Dir.z!=0.0f) {
			if(fabs(Dir.x)<fabs(Dir.z))
				step=1.0f/fabs(Dir.x);
			else
				step=1.0f/fabs(Dir.z);
			}
		int nb=0;
		float dwm=map_w*0.5f;
		float dhm=map_h*0.5f;
		while(sealvl<Pos.y && get_unit_h(Pos.x,Pos.z)<Pos.y) {
			if(nb>=1000)
				return Pos;
			nb++;
			Pos=Pos+step*Dir;
			if(fabs(Pos.x)>dwm || fabs(Pos.z)>dhm)		// Pas de résultat
				return Pos;
			}
		Pos=Pos-step*Dir;		// On recommence la dernière opération mais avec plus de précision
		step*=0.1f;
		nb=0;
		while(sealvl<Pos.y && get_unit_h(Pos.x,Pos.z)<Pos.y) {
			if(nb>=10)
				break;
			nb++;
			Pos=Pos+step*Dir;
			}
		return Pos;		// Meilleure solution approximative trouvée
	}

	void UNIT::add_mission(int mission_type,VECTOR *target,bool step,int dat,void *pointer)
	{
		MISSION *new_mission=(MISSION*) malloc(sizeof(MISSION));
		new_mission->next=NULL;
		new_mission->mission=mission_type;
		new_mission->step=step;
		new_mission->time=Atimer;
		new_mission->data=dat;
		new_mission->p=pointer;

		if(target)
			new_mission->target=*target;

		MISSION *stop=(MISSION*) malloc(sizeof(MISSION));
		stop->mission=MISSION_STOP;
		stop->step=true;
		stop->time=Atimer;
		stop->p=NULL;
		stop->data=0;
		if(step) {
			stop->next=NULL;
			new_mission->next=stop;
			}
		else {
			stop->next=new_mission;
			new_mission=stop;
			}

		if(step && mission) {
//			new_mission->next=mission;
			stop->next=mission;
			mission=new_mission;
			start_mission_script(mission->mission);
			}
		else {
			if(mission) {				// Ajoute l'ordre aux autres
				MISSION *cur=mission;
				while(cur->next!=NULL)	cur=cur->next;
				cur->next=new_mission;
				}
			else {
				mission=new_mission;
				start_mission_script(mission->mission);
				}
			}
	}

	void UNIT::set_mission(int mission_type,VECTOR *target,bool step,int dat,bool stopit,void *pointer)
	{
		while(mission!=NULL) 	clear_mission();			// Efface les ordres précédents

		mission=(MISSION*) malloc(sizeof(MISSION));
		mission->next=NULL;
		mission->mission=mission_type;
		mission->step=step;
		mission->time=Atimer;
		mission->data=dat;
		mission->p=pointer;
		if(target)
			mission->target=*target;

		if(stopit) {
			MISSION *stop=(MISSION*) malloc(sizeof(MISSION));
			stop->next=mission;
			stop->mission=MISSION_STOP;
			stop->step=true;
			stop->time=Atimer;
			stop->p=NULL;
			stop->data=0;
			mission=stop;
			}
		else
			start_mission_script(mission->mission);
	}

	void UNIT::draw(CAMERA *cam)
	{
		if(model==NULL) return;		// S'il n'y a pas de modèle associé, on quitte la fonction

		VECTOR D;
		D=Pos-cam->Pos;		// Vecteur "viseur unité" partant de la caméra vers l'unité

		float dist=D.Sq();
		if(dist>=16384.0f && (D%cam->Dir)<=0.0f)		return;
		if((D%cam->Dir)>cam->zfar2)	return;		// Si l'objet est hors champ on ne le dessine pas

		cam->SetView();					// Dessine le modèle
		glTranslatef(Pos.x,Pos.y,Pos.z);
		glRotatef(Angle.x,1.0f,0.0f,0.0f);
		glRotatef(Angle.y,0.0f,1.0f,0.0f);
		glRotatef(Angle.z,0.0f,0.0f,1.0f);
		float scale=unit_manager.unit_type[type_id].Scale;
		glScalef(scale,scale,scale);
		if(port[BUILD_PERCENT_LEFT]>66.0f)
			glPolygonMode (GL_FRONT_AND_BACK, GL_LINE);
		else
			glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);
		model->draw(&data,owner_id==0 && sel,port[BUILD_PERCENT_LEFT]>0.0f);
		glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);
	}

	void UNIT::draw_shadow(CAMERA *cam,VECTOR Dir)
	{
		if(model==NULL) return;		// S'il n'y a pas de modèle associé, on quitte la fonction

		VECTOR D;
		D=Pos-cam->Pos;		// Vecteur "viseur unité" partant de la caméra vers l'unité

		float dist=D.Sq();
		if(dist>=16384.0f && (D%cam->Dir)<=0.0f)		return;
		if((D%cam->Dir)>cam->zfar2)	return;		// Si l'objet est hors champ on ne le dessine pas

		cam->SetView();					// Dessine le modèle
		glTranslatef(Pos.x,Pos.y,Pos.z);
		glRotatef(Angle.x,1.0f,0.0f,0.0f);
		glRotatef(Angle.y,0.0f,1.0f,0.0f);
		glRotatef(Angle.z,0.0f,0.0f,1.0f);
		float scale=unit_manager.unit_type[type_id].Scale;
		glScalef(scale,scale,scale);
		glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);
		model->draw_shadow(Dir*RotateZ(-Angle.z*DEG2RAD)*RotateY(-Angle.y*DEG2RAD)*RotateX(-Angle.x*DEG2RAD),&data);
	}

	void UNIT::run_script(float dt,int id)			// Interprète les scripts liés à l'unité
	{
		if(!script_env[id].running)	return;
		if(script_env[id].wait>0.0f) {
			script_env[id].wait-=dt;
			return;
			}
		if(script==NULL || script_env[id].cur==NULL) {
			script_env[id].running=false;
			return;	// S'il n'y a pas de script associé on quitte la fonction
			}
		int script_id=(script_env[id].cur->val&0xFF);			// Récupère l'identifiant du script en cours d'éxecution et la position d'éxecution
		int pos=(script_env[id].cur->val>>8);

		if(script_id<0 || script_id>=script->nb_script)	{
			script_env[id].running=false;
			return;		// Erreur, ce n'est pas un script repertorié
			}

		port[HEALTH]=hp*100/unit_manager.unit_type[type_id].MaxDamage;

		float divisor=1.0f/65536.0f;
		POINTF O;
		O.x=O.y=O.z=0.0f;
		bool done=false;
		int nb_code=0;

//#define PRINT_CODE			// Affiche le code éxécuté dans la console

		do
		{
			int code=script->script_code[script_id][pos];			// Lit un code
			pos++;
			nb_code++;
			if(nb_code>=1) done=true;			// Pour éviter que le programme ne fige à cause d'un script
			switch(code)			// Code de l'interpréteur
			{
			case SCRIPT_MOVE_OBJECT:
#ifdef PRINT_CODE
				printf("MOVE_OBJECT\n");
#endif
				{
					int obj=script->script_code[script_id][pos++];
					int axis=script->script_code[script_id][pos++];
					int v1=script_env[id].pop();
					int v2=script_env[id].pop();
					if(axis==0)
						v1=-v1;
					data.axe[axis][obj].move_distance=v1*divisor;
					data.axe[axis][obj].move_distance-=data.axe[axis][obj].pos;
					if(data.axe[axis][obj].move_distance<0.0f)
						data.axe[axis][obj].move_speed=-fabs(v2*divisor*0.1f);
					else
						data.axe[axis][obj].move_speed=fabs(v2*divisor*0.1f);
				}
				break;
			case SCRIPT_WAIT_FOR_TURN:
#ifdef PRINT_CODE
				printf("WAIT_FOR_TURN\n");
#endif
				{
					int obj=script->script_code[script_id][pos++];
					int axis=script->script_code[script_id][pos++];
					float a=data.axe[axis][obj].rot_angle;
					if((data.axe[axis][obj].rot_speed!=0.0f || data.axe[axis][obj].rot_accel!=0.0f) && (a!=0.0f && data.axe[axis][obj].rot_limit))
						pos-=3;
					else if(data.axe[axis][obj].rot_speed!=data.axe[axis][obj].rot_target_speed && data.axe[axis][obj].rot_speed_limit)
						pos-=3;
				}
				done=true;
				break;
			case SCRIPT_RANDOM_NUMBER:
#ifdef PRINT_CODE
				printf("RANDOM_NUMBER\n");
#endif
				{
					int high=script_env[id].pop();
					int low=script_env[id].pop();
					script_env[id].push((rand()%(high-low+1))+low);
				}
				break;
			case SCRIPT_LESS:
#ifdef PRINT_CODE
				printf("LESS\n");
#endif
				{
					int v2=script_env[id].pop();
					int v1=script_env[id].pop();
					script_env[id].push(v1<v2 ? 1 : 0);
				}
				break;
			case SCRIPT_EXPLODE:
#ifdef PRINT_CODE
				printf("EXPLODE\n");
#endif
				particle_engine.make_fire(O+Pos,1,(int)log(1+script_env[id].pop())<<2,45.0f);
				pos++;
				break;
			case SCRIPT_TURN_OBJECT:
#ifdef PRINT_CODE
				printf("TURN_OBJECT\n");
#endif
				{
					int obj=script->script_code[script_id][pos++];
					int axis=script->script_code[script_id][pos++];
					int v1=script_env[id].pop();
					int v2=script_env[id].pop();
					if(axis!=2) {
						v1=-v1;
						v2=-v2;
						}
					data.axe[axis][obj].rot_angle=-v1*divisor*360.0f;
					data.axe[axis][obj].rot_accel=0.0f;
					data.axe[axis][obj].rot_angle-=data.axe[axis][obj].angle;
					if(data.axe[axis][obj].rot_angle>0.0f)
						data.axe[axis][obj].rot_speed=fabs(v2*divisor*36.0f);
					else
						data.axe[axis][obj].rot_speed=-fabs(v2*divisor*36.0f);
					data.axe[axis][obj].rot_limit=true;
					data.axe[axis][obj].rot_speed_limit=false;
				}
				break;
			case SCRIPT_WAIT_FOR_MOVE:
#ifdef PRINT_CODE
				printf("WAIT_FOR_MOVE\n");
#endif
				{
					int obj=script->script_code[script_id][pos++];
					int axis=script->script_code[script_id][pos++];
					float a=data.axe[axis][obj].rot_angle;
					if(data.axe[axis][obj].move_distance!=0.0f)
						pos-=3;
				}
				done=true;
				break;
			case SCRIPT_CREATE_LOCAL_VARIABLE:
#ifdef PRINT_CODE
				printf("CREATE_LOCAL_VARIABLE\n");
#endif
				break;
			case SCRIPT_SUBTRACT:
#ifdef PRINT_CODE
				printf("SUBSTRACT\n");
#endif
				{
					int v1=script_env[id].pop();
					int v2=script_env[id].pop();
					script_env[id].push(v1-v2);
				}
				break;
			case SCRIPT_GET_VALUE_FROM_PORT:
#ifdef PRINT_CODE
				printf("GET_VALUE_FROM_PORT\n");
#endif
				{
					int value=script_env[id].pop();
					switch(value)
					{
					case ATAN:
						{
							int v1=script_env[id].pop();
							int v2=script_env[id].pop();
							port[value]=atan((float)v1/v2);
						}
						break;
					case HYPOT:
						{
							int v1=script_env[id].pop();
							int v2=script_env[id].pop();
							port[value]=sqrt(v1*v1+v2*v2);
						}
						break;
					case BUGGER_OFF:
						port[value]=0;
						break;
					case YARD_OPEN:
					case HEALTH:
					case BUILD_PERCENT_LEFT:
						break;
					default:
						{
							char *op[]={"INCONNU","ACTIVATION","STANDINGMOVEORDERS","STANDINGFIREORDERS","HEALTH","INBUILDSTANCE","BUSY","PIECE_XZ","PIECE_Y",
							"UNIT_XZ","UNIT_Y","UNIT_HEIGHT","XZ_ATAN","XZ_HYPOT","ATAN","HYPOT","GROUND_HEIGHT","BUILD_PERCENT_LEFT","YARD_OPEN",
							"BUGGER_OFF","ARMORED"};
							if(value>20)
								value=0;
							printf("opération non gérée : %s\n",op[value]);
						}
						break;
					};
					script_env[id].push(port[value]);
				}
				break;
			case SCRIPT_LESS_EQUAL:
#ifdef PRINT_CODE
				printf("LESS_EQUAL\n");
#endif
				{
					int v1=script_env[id].pop();
					int v2=script_env[id].pop();
					script_env[id].push(v1<=v2 ? 1 : 0);
				}
				break;
			case SCRIPT_SPIN_OBJECT:
#ifdef PRINT_CODE
				printf("SPIN_OBJECT\n");
#endif
				{
					int obj=script->script_code[script_id][pos++];
					int axis=script->script_code[script_id][pos++];
					int v1=script_env[id].pop();
					int v2=script_env[id].pop();
					if(axis!=2) {
						v1=-v1;
						v2=-v2;
						}
					data.axe[axis][obj].rot_limit=false;
					data.axe[axis][obj].rot_speed_limit=true;
					data.axe[axis][obj].rot_target_speed=v1;
					if(v1) {
						if(data.axe[axis][obj].rot_target_speed>data.axe[axis][obj].rot_speed)
							data.axe[axis][obj].rot_accel=abs(v2);
						else
							data.axe[axis][obj].rot_accel=-abs(v2);
						}
					else
						data.axe[axis][obj].rot_accel=v2;
				}
				break;
			case SCRIPT_SLEEP:
#ifdef PRINT_CODE
				printf("SLEEP\n");
#endif
				script_env[id].wait=script_env[id].pop()*0.001f;
				done=true;
				break;
			case SCRIPT_MULTIPLY:
#ifdef PRINT_CODE
				printf("MULTIPLY\n");
#endif
				{
					int v1=script_env[id].pop();
					int v2=script_env[id].pop();
					script_env[id].push(v1*v2);
				}
				break;
			case SCRIPT_CALL_SCRIPT:
#ifdef PRINT_CODE
				printf("CALL_SCRIPT\n");
#endif
				{
				int function_id=script->script_code[script_id][pos];			// Lit un code
				pos++;
				int num_param=script->script_code[script_id][pos];			// Lit un code
				pos++;
				script_env[id].push_vars();			// mémorise les variables
				for(int i=0;i<num_param;i++)		// Lit les paramètres
					script_env[id].var[i]=script_env[id].pop();
				script_env[id].cur->val=script_id+(pos<<8);
				SCRIPT_STACK *old=script_env[id].cur;
				script_env[id].cur=new SCRIPT_STACK;
				script_env[id].cur->next=old;
				script_env[id].cur->val=function_id;
				done=true;
				pos=0;
				script_id=function_id;
				}
				break;
			case SCRIPT_SHOW_OBJECT:
#ifdef PRINT_CODE
				printf("SHOW_OBJECT\n");
#endif
				data.flag[script->script_code[script_id][pos++]]&=(~FLAG_HIDE);
				break;
			case SCRIPT_EQUAL:
#ifdef PRINT_CODE
				printf("EQUAL\n");
#endif
				{
					int v1=script_env[id].pop();
					int v2=script_env[id].pop();
					script_env[id].push(v1==v2 ? 1 : 0);
				}
				break;
			case SCRIPT_NOT_EQUAL:
#ifdef PRINT_CODE
				printf("NOT_EQUAL\n");
#endif
				{
					int v1=script_env[id].pop();
					int v2=script_env[id].pop();
					script_env[id].push(v1!=v2 ? 1 : 0);
				}
				break;
			case SCRIPT_IF:
#ifdef PRINT_CODE
				printf("IF\n");
#endif
				if(script_env[id].pop()!=0)
					pos++;
				else {
					int target_offset=script->script_code[script_id][pos];			// Lit un code
					pos=target_offset-script->dec_offset[script_id];								// Déplace l'éxecution
					}
				break;
			case SCRIPT_HIDE_OBJECT:
#ifdef PRINT_CODE
				printf("HIDE_OBJECT\n");
#endif
				data.flag[script->script_code[script_id][pos++]]|=FLAG_HIDE;
				break;
			case SCRIPT_SIGNAL:
#ifdef PRINT_CODE
				printf("SIGNAL\n");
#endif
				if(script_env[id].pop()==script_env[id].signal_mask)
					script_env[id].running=false;
				break;
			case SCRIPT_DONT_CACHE:
#ifdef PRINT_CODE
				printf("DONT_CACHE\n");
#endif
				pos++;
				break;
			case SCRIPT_SET_SIGNAL_MASK:
#ifdef PRINT_CODE
				printf("SET_SIGNAL_MASK\n");
#endif
				script_env[id].signal_mask=script_env[id].pop();
				break;
			case SCRIPT_NOT:
#ifdef PRINT_CODE
				printf("NOT\n");
#endif
				script_env[id].push(~script_env[id].pop());
				break;
			case SCRIPT_DONT_SHADE:
#ifdef PRINT_CODE
				printf("DONT_SHADE\n");
#endif
				pos++;
				break;
			case SCRIPT_EMIT_SFX:
#ifdef PRINT_CODE
				printf("EMIT_SFX\n");
#endif
				{
					int smoke_type=script_env[id].pop();
//					printf("smoke_type %d\n",smoke_type);
					int from_piece=script->script_code[script_id][pos++];
					if(smoke_type==2 || smoke_type==3) {
						VECTOR Dir;
						Dir.x=-sin(Angle.y*PI/180.0f);
						Dir.z=-cos(Angle.y*PI/180.0f);
						Dir.y=0.0f;
						particle_engine.emit_part(O+Pos+data.pos[from_piece],Dir,0,1,10.0f);
						}
					else
						particle_engine.make_smoke(O+Pos+data.pos[from_piece],0,1,10.0f);
				}
				break;
			case SCRIPT_PUSH_CONST:
#ifdef PRINT_CODE
				printf("PUSH_CONST\n");
#endif
				script_env[id].push(script->script_code[script_id][pos]);
				pos++;
				break;
			case SCRIPT_PUSH_VAR:
#ifdef PRINT_CODE
				printf("PUSH_VAR\n");
#endif
				script_env[id].push(script_env[id].var[script->script_code[script_id][pos]]);
				pos++;
				break;
			case SCRIPT_SET_VAR:
#ifdef PRINT_CODE
				printf("SET_VAR\n");
#endif
				script_env[id].var[script->script_code[script_id][pos]]=script_env[id].pop();
				pos++;
				break;
			case SCRIPT_PUSH_STATIC_VAR:
#ifdef PRINT_CODE
				printf("PUSH_STATIC_VAR\n");
#endif
				script_env[id].push(s_var[script->script_code[script_id][pos]]);
				pos++;
				break;
			case SCRIPT_SET_STATIC_VAR:
#ifdef PRINT_CODE
				printf("SET_STATIC_VAR\n");
#endif
				s_var[script->script_code[script_id][pos]]=script_env[id].pop();
				pos++;
				break;
			case SCRIPT_OR:
#ifdef PRINT_CODE
				printf("OR\n");
#endif
				{
					int v1=script_env[id].pop(),v2=script_env[id].pop();
					script_env[id].push(v1|v2);
				}
				break;
			case SCRIPT_START_SCRIPT:				// Transfère l'éxecution à un autre script
#ifdef PRINT_CODE
				printf("START_SCRIPT\n");
#endif
				{
				int function_id=script->script_code[script_id][pos];			// Lit un code
				pos++;
				int num_param=script->script_code[script_id][pos];			// Lit un code
				pos++;
				for(int i=0;i<num_param;i++)		// Lit les paramètres
					script_env[id].var[i]=script_env[id].pop();
				if(script_env[id].cur)			// Détruit la pile
					do
					{
						SCRIPT_STACK *next=script_env[id].cur->next;
						delete script_env[id].cur;
						script_env[id].cur=next;
					}while(script_env[id].cur);
				script_env[id].cur=new SCRIPT_STACK;
				script_env[id].cur->next=NULL;
				script_env[id].cur->val=function_id;
				done=true;
				pos=0;
				script_id=function_id;
				}
				break;
			case SCRIPT_RETURN:		// Retourne au script précédent
#ifdef PRINT_CODE
				printf("RETURN\n");
#endif
				{
					SCRIPT_STACK *old=script_env[id].cur;
					script_env[id].cur=script_env[id].cur->next;
					delete old;
					script_env[id].pop();		// Enlève la valeur retournée
					script_env[id].pop_vars();
				}
				if(script_env[id].cur) {
					script_id=(script_env[id].cur->val&0xFF);			// Récupère l'identifiant du script en cours d'éxecution et la position d'éxecution
					pos=(script_env[id].cur->val>>8);
					}
				done=true;
				break;
			case SCRIPT_JUMP:						// Commande de saut
#ifdef PRINT_CODE
				printf("JUMP\n");
#endif
				{
					int target_offset=script->script_code[script_id][pos];			// Lit un code
					pos=target_offset-script->dec_offset[script_id];								// Déplace l'éxecution
//					printf("GOTO %d\n",pos);
				}
				break;
			case SCRIPT_ADD:
#ifdef PRINT_CODE
				printf("ADD\n");
#endif
				{
					int v1=script_env[id].pop();
					int v2=script_env[id].pop();
					script_env[id].push(v1+v2);
				}
				break;	//added
			case SCRIPT_STOP_SPIN:
#ifdef PRINT_CODE
				printf("STOP_SPIN\n");
#endif
				{
					int obj=script->script_code[script_id][pos++];
					int axis=script->script_code[script_id][pos++];
					int v=script_env[id].pop();
					if(axis!=2)
						v=-v;
					data.axe[axis][obj].rot_limit=false;
					data.axe[axis][obj].rot_speed_limit=true;
					data.axe[axis][obj].rot_target_speed=0.0f;
					if(data.axe[axis][obj].rot_speed>0.0f)
						data.axe[axis][obj].rot_accel=-abs(v);
					else
						data.axe[axis][obj].rot_accel=abs(v);
				}
				break;	//added
			case SCRIPT_DIVIDE:
#ifdef PRINT_CODE
				printf("DIVIDE\n");
#endif
				{
					int v1=script_env[id].pop();
					int v2=script_env[id].pop();
					script_env[id].push(v2 / v1);
				}
				break;	//added
			case SCRIPT_MOVE_PIECE_NOW:
#ifdef PRINT_CODE
				printf("MOVE_PIECE_NOW\n");
#endif
				{
					int obj=script->script_code[script_id][pos++];
					int axis=script->script_code[script_id][pos++];
					if(axis==0)
						data.axe[axis][obj].pos=-script_env[id].pop()*divisor;
					else
						data.axe[axis][obj].pos=script_env[id].pop()*divisor;
				}
				break;	//added
			case SCRIPT_TURN_PIECE_NOW:
#ifdef PRINT_CODE
				printf("TURN_PIECE_NOW\n");
#endif
				{
					int obj=script->script_code[script_id][pos++];
					int axis=script->script_code[script_id][pos++];
					int v=script_env[id].pop();
					if(axis!=2)
						v=-v;
					data.axe[axis][obj].angle=-v*divisor*360.0f;
				}
				break;	//added
			case SCRIPT_CACHE:
#ifdef PRINT_CODE
				printf("CACHE\n");
#endif
				pos++;
				break;	//added
			case SCRIPT_COMPARE_AND:
#ifdef PRINT_CODE
				printf("COMPARE_AND\n");
#endif
				{
					int v1=script_env[id].pop();
					int v2=script_env[id].pop();
					script_env[id].push(v1 && v2);
				}
				break;	//added
			case SCRIPT_COMPARE_OR:
#ifdef PRINT_CODE
				printf("COMPARE_OR\n");
#endif
				{
					int v1=script_env[id].pop();
					int v2=script_env[id].pop();
					script_env[id].push(v1 || v2);
				}
				break;	//added
			case SCRIPT_CALL_FUNCTION:
#ifdef PRINT_CODE
				printf("CALL_FUNCTION\n");
#endif
				{
				int function_id=script->script_code[script_id][pos];			// Lit un code
				pos++;
				int num_param=script->script_code[script_id][pos];			// Lit un code
				pos++;
				script_env[id].push_vars();
				for(int i=0;i<num_param;i++)		// Lit les paramètres
					script_env[id].var[i]=script_env[id].pop();
				script_env[id].cur->val=script_id+(pos<<8);
				SCRIPT_STACK *old=script_env[id].cur;
				script_env[id].cur=new SCRIPT_STACK;
				script_env[id].cur->next=old;
				script_env[id].cur->val=function_id;
				done=true;
				pos=0;
				script_id=function_id;
				}
				break;	//added
			case SCRIPT_GET:
				printf("GET *\n");
				break;	//added
			case SCRIPT_SET_VALUE:
				printf("SET_VALUE *\n");
				{
					int v1=script_env[id].pop();
					int v2=script_env[id].pop();
					printf("%d %d\n",v1,v2);
					port[v2]=v1;
				}
				break;	//added
			case SCRIPT_ATTACH_UNIT:
				printf("ATTACH_UNIT *\n");
				pos++;
				break;	//added
			case SCRIPT_DROP_UNIT:
				printf("DROP_UNIT *\n");
				pos++;
				break;	//added
			default:
				printf("UNKNOWN %d, STOPPING SCRIPT\n",code);
				{
					SCRIPT_STACK *old=script_env[id].cur;
					script_env[id].cur=script_env[id].cur->next;
					delete old;
				}
				if(script_env[id].cur) {
					script_id=(script_env[id].cur->val&0xFF);			// Récupère l'identifiant du script en cours d'éxecution et la position d'éxecution
					pos=(script_env[id].cur->val>>8);
					}
				else
					script_env[id].running=false;
				done=true;
			};
		}while(!done);
		if(script_env[id].cur)
			script_env[id].cur->val=script_id+(pos<<8);
	}

	void UNIT::move(float dt,MAP *map)
	{
		if(port[BUILD_PERCENT_LEFT]>0.0f)	return;		// L'unité n'est pas finie
		if(data.nb_piece>0)
			data.move(dt);

		V_Angle.x=V_Angle.y=V_Angle.z=0.0f;
		if(mission) {
				switch(mission->mission)						// Commandes générales
				{
				case MISSION_STOP:
					launch_script(get_script_index("StopMoving"));
					if(mission->data!=0)
						next_mission();
					else
						mission->data++;
					break;
				case MISSION_REPAIR:
					{
						UNIT *target_unit=(UNIT*) mission->p;
						if(target_unit!=NULL && target_unit->flags && target_unit->port[BUILD_PERCENT_LEFT]==0.0f) {
							if(target_unit->hp>=unit_manager.unit_type[target_unit->type_id].MaxDamage || unit_manager.unit_type[type_id].TEDclass==CLASS_PLANT) {
								if(unit_manager.unit_type[type_id].TEDclass!=CLASS_PLANT)
									target_unit->hp=unit_manager.unit_type[target_unit->type_id].MaxDamage;
								next_mission();
								}
							else {
								VECTOR Dir=target_unit->Pos-Pos;
								mission->target=target_unit->Pos;
								float dist=Dir.Sq();
								int maxdist=unit_manager.unit_type[type_id].BuildDistance;
								if(dist>maxdist*maxdist && unit_manager.unit_type[type_id].TEDclass!=CLASS_PLANT) {	// Si l'unité est trop loin du chantier
									VECTOR target;
									dist=sqrt(dist);
									target=Pos+(dist-maxdist)/dist*Dir;
									add_mission(MISSION_MOVE,&target,true);		// On lui demande de se rapprocher
									}
								else {
									if(dist!=0.0f)
										Dir=1.0f/sqrt(dist)*Dir;
									float dest_angle=acos(Dir.z)*RAD2DEG;
									if(Dir.x<0.0f) dest_angle=-dest_angle;
									while(fabs(dest_angle-Angle.y)>180.0f) {
										if(dest_angle<Angle.y)
											Angle.y-=360.0f;
										else
											Angle.y+=360.0f;
										}
									if(dist>0.0f && fabs(dest_angle-Angle.y)>=unit_manager.unit_type[type_id].BuildAngle) {
										float aspeed=unit_manager.unit_type[type_id].TurnRate;
										if(dest_angle<Angle.y)
											aspeed=-aspeed;
										float a=dest_angle-Angle.y;
										V_Angle.y=aspeed*0.1f;
										float b=dest_angle-(Angle.y+dt*V_Angle.y);
										if((a<0.0f && b>0.0f) || (a>0.0f && b<0.0f)) {
											V_Angle.y=0.0f;
											Angle.y=dest_angle;
											}
										}
									else
										target_unit->hp+=dt*unit_manager.unit_type[type_id].WorkerTime*unit_manager.unit_type[target_unit->type_id].MaxDamage/unit_manager.unit_type[target_unit->type_id].HealTime;
									}
								}
							}
						else if(target_unit!=NULL && target_unit->flags) {
							VECTOR Dir=target_unit->Pos-Pos;
							mission->target=target_unit->Pos;
							float dist=Dir.Sq();
							int maxdist=unit_manager.unit_type[type_id].BuildDistance;
							if(dist>maxdist*maxdist && unit_manager.unit_type[type_id].TEDclass!=CLASS_PLANT) {	// Si l'unité est trop loin du chantier
								VECTOR target;
								dist=sqrt(dist);
								target=Pos+(dist-maxdist)/dist*Dir;
								add_mission(MISSION_MOVE,&target,true);		// On lui demande de se rapprocher
								}
							else {
								if(dist!=0.0f)
									Dir=1.0f/sqrt(dist)*Dir;
								float dest_angle=acos(Dir.z)*RAD2DEG;
								if(Dir.x<0.0f) dest_angle=-dest_angle;
								while(fabs(dest_angle-Angle.y)>180.0f) {
									if(dest_angle<Angle.y)
										Angle.y-=360.0f;
									else
										Angle.y+=360.0f;
									}
								if(dist>0.0f && fabs(dest_angle-Angle.y)>=unit_manager.unit_type[type_id].BuildAngle && unit_manager.unit_type[type_id].TEDclass!=CLASS_PLANT) {
									float aspeed=unit_manager.unit_type[type_id].TurnRate;
									if(dest_angle<Angle.y)
										aspeed=-aspeed;
									float a=dest_angle-Angle.y;
									V_Angle.y=aspeed*0.1f;
									float b=dest_angle-(Angle.y+dt*V_Angle.y);
									if((a<0.0f && b>0.0f) || (a>0.0f && b<0.0f)) {
										V_Angle.y=0.0f;
										Angle.y=dest_angle;
										}
									}
								else
									mission->mission=MISSION_BUILD_2;		// Change de type de mission
								}
							}
					}
					break;
				case MISSION_BUILD_2:
					{
						UNIT *target_unit=(UNIT*) mission->p;
						if(target_unit->flags) {
							if(target_unit->port[BUILD_PERCENT_LEFT]<=0.0f) {
								target_unit->port[BUILD_PERCENT_LEFT]=0.0f;
								target_unit->launch_script(target_unit->get_script_index("create"));		// Script post création
								if(unit_manager.unit_type[target_unit->type_id].ActivateWhenBuilt)		// Activation automatique
									target_unit->activate();
								if(unit_manager.unit_type[type_id].TEDclass==CLASS_PLANT) {		// Ordre de se déplacer
									VECTOR target=Pos;
									target.z+=128.0f;
									target_unit->set_mission(MISSION_MOVE,&target);		// Fait sortir l'unité du bâtiment
									}
								mission->p=NULL;
								next_mission();
								deactivate();
								}
							else {
								target_unit->port[BUILD_PERCENT_LEFT]-=dt*unit_manager.unit_type[type_id].WorkerTime*100.0f/unit_manager.unit_type[target_unit->type_id].BuildTime;
								target_unit->hp=(1.0f-0.01f*target_unit->port[BUILD_PERCENT_LEFT])*unit_manager.unit_type[target_unit->type_id].MaxDamage;
								}
							}
						else {
							deactivate();
							next_mission();
							}
						}
					break;
				case MISSION_BUILD:
					if(mission->p) {
						if(!is_running(get_script_index("Activate")))
							mission->mission=MISSION_BUILD_2;		// Change les objectifs
						}
					else {
						VECTOR Dir=mission->target-Pos;
						float dist=Dir.Sq();
						int maxdist=unit_manager.unit_type[type_id].BuildDistance;
						if(dist>maxdist*maxdist && unit_manager.unit_type[type_id].TEDclass!=CLASS_PLANT) {	// Si l'unité est trop loin du chantier
							VECTOR target;
							dist=sqrt(dist);
							target=Pos+(dist-maxdist)/dist*Dir;
							add_mission(MISSION_MOVE,&target,true);		// On lui demande de se rapprocher
							}
						else {
							if(dist!=0.0f)
								Dir=1.0f/sqrt(dist)*Dir;
							float dest_angle=acos(Dir.z)*RAD2DEG;
							if(Dir.x<0.0f) dest_angle=-dest_angle;
							while(fabs(dest_angle-Angle.y)>180.0f) {
								if(dest_angle<Angle.y)
									Angle.y-=360.0f;
								else
									Angle.y+=360.0f;
								}
							if(dist>0.0f && fabs(dest_angle-Angle.y)>=unit_manager.unit_type[type_id].BuildAngle && unit_manager.unit_type[type_id].TEDclass!=CLASS_PLANT) {
								float aspeed=unit_manager.unit_type[type_id].TurnRate;
								if(dest_angle<Angle.y)
									aspeed=-aspeed;
								float a=dest_angle-Angle.y;
								V_Angle.y=aspeed*0.1f;
								float b=dest_angle-(Angle.y+dt*V_Angle.y);
								if((a<0.0f && b>0.0f) || (a>0.0f && b<0.0f)) {
									V_Angle.y=0.0f;
									Angle.y=dest_angle;
									}
								}
							else {
								mission->p=create_unit(mission->data,owner_id,mission->target);
								activate();
								}
							}
						}
					break;
				};

			switch(unit_manager.unit_type[type_id].TEDclass)			// Commandes particulières
			{
			case CLASS_SHIP:
				switch(mission->mission)
				{
				case MISSION_STANDBY:
					next_mission();
					V=exp(-dt)*V;			// Frottements
					break;
				case MISSION_MOVE:
					{
						VECTOR J,I;
						J=mission->target-Pos;
						float dist=J.Norm();
						if(dist>0.0f)
							J=1.0f/dist*J;
						float dest_angle=acos(J.z)*RAD2DEG;
						if(J.x<0.0f) dest_angle=-dest_angle;
						while(fabs(dest_angle-Angle.y)>180.0f) {
							if(dest_angle<Angle.y)
								Angle.y-=360.0f;
							else
								Angle.y+=360.0f;
							}
						if(dist>0.0f && fabs(dest_angle-Angle.y)>=1.0f) {
							float aspeed=unit_manager.unit_type[type_id].TurnRate;
							if(dest_angle<Angle.y)
								aspeed=-aspeed;
							float a=dest_angle-Angle.y;
							V_Angle.y=aspeed*0.1f;
							float b=dest_angle-(Angle.y+dt*V_Angle.y);
							if((a<0.0f && b>0.0f) || (a>0.0f && b<0.0f)) {
								V_Angle.y=0.0f;
								Angle.y=dest_angle;
								}
							}

						J.z=cos(Angle.y*DEG2RAD);
						J.x=sin(Angle.y*DEG2RAD);
						J.y=0.0f;
						I.z=-J.x;
						I.x=J.z;
						I.y=0.0f;
						V=(V%J)*J+exp(-dt)*(V%I)*I;
						if(!(dist<128.0f && fabs(Angle.y-dest_angle)>=1)) {
							float speed=V.Norm();
							float time_to_stop=speed/unit_manager.unit_type[type_id].BrakeRate;
							float min_dist=time_to_stop*(speed-unit_manager.unit_type[type_id].BrakeRate*0.5f*time_to_stop);
							if(min_dist>=dist)			// Freine si nécessaire
								V=V-unit_manager.unit_type[type_id].BrakeRate*dt*J;
							else if(speed<unit_manager.unit_type[type_id].MaxVelocity)
								V=V+unit_manager.unit_type[type_id].Acceleration*dt*J;
							if(dist<10.0f) {	
								next_mission();
								launch_script(get_script_index("stopmoving"));
								}
							}
						else
							V=exp(-dt)*V;			// Frottements
					}
					break;
				default:
					V=exp(-dt)*V;			// Frottements
				};
				break;
			case CLASS_PLANT:
				switch(mission->mission)
				{
				case MISSION_STANDBY:
				case MISSION_BUILD:
				case MISSION_BUILD_2:
					break;
				default:
					next_mission();
				};
				break;
			case CLASS_VTOL:
			case CLASS_KBOT:
			case CLASS_COMMANDER:
			case CLASS_TANK:
			case CLASS_CNSTR:
				{
				VECTOR K,PV;
				K.x=0.0f;
				K.y=1.0f;
				K.z=0.0f;
				PV=(V%K)*K;
				V=exp(-2.0f*dt)*(V-PV)+PV;		// N'altère pas la déscente des unités
				switch(mission->mission)
				{
				case MISSION_MOVE:
					{
						VECTOR J,I;
						J=mission->target-Pos;
						J.y=0.0f;
						float dist=J.Norm();
						if(dist>0.0f)
							J=1.0f/dist*J;
						float dest_angle=acos(J.z)*RAD2DEG;
						if(J.x<0.0f) dest_angle=-dest_angle;
						while(fabs(dest_angle-Angle.y)>180.0f) {
							if(dest_angle<Angle.y)
								Angle.y-=360.0f;
							else
								Angle.y+=360.0f;
							}
						if(dist>0.0f && fabs(dest_angle-Angle.y)>=1.0f) {
							float aspeed=unit_manager.unit_type[type_id].TurnRate;
							if(dest_angle<Angle.y)
								aspeed=-aspeed;
							float a=dest_angle-Angle.y;
							V_Angle.y=aspeed*0.1f;
							float b=dest_angle-(Angle.y+dt*V_Angle.y);
							if((a<0.0f && b>0.0f) || (a>0.0f && b<0.0f)) {
								V_Angle.y=0.0f;
								Angle.y=dest_angle;
								}
							}

						J.z=cos(Angle.y*DEG2RAD);
						J.x=sin(Angle.y*DEG2RAD);
						J.y=0.0f;
						I.z=-J.x;
						I.x=J.z;
						I.y=0.0f;
						V=(V%K)*K+(V%J)*J+exp(-dt)*(V%I)*I;
						if(!(dist<128.0f && fabs(Angle.y-dest_angle)>=1)) {
							float speed=V.Norm();
							float time_to_stop=speed/unit_manager.unit_type[type_id].BrakeRate;
							float min_dist=time_to_stop*(speed-unit_manager.unit_type[type_id].BrakeRate*0.5f*time_to_stop);
							if(min_dist>=dist)			// Freine si nécessaire
								V=V-unit_manager.unit_type[type_id].BrakeRate*dt*J;
							else if(speed<unit_manager.unit_type[type_id].MaxVelocity)
								V=V+unit_manager.unit_type[type_id].Acceleration*dt*J;
							if(dist<10.0f) {	
								next_mission();
								launch_script(get_script_index("stopmoving"));
								}
							}
					}
					break;
					};
				}
				break;
			};
			}

		if(port[BUILD_PERCENT_LEFT]==0.0f) {
			Angle=Angle+dt*V_Angle;
			V.y-=dt*9.81;			// L'unité subit la force de gravitation
			Pos=Pos+dt*V;			// Déplace l'unité
			}
		switch(unit_manager.unit_type[type_id].TEDclass)
		{
		case CLASS_SHIP:
			if(map->get_unit_h(Pos.x,Pos.z)>map->sealvl) {			// Les bateaux restent sur l'eau
				Pos=Pos-dt*V;
				V=0.0f*V;			// le bateau s'arrête
				if(Atimer-mission->time>=SECS_TO_TIMER(5))
				switch(mission->mission)		// Si cela compromet la mission
				{
				case MISSION_MOVE:
					{
						VECTOR Dir=mission->target-Pos;
						Dir.Unit();
						VECTOR step_target;
						VECTOR step_pos;
						step_target.x=Dir.z;
						step_target.y=Dir.y;
						step_target.z=-Dir.x;
						if(rand()&1)
							step_target=-step_target;
						step_pos=Pos+step_target;
						int nb=0;
						while(map->get_unit_h(step_pos.x,step_pos.z)<map->sealvl && nb<130) {
							step_pos=step_pos+step_target;
							nb++;
							}
						step_pos=step_pos-step_target;
						if(nb==0) {
							step_pos=Pos-step_target;
							while(map->get_unit_h(step_pos.x,step_pos.z)<map->sealvl && nb<130) {
								step_pos=step_pos-step_target;
								nb++;
								}
							step_pos=step_pos-step_target;
							}
						add_mission(MISSION_MOVE,&step_pos,true);		// Ajoute une étape
					}
					break;
				};
				}
			break;
		};
		if(map && port[BUILD_PERCENT_LEFT]==0.0f) {
			float min_h=map->get_unit_h(Pos.x,Pos.z);
			if((unit_manager.unit_type[type_id].WaterLine || unit_manager.unit_type[type_id].Floater || (unit_manager.unit_type[type_id].TEDclass&CLASS_SHIP)) && Pos.y<map->sealvl) {
				Pos.y=map->sealvl;
				if(V.y<0.0f)
					V.y=0.0f;
				}
			else if(min_h>Pos.y) {
				Pos.y=min_h;
				if(V.y<0.0f)
					V.y=0.0f;
				}
			port[GROUND_HEIGHT]=Pos.y-min_h;
			}
		for(int i=0;i<30;i++)
			run_script(dt,i);
	}

	void UNIT::show_orders()				// Dessine les ordres reçus
	{
		MISSION *cur=mission;
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
		VECTOR p_target=Pos;
		float rab=(Atimer%1000000)*0.000001;
		while(cur) {
			if(cur->step)				// S'il s'agit d'une étape on ne la montre pas
				cur=cur->next;
			{
			int curseur=anim_cursor(CURSOR_CROSS_LINK);
			float dx=0.5f*cursor.anm[CURSOR_CROSS_LINK].ofs_x[curseur];
			float dz=0.5f*cursor.anm[CURSOR_CROSS_LINK].ofs_y[curseur];
			float sx=0.5f*(cursor.anm[CURSOR_CROSS_LINK].bmp[curseur]->w-1);
			float sy=0.5f*(cursor.anm[CURSOR_CROSS_LINK].bmp[curseur]->h-1);
			float x,y,z;
			float dist=(cur->target-p_target).Norm();
			int rec=(int)(dist/30.0f);
			if(rec>0) {
				glBindTexture(GL_TEXTURE_2D,cursor.anm[CURSOR_CROSS_LINK].glbmp[curseur]);
				glBegin(GL_QUADS);
					for(int i=0;i<rec;i++) {
						x=p_target.x+(cur->target.x-p_target.x)*(i+rab)/rec;
						y=p_target.y+(cur->target.y-p_target.y)*(i+rab)/rec;
						z=p_target.z+(cur->target.z-p_target.z)*(i+rab)/rec;
						x-=dx;
						y+=0.75f;
						z-=dz;
						glTexCoord2f(0.0f,0.0f);		glVertex3f(x,y,z);
						glTexCoord2f(1.0f,0.0f);		glVertex3f(x+sx,y,z);
						glTexCoord2f(1.0f,1.0f);		glVertex3f(x+sx,y,z+sy);
						glTexCoord2f(0.0f,1.0f);		glVertex3f(x,y,z+sy);
						}
				glEnd();
				}
			p_target=cur->target;
			}
			switch(cur->mission)
			{
			case MISSION_MOVE:
				{
				int curseur=anim_cursor(CURSOR_MOVE);
				float x=cur->target.x-0.5f*cursor.anm[CURSOR_MOVE].ofs_x[curseur];
				float y=cur->target.y+1.0f;
				float z=cur->target.z-0.5f*cursor.anm[CURSOR_MOVE].ofs_y[curseur];
				float sx=0.5f*(cursor.anm[CURSOR_MOVE].bmp[curseur]->w-1);
				float sy=0.5f*(cursor.anm[CURSOR_MOVE].bmp[curseur]->h-1);
				glBindTexture(GL_TEXTURE_2D,cursor.anm[CURSOR_MOVE].glbmp[curseur]);
				glBegin(GL_QUADS);
					glTexCoord2f(0.0f,0.0f);		glVertex3f(x,y,z);
					glTexCoord2f(1.0f,0.0f);		glVertex3f(x+sx,y,z);
					glTexCoord2f(1.0f,1.0f);		glVertex3f(x+sx,y,z+sy);
					glTexCoord2f(0.0f,1.0f);		glVertex3f(x,y,z+sy);
				glEnd();
				}
				break;
			};
			cur=cur->next;
			}
		glDisable(GL_BLEND);
	}

	bool INGAME_UNITS::select(CAMERA *cam,int sel_x[],int sel_y[])
	{
		bool selected=false;
		cam->SetView();
		MATRIX_4x4 ModelView,Project,T;
		int	viewportCoords[4] = {0, 0, 0, 0};
		glGetIntegerv(GL_VIEWPORT, viewportCoords);
		glGetFloatv(GL_MODELVIEW_MATRIX, (float*)ModelView.E);
		glGetFloatv(GL_PROJECTION_MATRIX, (float*)Project.E);
		ModelView=Transpose(ModelView);
		Project=Transpose(Project);
		float VW=(viewportCoords[2]-viewportCoords[0])*0.5f;
		float VH=-(viewportCoords[3]-viewportCoords[1])*0.5f;
		T=ModelView*Project;		// Matrice de transformation

		POINTF UPos,O;
		O.x=O.y=O.z=0.0f;
		int X1,Y1,X2,Y2;
		X1=min(sel_x[0],sel_x[1]);
		Y1=min(sel_y[0],sel_y[1]);
		X2=max(sel_x[0],sel_x[1]);
		Y2=max(sel_y[0],sel_y[1]);

		for(int i=0;i<=last_index;i++)
			if(unit[i].flags!=0 && unit[i].owner_id==0 && unit[i].port[BUILD_PERCENT_LEFT]==0.0f) {		// Ne sélectionne que les unités achevées
				if(key[KEY_LSHIFT] && unit[i].sel) {
					selected=true;
					continue;
					}
				if(!key[KEY_LSHIFT])
					unit[i].sel=false;

				VECTOR Vec=unit[i].Pos-cam->Pos;
				float d=Vec.Sq();
				if(d>16384.0f && (Vec%cam->Dir)<=0.0f) continue;

				UPos=glNMult((O+unit[i].Pos),T);		// Transforme la position de l'unité
				UPos.x=UPos.x*VW+VW;
				UPos.y=UPos.y*VH-VH;

				if(X1<=UPos.x && X2>=UPos.x && Y1<=UPos.y && Y2>=UPos.y)
					selected=unit[i].sel=true;
				}
		return selected;
	}

	int INGAME_UNITS::pick(CAMERA *cam,int sensibility)
	{
		int index=-1;

		if(nb_unit<=0)
			return index;

		VECTOR Dir;
		Dir=cam->Dir+2.0f*(mouse_x-SCREEN_W*0.5f)/SCREEN_W*cam->Side-1.5f*(mouse_y-SCREEN_H*0.5f)/SCREEN_H*cam->Up;
		Dir.Unit();		// Direction pointée par le curseur

		bool detectable=false;
		int i;

		for(i=0;i<=last_index;i++) {
			if(unit[i].flags==0) continue;		// Si l'unité n'existe pas on la zappe
			unit[i].flags&=0xFD;	// Enlève l'indicateur de possibilité d'intersection
			VECTOR center=unit[i].model->center+unit[i].Pos-cam->Pos;
			float size=unit[i].model->size*unit_manager.unit_type[unit[i].type_id].Scale*unit_manager.unit_type[unit[i].type_id].Scale;
			center=Dir*center;
			float dist=center.Sq();
			if(dist<size) {
				detectable=true;
				unit[i].flags|=0x2;		// Unité détectable
				}
			}

		if(!detectable)			// Si aucune unité n'est à proximité du curseur, on zappe la méthode OpenGl
			return index;

		int BUFSIZE=max_unit<<2;

		GLuint selectBuf[BUFSIZE];
		GLint hits;

		glSelectBuffer (BUFSIZE, selectBuf);
		glRenderMode (GL_SELECT);

		glInitNames();
		glPushName(0);

		int	viewport[4] = {0, 0, 0, 0};
		glGetIntegerv(GL_VIEWPORT, viewport);

		if(cam->Ref!=NULL)
			cam->Pos=cam->RPos+(*(cam->Ref));
		else
			cam->Pos=cam->RPos;

		VECTOR	FP;
		FP=cam->Pos+cam->Dir;

		cam->zfar2=cam->zfar*cam->zfar;
		glMatrixMode (GL_PROJECTION);			// Matrice de projection
		glLoadIdentity ();

		gluPickMatrix(mouse_x, SCREEN_H-mouse_y, sensibility, sensibility, viewport);

		glFrustum (-1.0f*cam->znear, 1.0f*cam->znear, -0.75f*cam->znear, 0.75f*cam->znear, cam->znear, cam->zfar);

		glMatrixMode (GL_MODELVIEW);

		glLoadIdentity();			// Charge la matrice identité

		gluLookAt(cam->Pos.x,cam->Pos.y,cam->Pos.z,FP.x,FP.y,FP.z,cam->Up.x,cam->Up.y,cam->Up.z);

		glDisable(GL_LIGHTING);
		glDisable(GL_FOG);

		for(i=0;i<=last_index;i++)
			if((unit[i].flags&0x2)==0x2) {			// Si l'unité existe et est sélectionnable
				if(unit[i].model) {
					glLoadName(i);
					VECTOR D;
					D=unit[i].Pos-cam->Pos;		// Vecteur "viseur unité" partant de la caméra vers l'unité

					float dist=D.Sq();
					if(!((dist>=16384.0f && (D%cam->Dir)<=0.0f)||(D%cam->Dir)>cam->zfar2)) {
						glPushMatrix();

						glTranslatef(unit[i].Pos.x,unit[i].Pos.y,unit[i].Pos.z);
						glRotatef(unit[i].Angle.x,1.0f,0.0f,0.0f);
						glRotatef(unit[i].Angle.y,0.0f,1.0f,0.0f);
						glRotatef(unit[i].Angle.z,0.0f,0.0f,1.0f);
						float scale=unit_manager.unit_type[unit[i].type_id].Scale;
						glScalef(scale,scale,scale);
						unit[i].model->draw(&(unit[i].data),false,true);

						glPopMatrix();
						}
					}
				}

		glFlush ();

		glEnable(GL_LIGHTING);
		glEnable(GL_FOG);

		hits = glRenderMode (GL_RENDER);

		int j;
		GLuint names, *ptr;

		float nearest=2.0f;
		ptr = (GLuint *) selectBuf;
		for (i = 0; i < hits; i++) { /*  for each hit  */
			names = *ptr;
			ptr++;
			float z1=(float) *ptr/0x7fffffff; ptr++;
			float z2=(float) *ptr/0x7fffffff; ptr++;
			int v=*ptr; ptr++;

			if(z1<nearest) {
				nearest=z1;
				index=v;
				}
			}

		return index;
	}

void *create_unit(int type_id,int owner,VECTOR pos)
{
	int id=units.create(type_id,owner);
	units.unit[id].Pos=pos;
	units.unit[id].port[BUILD_PERCENT_LEFT]=100.0f;
	units.unit[id].reset_script();
	return &(units.unit[id]);
}
