#include <alleggl.h>
#include <string>

#include "../engine/TimeKeeper.h"
#include "../engine/TextRenderer.h"
#include "../misc/TextHandler.h"
#include "../misc/KeyReader.h"
#include "../misc/Random.h"
#include "../things/Planet.h"
#include "../things/Ship.h"
#include "../things/MenuWindow.h"
#include "../things/Panel.h"
#include "../things/Artifact.h"
#include "../sound/SoundSystem.h"
#include "../things/MineWindow.h"

#include "PlanetMenu.h"

/*****************************************/

struct EventData
{
	int type;
	std::string intro;
	std::string decision[2];
	double chance[2];
	std::string outcome[4];
};

const int EVENT_COUNT = 15;

EventData events[EVENT_COUNT] =
{
	// 0
	{0, 
	 "You encounter a strange hermit. At first you think he is human, but when you come near"
	 " you realise that his apparel is some sort of hologram. He asks you (surprisingly, in"
	 " plain English) politely to give him some food.",
	 {"Give him some (30 units)", "Refuse"},
	 {35, 50},
	 {"He smiles and promptly disappears. Something seems to be lying on the place where he was standing.",
	 "He thanks you and vanishes into thin air. Here's what you get for helping.",
	 "The old man grumbles something about \"the youth these days\" and disappears.",
	 "The old man growls and disappears. When you come back to your ship, you realize that almost all the food stored is gone."}
	},
	
	// 1
	{0,
	 "While you were taking a stroll outside your ship, some strange animals managed to sneak into the cargo bay"
	 " and steal some of your food. You see tracks leading into some dense vegetation.",
	 {"Follow the tracks", "Forget about the incident"},
	 {20, 101},
	 {"You find the animals and kill them. As it turns out, they seem to be edible themselves. You take them back"
	  " to your ship as food, along with the supplies they stole.",
	  "After you venture into the forest, you feel something heavy jumping on your back. You struggle for a moment,"
        " but in the end, the animal's teeth sever your carotid artery. You have died.",
	  "You back up to your ship and seal it.", ""}
	},
	
	// 2
	{0,
	 "You find something that appears to be a deserted Terran colony. It's in very poor condition.",
	 {"Venture inside", "Go back to the ship"},
	 {50, 101},
	 {"You find a recharge station inside. You hook it up to your ship and watch the energy bar go all the way to the top.",
	  "",
	  "It's probably a ruin anyway.",
	  ""}
	},
	
	// 3
	{1,
	 "You find some people frozen in Deep Sleep(TM). The containers appear to be very old.",
	 {"Wake them", "Leave them here"},
	 {50, 101},
	 {"The machines are defunct, and you don't succeed in reviving any of the humans. However, one of them"
	  " was holding a very interesting item.",
	  "The machines are defunct, and you don't succeed in reviving any of the humans.",
	  "Who knows what might happen, with all that old technology? Best not to bother.",
	  ""
	 }
	},
	
	// 4
	{0,
	 "The ground rumbles. It appears that a nearby volcano is going to erupt!",
	 {"Flee the planet", "Stay"},
	 {101, 25},
	 {"You managed to escape just in time before the eruption.",
	  "",
	  "The earthquake was very brief, and the volcano didn't erupt after all. However, a vein of hyperium"
	  " is revealed by a crack in the ground.",
	  "It's not very wise to watch a volcano erupt from closeby. The cargo bay got damaged by enormous"
	  " stones striking the hull."}
	},
	
	// 5
	{0,
	 "You stand before a gigantic stone fortress. It bears the symbol of Grammar Nazis.",
	 {"Enter", "Shit your pants"},
	 {-11, 101},
	 {"The High Grammar Priest notices the X-Ginator you carry. You return the precious relic to the"
	  " Nazis (they said they'd behead you if you didnt'). In return, the Priest gives you an item.",
	  "Ph34r the Nazis! The High Priest informs you that you"
	  " can't spell and that you are otherwise worthless. Seeing the obvious evidence of the futility of your"
	  " existence, you commit suicide.",
	  "Wise decision.",
	  ""}
	},
	
	// 6
	{0,
	 "You find a strange metal box. It is engraved, but the writings are in a language unsimilar to anything "
	 "you have ever seen.",
	 {"Open it", "Ignore it"},
	 {70, 101},
	 {"There is a loud fanfare as you open the box. Inside, amongst many awkwardly carved pieces of metal, "
	  "you notice a very peculiar item.",
        "The box appears to have been booby-trapped! A strange device inside started to beep rhytmically, and"
	  "with each beep, one symbol on a red display changed. You flee to your ship, and into orbit, just in time"
	  "to see the entire landing site where your ship rested wiped out from the face of the planet.",
	  "Who knows what might be inside.",
	  ""}
	},
	
	// 7
	{0,
	 "You find a strange metal box. It is engraved, but the writings are in a language unsimilar to anything "
	 "you have ever seen. As you approach it starts to open by itself.",
	 {"Come closer", "Run away"},
	 {50, 50},
	 {"As you come closer, you notice that the engravings are in fact familiar. In that instant, you realise that "
	  "you know this language - it is written in Advertising! Trying not to look on the letters, you peek inside. "
	  "You find a whole shipment of BigMacs with a note: 'We truly are EVERYWHERE!'.",
	  "When you come closer a lightning bolt shoots out. It misses you barely, but hits your ship instead. As you "
	  "examine the damage, you notice that some of the power cells are burnt and useless.",
	  "A lightning bolt misses you and hits a rock. As it cools, you notice something strange about its color - it's"
	  "hyperium! Molten or not, its useful. You take it to your ship.",
        "A man wearing a cape opens the box from the inside and flies to your ship. You realise in terror that this"
	  " must be the Intergalactic Supersizememan. Before you have a chance to act, he enters your cargo bay and"
	  " eats half of your food."}
	},
	
	// 8
	{0,
	 "You have found a strange device that looks like a ship module. What will you do with it?",
	 {"Try to connect it with your ship", "Toss it"},
	 {50, 101},
	 {"The device turns out to be a versatile power cell.",
	  "The device turns out to be a fuel compressor - thus allowing you to store more fuel.",
	  "You leave it right were you found it.", ""}
	},
	
	// 9
	{0,
	 "You have found a strange device that looks like a ship module. What will you do with it?",
	 {"Try to connect it with your ship", "Toss it"},
	 {50, 101},
	 {"The device turns out to be a one-use fuel enricher. The fuel that was inside your tank becomes more effective.",
	  "The device caused a detonation in your fuel tank. You manage to fix it, but at the expense of capacity.", 
	  "You leave it right where you found it.", ""}
	},
	
	// 10
	{0,
	 "An alien creature appears from nowhere and starts howling. It is two meters high, has beady yellow eyes and sword-sharp "
	 "teeth - not the type of alien you would negotiate a peace treaty with. You are cornered and can't run. What do you "
	 "do?",
	 {"Attack it", "Give it some food"},
	 {20, 90},
	 {"You manage to defeat the creature. It seems to be edible, and carried a very useful item.",
	  "You stand no chance against that creature without a good weapon. You are unwillingly but quickly transformed into mincemeat.",
	  "The creature sniffs your food, grabs it and vanishes behind a hill.",
	  "The creature swallows your food in one bite. After the appetizer, the alien consumes you for the main course."}
	},
	
	// 11
	{0,
	 "You notice an alien structure on the top of a nearby hill.",
	 {"Visit it", "Go to the ship"},
	 {50, 101},
	 {"", "", "You go to your ship, not looking that way anymore.", ""}
	},
	
	// 12
	{0,
	 "You notice an alien structure on the top of a nearby hill.",
	 {"Visit it", "Go to the ship"},
	 {50, 101},
	 {"@", "@", "You go to your ship, not looking that way anymore.", ""}
	},
	
	// 13
	{1,
	 "It looks like some sort of hive. The walls are organic, and you hear buzzing in the distance.",
	 {"Go further", "Withdraw"},
	 {60, 40},
	 {"It seems that the buzzing was only your imagination. If it indeed was something like a hive, it has been deserted"
	  " long ago. All that remains is dust and a single item of interest.",
	  "The buzzing was produced by creatures far worse than bees - small bearlike beings with big teeth and sharp claws. "
	  "Surprisingly, they don't attack you. Only one of them approaches you and scratches your forearm with his claw. You "
	  "fall unconscious, and when you wake up in your ship, you realise that those creatures have taken almost all of your fuel!",
	  "As you go out, you notice a ram hyperium stone embedded into the wall above the entrance. With some effort, you manage "
	  "to dislodge it.",
	  "You escape the structure in fear."}
	},
	
	// 14
	{1,
	 "It appears to be an alien supply station. Ship modules lie everywhere. Most are unusable, but you think that two of "
       "them are possible to attach to your ship with some tweaking. Which one do you take?",
	 {"The cargo module", "The cell module"},
	 {70, 60},
	 {"The module fits and works like a charm.",
	  "After all, you weren't able to make it work with your ship. What a shame.",
	  "After a bit of work you manage to install the module. It has big capacity.",
	  "After all, you weren't able to make it work with your ship. What a shame."}
	},
};

/***********************************************************
		EVENT SYSTEM
************************************************************/

enum {FOOD = 0, POWER = 1, FUEL = 2, HYPER = 3};

void PlanetMenu::grantArtifact()
{
	static std::string great[] = {"Great!"};
	
	
	int nr, cnt = 0;
	do
	{
		nr = randomInt(0, ARTIFACT_COUNT - 1);
		cnt++;
	} while(ship->hasArtifact[nr] || cnt == 50);
	
	ship->hasArtifact[nr] = true;
	
	std::string message = artifactData[nr][0] + " \n " + artifactData[nr][1];
	MenuWindow msg(message, great, 1, 0, 350);
	showMenu(&msg);
}

void PlanetMenu::addStuff(int what, double howMuch)
{
	double *val, *cap;
	
	switch(what)
	{
		case FOOD:
			val = &(ship->food);
			cap = &(ship->foodCap);
			break;
		case POWER:
			val = &(ship->power);
			cap = &(ship->powerCap);
			break;
		case FUEL:
			val = &(ship->fuel);
			cap = &(ship->fuelCap);
			break;
		case HYPER:
			val = &(ship->hyper);
			cap = &(ship->hyperCap);
			break;
	}
	
	(*val) += howMuch;
	if ((*val) > (*cap))
		(*val) = (*cap);
	if ((*val) < 0.0)
		(*val) = 0.0;
}

int PlanetMenu::askEvent(int nr)
{
	MenuWindow mWindow(events[nr].intro, events[nr].decision, 2, 0, 450);
	
	int choice = showMenu(&mWindow);
	return choice;
}

int PlanetMenu::determineOutcome(double chance)
{
	double rnd = randomFloat(0.0, 100.0);
	return (rnd < chance ? 0 : 1);
}

void PlanetMenu::showOutcome(int nr, int oc)
{
	std::string okOnly[] = {"OK"};
	
	MenuWindow mWindow(events[nr].outcome[oc], okOnly, 1, 0, 450);
	showMenu(&mWindow);
}

bool PlanetMenu::processEvent(int nr)
{
	bool retVal = false;
	
	/*********************************/

	switch(nr)
	{
		case 1: // animals
			addStuff(FOOD, -25.0);
			break;
	}

	/*********************************/

	int decision = askEvent(nr);
	double chance = events[nr].chance[decision];
	
	/*********************************/
	
	switch(nr)
	{
		case 0: // hermit
			if (!decision) addStuff(FOOD, -30.0);
			break;
		case 1: // animals
			if (!decision)
			{ if (ship->hasArtifact[RIFLE]) chance += 30;
			  if (ship->hasArtifact[LASER]) chance = 100;}
			break;
		case 5: // nazis
			if (!decision) if (ship->hasArtifact[XG]) chance = 101;
			break;
		case 10: // alien
			if (decision)
			{ if (ship->hasArtifact[RIFLE]) chance = 80;
			  if (ship->hasArtifact[LASER]) chance = 100;}
			break;
	}
	
	if (ship->hasArtifact[LUCK])
		chance += 10;
	
	/*********************************/
	
	int outcome = decision * 2 + determineOutcome(chance);
	
	/*********************************/
	
	switch(nr)
	{
		case 0: // hermit
			if (outcome == 0) {showOutcome(nr, outcome); outcome = -1; grantArtifact();};
			if (outcome == 3)
				if (ship->food < 15.0)
					outcome = 2;
				else
					ship->food = 15.0;
			break;
				
		case 1: // animals
			if (outcome == 0) addStuff(FOOD, 40.0);
			if (outcome == 1)
			{
				ship->aliveFlag = false;
				retVal = true;
			}
			break;
		case 2: // colony
			if (outcome == 0) ship->power = ship->powerCap;
			if (outcome == 1) { outcome = -1; processEvent(3); }
			break;
		case 3: // people
			if (outcome == 0) {showOutcome(nr, outcome); outcome = -1; grantArtifact();};
			break;
		case 4: // eruption
			if (outcome == 0) retVal = true;
			if (outcome == 2) planet->hyperValue += 30.0;
			if (outcome == 3) ship->foodCap -= 20.0;
			break;
		case 5: // nazis
			if (outcome == 0) {showOutcome(nr, outcome); outcome = -1; grantArtifact(); ship->hasArtifact[XG] = false; };
			if (outcome == 1) {ship->aliveFlag = false; retVal = true;};
			break;
		case 6: // box 1
			if (outcome == 0) {showOutcome(nr, outcome); outcome = -1; grantArtifact();};
			if (outcome == 1) retVal = true;
			break;
		case 7: // box 2
			if (outcome == 0) addStuff(FOOD, 30);
			if (outcome == 1) ship->powerCap *= 0.9;
			if (outcome == 2) addStuff(HYPER, 10);
			if (outcome == 3) ship->food *= 0.5;
			break;
		case 8: // device 1
			if (outcome == 0) ship->powerCap *= 1.1;
			if (outcome == 1) ship->fuelCap *= 1.15;
			break;
		case 9: // device 2
			if (outcome == 0) addStuff(FUEL, ship->fuel);
			if (outcome == 1) ship->fuelCap *= 0.85;
			break;
		case 10:
			if (outcome == 0) {addStuff(FOOD, 10); showOutcome(nr, outcome); outcome = -1; grantArtifact(); };
			if (outcome % 2 == 1) {ship->aliveFlag = false; retVal = true;};
			if (outcome == 2) addStuff(FOOD, -20);
			break;
		case 11:
		case 12:
			if (outcome == 0) {outcome = -1; processEvent(13);}
			if (outcome == 1) {outcome = -1; processEvent(14);}
			break;
		case 13:
			if (outcome == 0) {showOutcome(nr, outcome); outcome = -1; grantArtifact();};
			if (outcome == 1) ship->fuel *= 0.2;
			if (outcome == 2) addStuff(HYPER, 10);
			break;
		case 14:
			if (outcome == 0) ship->foodCap *= 1.2;
			if (outcome == 2) ship->powerCap *= 1.1;
			break;
	}
	/*********************************/
	
	if (ship->food > ship->foodCap)
		ship->food = ship->foodCap;
	if (ship->fuel > ship->fuelCap)
		ship->fuel = ship->fuelCap;
	if (ship->power > ship->powerCap)
		ship->power = ship->powerCap;

	if (outcome != -1)
		showOutcome(nr, outcome);
	
	return retVal;
}

/*****************************************/

PlanetMenu::PlanetMenu(Planet *planet, Ship *ship, Panel *panel)
	: planet(planet), ship(ship), panel(panel)
{}

/*****************************************/

void PlanetMenu::play()
{
	if (ship->thrustSound != -1)
	{
		FSOUND_StopSound(ship->thrustSound);
		ship->thrustSound = -1;
	}
	
	static char buf[512];
	bool alreadyHarvested = false, alreadyExplored = false, alreadyMined = false;
	
	/** The main planet menu **/
	std::string standardOps[] = {"Scan the surface", "Explore the planet on foot", "Harvest for food", "Mine for hyperium", "Launch"};
	std::string okOnly[] = {"OK"};
	
	MenuWindow mWindow("Your action?", standardOps, 5, 0, 350);
	
	int choice;
	do
	{
		mWindow.selected = -1;
		choice = showMenu(&mWindow);
		int amount;
		switch(choice)
		{
			case 0:
				amount = (ship->hasArtifact[EAGLE]) ? 12 : 15;

				if (ship->power > amount)
				{
					ship->power -= amount;
					
					sprintf(buf, "You have scanned the surface at the expense of %d units of power. ", amount);
					std::string message = buf;
					
					if (planet->foodValue > 0.8)
						message += "The planet is lush and you can easily resupply food here.";
					else if (planet->foodValue > 0.5)
						message += "There is a good chance of finding something edible on this planet.";
					else if (planet->foodValue > 0.25)
						message += "Harvesting food here may be less successful than on other, more Earth-like planets.";
					else
						message += "It's highly unlikely that you would find food here.";
					
					if (planet->hyperValue > 20.0)
						message += " There seems to be a lot of hyperium here.";
					else if (planet->hyperValue > 0.0)
						message += " It looks like some hyperium should be around.";
					else if (planet->hyperValue == 0.0)
						message += " There is no hyperium on this planet.";
					else
						message += " You have already extracted all the caches of hyperium on this planet.";
					
					if (planet->eventValue > 0.8)
						message += " You have a feeling that there is lot in store for you on this planet.";
					else if (planet->eventValue > 0.5)
						message += " It seems like a lot is happening here.";
					else if (planet->eventValue > 0.2)
						message += " The planet looks pretty quiet.";
					else
						message += " The surface is plain and very boring.";
					
					MenuWindow msg(message.c_str(), okOnly, 1, 0, 450);
					showMenu(&msg);
				}
				else
				{
					MenuWindow msg("You don't have enough power left to perform a scan.", okOnly, 1, 0, 450);
					showMenu(&msg);
				}
				break;
				
				
			case 1:
				if (!alreadyExplored)
				{
					alreadyExplored = true;
					if (randomFloat(0.0, 1.0) < planet->eventValue * 1.5)
					{
						int nr;
						do
							nr = randomInt(0, EVENT_COUNT - 1);
						while (events[nr].type != 0);
						
						bool brk;
						brk = processEvent(nr);
						if (brk)
							choice = 4;
					}
					else
					{
						MenuWindow msg("You look for something interesting, but it seems that your landing site has nothing to offer.", okOnly, 1, 0, 450);
						showMenu(&msg);
					}
				}
				else
				{
					MenuWindow msg("You've done enough exploring for now.", okOnly, 1, 0, 450);
					showMenu(&msg);
				}
				
				break;
				
				
			case 2:
				if (alreadyHarvested)
				{
					MenuWindow msg("You already did.", okOnly, 1, 0, 450);
					showMenu(&msg);
					break;
				}
				
				amount = (ship->hasArtifact[PLOUGH]) ? 16 : 20;
				
				if (ship->power > amount)
				{
					alreadyHarvested = true;
					ship->power -= amount;
					if (randomFloat(0.0, 0.8) < planet->foodValue)
					{
						double foodHarvested = randomFloat(35.0, 50.0) * (0.2 + planet->foodValue * 0.8);
						
						sprintf(buf, "You have harvested %d units of food at the expense of %d units of power.", (int)foodHarvested, amount);
							
						ship->food += foodHarvested;
						if (ship->food > ship->foodCap)
							ship->food = ship->foodCap;
						
						MenuWindow msg(buf, okOnly, 1, 0, 450);
						showMenu(&msg);
					}
					else
					{
						sprintf(buf, "You weren't able to find anything edible and lost %d units of power while trying.", amount);
						MenuWindow msg(buf, okOnly, 1, 0, 450);
						showMenu(&msg);
					}
					
					if (randomFloat(0.0, 1.0) < planet->eventValue)
					{
						int nr;
						do
							nr = randomInt(0, EVENT_COUNT - 1);
						while (events[nr].type != 0);
						
						bool brk;
						brk = processEvent(nr);
						if (brk)
							choice = 4;
					}
				}
				else
				{
					MenuWindow msg("You don't have enough power to harvest.", okOnly, 1, 0, 450);
					showMenu(&msg);
				}
				
				break;
				
			case 3:
				amount = (ship->hasArtifact[PICKAXE]) ? 25 : 30;
				
				if (alreadyMined)
				{
					MenuWindow msg("You have already mined for hyperium here. You must wait for the ground to cool off.", okOnly, 1, 0, 450);
					showMenu(&msg);
					break;
				}
				
				if (ship->power > amount)
				{
					ship->power -= amount;
					if (planet->hyperValue > 0.0)
					{
						alreadyMined = true;
						mine();
					}
					else
					{
						sprintf(buf, "Unfortunately, this planet seems to contain no hyperium. %d units of energy wasted.", amount);
						MenuWindow msg(buf, okOnly, 1, 0, 450);;
						showMenu(&msg);
					}
					
					if (randomFloat(0.0, 1.0) < planet->eventValue)
					{
						int nr;
						do
							nr = randomInt(0, EVENT_COUNT - 1);
						while (events[nr].type != 0);
						
						bool brk;
						brk = processEvent(nr);
						if (brk)
							choice = 4;
					}
				}
				else
				{
					MenuWindow msg("You don't have enough power left to mine for hyperium.", okOnly, 1, 0, 450);
					showMenu(&msg);
				}
				break;
		} 
	} while (choice != 4);
}

/*****************************************/

int PlanetMenu::showMenu(MenuWindow *mWindow)
{
	TimeKeeper tk(0.01, 0.025, 2, true);
	
	std::string oldName = planet->name;
	double oldX = planet->x, oldY = planet->y;
	planet->name = "";
	
	bool end = false;
	tk.reset();
	KeyReader::clearBuffer();
	
	while(!end)
	{
		double dt = tk.getDT();
		
		do
		{
			mWindow->step(dt);
			
			if (mWindow->selected > -1)
				end = true;

			KeyReader::update();
		}
		while(tk.getFramesToSkip() > 0);
		
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		
		glTranslatef(320.0, 200.0, 0.0);
		glScalef(2.0, 2.0, 1.0);
		planet->x = 0.0; planet->y = 0.0;
		planet->render();
		planet->x = oldX; planet->y = oldY;
		
		glLoadIdentity();
		
		TextRenderer *tr = TextHandler::getInstance()->getRenderer("wet");
		tr->setSize(0.5, 0.5);
		tr->render(320.0, 50.0, 0.5, 0.5, GLColor(1.0, 1.0, 1.0), oldName.c_str());
		
		mWindow->render();
		panel->render();
		
		allegro_gl_flip();
	}
	
	planet->name = oldName;
	
	return mWindow->selected;
}

void PlanetMenu::mine()
{
	TimeKeeper tk(0.01, 0.025, 2, true);
	
	std::string oldName = planet->name;
	double oldX = planet->x, oldY = planet->y;
	planet->name = "";
	
	bool end = false;
	tk.reset();
	KeyReader::clearBuffer();
	
	MineWindow *mWindow = new MineWindow(planet, ship);
	
	while(!end)
	{
		double dt = tk.getDT();
		
		do
		{
			mWindow->step(dt);
			
			if (mWindow->done)
				end = true;

			KeyReader::update();
		}
		while(tk.getFramesToSkip() > 0);
		
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		
		glTranslatef(320.0, 200.0, 0.0);
		glScalef(2.0, 2.0, 1.0);
		planet->x = 0.0; planet->y = 0.0;
		planet->render();
		planet->x = oldX; planet->y = oldY;
		
		glLoadIdentity();
		
		TextRenderer *tr = TextHandler::getInstance()->getRenderer("wet");
		tr->setSize(0.5, 0.5);
		tr->render(320.0, 50.0, 0.5, 0.5, GLColor(1.0, 1.0, 1.0), oldName.c_str());
		
		mWindow->render();
		panel->render();
		
		allegro_gl_flip();
	}
	
	planet->name = oldName;
	KeyReader::update();
	
	delete mWindow;
}
