#include <cmath>

#include "../entity/GameWorld.h"
#include "../things/Mover.h"
#include "../engine/GLBitmap.h"
#include "../misc/Random.h"
#include "../things/StarFlare.h"
#include "Sun.h"

/************************************************/

GLBitmap *Sun::sunBMP = NULL, *Sun::particleBMP;

/************************************************/

Sun::Sun()
	: Entity(0.0, 0.0)
{
	for (int i = 0; i < NUM_PARTICLES; i++)
		newParticle(&particle[i]);
}

/************************************************/

void Sun::targetShip()
{
	pulledObjects = parent->getEntitiesOfType(ETYPE_SHIP);
}

/************************************************/

void Sun::newParticle(SunParticle *p)
{
	p->x = p->y = 0.0;
	
	double angle = randomFloat(0.0, 6.28);
	double power = randomFloat(50.0, 150.0);
	
	p->vx = std::cos(angle) * power;
	p->vy = -std::sin(angle) * power;
	p->power = power;
	p->life = 1.0;
}

/************************************************/

void Sun::step(double dt)
{
	for (int i = 0; i < NUM_PARTICLES; i++)
	{
		particle[i].x += particle[i].vx * dt;
		particle[i].y += particle[i].vy * dt;
		
		particle[i].life -= 4.2 * (particle[i].power / 350.0) * dt;
		
		if (particle[i].life <= 0.0)
			newParticle(&particle[i]);
	}
	
	list<Entity*> toRemove;
	
	for (EL_ITERATOR it = pulledObjects.begin(); it != pulledObjects.end(); it++)
	{
		Mover *mv = (Mover*)(*it);
		
		if (!mv->isAlive())
			toRemove.push_back(mv);
			
		double distSq = mv->x * mv->x + mv->y * mv->y;
		if (distSq > 160000.0) continue;
		
		double dist = std::sqrt(distSq);
		//allegro_message("%f", (-mv->x / dist) * 1000000.0 * dt / distSq);
		mv->vx += (-mv->x / dist) * 3000000.0 * dt / distSq;
		mv->vy += (-mv->y / dist) * 3000000.0 * dt / distSq;
	}
	
	for (EL_ITERATOR it = toRemove.begin(); it != toRemove.end(); it++)
	{
		pulledObjects.remove(*it);
	}
	toRemove.clear();
	
	if (randomFloat(0.0, 100.0) < 10.0)
	{
		double angle = randomFloat(0.0, 6.28);
		double power = randomFloat(200.0, 300.0);
		double vx = std::cos(angle) * power;
		double vy = -std::sin(angle) * power;
		double deployAngle = randomFloat(0.0, 6.28);
		double x = std::cos(deployAngle) * 60.0;
		double y = -std::sin(deployAngle) * 60.0;
		
		StarFlare *starFlare = new StarFlare(x, y, vx, vy);
		parent->add(starFlare, true);
		pulledObjects.push_back(starFlare);
	}
}

/************************************************/

void Sun::render()
{
	sunBMP->setOrigin(0.5, 0.5);
	sunBMP->draw(x, y);
	
	particleBMP->setTransparencyMode(GLBitmap::TM_ALPHA_ADDITIVE);
	particleBMP->setScale(2.0, 2.0);
	particleBMP->setOrigin(0.5, 0.5);
	
	for (int i = 0; i < NUM_PARTICLES; i++)
	{
		particleBMP->setTint(GLColor(1.0, 1.0, 1.0, particle[i].life));
		particleBMP->draw(x + particle[i].x, y + particle[i].y);
	}
}

/************************************************/

bool Sun::contains(double xx, double yy)
{
	double distSq = xx * xx + yy * yy;
	return distSq < 72 * 72;
}
